# -*- coding: utf-8 -*-
"""
/***************************************************************************
 DownloadSentinel
                                 A QGIS plugin
 This plugin allows users to be able to download Sentinel 2 images.
 Generated by Plugin Builder: http://g-sherman.github.io/Qgis-Plugin-Builder/
                              -------------------
        begin                : 2024-06-05
        git sha              : $Format:%H$
        copyright            : (C) 2024 by Murat Çalışkan
        email                : caliskan.murat.20@gmail.com
 ***************************************************************************/

/***************************************************************************
 *                                                                         *
 *   This program is free software; you can redistribute it and/or modify  *
 *   it under the terms of the GNU General Public License as published by  *
 *   the Free Software Foundation; either version 2 of the License, or     *
 *   (at your option) any later version.                                   *
 *                                                                         *
 ***************************************************************************/
"""
from qgis.PyQt.QtCore import QSettings, QTranslator, QCoreApplication, QDate
from qgis.PyQt.QtGui import QIcon, QPixmap
from qgis.PyQt.QtWidgets import QAction, QFileDialog, QMessageBox, QLineEdit

# Initialize Qt resources from file resources.py
from .resources import *

# Import the code for the dialog
from .sentinel_downloader_dialog import DownloadSentinelDialog, CredsWindow, OptionsWindow

from .DrawRect import RectangleMapTool

from qgis.core import QgsVectorLayer, QgsProject, QgsWkbTypes, QgsMapLayer

from osgeo import osr, ogr
from pyproj import CRS, Transformer
from shapely.wkt import loads
from shapely.ops import transform
from datetime import datetime, timedelta, date
import requests, os
import numpy as np
from shapely.geometry import shape
import pandas as pd
import sys
from ast import literal_eval
from zipfile import ZipFile

class DownloadSentinel:
    """QGIS Plugin Implementation."""

    def __init__(self, iface):
       
        # Save reference to the QGIS interface
        self.iface = iface
        self.canvas = QgsProject.instance()
        # initialize plugin directory
        self.plugin_dir = os.path.dirname(__file__)
        # initialize locale
        locale = QSettings().value('locale/userLocale')[0:2]
        locale_path = os.path.join(
            self.plugin_dir,
            'i18n',
            'DownloadSentinel_{}.qm'.format(locale))

        if os.path.exists(locale_path):
            self.translator = QTranslator()
            self.translator.load(locale_path)
            QCoreApplication.installTranslator(self.translator)

        # Declare instance attributes
        self.actions = []
        self.menu = self.tr(u'&Sentinel 1 Image Downloader')

        # Check if plugin was started the first time in current QGIS session
        # Must be set in initGui() to survive plugin reloads
        self.first_start = None

    # noinspection PyMethodMayBeStatic
    def tr(self, message):
        
        return QCoreApplication.translate('DownloadSentinel', message)


    def add_action(
        self,
        icon_path,
        text,
        callback,
        enabled_flag=True,
        add_to_menu=True,
        add_to_toolbar=True,
        status_tip=None,
        whats_this=None,
        parent=None):
        
        icon = QIcon(icon_path)
        action = QAction(icon, text, parent)
        action.triggered.connect(callback)
        action.setEnabled(enabled_flag)

        if status_tip is not None:
            action.setStatusTip(status_tip)

        if whats_this is not None:
            action.setWhatsThis(whats_this)

        if add_to_toolbar:
            # Adds plugin icon to Plugins toolbar
            self.iface.addToolBarIcon(action)

        if add_to_menu:
            self.iface.addPluginToMenu(
                self.menu,
                action)

        self.actions.append(action)

        return action

    def initGui(self):
        """Create the menu entries and toolbar icons inside the QGIS GUI."""

        icon_path = ':/plugins/sentinel_1_downloader/icon.png'
        
        self.add_action(
            icon_path,
            text=self.tr(u'Sentinel 1 Image Downloader'),
            callback=self.run,
            parent=self.iface.mainWindow())

        # will be set False in run()
        self.first_start = True


    def unload(self):
        """Removes the plugin menu item and icon from QGIS GUI."""
        for action in self.actions:
            self.iface.removePluginMenu(
                self.tr(u'&Sentinel 1 Image Downloader'),
                action)
            self.iface.removeToolBarIcon(action)
    
    def selectQlFile(self):
        sender = self.dlg.sender()
        name = sender.objectName()
        
        if name == "btn_browse_ql":
            self.dlg.lw_input_ql.clear()
            filePath, _filter = QFileDialog.getOpenFileNames(self.dlg, "Select quicklook File(s)","", 'PNG(*.png *.PNG)')
            self.dlg.lw_input_ql.addItems(filePath)     

    def selectInputFile(self):
        sender = self.dlg.sender()
        name = sender.objectName()
        
        if name == "btn_browse_3":
            self.dlg.le_inputfile.setText("")
            filePath, _filter = QFileDialog.getOpenFileNames(self.dlg, "Select footprint File(s)","", 'GPKG(*.gpkg *.GPKG)')
            self.dlg.le_inputfile.setText(";".join(filePath))
            
        elif name == "btn_browse_4":
            self.dlg.le_inputfile_2.setText("")
            filePath, _filter = QFileDialog.getOpenFileNames(self.dlg, "Select footprint File(s)","", 'GPKG(*.gpkg *.GPKG)')
            self.dlg.le_inputfile_2.setText(";".join(filePath))

    def selectOutputFolder(self):
        sender = self.dlg.sender()
        object_name = sender.objectName()
        
        if object_name == "btn_browse":        
            self.dlg.le_outputFolder.setText("")
            self.dlg.le_outFileName.setText("")
            
            startDate = self.dlg.dt_startDate.dateTime().toPyDateTime().strftime("%Y-%m-%dT%H:%M:%S.%fZ")
            end_date_ = self.dlg.dt_endDate.dateTime().toPyDateTime()
            endDate = (end_date_ + timedelta(days=1) - timedelta(seconds=1)).strftime("%Y-%m-%dT%H:%M:%S.%fZ")
                        
            sd = datetime.strptime(startDate, "%Y-%m-%dT%H:%M:%S.%fZ").strftime("%Y%m%d")
            ed = datetime.strptime(endDate, "%Y-%m-%dT%H:%M:%S.%fZ").strftime("%Y%m%d")
                        
            output_dir = QFileDialog.getExistingDirectory(None, 'Open working directory', "", QFileDialog.ShowDirsOnly)
            self.dlg.le_outputFolder.setText(output_dir)
            outFolderPath = self.dlg.le_outputFolder.text()
            
            out_name_ = f""" footprints_{sd}_{ed} """.strip()
            
            if not os.path.isfile(os.path.join(f"{outFolderPath}", f"{out_name_}.gpkg")):
                out_name = out_name_
            else:
                n = 2
                while True:
                    out_name = out_name_ + "_" + str(n)
                    if not os.path.isfile(os.path.join(f"{outFolderPath}", f"{out_name}.gpkg")):
                        break
                    else:
                        n += 1
            
            self.dlg.le_outFileName.setText(out_name)
        
        elif object_name == "btn_browse_2":        
            self.dlg.le_outputFolder_2.setText("")
            output_dir = QFileDialog.getExistingDirectory(None, 'Open working directory', "", QFileDialog.ShowDirsOnly)
            self.dlg.le_outputFolder_2.setText(output_dir)
    
    def getTransformedGeometry(self, geom_wkt, srs_wkt, env=False):
        geom = loads(geom_wkt)
        
        from_crs = CRS.from_wkt(srs_wkt)
        to_crs = CRS.from_epsg(4326)
        
        transformer = Transformer.from_crs(from_crs, to_crs, always_xy=True)
        projected = transform(transformer.transform, geom)
        
        proj_geom = ogr.CreateGeometryFromWkt(projected.wkt)       
        
        if env:
            return proj_geom.GetEnvelope()
        else:
            return proj_geom
    
    def createGPKG(self, products_all, outfile, driver_name = "GPKG"):
        driver = ogr.GetDriverByName(driver_name)
        ds = driver.CreateDataSource(outfile)
        srs = osr.SpatialReference()
        srs.ImportFromEPSG(4326)
        
        layer = ds.CreateLayer("footprints", srs, ogr.wkbPolygon)
        
        for c, d in products_all.dtypes.items():
            if d.name == "object":
                t = ogr.OFTString
            elif "date" in d.name:
                t = ogr.OFTDateTime
            elif "int" in d.name:
                t = ogr.OFTInteger64
            elif "float" in d.name:
                t = ogr.OFTReal
            
            
            layer.CreateField(ogr.FieldDefn(c, t))
       
        
        defn = layer.GetLayerDefn()
        
        for e, (_,row) in enumerate(products_all.iterrows()):
            feat = ogr.Feature(defn)
            
            for k,v in row.items():
                if k == "footprint_wkt":
                    feat.SetField(k, v)
                    feat.SetGeometry(ogr.CreateGeometryFromWkt(v))
                else:
                    try:
                        if "date" in k:
                            feat.SetField(k, str(v))
                        else:
                            feat.SetField(k, v)
                    except:
                        message = f"ERROR: {str(e)}"
                        self.createLog(message)
        
            layer.CreateFeature(feat)
        
        ds.FlushCache()
        ds = layer = defn = feat = None
    
    def getCanvasExtent(self):
        self.dlg.cb_feat_bounds.setChecked(False)
        canvas = QgsProject.instance()
        srs_wkt = canvas.crs().toWkt()
        ext = self.iface.mapCanvas().extent()
        
        minx, maxx, miny, maxy = self.getTransformedGeometry(ext.asWktPolygon(), srs_wkt, env=True)
            
        self.dlg.sb_extent_minx.setValue(minx)
        self.dlg.sb_extent_miny.setValue(miny)
        self.dlg.sb_extent_maxx.setValue(maxx)
        self.dlg.sb_extent_maxy.setValue(maxy)
     
    def getLayerExtent(self):
        self.dlg.cb_feat_bounds.setChecked(False)
        layerName = self.dlg.cb_layers.currentText()
        layers = QgsProject.instance().mapLayersByName(layerName)
        if len(layers) > 0:
            srs_wkt = layers[0].crs().toWkt()        
            ext = layers[0].extent()
            
            minx, maxx, miny, maxy = self.getTransformedGeometry(ext.asWktPolygon(), srs_wkt, env=True)
                        
            self.dlg.sb_extent_minx.setValue(minx)
            self.dlg.sb_extent_miny.setValue(miny)
            self.dlg.sb_extent_maxx.setValue(maxx)
            self.dlg.sb_extent_maxy.setValue(maxy)
    
    def checkLayer(self):
        self.dlg.cb_feat_bounds.setChecked(False)
        self.resetExtent()
        
        layerName = self.dlg.cb_layers.currentText()
        layers = QgsProject.instance().mapLayersByName(layerName)
        if len(layers) > 0:
            if layers[0].type() == QgsMapLayer.VectorLayer:
                self.dlg.cb_feat_bounds.setEnabled(True)
                no_feats = layers[0].featureCount()
                self.dlg.lbl_no_feats.setText("# of features : " + str(no_feats) + " ")
            else:
                self.dlg.cb_feat_bounds.setEnabled(False)
                self.dlg.lbl_no_feats.setText("")
    
    def checkExtract(self):
        sender = self.dlg.sender()
        oname = sender.objectName()
        
        if oname == "cb_extract":
            if self.dlg.cb_extract.isChecked():
                self.dlg.cb_delzip.setEnabled(True)
            else:
                self.dlg.cb_delzip.setChecked(False)
                self.dlg.cb_delzip.setEnabled(False)
        
        elif oname == "rb_image":
            if self.dlg.rb_image.isChecked():
                self.dlg.cb_extract.setEnabled(True)
            else:
                self.dlg.cb_extract.setChecked(False)
                self.dlg.cb_delzip.setChecked(False)
                self.dlg.cb_extract.setEnabled(False)
                self.dlg.cb_delzip.setEnabled(False)
        
    
    def resetExtent(self):
        if self.dlg.cb_feat_bounds.isChecked():
            self.dlg.sb_extent_minx.setValue(0)
            self.dlg.sb_extent_miny.setValue(0)
            self.dlg.sb_extent_maxx.setValue(0)
            self.dlg.sb_extent_maxy.setValue(0)
            
        self.checkExtent()
    
    def getDrawnCoor(self, canvas):
        self.dlg.showMinimized()
        self.rect = RectangleMapTool(canvas, self.dlg)
        canvas.setMapTool(self.rect)
            
    def outFolderCheck(self):
        sender = self.dlg.sender()
        object_name = sender.objectName()
        
        if object_name == "le_outputFolder":
            folder_path = self.dlg.le_outputFolder.text()
            
            if os.path.isdir(folder_path):
                self.folderCheck = True
                self.dlg.lbl_message_3.setText("")
            else:
                self.folderCheck = False
                self.dlg.lbl_message_3.setText('<html><head/><body><p><span style=" color:#ff0000;"> Invalid folder path! </span></p></body></html>')
            
            self.dlg.btn_execute.setEnabled(all((self.dateCheck, self.extentCheck, self.folderCheck)))
            
        elif object_name == "le_outputFolder_2":
            folder_path = self.dlg.le_outputFolder_2.text()
            
            if os.path.isdir(folder_path):
                self.folderCheck2 = True
                self.dlg.lbl_message_4.setText("")
            else:
                self.folderCheck2 = False
                self.dlg.lbl_message_4.setText('<html><head/><body><p><span style=" color:#ff0000;"> Invalid folder path! </span></p></body></html>')
            
            self.dlg.btn_execute_2.setEnabled(all((self.fileCheck2, self.folderCheck2, self.loginCheck2)))
        
    def inputFileCheck(self):
        file_paths = self.dlg.le_inputfile.text().split(";")
        try:
            res = all([ogr.Open(p) for p in file_paths])
        except:
            res = None
        
        if res:
            self.fileCheck2 = True
            self.dlg.lbl_message_5.setText("")
        else:
            self.fileCheck2 = False
            self.dlg.lbl_message_5.setText('<html><head/><body><p><span style=" color:#ff0000;"> Invalid file path! </span></p></body></html>')
        
        self.dlg.btn_execute_2.setEnabled(all((self.fileCheck2, self.folderCheck2, self.loginCheck2)))
    
    def showPassword(self, *arg, **kwargs):
        self.dlg.lbl_img.setPixmap(self.pixmap_hide)
        self.dlg.le_password.setEchoMode(QLineEdit.Normal)
    
    def hidePassword(self, *arg, **kwargs):
        self.dlg.lbl_img.setPixmap(self.pixmap_show)   
        self.dlg.le_password.setEchoMode(QLineEdit.Password)
    
    def showPassword2(self, *arg, **kwargs):
        self.dlg3.lbl_img.setPixmap(self.pixmap_hide)
        self.dlg3.le_password.setEchoMode(QLineEdit.Normal)
    
    def hidePassword2(self, *arg, **kwargs):
        self.dlg3.lbl_img.setPixmap(self.pixmap_show)   
        self.dlg3.le_password.setEchoMode(QLineEdit.Password)
    
    def deleteCred(self):
        if self.dlg3.lw_creds.count() > 0:
            selected_row = self.dlg3.lw_creds.currentRow()
            selected_text = self.dlg3.lw_creds.currentItem().text()
            
            self.dlg3.lw_creds.takeItem(selected_row)
            
            self.creds.pop(selected_text)
            
            with open(os.path.join(os.path.dirname(__file__), 'credentials.txt'), "w") as f:
                f.write(str(self.creds))
            
            self.dlg.cb_username.clear()
            self.dlg.cb_username.addItems(sorted(self.creds.keys()))
            self.dlg.cb_username.setCurrentText("")
                
    def addCred(self):
        usr = self.dlg3.le_username.text()
        pr = self.dlg3.le_password.text()
        
        if (usr == "") or (pr == ""):
            pass
        else:
            res = QMessageBox.question(None, "Message", """Username and Password will be stored as plain text. Are you sure?""")
            if res == QMessageBox.Yes:
                if self.creds.get(usr) is None:
                    self.creds[usr] = pr
                    self.dlg3.lw_creds.addItem(usr)
                
                    with open(os.path.join(os.path.dirname(__file__), 'credentials.txt'), "w") as f:
                        f.write(str(self.creds))
                    
                    self.dlg3.le_username.setText("")
                    self.dlg3.le_password.setText("")
                
                    self.dlg.cb_username.clear()
                    self.dlg.cb_username.addItems(sorted(self.creds.keys()))
                    self.dlg.cb_username.setCurrentText("")
            
            else:
                self.dlg3.le_username.setText("")
                self.dlg3.le_password.setText("")
            
    def checkExtent(self):
        minx = float(self.dlg.sb_extent_minx.value())
        maxx = float(self.dlg.sb_extent_maxx.value())
        miny = float(self.dlg.sb_extent_miny.value())
        maxy = float(self.dlg.sb_extent_maxy.value())  
        
        controls = [maxx > minx,
                    maxy > miny]
                
        if all(controls) or self.dlg.cb_feat_bounds.isChecked():
            self.extentCheck = True
            self.dlg.lbl_message_2.setText("")
        else:
            self.extentCheck = False
            self.dlg.lbl_message_2.setText('<html><head/><body><p><span style=" color:#ff0000;"> Invalid extent value! </span></p></body></html>')
        
        
        self.dlg.btn_execute.setEnabled(all((self.dateCheck, self.extentCheck, self.folderCheck)))
    
    def tabChange(self):
        if self.dlg.tabWidget.currentIndex() == 0:
            self.dlg.pb_download.setVisible(False)
            self.dlg.pe_log.resize(381, 504)
            
        else:
            self.dlg.pb_download.setVisible(True)
            self.dlg.pe_log.resize(381, 481)
    
    def checkDates(self):
        enabled = []
        if self.dlg.dt_endDate.dateTime().toPyDateTime() < self.dlg.dt_startDate.dateTime().toPyDateTime():
            enabled.append(False)
            message = '<html><head/><body><p><span style=" color:#ff0000;"> Invalid Start or End date! </span></p></body></html>'
        else:
            enabled.append(True)
            message = ""
            
        if self.dlg.dt_startDate.dateTime().toPyDateTime() > datetime.now():
            enabled.append(False)
            message = '<html><head/><body><p><span style=" color:#ff0000;"> Invalid Start date! </span></p></body></html>'
        else:
            enabled.append(True)
            
        if all(enabled):
            self.dateCheck = True
        else:
            self.dateCheck = False
            
        self.dlg.btn_execute.setEnabled(all((self.dateCheck, self.extentCheck, self.folderCheck)))
        self.dlg.lbl_message.setText(message)
                
    
    def createLog(self, message, end=False, clear=False):
        try:
            txt = f"-- {str(datetime.now())}\n{message} - {sys.exc_info()[-1].tb_lineno}\n------------------------------------------------------------------"
        except:
            txt = f"-- {str(datetime.now())}\n{message}\n------------------------------------------------------------------"
        
        if clear:
            self.dlg.pe_log.clear() 
            
        self.dlg.pe_log.appendPlainText(txt + "\n")
        self.dlg.processEvents()
    
    def fillPassword(self):
        self.dlg.le_password.setText("")
        
        usr = self.le_username.text()
        pss = self.creds.get(usr)
        
        if pss:
            self.dlg.le_password.setText(pss)
    
    def cleanFootprints(self):
        self.dlg.btn_execute_3.setText("RUNNING...")
        self.dlg.processEvents()
        
        ql_count = self.dlg.lw_input_ql.count()
        file_txt = self.dlg.le_inputfile_2.text()
        
        ql_file_list = []
        for i in range(ql_count):
            ql_path = self.dlg.lw_input_ql.item(i).text()
            ql_name = "_".join(os.path.split(ql_path)[-1].split("_")[:-1])
            
            ql_file_list.append(ql_name)
        
        if len(ql_file_list) > 0:
            ql_file_query = ", ".join([f"'{i}'" for i in ql_file_list])
            if file_txt:
                file_list = file_txt.split(";")
                for p in file_list:
                    if (os.path.isfile(p)) and (p.endswith("gpkg")):
                        new_data_list = []
                        
                        data = ogr.Open(p)
                        layer = data.GetLayer()
                        layer_defn = layer.GetLayerDefn()
                        lyr_name = layer.GetName()
                        
                        cols = [f"{layer_defn.GetFieldDefn(f).GetName()}" for f in range(layer_defn.GetFieldCount())]
                        
                        query = f"SELECT * FROM {lyr_name} WHERE prod_identifier IN ({ql_file_query})"
                        res = data.ExecuteSQL(query)
                        
                        for feat in res:
                            geom = feat.GetGeometryRef()
                            wkt = geom.ExportToWkt()
                            new_data_list.append([*[feat.GetField(c) for c in cols], wkt])
                
                        df_new_data = pd.DataFrame(new_data_list, columns=[*cols, "wkt"])
                        
                        new_outfile = p.replace(".gpkg", "_clean.gpkg")
                        out_name = os.path.split(new_outfile)[-1]
                        self.createGPKG( df_new_data, new_outfile, driver_name = "GPKG")       
        
                        if 1:
                            try:
                                layer = QgsVectorLayer(new_outfile, out_name, "ogr")
                                QgsProject.instance().addMapLayer(layer)
                            except:
                                pass
        
        self.dlg.btn_execute_3.setText("RUN")
        self.dlg.processEvents()
        
    
    def checkCredentials(self):
        self.username = self.le_username.text()
        self.password = self.dlg.le_password.text()
        
        data = {
            "client_id": "cdse-public",
            "username": self.username,
            "password": self.password,
            "grant_type": "password",
            }
        try:
            r = requests.post("https://identity.dataspace.copernicus.eu/auth/realms/CDSE/protocol/openid-connect/token",
            data=data,
            )
            
            if r.status_code == 200:
                self.bear = r.json()["access_token"]
                self.dlg.lbl_message_6.setText('<html><head/><body><p><span style=" color:#05eb14;"> Login successful! </span></p></body></html>')
                self.createLog("Login successful!")
                self.loginCheck2 = True
            else:
                message = f"ERROR!!!\nStatus Code : {r.status_code}\n{r.text}"
                self.createLog(message, end=False, clear=False)
        
        except Exception as e:
            message = f"ERROR: {str(e)}"
            self.createLog(message, end=False, clear=False)
        
        self.dlg.btn_execute_2.setEnabled(all((self.fileCheck2, self.folderCheck2, self.loginCheck2)))
        
    
    def getUrl(self, wkt, start, end, limit, offset, instrumentShortName, platformSerialIdentifiers_, orbitDirections_, operationalModes_, polarisationChannels_, levels_):

        levels_dict = {
            "raw" : "contains(Name,%27RAW%27)",
            "slc" : "contains(Name,%27SLC%27)",
            "grd" : "contains(Name,%27GRD%27)%20and%20not%20contains(Name,%27_COG%27)",
            "grd_cog" : "contains(Name,%27GRD%27)%20and%20contains(Name,%27_COG%27)",
            "ocn" : "contains(Name,%27OCN%27)"
        }
        
        platformSerialIdentifiers = [f"Attributes/OData.CSC.StringAttribute/any(att:att/Name%20eq%20%27platformSerialIdentifier%27%20and%20att/OData.CSC.StringAttribute/Value%20eq%20%27{i}%27)" for i in platformSerialIdentifiers_]
        platformSerialIdentifiers_query = "%20or%20".join(platformSerialIdentifiers)
        
        
        orbitDirections = [f"Attributes/OData.CSC.StringAttribute/any(att:att/Name%20eq%20%27orbitDirection%27%20and%20att/OData.CSC.StringAttribute/Value%20eq%20%27{i}%27)" for i in orbitDirections_]
        orbitDirections_query = "%20or%20".join(orbitDirections)
        
        
        operationalModes = [f"Attributes/OData.CSC.StringAttribute/any(att:att/Name%20eq%20%27operationalMode%27%20and%20att/OData.CSC.StringAttribute/Value%20eq%20%27{i}%27)" for i in operationalModes_]
        operationalModes_query = "%20or%20".join(operationalModes)
        
        
        polarisationChannels = [f"Attributes/OData.CSC.StringAttribute/any(att:att/Name%20eq%20%27polarisationChannels%27%20and%20att/OData.CSC.StringAttribute/Value%20eq%20%27{i}%27)" for i in polarisationChannels_]
        polarisationChannels_query = "%20or%20".join(polarisationChannels)
        
        
        if len(levels_) == 0:
            queries_1 = f"OData.CSC.Intersects(area=geography%27SRID=4326;{wkt}%27)"
        else:
            levels = [f"({levels_dict[i]}%20and%20OData.CSC.Intersects(area=geography%27SRID=4326;{wkt}%27))" for i in levels_]
            queries_1 = f"({'%20or%20'.join(levels)})"
        
        
        queries_2 = "%20and%20".join(["Online%20eq%20true",
                                    f"({platformSerialIdentifiers_query})",
                                    f"({orbitDirections_query})",
                                    f"({operationalModes_query})",
                                    f"({polarisationChannels_query})"])
        
        
        url_ = """https://catalogue.dataspace.copernicus.eu/odata/v1/Products?&$filter=
        (
        	(
        		Collection/Name%20eq%20%27SENTINEL-1%27
        		%20and%20
        		(
        			Attributes/OData.CSC.StringAttribute/any(att:att/Name%20eq%20%27instrumentShortName%27%20and%20att/OData.CSC.StringAttribute/Value%20eq%20%27{instrumentShortName}%27)
        			%20and%20
        			{queries_1}
        		)
        		%20and%20
        		(
        			{queries_2}
        		)
        	)
        	%20and%20
        	ContentDate/Start%20ge%20{start}
        	%20and%20
        	ContentDate/Start%20lt%20{end}
        )
        &$orderby=ContentDate/Start%20desc
        &$expand=Attributes
        &$count=true
        &$top={top}
        &$expand=Assets
        &$skip={skip}"""
        
        url = "".join(url_.split()).format(instrumentShortName=instrumentShortName, wkt=wkt, queries_1=queries_1, queries_2=queries_2, start=start, end=end, top=limit, skip=offset)
        
        return url
    
    def getData(self, total_results, wkt, start, end, limit, offset, instrumentShortName, platformSerialIdentifiers_, orbitDirections_, operationalModes_, polarisationChannels_, levels_):
        try:
            
            url = self.getUrl(wkt, start, end, limit, offset, instrumentShortName, platformSerialIdentifiers_, orbitDirections_, operationalModes_, polarisationChannels_, levels_)
            
            resp = requests.get(url)
                    
            if resp.status_code == 200:
                results_json = resp.json()
                if total_results:
                    return results_json.get("@odata.count")
                else:
                    return results_json.get("value")
            else:
                if total_results:
                    return f"ERROR! --> Status Code:{resp.status_code} (total_results)"
                else:
                    return f"ERROR! --> Status Code:{resp.status_code} (get data)"
    
        except Exception as e:
            return f"ERROR! --> {str(e)}"
        

    def getProducts(self, products):
        product_list = []
        for prod in products:
            attributes = {}
            
            assets = [i for i in prod["Assets"] if i.get("Type")=="QUICKLOOK"]
            
            if len(assets) > 0:
                attributes["quicklook_download_url"] = assets[0].get("DownloadLink")
            
            prod_id = prod["Id"]
            attributes["prod_id"] = prod_id
            attributes["prod_identifier"] = prod["Name"].rstrip(".SAFE")
            
            attributes["size_mb"] = int(round(prod.get("ContentLength",0)/(1024*1024)))
                
            attributes["prod_download_url"] = f"https://download.dataspace.copernicus.eu/odata/v1/Products({prod_id})/$value"
            
            attributes.update({i["Name"]:i["Value"] for i in prod["Attributes"]})
            
            transformer = Transformer.from_crs(CRS('EPSG:4326'), CRS('EPSG:3035'), always_xy=True).transform
            attributes["area_km"] = round(transform(transformer, shape(prod["GeoFootprint"])).area/1000000)
            
            attributes["footprint_wkt"] = shape(prod["GeoFootprint"]).wkt
            
            product_list.append(attributes)
        
        df = pd.DataFrame(product_list)
        
        try:
            df["beginningDateTime"] = pd.to_datetime(df["beginningDateTime"].str.replace("Z","").str.replace("T"," ")).dt.floor("s")
        except:
            pass
        try:
            df["endingDateTime"] = pd.to_datetime(df["endingDateTime"].str.replace("Z","").str.replace("T"," ")).dt.floor("s")
        except:
            pass
        try:
            df["processingDate"] = pd.to_datetime(df["processingDate"].str.replace("Z","").str.replace("T"," ")).dt.floor("s")
        except:
            pass
        
        df.columns = [i.lower() for i in df.columns]
        
        cols = ['prod_id', 'prod_identifier', 'platformserialidentifier', 'orbitdirection', 'polarisationchannels', 'operationalmode',
                'swathidentifier', 'producttype', 
                'beginningdatetime', 'endingdatetime', 'processingdate', 'size_mb', 'prod_download_url', 'quicklook_download_url', 
                'origin', 'datatakeid', 'timeliness', 'cyclenumber', 'orbitnumber',
                 'slicenumber', 'totalslices', 'productclass', 'processorname',
                  'processinglevel', 'processingcenter', 'processorversion',
                 'segmentstarttime', 'sliceproductflag', 'platformshortname', 'productcomposition', 'instrumentshortname',
                 'relativeorbitnumber', 'instrumentconfigurationid', 
                 'starttimefromascendingnode', 'completiontimefromascendingnode', 'area_km',
                 'footprint_wkt']
        
        df = df[[i for i in cols if i in df.columns]]
        
        return df
    
    
    def executeFootprints(self):
        missings = []
        
        platformSerialIdentifier_check = any([self.dlg2.cb_s1a.isChecked(),
                                              self.dlg2.cb_s1b.isChecked(),
                                              self.dlg2.cb_s1c.isChecked()])
        if not platformSerialIdentifier_check:
            missings.append("- Select at least 1 Platform Serial Identifier")        
            
            
        orbitDirection_check = any([self.dlg2.cb_ascending.isChecked(),
                                    self.dlg2.cb_descending.isChecked()])
        if not orbitDirection_check:
            missings.append("- Select at least 1 Orbit Direction")  
            
            
        operationalMode_check = any([self.dlg2.cb_mode_sm.isChecked(),
                                     self.dlg2.cb_mode_iw.isChecked(),
                                     self.dlg2.cb_mode_ew.isChecked(),
                                     self.dlg2.cb_mode_wv.isChecked()])
        if not operationalMode_check:
            missings.append("- Select at least 1 Operational Mode")
            
            
        polarisationChannels_check = any([self.dlg2.cb_vv_vh.isChecked(),
                                          self.dlg2.cb_hh_hv.isChecked(),
                                          self.dlg2.cb_vv.isChecked(),
                                          self.dlg2.cb_hh.isChecked()])
        if not polarisationChannels_check:
            missings.append("- Select at least 1 Polarisation Channel")            
            
            
        levels_check = any([self.dlg2.cb_raw.isChecked(),
                            self.dlg2.cb_slc.isChecked(),
                            self.dlg2.cb_grd.isChecked(),
                            self.dlg2.cb_grd_cog.isChecked(),
                            self.dlg2.cb_ocn.isChecked()])
        if not levels_check:
            missings.append("- Select at least 1 Level")
        
        if len(missings) > 0:        
            missings_text = "\n".join(missings)                    
            QMessageBox.information(None, "Missing parameters.", missings_text)
            return
        
        
        self.dlg.btn_execute.setText("Running...")
        self.dlg.pb_download.setValue(0)
        self.dlg.processEvents()
        
        instrumentShortName = "SAR"
        
        platformSerialIdentifiers_ = []
        if self.dlg2.cb_s1a.isChecked():platformSerialIdentifiers_.append("A")
        if self.dlg2.cb_s1b.isChecked():platformSerialIdentifiers_.append("B")
        if self.dlg2.cb_s1c.isChecked():platformSerialIdentifiers_.append("C")
            
        orbitDirections_ = []
        if self.dlg2.cb_ascending.isChecked():orbitDirections_.append("ASCENDING")
        if self.dlg2.cb_descending.isChecked():orbitDirections_.append("DESCENDING")
            
        operationalModes_ = []
        if self.dlg2.cb_mode_sm.isChecked():operationalModes_.append("SM")
        if self.dlg2.cb_mode_iw.isChecked():operationalModes_.append("IW")
        if self.dlg2.cb_mode_ew.isChecked():operationalModes_.append("EW")
        if self.dlg2.cb_mode_wv.isChecked():operationalModes_.append("WV")
            
        polarisationChannels_ = []
        if self.dlg2.cb_vv_vh.isChecked():polarisationChannels_.append("VV%26VH")
        if self.dlg2.cb_hh_hv.isChecked():polarisationChannels_.append("HH%26HV")
        if self.dlg2.cb_vv.isChecked():polarisationChannels_.append("VV")
        if self.dlg2.cb_hh.isChecked():polarisationChannels_.append("HH")
            
        levels_ = []
        if self.dlg2.cb_raw.isChecked():levels_.append("raw")
        if self.dlg2.cb_slc.isChecked():levels_.append("slc")
        if self.dlg2.cb_grd.isChecked():levels_.append("grd")
        if self.dlg2.cb_grd_cog.isChecked():levels_.append("grd_cog")
        if self.dlg2.cb_ocn.isChecked():levels_.append("ocn")
        
        wkt_list_download = []
        if self.dlg.cb_feat_bounds.isChecked():
            layerName = self.dlg.cb_layers.currentText()
            layer = QgsProject.instance().mapLayersByName(layerName)[0]
            srs_wkt = layer.crs().toWkt()
            if not srs_wkt:
                QMessageBox.critical(None, "ERROR", """Invalid CRS!""")
                return
            
            features = layer.getFeatures()
            wkt_list = []
            for feat in features:
                geom = feat.geometry()
                geom_type = geom.wkbType()
                geom_wkt = geom.asWkt()
                
                geom2 = self.getTransformedGeometry(geom_wkt, srs_wkt, env=False)
                
                if geom_type in [QgsWkbTypes.LineString, QgsWkbTypes.MultiLineString, QgsWkbTypes.Point, QgsWkbTypes.MultiPoint]:
                    wkt_list.append(geom2.ExportToWkt())
                else:
                    minx, maxx, miny, maxy = geom2.GetEnvelope()
                    wkt_list.append(f"POLYGON(({minx} {maxy}, {maxx} {maxy}, {maxx} {miny}, {minx} {miny}, {minx} {maxy}))")
        
        else:        
            minx = float(self.dlg.sb_extent_minx.value())
            maxx = float(self.dlg.sb_extent_maxx.value())
            miny = float(self.dlg.sb_extent_miny.value())
            maxy = float(self.dlg.sb_extent_maxy.value())
            wkt = f"POLYGON(({minx} {maxy}, {maxx} {maxy}, {maxx} {miny}, {minx} {miny}, {minx} {maxy}))"
            wkt_list = [wkt]
        
        startDate = self.dlg.dt_startDate.dateTime().toPyDateTime().strftime("%Y-%m-%dT%H:%M:%S.%fZ")
        end_date_ = self.dlg.dt_endDate.dateTime().toPyDateTime()
        endDate = (end_date_ + timedelta(days=1) - timedelta(seconds=1)).strftime("%Y-%m-%dT%H:%M:%S.%fZ")
        
        limit = 100
        outFolderPath = self.dlg.le_outputFolder.text()
        
        self.createLog("Process Started", clear=True)
        
        total_results = 0
        offset_lists = []
        for we, wkt in enumerate(wkt_list):
            total = self.getData(True, wkt, startDate, endDate, limit, 0, instrumentShortName, platformSerialIdentifiers_, orbitDirections_, operationalModes_, polarisationChannels_, levels_)
            
            if isinstance(total, str):
                message = f"{total} - {wkt}"
                self.createLog(message)
                
            else:
                if total > 0:
                    message = f"{we+1} - {total} features"
                    self.createLog(message)
                    
                    wkt_list_download.append(wkt_list[we])
                    total_results += total
                    offset_lists.append(np.arange(0, total, limit))
        
        if isinstance(total_results, str):
            message = f"total_results couldn't retrieve - {total_results}"
            self.createLog(message)
        else:
            if total_results == 0:
                QMessageBox.information(None, "Number of Image", """No images returned after query. Change the parameters and try again.""")
            else:
                message = f"total_results : {total_results}"
                self.createLog(message)
                
                res = QMessageBox.question(None, "Number of Image", f"""{total_results} images from {len(wkt_list_download)} extents returned after query. This number may decrease after deleting the duplicates.\nDo you want to create footprints?""")
                if res == QMessageBox.Yes:
                    
                    self.createLog("Creating footprints...") 
                    
                    products_all = pd.DataFrame()
                    for en, (wkt, offset_list) in enumerate(zip(wkt_list_download, offset_lists), 1):
                        for i, o in enumerate(offset_list, 1):
                            self.createLog(f"{en} -- {i} / {len(offset_list)}")
                            products = self.getData(False, wkt, startDate, endDate, limit, o, instrumentShortName, platformSerialIdentifiers_, orbitDirections_, operationalModes_, polarisationChannels_, levels_)
                            if isinstance(products, str):
                                message = f"{products}"
                                self.createLog(message)
                            else:
                                df = self.getProducts(products)
                                df["query_wkt"] = wkt
                                
                                products_all = pd.concat([products_all, df], axis=0)                
                    
                    products_all = products_all.drop_duplicates(subset="prod_id")         
                    
                    out_name = self.dlg.le_outFileName.text()
                    
                    if os.path.isfile(os.path.join(f"{outFolderPath}", f"{out_name}.gpkg")):
                        QMessageBox.critical(None, "ERROR", "File already exists.")
                        self.dlg.btn_execute.setText("Generate Footprints")
                        self.dlg.processEvents()
                        return
                    
                    self.createGPKG(products_all, os.path.join(f"{outFolderPath}", f"{out_name}.gpkg"), driver_name = "GPKG")
                    self.createLog(f"Exported to {outFolderPath}/{out_name}.gpkg")
                    
                    self.createLog("Footprints created.")
                    QMessageBox.information(None, "Created", f"""Footprints file created.\n Number of footprints : {products_all.shape[0]}""")
                    
                    if self.dlg.cb_add_to_layer.isChecked():
                        try:
                            layer = QgsVectorLayer(os.path.join(f"{outFolderPath}", f"{out_name}.gpkg"), out_name, "ogr")
                            QgsProject.instance().addMapLayer(layer)
                        except:
                            pass
        
        self.dlg.btn_execute.setText("Generate Footprints")
        self.dlg.processEvents()
                        
    def get_access_token(self):
        data = {
            "client_id": "cdse-public",
            "username": self.username,
            "password": self.password,
            "grant_type": "password",
            }
        try:
            r = requests.post("https://identity.dataspace.copernicus.eu/auth/realms/CDSE/protocol/openid-connect/token",
            data=data,
            )
            
            return r.json()["access_token"]
        
        except Exception as e:
            return f"ERROR: {str(e)}"
        
    def downloadImage(self, url, name):
        outpath = os.path.join(f"{self.dlg.le_outputFolder_2.text()}", f"{name}.zip")
        if os.path.isfile(outpath):
            self.createLog(f"{name} - Skipping... File already exists.")
            return
        else:
            self.createLog(f"{name}")
                               
        session = requests.Session()
        if self.bear is None:
            self.bear = self.get_access_token()
        session.headers.update({"Authorization": f"Bearer {self.bear}"})
        
        try:
            response = session.get(url, stream=True)            
            
            if response.status_code == 200:       
                
                with open(outpath, "wb") as f:
                    for chunk in response.iter_content(chunk_size=250 * 1024):
                        f.write(chunk)
                message = "Image Downloaded!"
                
            elif response.status_code == 401:
                try:
                    self.bear = self.get_access_token()
                    session.headers.update({"Authorization": f"Bearer {self.bear}"})                    
                    response = session.get(url, stream=True)
                    
                    with open(outpath, "wb") as f:
                        for chunk in response.iter_content(chunk_size=250 * 1024):
                            f.write(chunk)
                    
                    message = "Image Downloaded!"
                
                except Exception as e:
                    message = f"ERROR!!!\n{str(e)}"
            
            else:
                message =  f"ERROR!!!\nDownload --> Image Couldn't be downloaded - Status Code:{response.status_code}"
                
        except Exception as e:
            message = f"ERROR!!!\n{str(e)}"
        
        self.createLog(message)
        
    def showCredsSettings(self):
        self.dlg3.pb_delete_cred.setEnabled(False)
        self.dlg3.lw_creds.clear()
        
        with open(os.path.join(os.path.dirname(__file__), 'credentials.txt')) as f:
            self.creds = literal_eval(f.read())
        
        self.dlg3.lw_creds.addItems(sorted(self.creds.keys()))
            
        self.dlg3.show()

    def executeDownloadImages(self):
        
        self.dlg.pb_download.setValue(0)
        self.dlg.processEvents()      
        
        img_paths = self.dlg.le_inputfile.text().split(";")
        total_images = 0
        for ip in img_paths:
            if ip.endswith("gpkg"):
                f = ogr.Open(ip)
                lyr = f.GetLayer()
                fc = lyr.GetFeatureCount()
                total_images += fc
        
        res = QMessageBox.question(None, "Number of Image", f"""{total_images} frame(s) will be downloaded. Do you want to continue?""")
        if res == QMessageBox.No:
            return
        
        self.createLog("DOWNLOAD STARTED")
        
        self.dlg.btn_execute_2.setText("Running...")
        self.dlg.processEvents()
        
        download_list = []
        file_paths = self.dlg.le_inputfile.text().split(";")
        for file_path in file_paths:
            data = ogr.Open(file_path)
            layer = data.GetLayer()
            lyr_defn = layer.GetLayerDefn()
            field_cnt = lyr_defn.GetFieldCount()
            fields = [lyr_defn.GetFieldDefn(i).GetName() for i in range(field_cnt)]
            
            cons = [("prod_id" in fields),
                    ("prod_download_url" in fields),
                    ("quicklook_download_url" in fields),
                    ("prod_identifier" in fields),
                    ("query_wkt" in fields)
                    ]
    
            if not all(cons):
                QMessageBox.critical(None, "ERROR", """Invalid input file. Use footprins file that created by this plugin!""")
                self.dlg.btn_execute_2.setText("Download Images")
                self.dlg.processEvents()
                
                return
            
            for feat in layer:
                geom_ = feat.GetGeometryRef()
                geom_wkt = geom_.ExportToWkt()
                
                prod_url = feat.GetField("prod_download_url")
                ql_url = feat.GetField("quicklook_download_url")
                img_name = feat.GetField("prod_identifier")               
                
                download_list.append([geom_wkt, prod_url, ql_url, img_name])
        
            data.FlushCache()
            data = layer = None
            del data, layer
        
        for f_row, ddd in enumerate(download_list,1):
            self.dlg.pb_download.setValue(int(((f_row-0.5)/len(download_list))*100))
            self.dlg.processEvents()
            
            geom_wkt, prod_url, ql_url, img_name = ddd
            
            if self.dlg.rb_ql.isChecked():
                if os.path.isfile(os.path.join(f"{self.dlg.le_outputFolder_2.text()}", f"{img_name}_ql.tif")):
                    self.createLog(f"{img_name} - Skipping... File already exists.")
                    continue               
                
                try:                        
                    self.createLog(f"{img_name}")
                    
                    response = requests.get(ql_url)
                    if response.status_code == 200:
                        outdir_path_ql = os.path.join(f"{self.dlg.le_outputFolder_2.text()}", "quicklook")
                        os.makedirs(outdir_path_ql, exist_ok=True)
                        
                        f = response.content
                        
                        with open(os.path.join(outdir_path_ql, f"{img_name}_ql.png"), "wb") as ff:
                            ff.write(f)
                            
                except Exception as e:
                    self.createLog(str(e))
                
                        
            elif self.dlg.rb_image.isChecked():
                try:
                    self.downloadImage(prod_url, img_name)
                    
                    if self.dlg.cb_extract.isChecked():
                        inpath = os.path.join(f"{self.dlg.le_outputFolder_2.text()}", f"{img_name}.zip")
                        with ZipFile(inpath, 'r') as z:
                            z.extractall(path = self.dlg.le_outputFolder_2.text())
                        
                        if self.dlg.cb_delzip.isChecked():
                            os.remove(inpath)
                    
                except Exception as e:
                    self.createLog(str(e))
            
            self.dlg.pb_download.setValue(int((f_row/len(download_list))*100))
            self.dlg.processEvents()
        
        self.dlg.btn_execute_2.setText("Download Images")
        self.dlg.processEvents()
        
        QMessageBox.information(None, "Created", """Images Downloaded.""")
        
    def onCloseEvent(self, event):
        self.dlg2.close()
        self.dlg3.close()
    
    def clearAllChecks(self):
        cb_list_1 = self.dlg2.layoutWidget.children()
        cb_list_2 = self.dlg2.layoutWidget1.children()
        cb_list_3 = self.dlg2.layoutWidget2.children()
        cb_list_4 = self.dlg2.layoutWidget_2.children()
        cb_list_5 = self.dlg2.layoutWidget_3.children()
        
        cb_list = [*cb_list_1, *cb_list_2, *cb_list_3, *cb_list_4, *cb_list_5]
        
        for i in cb_list:
            if i.objectName().startswith("cb"):
                i.setChecked(False)
                
    def removeQlList(self):
        if self.dlg.lw_input_ql.count() > 0:
            selected_items = self.dlg.lw_input_ql.selectedItems()
            
            if len(selected_items) > 0:
                for item in selected_items:
                    self.dlg.lw_input_ql.takeItem(self.dlg.lw_input_ql.row(item))
    
    def run(self):
        """Run method that performs all the real work"""

        # Create the dialog with elements (after translation) and keep reference
        # Only create GUI ONCE in callback, so that it will only load when the plugin is started
        if self.first_start == True:
            # self.first_start = False
            self.dlg = DownloadSentinelDialog()
            self.dlg2 = OptionsWindow()
            self.dlg3 = CredsWindow()
            
            self.dlg.closeEvent = self.onCloseEvent
            
            if any([
                (date.today().day == 18 and date.today().month == 3),
                (date.today().day == 23 and date.today().month == 4),
                (date.today().day == 19 and date.today().month == 5),
                (date.today().day == 30 and date.today().month == 8),
                (date.today().day == 29 and date.today().month == 10),
                (date.today().day == 10 and date.today().month == 11)
                ]):
                
                self.dlg.setWindowIcon(QIcon(':/plugins/sentinel_downloader/mka.png'))
                self.dlg2.setWindowIcon(QIcon(':/plugins/sentinel_downloader/mka.png')) 
                self.dlg3.setWindowIcon(QIcon(':/plugins/sentinel_downloader/mka.png'))            
            
            self.dateCheck = True
            self.extentCheck = False
            self.folderCheck = False
            
            self.fileCheck2 = False
            self.folderCheck2 = False
            self.loginCheck2 = False
            
            self.dlg.pb_download.setVisible(False)
            self.dlg.pe_log.resize(381, 504)
            
            self.bear = None
            
            self.pixmap_hide = QPixmap(':/plugins/sentinel_downloader/show.png')
            self.pixmap_show = QPixmap(':/plugins/sentinel_downloader/hide.png')
            
            self.dlg.lbl_img.setPixmap(self.pixmap_show)
            self.dlg3.lbl_img.setPixmap(self.pixmap_show)
            
            self.dlg.lbl_info.setText("""<html><head/><body><a href="https://github.com/caliskanmurat/qgis_sentinel1_image_downloader_plugin"><img width="20" height="20" src=":/plugins/sentinel_downloader/info.png"/></a></body></html>""")
            
            ed = QDate.currentDate()       
            self.dlg.dt_endDate.setDate(ed)
            
            sd = QDate.currentDate().addDays(-10)
            self.dlg.dt_startDate.setDate(sd)
                        
            layers = [v.name() for v in QgsProject.instance().mapLayers().values()]
            self.dlg.cb_layers.clear()
            self.dlg.cb_layers.addItems(layers)
            
            layerName = self.dlg.cb_layers.currentText()
            selected_layers = QgsProject.instance().mapLayersByName(layerName)
            if len(selected_layers) > 0:
                if selected_layers[0].type() == QgsMapLayer.VectorLayer:
                    self.dlg.cb_feat_bounds.setEnabled(True)
                    no_feats = selected_layers[0].featureCount()
                    self.dlg.lbl_no_feats.setText("# of features : " + str(no_feats) + " ")
                
                else:
                    self.dlg.cb_feat_bounds.setEnabled(False)     
            
            self.dlg.lbl_message_2.setText('<html><head/><body><p><span style=" color:#ff0000;"> Invalid extent value! </span></p></body></html>')
            self.dlg.lbl_message_3.setText('<html><head/><body><p><span style=" color:#ff0000;"> Invalid folder path! </span></p></body></html>')
            self.dlg.lbl_message_4.setText('<html><head/><body><p><span style=" color:#ff0000;"> Invalid folder path! </span></p></body></html>')
            self.dlg.lbl_message_5.setText('<html><head/><body><p><span style=" color:#ff0000;"> Invalid file path! </span></p></body></html>')
            self.dlg.lbl_message_6.setText('<html><head/><body><p><span style=" color:#ff0000;"> Login to download! </span></p></body></html>')
            
            self.dlg.btn_canvasExtent.clicked.connect(self.getCanvasExtent)
            self.dlg.btn_layerextent.clicked.connect(self.getLayerExtent)
            self.dlg.tbtn_draw.clicked.connect(lambda x:self.getDrawnCoor(self.iface.mapCanvas()))            
         
            self.dlg.btn_execute.clicked.connect(self.executeFootprints)
            self.dlg.btn_execute_2.clicked.connect(self.executeDownloadImages)
            self.dlg.btn_execute_3.clicked.connect(self.cleanFootprints)
            self.dlg.btn_browse.clicked.connect(self.selectOutputFolder)
            self.dlg.btn_browse_2.clicked.connect(self.selectOutputFolder)
            self.dlg.btn_browse_3.clicked.connect(self.selectInputFile)
            self.dlg.btn_browse_4.clicked.connect(self.selectInputFile)
            self.dlg.btn_browse_ql.clicked.connect(self.selectQlFile)
            self.dlg.pb_remove.clicked.connect(self.removeQlList)
            
            self.dlg.dt_startDate.dateChanged.connect(self.checkDates)
            self.dlg.dt_endDate.dateChanged.connect(self.checkDates)
            self.dlg.sb_extent_minx.valueChanged.connect(self.checkExtent)
            self.dlg.sb_extent_maxx.valueChanged.connect(self.checkExtent)
            self.dlg.sb_extent_miny.valueChanged.connect(self.checkExtent)
            self.dlg.sb_extent_maxy.valueChanged.connect(self.checkExtent)
            
            self.dlg.le_outputFolder.textChanged[str].connect(self.outFolderCheck)
            self.dlg.le_outputFolder_2.textChanged[str].connect(self.outFolderCheck)
            self.dlg.le_inputfile.textChanged[str].connect(self.inputFileCheck)
            
            self.dlg.btn_checkcreds.clicked.connect(self.checkCredentials)
            
            self.dlg.lbl_img.mousePressEvent = self.showPassword
            self.dlg.lbl_img.mouseReleaseEvent = self.hidePassword
            
            self.dlg3.lbl_img.mousePressEvent = self.showPassword2
            self.dlg3.lbl_img.mouseReleaseEvent = self.hidePassword2
            
            self.dlg.pb_creds_setting.clicked.connect(self.showCredsSettings)
            
            self.dlg.btn_clearLogs.clicked.connect(lambda x:self.dlg.pe_log.clear())
                        
            self.dlg.cb_layers.currentTextChanged.connect(self.checkLayer)
            self.dlg.cb_feat_bounds.clicked.connect(self.resetExtent)
            self.dlg.cb_extract.stateChanged.connect(self.checkExtract)
            self.dlg.rb_image.toggled.connect(self.checkExtract)
            
            self.dlg.tabWidget.currentChanged.connect(self.tabChange)
            
            self.dlg3.lw_creds.itemPressed.connect(lambda x:self.dlg3.pb_delete_cred.setEnabled(True))
            self.dlg3.pb_delete_cred.clicked.connect(self.deleteCred)
            self.dlg3.pb_add_cred.clicked.connect(self.addCred)
            
            with open(os.path.join(os.path.dirname(__file__), 'credentials.txt')) as f:
                self.creds = literal_eval(f.read())
            
            self.dlg.cb_username.addItems(sorted(self.creds.keys()))
            
            self.le_username = QLineEdit()
            self.le_username.setPlaceholderText("Email")
            self.dlg.cb_username.setLineEdit(self.le_username)
            self.dlg.cb_username.setCurrentIndex(-1)
            
            self.le_username.textChanged[str].connect(self.fillPassword)
            
            self.dlg.pb_options.clicked.connect(lambda x:self.dlg2.show())
            self.dlg2.btn_clearOptions.clicked.connect(self.clearAllChecks)
            self.dlg2.btn_close.clicked.connect(lambda x:self.dlg2.hide())
            
        self.dlg.show()