# -*- coding: utf-8 -*-
# """
# /***************************************************************************
#  Terra Tools
#                                  A QGIS plugin
#  This window has access to Tools for Terra/SCM application.
#  Generated by Plugin Builder: http://g-sherman.github.io/Qgis-Plugin-Builder/
#                              -------------------
#         begin                : 2022-08-25
#         git sha              : $Format:%H$
#         copyright            : (C) 2022 by SenseHawk
#         email                : kiranh@sensehawk.com
#  ***************************************************************************/
#
# /***************************************************************************
#  *                                                                         *
#  *   This program is free software; you can redistribute it and/or modify  *
#  *   it under the terms of the GNU General Public License as published by  *
#  *   the Free Software Foundation; either version 2 of the License, or     *
#  *   (at your option) any later version.                                   *
#  *                                                                         *
#  ***************************************************************************/
# """

from ..sensehawk_apis.core_apis import save_project_geojson, get_project_geojson
from ..sensehawk_apis.scm_apis import get_models_list, detect
from ..event_filters import KeypressFilter, KeypressEmitter, KeypressShortcut, MousepressFilter
from ..tasks import clipRequest, detectionTask, approveTask

from qgis.PyQt import QtWidgets, uic
from qgis.core import QgsMessageLog, Qgis, QgsApplication, QgsTask, QgsFeatureRequest, QgsPoint
from qgis.PyQt.QtCore import Qt
import os
import qgis
from PyQt5.QtGui import QKeySequence

from .ml_service_map import MLServiceMapWindow

from PyQt5.QtWidgets import QApplication

import json


TERRA_TOOLS_UI, _ = uic.loadUiType(os.path.join(os.path.dirname(__file__), 'terra_tools.ui'))


class TerraToolsWindow(QtWidgets.QDockWidget, TERRA_TOOLS_UI):

    def __init__(self, load_window, iface):
        """Constructor."""
        super(TerraToolsWindow, self).__init__()
        self.setupUi(self)
        self.backButton.clicked.connect(self.show_load_window)
        self.loadModelsButton.clicked.connect(self.load_models)
        self.detectButton.clicked.connect(self.start_detect_task)
        self.approveButton.clicked.connect(self.start_approve_task)
        self.clipButton.clicked.connect(self.start_clip_task)
        self.requestModelButton.clicked.connect(self.request_model)
        self.saveProject.clicked.connect(self.save_project)
        self.load_window = load_window
        self.core_token = self.load_window.core_token
        self.project_details = self.load_window.project_details
        self.class_maps = self.load_window.class_maps
        self.class_groups = self.load_window.class_groups
        self.iface = iface
        self.canvas = self.iface.mapCanvas()
        self.active_layer = self.iface.activeLayer()
        # Add to the left docking area by default
        self.iface.addDockWidget(Qt.LeftDockWidgetArea, self)
        self.models_dict = {}
        ## Keyboard shortcuts
        self.keyboard_shortcuts = {}
        # Create the feature change shortcuts for this particular project using class maps
        self.create_feature_change_shortcuts()
        # Create QGIS tools shortcuts
        self.create_qgis_shortcuts()
        # If there is an existing keyboard shortcuts csv, remap shortcuts in case there are custom changes
        self.keyboard_shortcut_details_path = os.path.join(os.path.dirname(__file__), 'terra_keyboard_shortcuts.csv')
        if os.path.exists(self.keyboard_shortcut_details_path):
            self.remap_keyboard_shortcuts()
        # Create a key emitter that sends the key presses
        self.key_emitter = KeypressEmitter()
        # Connect the key emitter to the key eater that performs required shortcuts
        self.key_emitter.signal.connect(lambda x: self.key_eater(x))
        # Create keypress event filter to consume the key presses from iface and send it to key_emitter
        self.keypress_filter = KeypressFilter(self.key_emitter)
        # Install key press filter to iface's map canvas
        self.iface.mapCanvas().installEventFilter(self.keypress_filter)
        self.export_keyboard_shortcut_details()

        # Mouse press filter
        self.mouse_emitter = KeypressEmitter()
        self.mousepress_filter = MousepressFilter(self.mouse_emitter)

        # ML Service Map
        self.ml_service_map_window = None

    def logger(self, message, level=Qgis.Info):
        QgsMessageLog.logMessage(message, 'SenseHawk QC', level=level)

    def show_load_window(self):
        self.load_window.terra_tools_window = self
        self.load_window.show()
        self.hide()

    def request_model(self):
        self.ml_service_map_window = MLServiceMapWindow(self.iface, self.class_groups, self)
        self.ml_service_map_window.show()
        self.hide()

    def load_models(self):
        # Get list of available models
        self.models_dict = get_models_list(self.load_window.project_uid, self.core_token)
        models_list = list(self.models_dict.keys())
        if models_list:
            list_items = models_list
        else:
            list_items = ["No models available"]
        # Clear list to avoid duplicates
        self.detectionModel.clear()
        self.detectionModel.addItems(list_items)

    def add_feature(self):
        if not isinstance(self.active_layer, qgis._core.QgsVectorLayer):
            self.logger("Select a vector layer to add feature...")
            return None
        self.active_layer.startEditing()
        self.iface.actionAddFeature().trigger()

    # Shortcut
    def save_layer(self):
        if not isinstance(self.active_layer, qgis._core.QgsVectorLayer):
            return None
        # Deselect any selected features
        self.active_layer.removeSelection()
        self.active_layer.commitChanges()
        # In case duplicate feature was activated, disconnect copy-in-place on left mouse click
        try:
            self.mouse_emitter.signal.disconnect()
            self.iface.mapCanvas().viewport().removeEventFilter(self.mousepress_filter)
        except Exception:
            pass
        self.active_layer.triggerRepaint()

    def start_clip_task(self):
        def callback(task, logger):
            result = task.returned_values
            if result:
                logger(str(result["message"]))
        self.logger("Clip task starting...")
        clip_task_inputs = self.logger, self.project_details, self.load_window.geojson_path, self.class_maps, self.core_token
        clip_task = QgsTask.fromFunction("Clip Request", clipRequest, clip_task_input=clip_task_inputs)
        clip_task.statusChanged.connect(lambda:callback(clip_task, self.logger))
        QgsApplication.taskManager().addTask(clip_task)

    def start_detect_task(self):
        def callback(task, logger):
            result = task.returned_values
            if result:
                logger(str(result))
        self.logger("Detection called..")
        geojson = get_project_geojson(self.project_details.get("uid", None), self.core_token, "terra")
        self.logger("Getting model information...")
        model_name = self.detectionModel.currentText()
        if model_name not in self.models_dict:
            self.logger("Invalid model...")
            return None
        model_url = self.models_dict[model_name]
        model_details = [model_name, model_url]
        self.logger("Initiating detection request task...")
        detection_task = QgsTask.fromFunction("Detect", detectionTask,
                                              detection_task_input=[self.project_details, geojson,
                                                                    model_details, self.load_window.user_email,
                                                                    self.core_token])
        detection_task.statusChanged.connect(lambda:callback(detection_task, self.logger))
        QgsApplication.taskManager().addTask(detection_task)

    def start_approve_task(self):
        def callback(task, logger):
            result = task.returned_values
            if result:
                logger(str(result))
        self.logger("Approve called...")
        geojson = get_project_geojson(self.project_details.get("uid", None), self.core_token, "terra")
        approve_task = QgsTask.fromFunction("Approve", approveTask,
                                            approve_task_input=[self.project_details, geojson,
                                                                self.load_window.user_email, self.core_token])
        approve_task.statusChanged.connect(lambda:callback(approve_task, self.logger))
        QgsApplication.taskManager().addTask(approve_task)

    # Shortcut function
    def change_feature_type(self, class_name):
        # Change feature type of the selected features or the last added feature
        if not isinstance(self.active_layer, qgis._core.QgsVectorLayer):
            self.logger("Activate a vector layer or select feature to change feature type...")
            return None
        self.active_layer.startEditing()
        # If there are selected items, change feature type for those or else change feature type of last added feature
        selected_features = list(self.active_layer.selectedFeatures())
        if selected_features:
            for feature in selected_features:
                name = self.class_maps.get(class_name, {}).get("name", None)
                feature.setAttribute("class", name)
                feature.setAttribute("class_name", class_name)
                class_id = self.class_maps.get(class_name, {}).get("id", None)
                feature.setAttribute("class_id", int(class_id))
                self.active_layer.updateFeature(feature)
        else:
            features = list(self.active_layer.getFeatures())
            last_feature_index = -1
            try:
                last_feature = features[last_feature_index]
            except IndexError:
                self.logger("No feature selected or new feature added...")
                return None
            self.logger("Changing class_name of last added feature to {}".format(class_name))
            name = self.class_maps.get(class_name, {}).get("name", None)
            last_feature.setAttribute("class", name)
            last_feature.setAttribute("class_name", class_name)
            class_id = self.class_maps.get(class_name, {}).get("id", None)
            last_feature.setAttribute("class_id", int(class_id))
            self.active_layer.updateFeature(last_feature)
        self.active_layer.triggerRepaint()

    def create_feature_change_shortcuts(self):
        # Populate shortcuts dictionary for feature type change with keys
        shortcuts_dict = {self.class_maps[i]["name"]: {"key": None,
                                                       "class_name": i,
                                                       "key_code": None,
                                                       "name": self.class_maps[i]["name"],
                                                       "function": self.change_feature_type,
                                                       "function_args": [i, ],
                                                       "shortcut_type": "Feature type change"} for i in self.class_maps}
        shortcuts_dict.get("clip_boundary", shortcuts_dict.get("Clip_boundary", {}))["key"] = "C"
        shortcuts_dict.get("train_boundary", shortcuts_dict.get("Train_boundary", {}))["key"] = "T"
        # We will give shortcuts to other classes starting from 1 in alphabetical order
        count = 1
        remaining_classes = sorted([i for i in shortcuts_dict if not shortcuts_dict[i]["key"]])
        for c in remaining_classes:
            shortcuts_dict[c]["key"] = str(count)
            count += 1
        # Generate keycode which is communicated from keyboard
        for s in shortcuts_dict:
            key = shortcuts_dict[s]["key"]
            key_code = eval("Qt.Key_{}".format(key.upper()))
            shortcuts_dict[s]["key_code"] = key_code

        for i in shortcuts_dict:
            self.keyboard_shortcuts[shortcuts_dict[i]["key_code"]] = KeypressShortcut(shortcuts_dict[i])

    def duplicate_feature(self):
        # Set layer as editable
        self.active_layer.startEditing()
        # Install mouse press filter to iface's map canvas
        self.iface.mapCanvas().viewport().installEventFilter(self.mousepress_filter)
        move_function = self.iface.actionMoveFeature()
        move_function.trigger()
        # Connect mouse event to copy in place function
        self.mouse_emitter.signal.connect(lambda mouse_event: copy_in_place(mouse_event))
        def copy_in_place(mouse_event):
            mouse_button = mouse_event.button()
            point = self.canvas.getCoordinateTransform().toMapCoordinates(mouse_event.pos())
            x, y = point.x(), point.y()
            if mouse_button == 1:
                # If anything is selected, deselect and do nothing else
                if self.active_layer.selectedFeatures():
                    self.active_layer.removeSelection()
                    return None

                # Left button repeats function if nothing is selected currently
                feature_request = QgsFeatureRequest()
                feature_request.setFilterRect(QgsPoint(x, y).boundingBox())
                features = self.active_layer.getFeatures(feature_request)
                # Sort features by area and select the least area feature for copy and move
                features = sorted([f for f in features], key=lambda f: f.geometry().area())
                if not features:
                    self.active_layer.removeSelection()
                    return False
                selected_feature = features[0]
                f_id = selected_feature.id()
                self.active_layer.selectByIds([f_id])
                self.iface.actionCopyFeatures().trigger()
                self.iface.actionPasteFeatures().trigger()

    def create_qgis_shortcuts(self):
        # 'Enter' key saves the active layer
        self.keyboard_shortcuts[16777220] = KeypressShortcut({"key_code": 16777220,
                                                              "name": "Save layer changes",
                                                              "function": self.save_layer,
                                                              "shortcut_type": "QGIS tools"})
        # 'S' key for select feature
        self.keyboard_shortcuts[83] = KeypressShortcut({"key_code": 83,
                                                        "name": "Select features",
                                                        "function": self.iface.actionSelect().trigger,
                                                        "shortcut_type": "QGIS tools"})
        # 'F' key to add new feature
        self.keyboard_shortcuts[70] = KeypressShortcut({"key_code": 70,
                                                        "name": "Add new feature",
                                                        "function": self.add_feature,
                                                        "shortcut_type": "QGIS tools"})
        # 'P' key to change feature properties / show attributes table
        self.keyboard_shortcuts[80] = KeypressShortcut({"key_code": 80,
                                                        "name": "Show attribute table",
                                                        "function": "iface.showAttributeTable(iface.activeLayer())",
                                                        "shortcut_type": "QGIS tools"})
        # 'E' key to toggle editing of selected layer
        self.keyboard_shortcuts[69] = KeypressShortcut({"key_code": 69,
                                                        "name": "Turn on layer edit",
                                                        "function": self.active_layer.startEditing,
                                                        "shortcut_type": "QGIS tools"})

        # 'Z' key to zoom to layer
        self.keyboard_shortcuts[90] = KeypressShortcut({"key_code": 90,
                                                        "name": "Zoom to layer",
                                                        "function": self.iface.actionZoomToLayer().trigger,
                                                        "shortcut_type": "QGIS tools"})

        # 'D' key to duplicate feature
        self.keyboard_shortcuts[68] = KeypressShortcut({"key_code": 68,
                                                        "name": "Duplicate feature",
                                                        "function": self.duplicate_feature,
                                                        "shortcut_type": "QGIS tools"})


    def key_eater(self, key_code):
        # Is connected to the keyboard and will call the relevant functions from the shortcuts_dict when a key is pressed
        shortcut = self.keyboard_shortcuts.get(key_code, None)
        if not shortcut:
            return None
        self.logger("Keyboard shortcut: {}".format(shortcut.name))
        shortcut.run()

    def save_project(self):
        def save_task(task, save_task_input):
            geojson_path, core_token, project_uid, project_type = save_task_input
            cleaned_geojson_path = geojson_path.replace(".geojson", "_cleaned.geojson")
            # Delete any duplicate features
            qgis.processing.run('qgis:deleteduplicategeometries',
                                {"INPUT": geojson_path, "OUTPUT": cleaned_geojson_path})
            with open(cleaned_geojson_path, 'r') as fi:
                geojson = json.load(fi)
            # Clear UIDs to avoid duplicate error while saving to Therm
            for f in geojson["features"]:
                f["properties"]["uid"] = None
            # Upload vectors
            saved = save_project_geojson(geojson, project_uid, core_token,
                                         project_type=project_type)
            return {'status': str(saved), 'task': task.description()}

        def callback(task, logger):
            result = task.returned_values
            if result:
                logger(result["status"])

        if self.load_window.load_successful:
            st = QgsTask.fromFunction("Save", save_task,
                                      save_task_input=[self.load_window.geojson_path,
                                                       self.core_token,
                                                       self.load_window.project_uid,
                                                       self.load_window.project_type])
            QgsApplication.taskManager().addTask(st)
            st.statusChanged.connect(lambda: callback(st, self.logger))

    def export_keyboard_shortcut_details(self):
        self.logger("Exporting key board shortcut details...")
        csv_lines = ["Key, Shortcut Name, Shortcut Type\n"]
        for key_code in self.keyboard_shortcuts:
            shortcut = self.keyboard_shortcuts[key_code]
            name = shortcut.name
            shortcut_type = shortcut.shortcut_type
            key = QKeySequence(key_code).toString()
            csv_lines.append("{}, {}, {}\n".format(key, name, shortcut_type))
        with open(self.keyboard_shortcut_details_path, "w") as fi:
            for l in csv_lines:
                fi.write(l)
    def remap_keyboard_shortcuts(self):
        # Default keyboard shortcuts
        default_map = {i: self.keyboard_shortcuts[i].name for i in self.keyboard_shortcuts}
        # Keyboard shortcuts from csv
        csv_map = {}
        csv_lines = open(self.keyboard_shortcut_details_path).readlines()[1:]
        for l in csv_lines:
            key, name, _ = l.split(",")
            key, name = key.strip(), name.strip()
            key_code = eval("Qt.Key_{}".format(key))
            csv_map[name] = key_code
        # If the existing keyboard shortcuts do not contain the classes from this project, do not remap
        try:
            default_to_csv_map = {i: csv_map[default_map[i]] for i in default_map}
        except KeyError:
            return None
        for i in default_to_csv_map:
            old_keycode = i
            new_keycode = default_to_csv_map[i]
            if old_keycode == new_keycode:
                continue
            self.keyboard_shortcuts[new_keycode] = self.keyboard_shortcuts[old_keycode]
            del self.keyboard_shortcuts[old_keycode]