from __future__ import absolute_import
from builtins import range
from builtins import object
from qgis.PyQt.QtWidgets import QMenu, QAction, QWidget, QDockWidget, QToolBar, QToolButton, QListView, QVBoxLayout, QAbstractItemView 
from qgis.PyQt.QtGui import QStandardItemModel, QStandardItem
from qgis.PyQt.QtCore import Qt, QObject, pyqtSignal
from qgis.core import QgsProject, QgsMessageLog, QgsVectorLayer
from qgis.PyQt.QtGui import QIcon

from . import images
from .resources import *
import json
import zlib
import base64


def classFactory(iface):
    return SelectionSetsPlugin(iface)


class SelectionSetWidget(QWidget):
    saveSet = pyqtSignal()
    saveSetAll = pyqtSignal()
    setSelected = pyqtSignal(dict)
    modified = pyqtSignal()

    def __init__(self, parent=None):
        super(SelectionSetWidget, self).__init__(parent)

        self.saveAllAction = QAction(images.ADDALL, "Save selection set (All layers)", self)
        self.saveAction = QAction(images.ADD, "Save selection set (Active layer)", self)
        self.deleteAction = QAction(images.REMOVE, "Delete set", self)
        self.saveAllAction.triggered.connect(self.saveSetAll.emit)
        self.saveAction.triggered.connect(self.saveSet.emit)
        self.deleteAction.triggered.connect(self.deleteSet)

        self.menu = QMenu()
        self.menu.addAction(self.saveAction)
        self.menu.addAction(self.saveAllAction)
        self.toolbar = QToolBar()
        self.saveButton = QToolButton()
        self.toolbar.addWidget(self.saveButton)
        self.deleteAction = self.toolbar.addAction(self.deleteAction)

        self.saveButton.setMenu(self.menu)
        self.saveButton.setPopupMode(QToolButton.MenuButtonPopup)
        self.saveButton.setDefaultAction(self.saveAction)

        self.setList = MyListView()
        self.setModel = QStandardItemModel()
        self.setList.setModel(self.setModel)
        self.setList.selectionModel().currentChanged.connect(self._itemSelected)
        self.setList.clicked.connect(self.item_clicked)

        layout = QVBoxLayout()
        layout.setContentsMargins(0,0,0,0)
        self.setLayout(layout)
        layout.addWidget(self.toolbar)
        layout.addWidget(self.setList)

    def item_clicked(self, index):
        data = self._data_from_index(self.setList.selectionModel().currentIndex())
        if data is None:
            return
        self.setSelected.emit(data)

    def deleteSet(self):
        current = self.setList.selectionModel().currentIndex()
        self.setModel.removeRow(current.row())
        #QgsMessageLog.logMessage(str(current.row()))
        self.modified.emit()
        
    def removeLayerFromSet(self, layerId):
        rowlist=[]
        for row in reversed(range(self.setModel.rowCount())):
            item = self.setModel.item(row)
            name = item.text()
            itemdata = item.data()
            if layerId in itemdata.keys():
                if len(itemdata)==1: #remove whole row for single select sets
                    #print(f"name: {name}")
                    #print(f"itemdata: {itemdata}")
                    rowlist.append(row)
                else: #for multiselect sets delete only the appropriate layerid
                    #print(f"delete layer from multiselect: {name}") 
                    del itemdata[layerId]
                    item.setData(itemdata, Qt.UserRole + 1)
                    self.modified.emit()
                
        for row in rowlist:
            self.setModel.removeRow(row)
            self.modified.emit()

    def _data_from_index(self, index):
        data = index.data(Qt.UserRole + 1)
        return data

    def addSelectionSet(self, selectionset, notify=True):
        name = ",".join(layer.name() for layer in list(selectionset.keys()))
        length = sum(len(items) for items in list(selectionset.values()))
        name = f"{name} ({length})"
        self.itemFromData(name, selectionset, notify)

    def itemFromData(self, name, selectionset, notify=True):
        item = QStandardItem(name)

        data = {}
        for layer, ids in selectionset.items():
            # TODO This is just a hack because I'm lazy for now
            if isinstance(layer, QgsVectorLayer):
                data[layer.id()] = ids
            else:
                data[layer] = ids

        item.setData(data, Qt.UserRole + 1)
        self.setModel.appendRow(item)
        if notify:
            self.modified.emit()

       
    def _itemSelected(self, current, old):
        data = self._data_from_index(current)
        if data is None:
            return
        self.setSelected.emit(data)

    def dataForSaving(self):
        data = {}
        for row in range(self.setModel.rowCount()):
            item = self.setModel.item(row)
            name = item.text()
            itemdata = item.data()
            data[name] = itemdata
        return data

    def setFromLoaded(self, data):
        self.setModel.clear()
        for name, itemdata in data.items():
            self.itemFromData(name, itemdata, notify=False)

    def clear(self):
        self.setModel.clear()

class MyListView(QListView):
    def mousePressEvent(self, event):
        self._mouse_button = event.button()
        super(MyListView, self).mousePressEvent(event)


class SelectionSetsPlugin(object):
    def __init__(self, iface):
        self.iface = iface

    def initGui(self):
        self.action = QAction(QIcon(':/plugins/selectionsets/select_icon.png'),"Selection Sets", self.iface.mainWindow())
        self.action.triggered.connect(self.run)
        self.dock = QDockWidget("Selection Sets", self.iface.mainWindow())
        self.setWidget = SelectionSetWidget()
        self.setWidget.modified.connect(self.saveIntoProject)
        self.setWidget.saveSet.connect(self.saveSet)
        self.setWidget.saveSetAll.connect(self.saveSetAll)
        self.setWidget.setSelected.connect(self.updateSelection)
        self.dock.setWidget(self.setWidget)
        self.iface.addDockWidget(Qt.RightDockWidgetArea, self.dock)
        self.dock.hide()
        self.iface.addToolBarIcon(self.action)
        QgsProject.instance().readProject.connect(self.loadFromProject)
        QgsProject.instance().writeProject.connect(self.saveIntoProject)
        QgsProject.instance().layerWillBeRemoved.connect(self.on_layerWillBeRemoved)
        self.iface.newProjectCreated.connect(self.setWidget.clear)

    def unload(self):
        self.iface.removeToolBarIcon(self.action)
        self.iface.removeDockWidget(self.dock)
        del self.action

    def run(self):
        self.dock.show()
        
    def on_layerWillBeRemoved(self,layerId):
        self.setWidget.removeLayerFromSet(layerId)
        

    def saveIntoProject(self):
        data = self.setWidget.dataForSaving()
        datas = json.dumps(data).encode()
        datas_compressed = base64.b64encode(zlib.compress(datas)).decode("utf-8")
        QgsProject.instance().writeEntry("SelectionSets", "/sets", datas_compressed)

    def loadFromProject(self):
        datas_compressed = QgsProject.instance().readEntry("SelectionSets", "/sets")[0]
        try:
            datas = zlib.decompress(base64.b64decode(datas_compressed))
        except:
            self.setWidget.clear()
            return
            
        try:
            data = json.loads(datas)
            self.setWidget.setFromLoaded(data)
        except ValueError:
            self.setWidget.clear()

    def saveSet(self):
        layer = self.iface.activeLayer()
        if not isinstance(layer, QgsVectorLayer):
            return
        ids = layer.selectedFeatureIds()
        if len(ids)==0: 
            return
        data = {}
        data[layer] = ids
        self.setWidget.addSelectionSet(data)

    def saveSetAll(self):
        data = {}
        for layer in list(QgsProject.instance().mapLayers().values()):
            if not isinstance(layer, QgsVectorLayer):
                continue
            ids = layer.selectedFeatureIds()
            if not ids:
                continue
            data[layer] = ids
        self.setWidget.addSelectionSet(data)

    def updateSelection(self, data):
        for layer in list(QgsProject.instance().mapLayers().values()):
            if isinstance(layer, QgsVectorLayer):
                layer.removeSelection()
            try:
                ids = data[layer.id()]
                layer.select(ids)
                self.iface.layerTreeView().setCurrentLayer(layer)
            except KeyError:
                pass
