# -*- coding: utf-8 -*-

"""
/***************************************************************************
 SelectMatchingFeatures
                                 A QGIS plugin
 Clicking a feature will select all features in layer with same attribute value
 Generated by Plugin Builder: http://g-sherman.github.io/Qgis-Plugin-Builder/
                              -------------------
        begin                : 2025-11-05
        git sha              : $Format:%H$
        copyright            : (C) 2025 by Matt Needle
        email                : matt.needle.nl@gmail.com
 ***************************************************************************/

/***************************************************************************
 *                                                                         *
 *   This program is free software; you can redistribute it and/or modify  *
 *   it under the terms of the GNU General Public License as published by  *
 *   the Free Software Foundation; either version 2 of the License, or     *
 *   (at your option) any later version.                                   *
 *                                                                         *
 ***************************************************************************/
"""
import os
from qgis.PyQt.QtCore import QSettings, QTranslator, QCoreApplication, Qt, QLocale, pyqtSignal
from qgis.PyQt.QtGui import QIcon
from qgis.PyQt.QtWidgets import QAction, QToolBar
from qgis.core import Qgis, QgsGeometry
from qgis.gui import QgsMapToolIdentifyFeature, QgsMapToolIdentify

# Initialize Qt resources from file resources.py
from .resources import *

# Import utilities
from .utils import MessageHandler, SettingsManager, Validator, ValidationError, PluginLogger

# Import core classes
from .core import SelectionManager, ExpressionBuilder, HighlightManager

# Import GUI components
from .gui.select_matching_features_dockwidget import SelectMatchingFeaturesDockWidget


class CustomIdentifyTool(QgsMapToolIdentifyFeature):
    """Custom map tool that picks the closest feature and detects empty clicks"""

    noFeatureClicked = pyqtSignal()

    def __init__(self, canvas):
        super().__init__(canvas)
        self._feature_was_identified = False
        self._target_layer = None

    def setLayer(self, layer):
        """Override to store the layer reference"""
        super().setLayer(layer)
        self._target_layer = layer

    def canvasPressEvent(self, event):
        """Override to identify closest feature at click point"""
        if not self._target_layer:
            return

        # Identify all features at this point in our target layer
        results = self.identify(event.x(), event.y(), [self._target_layer], QgsMapToolIdentify.TopDownAll)

        if results:
            # Get click point in map coordinates
            click_point = self.toMapCoordinates(event.pos())
            click_geom = QgsGeometry.fromPointXY(click_point)

            # Find the closest feature to click point
            min_distance = float('inf')
            closest_feature = None

            for result in results:
                feature = result.mFeature
                distance = feature.geometry().distance(click_geom)
                if distance < min_distance:
                    min_distance = distance
                    closest_feature = feature

            # Emit for the closest feature only
            if closest_feature:
                self._feature_was_identified = True
                self.featureIdentified.emit(closest_feature)

    def canvasReleaseEvent(self, event):
        """Called when mouse button is released"""
        # If no feature was identified on press, it's an empty click
        if not self._feature_was_identified:
            self.noFeatureClicked.emit()

        # Reset for next click
        self._feature_was_identified = False


class SelectMatchingFeatures:
    """
    QGIS Plugin Implementation - Refactored Version.
    
    This refactored version eliminates code duplication and improves modularity by:
    - Using MessageHandler for all user messages
    - Using SelectionManager for selection/filtering (with bug fix)
    - Using ExpressionBuilder for expression generation
    - Using HighlightManager for feature highlights
    - Using Validator for input validation
    """
    
    def __init__(self, iface):
        """Constructor."""
        self.iface = iface
        self.dockwidget = None
        self.map_tool = None
        self.selection_toolbar = None
        self.reference_feature = None
        
        # Initialize utility classes
        self.message_handler = MessageHandler(iface)
        self.settings_manager = SettingsManager()
        self.highlight_manager = HighlightManager(iface.mapCanvas())

        # Initialize locale
        if QSettings().value('locale/overrideFlag', type=bool):
            locale = QSettings().value('locale/userLocale')
        else:
            locale = QLocale.system().name()

        locale_path = os.path.join(
            os.path.dirname(__file__),
            'i18n',
            'select_matching_features_{}.qm'.format(locale[0:2]))

        if os.path.exists(locale_path):
            self.translator = QTranslator()
            self.translator.load(locale_path)
            QCoreApplication.installTranslator(self.translator)

    def tr(self, message):
        """Get the translation for a string using Qt translation API."""
        return QCoreApplication.translate('SelectMatchingFeatures', message)

    def initGui(self):
        """Create the menu entries and toolbar icons inside the QGIS GUI."""
        icon_path = os.path.join(os.path.dirname(__file__), "select_icon.png")
        
        self.action = QAction(
            QIcon(icon_path),
            self.tr('Select Matching Features'),
            self.iface.mainWindow()
        )
        self.action.setToolTip(self.tr('Show/Hide Select Matching Features Panel'))
        self.action.setCheckable(True)
        self.action.triggered.connect(self.run)

        # Add to menu and toolbar
        self.iface.addPluginToVectorMenu(self.tr('&Select Matching Features'), self.action)
        
        # Find the Selection Toolbar
        for toolbar in self.iface.mainWindow().findChildren(QToolBar):
            if toolbar.windowTitle() == 'Selection Toolbar':
                self.selection_toolbar = toolbar
                break

        # Add to Selection Toolbar if found, otherwise use default plugins toolbar
        if self.selection_toolbar:
            self.selection_toolbar.addAction(self.action)
        else:
            self.iface.addToolBarIcon(self.action)

    def unload(self):
        """Removes the plugin menu item and icon from QGIS GUI."""
        self.iface.removePluginVectorMenu(self.tr('&Select Matching Features'), self.action)
        
        if self.selection_toolbar:
            self.selection_toolbar.removeAction(self.action)
        else:
            self.iface.removeToolBarIcon(self.action)
        
        if self.dockwidget:
            self.iface.removeDockWidget(self.dockwidget)
            self.dockwidget.deleteLater()
            self.dockwidget = None

    def run(self):
        """Show or hide the docked widget."""
        if self.dockwidget is None:
            # Create the docked widget with settings manager
            self.dockwidget = SelectMatchingFeaturesDockWidget(self.iface, self.settings_manager)

            # Connect signals
            self.dockwidget.toolActivated.connect(self.on_tool_activated)
            self.dockwidget.showAllFeaturesRequested.connect(self.on_show_all_features)
            self.dockwidget.visibilityChanged.connect(self.on_dockwidget_visibility_changed)
            self.dockwidget.mFieldCombo.fieldChanged.connect(self.on_field_changed)
            self.dockwidget.selectFilteredChanged.connect(self.on_select_filtered_changed)
            self.dockwidget.applyFilterToSelection.connect(self.on_apply_filter_to_selection)
            self.dockwidget.clearFilterAndSelectFiltered.connect(self.on_clear_filter_and_select)

            self.iface.addDockWidget(Qt.RightDockWidgetArea, self.dockwidget)
            self.dockwidget.show()
            return

        # Toggle visibility on subsequent clicks
        if self.dockwidget.isVisible():
            self.dockwidget.hide()
        else:
            self.dockwidget.show()

    def on_dockwidget_visibility_changed(self, visible):
        """Update the action checked state when dockwidget visibility changes."""
        self.action.setChecked(visible)
        
        if not visible and self.map_tool:
            self.deactivate_map_tool()

    def on_tool_activated(self, activated):
        """Handle when the activate tool button is clicked."""
        if activated:
            self.activate_map_tool()
        else:
            self.deactivate_map_tool()

    def activate_map_tool(self):
        """Activate the map tool for selecting matching features."""
        if not self.dockwidget:
            return
        
        layer = self.dockwidget.get_current_layer()
        field_name = self.dockwidget.get_current_field()
        
        # Validate using Validator class
        try:
            Validator.validate_layer_and_field(layer, field_name)
        except ValidationError as e:
            self.message_handler.show_warning(str(e))
            self.dockwidget.set_tool_active(False)
            return

        # Create map tool if needed
        if self.map_tool is None:
            self.map_tool = CustomIdentifyTool(self.iface.mapCanvas())
            self.map_tool.featureIdentified.connect(self.on_feature_clicked)
            self.map_tool.noFeatureClicked.connect(self.on_empty_click)
        
        self.map_tool.setLayer(layer)
        self.iface.mapCanvas().setMapTool(self.map_tool)
        
        # Show feedback using MessageHandler
        self.message_handler.show_info(
            f"Click a feature to select all matching features in field '{field_name}'",
            title="Tool Active"
        )
        
        PluginLogger.info(f"Selection tool activated for layer: {layer.name()}, field: {field_name}")

    def deactivate_map_tool(self):
        """Deactivate the map tool and clean up."""
        self.message_handler.clear()
        
        # Clear reference feature and highlights
        self.reference_feature = None
        self.highlight_manager.clear_all()

        if self.map_tool and self.iface.mapCanvas().mapTool() == self.map_tool:
            self.iface.mapCanvas().unsetMapTool(self.map_tool)
        
        if self.dockwidget:
            self.dockwidget.set_tool_active(False)
            
            layer = self.dockwidget.get_current_layer()
            if layer and SelectionManager.has_active_filter(layer):
                SelectionManager.clear_filter(layer)
                self.dockwidget.enable_show_all_button(False)
        
        PluginLogger.info("Selection tool deactivated")

    def perform_selection(self, feature, field_name):
        """
        Perform selection based on a feature and field name.
        Refactored to use ExpressionBuilder and SelectionManager.
        """
        if not self.dockwidget:
            return

        layer = self.dockwidget.get_current_layer()
        if not layer:
            return

        try:
            # Extract field value
            value = self._extract_field_value(feature, field_name)
            operator = self.dockwidget.get_operator()
            
            # Build expression using ExpressionBuilder
            try:
                expr, display_value = ExpressionBuilder.build_expression(field_name, operator, value)
            except ValueError as e:
                # Operator not compatible with NULL
                self.message_handler.show_warning(str(e))
                return
            
            # Clear any existing filter and select
            SelectionManager.clear_filter(layer)
            layer.selectByExpression(expr)
            count = layer.selectedFeatureCount()

            # Apply filter if in hide mode
            hide_mode = self.dockwidget.get_selection_mode()
            if hide_mode:
                success, count, error = SelectionManager.apply_filter_to_selection(layer)
                if not success:
                    self.message_handler.show_error(error)
                    return
                
                SelectionManager.clear_selection(layer)

                # If "select filtered" checkbox is checked, select all filtered features
                if self.dockwidget.get_select_filtered_state():
                    SelectionManager.select_all_in_filter(layer)

                action_text = "Filtered"
            else:
                action_text = "Selected"

            # Enable button if we have results or in hide mode
            if count > 0 or hide_mode:
                self.dockwidget.enable_show_all_button(True)

            # Show feedback - use warning level for NULL, success for others
            if value is None:
                level_method = self.message_handler.show_warning
                is_null_text = "IS NULL" if operator == "=" else "IS NOT NULL"
                message = f"{action_text} {count} feature(s) where {field_name} {is_null_text}"
            else:
                level_method = self.message_handler.show_success
                message = f"{action_text} {count} feature(s) with {field_name} {operator} {display_value}"
            
            level_method(message)
            PluginLogger.info(f"Selected {count} features with expression: {expr}")

        except KeyError:
            self.message_handler.show_warning(f"Feature doesn't have field '{field_name}'")
        except Exception as e:
            self.message_handler.show_error(f"Failed to select features: {str(e)}")
            PluginLogger.error(f"Selection error: {str(e)}")

    def _extract_field_value(self, feature, field_name):
        """
        Extract and normalize field value from feature.
        
        Args:
            feature: QgsFeature
            field_name: Name of field to extract
            
        Returns:
            Field value (None if NULL, otherwise the actual value)
        """
        raw_value = feature[field_name]
        
        # Handle QVariant NULL
        if raw_value is None or (hasattr(raw_value, 'isNull') and raw_value.isNull()):
            return None
        
        return raw_value

    def on_feature_clicked(self, feature):
        """Handle when a feature is clicked."""
        if not self.dockwidget:
            return

        layer = self.dockwidget.get_current_layer()
        field_name = self.dockwidget.get_current_field()

        if not layer or not field_name:
            return

        # Store reference feature for field change re-evaluation
        self.reference_feature = feature

        # Update highlight using HighlightManager
        self.highlight_manager.add_highlight('reference', feature.geometry(), layer)

        # Perform the selection
        self.perform_selection(feature, field_name)

    def on_field_changed(self, field_name):
        """Handle when the field selection changes - re-evaluate with reference feature"""
        # Only re-evaluate if we have a reference feature and tool is active
        if not self.reference_feature or not self.map_tool:
            return

        if not self.dockwidget:
            return

        if self.iface.mapCanvas().mapTool() != self.map_tool:
            return

        if not field_name:
            return

        # Perform the selection with the stored reference feature
        self.perform_selection(self.reference_feature, field_name)

    def on_select_filtered_changed(self, checked):
        """Handle when select filtered checkbox is toggled"""
        if not self.dockwidget:
            return

        layer = self.dockwidget.get_current_layer()
        if not layer:
            return

        if checked:
            if SelectionManager.has_active_filter(layer):
                count = SelectionManager.select_all_in_filter(layer)
                if count >= 0:
                    self.message_handler.show_success(f"Selected {count} filtered feature(s)")
        else:
            SelectionManager.clear_selection(layer)
            self.message_handler.show_success("Selection cleared")

    def on_apply_filter_to_selection(self):
        """Apply filter to the current selection when switching from select to filter mode"""
        if not self.dockwidget:
            return

        layer = self.dockwidget.get_current_layer()
        if not layer:
            return

        # Use SelectionManager to apply filter
        success, count, error = SelectionManager.apply_filter_to_selection(layer)
        
        if not success:
            self.message_handler.show_warning(error)
            return

        # Check if we should keep the selection or clear it
        select_filtered = self.dockwidget.get_select_filtered_state()

        if not select_filtered:
            SelectionManager.clear_selection(layer)
            self.message_handler.show_success(f"Filtered to {count} feature(s)")
        else:
            # Selection already kept by SelectionManager
            self.message_handler.show_success(f"Filtered to {count} selected feature(s) (selection kept)")

        PluginLogger.info(f"Filtered layer to {count} selected features")

    def on_clear_filter_and_select(self):
        """
        Clear filter and select the features that were filtered.
        Called when switching from filter mode to select mode.
        """
        if not self.dockwidget:
            return

        layer = self.dockwidget.get_current_layer()
        if not layer:
            return

        # Use SelectionManager to clear filter and select filtered features
        success, count, error = SelectionManager.clear_filter_and_select_features(layer)
        
        if not success:
            # If there's an error, just show a warning but don't block the mode switch
            self.message_handler.show_warning(error)
            return

        # Show success message
        self.message_handler.show_success(
            f"Filter cleared - {count} feature(s) now selected"
        )
        
        PluginLogger.info(f"Cleared filter and selected {count} features (mode switched to select)")

    def on_empty_click(self):
        """Handle when user clicks on empty space (no feature)"""
        if not self.dockwidget:
            return

        hide_mode = self.dockwidget.get_selection_mode()
        layer = self.dockwidget.get_current_layer()

        if hide_mode and layer and SelectionManager.has_active_filter(layer):
            # Clear the filter
            SelectionManager.clear_filter(layer)
            SelectionManager.clear_selection(layer)

            # Clear reference feature and highlights
            self.reference_feature = None
            self.highlight_manager.clear_all()

            # Disable the show all features button
            self.dockwidget.enable_show_all_button(False)

            self.message_handler.show_success(f"All features in '{layer.name()}' are now visible")
            PluginLogger.info(f"Filter cleared by empty click for layer: {layer.name()}")
            
        elif layer:
            # Not in filter mode, but still clear selection if it exists
            if layer.selectedFeatureCount() > 0:
                SelectionManager.clear_selection(layer)
                self.reference_feature = None
                self.highlight_manager.clear_all()
                self.message_handler.show_success("Selection cleared")

    def on_show_all_features(self):
        """Clear selection or filter depending on the mode."""
        if not self.dockwidget:
            return

        # Clear reference feature and highlights
        self.reference_feature = None
        self.highlight_manager.clear_all()

        layer = self.dockwidget.get_current_layer()
        hide_mode = self.dockwidget.get_selection_mode()

        if not layer:
            return

        try:
            self.message_handler.clear()

            if hide_mode:
                # Hide mode: Clear the filter and selection
                SelectionManager.clear_filter(layer)
                SelectionManager.clear_selection(layer)

                self.message_handler.show_success(f"All features in '{layer.name()}' are now visible")
                PluginLogger.info(f"Filter cleared for layer: {layer.name()}")
            else:
                # Select only mode: Just clear the selection
                SelectionManager.clear_selection(layer)
                self.message_handler.show_success(f"Selection cleared in '{layer.name()}'", duration=3)
                PluginLogger.info(f"Selection cleared for layer: {layer.name()}")
            
        except Exception as e:
            self.message_handler.show_error(f"Failed to clear: {str(e)}")
            PluginLogger.error(f"Error: {str(e)}")
