# -*- coding: utf-8 -*-
"""
/***************************************************************************
 SeisPosImport
                                 A QGIS plugin
 This plugin imports Seismic Positioning or Navigation data from various formats such as P1/11 (P111), P1/90 (P190), SPS.
 Generated by Plugin Builder: http://g-sherman.github.io/Qgis-Plugin-Builder/
                              -------------------
        begin                : 2020-06-30
        git sha              : $Format:%H$
        copyright            : (C) 2020 by Gurpreet Singh
        email                : gurpreet.s.singh@gmail.com
 ***************************************************************************/

/***************************************************************************
 *                                                                         *
 *   This program is free software; you can redistribute it and/or modify  *
 *   it under the terms of the GNU General Public License as published by  *
 *   the Free Software Foundation; either version 2 of the License, or     *
 *   (at your option) any later version.                                   *
 *                                                                         *
 ***************************************************************************/
"""
from PyQt5.QtCore import QSettings, QTranslator, qVersion, QCoreApplication, QFileInfo
from PyQt5.QtGui import QIcon
from PyQt5.QtWidgets import QAction, QFileDialog
from qgis.core import QgsProject, QgsVectorLayer
from qgis.core import QgsMessageLog
from qgis import processing
import os
import re
import csv

# Initialize Qt resources from file resources.py
from .resources import *
# Import the code for the dialog
from .seismic_importer_dialog import SeisPosImportDialog
import os.path


class SeisPosImport:
    """QGIS Plugin Implementation."""

    def __init__(self, iface):
        """Constructor.

        :param iface: An interface instance that will be passed to this class
            which provides the hook by which you can manipulate the QGIS
            application at run time.
        :type iface: QgsInterface
        """
        # Save reference to the QGIS interface
        self.iface = iface
        # initialize plugin directory
        self.plugin_dir = os.path.dirname(__file__)
        # initialize locale
        locale = QSettings().value('locale/userLocale')[0:2]
        locale_path = os.path.join(
            self.plugin_dir,
            'i18n',
            'SeisPosImport_{}.qm'.format(locale))

        if os.path.exists(locale_path):
            self.translator = QTranslator()
            self.translator.load(locale_path)

            if qVersion() > '4.3.3':
                QCoreApplication.installTranslator(self.translator)

        # Declare instance attributes
        self.actions = []
        self.menu = self.tr(u'&Seismic Positioning Data Importer')
        # TODO: We are going to let the user set this up in a future iteration
        self.toolbar = self.iface.addToolBar(u'Seismic_Importer')
        self.toolbar.setObjectName(u'Seismic_Importer')

        #This is written by @gsingh7 for initializing the plugin.
        
        self.dlg = SeisPosImportDialog()
        self.first_start = False
        
        #List the choices for importing and added to combo box, update variable importChoice whenever clicked
        
        self.importChoiceDict = {}
        self.importChoiceDict['receiverSPS'] = "Receiver SPS (*.R01)"
        self.importChoiceDict['sourceSPS'] = "Source SPS (*.S01)"
        self.importChoiceDict['p190V'] = "P190 V records Import (*.P190)"
        self.importChoiceDict['p190S'] = "P190 S Records Import (*.P190)"
        self.importChoiceDict['p190E'] = "P190 E Records Import (*.P190)"
        self.importChoiceDict['p190_Custom'] = "P190 Custom Records Import (*.P190)"
        self.importChoiceDict['preplot190']="P190 preplot import (*.190)"
        self.importChoiceDict['p111S']="P111 S Records Import (*.p111)"
        self.importChoiceDict['p111S_Geomerge']="P111 S Records Import for Geomerge (*.p111)"
        self.importChoiceDict['p111_Custom']="P111 Custom Records Import (*.p111)"
        self.importChoiceList = [v for v in self.importChoiceDict.values()]     
        self.importChoice = self.importChoiceList[0]
        self.dlg.comboBox.addItems(self.importChoiceList)
        self.dlg.comboBox.activated[str].connect(self.import_choice)

        #whenever pushButton is clicked connected to openFile function
        self.dlg.pushButton.clicked.connect(self.get_filenames)
        self.dlg.pushButton_output.clicked.connect(self.get_savefilename)
        self.customFilterRequired = False

        #self.firstLastOnly = False

        # Check if plugin was started the first time in current QGIS session
        # Must be set in initGui() to survive plugin reloads
        self.first_start = None

    # noinspection PyMethodMayBeStatic
    def tr(self, message):
        """Get the translation for a string using Qt translation API.

        We implement this ourselves since we do not inherit QObject.

        :param message: String for translation.
        :type message: str, QString

        :returns: Translated version of message.
        :rtype: QString
        """
        # noinspection PyTypeChecker,PyArgumentList,PyCallByClass
        return QCoreApplication.translate('SeisPosImport', message)


    def add_action(
        self,
        icon_path,
        text,
        callback,
        enabled_flag=True,
        add_to_menu=True,
        add_to_toolbar=True,
        status_tip=None,
        whats_this=None,
        parent=None):
        """Add a toolbar icon to the toolbar.

        :param icon_path: Path to the icon for this action. Can be a resource
            path (e.g. ':/plugins/foo/bar.png') or a normal file system path.
        :type icon_path: str

        :param text: Text that should be shown in menu items for this action.
        :type text: str

        :param callback: Function to be called when the action is triggered.
        :type callback: function

        :param enabled_flag: A flag indicating if the action should be enabled
            by default. Defaults to True.
        :type enabled_flag: bool

        :param add_to_menu: Flag indicating whether the action should also
            be added to the menu. Defaults to True.
        :type add_to_menu: bool

        :param add_to_toolbar: Flag indicating whether the action should also
            be added to the toolbar. Defaults to True.
        :type add_to_toolbar: bool

        :param status_tip: Optional text to show in a popup when mouse pointer
            hovers over the action.
        :type status_tip: str

        :param parent: Parent widget for the new action. Defaults None.
        :type parent: QWidget

        :param whats_this: Optional text to show in the status bar when the
            mouse pointer hovers over the action.

        :returns: The action that was created. Note that the action is also
            added to self.actions list.
        :rtype: QAction
        """

        icon = QIcon(icon_path)
        action = QAction(icon, text, parent)
        action.triggered.connect(callback)
        action.setEnabled(enabled_flag)

        if status_tip is not None:
            action.setStatusTip(status_tip)

        if whats_this is not None:
            action.setWhatsThis(whats_this)

        if add_to_toolbar:
            # Adds plugin icon to Plugins toolbar
            self.iface.addToolBarIcon(action)

        if add_to_menu:
            self.iface.addPluginToVectorMenu(
                self.menu,
                action)

        self.actions.append(action)

        return action

    def initGui(self):
        """Create the menu entries and toolbar icons inside the QGIS GUI."""

        icon_path = ':/plugins/seismic_importer/icon.png'
        self.add_action(
            icon_path,
            text=self.tr(u'Import Seismic Positioining Data'),
            callback=self.run,
            parent=self.iface.mainWindow())

        # will be set False in run()
        self.first_start = True


    def unload(self):
        """Removes the plugin menu item and icon from QGIS GUI."""
        for action in self.actions:
            self.iface.removePluginVectorMenu(
                self.tr(u'&Seismic Positioning Data Importer'),
                action)
            self.iface.removeToolBarIcon(action)
    def get_filenames(self):
        """Open P190/SPS from file dialog"""
        self.inFiles = []
        self.inFiles, self.typeFile = QFileDialog.getOpenFileNames(caption=self.tr("Open SPS/P190 files"), filter=self.tr(self.importChoice+";; All Files (*.*)"))
        self.dlg.textEdit.setText('Type of file: '+self.typeFile)
        for file in self.inFiles:
            self.dlg.textEdit.append(QFileInfo(file).fileName())
            #print (str(file))
        self.dlg.textEdit.append(self.importChoice)

    def get_savefilename(self):
        """Open the shapefilename to save"""
        self.outFile, _ = QFileDialog.getSaveFileName(caption = self.tr("Save File name"), filter = 'CSV(*.csv)')
        if not self.outFile:
            QgsMessageLog.logMessage("\n...Could not open the mentioned save Filename")
        else:
            self.dlg.textEdit.append("\n Output file name: \n"+QFileInfo(self.outFile).fileName())

    def import_choice(self, text):
        self.importChoice = text


    def read_schema(self):
        schemaDict={}
        schemaDict['fieldDictList'] = []
        schemaDict['LineNameFrom'] = 'Linename'
        schemaDict['pointnumFrom'] = 'pointnum'
        schemaDict['XFrom'] = 'X'
        schemaDict['YFrom'] = 'Y'
        with open(self.schemaPath,'r') as fschema:
            for line in fschema:
                fields = line.split(':')
                if(fields[0]=='schema'):
                    QgsMessageLog.logMessage("\n...Loading Schema: "+fields[1].strip())
                    schemaDict['schemaName']=fields[1].strip()
                if(fields[0]=='type'):
                    schemaDict['type']=fields[1].strip()
                if(fields[0]=='filter'):
                    schemaDict['filter']=fields[1].strip()
                if(fields[0]=='LineDirectionFrom'):
                    schemaDict['lineDirectionFrom']=fields[1].strip()
                if(fields[0]=='LineNameFrom'):
                    schemaDict['LineNameFrom']=fields[1].strip()
                if(fields[0]=='pointnumFrom'):
                    schemaDict['pointnumFrom']=fields[1].strip()
                if(fields[0]=='XFrom'):
                    schemaDict['XFrom']=fields[1].strip()
                if(fields[0]=='YFrom'):
                    schemaDict['YFrom']=fields[1].strip() 
                if(fields[0]=='field'):
                    fieldDict={'fieldName':fields[1].strip(),'fieldStart':(int(fields[2].strip())-1),'fieldEnd':int(fields[3].strip()),'fieldType':fields[4].strip()}
                    schemaDict['fieldDictList'].append(fieldDict)
        fschema.close()
        return schemaDict
                    
                    
            
        
    def import_file(self):
        schemaDict = self.read_schema()
        QgsMessageLog.logMessage("Schema"+str(schemaDict))
        pointList = []
        for file in self.inFiles:
            filePath = QFileInfo(file).absoluteFilePath()
            f = open(filePath, 'rb')
            #pointListPerFile = []
            pointListPerLine = []
            prevLinename = ''
            for line in f:
                line = unicode(line, errors='replace')
                if(self.customFilterRequired):
                    m = re.search(self.filter,line)
                else:
                    m = re.search(schemaDict['filter'],line)
                if m:
                    point={}
                    if(schemaDict['type']=='fixedWidth'):
                        for fieldDict in schemaDict['fieldDictList']:
                            tempread =(line[fieldDict['fieldStart']:fieldDict['fieldEnd']]).strip()
                            #QgsMessageLog.logMessage(tempread)
                            if(fieldDict['fieldType']=='int'):
                                point[fieldDict['fieldName']]=int(tempread)
                            elif(fieldDict['fieldType']=='str'):
                                point[fieldDict['fieldName']]=str(tempread)
                            elif(fieldDict['fieldType']=='float'):
                                point[fieldDict['fieldName']]=float(tempread)
                    else:
                        data = re.split(',|;', line)
                        #QgsMessageLog.logMessage(str(data))
                        for fieldDict in schemaDict['fieldDictList']:
                            #QgsMessageLog.logMessage("fieldDict"+str(fieldDict))
                            #QgsMessageLog.logMessage("fieldDictValue"+str(data[fieldDict['fieldStart']+1]))
                            tempread =(data[fieldDict['fieldStart']+1]).strip()
                            if(fieldDict['fieldType']=='int'):
                                point[fieldDict['fieldName']]=int(tempread)
                            elif(fieldDict['fieldType']=='str'):
                                point[fieldDict['fieldName']]=str(tempread)
                            elif(fieldDict['fieldType']=='float'):
                                point[fieldDict['fieldName']]=float(tempread)


                    #Handling the first point of the file
                    if prevLinename == '':
                        prevLinename = point[schemaDict['LineNameFrom']]

                    #case where previous point linename is equal to point linename
                    if prevLinename!= point[schemaDict['LineNameFrom']]:
                        #find the lineDirectionFrom field should be name of the field which has pointnum else 'none'
                        if not 'none' == schemaDict['lineDirectionFrom']:
                            if pointListPerLine[0][schemaDict['lineDirectionFrom']]<pointListPerLine[-1][schemaDict['lineDirectionFrom']]:
                                direction = "Upline"
                            else:
                                direction = "Downline"
                        else:
                            direction = "Undefined"



                        if not self.dlg.firstLastOnlyCheck.isChecked():
                            for p in pointListPerLine:
                                p['Direction'] = direction
                                pointList.append(p)
                        else:
                            pointListPerLine[0]['Direction']=direction
                            pointListPerLine[-1]['Direction']=direction
                            pointList.append(pointListPerLine[0])
                            pointList.append(pointListPerLine[-1])
                        pointListPerLine.clear()
                        
                    pointListPerLine.append(point)
                    prevLinename = point[schemaDict['LineNameFrom']]
                    

            #repeated else code above for the last line in the file
            #find the lineDirectionFrom field should be name of the field which has pointnum else 'none'
            if not 'none' == schemaDict['lineDirectionFrom']:
                QgsMessageLog.logMessage("First"+str(pointListPerLine[0]))
                if pointListPerLine[0][schemaDict['lineDirectionFrom']]<pointListPerLine[-1][schemaDict['lineDirectionFrom']]:
                    direction = "Upline"
                else:
                    direction = "Downline"
            else:
                direction = "Undefined"



            if not self.dlg.firstLastOnlyCheck.isChecked():
                for p in pointListPerLine:
                    p['Direction'] = direction
                    pointList.append(p)
            else:
                pointListPerLine[0]['Direction']=direction
                pointListPerLine[-1]['Direction']=direction
                pointList.append(pointListPerLine[0])
                pointList.append(pointListPerLine[-1])
            #QgsMessageLog.logMessage("First"+str(pointListPerLine[0]))
            #QgsMessageLog.logMessage("Last"+str(pointListPerLine[-1]))

            f.close() 

        with open(QFileInfo(self.outFile).absoluteFilePath(), 'w', newline = '') as out:
            fieldnames = list(map(lambda x : x['fieldName'], schemaDict['fieldDictList']))
            fieldnames.append('Direction')
            #QgsMessageLog.logMessage(str(schemaDict['fieldDictList']))
            #QgsMessageLog.logMessage(str(fieldnames))
            writer = csv.DictWriter(out, fieldnames=fieldnames)
            writer.writeheader()
            writer.writerows(pointList)
        out.close()

        #TO-DO print an error if X field and Y field are not present in the fieldnames
        uri = "file:///"+QFileInfo(self.outFile).absoluteFilePath()+"?delimiter={}&xField={}&yField={}".format(",",schemaDict['XFrom'],schemaDict['YFrom'])
        QgsMessageLog.logMessage("\n...Importing " + uri)
        vlayer = QgsVectorLayer(uri, QFileInfo(self.outFile).baseName(), "delimitedtext")
        QgsProject.instance().addMapLayer(vlayer)
        points2pathDict = {'GROUP_FIELD' : schemaDict['LineNameFrom'], 'INPUT' : uri, 'ORDER_FIELD' : schemaDict['pointnumFrom'], 'OUTPUT' : QFileInfo(self.outFile).absolutePath()+"/"+QFileInfo(self.outFile).baseName()+"_Lines.shp" }
        QgsMessageLog.logMessage(str(points2pathDict))
        resultPath = processing.run("qgis:pointstopath", points2pathDict)
        linesLayer = QgsVectorLayer(resultPath['OUTPUT'], QFileInfo(self.outFile).baseName()+"_Lines", "ogr")
        QgsMessageLog.logMessage("Adding Lines Layer From: "+resultPath['OUTPUT'])
        QgsProject.instance().addMapLayer(linesLayer)

    def import_receiverSPS(self):
        self.dlg.textEdit.append("\n...Importing Receiver SPS")
        self.schemaPath = os.path.join(self.plugin_dir,'schema/receiverSPSSchema.txt')
        self.layerName = 'receiverSPS_csv'
        self.import_file()
        
    def import_sourceSPS(self):
        self.dlg.textEdit.append("\n...Importing Source SPS")
        self.schemaPath = os.path.join(self.plugin_dir,'schema/sourceSPSSchema.txt')
        self.layerName = 'sourceSPS_csv'
        self.import_file()

    def import_p190V(self):
        self.dlg.textEdit.append("\n...Importing P190 V Only")
        self.schemaPath = os.path.join(self.plugin_dir,'schema/P190VOnlySchema.txt')
        self.layerName = 'P190V_csv'
        self.import_file()

    def import_p190S(self):
        self.dlg.textEdit.append("\n...Importing P190 S Only")
        self.schemaPath = os.path.join(self.plugin_dir,'schema/P190SOnlySchema.txt')
        self.layerName = 'P190S_csv'
        self.import_file()

    def import_p190E(self):
        self.dlg.textEdit.append("\n...Importing p190 E Only")
        self.schemaPath = os.path.join(self.plugin_dir,'schema/P190EOnlySchema.txt')
        self.layerName = 'p190E_csv'
        self.import_file()
        
    def import_p190Custom(self):
        self.dlg.textEdit.append("\n...Importing p190 Custom as in Filter")
        self.schemaPath = os.path.join(self.plugin_dir,'schema/P190Custom.txt')
        self.customFilterRequired = True
        self.filter = self.dlg.lineEdit.text()
        self.layerName = 'p190_csv'
        self.import_file()

    def import_preplot190(self):
        self.dlg.textEdit.append("\n...Importing preplot190")
        self.schemaPath = os.path.join(self.plugin_dir,'schema/preplot190Schema.txt')
        self.layerName = 'preplot190_csv'
        self.import_file()
        
    def import_p111S(self):
        self.dlg.textEdit.append("\n...Importing p111S")
        self.schemaPath = os.path.join(self.plugin_dir,'schema/P111SOnlySchema.txt')
        self.layerName = 'p111S_csv'
        self.import_file()

    def import_p111Custom(self):
        self.dlg.textEdit.append("\n...Importing p111 Custom as in Filter")
        self.schemaPath = os.path.join(self.plugin_dir,'schema/P111Custom.txt')
        self.customFilterRequired = True
        self.filter = self.dlg.lineEdit.text()
        self.layerName = 'p111_csv'
        self.import_file()

    def import_p111SGeomerge(self):
        self.dlg.textEdit.append("\n...Importing p111S Geomerge as in Filter")
        self.schemaPath = os.path.join(self.plugin_dir,'schema/P111SGeomerge.txt')
        self.customFilterRequired = False
        self.filter = self.dlg.lineEdit.text()
        self.layerName = 'p111S_Geomerge_csv'
        self.import_file()


        

    def run(self):
        """Run method that performs all the real work"""

        # Create the dialog with elements (after translation) and keep reference
        # Only create GUI ONCE in callback, so that it will only load when the plugin is started


        # show the dialog
        self.dlg.show()
        # Run the dialog event loop
        result = self.dlg.exec_()
        # See if OK was pressed
        if result:
            if self.importChoice == self.importChoiceDict['receiverSPS']:
                self.import_receiverSPS()
            elif self.importChoice == self.importChoiceDict['sourceSPS']:
                self.import_sourceSPS()
            elif self.importChoice == self.importChoiceDict['p190V']:
                self.import_p190V()
            elif self.importChoice == self.importChoiceDict['p190S']:
                self.import_p190S()
            elif self.importChoice == self.importChoiceDict['p190E']:
                self.import_p190E()
            elif self.importChoice == self.importChoiceDict['p190_Custom']:
                self.import_p190Custom()
            elif self.importChoice == self.importChoiceDict['preplot190']:
                self.import_preplot190()
            elif self.importChoice == self.importChoiceDict['p111S']:
                self.import_p111S()
            elif self.importChoice == self.importChoiceDict['p111_Custom']:
                self.import_p111Custom()
            elif self.importChoice == self.importChoiceDict['p111S_Geomerge']:
                self.import_p111SGeomerge()
