"""Signal management module for SecInterp main dialog.

This module handles all signal connections for the dialog,
separating signal setup from the main dialog class.
"""

from typing import TYPE_CHECKING

from qgis.PyQt.QtWidgets import QDialogButtonBox

from sec_interp.logger_config import get_logger

logger = get_logger(__name__)


if TYPE_CHECKING:
    pass


class DialogSignalManager:
    """Manages all signal connections for SecInterpDialog.

    This class organizes signal connections into logical groups,
    making the dialog initialization cleaner and more maintainable.
    """

    def __init__(self, dialog: "sec_interp.gui.main_dialog.SecInterpDialog"):
        """Initialize signal manager.

        Args:
            dialog: The :class:`sec_interp.gui.main_dialog.SecInterpDialog` instance

        """
        self.dialog = dialog

    def connect_all(self) -> None:
        """Connect all signals in organized groups."""
        self._connect_button_signals()
        self._connect_preview_signals()
        self._connect_page_signals()
        self._connect_tool_signals()

    def _connect_button_signals(self) -> None:
        """Connect dialog button signals."""
        # Main dialog buttons
        # Connect standard buttons explicitly
        # We avoid connecting accepting/rejected signals directly because 'Save' button
        # might have AcceptRole by default in some Qt versions/styles, triggering accept().

        ok_btn = self.dialog.button_box.button(QDialogButtonBox.Ok)
        if ok_btn:
            ok_btn.clicked.connect(self.dialog.accept_handler)

        cancel_btn = self.dialog.button_box.button(QDialogButtonBox.Cancel)
        if cancel_btn:
            cancel_btn.clicked.connect(self.dialog.reject_handler)

        save_btn = self.dialog.button_box.button(QDialogButtonBox.Save)
        if save_btn:
            # Save should ONLY trigger export, not close the dialog
            save_btn.clicked.connect(self.dialog.export_manager.export_data)

        self.dialog.button_box.helpRequested.connect(self.dialog.open_help)

        # Clear cache button
        self.dialog.clear_cache_btn.clicked.connect(self.dialog.clear_cache_handler)

    def _connect_preview_signals(self) -> None:
        """Connect preview-related signals."""
        # Preview action buttons
        self.dialog.preview_widget.btn_preview.clicked.connect(
            self.dialog.preview_profile_handler
        )
        self.dialog.preview_widget.btn_export.clicked.connect(
            self.dialog.export_preview
        )

        # Preview layer checkboxes
        self.dialog.preview_widget.chk_topo.stateChanged.connect(
            self.dialog.update_preview_from_checkboxes
        )
        self.dialog.preview_widget.chk_geol.stateChanged.connect(
            self.dialog.update_preview_from_checkboxes
        )
        self.dialog.preview_widget.chk_struct.stateChanged.connect(
            self.dialog.update_preview_from_checkboxes
        )
        self.dialog.preview_widget.chk_drillholes.stateChanged.connect(
            self.dialog.update_preview_from_checkboxes
        )
        self.dialog.preview_widget.chk_interpretations.stateChanged.connect(
            self.dialog.update_preview_from_checkboxes
        )

        # Preview settings
        self.dialog.preview_widget.spin_max_points.valueChanged.connect(
            self.dialog.update_preview_from_checkboxes
        )
        self.dialog.preview_widget.chk_auto_lod.toggled.connect(
            self.dialog.update_preview_from_checkboxes
        )
        self.dialog.preview_widget.chk_adaptive_sampling.toggled.connect(
            self.dialog.update_preview_from_checkboxes
        )

    def _connect_page_signals(self) -> None:
        """Connect page-specific signals for state updates."""
        # Output path changes
        self.dialog.output_widget.fileChanged.connect(self.dialog.update_button_state)

        # DEM page
        self.dialog.page_dem.raster_combo.layerChanged.connect(
            self.dialog.update_button_state
        )
        self.dialog.page_dem.raster_combo.layerChanged.connect(
            self.dialog.update_preview_checkbox_states
        )

        # Section page
        self.dialog.page_section.line_combo.layerChanged.connect(
            self.dialog.update_button_state
        )
        self.dialog.page_section.line_combo.layerChanged.connect(
            self.dialog.update_preview_checkbox_states
        )

        # Data pages
        self.dialog.page_geology.dataChanged.connect(
            self.dialog.update_preview_checkbox_states
        )
        self.dialog.page_struct.dataChanged.connect(
            self.dialog.update_preview_checkbox_states
        )
        self.dialog.page_drillhole.dataChanged.connect(
            self.dialog.update_preview_checkbox_states
        )

    def _connect_tool_signals(self) -> None:
        """Connect map tool signals."""
        # Measure tool
        self.dialog.preview_widget.btn_measure.toggled.connect(
            self.dialog.toggle_measure_tool
        )

        # Interpretation tool
        self.dialog.preview_widget.btn_interpret.toggled.connect(
            self.dialog.toggle_interpretation_tool
        )

        # Finalize button with debug wrapper
        def finalize_with_log() -> None:
            """Log finalization action and trigger measurement finalization."""
            logger.info("Finalize button clicked!")
            self.dialog.tool_manager.measure_tool.finalize_measurement()

        self.dialog.preview_widget.btn_finalize.clicked.connect(finalize_with_log)

        self.dialog.tool_manager.measure_tool.measurementChanged.connect(
            self.dialog.update_measurement_display
        )
        self.dialog.tool_manager.measure_tool.measurementCleared.connect(
            lambda: self.dialog.preview_widget.results_text.clear()
        )
