"""Drillhole Data Processing Service.

This module provides services for processing and projecting drillhole data,
including collar projection, trajectory calculation, and interval interpolation.
"""

from __future__ import annotations

import contextlib
from typing import Any

from qgis.core import (
    QgsCoordinateReferenceSystem,
    QgsDistanceArea,
    QgsFeature,
    QgsFeatureRequest,
    QgsGeometry,
    QgsPointXY,
    QgsRaster,
    QgsRasterLayer,
    QgsVectorLayer,
)

from sec_interp.core import utils as scu
from sec_interp.core.exceptions import DataMissingError, GeometryError
from sec_interp.core.interfaces.drillhole_interface import IDrillholeService
from sec_interp.core.types import DrillholeTaskInput, GeologySegment
from sec_interp.logger_config import get_logger

logger = get_logger(__name__)


class DrillholeService(IDrillholeService):
    """Service for processing drillhole data."""

    def project_collars(
        self,
        collar_layer: QgsVectorLayer,
        line_geom: QgsGeometry,
        line_start: QgsPointXY,
        distance_area: QgsDistanceArea,
        buffer_width: float,
        collar_id_field: str,
        use_geometry: bool,
        collar_x_field: str,
        collar_y_field: str,
        collar_z_field: str,
        collar_depth_field: str,
        dem_layer: QgsRasterLayer | None,
        line_crs: QgsCoordinateReferenceSystem | None = None,
    ) -> list[tuple[Any, float, float, float, float]]:
        """Project collar points onto section line using spatial optimization.

        Args:
            collar_layer: Vector layer containing drillhole collars.
            line_geom: Geometry of the cross-section line.
            line_start: Start point of the section line.
            distance_area: Distance calculation object.
            buffer_width: Search buffer distance in meters.
            collar_id_field: Field name for unique drillhole ID.
            use_geometry: Whether to use feature geometry for X/Y coordinates.
            collar_x_field: Field name for X coordinate (if not using geometry).
            collar_y_field: Field name for Y coordinate (if not using geometry).
            collar_z_field: Field name for collar elevation.
            collar_depth_field: Field name for total drillhole depth.
            dem_layer: Optional DEM layer for elevation if Z field is missing/zero.
            line_crs: CRS of the section line for spatial filtering.

        Returns:
            A list of tuples (hole_id, dist_along, z, offset, total_depth).

        """
        if not collar_layer:
            raise DataMissingError("Collar layer is not provided")

        from sec_interp.core.exceptions import ValidationError

        if buffer_width <= 0:
            raise ValidationError(f"Buffer width must be positive, got {buffer_width}")

        # Validate Fields
        if (
            collar_id_field
            and collar_layer.fields().indexFromName(collar_id_field) == -1
        ):
            raise ValidationError(
                f"Field '{collar_id_field}' not found in collar layer."
            )

        if not use_geometry:
            for f_name in [collar_x_field, collar_y_field]:
                if f_name and collar_layer.fields().indexFromName(f_name) == -1:
                    raise ValidationError(
                        f"Field '{f_name}' not found in collar layer."
                    )

        projected_collars = []
        logger.info(
            f"Projecting collars from {collar_layer.name()} with buffer {buffer_width}m"
        )

        # 1. Spatial Filtering
        # Create buffer zone around section line
        try:
            line_buffer = line_geom.buffer(buffer_width, 8)
        except Exception as e:
            raise GeometryError(
                "Failed to create section line buffer", {"buffer_width": buffer_width}
            ) from e

        # Use centralized filtering utility which handles CRS transformation
        candidate_features = scu.filter_features_by_buffer(
            collar_layer, line_buffer, line_crs
        )

        if not candidate_features:
            logger.info("No collars found within buffer area.")
            return []

        for collar_feat in candidate_features:
            result = self._project_single_collar(
                collar_feat,
                line_geom,
                line_start,
                distance_area,
                buffer_width,
                collar_id_field,
                use_geometry,
                collar_x_field,
                collar_y_field,
                collar_z_field,
                collar_depth_field,
                dem_layer,
            )
            if result:
                projected_collars.append(result)

        logger.info(
            f"DrillholeService.project_collars END: Found {len(projected_collars)} collars."
        )
        return projected_collars

    def prepare_task_input(
        self,
        line_geom: QgsGeometry,
        line_start: QgsPointXY,
        line_crs: QgsCoordinateReferenceSystem,
        section_azimuth: float,
        buffer_width: float,
        # Collar Params
        collar_layer: QgsVectorLayer,
        collar_id_field: str,
        use_geometry: bool,
        collar_x_field: str,
        collar_y_field: str,
        collar_z_field: str,
        collar_depth_field: str,
        # Child Data params
        survey_layer: QgsVectorLayer,
        survey_fields: dict[str, str],
        interval_layer: QgsVectorLayer,
        interval_fields: dict[str, str],
        # Optional
        dem_layer: QgsRasterLayer | None = None,
    ) -> DrillholeTaskInput:
        """Prepare detached data for async processing."""
        # --- Level 3 Validation: Domain Guards ---
        from sec_interp.core.exceptions import ValidationError
        from sec_interp.core.validation.validators import (
            validate_positive,
            validate_range,
        )

        # 1. Parameter Validation
        validate_positive("Buffer width")(buffer_width)
        validate_range(0, 360, "Section azimuth")(section_azimuth)

        # 2. Field Existence Validation
        # Helper to check fields in a layer
        def check_field(layer: QgsVectorLayer, field_name: str, layer_name: str):
            if not field_name:
                return  # Optional or handled elsewhere?
            if layer.fields().indexFromName(field_name) == -1:
                raise ValidationError(
                    f"Field '{field_name}' not found in {layer_name}."
                )

        if collar_id_field:
            check_field(collar_layer, collar_id_field, "collar layer")

        # We don't strictly validate X/Y/Z/Depth here because they might be optional or mapped differently
        # But if provided, they should exist.
        if use_geometry is False:
            check_field(collar_layer, collar_x_field, "collar layer")
            check_field(collar_layer, collar_y_field, "collar layer")

        # --- End Validation ---

        # 1. Filter Collars (Spatial)
        # Use buffer to limit feature fetching
        try:
            line_buffer = line_geom.buffer(buffer_width, 8)
        except Exception:
            # Fallback to no spatial filter or bbox
            line_buffer = None

        collar_bbox = (
            line_buffer.boundingBox() if line_buffer else line_geom.boundingBox()
        )

        # Prepare Request
        # We need attrs+geometry
        req = QgsFeatureRequest()
        if line_buffer:
            req.setFilterRect(collar_bbox)  # Optimization

        collar_data = []
        pre_sampled_z = {}
        processed_ids = set()

        # Iterate collars and detach
        for feat in collar_layer.getFeatures(req):
            # Check rigorous spatial intersection if buffer exists
            if line_buffer and not feat.geometry().intersects(line_buffer):
                continue

            hid = feat[collar_id_field]
            processed_ids.add(hid)

            # Detach geometry
            geom_copy = QgsGeometry(feat.geometry()) if feat.hasGeometry() else None

            # Detach attributes
            attrs = dict(zip(feat.fields().names(), feat.attributes(), strict=False))

            collar_data.append({"geometry": geom_copy, "attributes": attrs, "id": hid})

            # Pre-sample Z if DEM provided (Thread safety fix)
            if dem_layer:
                pt = self._extract_point(
                    feat, use_geometry, collar_x_field, collar_y_field
                )
                if pt:
                    # Check if Z is in attributes first? Logic mirrors _get_collar_info
                    z_val = 0.0
                    try:
                        if collar_z_field:
                            z_val = float(feat[collar_z_field] or 0.0)
                    except (ValueError, TypeError):
                        z_val = 0.0

                    if z_val == 0.0:
                        sampled = self._sample_dem(dem_layer, pt)
                        pre_sampled_z[hid] = sampled

        # 2. Bulk Fetch Child Data (Sync)
        # We fetch ALL data for the identified collars
        survey_data = self._fetch_bulk_data_detached(
            survey_layer, processed_ids, survey_fields
        )
        interval_data = self._fetch_bulk_data_detached(
            interval_layer, processed_ids, interval_fields
        )

        return DrillholeTaskInput(
            line_geometry=QgsGeometry(line_geom),
            line_start=QgsPointXY(line_start),
            line_crs_authid=line_crs.authid(),
            section_azimuth=section_azimuth,
            buffer_width=buffer_width,
            collar_id_field=collar_id_field,
            use_geometry=use_geometry,
            collar_x_field=collar_x_field,
            collar_y_field=collar_y_field,
            collar_z_field=collar_z_field,
            collar_depth_field=collar_depth_field,
            collar_data=collar_data,
            survey_data=survey_data,
            interval_data=interval_data,
            pre_sampled_z=pre_sampled_z,
        )

    def process_task_data(
        self, task_input: DrillholeTaskInput, feedback: Any | None = None
    ) -> Any:
        """Process drillholes using detached data (Thread-Safe)."""
        # Reconstruct Objects
        line_crs = QgsCoordinateReferenceSystem(task_input.line_crs_authid)
        da = scu.create_distance_area(line_crs)

        geol_data_all = []
        drillhole_data_all = []  # (hid, trace2d, trace3d, proj3d, geologic_segments)

        total = len(task_input.collar_data)

        for i, c_item in enumerate(task_input.collar_data):
            if feedback and feedback.isCanceled():
                return None

            # Build logical feature from detached data to reuse existing logic?
            # Or refactor _get_collar_info to work with dicts.
            # Adaptation: Inline/Refactor _get_collar_info logic for dicts

            hid = c_item["id"]
            attrs = c_item["attributes"]
            geom = c_item["geometry"]

            # --- Extract Point ---
            pt = None
            if task_input.use_geometry and geom:
                pt = geom.asPoint()
            else:
                try:
                    x = float(attrs.get(task_input.collar_x_field, 0.0))
                    y = float(attrs.get(task_input.collar_y_field, 0.0))
                    pt = QgsPointXY(x, y)
                except (ValueError, TypeError):
                    pass

            if not pt:
                continue

            # --- Extract Z/Depth ---
            z = 0.0
            try:
                z = float(attrs.get(task_input.collar_z_field, 0.0))
            except:
                z = 0.0

            # Fallback Z
            if z == 0.0 and hid in task_input.pre_sampled_z:
                z = task_input.pre_sampled_z[hid]

            depth = 0.0
            try:
                depth = float(attrs.get(task_input.collar_depth_field, 0.0))
            except:
                depth = 0.0

            # --- Project ---
            # Logic from _project_single_collar
            collar_geom_pt = QgsGeometry.fromPointXY(pt)
            nearest_point = task_input.line_geometry.nearestPoint(
                collar_geom_pt
            ).asPoint()

            offset = da.measureLine(pt, nearest_point)
            if offset > task_input.buffer_width:
                continue

            # --- Full Processing ---
            surveys = task_input.survey_data.get(hid, [])
            intervals = task_input.interval_data.get(hid, [])

            hole_geol, hole_tuple = self._process_single_hole(
                hole_id=hid,
                collar_point=pt,
                collar_z=z,
                given_depth=depth,
                survey_data=surveys,
                intervals=intervals,
                line_geom=task_input.line_geometry,
                line_start=task_input.line_start,
                distance_area=da,
                buffer_width=task_input.buffer_width,
                section_azimuth=task_input.section_azimuth,
            )

            if hole_geol:
                geol_data_all.extend(hole_geol)
            drillhole_data_all.append(hole_tuple)

            if feedback:
                feedback.setProgress((i / total) * 100)

        # Result Structure matches DrillholeService.process_intervals return
        return geol_data_all, drillhole_data_all

    def process_intervals(
        self,
        collar_points: list[tuple],
        collar_layer: QgsVectorLayer,
        survey_layer: QgsVectorLayer,
        interval_layer: QgsVectorLayer,
        collar_id_field: str,
        use_geometry: bool,
        collar_x_field: str,
        collar_y_field: str,
        line_geom: QgsGeometry,
        line_start: QgsPointXY,
        distance_area: QgsDistanceArea,
        buffer_width: float,
        section_azimuth: float,
        survey_fields: dict[str, str],
        interval_fields: dict[str, str],
    ) -> tuple[
        list[GeologySegment],
        list[
            tuple[
                Any,
                list[tuple[float, float]],
                list[tuple[float, float, float]],
                list[tuple[float, float, float]],
                list[GeologySegment],
            ]
        ],
    ]:
        """Generate drillhole trace and interval data and project onto the section.

        Args:
            collar_points: List of projected collar tuples from `project_collars`.
            collar_layer: The collar vector layer.
            survey_layer: The survey vector layer.
            interval_layer: The interval/geology vector layer.
            collar_id_field: Field name for hole ID in collar layer.
            use_geometry: Use geometry for collar coordinates.
            collar_x_field: Field name for X in collar layer.
            collar_y_field: Field name for Y in collar layer.
            line_geom: Section line geometry.
            line_start: Section line start point.
            distance_area: Distance calculation object.
            buffer_width: Section buffer width in meters.
            section_azimuth: Azimuth of the section line.
            survey_fields: Mapping of survey field roles to field names.
            interval_fields: Mapping of interval field roles to field names.

        Returns:
            A tuple of (geol_data, drillhole_data).

        """
        geol_data, drillhole_data = [], []

        # 1. Build collar coordinate map
        collar_coords = self._build_collar_coord_map(
            collar_layer, collar_id_field, use_geometry, collar_x_field, collar_y_field
        )

        # 2. Bulk fetch survey and interval data for all relevant holes
        hole_ids = {cp[0] for cp in collar_points}
        surveys_map = self._fetch_bulk_data(survey_layer, hole_ids, survey_fields)
        intervals_map = self._fetch_bulk_data(interval_layer, hole_ids, interval_fields)

        for hole_id, _dist, collar_z, _off, given_depth in collar_points:
            collar_point = collar_coords.get(hole_id)
            if not collar_point:
                continue

            self._safe_process_single_hole(
                hole_id,
                collar_point,
                collar_z,
                given_depth,
                surveys_map,
                intervals_map,
                line_geom,
                line_start,
                distance_area,
                buffer_width,
                section_azimuth,
                geol_data,
                drillhole_data,
            )

        return geol_data, drillhole_data

    def _fetch_bulk_data(
        self, layer: QgsVectorLayer, hole_ids: set[Any], fields: dict[str, str]
    ) -> dict[Any, list[tuple[Any, ...]]]:
        """Fetch data for multiple holes in a single pass."""
        if not layer or not layer.isValid():
            return {}

        id_f = fields.get("id")
        if not id_f:
            return {}

        is_survey = "depth" in fields
        required = ["depth", "azim", "incl"] if is_survey else ["from", "to", "lith"]

        # Validate fields
        for field_key in required:
            if not fields.get(field_key):
                return {}

        result_map: dict[Any, list[tuple]] = {}
        if not hole_ids:
            return {}

        ids_str = ", ".join([f"'{hid!s}'" for hid in hole_ids])
        request = QgsFeatureRequest().setFilterExpression(f'"{id_f}" IN ({ids_str})')

        for feat in layer.getFeatures(request):
            hole_id = feat[id_f]
            data = self._extract_data_tuple(feat, fields, is_survey)
            if data:
                if hole_id not in result_map:
                    result_map[hole_id] = []
                result_map[hole_id].append(data)

        # Sort surveys by depth
        if is_survey:
            for h_id in result_map:
                result_map[h_id].sort(key=lambda x: x[0])

        return result_map

    def _fetch_bulk_data_detached(
        self, layer: QgsVectorLayer, hole_ids: set[Any], fields: dict[str, str]
    ) -> dict[Any, list[tuple]]:
        """Fetch data into pure python structures (no QgsFeature dependency)."""
        # This is essentially the same as _fetch_bulk_data but guarantees primitive types
        # in the returned list, which _extract_data_tuple already does.
        # So we can reuse the logic but ensures it runs on Main Thread in prepare()
        return self._fetch_bulk_data(layer, hole_ids, fields)

    def _process_single_hole(
        self,
        hole_id: Any,
        collar_point: QgsPointXY,
        collar_z: float,
        given_depth: float,
        survey_data: list[tuple[float, float, float]],
        intervals: list[tuple[float, float, str]],
        line_geom: QgsGeometry,
        line_start: QgsPointXY,
        distance_area: QgsDistanceArea,
        buffer_width: float,
        section_azimuth: float,
    ) -> tuple[
        list[GeologySegment],
        tuple[
            Any,
            list[tuple[float, float]],
            list[tuple[float, float, float]],
            list[tuple[float, float, float]],
            list[GeologySegment],
        ],
    ]:
        """Process a single drillhole's trajectory and intervals.

        Returns:
            A tuple of (hole_geol_data, drillhole_tuple).

        """
        # 1. Determine Final Depth
        max_s_depth = max([s[0] for s in survey_data]) if survey_data else 0.0
        max_i_depth = max([i[1] for i in intervals]) if intervals else 0.0
        final_depth = max(given_depth, max_s_depth, max_i_depth)

        # 2. Trajectory and Projection
        trajectory = scu.calculate_drillhole_trajectory(
            collar_point,
            collar_z,
            survey_data,
            section_azimuth,
            total_depth=final_depth,
        )
        projected_traj = scu.project_trajectory_to_section(
            trajectory, line_geom, line_start, distance_area
        )

        # 3. Interpolate Intervals
        hole_geol_data = self._interpolate_hole_intervals(
            projected_traj, intervals, buffer_width
        )

        # 4. Store trace (2D and 3D)
        traj_points_2d = [(p[4], p[3]) for p in projected_traj]
        traj_points_3d = [(p[1], p[2], p[3]) for p in projected_traj]
        traj_points_3d_proj = [(p[6], p[7], p[3]) for p in projected_traj]

        return hole_geol_data, (
            hole_id,
            traj_points_2d,
            traj_points_3d,
            traj_points_3d_proj,
            hole_geol_data,
        )

    def _build_collar_coord_map(
        self,
        layer: QgsVectorLayer,
        id_field: str,
        use_geom: bool,
        x_field: str,
        y_field: str,
    ) -> dict[Any, QgsPointXY]:
        """Build a lookup map for collar coordinates.

        Args:
            layer: The collar vector layer.
            id_field: Field name for hole ID.
            use_geom: Whether to use feature geometry.
            x_field: Field name for X.
            y_field: Field name for Y.

        Returns:
            A dictionary mapping hole_id to QgsPointXY.

        """
        if not layer or not id_field:
            return {}
        coords = {}

        # Fetch only necessary attributes and geometry
        if use_geom:
            # Need geometry and id_field only
            request = QgsFeatureRequest().setSubsetOfAttributes(
                [id_field], layer.fields()
            )
        else:
            # Need id_field, x_field, y_field but no geometry
            request = QgsFeatureRequest().setSubsetOfAttributes(
                [id_field, x_field, y_field], layer.fields()
            )
            request.setFlags(QgsFeatureRequest.NoGeometry)

        for feat in layer.getFeatures(request):
            hole_id = feat[id_field]
            pt = self._extract_point(feat, use_geom, x_field, y_field)
            if pt:
                coords[hole_id] = pt
        return coords

    def _extract_point(
        self, feat: QgsFeature, use_geom: bool, x_f: str, y_f: str
    ) -> QgsPointXY | None:
        """Extract point from feature geometry or fields."""
        if use_geom:
            geom = feat.geometry()
            if geom:
                pt = geom.asPoint()
                if pt.x() != 0 or pt.y() != 0:
                    return pt
        else:
            try:
                x, y = float(feat[x_f]), float(feat[y_f])
                if x != 0 or y != 0:
                    return QgsPointXY(x, y)
            except (ValueError, TypeError, KeyError):
                pass
        return None

    def _get_survey_data(
        self, layer: QgsVectorLayer, hole_id: Any, fields: dict[str, str]
    ) -> list[tuple[Any, ...]]:
        """Legacy support - redirected to bulk fetch if needed."""
        res = self._fetch_bulk_data(layer, {hole_id}, fields)
        return res.get(hole_id, [])

    def _get_interval_data(
        self, layer: QgsVectorLayer, hole_id: Any, fields: dict[str, str]
    ) -> list[tuple[Any, ...]]:
        """Legacy support - redirected to bulk fetch if needed."""
        res = self._fetch_bulk_data(layer, {hole_id}, fields)
        return res.get(hole_id, [])

    def _interpolate_hole_intervals(
        self,
        traj: list[tuple[float, float, float, float, float]],
        intervals: list[tuple[float, float, str]],
        buffer_width: float,
    ) -> list[GeologySegment]:
        """Interpolate intervals along a trajectory and return GeologySegments.

        Args:
            traj: The projected trajectory tuples.
            intervals: List of (from, to, lith) tuples.
            buffer_width: Section buffer width.

        Returns:
            A list of GeologySegment objects.

        """
        if not intervals:
            return []

        rich_intervals = [
            (fd, td, {"unit": lith, "from": fd, "to": td}) for fd, td, lith in intervals
        ]
        # Scu returns (attr, points_2d, points_3d, points_3d_proj)
        tuples = scu.interpolate_intervals_on_trajectory(
            traj, rich_intervals, buffer_width
        )

        segments = []
        for attr, points_2d, points_3d, points_3d_proj in tuples:
            segments.append(
                GeologySegment(
                    unit_name=str(attr.get("unit", "Unknown")),
                    geometry=None,
                    attributes=attr,
                    points=points_2d,
                    points_3d=points_3d,
                    points_3d_projected=points_3d_proj,
                )
            )
        return segments

    def _project_single_collar(
        self,
        collar_feat: QgsFeature,
        line_geom: QgsGeometry,
        line_start: QgsPointXY,
        distance_area: QgsDistanceArea,
        buffer_width: float,
        collar_id_field: str,
        use_geometry: bool,
        collar_x_field: str,
        collar_y_field: str,
        collar_z_field: str,
        collar_depth_field: str,
        dem_layer: QgsRasterLayer | None,
    ) -> tuple[Any, float, float, float, float] | None:
        """Process and project a single collar feature.

        Args:
            collar_feat: The collar feature to process.
            line_geom: Section line geometry.
            line_start: Start point of the section line.
            distance_area: Distance calculation object.
            buffer_width: Buffer width for filtering.
            collar_id_field: Field for ID.
            use_geometry: Whether to use geometry for coords.
            collar_x_field: Field for X.
            collar_y_field: Field for Y.
            collar_z_field: Field for Z.
            collar_depth_field: Field for depth.
            dem_layer: Optional DEM layer.

        Returns:
            Tuple of (hole_id, dist_along, z, offset, total_depth) or None.

        """
        # 1. Get Collar Info
        collar_info = self._get_collar_info(
            collar_feat,
            collar_id_field,
            use_geometry,
            collar_x_field,
            collar_y_field,
            collar_z_field,
            collar_depth_field,
            dem_layer,
        )
        if not collar_info:
            return None

        hole_id, collar_point, z, depth = collar_info

        # 2. Project to section line
        collar_geom_pt = QgsGeometry.fromPointXY(collar_point)
        nearest_point = line_geom.nearestPoint(collar_geom_pt).asPoint()

        # Calculate distances
        dist_along = distance_area.measureLine(line_start, nearest_point)
        offset = distance_area.measureLine(collar_point, nearest_point)

        # Check if within buffer
        if offset <= buffer_width:
            return (hole_id, dist_along, z, offset, depth)

        return None

    def _get_collar_info(
        self,
        feat: QgsFeature,
        id_field: str,
        use_geom: bool,
        x_field: str,
        y_field: str,
        z_field: str,
        depth_field: str,
        dem_layer: QgsRasterLayer | None = None,
    ) -> tuple[Any, QgsPointXY, float, float] | None:
        """Extract collar ID, coordinate, Z and depth from a feature.

        Args:
            feat: The collar feature to parse.
            id_field: Field name for hole ID.
            use_geom: Whether to use geometry for coordinates.
            x_field: Field name for X coordinate.
            y_field: Field name for Y coordinate.
            z_field: Field name for Z coordinate.
            depth_field: Field name for total depth.
            dem_layer: Optional DEM layer for fallback elevation.

        Returns:
            A tuple of (hole_id, point, elevation, total_depth) or None if invalid.

        """
        if not id_field:
            return None
        hole_id = feat[id_field]

        # Point extraction
        pt = self._extract_point(feat, use_geom, x_field, y_field)
        if not pt:
            return None

        # Z logic
        z = 0.0
        if z_field:
            with contextlib.suppress(ValueError, TypeError):
                z = float(feat[z_field])

        if z == 0.0 and dem_layer:
            z = self._sample_dem(dem_layer, pt)

        # Depth
        depth = 0.0
        if depth_field:
            with contextlib.suppress(ValueError, TypeError):
                depth = float(feat[depth_field])

        return hole_id, pt, z, depth

    def _sample_dem(self, dem_layer: QgsRasterLayer, pt: QgsPointXY) -> float:
        """Sample elevation at point from DEM."""
        ident = dem_layer.dataProvider().identify(pt, QgsRaster.IdentifyFormatValue)
        if ident.isValid():
            val = ident.results().get(1)
            if val is not None:
                return float(val)
        return 0.0

    def _safe_process_single_hole(
        self,
        hole_id: Any,
        collar_point: QgsPointXY,
        collar_z: float,
        given_depth: float,
        surveys_map: dict[Any, list[tuple[float, float, float]]],
        intervals_map: dict[Any, list[tuple[float, float, str]]],
        line_geom: QgsGeometry,
        line_start: QgsPointXY,
        distance_area: QgsDistanceArea,
        buffer_width: float,
        section_azimuth: float,
        geol_data: list[GeologySegment],
        drillhole_data: list[
            tuple[Any, list[tuple[float, float]], list[GeologySegment]]
        ],
    ) -> None:
        """Safely process a single hole with error logging."""
        try:
            hole_geol, hole_drill = self._process_single_hole(
                hole_id=hole_id,
                collar_point=collar_point,
                collar_z=collar_z,
                given_depth=given_depth,
                survey_data=surveys_map.get(hole_id, []),
                intervals=intervals_map.get(hole_id, []),
                line_geom=line_geom,
                line_start=line_start,
                distance_area=distance_area,
                buffer_width=buffer_width,
                section_azimuth=section_azimuth,
            )

            if hole_geol:
                geol_data.extend(hole_geol)
            drillhole_data.append(hole_drill)
        except Exception as e:
            logger.exception(
                f"Failed to process hole {hole_id}: {type(e).__name__}: {e}"
            )
            import traceback

            logger.exception(traceback.format_exc())
            raise

    def _extract_data_tuple(
        self, feat: QgsFeature, fields: dict[str, str], is_survey: bool
    ) -> tuple[float, float, Any] | None:
        """Extract a data tuple from a feature based on its role."""
        try:
            if is_survey:
                return (
                    float(feat[fields["depth"]]),
                    float(feat[fields["azim"]]),
                    float(feat[fields["incl"]]),
                )
            else:
                return (
                    float(feat[fields["from"]]),
                    float(feat[fields["to"]]),
                    str(feat[fields["lith"]]),
                )
        except (ValueError, TypeError, KeyError):
            return None
