# /***************************************************************************
# SecInterpDialog
#                                 A QGIS plugin
# Data extraction for geological interpretation
# Generated by Plugin Builder: http://g-sherman.github.io/Qgis-Plugin-Builder/
#                             -------------------
#        begin                : 2025-11-15
#        git sha              : $Format:%H$
#        copyright            : (C) 2025 by Juan M Bernales
#        email                : juanbernales@gmail.com
# ***************************************************************************/
#
# /***************************************************************************
# *                                                                         *
# *   This program is free software; you can redistribute it and/or modify  *
# *   it under the terms of the GNU General Public License as published by  *
# *   the Free Software Foundation; either version 2 of the License, or     *
# *   (at your option) any later version.                                   *
# *                                                                         *
# ***************************************************************************/

"""Main Dialog Module.

Contains the SecInterpDialog class which is the primary UI for the plugin.
"""

import json
import traceback
from pathlib import Path
from typing import Any

from qgis.core import (
    Qgis,
    QgsPointXY,
    QgsProject,
)
from qgis.PyQt.QtCore import QUrl
from qgis.PyQt.QtGui import QDesktopServices
from qgis.PyQt.QtWidgets import (
    QDialog,
    QDialogButtonBox,
    QPushButton,
)

from sec_interp.core.exceptions import SecInterpError
from sec_interp.core.types import InterpretationPolygon
from sec_interp.gui.utils import show_user_message
from sec_interp.logger_config import get_logger


class _NoOpMessageBar:
    """Safe no-op messagebar when iface is not available."""

    def pushMessage(self, *_args, **_kwargs):
        """No-op implementation of pushMessage."""
        return None


from .legend_widget import LegendWidget

logger = get_logger(__name__)
from .main_dialog_cache_handler import CacheHandler
from .main_dialog_data import DialogDataAggregator
from .main_dialog_export import ExportManager
from .main_dialog_preview import PreviewManager
from .main_dialog_settings import DialogSettingsManager
from .main_dialog_signals import DialogSignalManager
from .main_dialog_status import DialogStatusManager
from .main_dialog_tools import DialogToolManager, NavigationManager
from .main_dialog_utils import DialogEntityManager
from .main_dialog_validation import DialogValidator
from .preview_layer_factory import PreviewLayerFactory
from .ui.main_window import SecInterpMainWindow


class SecInterpDialog(SecInterpMainWindow):
    """Dialog for the SecInterp QGIS plugin.

    This dialog provides the user interface and helper methods to populate
    combo boxes with layers from the current QGIS project (raster and vector
    layers filtered by geometry type). It also exposes the interface and
    plugin instance for interaction with the host application.

    Attributes:
        iface (QgsInterface): The QGIS interface instance.
        plugin_instance (SecInterp): The plugin instance that created this dialog.
        messagebar (QgsMessageBar): The message bar widget for notifications.

    """

    def __init__(self, iface=None, plugin_instance=None, parent=None):
        """Initialize the dialog."""
        # Initialize the base class which sets up the programmatic UI
        super().__init__(iface, parent)

        self.iface = iface
        self.plugin_instance = plugin_instance
        self.project = QgsProject.instance()

        # Provide a safe, no-op messagebar when iface is not available (tests)
        if self.iface is None:
            self.messagebar = _NoOpMessageBar()
        else:
            self.messagebar = self.iface.messageBar()

        # Initialize manager instances
        self._init_managers()

        # Create legend widget
        self.legend_widget = LegendWidget(self.preview_widget.canvas)

        # Store current preview data
        self.current_topo_data = None
        self.current_geol_data = None
        self.current_struct_data = None
        self.current_canvas = None
        self.current_layers = []
        self.interpretations = []  # Store InterpretationPolygon objects
        self._load_interpretations()

        # Add cache and reset buttons
        self.clear_cache_btn = QPushButton(self.tr("Clear Cache"))
        self.clear_cache_btn.setToolTip(self.tr("Clear cached data to force re-processing."))
        self.button_box.addButton(self.clear_cache_btn, QDialogButtonBox.ActionRole)

        self.reset_defaults_btn = QPushButton(self.tr("Reset Defaults"))
        self.reset_defaults_btn.setToolTip(self.tr("Reset all inputs to their default values."))
        self.button_box.addButton(self.reset_defaults_btn, QDialogButtonBox.ActionRole)

        # Initialize map tools via tool_manager
        self.tool_manager.initialize_tools()

        # Connect all signals
        self.signal_manager = DialogSignalManager(self)
        self.signal_manager.connect_all()

        # Connect extra tool buttons
        self.clear_cache_btn.clicked.connect(self.clear_cache_handler)
        self.reset_defaults_btn.clicked.connect(self.reset_defaults_handler)

        # Initial state update
        # Initial state update
        self.status_manager.update_all()
        self.settings_manager.load_settings()

        # Flag to control saving settings on close
        self._save_on_close = True

    def _init_managers(self):
        """Initialize all manager instances."""
        from sec_interp.core.services.preview_service import PreviewService

        self.validator = DialogValidator(self)
        self.preview_manager = PreviewManager(self, PreviewService(self.plugin_instance.controller))
        self.export_manager = ExportManager(self)
        self.cache_handler = CacheHandler(self)
        self.data_aggregator = DialogDataAggregator(self)
        self.settings_manager = DialogSettingsManager(self)
        self.status_manager = DialogStatusManager(self)
        self.status_manager.setup_indicators()
        self.tool_manager = DialogToolManager(self)
        self.navigation_manager = NavigationManager(self)
        self.layer_factory = PreviewLayerFactory()

    def handle_error(self, error: Exception, title: str = "Error"):
        """Centralized error handling for the dialog.

        Args:
            error: The exception to handle.
            title: Title for the error message box.

        """
        if isinstance(error, SecInterpError):
            msg = str(error)
            logger.warning(f"{title}: {msg} - Details: {error.details}")
            show_user_message(self, title, msg, level="warning")
        else:
            msg = self.tr("An unexpected error occurred: {}").format(error)
            details = traceback.format_exc()
            logger.error(f"{title}: {msg}\n{details}")
            show_user_message(
                self,
                title,
                self.tr("{}\n\nPlease check the logs for details.").format(msg),
                level="critical",
            )

    def wheelEvent(self, event: Any) -> None:
        """Handle mouse wheel for zooming in preview via navigation_manager."""
        if self.navigation_manager.handle_wheel_event(event):
            return
        super().wheelEvent(event)

    def closeEvent(self, event: Any) -> None:
        """Handle dialog close event to clean up resources."""
        if self._save_on_close:
            self.settings_manager.save_settings()

        logger.info("Closing dialog, cleaning up resources...")
        self._save_interpretations()
        self.preview_manager.cleanup()
        super().closeEvent(event)

    def open_help(self):
        """Open the help file in the default browser."""
        # Fix: help is at project root, main_dialog is in gui/
        help_file = Path(__file__).parent.parent / "help" / "html" / "index.html"
        if help_file.exists():
            QDesktopServices.openUrl(QUrl.fromLocalFile(str(help_file)))
        else:
            self.messagebar.pushMessage(
                self.tr("Error"),
                self.tr("Help file not found. Please run 'make doc' to generate it."),
                level=Qgis.Warning,
            )

    def toggle_measure_tool(self, checked: bool) -> None:
        """Toggle measurement tool via tool_manager."""
        self.tool_manager.toggle_measure_tool(checked)

    def update_measurement_display(self, metrics: dict[str, Any]) -> None:
        """Display measurement results from multi-point tool via tool_manager."""
        self.tool_manager.update_measurement_display(metrics)

    def toggle_interpretation_tool(self, checked: bool) -> None:
        """Toggle interpretation tool via tool_manager."""
        if checked:
            # Deactivate measure tool if active
            self.preview_widget.btn_measure.setChecked(False)
        self.tool_manager.toggle_interpretation_tool(checked)

    def on_interpretation_finished(self, interpretation: InterpretationPolygon) -> None:
        """Handle finalized interpretation polygon.

        Args:
            interpretation: InterpretationPolygon object from the tool

        """
        from sec_interp.logger_config import log_critical_operation

        from .dialogs.interpretation_properties_dialog import (
            InterpretationPropertiesDialog,
        )

        log_critical_operation(
            logger,
            "on_interpretation_finished",
            polygon_id=interpretation.id,
            vertices=len(interpretation.vertices_2d),
        )

        # 1. Prepare for inheritance
        interp_config = self.page_interpretation.get_data()

        # Try to inherit attributes if enabled
        if interp_config.get("inherit_geology") or interp_config.get("inherit_drillholes"):
            self._apply_attribute_inheritance(interpretation, interp_config)

        # 2. Show properties dialog
        dlg = InterpretationPropertiesDialog(
            interpretation, interp_config.get("custom_fields"), self
        )

        if dlg.exec_() != QDialog.Accepted:
            logger.info(f"Interpretation canceled by user: {interpretation.id}")
            # Deactivate interpretation tool anyway
            self.preview_widget.btn_interpret.setChecked(False)
            return

        # Store interpretation
        self.interpretations.append(interpretation)
        self._save_interpretations()
        logger.info(
            f"Interpretation polygon added: {interpretation.id} "
            f"({len(interpretation.vertices_2d)} vertices)"
        )

        # Display feedback in results area
        msg = (
            f"<b>Interpretación Finalizada</b><br>"
            f"<b>Nombre:</b> {interpretation.name}<br>"
            f"<b>Vértices:</b> {len(interpretation.vertices_2d)}<br>"
            f"<b>ID:</b> {interpretation.id[:8]}..."
        )
        self.preview_widget.results_text.setHtml(msg)
        self.preview_widget.results_group.setCollapsed(False)

        # Deactivate interpretation tool
        self.preview_widget.btn_interpret.setChecked(False)

        # Update preview to show the new polygon
        self.update_preview_from_checkboxes()

    def _apply_attribute_inheritance(
        self, interpretation: InterpretationPolygon, config: dict[str, Any]
    ) -> None:
        """Inherit attributes from nearest geology or drillhole data."""
        from qgis.core import QgsGeometry

        # Use centroid or first vertex as reference point
        poly_geom = QgsGeometry.fromPolygonXY(
            [[QgsPointXY(x, y) for x, y in interpretation.vertices_2d]]
        )
        ref_point = poly_geom.centroid().asPoint()

        best_match = None
        min_dist = float("inf")

        # 1. Check Geology Data
        if config.get("inherit_geology") and self.preview_manager.cached_data.get("geol"):
            for segment in self.preview_manager.cached_data["geol"]:
                # Check min distance to any point in the segment
                if not segment.points:
                    continue

                # Find closest point in this segment
                seg_min_dist = float("inf")
                for p_dist, p_elev in segment.points:
                    d = ref_point.distance(QgsPointXY(p_dist, p_elev))
                    seg_min_dist = min(d, seg_min_dist)

                if seg_min_dist < min_dist:
                    min_dist = seg_min_dist
                    best_match = {
                        "name": segment.unit_name,
                        "type": "geology",
                        "attrs": segment.attributes,
                    }

        # 2. Check Drillhole Data (Intervals)
        if config.get("inherit_drillholes") and self.preview_manager.cached_data.get("drillhole"):
            for dh in self.preview_manager.cached_data["drillhole"]:
                # Handle both tuple (id, trace, intervals) and object (with .intervals)
                extracted_intervals = []
                if isinstance(dh, tuple) and len(dh) >= 3:
                    extracted_intervals = dh[2]
                elif hasattr(dh, "intervals"):
                    extracted_intervals = dh.intervals

                if not extracted_intervals:
                    continue

                for interval in extracted_intervals:
                    if not interval.points:
                        continue

                    # Find closest point in this interval
                    int_min_dist = float("inf")
                    for p_dist, p_elev in interval.points:
                        d = ref_point.distance(QgsPointXY(p_dist, p_elev))
                        int_min_dist = min(d, int_min_dist)

                    if int_min_dist < min_dist:
                        min_dist = int_min_dist
                        # Polymorphic access: DrillholeInterval uses 'rock_unit', GeologySegment uses 'unit_name'
                        unit_name = getattr(
                            interval,
                            "rock_unit",
                            getattr(interval, "unit_name", "Unknown"),
                        )

                        best_match = {
                            "name": unit_name,
                            "type": "drillhole",
                            "attrs": interval.attributes,
                        }

        if best_match:
            logger.info(f"Inherited attributes from {best_match['type']}: {best_match['name']}")
            interpretation.name = best_match["name"]
            interpretation.type = best_match["type"]
            # Copy all attributes from source
            if best_match["attrs"]:
                interpretation.attributes.update(best_match["attrs"])
            # Update color to match unit color if possible
            interpretation.color = self.layer_factory.get_color_for_unit(best_match["name"]).name()

    def update_preview_checkbox_states(self):
        """Enable or disable preview checkboxes via status_manager."""
        self.status_manager.update_preview_checkbox_states()

    def update_button_state(self):
        """Enable or disable buttons via status_manager."""
        self.status_manager.update_button_state()

    def get_selected_values(self):
        """Get the selected values from the dialog.

        Returns:
            Dictionary with all dialog values in legacy flat format

        """
        return self.data_aggregator.get_all_values()

    def get_preview_options(self):
        """Return the state of preview layer checkboxes.

        Returns:
            dict: Keys 'show_topo', 'show_geol', 'show_struct' with boolean values.

        """
        return {
            "show_topo": bool(self.preview_widget.chk_topo.isChecked()),
            "show_geol": bool(self.preview_widget.chk_geol.isChecked()),
            "show_struct": bool(self.preview_widget.chk_struct.isChecked()),
            "show_drillholes": bool(self.preview_widget.chk_drillholes.isChecked()),
            "show_interpretations": bool(self.preview_widget.chk_interpretations.isChecked()),
            "max_points": self.preview_widget.spin_max_points.value(),
            "auto_lod": self.preview_widget.chk_auto_lod.isChecked(),
            "use_adaptive_sampling": bool(self.preview_widget.chk_adaptive_sampling.isChecked()),
        }

    def update_preview_from_checkboxes(self):
        """Update preview when checkboxes change.

        This method delegates to PreviewManager for preview updates.
        """
        self.preview_manager.update_from_checkboxes()

    def preview_profile_handler(self):
        """Generate a quick preview with topographic, geological, and structural data.

        This method delegates to PreviewManager for preview generation.
        """
        success, message = self.preview_manager.generate_preview()
        if success:
            # Auto-save settings on successful preview
            self.settings_manager.save_settings()

        if not success and message:
            self.messagebar.pushMessage("Preview Error", message, level=2)

    def export_preview(self):
        """Export the current preview to a file using ExportManager."""
        self.export_manager.export_preview()

    def accept_handler(self):
        """Handle the accept button click event."""
        # Proactively save settings as UI state, even if validation fails
        self.settings_manager.save_settings()

        # When running without a QGIS iface (tests), skip strict validation
        if self.iface is None:
            self.accept()
            return

        if not self.validate_inputs():
            return

        self._save_interpretations()
        self.accept()

    def reject_handler(self):
        """Handle the reject button click event."""
        self._save_on_close = False
        self.close()

    def validate_inputs(self):
        """Validate the inputs from the dialog.

        This method delegates to DialogValidator for input validation.
        """
        is_valid, error_message = self.validator.validate_inputs()
        if not is_valid:
            show_user_message(self, "Validation Error", error_message)
        return is_valid

    def clear_cache_handler(self):
        """Clear cached data and notify user."""
        if hasattr(self, "plugin_instance") and self.plugin_instance:
            self.plugin_instance.controller.data_cache.clear()
            self.preview_widget.results_text.append(
                self.tr("✓ Cache cleared - next preview will re-process data")
            )
            # context specific usage
            logger = get_logger(__name__)
            logger.info("Cache cleared by user")
        else:
            self.preview_widget.results_text.append(self.tr("⚠ Cache not available"))

    def reset_defaults_handler(self):
        """Reset all dialog inputs and notify user."""
        self.settings_manager.reset_to_defaults()
        self.preview_widget.results_text.append(self.tr("✓ Form reset to default values"))
        logger.info("Dialog reset to defaults by user")

    def _populate_field_combobox(self, source_combobox: Any, target_combobox: Any) -> None:
        """Populate a combobox with field names."""
        DialogEntityManager.populate_field_combobox(source_combobox, target_combobox)

    def get_layer_names_by_type(self, layer_type) -> list[str]:
        """Get layer names by type."""
        return DialogEntityManager.get_layer_names_by_type(layer_type)

    def get_layer_names_by_geometry(self, geometry_type) -> list[str]:
        """Get layer names by geometry."""
        return DialogEntityManager.get_layer_names_by_geometry(geometry_type)

    def getThemeIcon(self, name: str) -> Any:
        """Get a theme icon via DialogEntityManager."""
        return DialogEntityManager.get_theme_icon(name)

    def _load_user_settings(self):
        """Load user settings via settings_manager."""
        self.settings_manager.load_settings()

    def _save_user_settings(self):
        """Save user settings via settings_manager."""
        self.settings_manager.save_settings()

    def _save_interpretations(self):
        """Save interpretations to the QGIS project."""
        if not self.project:
            return

        data = []
        for interp in self.interpretations:
            data.append(
                {
                    "id": interp.id,
                    "name": interp.name,
                    "type": interp.type,
                    "vertices_2d": interp.vertices_2d,
                    "attributes": interp.attributes,
                    "color": interp.color,
                    "created_at": interp.created_at,
                }
            )

        def json_serial(obj):
            """JSON serializer for objects not serializable by default json code."""
            if hasattr(obj, "isNull"):  # Handle QVariant (PyQt5/PyQGIS)
                if obj.isNull():
                    return None
                return obj.value()
            return str(obj)

        json_data = json.dumps(data, default=json_serial)
        self.project.writeEntry("SecInterp", "interpretations", json_data)
        logger.debug(f"Saved {len(data)} interpretations to project")

    def _load_interpretations(self):
        """Load interpretations from the QGIS project."""
        if not self.project:
            return

        json_data, ok = self.project.readEntry("SecInterp", "interpretations", "[]")
        if not ok or not json_data:
            return

        try:
            data = json.loads(json_data)
            self.interpretations = []
            for item in data:
                interp = InterpretationPolygon(
                    id=item.get("id", ""),
                    name=item.get("name", ""),
                    type=item.get("type", "lithology"),
                    vertices_2d=[tuple(v) for v in item.get("vertices_2d", [])],
                    attributes=item.get("attributes", {}),
                    color=item.get("color", "#FF0000"),
                    created_at=item.get("created_at", ""),
                )
                self.interpretations.append(interp)
            logger.info(f"Loaded {len(self.interpretations)} interpretations from project")
        except Exception:
            logger.exception("Failed to load interpretations")
