# /***************************************************************************
# SecInterpDialog
#                                 A QGIS plugin
# Data extraction for geological interpretation
# Generated by Plugin Builder: http://g-sherman.github.io/Qgis-Plugin-Builder/
#                             -------------------
#        begin                : 2025-11-15
#        git sha              : $Format:%H$
#        copyright            : (C) 2025 by Juan M Bernales
#        email                : juanbernales@gmail.com
# ***************************************************************************/
#
# /***************************************************************************
# *                                                                         *
# *   This program is free software; you can redistribute it and/or modify  *
# *   it under the terms of the GNU General Public License as published by  *
# *   the Free Software Foundation; either version 2 of the License, or     *
# *   (at your option) any later version.                                   *
# *                                                                         *
# ***************************************************************************/

"""Main Dialog Module.

Contains the SecInterpDialog class which is the primary UI for the plugin.
"""

from pathlib import Path
import tempfile
import traceback

from qgis.core import (
    Qgis,
    QgsApplication,
    QgsCoordinateTransform,
    QgsMapLayer,
    QgsMapLayerProxyModel,
    QgsMapSettings,
    QgsPointXY,
    QgsProject,
    QgsSettings,
    QgsUnitTypes,
    QgsWkbTypes,
)
from qgis.gui import QgsFileWidget, QgsMapCanvas
from qgis.PyQt.QtCore import QSize, QUrl, QVariant
from qgis.PyQt.QtGui import QColor, QDesktopServices
from qgis.PyQt.QtWidgets import (
    QApplication,
    QDialog,
    QDialogButtonBox,
    QFileDialog,
    QLabel,
    QPushButton,
    QStyle,
)

from sec_interp.core import utils as scu
from sec_interp.core import validation as vu
from sec_interp.core.exceptions import SecInterpError
from sec_interp.exporters import get_exporter
from sec_interp.gui.utils import show_user_message
from sec_interp.logger_config import get_logger


class _NoOpMessageBar:
    """Safe no-op messagebar when iface is not available."""

    def pushMessage(self, *_args, **_kwargs):
        """No-op implementation of pushMessage."""
        return None


from .legend_widget import LegendWidget


logger = get_logger(__name__)
from .main_dialog_cache_handler import CacheHandler
from .main_dialog_config import DialogDefaults
from .main_dialog_data import DialogDataAggregator
from .main_dialog_export import ExportManager
from .main_dialog_preview import PreviewManager
from .main_dialog_settings import DialogSettingsManager
from .main_dialog_signals import DialogSignalManager
from .main_dialog_status import DialogStatusManager
from .main_dialog_tools import DialogToolManager, NavigationManager
from .main_dialog_utils import DialogEntityManager
from .main_dialog_validation import DialogValidator
from .ui.main_window import SecInterpMainWindow


class SecInterpDialog(SecInterpMainWindow):
    """Dialog for the SecInterp QGIS plugin.

    This dialog provides the user interface and helper methods to populate
    combo boxes with layers from the current QGIS project (raster and vector
    layers filtered by geometry type). It also exposes the interface and
    plugin instance for interaction with the host application.

    Attributes:
        iface (QgsInterface): The QGIS interface instance.
        plugin_instance (SecInterp): The plugin instance that created this dialog.
        messagebar (QgsMessageBar): The message bar widget for notifications.

    """

    def __init__(self, iface=None, plugin_instance=None, parent=None):
        """Initialize the dialog."""
        # Initialize the base class which sets up the programmatic UI
        super().__init__(iface, parent)

        self.iface = iface
        self.plugin_instance = plugin_instance
        self.project = QgsProject.instance()

        # Provide a safe, no-op messagebar when iface is not available (tests)
        if self.iface is None:
            self.messagebar = _NoOpMessageBar()
        else:
            self.messagebar = self.iface.messageBar()

        # Initialize manager instances
        self._init_managers()

        # Create legend widget
        self.legend_widget = LegendWidget(self.preview_widget.canvas)

        # Store current preview data
        self.current_topo_data = None
        self.current_geol_data = None
        self.current_struct_data = None
        self.current_canvas = None
        self.current_layers = []
        self.interpretations = []  # Store InterpretationPolygon objects

        # Add cache and reset buttons
        self.clear_cache_btn = QPushButton(self.tr("Clear Cache"))
        self.clear_cache_btn.setToolTip(self.tr("Clear cached data to force re-processing."))
        self.button_box.addButton(self.clear_cache_btn, QDialogButtonBox.ActionRole)

        self.reset_defaults_btn = QPushButton(self.tr("Reset Defaults"))
        self.reset_defaults_btn.setToolTip(self.tr("Reset all inputs to their default values."))
        self.button_box.addButton(self.reset_defaults_btn, QDialogButtonBox.ActionRole)

        # Initialize map tools via tool_manager
        self.tool_manager.initialize_tools()

        # Connect all signals
        self.signal_manager = DialogSignalManager(self)
        self.signal_manager.connect_all()

        # Connect extra tool buttons
        self.clear_cache_btn.clicked.connect(self.clear_cache_handler)
        self.reset_defaults_btn.clicked.connect(self.reset_defaults_handler)

        # Initial state update
        # Initial state update
        self.status_manager.update_all()
        self.settings_manager.load_settings()

        # Flag to control saving settings on close
        self._save_on_close = True

    def _init_managers(self):
        """Initialize all manager instances."""
        from sec_interp.core.services.preview_service import PreviewService

        self.validator = DialogValidator(self)
        self.preview_manager = PreviewManager(self, PreviewService(self.plugin_instance.controller))
        self.export_manager = ExportManager(self)
        self.cache_handler = CacheHandler(self)
        self.data_aggregator = DialogDataAggregator(self)
        self.settings_manager = DialogSettingsManager(self)
        self.status_manager = DialogStatusManager(self)
        self.status_manager.setup_indicators()
        self.tool_manager = DialogToolManager(self)
        self.navigation_manager = NavigationManager(self)

    def handle_error(self, error: Exception, title: str = "Error"):
        """Centralized error handling for the dialog.

        Args:
            error: The exception to handle.
            title: Title for the error message box.

        """
        if isinstance(error, SecInterpError):
            msg = str(error)
            logger.warning(f"{title}: {msg} - Details: {error.details}")
            show_user_message(self, title, msg, level="warning")
        else:
            msg = self.tr("An unexpected error occurred: {}").format(error)
            details = traceback.format_exc()
            logger.error(f"{title}: {msg}\n{details}")
            show_user_message(
                self,
                title,
                self.tr("{}\n\nPlease check the logs for details.").format(msg),
                level="critical",
            )

    def wheelEvent(self, event):
        """Handle mouse wheel for zooming in preview via navigation_manager."""
        if self.navigation_manager.handle_wheel_event(event):
            return
        super().wheelEvent(event)

    def closeEvent(self, event):
        """Handle dialog close event to clean up resources."""
        if self._save_on_close:
            self.settings_manager.save_settings()

        logger.info("Closing dialog, cleaning up resources...")
        self.preview_manager.cleanup()
        super().closeEvent(event)

    def open_help(self):
        """Open the help file in the default browser."""
        # Fix: help is at project root, main_dialog is in gui/
        help_file = Path(__file__).parent.parent / "help" / "html" / "index.html"
        if help_file.exists():
            QDesktopServices.openUrl(QUrl.fromLocalFile(str(help_file)))
        else:
            self.messagebar.pushMessage(
                self.tr("Error"),
                self.tr("Help file not found. Please run 'make doc' to generate it."),
                level=Qgis.Warning,
            )

    def toggle_measure_tool(self, checked):
        """Toggle measurement tool via tool_manager."""
        self.tool_manager.toggle_measure_tool(checked)

    def update_measurement_display(self, metrics):
        """Display measurement results from multi-point tool via tool_manager."""
        self.tool_manager.update_measurement_display(metrics)

    def toggle_interpretation_tool(self, checked):
        """Toggle interpretation tool via tool_manager."""
        if checked:
            # Deactivate measure tool if active
            self.preview_widget.btn_measure.setChecked(False)
        self.tool_manager.toggle_interpretation_tool(checked)

    def on_interpretation_finished(self, interpretation):
        """Handle finalized interpretation polygon.

        Args:
            interpretation: InterpretationPolygon object from the tool

        """
        from sec_interp.logger_config import log_critical_operation

        log_critical_operation(
            logger,
            "on_interpretation_finished",
            polygon_id=interpretation.id,
            vertices=len(interpretation.vertices_2d),
        )

        # Store interpretation
        self.interpretations.append(interpretation)
        logger.info(
            f"Interpretation polygon added: {interpretation.id} "
            f"({len(interpretation.vertices_2d)} vertices)"
        )

        # Display feedback in results area
        msg = (
            f"<b>Interpretación Finalizada</b><br>"
            f"<b>Vértices:</b> {len(interpretation.vertices_2d)}<br>"
            f"<b>ID:</b> {interpretation.id[:8]}..."
        )
        self.preview_widget.results_text.setHtml(msg)
        self.preview_widget.results_group.setCollapsed(False)

        # Deactivate interpretation tool
        self.preview_widget.btn_interpret.setChecked(False)

        # Update preview to show the new polygon
        self.update_preview_from_checkboxes()

    def update_preview_checkbox_states(self):
        """Enable or disable preview checkboxes via status_manager."""
        self.status_manager.update_preview_checkbox_states()

    def update_button_state(self):
        """Enable or disable buttons via status_manager."""
        self.status_manager.update_button_state()

    def get_selected_values(self):
        """Get the selected values from the dialog.

        Returns:
            Dictionary with all dialog values in legacy flat format

        """
        return self.data_aggregator.get_all_values()

    def get_preview_options(self):
        """Return the state of preview layer checkboxes.

        Returns:
            dict: Keys 'show_topo', 'show_geol', 'show_struct' with boolean values.

        """
        return {
            "show_topo": bool(self.preview_widget.chk_topo.isChecked()),
            "show_geol": bool(self.preview_widget.chk_geol.isChecked()),
            "show_struct": bool(self.preview_widget.chk_struct.isChecked()),
            "show_drillholes": bool(self.preview_widget.chk_drillholes.isChecked()),
            "show_interpretations": bool(self.preview_widget.chk_interpretations.isChecked()),
            "max_points": self.preview_widget.spin_max_points.value(),
            "auto_lod": self.preview_widget.chk_auto_lod.isChecked(),
            "use_adaptive_sampling": bool(self.preview_widget.chk_adaptive_sampling.isChecked()),
        }

    def update_preview_from_checkboxes(self):
        """Update preview when checkboxes change.

        This method delegates to PreviewManager for preview updates.
        """
        self.preview_manager.update_from_checkboxes()

    def preview_profile_handler(self):
        """Generate a quick preview with topographic, geological, and structural data.

        This method delegates to PreviewManager for preview generation.
        """
        success, message = self.preview_manager.generate_preview()
        if not success and message:
            self.messagebar.pushMessage("Preview Error", message, level=2)

    def export_preview(self):
        """Export the current preview to a file using ExportManager."""
        self.export_manager.export_preview()

    def accept_handler(self):
        """Handle the accept button click event."""
        # When running without a QGIS iface (tests), skip strict validation
        if self.iface is None:
            self.accept()
            return

        if not self.validate_inputs():
            return

        # Settings saved in closeEvent
        self.accept()

    def reject_handler(self):
        """Handle the reject button click event."""
        self._save_on_close = False
        self.close()

    def validate_inputs(self):
        """Validate the inputs from the dialog.

        This method delegates to DialogValidator for input validation.
        """
        is_valid, error_message = self.validator.validate_inputs()
        if not is_valid:
            scu.show_user_message(self, "Validation Error", error_message)
        return is_valid

    def clear_cache_handler(self):
        """Clear cached data and notify user."""
        if hasattr(self, "plugin_instance") and self.plugin_instance:
            self.plugin_instance.controller.data_cache.clear()
            self.preview_widget.results_text.append(
                self.tr("✓ Cache cleared - next preview will re-process data")
            )
            # context specific usage
            logger = get_logger(__name__)
            logger.info("Cache cleared by user")
        else:
            self.preview_widget.results_text.append(self.tr("⚠ Cache not available"))

    def reset_defaults_handler(self):
        """Reset all dialog inputs and notify user."""
        self.settings_manager.reset_to_defaults()
        self.preview_widget.results_text.append(self.tr("✓ Form reset to default values"))
        logger.info("Dialog reset to defaults by user")

    def _populate_field_combobox(self, source_combobox, target_combobox):
        """Populate a combobox with field names."""
        DialogEntityManager.populate_field_combobox(source_combobox, target_combobox)

    def get_layer_names_by_type(self, layer_type) -> list[str]:
        """Get layer names by type."""
        return DialogEntityManager.get_layer_names_by_type(layer_type)

    def get_layer_names_by_geometry(self, geometry_type) -> list[str]:
        """Get layer names by geometry."""
        return DialogEntityManager.get_layer_names_by_geometry(geometry_type)

    def getThemeIcon(self, name):
        """Get a theme icon via DialogEntityManager."""
        return DialogEntityManager.get_theme_icon(name)

    def _load_user_settings(self):
        """Load user settings via settings_manager."""
        self.settings_manager.load_settings()

    def _save_user_settings(self):
        """Save user settings via settings_manager."""
        self.settings_manager.save_settings()
