"""Common type aliases for SecInterp plugin.

This module defines type aliases used throughout the codebase to improve
type hint readability and maintainability.
"""

from enum import IntEnum
from typing import Any, Optional

from qgis.core import QgsPointXY, QgsRasterLayer, QgsVectorLayer


class FieldType(IntEnum):
    """Core-safe field types mapping to QVariant.Type values.

    This allows the core module to perform type validation WITHOUT
    direct dependencies on PyQt components.
    """

    NULL = 0
    BOOL = 1
    INT = 2
    DOUBLE = 6
    STRING = 10
    LONG_LONG = 4
    DATE = 14
    DATE_TIME = 16


# Profile data types
ProfileData = list[tuple[float, float]]
"""List of (distance, elevation) tuples representing a topographic profile."""

GeologyData = list[tuple[float, float, str]]
"""List of (distance, elevation, unit_name) tuples representing geological intersections."""

StructureData = list[tuple[float, float]]
"""List of (distance, apparent_dip) tuples representing projected structural measurements."""

# Layer collections
LayerDict = dict[str, QgsVectorLayer]
"""Dictionary mapping layer names to QgsVectorLayer objects."""

# Settings and configuration
SettingsDict = dict[str, Any]
"""Dictionary of plugin settings and configuration values."""

ExportSettings = dict[str, Any]
"""Dictionary of export configuration parameters."""

# Validation results
ValidationResult = tuple[bool, str]
"""Tuple of (is_valid, error_message) from validation functions."""

# Point data
PointList = list[QgsPointXY]
"""List of QgsPointXY objects."""

from dataclasses import dataclass


@dataclass
class StructureMeasurement:
    """Represents a projected structural measurement."""

    distance: float
    elevation: float
    apparent_dip: float
    original_dip: float
    original_strike: float
    attributes: dict[str, Any]


@dataclass
class GeologySegment:
    """Represents a geological unit segment along the profile."""

    unit_name: str
    geometry: "QgsGeometry"  # Forward reference
    attributes: dict[str, Any]
    # Sampled points for rendering (distance, elevation)
    points: list[tuple[float, float]]


# Updated type aliases
StructureData = list[StructureMeasurement]
GeologyData = list[GeologySegment]
