from qgis.core import QgsMapLayer, QgsMapLayerProxyModel, QgsProject, QgsWkbTypes, QgsVectorLayer
from qgis.gui import QgsMapLayerComboBox, QgsFieldComboBox
from qgis.PyQt.QtWidgets import QGridLayout, QLabel, QGroupBox, QVBoxLayout
from qgis.PyQt.QtCore import pyqtSignal, QModelIndex, Qt

from sec_interp.logger_config import get_logger

logger = get_logger(__name__)
logger.info("DrillPage.py loaded - Timestamp: 20251214_171700")

from .base_page import BasePage


class DrillPage(BasePage):
    """Configuration page for Drill Hole settings (Collars, Survey, Intervals)."""
    
    dataChanged = pyqtSignal()

    def __init__(self, parent=None):
        """Initialize Drill page."""
        super().__init__("Drill Logs", parent)

    def _setup_ui(self):
        super()._setup_ui()

# Define the custom proxy model class
class DrillLayerProxyModel(QgsMapLayerProxyModel):
    def __init__(self, parent=None, allowed_filters: QgsMapLayerProxyModel.Filter = QgsMapLayerProxyModel.AllLayers):
        super().__init__(parent)
        self._allowed_filters = allowed_filters
        self.setFilterCaseSensitivity(Qt.CaseInsensitive) # Default to case insensitive filter

    def filterAcceptsRow(self, source_row: int, source_parent: QModelIndex) -> bool:
        # Get the QgsMapLayer object for the current row
        layer = self.sourceModel().mapLayer(self.mapFromSource(source_row, source_parent).internalPointer())

        if not layer:
            return False

        # Apply the combined filters
        accept = True

        # Check for QgsMapLayerProxyModel.NoGeometry
        if (self._allowed_filters & QgsMapLayerProxyModel.NoGeometry) and layer.wkbType() == QgsWkbTypes.Unknown:
            accept = True # Accept if it's a non-spatial layer and filter allows it
        elif (self._allowed_filters & QgsMapLayerProxyModel.PointLayer) and isinstance(layer, QgsVectorLayer) and QgsWkbTypes.geometryType(layer.wkbType()) == QgsWkbTypes.PointGeometry:
            accept = True # Accept if it's a point layer and filter allows it
        else:
            accept = False # Reject by default if no specific filter matches

        return accept

# Rest of the _setup_ui method below...

        # Layout for the main content area (inside the inherited group box)
        # Actually BasePage has self.group_layout attached to self.group_box
        # We can use sub-groupboxes for clarity.
        
        main_layout = QVBoxLayout()
        # Reparent the layout of the group_box if needed or just add to it.
        # BasePage uses QGridLayout for self.group_layout.
        # Let's clean it up. BasePage initializes self.group_box with a layout?
        # Looking at dem_page.py: self.group_layout = QGridLayout(self.group_box)
        # So we can do the same.
        
        # We will split into 3 sections: Collar, Survey, Intervals
        
        # --- COLLAR SECTION ---
        self.collar_group = QGroupBox("Collar Data (Location)")
        collar_layout = QGridLayout(self.collar_group)
        
        collar_layout.addWidget(QLabel("Layer:"), 0, 0)
        self.collar_layer = QgsMapLayerComboBox()
        self.collar_layer.setFilters(QgsMapLayerProxyModel.PointLayer)
        self.collar_layer.setAllowEmptyLayer(True)
        collar_layout.addWidget(self.collar_layer, 0, 1)

        collar_layout.addWidget(QLabel("Hole ID:"), 1, 0)
        self.collar_id = QgsFieldComboBox()
        collar_layout.addWidget(self.collar_id, 1, 1)
        
        collar_layout.addWidget(QLabel("Elevation (Z):"), 2, 0)
        self.collar_z = QgsFieldComboBox()
        self.collar_z.setAllowEmptyFieldName(True)
        self.collar_z.setToolTip("Leave empty to use elevation from DEM (if available)")
        collar_layout.addWidget(self.collar_z, 2, 1)

        collar_layout.addWidget(QLabel("Easting (X) [Optional]:"), 3, 0)
        self.collar_x = QgsFieldComboBox()
        self.collar_x.setAllowEmptyFieldName(True)
        self.collar_x.setToolTip("Leave empty to use layer geometry X coordinate")
        collar_layout.addWidget(self.collar_x, 3, 1)

        collar_layout.addWidget(QLabel("Northing (Y) [Optional]:"), 4, 0)
        self.collar_y = QgsFieldComboBox()
        self.collar_y.setAllowEmptyFieldName(True)
        self.collar_y.setToolTip("Leave empty to use layer geometry Y coordinate")
        collar_layout.addWidget(self.collar_y, 4, 1)

        # --- SURVEY SECTION ---
        self.survey_group = QGroupBox("Survey Data (Geometry)")
        survey_layout = QGridLayout(self.survey_group)

        survey_layout.addWidget(QLabel("Layer:"), 0, 0)
        self.survey_layer = QgsMapLayerComboBox(self) # Initialize combo box
        survey_proxy_model = DrillLayerProxyModel(self.survey_layer, allowed_filters=QgsMapLayerProxyModel.NoGeometry | QgsMapLayerProxyModel.PointLayer)
        logger.debug(f"Created survey_proxy_model type: {type(survey_proxy_model)}")
        self.survey_layer.setModel(survey_proxy_model) # Set configured model
        logger.debug(f"After setModel, survey_layer.model() type: {type(self.survey_layer.model())}")
        self.survey_layer.setAllowEmptyLayer(True)
        survey_layout.addWidget(self.survey_layer, 0, 1)

        survey_layout.addWidget(QLabel("Hole ID:"), 1, 0)
        self.survey_id = QgsFieldComboBox()
        survey_layout.addWidget(self.survey_id, 1, 1)

        survey_layout.addWidget(QLabel("Depth:"), 2, 0)
        self.survey_depth = QgsFieldComboBox()
        survey_layout.addWidget(self.survey_depth, 2, 1)

        survey_layout.addWidget(QLabel("Azimuth:"), 3, 0)
        self.survey_az = QgsFieldComboBox()
        survey_layout.addWidget(self.survey_az, 3, 1)
        
        survey_layout.addWidget(QLabel("Dip:"), 4, 0)
        self.survey_dip = QgsFieldComboBox()
        survey_layout.addWidget(self.survey_dip, 4, 1)

        # --- INTERVAL SECTION ---
        self.interval_group = QGroupBox("Interval Data (Lithology/Assay)")
        interval_layout = QGridLayout(self.interval_group)

        interval_layout.addWidget(QLabel("Layer:"), 0, 0)
        self.interval_layer = QgsMapLayerComboBox(self) # Initialize combo box
        interval_proxy_model = DrillLayerProxyModel(self.interval_layer, allowed_filters=QgsMapLayerProxyModel.NoGeometry | QgsMapLayerProxyModel.PointLayer)
        logger.debug(f"Created interval_proxy_model type: {type(interval_proxy_model)}")
        logger.debug(f"After setModel, interval_layer.model() type: {type(self.interval_layer.model())}")
        self.interval_layer.setAllowEmptyLayer(True)
        interval_layout.addWidget(self.interval_layer, 0, 1)

        interval_layout.addWidget(QLabel("Hole ID:"), 1, 0)
        self.interval_id = QgsFieldComboBox()
        interval_layout.addWidget(self.interval_id, 1, 1)

        interval_layout.addWidget(QLabel("From:"), 2, 0)
        self.interval_from = QgsFieldComboBox()
        interval_layout.addWidget(self.interval_from, 2, 1)

        interval_layout.addWidget(QLabel("To:"), 3, 0)
        self.interval_to = QgsFieldComboBox()
        interval_layout.addWidget(self.interval_to, 3, 1)

        interval_layout.addWidget(QLabel("Value (Lith/Grade):"), 4, 0)
        self.interval_val = QgsFieldComboBox()
        interval_layout.addWidget(self.interval_val, 4, 1)

        # Add groups to main layout of the page
        # We attach to self.main_layout instead of group_box to have multiple blocks?
        # BasePage puts everything in self.group_box. 
        # Let's put these groups inside the self.group_box layout.
        
        self.group_layout = QVBoxLayout(self.group_box)
        self.group_layout.addWidget(self.collar_group)
        self.group_layout.addWidget(self.survey_group)
        self.group_layout.addWidget(self.interval_group)

        # Connect Layer Combos to Field Combos
        self.collar_layer.layerChanged.connect(self.collar_id.setLayer)
        self.collar_layer.layerChanged.connect(self.collar_z.setLayer)
        self.collar_layer.layerChanged.connect(self.collar_x.setLayer)
        self.collar_layer.layerChanged.connect(self.collar_y.setLayer)

        self.survey_layer.layerChanged.connect(self.survey_id.setLayer)
        self.survey_layer.layerChanged.connect(self.survey_depth.setLayer)
        self.survey_layer.layerChanged.connect(self.survey_az.setLayer)
        self.survey_layer.layerChanged.connect(self.survey_dip.setLayer)

        self.interval_layer.layerChanged.connect(self.interval_id.setLayer)
        self.interval_layer.layerChanged.connect(self.interval_from.setLayer)
        self.interval_layer.layerChanged.connect(self.interval_to.setLayer)
        self.interval_layer.layerChanged.connect(self.interval_val.setLayer)
        
        # Signal Connections for State Updates
        self.collar_layer.layerChanged.connect(self.dataChanged)
        self.survey_layer.layerChanged.connect(self.dataChanged)
        self.interval_layer.layerChanged.connect(self.dataChanged)
        
    def get_data(self) -> dict:
        """Get Drill configuration."""
        return {
            "collar_layer": self.collar_layer.currentLayer(),
            "collar_id_field": self.collar_id.currentField(),
            "collar_z_field": self.collar_z.currentField(),
            "collar_x_field": self.collar_x.currentField(),
            "collar_y_field": self.collar_y.currentField(),
            
            "survey_layer": self.survey_layer.currentLayer(),
            "survey_id_field": self.survey_id.currentField(),
            "survey_depth_field": self.survey_depth.currentField(),
            "survey_az_field": self.survey_az.currentField(),
            "survey_dip_field": self.survey_dip.currentField(),
            
            "interval_layer": self.interval_layer.currentLayer(),
            "interval_id_field": self.interval_id.currentField(),
            "interval_from_field": self.interval_from.currentField(),
            "interval_to_field": self.interval_to.currentField(),
            "interval_val_field": self.interval_val.currentField(),
        }

    def is_complete(self) -> bool:
        """Check if minimum required fields are filled (Collar is absolute minimum)."""
        # We need at least collars to do anything.
        if not self.collar_layer.currentLayer():
            return False
            
        # We need HoleID and Z. X/Y are optional (fallback to geometry)
        if not self.collar_id.currentField(): return False
        # Z is now optional (can use DEM)
        # if not self.collar_z.currentField(): return False
        
        # Check if layer is spatial
        layer = self.collar_layer.currentLayer()
        if layer:
            is_spatial = layer.isSpatial()
            # If not spatial (e.g. CSV table), we MUST have X and Y
            if not is_spatial:
                if not self.collar_x.currentField(): return False
                if not self.collar_y.currentField(): return False

        return True
