# /***************************************************************************
# SecInterpDialog
#                                 A QGIS plugin
# Data extraction for geological interpretation
# Generated by Plugin Builder: http://g-sherman.github.io/Qgis-Plugin-Builder/
#                             -------------------
#        begin                : 2025-11-15
#        git sha              : $Format:%H$
#        copyright            : (C) 2025 by Juan M Bernales
#        email                : juanbernales@gmail.com
# ***************************************************************************/
#
# /***************************************************************************
# *                                                                         *
# *   This program is free software; you can redistribute it and/or modify  *
# *   it under the terms of the GNU General Public License as published by  *
# *   the Free Software Foundation; either version 2 of the License, or     *
# *   (at your option) any later version.                                   *
# *                                                                         *
# ***************************************************************************/

"""Main Dialog Module.

Contains the SecInterpDialog class which is the primary UI for the plugin.
"""

from pathlib import Path
import tempfile
import traceback

from qgis.core import (
    Qgis,
    QgsApplication,
    QgsCoordinateTransform,
    QgsMapLayer,
    QgsMapLayerProxyModel,
    QgsMapSettings,
    QgsPointXY,
    QgsProject,
    QgsSettings,
    QgsUnitTypes,
    QgsWkbTypes,
)
from qgis.gui import QgsFileWidget, QgsMapCanvas, QgsMapToolPan
from qgis.PyQt.QtCore import QSize, QVariant, QUrl
from qgis.PyQt.QtGui import QColor, QDesktopServices
from qgis.PyQt.QtWidgets import (
    QApplication,
    QDialog,
    QDialogButtonBox,
    QFileDialog,
    QLabel,
    QPushButton,
    QStyle,
)

from sec_interp.core import utils as scu
from sec_interp.core import validation as vu
from sec_interp.exporters import get_exporter
from sec_interp.gui.utils import show_user_message
from sec_interp.logger_config import get_logger

from .tools.measure_tool import ProfileMeasureTool
from .legend_widget import LegendWidget
from .main_dialog_cache_handler import CacheHandler
from .main_dialog_config import DialogDefaults
from .main_dialog_data import DialogDataAggregator
from .main_dialog_export import ExportManager
from .main_dialog_preview import PreviewManager
from .main_dialog_signals import DialogSignalManager
from .main_dialog_validation import DialogValidator
from .main_dialog_settings import DialogSettingsManager
from .main_dialog_status import DialogStatusManager
from .main_dialog_utils import DialogEntityManager


from .ui.main_window import SecInterpMainWindow


class SecInterpDialog(SecInterpMainWindow):
    """Dialog for the SecInterp QGIS plugin.

    This dialog provides the user interface and helper methods to populate
    combo boxes with layers from the current QGIS project (raster and vector
    layers filtered by geometry type). It also exposes the interface and
    plugin instance for interaction with the host application.

    Attributes:
        iface (QgsInterface): The QGIS interface instance.
        plugin_instance (SecInterp): The plugin instance that created this dialog.
        messagebar (QgsMessageBar): The message bar widget for notifications.
    """

    def __init__(self, iface=None, plugin_instance=None, parent=None):
        """Constructor."""
        # Initialize the base class which sets up the programmatic UI
        super().__init__(iface, parent)

        self.iface = iface
        self.plugin_instance = plugin_instance

        # Provide a safe, no-op messagebar when iface is not available (tests)
        if self.iface is None:

            class _NoOpMessageBar:
                def pushMessage(self, *_args, **_kwargs):
                    """No-op implementation of pushMessage."""
                    return None

            self.messagebar = _NoOpMessageBar()
        else:
            self.messagebar = self.iface.messageBar()

        # Initialize manager instances
        self._init_managers()

        # Create legend widget
        self.legend_widget = LegendWidget(self.preview_widget.canvas)

        # Store current preview data
        self.current_topo_data = None
        self.current_geol_data = None
        self.current_struct_data = None
        self.current_canvas = None
        self.current_layers = []

        # Add clear cache button
        self.clear_cache_btn = QPushButton("Clear Cache")
        self.clear_cache_btn.setToolTip("Clear cached data to force re-processing.")
        self.button_box.addButton(self.clear_cache_btn, QDialogButtonBox.ActionRole)

        # Initialize map tools
        self._init_tools()

        # Connect all signals
        self.signal_manager = DialogSignalManager(self)
        self.signal_manager.connect_all()

        # Initial state update
        self.status_manager.update_all()
        self.settings_manager.load_settings()

    def _init_managers(self):
        """Initialize all manager instances."""
        self.validator = DialogValidator(self)
        self.preview_manager = PreviewManager(self)
        self.export_manager = ExportManager(self)
        self.cache_handler = CacheHandler(self)
        self.data_aggregator = DialogDataAggregator(self)
        self.settings_manager = DialogSettingsManager(self)
        self.status_manager = DialogStatusManager(self)
        self.status_manager.setup_indicators()

    def _init_tools(self):
        """Initialize map tools."""
        self.pan_tool = QgsMapToolPan(self.preview_widget.canvas)
        self.measure_tool = ProfileMeasureTool(self.preview_widget.canvas)
        self.preview_widget.canvas.setMapTool(self.pan_tool)

    def wheelEvent(self, event):
        """Handle mouse wheel for zooming in preview."""
        # Check if mouse is over preview widget
        if self.preview_widget.canvas.underMouse():
            # QgsMapCanvas has built-in zoom with wheel
            # We can customize the zoom factor if needed
            if event.angleDelta().y() > 0:
                self.preview_widget.canvas.zoomIn()
            else:
                self.preview_widget.canvas.zoomOut()
            event.accept()
        else:
            super().wheelEvent(event)

    def open_help(self):
        """Open the help file in the default browser."""
        # Fix: help is at project root, main_dialog is in gui/
        help_file = Path(__file__).parent.parent / "help" / "html" / "index.html"
        if help_file.exists():
            QDesktopServices.openUrl(QUrl.fromLocalFile(str(help_file)))
        else:
            self.messagebar.pushMessage(
                "Error",
                "Help file not found. Please run 'make doc' to generate it.",
                level=Qgis.Warning,
            )

    def toggle_measure_tool(self, checked):
        """Toggle measurement tool."""
        if checked:
            self.preview_widget.canvas.setMapTool(self.measure_tool)
            self.measure_tool.activate()
        else:
            self.preview_widget.canvas.setMapTool(self.pan_tool)
            self.pan_tool.activate()
            # Clear results when turning off? Maybe optional
            # self.preview_widget.results_text.clear()

    def update_measurement_display(self, dx, dy, dist, slope):
        """Display measurement results."""
        msg = (
            f"<b>Measurement Result:</b><br>"
            f"Distance: {dist:.2f} m<br>"
            f"Horizontal (dx): {dx:.2f} m<br>"
            f"Vertical (dy): {dy:.2f} m<br>"
            f"Slope: {slope:.1f}°"
        )
        self.preview_widget.results_text.setHtml(msg)
        # Ensure results group is expanded
        self.preview_widget.results_group.setCollapsed(False)

    def update_preview_checkbox_states(self):
        """Enable or disable preview checkboxes via status_manager."""
        self.status_manager.update_preview_checkbox_states()

    def update_button_state(self):
        """Enable or disable buttons via status_manager."""
        self.status_manager.update_button_state()

    def get_selected_values(self):
        """Get the selected values from the dialog.
        
        Returns:
            Dictionary with all dialog values in legacy flat format
        """
        return self.data_aggregator.get_all_values()

    def get_preview_options(self):
        """Return the state of preview layer checkboxes.

        Returns:
            dict: Keys 'show_topo', 'show_geol', 'show_struct' with boolean values.
        """
        return {
            "show_topo": bool(self.preview_widget.chk_topo.isChecked()),
            "show_geol": bool(self.preview_widget.chk_geol.isChecked()),
            "show_struct": bool(self.preview_widget.chk_struct.isChecked()),
            "show_drillholes": bool(self.preview_widget.chk_drillholes.isChecked()),
            "max_points": self.preview_widget.spin_max_points.value(),
            "auto_lod": self.preview_widget.chk_auto_lod.isChecked(),
            "use_adaptive_sampling": bool(self.preview_widget.chk_adaptive_sampling.isChecked()),
        }

    def update_preview_from_checkboxes(self):
        """Update preview when checkboxes change.

        This method delegates to PreviewManager for preview updates.
        """
        self.preview_manager.update_from_checkboxes()

    def preview_profile_handler(self):
        """Generate a quick preview with topographic, geological, and structural data.

        This method delegates to PreviewManager for preview generation.
        """
        success, message = self.preview_manager.generate_preview()
        if not success and message:
            self.messagebar.pushMessage("Preview Error", message, level=2)

        # Update CRS label in status bar
        try:
            # Use line layer CRS as the primary reference for the preview
            line_layer = self.page_section.line_combo.currentLayer()
            if line_layer:
                auth_id = line_layer.crs().authid()
                self.preview_widget.lbl_crs.setText(f"CRS: {auth_id}")
        except Exception:
            self.preview_widget.lbl_crs.setText("CRS: Unknown")

    def export_preview(self):
        """Export the current preview to a file using ExportManager."""
        self.export_manager.export_preview()

    def accept_handler(self):
        """Handle the accept button click event."""
        # When running without a QGIS iface (tests), skip strict validation
        if self.iface is None:
            self.accept()
            return

        if not self.validate_inputs():
            return

        # Save user settings before closing via settings_manager
        self.settings_manager.save_settings()
        self.accept()

    def reject_handler(self):
        """Handle the reject button click event."""
        self.close()

    def validate_inputs(self):
        """Validate the inputs from the dialog.

        This method delegates to DialogValidator for input validation.
        """
        is_valid, error_message = self.validator.validate_inputs()
        if not is_valid:
            scu.show_user_message(self, "Validation Error", error_message)
        return is_valid

    def clear_cache_handler(self):
        """Clear cached data and notify user."""
        if hasattr(self, "plugin_instance") and self.plugin_instance:
            self.plugin_instance.data_cache.clear()
            self.preview_widget.results_text.append(
                "✓ Cache cleared - next preview will re-process data"
            )
            # context specific usage
            logger = get_logger(__name__)
            logger.info("Cache cleared by user")
        else:
            self.preview_widget.results_text.append("⚠ Cache not available")

    def _populate_field_combobox(self, source_combobox, target_combobox):
        """Helper function to populate a combobox with field names."""
        DialogEntityManager.populate_field_combobox(source_combobox, target_combobox)

    def get_layer_names_by_type(self, layer_type) -> list[str]:
        """Get layer names by type."""
        return DialogEntityManager.get_layer_names_by_type(layer_type)

    def get_layer_names_by_geometry(self, geometry_type) -> list[str]:
        """Get layer names by geometry."""
        return DialogEntityManager.get_layer_names_by_geometry(geometry_type)

    def getThemeIcon(self, name):
        """Get a theme icon via DialogEntityManager."""
        return DialogEntityManager.get_theme_icon(name)

    def _load_user_settings(self):
        """Load user settings via settings_manager."""
        self.settings_manager.load_settings()

    def _save_user_settings(self):
        """Save user settings via settings_manager."""
        self.settings_manager.save_settings()
