import os.path

from qgis.core import (
    Qgis,
    QgsGeometry,
    QgsMessageLog,
    QgsPointXY,
    QgsRaster,
    QgsWkbTypes,
)
from qgis.PyQt.QtGui import QIcon
from qgis.PyQt.QtWidgets import QAction, QFileDialog

from . import si_core_utils as scu
from .drilsec_dialog import DrillSecDialog
from .preview_renderer import PreviewRenderer
from sec_interp.core.controller import ProfileController


class DrillSecPlugin:
    def __init__(self, iface):
        self.iface = iface
        self.plugin_dir = os.path.dirname(__file__)
        self.dlg = DrillSecDialog()
        self.preview_renderer = None
        self.controller = ProfileController()
        # Store data for export
        self.last_topo_points = None
        self.last_drillhole_data = None
        self.last_collar_points = None

    def log_message(self, message, level=Qgis.Info):
        """Log message to QGIS Message Log and UI Log Viewer."""
        # Log to QGIS
        QgsMessageLog.logMessage(message, "DrillSec", level)

        # Log to UI
        if hasattr(self.dlg, "mTextEdit_logs"):
            prefix = ""
            color = "black"
            if level == Qgis.Warning:
                prefix = "[WARNING] "
                color = "orange"
            elif level == Qgis.Critical:
                prefix = "[ERROR] "
                color = "red"
            elif level == Qgis.Success:
                prefix = "[SUCCESS] "
                color = "green"

            from datetime import datetime

            timestamp = datetime.now().strftime("%H:%M:%S")

            html = f'<span style="color:gray">[{timestamp}]</span> <span style="color:{color}">{prefix}{message}</span>'
            self.dlg.mTextEdit_logs.append(html)

    def initGui(self):
        icon_path = os.path.join(self.plugin_dir, "icon.png")
        self.action = QAction(QIcon(icon_path), "DrillSec", self.iface.mainWindow())
        self.action.triggered.connect(self.run)
        self.iface.addToolBarIcon(self.action)
        self.iface.addPluginToMenu("&DrillSec", self.action)
        # Connect dialog signals
        self.dlg.mButton_generate.clicked.connect(self.generate_profile)
        self.dlg.mButton_export_shp.clicked.connect(self.export_shapefiles)
        # Connect layer combo boxes to field combo boxes
        self.dlg.mMapLayerComboBox_collar.layerChanged.connect(
            self.dlg.mFieldComboBox_collar.setLayer
        )
        self.dlg.mMapLayerComboBox_collar.layerChanged.connect(
            self.dlg.mFieldComboBox_collar_x.setLayer
        )
        self.dlg.mMapLayerComboBox_collar.layerChanged.connect(
            self.dlg.mFieldComboBox_collar_y.setLayer
        )
        self.dlg.mMapLayerComboBox_collar.layerChanged.connect(
            self.dlg.mFieldComboBox_collar_z.setLayer
        )
        self.dlg.mMapLayerComboBox_collar.layerChanged.connect(
            self.dlg.mFieldComboBox_collar_depth.setLayer
        )
        self.dlg.mMapLayerComboBox_survey.layerChanged.connect(
            self.dlg.mFieldComboBox_survey.setLayer
        )
        self.dlg.mMapLayerComboBox_interval.layerChanged.connect(
            self.dlg.mFieldComboBox_interval.setLayer
        )

        # Connect UI logic signals
        self.dlg.mMapLayerComboBox_collar.layerChanged.connect(self.update_collar_ui)
        self.dlg.mCheckBox_use_geometry.toggled.connect(self.toggle_coordinate_fields)

    def update_collar_ui(self):
        layer = self.dlg.mMapLayerComboBox_collar.currentLayer()
        if not layer:
            return

        is_point = layer.geometryType() == QgsWkbTypes.PointGeometry
        self.dlg.mCheckBox_use_geometry.setEnabled(is_point)
        self.dlg.mCheckBox_use_geometry.setChecked(is_point)
        self.toggle_coordinate_fields(is_point)

    def toggle_coordinate_fields(self, checked):
        self.dlg.mFieldComboBox_collar_x.setEnabled(not checked)
        self.dlg.mFieldComboBox_collar_y.setEnabled(not checked)

    def unload(self):
        self.iface.removeToolBarIcon(self.action)
        self.iface.removePluginMenu("&DrillSec", self.action)
        del self.action

    def run(self):
        # Initialize field combo boxes with current layers
        collar_layer = self.dlg.mMapLayerComboBox_collar.currentLayer()
        if collar_layer:
            self.dlg.mFieldComboBox_collar.setLayer(collar_layer)
            self.dlg.mFieldComboBox_collar_x.setLayer(collar_layer)
            self.dlg.mFieldComboBox_collar_y.setLayer(collar_layer)
            self.dlg.mFieldComboBox_collar_z.setLayer(collar_layer)
            self.dlg.mFieldComboBox_collar_depth.setLayer(collar_layer)

        survey_layer = self.dlg.mMapLayerComboBox_survey.currentLayer()
        if survey_layer:
            self.dlg.mFieldComboBox_survey.setLayer(survey_layer)
            self.dlg.mFieldComboBox_survey_depth.setLayer(survey_layer)
            self.dlg.mFieldComboBox_survey_azim.setLayer(survey_layer)
            self.dlg.mFieldComboBox_survey_incl.setLayer(survey_layer)

        interval_layer = self.dlg.mMapLayerComboBox_interval.currentLayer()
        if interval_layer:
            self.dlg.mFieldComboBox_interval.setLayer(interval_layer)
            self.dlg.mFieldComboBox_interval_from.setLayer(interval_layer)
            self.dlg.mFieldComboBox_interval_to.setLayer(interval_layer)
            self.dlg.mFieldComboBox_interval_lith.setLayer(interval_layer)
            self.dlg.mFieldComboBox_interval_from.setLayer(interval_layer)
            self.dlg.mFieldComboBox_interval_to.setLayer(interval_layer)

        self.dlg.show()
        result = self.dlg.exec_()
        if result:
            pass

    def generate_profile(self):
        # Clear logs
        self.dlg.mTextEdit_logs.clear()
        self.log_message("Starting profile generation...", Qgis.Info)

        # 1. Validate and get essential layers and parameters
        section_layer = self.dlg.mMapLayerComboBox_section.currentLayer()
        dem_layer = self.dlg.mMapLayerComboBox_dem.currentLayer()
        if not section_layer or not dem_layer:
            self.log_message(
                "Please select DEM and Section Line layers", level=Qgis.Critical
            )
            return

        buffer_width = self.dlg.mDoubleSpinBox_buffer.value()

        # 2. Prepare profile context (line geometry, start point, distance area)
        try:
            line_geom, line_start, distance_area = scu.prepare_profile_context(
                section_layer
            )
        except ValueError as e:
            self.log_message(str(e), level=Qgis.Critical)
            return

        # 3. Prepare inputs for Controller
        values = {
            "raster_layer_obj": dem_layer,
            "line_layer_obj": section_layer,
            "selected_band": 1,
            "buffer_distance": buffer_width,
            "section_geom": line_geom,
            "section_start": line_start,
            "distance_area": distance_area,
        }

        # Drillhole Data Inputs
        collar_layer = self.dlg.mMapLayerComboBox_collar.currentLayer()
        if collar_layer:
            values["collar_layer_obj"] = collar_layer
            values["collar_id_field"] = self.dlg.mFieldComboBox_collar.currentField()
            values["collar_use_geometry"] = self.dlg.mCheckBox_use_geometry.isChecked()
            values["collar_x_field"] = self.dlg.mFieldComboBox_collar_x.currentField()
            values["collar_y_field"] = self.dlg.mFieldComboBox_collar_y.currentField()
            values["collar_z_field"] = self.dlg.mFieldComboBox_collar_z.currentField()
            values["collar_depth_field"] = self.dlg.mFieldComboBox_collar_depth.currentField()
            
            # Survey & Interval
            survey_layer = self.dlg.mMapLayerComboBox_survey.currentLayer()
            interval_layer = self.dlg.mMapLayerComboBox_interval.currentLayer()
            
            if survey_layer:
                values["survey_layer_obj"] = survey_layer
                values["survey_id_field"] = self.dlg.mFieldComboBox_survey.currentField()
                values["survey_depth_field"] = self.dlg.mFieldComboBox_survey_depth.currentField()
                values["survey_azim_field"] = self.dlg.mFieldComboBox_survey_azim.currentField()
                values["survey_incl_field"] = self.dlg.mFieldComboBox_survey_incl.currentField()
                
            if interval_layer:
                values["interval_layer_obj"] = interval_layer
                values["interval_id_field"] = self.dlg.mFieldComboBox_interval.currentField()
                values["interval_from_field"] = self.dlg.mFieldComboBox_interval_from.currentField()
                values["interval_to_field"] = self.dlg.mFieldComboBox_interval_to.currentField()
                values["interval_lith_field"] = self.dlg.mFieldComboBox_interval_lith.currentField()

        # 4. Generate Data via Controller
        topo_points, geol_data, struct_data, drillhole_data, messages = self.controller.generate_profile_data(values)

        # Log Controller Messages
        for msg in messages:
            level = Qgis.Info
            if "Error" in msg or "⚠" in msg:
                level = Qgis.Warning
            self.log_message(msg, level=level)

        if not topo_points:
             return

        # 5. Process Results for Preview/Export
        self.last_topo_points = topo_points
        
        # Merge drillhole segments into preview geology data
        preview_geol_data = []
        if drillhole_data:
             self.last_drillhole_data = drillhole_data
             # drillhole_data is structured: (hole_id, traces, segments)
             # PreviewRenderer expects geol_data as list of segments: (unit, points)
             for hole_id, traces, segments in drillhole_data:
                 preview_geol_data.extend(segments)
             # Also need collar points?
             # Logic for collars export relied on self.last_collar_points
             # Controller doesn't return collars list directly in Tuple, but prints valid count.
             # If I want to export collars, I might need to extract them from drillhole service or re-project?
             # Or `controller` should return them.
             # Wait, `DrillholeService.project_collars` returns list of tuples.
             # `controller` only returns `drillhole_data` (traces/segments).
             # I might need to update Controller to return `collars` list too if I want to maintain exact feature parity for export.
             # For now, let's allow export of traces/geology. Collars export might be temporarily disabled or I need to fetch them.
        else:
            self.last_drillhole_data = None
            
        # 6. Render Preview
        self.dlg.mButton_export_shp.setEnabled(True)
        self._render_preview(topo_points, preview_geol_data)

    def _render_preview(self, topo_points, geol_data):
        """Render the preview canvas with topographic and geological data."""
        # Ensure preview renderer exists
        if not self.preview_renderer:
            self.preview_renderer = PreviewRenderer(self.dlg.mMapCanvas_preview)
        canvas, layers = self.preview_renderer.render(
            topo_data=topo_points,
            geol_data=geol_data if geol_data else None,
            vert_exag=1.0,
        )
        # Refresh canvas and adjust view
        if canvas:
            try:
                canvas.refresh()
                # Zoom to full extent if possible
                if hasattr(canvas, "zoomToFullExtent"):
                    canvas.zoomToFullExtent()
            except Exception as e:
                self.log_message(
                    f"Failed to refresh preview canvas: {e}", level=Qgis.Warning
                )

        if canvas and layers:
            self.log_message(
                f"Preview canvas configured with {len(layers)} layers", level=Qgis.Info
            )
        else:
            self.log_message(
                "Preview rendering failed - no canvas or layers returned",
                level=Qgis.Warning,
            )
        self.log_message(
            f"Rendering preview: {len(topo_points)} topo, {len(geol_data) if geol_data else 0} geol points",
            level=Qgis.Info,
        )

    def export_shapefiles(self):
        """Export current profile data to shapefiles."""
        if not self.last_topo_points and not self.last_drillhole_data:
            self.log_message(
                "No data to export. Generate profile first.", level=Qgis.Warning
            )
            return

        # Get output folder
        output_folder = QFileDialog.getExistingDirectory(
            self.dlg, "Select Output Folder"
        )
        if not output_folder:
            return

        # Get section layer CRS
        section_layer = self.dlg.mMapLayerComboBox_section.currentLayer()
        crs = section_layer.crs() if section_layer else None

        self.log_message(f"Exporting shapefiles to {output_folder}...", level=Qgis.Info)
        if crs:
            self.log_message(f"Using CRS: {crs.authid()}", level=Qgis.Info)

        try:
            exporter = scu.ShapefileExporter(output_folder, "DrillSec_Profile", crs)
            created_files = exporter.export_profile_data(
                self.last_topo_points, self.last_drillhole_data, self.last_collar_points
            )

            if created_files:
                self.log_message(
                    f"Successfully exported {len(created_files)} files:",
                    level=Qgis.Success,
                )
                for path in created_files:
                    self.log_message(f"  - {os.path.basename(path)}", level=Qgis.Info)
            else:
                self.log_message("No files were exported.", level=Qgis.Warning)

        except Exception as e:
            self.log_message(f"Export failed: {e!s}", level=Qgis.Critical)

