from qgis.core import QgsMapLayerProxyModel
from qgis.gui import QgsFieldComboBox, QgsMapCanvas, QgsMapLayerComboBox
from qgis.PyQt import QtWidgets


class DrillSecDialog(QtWidgets.QDialog):
    """Dialog for the DrillSec QGIS plugin.

    This dialog provides the user interface for selecting input layers,
    configuring processing parameters, and displaying results for drillhole
    data visualization on topographic profiles.
    """

    def __init__(self, parent=None):
        """Constructor for DrillSecDialog.

        Sets up the entire user interface, including layer selection comboboxes,
        field selection comboboxes, spin boxes for numeric input, a preview
        map canvas, a log display, and action buttons.

        Args:
            parent: Parent widget for this dialog.
        """
        super().__init__(parent)
        self.setWindowTitle("DrillSec")
        self.resize(400, 450)  # Adjusted height

        # Main Layout
        self.main_layout = QtWidgets.QVBoxLayout(self)

        # --- Group: General Configuration ---
        self.group_general = QtWidgets.QGroupBox("General Configuration")
        self.layout_general = QtWidgets.QGridLayout()
        self.group_general.setLayout(self.layout_general)

        # DEM Layer Selection
        self.label_dem = QtWidgets.QLabel("DEM Layer:")
        self.layout_general.addWidget(self.label_dem, 0, 0)

        self.mMapLayerComboBox_dem = QgsMapLayerComboBox(self)
        self.mMapLayerComboBox_dem.setFilters(QgsMapLayerProxyModel.RasterLayer)
        self.mMapLayerComboBox_dem.setAllowEmptyLayer(True, "Select DEM Layer")
        self.layout_general.addWidget(self.mMapLayerComboBox_dem, 0, 1)

        # Section Line Layer Selection
        self.label_section = QtWidgets.QLabel("Section Line Layer:")
        self.layout_general.addWidget(self.label_section, 1, 0)

        self.mMapLayerComboBox_section = QgsMapLayerComboBox(self)
        self.mMapLayerComboBox_section.setFilters(QgsMapLayerProxyModel.LineLayer)
        self.mMapLayerComboBox_section.setAllowEmptyLayer(True, "Select Section Line")
        self.layout_general.addWidget(self.mMapLayerComboBox_section, 1, 1)

        # Buffer Width Selection
        self.label_buffer = QtWidgets.QLabel("Buffer Width (m):")
        self.layout_general.addWidget(self.label_buffer, 2, 0)

        self.mDoubleSpinBox_buffer = QtWidgets.QDoubleSpinBox(self)
        self.mDoubleSpinBox_buffer.setRange(0, 10000)
        self.mDoubleSpinBox_buffer.setDecimals(2)
        self.mDoubleSpinBox_buffer.setValue(50.00)
        self.mDoubleSpinBox_buffer.setSuffix(" m")
        self.layout_general.addWidget(self.mDoubleSpinBox_buffer, 2, 1)

        self.main_layout.addWidget(self.group_general)

        # --- Group: Drillhole Data ---
        self.group_drillhole = QtWidgets.QGroupBox("Drillhole Data")
        self.layout_drillhole = QtWidgets.QGridLayout()
        self.group_drillhole.setLayout(self.layout_drillhole)

        # Collar Layer Selection
        self.label_collar = QtWidgets.QLabel("Collar Layer:")
        self.layout_drillhole.addWidget(self.label_collar, 0, 0)

        self.mMapLayerComboBox_collar = QgsMapLayerComboBox(self)
        self.mMapLayerComboBox_collar.setFilters(QgsMapLayerProxyModel.VectorLayer)
        self.mMapLayerComboBox_collar.setAllowEmptyLayer(True, "Select Collar Layer")
        # Accept tables too
        self.layout_drillhole.addWidget(self.mMapLayerComboBox_collar, 0, 1)

        self.mFieldComboBox_collar = QgsFieldComboBox(self)
        self.mFieldComboBox_collar.setToolTip("Select Hole ID Field")
        self.layout_drillhole.addWidget(self.mFieldComboBox_collar, 0, 2)

        # Collar Coordinates (X, Y)
        self.mCheckBox_use_geometry = QtWidgets.QCheckBox("Use Layer Geometry", self)
        self.mCheckBox_use_geometry.setChecked(True)
        self.layout_drillhole.addWidget(self.mCheckBox_use_geometry, 1, 0, 1, 2)

        self.label_collar_x = QtWidgets.QLabel("Easting (X):")
        self.layout_drillhole.addWidget(self.label_collar_x, 2, 0)
        self.mFieldComboBox_collar_x = QgsFieldComboBox(self)
        self.mFieldComboBox_collar_x.setEnabled(False)
        self.layout_drillhole.addWidget(self.mFieldComboBox_collar_x, 2, 1)

        self.label_collar_y = QtWidgets.QLabel("Northing (Y):")
        self.layout_drillhole.addWidget(self.label_collar_y, 3, 0)
        self.mFieldComboBox_collar_y = QgsFieldComboBox(self)
        self.mFieldComboBox_collar_y.setEnabled(False)
        self.layout_drillhole.addWidget(self.mFieldComboBox_collar_y, 3, 1)

        # Collar Elevation (Z)
        self.label_collar_z = QtWidgets.QLabel("Elevation (Z) [Optional]:")
        self.layout_drillhole.addWidget(self.label_collar_z, 4, 0)
        self.mFieldComboBox_collar_z = QgsFieldComboBox(self)
        self.mFieldComboBox_collar_z.setAllowEmptyFieldName(True)
        self.layout_drillhole.addWidget(self.mFieldComboBox_collar_z, 4, 1)

        # Collar Total Depth
        self.label_collar_depth = QtWidgets.QLabel("Total Depth [Optional]:")
        self.layout_drillhole.addWidget(self.label_collar_depth, 5, 0)
        self.mFieldComboBox_collar_depth = QgsFieldComboBox(self)
        self.mFieldComboBox_collar_depth.setAllowEmptyFieldName(True)
        self.layout_drillhole.addWidget(self.mFieldComboBox_collar_depth, 5, 1)

        # Survey Layer Selection
        self.label_survey = QtWidgets.QLabel("Survey Layer:")
        self.layout_drillhole.addWidget(self.label_survey, 6, 0)

        self.mMapLayerComboBox_survey = QgsMapLayerComboBox(self)
        self.mMapLayerComboBox_survey.setFilters(QgsMapLayerProxyModel.VectorLayer)
        self.mMapLayerComboBox_survey.setAllowEmptyLayer(True, "Select Survey Layer")
        self.layout_drillhole.addWidget(self.mMapLayerComboBox_survey, 6, 1)

        self.mFieldComboBox_survey = QgsFieldComboBox(self)
        self.mFieldComboBox_survey.setToolTip("Select Hole ID Field")
        self.layout_drillhole.addWidget(self.mFieldComboBox_survey, 6, 2)

        # Survey Fields (Depth, Azimuth, Inclination)
        self.label_survey_depth = QtWidgets.QLabel("Depth:")
        self.layout_drillhole.addWidget(self.label_survey_depth, 7, 0)
        self.mFieldComboBox_survey_depth = QgsFieldComboBox(self)
        self.layout_drillhole.addWidget(self.mFieldComboBox_survey_depth, 7, 1)

        self.label_survey_azim = QtWidgets.QLabel("Azimuth:")
        self.layout_drillhole.addWidget(self.label_survey_azim, 8, 0)
        self.mFieldComboBox_survey_azim = QgsFieldComboBox(self)
        self.layout_drillhole.addWidget(self.mFieldComboBox_survey_azim, 8, 1)

        self.label_survey_incl = QtWidgets.QLabel("Inclination:")
        self.layout_drillhole.addWidget(self.label_survey_incl, 9, 0)
        self.mFieldComboBox_survey_incl = QgsFieldComboBox(self)
        self.layout_drillhole.addWidget(self.mFieldComboBox_survey_incl, 9, 1)

        # Interval Layer Selection
        self.label_interval = QtWidgets.QLabel("Interval Layer:")
        self.layout_drillhole.addWidget(self.label_interval, 10, 0)

        self.mMapLayerComboBox_interval = QgsMapLayerComboBox(self)
        self.mMapLayerComboBox_interval.setFilters(QgsMapLayerProxyModel.VectorLayer)
        self.mMapLayerComboBox_interval.setAllowEmptyLayer(True, "Select Interval Layer")
        self.layout_drillhole.addWidget(self.mMapLayerComboBox_interval, 10, 1)

        self.mFieldComboBox_interval = QgsFieldComboBox(self)
        self.mFieldComboBox_interval.setToolTip("Select Hole ID Field")
        self.layout_drillhole.addWidget(self.mFieldComboBox_interval, 10, 2)

        # Interval Fields (From, To)
        self.label_interval_from = QtWidgets.QLabel("From:")
        self.layout_drillhole.addWidget(self.label_interval_from, 11, 0)
        self.mFieldComboBox_interval_from = QgsFieldComboBox(self)
        self.layout_drillhole.addWidget(self.mFieldComboBox_interval_from, 11, 1)

        self.label_interval_to = QtWidgets.QLabel("To:")
        self.layout_drillhole.addWidget(self.label_interval_to, 12, 0)
        self.mFieldComboBox_interval_to = QgsFieldComboBox(self)
        self.layout_drillhole.addWidget(self.mFieldComboBox_interval_to, 12, 1)

        self.label_interval_lith = QtWidgets.QLabel("Lithology/Attribute:")
        self.layout_drillhole.addWidget(self.label_interval_lith, 13, 0)
        self.mFieldComboBox_interval_lith = QgsFieldComboBox(self)
        self.layout_drillhole.addWidget(self.mFieldComboBox_interval_lith, 13, 1)

        # Connect Signals
        self.mMapLayerComboBox_collar.layerChanged.connect(self.mFieldComboBox_collar.setLayer)
        self.mMapLayerComboBox_collar.layerChanged.connect(self.mFieldComboBox_collar_x.setLayer)
        self.mMapLayerComboBox_collar.layerChanged.connect(self.mFieldComboBox_collar_y.setLayer)
        self.mMapLayerComboBox_collar.layerChanged.connect(self.mFieldComboBox_collar_z.setLayer)
        self.mMapLayerComboBox_collar.layerChanged.connect(
            self.mFieldComboBox_collar_depth.setLayer
        )

        self.mMapLayerComboBox_survey.layerChanged.connect(self.mFieldComboBox_survey.setLayer)
        self.mMapLayerComboBox_survey.layerChanged.connect(
            self.mFieldComboBox_survey_depth.setLayer
        )
        self.mMapLayerComboBox_survey.layerChanged.connect(self.mFieldComboBox_survey_azim.setLayer)
        self.mMapLayerComboBox_survey.layerChanged.connect(self.mFieldComboBox_survey_incl.setLayer)

        self.mMapLayerComboBox_interval.layerChanged.connect(self.mFieldComboBox_interval.setLayer)
        self.mMapLayerComboBox_interval.layerChanged.connect(
            self.mFieldComboBox_interval_from.setLayer
        )
        self.mMapLayerComboBox_interval.layerChanged.connect(
            self.mFieldComboBox_interval_to.setLayer
        )
        self.mMapLayerComboBox_interval.layerChanged.connect(
            self.mFieldComboBox_interval_lith.setLayer
        )

        self.main_layout.addWidget(self.group_drillhole)

        # --- Group: Preview ---
        self.group_preview = QtWidgets.QGroupBox("Preview")
        self.layout_preview = QtWidgets.QVBoxLayout()
        self.group_preview.setLayout(self.layout_preview)

        # Map Canvas for preview
        self.mMapCanvas_preview = QgsMapCanvas(self)
        self.mMapCanvas_preview.setMinimumHeight(300)
        self.mMapCanvas_preview.setMaximumHeight(400)
        self.layout_preview.addWidget(self.mMapCanvas_preview)

        self.main_layout.addWidget(self.group_preview)

        # --- Group: Results & Logs ---
        self.group_logs = QtWidgets.QGroupBox("Results & Logs")
        self.layout_logs = QtWidgets.QVBoxLayout()
        self.group_logs.setLayout(self.layout_logs)

        self.mTextEdit_logs = QtWidgets.QTextEdit()
        self.mTextEdit_logs.setReadOnly(True)
        self.mTextEdit_logs.setMinimumHeight(100)
        self.layout_logs.addWidget(self.mTextEdit_logs)

        self.main_layout.addWidget(self.group_logs)

        # --- Spacer ---
        self.main_layout.addStretch()

        # --- Action Buttons ---
        # Generate Button
        self.mButton_generate = QtWidgets.QPushButton("Generate Profile")
        self.mButton_generate.setFixedHeight(40)  # Make it bigger
        self.main_layout.addWidget(self.mButton_generate)

        # Export Button
        self.mButton_export_shp = QtWidgets.QPushButton("Export to SHP")
        self.mButton_export_shp.setFixedHeight(40)
        self.mButton_export_shp.setEnabled(False)  # Disabled until profile is generated
        self.main_layout.addWidget(self.mButton_export_shp)

        # Standard Buttons (Ok/Cancel)
        button_flags = QtWidgets.QDialogButtonBox.Ok | QtWidgets.QDialogButtonBox.Cancel
        self.button_box = QtWidgets.QDialogButtonBox(button_flags)
        self.button_box.accepted.connect(self.accept)
        self.button_box.rejected.connect(self.reject)
        self.main_layout.addWidget(self.button_box)
