"""/***************************************************************************
 SecInterp
                                 A QGIS plugin
 Data extraction for geological interpretation.
 Generated by Plugin Builder: http://g-sherman.github.io/Qgis-Plugin-Builder/.
                              -------------------
        begin                : 2025-11-15
        git sha              : $Format:%H$
        copyright            : (C) 2025 by Juan M Bernales
        email                : juanbernales@gmail.com
 ***************************************************************************/

/***************************************************************************
 *                                                                         *
 *   This program is free software; you can redistribute it and/or modify  *
 *   it under the terms of the GNU General Public License as published by  *
 *   the Free Software Foundation; either version 2 of the License, or     *
 *   (at your option) any later version.                                   *
 *                                                                         *
 ***************************************************************************/
"""

import contextlib
from pathlib import Path

from qgis.core import (
    QgsGeometry,
    QgsRaster,
    QgsRasterLayer,
    QgsVectorLayer,
    QgsWkbTypes,
)
from qgis.PyQt.QtCore import (
    QCoreApplication,
    QSettings,
    QTranslator,
)
from qgis.PyQt.QtGui import QIcon
from qgis.PyQt.QtWidgets import QAction, QDialogButtonBox, QMessageBox

from sec_interp.core import utils as scu
from sec_interp.core import validation as vu
from sec_interp.exporters import (
    AxesShpExporter,
    CSVExporter,
    GeologyShpExporter,
    ProfileLineShpExporter,
    StructureShpExporter,
)
from sec_interp.gui.main_dialog import SecInterpDialog
from sec_interp.gui.preview_renderer import PreviewRenderer
from sec_interp.logger_config import get_logger

from .data_cache import DataCache
from .services import GeologyService, ProfileService, StructureService


logger = get_logger(__name__)


# DataCache has been moved to core/data_cache.py


class SecInterp:
    """QGIS Plugin Implementation for Geological Data Extraction.

    This class implements the main logic of the SecInterp plugin, handling
    initialization, UI integration, and orchestration of data processing tasks.
    It connects the QGIS interface with the plugin's dialog and processing algorithms.
    """

    def __init__(self, iface):
        """Constructor.

        Args:
            iface (QgsInterface): An interface instance that will be passed to this class
                which provides the hook by which you can manipulate the QGIS
                application at run time.
        """
        # Save reference to the QGIS interface
        self.iface = iface
        # initialize plugin directory
        self.plugin_dir = Path(__file__).resolve().parent
        # initialize locale
        locale = QSettings().value("locale/userLocale")[0:2]
        locale_path = self.plugin_dir / f"i18n/SecInterp_{locale}.qm"

        if locale_path.exists():
            self.translator = QTranslator()
            self.translator.load(str(locale_path))
            QCoreApplication.installTranslator(self.translator)

        # Create the dialog (after translation) and keep reference
        self.dlg = SecInterpDialog(self.iface, self)
        self.dlg.plugin_instance = self

        # Create preview renderer
        self.preview_renderer = PreviewRenderer()

        # Initialize services
        self.profile_service = ProfileService()
        self.geology_service = GeologyService()
        self.structure_service = StructureService()

        # Initialize data cache for performance
        self.data_cache = DataCache()
        logger.debug("Data cache initialized")

        # Declare instance attributes
        self.actions = []
        self.menu = self.tr("&Sec Interp")
        # Create custom toolbar and make it visible
        self.toolbar = self.iface.addToolBar("Sec Interp")
        self.toolbar.setObjectName("SecInterp")
        self.toolbar.setVisible(True)  # Ensure toolbar is visible

        # Check if plugin was started the first time in current QGIS session
        # Must be set in initGui() to survive plugin reloads
        self.first_start = None

    # noinspection PyMethodMayBeStatic
    def tr(self, message):
        """Get the translation for a string using Qt translation API.

        We implement this ourselves since we do not inherit QObject.

        Args:
            message (str): String for translation.

        Returns:
            str: Translated string (or original if no translation found).
        """
        # noinspection PyTypeChecker,PyArgumentList,PyCallByClass
        return QCoreApplication.translate("SecInterp", message)

    def add_action(
        self,
        icon_path,
        text,
        callback,
        enabled_flag=True,
        add_to_menu=True,
        add_to_toolbar=True,
        status_tip=None,
        whats_this=None,
        parent=None,
    ):
        """Add a toolbar icon to the toolbar.

        Args:
            icon_path (str): Path to the icon for this action.
            text (str): Text that should be shown in menu items for this action.
            callback (function): Function to be called when the action is triggered.
            enabled_flag (bool): A flag indicating if the action should be enabled
                by default. Defaults to True.
            add_to_menu (bool): Flag indicating whether the action should also
                be added to the menu. Defaults to True.
            add_to_toolbar (bool): Flag indicating whether the action should also
                be added to the toolbar. Defaults to True.
            status_tip (str): Optional text to show in a popup when mouse pointer
                hovers over the action.
            whats_this (str): Optional text to show in the status bar when the
                mouse pointer hovers over the action.
            parent (QWidget): Parent widget for the new action. Defaults None.

        Returns:
            QAction: The action that was created. Note that the action is also
                added to self.actions list.
        """
        icon = QIcon(icon_path)
        action = QAction(icon, text, parent)
        action.triggered.connect(callback)
        action.setEnabled(enabled_flag)

        if status_tip is not None:
            action.setStatusTip(status_tip)

        if whats_this is not None:
            action.setWhatsThis(whats_this)

        if add_to_toolbar:
            # Add to custom toolbar
            self.toolbar.addAction(action)
            # Also add to Plugins toolbar for visibility
            self.iface.addToolBarIcon(action)

        if add_to_menu:
            self.iface.addPluginToMenu(self.menu, action)

        self.actions.append(action)

        return action

    def initGui(self):
        """Create the menu entries and toolbar icons inside the QGIS GUI."""
        # Use absolute path to icon file to ensure it loads in toolbar
        icon_path = str(self.plugin_dir.parent / "icon.png")
        self.add_action(
            icon_path,
            text=self.tr("Geological data extraction"),
            callback=self.run,
            parent=self.iface.mainWindow(),
        )

        # will be set False in run()
        self.first_start = True

    def unload(self):
        """Removes the plugin menu item and icon from QGIS GUI."""
        for action in self.actions:
            self.iface.removePluginMenu(self.tr("&Sec Interp"), action)
            self.iface.removeToolBarIcon(action)

        # Remove custom toolbar
        if self.toolbar:
            del self.toolbar
            self.toolbar = None

    def run(self):
        """Run method that performs all the real work.

        This method initializes the dialog, connects signals and slots,
        and executes the main event loop for the plugin dialog.
        """
        # Create the dialog with elements (after translation) and keep reference
        # Only create GUI ONCE in callback, so that it will only load when the
        # plugin is started
        if self.first_start is True:
            self.first_start = False
            self.dlg = SecInterpDialog(self.iface, self)
            # Update preview renderer with the new dialog's canvas
            self.preview_renderer.canvas = self.dlg.preview_widget.canvas
        # show the dialog
        self.dlg.show()

        # Disconnect default accepted signal to prevent closing on Save
        with contextlib.suppress(TypeError):
            self.dlg.button_box.accepted.disconnect()

        # Connect OK button to process and close
        self.dlg.button_box.button(QDialogButtonBox.Ok).clicked.connect(
            self.process_data
        )
        self.dlg.button_box.button(QDialogButtonBox.Ok).clicked.connect(self.dlg.accept)

        # Connect Save button to save only
        self.dlg.button_box.button(QDialogButtonBox.Save).clicked.connect(
            self.save_profile_line
        )
        # Run the dialog event loop
        result = self.dlg.exec_()
        # See if OK was pressed
        if result:
            # Do something useful here - delete the line containing pass and
            # substitute with your code.
            pass

    def _get_and_validate_inputs(self):
        """Retrieve and validate inputs from the dialog.

        Returns:
            dict: Validated inputs including layer objects, or None if validation fails.
        """
        # Get values from the dialog using the helper method
        values = self.dlg.get_selected_values()

        is_valid, error_msg, validated_values = vu.validate_and_get_layers(values)

        if not is_valid:
            scu.show_user_message(self.dlg, self.tr("Error"), self.tr(error_msg))
            return None

        return validated_values

    def process_data(self):
        """Process profile data from selected layers with caching.

        Uses cache to avoid re-processing when only visualization
        parameters change.
        """
        # Get and validate inputs once
        inputs = self._get_and_validate_inputs()
        if not inputs:
            return None

        validated_values = inputs

        # Check for reasonable parameter ranges
        warnings = vu.validate_reasonable_ranges(validated_values)
        if warnings:
            logger.warning("Parameter range warnings: %s", warnings)
            self.dlg.preview_widget.results_text.append("\n⚠️ Validation Warnings:")
            for warning in warnings:
                self.dlg.preview_widget.results_text.append(f"  {warning}")
            self.dlg.preview_widget.results_text.append("")  # Empty line

        # Show CRS warning if present
        if "_crs_warning" in validated_values:
            self.dlg.preview_widget.results_text.append("\n" + validated_values["_crs_warning"])
            self.dlg.preview_widget.results_text.append("")  # Empty line

        # Check if we can use cached data
        cache_key = self.data_cache.get_cache_key(validated_values)
        cached_data = self.data_cache.get(cache_key)

        profile_data = None
        geol_data = None
        struct_data = None
        msgs = []

        if cached_data:
            logger.info("⚡ Using cached profile data")
            self.dlg.preview_widget.results_text.append("⚡ Using cached data (fast!)")
            profile_data = cached_data["profile_data"]
            geol_data = cached_data.get("geol_data")
            struct_data = cached_data.get("struct_data")
            msgs = cached_data.get("msgs", [])
        else:
            logger.info("🔄 Processing new profile data...")
            self.dlg.preview_widget.results_text.append("🔄 Processing data...")

            try:
                # Generate data using unified method
                profile_data, geol_data, struct_data, msgs = self._generate_profile_data(
                    validated_values
                )

                if not profile_data:
                    if not msgs: # If no messages, likely complete failure
                         scu.show_user_message(
                            self.dlg,
                            self.tr("Error"),
                            self.tr(
                                "No profile data was generated. "
                                "Check that the line intersects the raster."
                            ),
                        )
                         return None
                
                # Cache the processed data
                self.data_cache.set(
                    cache_key,
                    {
                        "profile_data": profile_data,
                        "geol_data": geol_data,
                        "struct_data": struct_data,
                        "msgs": msgs,
                    },
                )
                logger.info("✓ Data cached for future use")

            except Exception as e:
                self._handle_processing_error(e)
                return None

        # Update results text with messages from generation
        if msgs:
            self.dlg.preview_widget.results_text.setPlainText("\n".join(msgs))
        
        # Always append tip
        self.dlg.preview_widget.results_text.append(
            "\n\n💡 Use 'Save' button to export data to CSV and Shapefile."
        )

        # Draw preview with all data
        self.draw_preview(profile_data, geol_data, struct_data)

        return profile_data, geol_data, struct_data

    def _generate_profile_data(self, values):
        """Unified method to generate all profile data components.
        
        Args:
            values (dict): Validated input values.

        Returns:
            tuple: (profile_data, geol_data, struct_data, messages)
        """
        profile_data = []
        geol_data = None
        struct_data = None
        messages = []

        raster_layer = values["raster_layer_obj"]
        line_layer = values["line_layer_obj"]
        outcrop_layer = values["outcrop_layer_obj"]
        structural_layer = values["structural_layer_obj"]
        selected_band = values["selected_band"]
        buffer_dist = values["buffer_distance"]

        # 1. Topography
        profile_data = self.profile_service.generate_topographic_profile(
            line_layer, raster_layer, selected_band
        )

        if not profile_data:
            return None, None, None, ["Error: No profile data generated."]

        messages.append(f"✓ Data processed successfully!\n\nTopography: {len(profile_data)} points")

        # 2. Geology
        if outcrop_layer:
            outcrop_name_field = values["outcrop_name_field"]
            if outcrop_name_field:
                geol_data = self.geology_service.generate_geological_profile(
                    line_layer,
                    raster_layer,
                    outcrop_layer,
                    outcrop_name_field,
                    selected_band,
                )
                if geol_data:
                    messages.append(f"Geology: {len(geol_data)} segments")
                else:
                    messages.append("Geology: No intersections")
            else:
                 messages.append("\n⚠ Outcrop layer selected but no geology field specified.")

        # 3. Structure
        if structural_layer:
            dip_field = values["dip_field"]
            strike_field = values["strike_field"]

            if dip_field and strike_field:
                line_feat = next(line_layer.getFeatures(), None)
                if line_feat:
                    line_geom = line_feat.geometry()
                    if line_geom and not line_geom.isNull():
                        line_azimuth = scu.calculate_line_azimuth(line_geom)
                        struct_data = self.structure_service.project_structures(
                            line_layer,
                            raster_layer,  # Added argument
                            structural_layer,
                            buffer_dist,
                            line_azimuth,
                            dip_field,
                            strike_field,
                            selected_band, # Added argument
                        )
                        
                        if struct_data:
                            messages.append(f"Structures: {len(struct_data)} points")
                        else:
                            messages.append(f"Structures: None in {buffer_dist}m buffer")
            else:
                 messages.append("\n⚠ Structural layer selected but dip/strike fields not specified.")

        return profile_data, geol_data, struct_data, messages

    def _handle_processing_error(self, e):
        """Unified error handler."""
        import traceback
        error_details = traceback.format_exc()

        if isinstance(e, OSError):
             scu.show_user_message(
                self.dlg,
                self.tr("File System Error"),
                self.tr(f"Failed to access temporary files: {e!s}"),
                "error",
            )
        elif isinstance(e, ValueError):
             scu.show_user_message(
                self.dlg,
                self.tr("Data Validation Error"),
                self.tr(f"Invalid data encountered: {e!s}"),
            )
        elif isinstance(e, KeyError):
            scu.show_user_message(
                self.dlg,
                self.tr("Field Error"),
                self.tr(f"Required field not found: {e!s}"),
            )
        else:
             scu.show_user_message(
                self.dlg,
                self.tr("Unexpected Error"),
                self.tr(f"An unexpected error occurred: {e!s}"),
                "error",
            )
             logger.exception("Unexpected error in process_data: %s", error_details)
        
        self.dlg.preview_widget.results_text.append(f"Error: {e!s}")

    def save_profile_line(self):
        """Save all profile data to CSV and Shapefile formats by delegating to exporters."""
        try:
            inputs = self._get_and_validate_inputs()
            if not inputs:
                return

            values = inputs
            output_folder = Path(values["output_path"])
            is_valid, error, _ = vu.validate_output_path(str(output_folder))
            if not is_valid:
                scu.show_user_message(self.dlg, self.tr("Error"), self.tr(error))
                return

            self.dlg.preview_widget.results_text.setPlainText("✓ Generating data for export...")

            # 1. Generate all data using unified method
            try:
                profile_data, geol_data, struct_data, _ = self._generate_profile_data(values)
            except Exception as e:
                 self._handle_processing_error(e)
                 return

            if not profile_data:
                scu.show_user_message(
                    self.dlg,
                    self.tr("Error"),
                    self.tr("No profile data generated, cannot save."),
                )
                return

            # 2. Orchestrate saving using exporters
            result_msg = ["✓ Saving files..."]
            csv_exporter = CSVExporter({})
            line_crs = values["line_layer_obj"].crs()

            # Export Topography
            logger.info("✓ Saving topographic profile...")
            csv_exporter.export(
                output_folder / "topo_profile.csv",
                {"headers": ["dist", "elev"], "rows": profile_data},
            )
            result_msg.append("  - topo_profile.csv")
            ProfileLineShpExporter({}).export(
                output_folder / "profile_line.shp",
                {"profile_data": profile_data, "crs": line_crs},
            )
            result_msg.append("  - profile_line.shp")

            # Export Geology
            if geol_data:
                logger.info("✓ Saving geological profile...")
                # CSV needs flattening? data structure is now segments
                # We can flatten it manually for CSV
                geol_rows = []
                for s in geol_data:
                    for p in s.points:
                        geol_rows.append((p[0], p[1], s.unit_name))
                
                csv_exporter.export(
                    output_folder / "geol_profile.csv",
                    {"headers": ["dist", "elev", "geology"], "rows": geol_rows},
                )
                result_msg.append("  - geol_profile.csv")
                
                GeologyShpExporter({}).export(
                    output_folder / "geol_profile.shp",
                    {
                        "geology_data": geol_data,
                        "crs": line_crs,
                    },
                )
                result_msg.append("  - geol_profile.shp")

            # Export Structures
            if struct_data:
                logger.info("✓ Saving structural profile...")
                # CSV needs simple rows
                struct_rows = [(s.distance, s.apparent_dip) for s in struct_data]
                
                csv_exporter.export(
                    output_folder / "structural_profile.csv",
                    {"headers": ["dist", "apparent_dip"], "rows": struct_rows},
                )
                result_msg.append("  - structural_profile.csv")
                
                StructureShpExporter({}).export(
                    output_folder / "structural_profile.shp",
                    {
                        "structural_data": struct_data,
                        "crs": line_crs,
                        "dip_scale_factor": values["dip_scale_factor"],
                        "raster_res": values["raster_layer_obj"].rasterUnitsPerPixelX(),
                    },
                )
                result_msg.append("  - structural_profile.shp")

            # Export Axes
            logger.info("✓ Saving profile axes...")
            AxesShpExporter({}).export(
                output_folder / "profile_axes.shp",
                {"profile_data": profile_data, "crs": line_crs},
            )
            result_msg.append("  - profile_axes.shp")

            result_msg.append(f"\n✓ All files saved to:\n{output_folder}")
            self.dlg.preview_widget.results_text.setPlainText("\n".join(result_msg))
            QMessageBox.information(
                self.dlg,
                self.tr("Success"),
                self.tr(f"All data saved to:\n{output_folder}"),
            )

        except Exception as e:
            scu.show_user_message(
                self.dlg,
                self.tr("Export Error"),
                self.tr(f"Failed to export data: {e!s}"),
                "error",
            )
            logger.error("Export error in save_profile_line: %s", str(e), exc_info=True)

    def draw_preview(self, topo_data, geol_data=None, struct_data=None, max_points=1000, **kwargs):
        """Draw enhanced interactive preview using native PyQGIS renderer.

        Args:
            topo_data: List of (dist, elev) tuples for topographic profile
            geol_data: Optional list of (dist, elev, geology_name) tuples
            struct_data: Optional list of (dist, app_dip) tuples
            max_points (int): Maximum number of points for simplified preview (LOD)
            **kwargs: Additional arguments passed to renderer (e.g. preserve_extent)
        """
        logger.debug("draw_preview called with:")
        logger.debug("  - topo_data: %d points", len(topo_data) if topo_data else 0)
        logger.debug("  - geol_data: %d points", len(geol_data) if geol_data else 0)
        logger.debug(
            "  - struct_data: %d points", len(struct_data) if struct_data else 0
        )
        logger.debug("  - max_points: %d", max_points)

        # Store data in dialog for re-rendering when checkboxes change
        self.dlg.current_topo_data = topo_data
        self.dlg.current_geol_data = geol_data
        self.dlg.current_struct_data = struct_data

        # Get preview options from dialog checkboxes
        options = self.dlg.get_preview_options()
        logger.debug("Preview options: %s", options)

        # Filter data based on checkbox states
        filtered_topo = topo_data if options.get("show_topo", True) else None
        filtered_geol = geol_data if options.get("show_geol", True) else None
        filtered_struct = struct_data if options.get("show_struct", True) else None

        logger.debug("Filtered data:")
        logger.debug(
            "  - filtered_topo: %d points", len(filtered_topo) if filtered_topo else 0
        )
        logger.debug(
            "  - filtered_geol: %d points", len(filtered_geol) if filtered_geol else 0
        )
        logger.debug(
            "  - filtered_struct: %d points",
            len(filtered_struct) if filtered_struct else 0,
        )

        # Get vertical exaggeration from dialog
        
        # Use new numeric method or direct accessor from Page
        vert_exag = self.dlg.page_dem.vertexag_spin.value()
        
        logger.debug("Vertical exaggeration: %.2f", vert_exag)

        # Calculate dip line length based on scale factor and raster resolution
        dip_line_length = None
        if filtered_struct:
            dip_scale = self.dlg.page_struct.scale_spin.value()

            if dip_scale > 0:
                raster_layer = self.dlg.page_dem.raster_combo.currentLayer()
                if raster_layer and raster_layer.isValid():
                    res = raster_layer.rasterUnitsPerPixelX()
                    if res > 0:
                        dip_line_length = res * dip_scale
            logger.debug("Dip line length: %s (scale: %.2f)", dip_line_length, dip_scale)

        # Render using native PyQGIS
        canvas, layers = self.preview_renderer.render(
            filtered_topo, 
            filtered_geol, 
            filtered_struct, 
            vert_exag, 
            dip_line_length,
            max_points=max_points,
            preserve_extent=kwargs.get('preserve_extent', False)
        )

        # Store canvas and layers for export
        self.dlg.current_canvas = canvas
        self.dlg.current_layers = layers

        # Update legend
        if hasattr(self.dlg, "legend_widget"):
            self.dlg.legend_widget.update_legend(self.preview_renderer)

        logger.debug("Preview rendered with %d layers", len(layers) if layers else 0)
