"""SecInterpDialog — A QGIS plugin.

/***************************************************************************
SecInterpDialog
                                A QGIS plugin
Data extraction for geological interpretation
Generated by Plugin Builder: http://g-sherman.github.io/Qgis-Plugin-Builder/
                            -------------------
       begin                : 2025-11-15
       git sha              : $Format:%H$
       copyright            : (C) 2025 by Juan M Bernales
       email                : juanbernales@gmail.com
***************************************************************************/

/***************************************************************************
*                                                                         *
*   This program is free software; you can redistribute it and/or modify  *
*   it under the terms of the GNU General Public License as published by  *
*   the Free Software Foundation; either version 2 of the License, or     *
*   (at your option) any later version.                                   *
*                                                                         *
***************************************************************************/
"""

from pathlib import Path
import tempfile
import traceback
import webbrowser

from qgis.core import (
    Qgis,
    QgsApplication,
    QgsCoordinateTransform,
    QgsMapLayer,
    QgsMapLayerProxyModel,
    QgsMapSettings,
    QgsPointXY,
    QgsProject,
    QgsSettings,
    QgsUnitTypes,
    QgsWkbTypes,
)
from qgis.gui import QgsFileWidget, QgsMapCanvas
from qgis.PyQt.QtCore import QSize, QVariant
from qgis.PyQt.QtGui import QColor
from qgis.PyQt.QtWidgets import (
    QApplication,
    QDialog,
    QDialogButtonBox,
    QFileDialog,
    QLabel,
    QPushButton,
    QStyle,
)

from sec_interp.core import utils as scu
from sec_interp.core import validation as vu
from sec_interp.exporters import get_exporter
from sec_interp.logger_config import get_logger

from .legend_widget import LegendWidget
from .main_dialog_cache_handler import CacheHandler
from .main_dialog_config import DialogDefaults
from .main_dialog_export import ExportManager
from .main_dialog_preview import PreviewManager
from .main_dialog_validation import DialogValidator
from .ui.main_dialog_base import Ui_SecInterpDialogBase


class SecInterpDialog(QDialog, Ui_SecInterpDialogBase):
    """Dialog for the SecInterp QGIS plugin.

    This dialog provides the user interface and helper methods to populate
    combo boxes with layers from the current QGIS project (raster and vector
    layers filtered by geometry type). It also exposes the interface and
    plugin instance for interaction with the host application.

    Attributes:
    ----------
    iface :
        The QGIS interface instance passed by the host application.
    plugin_instance :
        The plugin instance that created this dialog.
    messagebar :
        The QGIS message bar widget used to display notifications and errors.
    """

    def __init__(self, iface=None, plugin_instance=None, parent=None):
        """Constructor."""
        super().__init__(parent)
        # iface and plugin_instance are optional to make the dialog testable
        # in environments where a QGIS iface is not available.
        self.iface = iface
        self.plugin_instance = plugin_instance
        # Set up the user interface from Designer through the base class.
        self.setupUi(self)
        # Provide a safe, no-op messagebar when iface is not available (tests)
        if self.iface is None:

            class _NoOpMessageBar:
                def pushMessage(self, *_args, **_kwargs):
                    """No-op implementation of pushMessage."""
                    return None

            self.messagebar = _NoOpMessageBar()
        else:
            self.messagebar = self.iface.messageBar()

        # Initialize attributes
        self.action = None

        # Initialize manager instances (composition pattern)
        self.validator = DialogValidator(self)
        self.preview_manager = PreviewManager(self)
        self.export_manager = ExportManager(self)
        self.cache_handler = CacheHandler(self)

        # Set default values from config
        self.scale.setText(DialogDefaults.SCALE)
        self.vertexag.setText(DialogDefaults.VERTICAL_EXAGGERATION)
        self.buffer_distance.setText("100")
        self.dip_scale_factor.setText("4")

        # Connect sidebar navigation
        self.listWidget.currentRowChanged.connect(self.stackedWidget.setCurrentIndex)

        # Set icons for sidebar items
        # DEM / Raster
        self.listWidget.item(0).setIcon(self.getThemeIcon("mIconRaster.svg"))
        # Section Line
        self.listWidget.item(1).setIcon(self.getThemeIcon("mIconLineLayer.svg"))
        # Geology
        self.listWidget.item(2).setIcon(self.getThemeIcon("mIconPolygonLayer.svg"))
        # Structural
        self.listWidget.item(3).setIcon(self.getThemeIcon("mIconPointLayer.svg"))

        # Replace QGraphicsView with QgsMapCanvas for preview
        # Remove the old QGraphicsView widget
        old_preview = self.preview
        old_preview.geometry()
        old_preview.parent()
        # Replace QGraphicsView with QgsMapCanvas for preview
        # Remove the old QGraphicsView widget
        old_preview = self.preview

        # Create new QgsMapCanvas
        self.preview = QgsMapCanvas(old_preview.parent())
        self.preview.setCanvasColor(QColor(255, 255, 255))  # White background

        # Replace in layout
        self.verticalLayout_preview.replaceWidget(old_preview, self.preview)

        old_preview.setParent(None)
        old_preview.deleteLater()

        # Create legend widget
        self.legend_widget = LegendWidget(self.preview)
        self.legend_widget.resize(self.preview.size())

        # Store current preview data for re-rendering when checkboxes change
        self.current_topo_data = None
        self.current_geol_data = None
        self.current_struct_data = None
        self.current_canvas = None
        self.current_layers = []

        # Configure outcrop layer combo box
        self.outcrop.setFilters(QgsMapLayerProxyModel.PolygonLayer)
        self.outcrop.setAllowEmptyLayer(True, "Do not use Outcrop layer")
        self.outcrop.setLayer(None)  # Default to empty
        self.outcrop.layerChanged.connect(self.ocropname.setLayer)

        # Initialize field combo box with current layer (will be None)
        self.ocropname.setLayer(self.outcrop.currentLayer())

        # Configure structural layer combo box
        self.structural.setFilters(QgsMapLayerProxyModel.PointLayer)
        self.structural.setAllowEmptyLayer(True, "Do not use Structural layer")
        self.structural.setLayer(None)  # Default to empty
        self.structural.layerChanged.connect(self.dip.setLayer)
        self.structural.layerChanged.connect(self.strike.setLayer)

        # Initialize field combo boxes with current layer (will be None)
        self.dip.setLayer(self.structural.currentLayer())
        self.strike.setLayer(self.structural.currentLayer())

        # Configure crossline layer combo box
        self.crossline.setFilters(QgsMapLayerProxyModel.LineLayer)
        self.crossline.setAllowEmptyLayer(True, "Select Crossline Layer")
        self.crossline.setLayer(None)  # Default to empty
        self.crossline.layerChanged.connect(self.update_button_state)

        # Configure output folder widget
        self.dest_fold.setStorageMode(QgsFileWidget.GetDirectory)
        self.dest_fold.setDialogTitle("Select Output Folder")

        # Configure raster layer combo box
        self.rasterdem.setFilters(QgsMapLayerProxyModel.RasterLayer)
        self.rasterdem.setAllowEmptyLayer(True, "Select DEM Layer")
        self.rasterdem.setLayer(None)  # Default to empty

        # Connect raster layer change to band combo box
        self.rasterdem.layerChanged.connect(self.band.setLayer)

        # Initialize band combo box with current raster if any
        if self.rasterdem.currentLayer():
            self.band.setLayer(self.rasterdem.currentLayer())

        # Update resolution only if iface or canvas isn't required (guarded in method)
        self.update_resolution_field()
        self.rasterdem.layerChanged.connect(self.update_resolution_field)
        self.rasterdem.layerChanged.connect(self.update_button_state)

        # Add Clear Cache button
        self.clear_cache_btn = QPushButton("Clear Cache")
        self.clear_cache_btn.setToolTip(
            "Clear cached data to force re-processing.\n"
            "Use this if you've changed data outside the plugin."
        )
        self.clear_cache_btn.clicked.connect(self.clear_cache_handler)

        # Add button to button box
        self.button_box.addButton(self.clear_cache_btn, QDialogButtonBox.ActionRole)

        # Add cache status label
        self.cache_status_label = QLabel("")
        self.cache_status_label.setStyleSheet("color: #666; font-size: 10px;")

        # Connect signals
        self.button_box.accepted.connect(self.accept_handler)
        self.button_box.rejected.connect(self.reject_handler)
        self.button_box.helpRequested.connect(self.open_help)
        self.dest_fold.fileChanged.connect(self.update_button_state)
        self.preview_button.clicked.connect(self.preview_profile_handler)
        self.update_button_state()

        # Connect preview checkboxes to update preview when toggled
        self.show_topo_cb.stateChanged.connect(self.update_preview_from_checkboxes)
        self.show_geol_cb.stateChanged.connect(self.update_preview_from_checkboxes)
        self.show_struct_cb.stateChanged.connect(self.update_preview_from_checkboxes)

        # Connect export button
        self.export_pre.clicked.connect(self.export_preview)

        # Setup required field indicators
        self._setup_required_field_indicators()

        # Set column stretch for grid layouts to minimize icon/label spacing
        # DEM section: columns 0,1 (icon, label) should not stretch, rest can expand
        self.gridLayout_dem.setColumnStretch(0, 0)  # icon
        self.gridLayout_dem.setColumnStretch(1, 0)  # label
        self.gridLayout_dem.setColumnStretch(2, 1)  # combo (expand)

        # Section line: columns 0,1 (icon, label) should not stretch, combo expands
        self.gridLayout_section.setColumnStretch(0, 0)  # icon
        self.gridLayout_section.setColumnStretch(1, 0)  # label
        self.gridLayout_section.setColumnStretch(2, 1)  # combo (expand)

        # Load user settings from previous session
        self._load_user_settings()

    def wheelEvent(self, event):
        """Handle mouse wheel for zooming in preview."""
        # Check if mouse is over preview widget
        if self.preview.underMouse():
            # QgsMapCanvas has built-in zoom with wheel
            # We can customize the zoom factor if needed
            if event.angleDelta().y() > 0:
                self.preview.zoomIn()
            else:
                self.preview.zoomOut()
            event.accept()
        else:
            super().wheelEvent(event)

    def open_help(self):
        """Open the help file in the default browser."""
        help_file = Path(__file__).parent / "help" / "html" / "index.html"
        if help_file.exists():
            webbrowser.open(help_file.as_uri())
        else:
            self.messagebar.pushMessage(
                "Error",
                "Help file not found. Please run 'make doc' to generate it.",
                level=Qgis.Warning,
            )

    def update_button_state(self):
        """Enable or disable buttons based on input validity.

        - Preview and Ok buttons require: DEM + Cross-section line
        - Save button requires: DEM + Cross-section line + Output path
        """
        has_output_path = bool(self.dest_fold.filePath())
        has_raster = bool(self.rasterdem.currentLayer())
        has_line = bool(self.crossline.currentLayer())

        # Preview and Ok require at least DEM and line
        can_preview = has_raster and has_line

        # Save requires DEM, line, and output path
        can_save = can_preview and has_output_path

        # Update Preview button
        preview_btn = self.preview_button
        if preview_btn:
            preview_btn.setEnabled(can_preview)

        # Update Ok button
        ok_btn = self.button_box.button(QDialogButtonBox.Ok)
        if ok_btn:
            ok_btn.setEnabled(can_preview)

        # Update Save button
        save_btn = self.button_box.button(QDialogButtonBox.Save)
        if save_btn:
            save_btn.setEnabled(can_save)

    def get_selected_values(self):
        """Get the selected values from the dialog with safe type conversion."""
        # Safely convert numeric inputs with defaults
        _, _, scale = vu.validate_numeric_input(
            self.scale.text(), field_name="Scale", allow_empty=True
        )
        _, _, vertexag = vu.validate_numeric_input(
            self.vertexag.text(), field_name="Vertical exaggeration", allow_empty=True
        )
        _, _, buffer_dist = vu.validate_numeric_input(
            self.buffer_distance.text(), field_name="Buffer distance", allow_empty=True
        )
        _, _, dip_scale = vu.validate_numeric_input(
            self.dip_scale_factor.text(),
            field_name="Dip scale factor",
            allow_empty=True,
        )
        # Band number is now handled by QgsRasterBandComboBox which returns int band number directly via currentBand()
        # But for consistency with existing code structure we can get it here
        band_num = self.band.currentBand()

        return {
            "raster_layer": self.rasterdem.currentLayer(),
            "outcrop_layer": self.outcrop.currentLayer(),
            "structural_layer": self.structural.currentLayer(),
            "crossline_layer": self.crossline.currentLayer(),
            "dip_field": self.dip.currentField(),
            "strike_field": self.strike.currentField(),
            "outcrop_name_field": self.ocropname.currentField(),
            "scale": scale if scale is not None else 50000,
            "vertexag": vertexag if vertexag is not None else 1.0,
            "selected_band": band_num if band_num is not None else 1,
            "buffer_distance": buffer_dist if buffer_dist is not None else 100.0,
            "dip_scale_factor": dip_scale if dip_scale is not None else 4.0,
            "output_path": self.dest_fold.filePath(),
        }

    def get_preview_options(self):
        """Return the state of preview layer checkboxes.

        Returns:
            dict: Keys 'show_topo', 'show_geol', 'show_struct' with boolean values.
        """
        return {
            "show_topo": bool(self.show_topo_cb.isChecked()),
            "show_geol": bool(self.show_geol_cb.isChecked()),
            "show_struct": bool(self.show_struct_cb.isChecked()),
        }

    def update_preview_from_checkboxes(self):
        """Update preview when checkboxes change.

        This method delegates to PreviewManager for preview updates.
        """
        self.preview_manager.update_from_checkboxes()

    def preview_profile_handler(self):
        """Generate a quick preview with topographic, geological, and structural data.

        This method delegates to PreviewManager for preview generation.
        """
        success, message = self.preview_manager.generate_preview()
        if not success and message:
            self.messagebar.pushMessage("Preview Error", message, level=2)

    def export_preview(self):
        """Export the current preview to a file using dedicated exporters."""
        if not self.current_layers or not self.current_canvas:
            self.results.setPlainText(
                "⚠ No preview to export. Please generate a preview first."
            )
            return

        # Determine default directory
        settings = QgsSettings()
        dest_folder = self.dest_fold.filePath().strip()

        if dest_folder:
            default_path = str(Path(dest_folder) / "preview.png")
        else:
            last_dir = settings.value("SecInterp/lastExportDir", "", type=str)
            default_path = (
                str(Path(last_dir) / "preview.png") if last_dir else "preview.png"
            )

        filename, selected_filter = QFileDialog.getSaveFileName(
            self,
            "Export Preview",
            default_path,
            "PNG Image (*.png);;JPEG Image (*.jpg);;Scalable Vector Graphics (*.svg);;PDF Documents (*.pdf)",
        )

        if not filename:
            return

        # Determine the correct extension
        filter_ext_map = {
            "PNG Image (*.png)": ".png",
            "JPEG Image (*.jpg)": ".jpg",
            "Scalable Vector Graphics (*.svg)": ".svg",
            "PDF Documents (*.pdf)": ".pdf",
        }

        current_ext = Path(filename).suffix.lower()
        expected_ext = filter_ext_map.get(selected_filter, "")

        if (
            not current_ext
            or current_ext
            not in [
                ".png",
                ".jpg",
                ".jpeg",
                ".svg",
                ".pdf",
            ]
        ) and expected_ext:
            filename = str(Path(filename).with_suffix(expected_ext))

        ext = Path(filename).suffix.lower()
        settings.setValue("SecInterp/lastExportDir", str(Path(filename).parent))

        try:
            self.results.setPlainText(f"Exporting preview to {filename}...")

            # Get extent from canvas
            extent = self.current_canvas.extent()

            # Export dimensions (use canvas size as reference)
            width = self.preview.width()
            height = self.preview.height()
            dpi = 96

            # For raster formats, use higher resolution
            if ext in [".png", ".jpg", ".jpeg"]:
                width = width * 3  # 3x resolution for better quality
                height = height * 3
                dpi = 300

            # Prepare export settings
            export_settings = {
                "width": width,
                "height": height,
                "dpi": dpi,
                "background_color": QColor(255, 255, 255),
                "legend_renderer": (
                    self.plugin_instance.preview_renderer
                    if hasattr(self.plugin_instance, "preview_renderer")
                    else None
                ),
                "title": "Section Interpretation Preview",
                "description": "Generated by SecInterp QGIS Plugin",
            }

            # Get map settings
            map_settings = self._get_export_settings(width, height, dpi, extent)

            # Use exporter factory
            exporter = get_exporter(ext, export_settings)
            success = exporter.export(Path(filename), map_settings)

            if success:
                self.results.setPlainText(
                    f"✓ Preview exported successfully to:\n{filename}"
                )
            else:
                self.results.setPlainText(f"⚠ Error exporting preview to {filename}")

        except ValueError as e:
            self.results.setPlainText(f"⚠ {e!s}")
        except Exception as e:
            error_details = traceback.format_exc()
            self.results.setPlainText(
                f"⚠ Error exporting preview: {e!s}\n\nDetails:\n{error_details}"
            )

    def accept_handler(self):
        """Handle the accept button click event."""
        # When running without a QGIS iface (tests), skip strict validation
        if self.iface is None:
            self.accept()
            return

        if not self.validate_inputs():
            return

        # Save user settings before closing
        self._save_user_settings()
        self.accept()

    def reject_handler(self):
        """Handle the reject button click event."""
        self.close()

    def validate_inputs(self):
        """Validate the inputs from the dialog.

        This method delegates to DialogValidator for input validation.
        """
        is_valid, error_message = self.validator.validate_inputs()
        if not is_valid:
            scu.show_user_message(self, "Validation Error", error_message)
        return is_valid

    def clear_cache_handler(self):
        """Clear cached data and notify user."""
        if hasattr(self, "plugin_instance") and self.plugin_instance:
            self.plugin_instance.data_cache.clear()
            self.results.append("✓ Cache cleared - next preview will re-process data")
            # context specific usage
            logger = get_logger(__name__)
            logger.info("Cache cleared by user")
        else:
            self.results.append("⚠ Cache not available")

    def _populate_field_combobox(self, source_combobox, target_combobox):
        """Helper function to populate a combobox with field names from a selected vector layer."""
        try:
            selected_layer_name = source_combobox.currentData()
            target_combobox.clear()

            if not selected_layer_name:
                return

            layers = QgsProject.instance().mapLayersByName(selected_layer_name)
            if not layers:
                return

            vector_layer = layers[0]
            # Use addItem with data parameter so currentData() returns the field name
            for field in vector_layer.fields():
                target_combobox.addItem(field.name(), field.name())
        except Exception as e:
            self.messagebar.pushMessage("Error", str(e), level=Qgis.Critical)

    def get_layer_names_by_type(self, layer_type) -> list[str]:
        """Get a list of layer names filtered by the specified layer type.

        This method scans all layers in the current project.

        Args:
              layer_type: The QgsMapLayer type to filter by (e.g.,
        QgsMapLayer.RasterLayer)

        Returns:
              A list of layer names matching the specified type
        """
        layers = QgsProject.instance().mapLayers().values()
        return [layer.name() for layer in layers if layer.type() == layer_type]

    def get_layer_names_by_geometry(self, geometry_type) -> list[str]:
        """Get a list of layer names filtered by the specified geometry type.

        This method scans all layers in the current project.

        Args:
              geometry_type: The QgsWkbTypes geometry type to filter by
        (e.g., QgsWkbTypes.LineGeometry)

        Returns:
              A list of layer names matching the specified geometry type
        """
        layers = QgsProject.instance().mapLayers().values()
        return [
            layer.name()
            for layer in layers
            if layer.type() == QgsMapLayer.VectorLayer
            and QgsWkbTypes.geometryType(layer.wkbType()) == geometry_type
        ]

    def update_resolution_field(self):
        """Calculate and update the resolution field for the selected raster layer."""
        try:
            raster_layer = self.rasterdem.currentLayer()
            if not raster_layer:
                self.resln.clear()
                self.unts.clear()
                return
            raster_crs = raster_layer.crs()
            # Determine map CRS safely: if iface or canvas isn't available
            # (e.g., during unit tests), fall back to using raster CRS.
            try:
                if (
                    self.iface is None
                    or not hasattr(self.iface, "mapCanvas")
                    or self.iface.mapCanvas() is None
                ):
                    map_crs = raster_crs
                else:
                    map_crs = self.iface.mapCanvas().mapSettings().destinationCrs()
            except AttributeError:
                # Defensive: if anything goes wrong accessing iface/canvas, use raster CRS.
                map_crs = raster_crs
            native_res = raster_layer.rasterUnitsPerPixelX()

            # Validate native resolution
            if native_res <= 0:
                self.resln.setText("Invalid")
                self.unts.setText("")
                self.messagebar.pushMessage(
                    "Warning",
                    "Raster resolution is invalid or zero",
                    level=Qgis.Warning,
                )
                return

            resolution_in_meters = 0

            if raster_crs == map_crs:
                if raster_crs.isGeographic():
                    self.resln.setText(f"{native_res:.6f}")
                    self.unts.setText("°")
                    # For scale calculation, we need resolution in meters.
                    # This is an approximation at the equator.
                    resolution_in_meters = native_res * 111320
                else:
                    self.resln.setText(f"{native_res:.2f}")
                    self.unts.setText(QgsUnitTypes.toString(raster_crs.mapUnits()))
                    if raster_crs.mapUnits() == QgsUnitTypes.DistanceUnit.Meters:
                        resolution_in_meters = native_res
                    elif raster_crs.mapUnits() == QgsUnitTypes.DistanceUnit.Feet:
                        resolution_in_meters = native_res * 0.3048

            else:
                transform = QgsCoordinateTransform(
                    raster_crs, map_crs, QgsProject.instance()
                )
                center = raster_layer.extent().center()
                p1 = center
                p2 = QgsPointXY(center.x() + native_res, center.y())
                p1_transformed = transform.transform(p1)
                p2_transformed = transform.transform(p2)

                if map_crs.isGeographic():
                    # Raster is projected, map is geographic
                    resolution = abs(p2_transformed.x() - p1_transformed.x())
                    self.resln.setText(f"{resolution:.6f}")
                    self.unts.setText("°")
                    # We need to transform back to the raster's projected CRS to get meters
                    # This is a bit complex, let's use the native resolution for scale for now
                    if raster_crs.mapUnits() == QgsUnitTypes.DistanceUnit.Meters:
                        resolution_in_meters = native_res
                    elif raster_crs.mapUnits() == QgsUnitTypes.DistanceUnit.Feet:
                        resolution_in_meters = native_res * 0.3048
                else:
                    # Raster is geographic, map is projected
                    resolution = p1_transformed.distance(p2_transformed)
                    self.resln.setText(f"{resolution:.2f}")
                    self.unts.setText(QgsUnitTypes.toString(map_crs.mapUnits()))
                    if map_crs.mapUnits() == QgsUnitTypes.DistanceUnit.Meters:
                        resolution_in_meters = resolution
                    elif map_crs.mapUnits() == QgsUnitTypes.DistanceUnit.Feet:
                        resolution_in_meters = resolution * 0.3048

            # Calculate suggested scale only if resolution is valid
            if resolution_in_meters > 0:
                suggested_scale = round((resolution_in_meters * 2000) / 1000) * 1000
                self.scale.setText(str(suggested_scale))

        except Exception as e:
            self.messagebar.pushMessage("Error", str(e), level=Qgis.Critical)

    def getThemeIcon(self, name):
        """Get a theme icon from QGIS.

        :param name: The name of the icon (e.g. 'mActionFileOpen.svg')
        :return: QIcon
        """
        return QgsApplication.getThemeIcon(name)

    def _validate_preview_requirements(self):
        """Validate inputs required for preview generation.

        Returns:
            tuple: (raster_layer, line_layer, band_num) if valid.

        Raises:
            ValueError: If validation fails, with a user-friendly message.
        """
        if self.plugin_instance is None:
            raise ValueError("Plugin instance not initialized.")

        raster_layer = self.rasterdem.currentLayer()
        if not raster_layer:
            raise ValueError("Please select a raster layer for preview.")

        line_layer = self.crossline.currentLayer()
        if not line_layer:
            raise ValueError("Please select a cross-section line for preview.")

        band_num = self.band.currentBand()
        if not band_num:
            raise ValueError("Please select a band number.")

        is_valid, error = vu.validate_raster_band(raster_layer, band_num)
        if not is_valid:
            raise ValueError(error)

        return raster_layer, line_layer, band_num

    def _generate_topography(self, line_layer, raster_layer, band_num):
        """Generate topographic profile data."""
        return self.plugin_instance.profile_service.generate_topographic_profile(
            line_layer, raster_layer, band_num
        )

    def _generate_geology(self, line_layer, raster_layer, band_num):
        """Generate geological profile data if outcrop layer is selected."""
        logger = get_logger(__name__)

        outcrop_layer = self.outcrop.currentLayer()
        if not outcrop_layer:
            logger.debug("No outcrop layer selected")
            return None

        outcrop_name_field = self.ocropname.currentField()
        if not outcrop_name_field:
            logger.debug("No outcrop name field selected")
            return None

        try:
            logger.info(
                f"Generating geological profile with field: {outcrop_name_field}"
            )
            result = self.plugin_instance.geology_service.generate_geological_profile(
                line_layer,
                raster_layer,
                outcrop_layer,
                outcrop_name_field,
                band_num,
            )
            logger.info(
                f"Geological profile result: {len(result) if result else 0} points"
            )
            return result
        except Exception as e:
            logger.error(f"Error generating geological profile: {e}", exc_info=True)
            return None

    def _generate_structures(self, line_layer, buffer_dist):
        """Generate structural data if structural layer is selected."""
        structural_layer = self.structural.currentLayer()
        if not structural_layer:
            return None

        dip_field = self.dip.currentField()
        strike_field = self.strike.currentField()

        if not dip_field or not strike_field:
            return None

        # Get line azimuth
        line_feat = next(line_layer.getFeatures(), None)
        if not line_feat:
            return None

        line_geom = line_feat.geometry()
        line_azimuth = scu.calculate_line_azimuth(line_geom)

        return self.plugin_instance.structure_service.project_structures(
            line_layer,
            structural_layer,
            buffer_dist,
            line_azimuth,
            dip_field,
            strike_field,
        )

    def _get_export_settings(self, width, height, dpi, extent):
        """Create QgsMapSettings for export.

        Args:
            width: Output width in pixels
            height: Output height in pixels
            dpi: Dots per inch
            extent: Map extent (QgsRectangle)

        Returns:
            Configured QgsMapSettings instance
        """
        settings = QgsMapSettings()
        settings.setLayers(self.current_layers)
        settings.setExtent(extent)
        settings.setOutputSize(QSize(width, height))
        settings.setOutputDpi(dpi)
        settings.setBackgroundColor(QColor(255, 255, 255))
        return settings

    def _setup_required_field_indicators(self):
        """Setup required field indicators with warning icons.

        Initializes status icons for required fields (Raster Layer and Section Line)
        and connects signals to update icons when fields are filled/cleared.
        """
        # Get standard Qt icons which are more reliable
        style = QApplication.style()
        self.warning_icon = style.standardIcon(QStyle.SP_MessageBoxWarning)
        self.success_icon = style.standardIcon(QStyle.SP_DialogApplyButton)

        # Set initial warning icons
        self.raster_status_icon.setPixmap(self.warning_icon.pixmap(16, 16))
        self.section_status_icon.setPixmap(self.warning_icon.pixmap(16, 16))

        # Connect signals to update status
        self.rasterdem.layerChanged.connect(self._update_raster_status)
        self.crossline.layerChanged.connect(self._update_section_status)

        # Initial update
        self._update_raster_status()
        self._update_section_status()

    def _update_raster_status(self):
        """Update raster layer status icon based on selection."""
        if self.rasterdem.currentLayer():
            self.raster_status_icon.setPixmap(self.success_icon.pixmap(16, 16))
        else:
            self.raster_status_icon.setPixmap(self.warning_icon.pixmap(16, 16))

    def _update_section_status(self):
        """Update section line status icon based on selection."""
        if self.crossline.currentLayer():
            self.section_status_icon.setPixmap(self.success_icon.pixmap(16, 16))
        else:
            self.section_status_icon.setPixmap(self.warning_icon.pixmap(16, 16))

    def _load_user_settings(self):
        """Load user settings from previous session.

        Restores numeric values (scale, vertical exaggeration, buffer distance,
        dip scale factor) and output folder path from QgsSettings.
        """
        settings = QgsSettings()

        # Load numeric values with validation
        try:
            scale = settings.value("SecInterp/scale", 50000, type=int)
            # Validate reasonable range (1:1,000 to 1:1,000,000)
            if 1000 <= scale <= 1000000:
                self.scale.setText(str(scale))
            else:
                self.scale.setText("50000")
        except (ValueError, TypeError):
            self.scale.setText("50000")

        try:
            vertexag = settings.value("SecInterp/vertexag", 1.0, type=float)
            # Validate reasonable range (0.1 to 100)
            if 0.1 <= vertexag <= 100.0:
                self.vertexag.setText(str(vertexag))
            else:
                self.vertexag.setText("1.0")
        except (ValueError, TypeError):
            self.vertexag.setText("1.0")

        try:
            buffer_dist = settings.value("SecInterp/bufferDistance", 100.0, type=float)
            # Validate reasonable range (0 to 10,000)
            if 0.0 <= buffer_dist <= 10000.0:
                self.buffer_distance.setText(str(buffer_dist))
            else:
                self.buffer_distance.setText("100.0")
        except (ValueError, TypeError):
            self.buffer_distance.setText("100.0")

        try:
            dip_scale = settings.value("SecInterp/dipScaleFactor", 4.0, type=float)
            # Validate reasonable range (0.1 to 20)
            if 0.1 <= dip_scale <= 20.0:
                self.dip_scale_factor.setText(str(dip_scale))
            else:
                self.dip_scale_factor.setText("4.0")
        except (ValueError, TypeError):
            self.dip_scale_factor.setText("4.0")

        # Load output folder
        last_output = settings.value("SecInterp/lastOutputFolder", "", type=str)
        if last_output:
            self.dest_fold.setFilePath(last_output)

    def _save_user_settings(self):
        """Save user settings for next session.

        Persists numeric values (scale, vertical exaggeration, buffer distance,
        dip scale factor) and output folder path to QgsSettings.
        """
        settings = QgsSettings()

        # Save numeric values (only if valid)
        try:
            if self.scale.text():
                scale_val = int(self.scale.text())
                if 1000 <= scale_val <= 1000000:
                    settings.setValue("SecInterp/scale", scale_val)
        except ValueError:
            pass  # Don't save invalid values

        try:
            if self.vertexag.text():
                vertexag_val = float(self.vertexag.text())
                if 0.1 <= vertexag_val <= 100.0:
                    settings.setValue("SecInterp/vertexag", vertexag_val)
        except ValueError:
            pass

        try:
            if self.buffer_distance.text():
                buffer_val = float(self.buffer_distance.text())
                if 0.0 <= buffer_val <= 10000.0:
                    settings.setValue("SecInterp/bufferDistance", buffer_val)
        except ValueError:
            pass

        try:
            if self.dip_scale_factor.text():
                dip_scale_val = float(self.dip_scale_factor.text())
                if 0.1 <= dip_scale_val <= 20.0:
                    settings.setValue("SecInterp/dipScaleFactor", dip_scale_val)
        except ValueError:
            pass

        # Save output folder
        if self.dest_fold.filePath():
            settings.setValue("SecInterp/lastOutputFolder", self.dest_fold.filePath())
