"""/***************************************************************************
 SecInterp
                                 A QGIS plugin
 Data extraction for geological interpretation.
 Generated by Plugin Builder: http://g-sherman.github.io/Qgis-Plugin-Builder/.
                              -------------------
        begin                : 2025-11-15
        git sha              : $Format:%H$
        copyright            : (C) 2025 by Juan M Bernales
        email                : juanbernales@gmail.com
 ***************************************************************************/

/***************************************************************************
 *                                                                         *
 *   This program is free software; you can redistribute it and/or modify  *
 *   it under the terms of the GNU General Public License as published by  *
 *   the Free Software Foundation; either version 2 of the License, or     *
 *   (at your option) any later version.                                   *
 *                                                                         *
 ***************************************************************************/
"""

import contextlib
from pathlib import Path

from qgis.core import (
    QgsGeometry,
    QgsRaster,
    QgsRasterLayer,
    QgsVectorLayer,
    QgsWkbTypes,
)
from qgis.PyQt.QtCore import (
    QCoreApplication,
    QSettings,
    QTranslator,
)
from qgis.PyQt.QtGui import QIcon
from qgis.PyQt.QtWidgets import QAction, QDialogButtonBox, QMessageBox

from sec_interp.core import utils as scu
from sec_interp.core import validation as vu
from sec_interp.exporters import (
    AxesShpExporter,
    CSVExporter,
    GeologyShpExporter,
    ProfileLineShpExporter,
    StructureShpExporter,
)
from sec_interp.gui.main_dialog import SecInterpDialog
from sec_interp.gui.preview_renderer import PreviewRenderer
from sec_interp.logger_config import get_logger

from .data_cache import DataCache
from .services import GeologyService, ProfileService, StructureService


logger = get_logger(__name__)


# DataCache has been moved to core/data_cache.py


class SecInterp:
    """QGIS Plugin Implementation for Geological Data Extraction.

    This class implements the main logic of the SecInterp plugin, handling
    initialization, UI integration, and orchestration of data processing tasks.
    It connects the QGIS interface with the plugin's dialog and processing algorithms.
    """

    def __init__(self, iface):
        """Constructor.

        Args:
            iface (QgsInterface): An interface instance that will be passed to this class
                which provides the hook by which you can manipulate the QGIS
                application at run time.
        """
        # Save reference to the QGIS interface
        self.iface = iface
        # initialize plugin directory
        self.plugin_dir = Path(__file__).resolve().parent
        # initialize locale
        locale = QSettings().value("locale/userLocale")[0:2]
        locale_path = self.plugin_dir / f"i18n/SecInterp_{locale}.qm"

        if locale_path.exists():
            self.translator = QTranslator()
            self.translator.load(str(locale_path))
            QCoreApplication.installTranslator(self.translator)

        # Create the dialog (after translation) and keep reference
        self.dlg = SecInterpDialog(self.iface, self)
        self.dlg.plugin_instance = self

        # Create preview renderer
        self.preview_renderer = PreviewRenderer()

        # Initialize services
        self.profile_service = ProfileService()
        self.geology_service = GeologyService()
        self.structure_service = StructureService()

        # Initialize data cache for performance
        self.data_cache = DataCache()
        logger.debug("Data cache initialized")

        # Declare instance attributes
        self.actions = []
        self.menu = self.tr("&Sec Interp")
        # Create custom toolbar and make it visible
        self.toolbar = self.iface.addToolBar("Sec Interp")
        self.toolbar.setObjectName("SecInterp")
        self.toolbar.setVisible(True)  # Ensure toolbar is visible

        # Check if plugin was started the first time in current QGIS session
        # Must be set in initGui() to survive plugin reloads
        self.first_start = None

    # noinspection PyMethodMayBeStatic
    def tr(self, message):
        """Get the translation for a string using Qt translation API.

        We implement this ourselves since we do not inherit QObject.

        Args:
            message (str): String for translation.

        Returns:
            str: Translated string (or original if no translation found).
        """
        # noinspection PyTypeChecker,PyArgumentList,PyCallByClass
        return QCoreApplication.translate("SecInterp", message)

    def add_action(
        self,
        icon_path,
        text,
        callback,
        enabled_flag=True,
        add_to_menu=True,
        add_to_toolbar=True,
        status_tip=None,
        whats_this=None,
        parent=None,
    ):
        """Add a toolbar icon to the toolbar.

        Args:
            icon_path (str): Path to the icon for this action.
            text (str): Text that should be shown in menu items for this action.
            callback (function): Function to be called when the action is triggered.
            enabled_flag (bool): A flag indicating if the action should be enabled
                by default. Defaults to True.
            add_to_menu (bool): Flag indicating whether the action should also
                be added to the menu. Defaults to True.
            add_to_toolbar (bool): Flag indicating whether the action should also
                be added to the toolbar. Defaults to True.
            status_tip (str): Optional text to show in a popup when mouse pointer
                hovers over the action.
            whats_this (str): Optional text to show in the status bar when the
                mouse pointer hovers over the action.
            parent (QWidget): Parent widget for the new action. Defaults None.

        Returns:
            QAction: The action that was created. Note that the action is also
                added to self.actions list.
        """
        icon = QIcon(icon_path)
        action = QAction(icon, text, parent)
        action.triggered.connect(callback)
        action.setEnabled(enabled_flag)

        if status_tip is not None:
            action.setStatusTip(status_tip)

        if whats_this is not None:
            action.setWhatsThis(whats_this)

        if add_to_toolbar:
            # Add to custom toolbar
            self.toolbar.addAction(action)
            # Also add to Plugins toolbar for visibility
            self.iface.addToolBarIcon(action)

        if add_to_menu:
            self.iface.addPluginToMenu(self.menu, action)

        self.actions.append(action)

        return action

    def initGui(self):
        """Create the menu entries and toolbar icons inside the QGIS GUI."""
        # Use absolute path to icon file to ensure it loads in toolbar
        icon_path = str(self.plugin_dir.parent / "icon.png")
        self.add_action(
            icon_path,
            text=self.tr("Geological data extraction"),
            callback=self.run,
            parent=self.iface.mainWindow(),
        )

        # will be set False in run()
        self.first_start = True

    def unload(self):
        """Removes the plugin menu item and icon from QGIS GUI."""
        for action in self.actions:
            self.iface.removePluginMenu(self.tr("&Sec Interp"), action)
            self.iface.removeToolBarIcon(action)

        # Remove custom toolbar
        if self.toolbar:
            del self.toolbar
            self.toolbar = None

    def run(self):
        """Run method that performs all the real work.

        This method initializes the dialog, connects signals and slots,
        and executes the main event loop for the plugin dialog.
        """
        # Create the dialog with elements (after translation) and keep reference
        # Only create GUI ONCE in callback, so that it will only load when the
        # plugin is started
        if self.first_start is True:
            self.first_start = False
            self.dlg = SecInterpDialog(self.iface, self)
            # Update preview renderer with the new dialog's canvas
            self.preview_renderer.canvas = self.dlg.preview
        # show the dialog
        self.dlg.show()

        # Disconnect default accepted signal to prevent closing on Save
        with contextlib.suppress(TypeError):
            self.dlg.button_box.accepted.disconnect()

        # Connect OK button to process and close
        self.dlg.button_box.button(QDialogButtonBox.Ok).clicked.connect(
            self.process_data
        )
        self.dlg.button_box.button(QDialogButtonBox.Ok).clicked.connect(self.dlg.accept)

        # Connect Save button to save only
        self.dlg.button_box.button(QDialogButtonBox.Save).clicked.connect(
            self.save_profile_line
        )
        # Run the dialog event loop
        result = self.dlg.exec_()
        # See if OK was pressed
        if result:
            # Do something useful here - delete the line containing pass and
            # substitute with your code.
            pass

    def _get_and_validate_inputs(self):
        """Retrieve and validate inputs from the dialog.

        Returns:
            dict: Validated inputs including layer objects, or None if validation fails.
        """
        # Get values from the dialog using the helper method
        values = self.dlg.get_selected_values()

        is_valid, error_msg, validated_values = vu.validate_and_get_layers(values)

        if not is_valid:
            scu.show_user_message(self.dlg, self.tr("Error"), self.tr(error_msg))
            return None

        return validated_values

    def process_data(self):
        """Process profile data from selected layers with caching.

        Uses cache to avoid re-processing when only visualization
        parameters change.
        """
        values = self.dlg.get_selected_values()

        is_valid, error_msg, validated_values = vu.validate_and_get_layers(values)

        if not is_valid:
            scu.show_user_message(self.dlg, self.tr("Error"), self.tr(error_msg))
            return None

        # Check for reasonable parameter ranges
        warnings = vu.validate_reasonable_ranges(validated_values)
        if warnings:
            logger.warning("Parameter range warnings: %s", warnings)
            self.dlg.results.append("\n⚠️ Validation Warnings:")
            for warning in warnings:
                self.dlg.results.append(f"  {warning}")
            self.dlg.results.append("")  # Empty line

        # Show CRS warning if present
        if "_crs_warning" in validated_values:
            self.dlg.results.append("\n" + validated_values["_crs_warning"])
            self.dlg.results.append("")  # Empty line

        # Check if we can use cached data
        cache_key = self.data_cache.get_cache_key(validated_values)
        cached_data = self.data_cache.get(cache_key)

        profile_data = None
        geol_data = None
        struct_data = None

        if cached_data:
            logger.info("⚡ Using cached profile data")
            self.dlg.results.append("⚡ Using cached data (fast!)")
            profile_data = cached_data["profile_data"]
            geol_data = cached_data.get("geol_data")
            struct_data = cached_data.get("struct_data")
        else:
            logger.info("🔄 Processing new profile data...")
            self.dlg.results.append("🔄 Processing data...")

            # Process data (existing code)
            profile_data, geol_data, struct_data = self._process_profile_data(
                validated_values
            )

            if not profile_data:
                scu.show_user_message(
                    self.dlg,
                    self.tr("Error"),
                    self.tr(
                        "No profile data was generated. "
                        "Check that the line intersects the raster."
                    ),
                )
                return

            # Cache the processed data
            self.data_cache.set(
                cache_key,
                {
                    "profile_data": profile_data,
                    "geol_data": geol_data,
                    "struct_data": struct_data,
                },
            )
            logger.info("✓ Data cached for future use")

        # Draw preview with current visualization parameters
        self.draw_preview(profile_data, geol_data, struct_data)

        return profile_data, geol_data, struct_data

    def _process_profile_data(self, values):
        """Main data processing method triggered by the OK button.

        This method orchestrates the entire data extraction workflow:
        1. Validates user inputs (layers, fields, parameters).
        2. Generates the topographic profile.
        3. Generates the geological profile (if selected).
        4. Projects structural data onto the section (if selected).
        5. Updates the results text area.
        6. Draws the interactive preview.

        Handles various exceptions and displays appropriate error messages to the user.
        """
        try:
            # Get and validate inputs
            inputs = self._get_and_validate_inputs()
            if not inputs:
                return

            # Extract values for easier access
            values = inputs
            raster_layer = inputs["raster_layer_obj"]
            line_layer = inputs["line_layer_obj"]
            outcrop_layer = inputs["outcrop_layer_obj"]
            structural_layer = inputs["structural_layer_obj"]

            selected_band = values["selected_band"]
            buffer_dist = values["buffer_distance"]

            # Process topographic profile
            profile_data = self.profile_service.generate_topographic_profile(
                line_layer, raster_layer, selected_band
            )

            # Validate profile data was generated
            if not profile_data:
                scu.show_user_message(
                    self.dlg,
                    self.tr("Error"),
                    self.tr(
                        "No profile data was generated. Check that the line intersects the raster."
                    ),
                )
                return
                return [], [], []  # Return empty lists if no profile data

            # Initialize result text
            result_text = f"✓ Data processed successfully!\n\nTopography: {len(profile_data)} points"
            self.dlg.results.setPlainText(result_text)

            # Process outcrop data
            if outcrop_layer:
                outcrop_name_field = values["outcrop_name_field"]
                if outcrop_name_field:
                    geol_data = self.geology_service.generate_geological_profile(
                        line_layer,
                        raster_layer,
                        outcrop_layer,
                        outcrop_name_field,
                        selected_band,
                    )

                    if geol_data:
                        result_text += f"\nGeology: {len(geol_data)} points"
                        self.dlg.results.setPlainText(result_text)
                    else:
                        result_text += "\nGeology: No intersections"
                        self.dlg.results.setPlainText(result_text)
                else:
                    result_text += (
                        "\n⚠ Outcrop layer selected but no geology field specified."
                    )
                    self.dlg.results.setPlainText(result_text)

            # Process structural data
            if structural_layer:
                dip_field = values["dip_field"]
                strike_field = values["strike_field"]

                if dip_field and strike_field:
                    # Get the azimuth of the cross-section line
                    line_feat = next(line_layer.getFeatures(), None)
                    if not line_feat:
                        scu.show_user_message(
                            self.dlg,
                            self.tr("Error"),
                            self.tr("Line layer has no features."),
                        )
                        return

                    line_geom = line_feat.geometry()
                    if not line_geom or line_geom.isNull():
                        scu.show_user_message(
                            self.dlg,
                            self.tr("Error"),
                            self.tr("Line geometry is not valid."),
                        )
                        return

                    line_azimuth = scu.calculate_line_azimuth(line_geom)

                    struct_data = self.structure_service.project_structures(
                        line_layer,
                        structural_layer,
                        buffer_dist,
                        line_azimuth,
                        dip_field,
                        strike_field,
                    )

                    if struct_data:
                        result_text += f"\nStructures: {len(struct_data)} points"
                        self.dlg.results.setPlainText(result_text)
                    else:
                        result_text += f"\nStructures: None in {buffer_dist}m buffer"
                        self.dlg.results.setPlainText(result_text)
                else:
                    result_text += "\n⚠ Structural layer selected but dip/strike fields not specified."
                    self.dlg.results.setPlainText(result_text)

            # Draw preview with all data
            logger.debug("About to draw preview:")
            logger.debug(
                "  - profile_data: %d points", len(profile_data) if profile_data else 0
            )
            logger.debug("  - geol_data: %d points", len(geol_data) if geol_data else 0)
            logger.debug(
                "  - struct_data: %d points", len(struct_data) if struct_data else 0
            )
            self.draw_preview(profile_data, geol_data, struct_data)

            result_text += (
                "\n\n💡 Use 'Save' button to export data to CSV and Shapefile."
            )
            self.dlg.results.setPlainText(result_text)

            # Return processed data for caching
            return (
                profile_data,
                geol_data if geol_data else [],
                struct_data if struct_data else [],
            )

        except OSError as e:
            # File system errors (temp file creation, permissions, etc.)
            scu.show_user_message(
                self.dlg,
                self.tr("File System Error"),
                self.tr(
                    f"Failed to access temporary files: {e!s}\n\n"
                    f"Please check disk space and permissions."
                ),
                "error",
            )
            self.dlg.results.append(f"File system error: {e!s}")
            logger.error("File system error in process_data: %s", str(e), exc_info=True)

        except ValueError as e:
            # Data validation errors (invalid geometry, empty data, etc.)
            scu.show_user_message(
                self.dlg,
                self.tr("Data Validation Error"),
                self.tr(
                    f"Invalid data encountered: {e!s}\n\n"
                    f"Please verify your input layers and try again."
                ),
            )
            self.dlg.results.append(f"Validation error: {e!s}")
            logger.error("Validation error in process_data: %s", str(e), exc_info=True)

        except RuntimeError as e:
            # Runtime errors (geometry operations, CRS transformations, etc.)
            scu.show_user_message(
                self.dlg,
                self.tr("Processing Error"),
                self.tr(
                    f"Error during data processing: {e!s}\n\n"
                    f"This may be due to invalid geometries or CRS issues."
                ),
            )
            self.dlg.results.append(f"Processing error: {e!s}")
            logger.error("Runtime error in process_data: %s", str(e), exc_info=True)

        except KeyError as e:
            # Missing field or attribute errors
            scu.show_user_message(
                self.dlg,
                self.tr("Field Error"),
                self.tr(
                    f"Required field not found: {e!s}\n\n"
                    f"Please verify that all required fields exist in your layers."
                ),
            )
            self.dlg.results.append(f"Field error: {e!s}")
            logger.error("Field error in process_data: %s", str(e), exc_info=True)

        except Exception as e:
            # Catch-all for unexpected errors
            import traceback

            error_details = traceback.format_exc()
            scu.show_user_message(
                self.dlg,
                self.tr("Unexpected Error"),
                self.tr(
                    f"An unexpected error occurred: {e!s}\n\nDetails:\n{error_details}"
                ),
                "error",
            )
            self.dlg.results.append(f"Unexpected error: {e!s}")
            logger.exception("Unexpected error in process_data: %s", error_details)

    # Methods topographic_profile, geol_profile, and project_structures
    # have been moved to ProfileService, GeologyService, and StructureService respectively

    def save_profile_line(self):
        """Save all profile data to CSV and Shapefile formats by delegating to exporters."""
        try:
            inputs = self._get_and_validate_inputs()
            if not inputs:
                return

            values = inputs
            output_folder = Path(values["output_path"])
            is_valid, error, _ = vu.validate_output_path(str(output_folder))
            if not is_valid:
                scu.show_user_message(self.dlg, self.tr("Error"), self.tr(error))
                return

            self.dlg.results.setPlainText("✓ Generating data for export...")

            # 1. Generate all data in memory first
            profile_data = self.profile_service.generate_topographic_profile(
                values["line_layer_obj"],
                values["raster_layer_obj"],
                values["selected_band"],
            )
            if not profile_data:
                scu.show_user_message(
                    self.dlg,
                    self.tr("Error"),
                    self.tr("No profile data generated, cannot save."),
                )
                return

            geol_data = None
            if values["outcrop_layer_obj"] and values["outcrop_name_field"]:
                geol_data = self.geology_service.generate_geological_profile(
                    values["line_layer_obj"],
                    values["raster_layer_obj"],
                    values["outcrop_layer_obj"],
                    values["outcrop_name_field"],
                    values["selected_band"],
                )

            struct_data = None
            if (
                values["structural_layer_obj"]
                and values["dip_field"]
                and values["strike_field"]
            ):
                line_feat = next(values["line_layer_obj"].getFeatures(), None)
                if line_feat:
                    line_geom = line_feat.geometry()
                    line_azimuth = scu.calculate_line_azimuth(line_geom)
                    struct_data = self.structure_service.project_structures(
                        values["line_layer_obj"],
                        values["structural_layer_obj"],
                        values["buffer_distance"],
                        line_azimuth,
                        values["dip_field"],
                        values["strike_field"],
                    )

            # 2. Orchestrate saving using exporters
            result_msg = ["✓ Saving files..."]
            csv_exporter = CSVExporter({})
            line_crs = values["line_layer_obj"].crs()

            # Export Topography
            logger.info("✓ Saving topographic profile...")
            csv_exporter.export(
                output_folder / "topo_profile.csv",
                {"headers": ["dist", "elev"], "rows": profile_data},
            )
            result_msg.append("  - topo_profile.csv")
            ProfileLineShpExporter({}).export(
                output_folder / "profile_line.shp",
                {"profile_data": profile_data, "crs": line_crs},
            )
            result_msg.append("  - profile_line.shp")

            # Export Geology
            if geol_data:
                logger.info("✓ Saving geological profile...")
                csv_exporter.export(
                    output_folder / "geol_profile.csv",
                    {"headers": ["dist", "elev", "geology"], "rows": geol_data},
                )
                result_msg.append("  - geol_profile.csv")
                GeologyShpExporter({}).export(
                    output_folder / "geol_profile.shp",
                    {
                        "line_lyr": values["line_layer_obj"],
                        "raster_lyr": values["raster_layer_obj"],
                        "outcrop_lyr": values["outcrop_layer_obj"],
                        "band_number": values["selected_band"],
                    },
                )
                result_msg.append("  - geol_profile.shp")

            # Export Structures
            if struct_data:
                logger.info("✓ Saving structural profile...")
                csv_exporter.export(
                    output_folder / "structural_profile.csv",
                    {"headers": ["dist", "apparent_dip"], "rows": struct_data},
                )
                result_msg.append("  - structural_profile.csv")
                StructureShpExporter({}).export(
                    output_folder / "structural_profile.shp",
                    {
                        "line_lyr": values["line_layer_obj"],
                        "raster_lyr": values["raster_layer_obj"],
                        "struct_lyr": values["structural_layer_obj"],
                        "dip_field": values["dip_field"],
                        "strike_field": values["strike_field"],
                        "band_number": values["selected_band"],
                        "buffer_distance": values["buffer_distance"],
                        "dip_scale_factor": values["dip_scale_factor"],
                    },
                )
                result_msg.append("  - structural_profile.shp")

            # Export Axes
            logger.info("✓ Saving profile axes...")
            AxesShpExporter({}).export(
                output_folder / "profile_axes.shp",
                {"profile_data": profile_data, "crs": line_crs},
            )
            result_msg.append("  - profile_axes.shp")

            result_msg.append(f"\n✓ All files saved to:\n{output_folder}")
            self.dlg.results.setPlainText("\n".join(result_msg))
            QMessageBox.information(
                self.dlg,
                self.tr("Success"),
                self.tr(f"All data saved to:\n{output_folder}"),
            )

        except Exception as e:
            scu.show_user_message(
                self.dlg,
                self.tr("Export Error"),
                self.tr(f"Failed to export data: {e!s}"),
                "error",
            )
            logger.error("Export error in save_profile_line: %s", str(e), exc_info=True)

    def draw_preview(self, topo_data, geol_data=None, struct_data=None):
        """Draw enhanced interactive preview using native PyQGIS renderer.

        Args:
            topo_data: List of (dist, elev) tuples for topographic profile
            geol_data: Optional list of (dist, elev, geology_name) tuples
            struct_data: Optional list of (dist, app_dip) tuples
        """
        logger.debug("draw_preview called with:")
        logger.debug("  - topo_data: %d points", len(topo_data) if topo_data else 0)
        logger.debug("  - geol_data: %d points", len(geol_data) if geol_data else 0)
        logger.debug(
            "  - struct_data: %d points", len(struct_data) if struct_data else 0
        )

        # Store data in dialog for re-rendering when checkboxes change
        self.dlg.current_topo_data = topo_data
        self.dlg.current_geol_data = geol_data
        self.dlg.current_struct_data = struct_data

        # Get preview options from dialog checkboxes
        options = self.dlg.get_preview_options()
        logger.debug("Preview options: %s", options)

        # Filter data based on checkbox states
        filtered_topo = topo_data if options.get("show_topo", True) else None
        filtered_geol = geol_data if options.get("show_geol", True) else None
        filtered_struct = struct_data if options.get("show_struct", True) else None

        logger.debug("Filtered data:")
        logger.debug(
            "  - filtered_topo: %d points", len(filtered_topo) if filtered_topo else 0
        )
        logger.debug(
            "  - filtered_geol: %d points", len(filtered_geol) if filtered_geol else 0
        )
        logger.debug(
            "  - filtered_struct: %d points",
            len(filtered_struct) if filtered_struct else 0,
        )

        # Get vertical exaggeration from dialog

        _, _, vert_exag = vu.validate_numeric_input(
            self.dlg.vertexag.text(),
            field_name="Vertical exaggeration",
            allow_empty=True,
        )
        vert_exag = vert_exag if vert_exag is not None else 1.0
        logger.debug("Vertical exaggeration: %.2f", vert_exag)

        # Calculate dip line length based on scale factor and raster resolution
        dip_line_length = None
        if filtered_struct:
            _, _, dip_scale = vu.validate_numeric_input(
                self.dlg.dip_scale_factor.text(),
                field_name="Dip scale factor",
                allow_empty=True,
            )
            dip_scale = dip_scale if dip_scale is not None else 4.0

            if dip_scale > 0:
                raster_layer = self.dlg.rasterdem.currentLayer()
                if raster_layer and raster_layer.isValid():
                    res = raster_layer.rasterUnitsPerPixelX()
                    if res > 0:
                        dip_line_length = res * dip_scale
            logger.debug("Dip line length: %s (scale: %.2f)", dip_line_length, dip_scale)

        # Render using native PyQGIS
        canvas, layers = self.preview_renderer.render(
            filtered_topo, filtered_geol, filtered_struct, vert_exag, dip_line_length
        )

        # Store canvas and layers for export
        self.dlg.current_canvas = canvas
        self.dlg.current_layers = layers

        # Update legend
        if hasattr(self.dlg, "legend_widget"):
            self.dlg.legend_widget.update_legend(self.preview_renderer)

        logger.debug("Preview rendered with %d layers", len(layers) if layers else 0)
