"""SecInterpDialog — A QGIS plugin.

/***************************************************************************
SecInterpDialog
                                A QGIS plugin
Data extraction for geological interpretation
Generated by Plugin Builder: http://g-sherman.github.io/Qgis-Plugin-Builder/
                            -------------------
       begin                : 2025-11-15
       git sha              : $Format:%H$
       copyright            : (C) 2025 by Juan M Bernales
       email                : juanbernales@gmail.com
***************************************************************************/

/***************************************************************************
*                                                                         *
*   This program is free software; you can redistribute it and/or modify  *
*   it under the terms of the GNU General Public License as published by  *
*   the Free Software Foundation; either version 2 of the License, or     *
*   (at your option) any later version.                                   *
*                                                                         *
***************************************************************************/
"""

import tempfile
import traceback
import webbrowser
from pathlib import Path
from typing import List

from qgis.core import (
    Qgis,
    QgsCoordinateTransform,
    QgsLayerTreeLayer,
    QgsMapLayer,
    QgsMapLayerProxyModel,
    QgsMapRendererCustomPainterJob,
    QgsMapSettings,
    QgsPointXY,
    QgsProject,
    QgsSettings,
    QgsUnitTypes,
    QgsWkbTypes,
)
from qgis.gui import QgsFileWidget, QgsMapCanvas, QgsMessageBar
from qgis.PyQt import QtCore
from qgis.PyQt.QtCore import QMarginsF, QRectF, QSize, QSizeF, Qt, QVariant
from qgis.PyQt.QtGui import QColor, QImage, QPainter, QPageSize
from qgis.PyQt.QtPrintSupport import QPrinter
from qgis.PyQt.QtSvg import QSvgGenerator
from qgis.PyQt.QtWidgets import (
    QDialog,
    QDialogButtonBox,
    QFileDialog,
    QMessageBox,
    QWidget,
)

from .ui.main_dialog_base import Ui_SecInterpDialogBase
from ..core import utils as scu
from ..core import validation as vu
from .preview_renderer import PreviewRenderer
from sec_interp.logger_config import get_logger


class LegendWidget(QWidget):
    """Widget to display the geological legend over the map canvas."""

    def __init__(self, parent=None):
        super().__init__(parent)
        self.renderer = None
        self.setAttribute(Qt.WA_TranslucentBackground)
        self.setAttribute(Qt.WA_TransparentForMouseEvents)  # Let clicks pass through
        self.setAutoFillBackground(False)  # Don't fill background
        self.hide()

    def update_legend(self, renderer):
        """Update legend with data from renderer."""
        self.renderer = renderer
        self.update()
        self.show()

    def paintEvent(self, event):
        if not self.renderer or not self.renderer.active_units:
            return

        painter = QPainter(self)
        painter.setRenderHint(QPainter.Antialiasing)

        # Draw legend using the shared method
        self.renderer.draw_legend(painter, QRectF(self.rect()))


class SecInterpDialog(QDialog, Ui_SecInterpDialogBase):
    """Dialog for the SecInterp QGIS plugin.

    This dialog provides the user interface and helper methods to populate
    combo boxes with layers from the current QGIS project (raster and vector
    layers filtered by geometry type). It also exposes the interface and
    plugin instance for interaction with the host application.

    Attributes:
    ----------
    iface :
        The QGIS interface instance passed by the host application.
    plugin_instance :
        The plugin instance that created this dialog.
    messagebar :
        The QGIS message bar widget used to display notifications and errors.
    """

    def __init__(self, iface=None, plugin_instance=None, parent=None):
        """Constructor."""
        super(SecInterpDialog, self).__init__(parent)
        # iface and plugin_instance are optional to make the dialog testable
        # in environments where a QGIS iface is not available.
        self.iface = iface
        self.plugin_instance = plugin_instance
        # Set up the user interface from Designer through the base class.
        self.setupUi(self)
        # Provide a safe, no-op messagebar when iface is not available (tests)
        if self.iface is None:

            class _NoOpMessageBar:
                def pushMessage(self, *_args, **_kwargs):
                    return None

            self.messagebar = _NoOpMessageBar()
        else:
            self.messagebar = self.iface.messageBar()
        self.scale.setText("50000")
        self.vertexag.setText("1")
        self.buffer_distance.setText("100")
        self.dip_scale_factor.setText("4")

        # Replace QGraphicsView with QgsMapCanvas for preview
        # Remove the old QGraphicsView widget
        old_preview = self.preview
        preview_geometry = old_preview.geometry()
        preview_parent = old_preview.parent()
        old_preview.setParent(None)
        old_preview.deleteLater()

        # Create new QgsMapCanvas
        self.preview = QgsMapCanvas(preview_parent)
        self.preview.setGeometry(preview_geometry)
        self.preview.setCanvasColor(QColor(255, 255, 255))  # White background

        # Create legend widget
        self.legend_widget = LegendWidget(self.preview)
        self.legend_widget.resize(self.preview.size())

        # Store current preview data for re-rendering when checkboxes change
        self.current_topo_data = None
        self.current_geol_data = None
        self.current_struct_data = None
        self.current_canvas = None
        self.current_layers = []

        # Configure outcrop layer combo box
        self.outcrop.setFilters(QgsMapLayerProxyModel.PolygonLayer)
        self.outcrop.setAllowEmptyLayer(True, "Do not use Outcrop layer")
        self.outcrop.setLayer(None)  # Default to empty
        self.outcrop.layerChanged.connect(self.ocropname.setLayer)

        # Initialize field combo box with current layer (will be None)
        self.ocropname.setLayer(self.outcrop.currentLayer())

        # Configure structural layer combo box
        self.structural.setFilters(QgsMapLayerProxyModel.PointLayer)
        self.structural.setAllowEmptyLayer(True, "Do not use Structural layer")
        self.structural.setLayer(None)  # Default to empty
        self.structural.layerChanged.connect(self.dip.setLayer)
        self.structural.layerChanged.connect(self.strike.setLayer)

        # Initialize field combo boxes with current layer (will be None)
        self.dip.setLayer(self.structural.currentLayer())
        self.strike.setLayer(self.structural.currentLayer())

        # Configure crossline layer combo box
        self.crossline.setFilters(QgsMapLayerProxyModel.LineLayer)
        self.crossline.setAllowEmptyLayer(True, "Select Crossline Layer")
        self.crossline.setLayer(None)  # Default to empty
        self.crossline.layerChanged.connect(self.update_button_state)

        # Configure output folder widget
        self.dest_fold.setStorageMode(QgsFileWidget.GetDirectory)
        self.dest_fold.setDialogTitle("Select Output Folder")

        # Configure raster layer combo box
        self.rasterdem.setFilters(QgsMapLayerProxyModel.RasterLayer)
        self.rasterdem.setAllowEmptyLayer(True, "Select DEM Layer")
        self.rasterdem.setLayer(None)  # Default to empty

        # Connect raster layer change to band combo box
        self.rasterdem.layerChanged.connect(self.band.setLayer)

        # Initialize band combo box with current raster if any
        if self.rasterdem.currentLayer():
            self.band.setLayer(self.rasterdem.currentLayer())

        # Update resolution only if iface or canvas isn't required (guarded in method)
        self.update_resolution_field()
        self.rasterdem.layerChanged.connect(self.update_resolution_field)
        self.rasterdem.layerChanged.connect(self.update_button_state)
        self.button_box.accepted.connect(self.accept_handler)
        self.button_box.rejected.connect(self.reject_handler)
        self.button_box.helpRequested.connect(self.open_help)
        self.dest_fold.fileChanged.connect(self.update_button_state)
        self.preview_button.clicked.connect(self.preview_profile_handler)
        self.update_button_state()

        # Connect preview checkboxes to update preview when toggled
        self.show_topo_cb.stateChanged.connect(self.update_preview_from_checkboxes)
        self.show_geol_cb.stateChanged.connect(self.update_preview_from_checkboxes)
        self.show_struct_cb.stateChanged.connect(self.update_preview_from_checkboxes)

        # Connect export button
        self.export_pre.clicked.connect(self.export_preview)

    def wheelEvent(self, event):
        """Handle mouse wheel for zooming in preview."""
        # Check if mouse is over preview widget
        if self.preview.underMouse():
            # QgsMapCanvas has built-in zoom with wheel
            # We can customize the zoom factor if needed
            if event.angleDelta().y() > 0:
                self.preview.zoomIn()
            else:
                self.preview.zoomOut()
            event.accept()
        else:
            super().wheelEvent(event)

    def open_help(self):
        """Open the help file in the default browser."""
        help_file = Path(__file__).parent / "help" / "html" / "index.html"
        if help_file.exists():
            webbrowser.open(help_file.as_uri())
        else:
            self.messagebar.pushMessage(
                "Error",
                "Help file not found. Please run 'make doc' to generate it.",
                level=Qgis.Warning,
            )

    def update_button_state(self):
        """Enable or disable buttons based on input validity.

        - Preview and Ok buttons require: DEM + Cross-section line
        - Save button requires: DEM + Cross-section line + Output path
        """
        has_output_path = bool(self.dest_fold.filePath())
        has_raster = bool(self.rasterdem.currentLayer())
        has_line = bool(self.crossline.currentLayer())

        # Preview and Ok require at least DEM and line
        can_preview = has_raster and has_line

        # Save requires DEM, line, and output path
        can_save = can_preview and has_output_path

        # Update Preview button
        preview_btn = self.preview_button
        if preview_btn:
            preview_btn.setEnabled(can_preview)

        # Update Ok button
        ok_btn = self.button_box.button(QDialogButtonBox.Ok)
        if ok_btn:
            ok_btn.setEnabled(can_preview)

        # Update Save button
        save_btn = self.button_box.button(QDialogButtonBox.Save)
        if save_btn:
            save_btn.setEnabled(can_save)

    def get_selected_values(self):
        """Get the selected values from the dialog with safe type conversion."""

        # Safely convert numeric inputs with defaults
        _, _, scale = vu.validate_numeric_input(
            self.scale.text(), field_name="Scale", allow_empty=True
        )
        _, _, vertexag = vu.validate_numeric_input(
            self.vertexag.text(), field_name="Vertical exaggeration", allow_empty=True
        )
        _, _, buffer_dist = vu.validate_numeric_input(
            self.buffer_distance.text(), field_name="Buffer distance", allow_empty=True
        )
        _, _, dip_scale = vu.validate_numeric_input(
            self.dip_scale_factor.text(),
            field_name="Dip scale factor",
            allow_empty=True,
        )
        # Band number is now handled by QgsRasterBandComboBox which returns int band number directly via currentBand()
        # But for consistency with existing code structure we can get it here
        band_num = self.band.currentBand()

        return {
            "raster_layer": self.rasterdem.currentLayer(),
            "outcrop_layer": self.outcrop.currentLayer(),
            "structural_layer": self.structural.currentLayer(),
            "crossline_layer": self.crossline.currentLayer(),
            "dip_field": self.dip.currentField(),
            "strike_field": self.strike.currentField(),
            "outcrop_name_field": self.ocropname.currentField(),
            "scale": scale if scale is not None else 50000,
            "vertexag": vertexag if vertexag is not None else 1.0,
            "selected_band": band_num if band_num is not None else 1,
            "buffer_distance": buffer_dist if buffer_dist is not None else 100.0,
            "dip_scale_factor": dip_scale if dip_scale is not None else 4.0,
            "output_path": self.dest_fold.filePath(),
        }

    def get_preview_options(self):
        """Return the state of preview layer checkboxes.

        Returns:
            dict: Keys 'show_topo', 'show_geol', 'show_struct' with boolean values.
        """
        return {
            "show_topo": bool(self.show_topo_cb.isChecked()),
            "show_geol": bool(self.show_geol_cb.isChecked()),
            "show_struct": bool(self.show_struct_cb.isChecked()),
        }

    def update_preview_from_checkboxes(self):
        """Update preview when checkboxes change.

        This method is called when any of the preview checkboxes are toggled.
        It re-renders the preview using the stored data and current checkbox states.
        """
        # Only update if we have data to display and a plugin instance capable
        # of drawing the preview (tests may construct the dialog without one).
        if (
            self.current_topo_data is not None
            and self.plugin_instance is not None
            and hasattr(self.plugin_instance, "draw_preview")
            and callable(getattr(self.plugin_instance, "draw_preview"))
        ):
            # Call the plugin's draw_preview method with stored data
            self.plugin_instance.draw_preview(
                self.current_topo_data, self.current_geol_data, self.current_struct_data
            )

    def preview_profile_handler(self):
        """Generate a quick preview with topographic, geological, and structural data.

        This method validates inputs and generates a preview with all available data layers
        without saving files to disk.
        """

        # Skip if no plugin instance (e.g., in tests)
        if self.plugin_instance is None:
            return

        # Validate raster layer
        raster_layer = self.rasterdem.currentLayer()
        if not raster_layer:
            self.results.setPlainText("⚠ Please select a raster layer for preview.")
            return

        # Validate crossline layer
        line_layer = self.crossline.currentLayer()
        if not line_layer:
            self.results.setPlainText(
                "⚠ Please select a cross-section line for preview."
            )
            return

        # Validate band number
        band_num = self.band.currentBand()
        if not band_num:
            self.results.setPlainText("⚠ Please select a band number.")
            return

        # Validate band exists in raster
        is_valid, error = vu.validate_raster_band(raster_layer, band_num)
        if not is_valid:
            self.results.setPlainText(f"⚠ {error}")
            return

        try:
            from pathlib import Path
            import tempfile
            from qgis.core import QgsProject

            self.results.setPlainText("Generating preview...")

            # Generate topographic profile
            with tempfile.NamedTemporaryFile(
                mode="w", suffix=".csv", delete=False
            ) as tmp:
                tmp_path = Path(tmp.name)

            profile_data = self.plugin_instance.topographic_profile(
                line_layer, raster_layer, tmp_path, band_num
            )
            tmp_path.unlink()

            if not profile_data or len(profile_data) < 2:
                self.results.setPlainText(
                    "⚠ No profile data generated. Check that the line intersects the raster."
                )
                return

            # Initialize result message
            result_msg = (
                f"✓ Preview generated!\n\nTopography: {len(profile_data)} points\n"
            )

            # Process geological data if outcrop layer is selected
            geol_data = None
            outcrop_layer = self.outcrop.currentLayer()
            if outcrop_layer:
                outcrop_name_field = self.ocropname.currentField()

                if outcrop_name_field:
                    with tempfile.NamedTemporaryFile(
                        mode="w", suffix=".csv", delete=False
                    ) as tmp:
                        tmp_path = Path(tmp.name)

                    geol_data = self.plugin_instance.geol_profile(
                        line_layer,
                        raster_layer,
                        outcrop_layer,
                        outcrop_name_field,
                        tmp_path,
                        band_num,
                    )
                    tmp_path.unlink()

                    if geol_data:
                        result_msg += f"Geology: {len(geol_data)} points\n"
                    else:
                        result_msg += "Geology: No intersections\n"

            # Process structural data if structural layer is selected
            struct_data = None
            structural_layer = self.structural.currentLayer()
            if structural_layer:
                dip_field = self.dip.currentField()
                strike_field = self.strike.currentField()

                if dip_field and strike_field:
                    # Get buffer distance
                    _, _, buffer_dist = vu.validate_numeric_input(
                        self.buffer_distance.text(),
                        field_name="Buffer distance",
                        allow_empty=True,
                    )
                    buffer_dist = buffer_dist if buffer_dist is not None else 100.0

                    # Get line azimuth
                    line_feat = next(line_layer.getFeatures(), None)
                    if line_feat:
                        line_geom = line_feat.geometry()
                        line_azimuth = scu.calculate_line_azimuth(line_geom)

                        with tempfile.NamedTemporaryFile(
                            mode="w", suffix=".csv", delete=False
                        ) as tmp:
                            tmp_path = Path(tmp.name)

                        struct_data = self.plugin_instance.project_structures(
                            line_layer,
                            structural_layer,
                            buffer_dist,
                            line_azimuth,
                            dip_field,
                            strike_field,
                            tmp_path,
                        )
                        tmp_path.unlink()

                        if struct_data:
                            result_msg += f"Structures: {len(struct_data)} points\n"
                        else:
                            result_msg += f"Structures: None in {buffer_dist}m buffer\n"

            # Draw preview with all available data
            self.plugin_instance.draw_preview(profile_data, geol_data, struct_data)

            # Add distance and elevation ranges
            result_msg += (
                f"\nDistance: {profile_data[0][0]:.1f} - {profile_data[-1][0]:.1f} m\n"
            )
            result_msg += f"Elevation: {min(p[1] for p in profile_data):.1f} - {max(p[1] for p in profile_data):.1f} m\n\n"
            result_msg += "Adjust 'Vert. Exag.' and click Preview to update."

            self.results.setPlainText(result_msg)

        except Exception as e:
            import traceback

            error_details = traceback.format_exc()
            self.results.setPlainText(
                f"⚠ Error generating preview: {str(e)}\n\nDetails:\n{error_details}"
            )

    def export_preview(self):
        """Export the current preview to a file (SVG, PDF, PNG, JPG) using PyQGIS native rendering."""
        if not self.current_layers or not self.current_canvas:
            self.results.setPlainText(
                "⚠ No preview to export. Please generate a preview first."
            )
            return

        # Determine default directory

        settings = QgsSettings()

        dest_folder = self.dest_fold.filePath().strip()
        if dest_folder:
            default_path = str(Path(dest_folder) / "preview.png")
        else:
            last_dir = settings.value("SecInterp/lastExportDir", "", type=str)
            default_path = (
                str(Path(last_dir) / "preview.png") if last_dir else "preview.png"
            )

        filename, selected_filter = QFileDialog.getSaveFileName(
            self,
            "Export Preview",
            default_path,
            "PNG Image (*.png);;JPEG Image (*.jpg);;Scalable Vector Graphics (*.svg);;PDF Documents (*.pdf)",
        )

        if not filename:
            return

        # Determine the correct extension
        filter_ext_map = {
            "PNG Image (*.png)": ".png",
            "JPEG Image (*.jpg)": ".jpg",
            "Scalable Vector Graphics (*.svg)": ".svg",
            "PDF Documents (*.pdf)": ".pdf",
        }

        current_ext = Path(filename).suffix.lower()
        expected_ext = filter_ext_map.get(selected_filter, "")

        if not current_ext or current_ext not in [
            ".png",
            ".jpg",
            ".jpeg",
            ".svg",
            ".pdf",
        ]:
            if expected_ext:
                filename = str(Path(filename).with_suffix(expected_ext))

        ext = Path(filename).suffix.lower()
        settings.setValue("SecInterp/lastExportDir", str(Path(filename).parent))

        try:
            self.results.setPlainText(f"Exporting preview to {filename}...")

            # Get extent from canvas
            extent = self.current_canvas.extent()

            # Export dimensions (use canvas size as reference)
            width = self.preview.width()
            height = self.preview.height()

            # For raster formats, use higher resolution
            if ext in [".png", ".jpg", ".jpeg"]:
                width = width * 3  # 3x resolution for better quality
                height = height * 3
                dpi = 300
            else:
                dpi = 96

            # Use PyQGIS native export
            from qgis.core import QgsMapSettings, QgsMapRendererCustomPainterJob

            settings = QgsMapSettings()
            settings.setLayers(self.current_layers)
            settings.setExtent(extent)
            settings.setOutputSize(QSize(width, height))
            settings.setOutputDpi(dpi)
            settings.setBackgroundColor(QColor(255, 255, 255))

            if ext in [".png", ".jpg", ".jpeg"]:
                # Raster export
                image = QImage(QSize(width, height), QImage.Format_ARGB32)

                if ext in [".jpg", ".jpeg"]:
                    image.fill(QColor(255, 255, 255))  # White background
                else:
                    image.fill(QColor(255, 255, 255))  # White background for PNG too

                painter = QPainter(image)
                painter.setRenderHint(QPainter.Antialiasing)
                painter.setRenderHint(QPainter.SmoothPixmapTransform)

                job = QgsMapRendererCustomPainterJob(settings, painter)
                job.start()
                job.waitForFinished()

                # Draw legend
                if hasattr(self.plugin_instance, "preview_renderer"):
                    self.plugin_instance.preview_renderer.draw_legend(
                        painter, QRectF(0, 0, width, height)
                    )

                painter.end()

                quality = 95 if ext in [".jpg", ".jpeg"] else -1
                if not image.save(filename, None, quality):
                    raise Exception(f"Failed to save image to {filename}")

            elif ext == ".svg":
                # SVG export
                generator = QSvgGenerator()
                generator.setFileName(filename)
                generator.setSize(QSize(width, height))
                generator.setViewBox(QRectF(0, 0, width, height))
                generator.setTitle("Section Interpretation Preview")
                generator.setDescription("Generated by SecInterp QGIS Plugin")

                painter = QPainter()
                if painter.begin(generator):
                    painter.setRenderHint(QPainter.Antialiasing)

                    job = QgsMapRendererCustomPainterJob(settings, painter)
                    job.start()
                    job.waitForFinished()

                    # Draw legend
                    if hasattr(self.plugin_instance, "preview_renderer"):
                        self.plugin_instance.preview_renderer.draw_legend(
                            painter, QRectF(0, 0, width, height)
                        )

                    painter.end()
                else:
                    raise Exception("Failed to initialize SVG painter")

            elif ext == ".pdf":
                # PDF export
                printer = QPrinter(QPrinter.HighResolution)
                printer.setOutputFormat(QPrinter.PdfFormat)
                printer.setOutputFileName(filename)
                printer.setPageSize(QPageSize(QSizeF(width, height), QPageSize.Point))
                printer.setPageMargins(0.0, 0.0, 0.0, 0.0, QPrinter.Point)
                printer.setFullPage(True)

                painter = QPainter()
                if painter.begin(printer):
                    painter.setRenderHint(QPainter.Antialiasing)

                    # Update settings with actual printer device dimensions and DPI
                    # This ensures the map fills the page regardless of HighResolution setting
                    dev = painter.device()
                    settings.setOutputSize(QSize(dev.width(), dev.height()))
                    settings.setOutputDpi(printer.resolution())

                    job = QgsMapRendererCustomPainterJob(settings, painter)
                    job.start()
                    job.waitForFinished()

                    # Draw legend
                    if hasattr(self.plugin_instance, "preview_renderer"):
                        self.plugin_instance.preview_renderer.draw_legend(
                            painter, QRectF(0, 0, dev.width(), dev.height())
                        )

                    painter.end()
                else:
                    raise Exception("Failed to initialize PDF printer")
            else:
                raise Exception(f"Unsupported file extension: {ext}")

            self.results.setPlainText(
                f"✓ Preview exported successfully to:\n{filename}"
            )

        except Exception as e:
            import traceback

            error_details = traceback.format_exc()
            self.results.setPlainText(
                f"⚠ Error exporting preview: {str(e)}\n\nDetails:\n{error_details}"
            )

    def accept_handler(self):
        """Handle the accept button click event."""
        # When running without a QGIS iface (tests), skip strict validation
        if self.iface is None:
            self.accept()
            return

        if not self.validate_inputs():
            return

        self.accept()

    def reject_handler(self):
        """Handle the reject button click event."""
        self.close()

    def validate_inputs(self):
        """Validate the inputs from the dialog."""

        errors = []

        # Validate raster layer selection
        raster_layer = self.rasterdem.currentLayer()
        if not raster_layer:
            errors.append("No raster layer selected.")
        else:
            # Validate band number
            band_num = self.band.currentBand()
            if band_num:
                is_valid, error = vu.validate_raster_band(raster_layer, band_num)
                if not is_valid:
                    errors.append(error)
            else:
                errors.append("No band selected.")

        # Validate crossline layer selection
        line_layer = self.crossline.currentLayer()
        if not line_layer:
            errors.append("No crossline layer selected")
        else:
            # Validate geometry type (QgsMapLayerComboBox with LineLayer filter should handle this, but good to double check)
            is_valid, error = vu.validate_layer_geometry(
                line_layer, QgsWkbTypes.LineGeometry
            )
            if not is_valid:
                errors.append(error)
            else:
                # Validate has features
                is_valid, error = vu.validate_layer_has_features(line_layer)
                if not is_valid:
                    errors.append(error)

        # Validate output folder
        if not self.dest_fold.filePath():
            errors.append("Output folder is required")
        else:
            is_valid, error, _ = vu.validate_output_path(self.dest_fold.filePath())
            if not is_valid:
                errors.append(error)

        # Validate numeric inputs
        # Scale
        is_valid, error, _ = vu.validate_numeric_input(
            self.scale.text(), min_val=1, field_name="Scale"
        )
        if not is_valid:
            errors.append(error)

        # Vertical exaggeration
        is_valid, error, _ = vu.validate_numeric_input(
            self.vertexag.text(), min_val=0.1, field_name="Vertical exaggeration"
        )
        if not is_valid:
            errors.append(error)

        # Buffer distance
        is_valid, error, _ = vu.validate_numeric_input(
            self.buffer_distance.text(), min_val=0, field_name="Buffer distance"
        )
        if not is_valid:
            errors.append(error)

        # Dip scale factor
        is_valid, error, _ = vu.validate_numeric_input(
            self.dip_scale_factor.text(), min_val=0.1, field_name="Dip scale factor"
        )
        if not is_valid:
            errors.append(error)

        # Validate outcrop layer if selected
        outcrop_layer = self.outcrop.currentLayer()
        if outcrop_layer:
            # Validate geometry type
            is_valid, error = vu.validate_layer_geometry(
                outcrop_layer, QgsWkbTypes.PolygonGeometry
            )
            if not is_valid:
                errors.append(error)
            else:
                # Validate has features
                is_valid, error = vu.validate_layer_has_features(outcrop_layer)
                if not is_valid:
                    errors.append(error)

                # Validate outcrop name field
                if not self.ocropname.currentText():
                    errors.append(
                        "Outcrop name field is required when outcrop layer is selected"
                    )
                else:
                    field_name = self.ocropname.currentData()
                    is_valid, error = vu.validate_field_exists(
                        outcrop_layer, field_name
                    )
                    if not is_valid:
                        errors.append(error)

        # Validate structural layer if selected
        struct_layer = self.structural.currentLayer()
        if struct_layer:
            # Validate geometry type
            is_valid, error = vu.validate_layer_geometry(
                struct_layer, QgsWkbTypes.PointGeometry
            )
            if not is_valid:
                errors.append(error)
            else:
                # Validate has features
                is_valid, error = vu.validate_layer_has_features(struct_layer)
                if not is_valid:
                    errors.append(error)

                # Validate dip field
                if not self.dip.currentText():
                    errors.append(
                        "Dip field is required when structural layer is selected"
                    )
                else:
                    dip_field = self.dip.currentData()
                    is_valid, error = vu.validate_field_exists(struct_layer, dip_field)
                    if not is_valid:
                        errors.append(error)
                    else:
                        # Validate field type (should be numeric)
                        is_valid, error = vu.validate_field_type(
                            struct_layer, dip_field, [QVariant.Int, QVariant.Double]
                        )
                        if not is_valid:
                            errors.append(error)

                # Validate strike field
                if not self.strike.currentText():
                    errors.append(
                        "Strike field is required when structural layer is selected"
                    )
                else:
                    strike_field = self.strike.currentData()
                    is_valid, error = vu.validate_field_exists(
                        struct_layer, strike_field
                    )
                    if not is_valid:
                        errors.append(error)
                    else:
                        # Validate field type (should be numeric)
                        is_valid, error = vu.validate_field_type(
                            struct_layer,
                            strike_field,
                            [QVariant.Int, QVariant.Double],
                        )
                        if not is_valid:
                            errors.append(error)

        if errors:
            QMessageBox.warning(self, "Validation Error", "\n".join(errors))
            return False

        return True

    def _populate_field_combobox(self, source_combobox, target_combobox):
        """Helper function to populate a combobox with field names from a selected vector layer."""
        try:
            selected_layer_name = source_combobox.currentData()
            target_combobox.clear()

            if not selected_layer_name:
                return

            layers = QgsProject.instance().mapLayersByName(selected_layer_name)
            if not layers:
                return

            vector_layer = layers[0]
            # Use addItem with data parameter so currentData() returns the field name
            for field in vector_layer.fields():
                target_combobox.addItem(field.name(), field.name())
        except Exception as e:
            self.messagebar.pushMessage("Error", str(e), level=Qgis.Critical)

    def get_layer_names_by_type(self, layer_type) -> List[str]:
        """Get a list of layer names filtered by the specified layer type.

        This method scans all layers in the current project.

        Args:
              layer_type: The QgsMapLayer type to filter by (e.g.,
        QgsMapLayer.RasterLayer)

        Returns:
              A list of layer names matching the specified type
        """
        layers = QgsProject.instance().mapLayers().values()
        return [layer.name() for layer in layers if layer.type() == layer_type]

    def get_layer_names_by_geometry(self, geometry_type) -> List[str]:
        """Get a list of layer names filtered by the specified geometry type.

        This method scans all layers in the current project.

        Args:
              geometry_type: The QgsWkbTypes geometry type to filter by
        (e.g., QgsWkbTypes.LineGeometry)

        Returns:
              A list of layer names matching the specified geometry type
        """
        layers = QgsProject.instance().mapLayers().values()
        return [
            layer.name()
            for layer in layers
            if layer.type() == QgsMapLayer.VectorLayer
            and QgsWkbTypes.geometryType(layer.wkbType()) == geometry_type
        ]

    def update_resolution_field(self):
        """Calculate and update the resolution field for the selected raster layer."""
        try:
            raster_layer = self.rasterdem.currentLayer()
            if not raster_layer:
                self.resln.clear()
                self.unts.clear()
                return
            raster_crs = raster_layer.crs()
            # Determine map CRS safely: if iface or canvas isn't available
            # (e.g., during unit tests), fall back to using raster CRS.
            try:
                if (
                    self.iface is None
                    or not hasattr(self.iface, "mapCanvas")
                    or self.iface.mapCanvas() is None
                ):
                    map_crs = raster_crs
                else:
                    map_crs = self.iface.mapCanvas().mapSettings().destinationCrs()
            except AttributeError:
                # Defensive: if anything goes wrong accessing iface/canvas, use raster CRS.
                map_crs = raster_crs
            native_res = raster_layer.rasterUnitsPerPixelX()

            # Validate native resolution
            if native_res <= 0:
                self.resln.setText("Invalid")
                self.unts.setText("")
                self.messagebar.pushMessage(
                    "Warning",
                    "Raster resolution is invalid or zero",
                    level=Qgis.Warning,
                )
                return

            resolution_in_meters = 0

            if raster_crs == map_crs:
                if raster_crs.isGeographic():
                    self.resln.setText(f"{native_res:.6f}")
                    self.unts.setText("°")
                    # For scale calculation, we need resolution in meters.
                    # This is an approximation at the equator.
                    resolution_in_meters = native_res * 111320
                else:
                    self.resln.setText(f"{native_res:.2f}")
                    self.unts.setText(QgsUnitTypes.toString(raster_crs.mapUnits()))
                    if raster_crs.mapUnits() == QgsUnitTypes.DistanceUnit.Meters:
                        resolution_in_meters = native_res
                    elif raster_crs.mapUnits() == QgsUnitTypes.DistanceUnit.Feet:
                        resolution_in_meters = native_res * 0.3048

            else:
                transform = QgsCoordinateTransform(
                    raster_crs, map_crs, QgsProject.instance()
                )
                center = raster_layer.extent().center()
                p1 = center
                p2 = QgsPointXY(center.x() + native_res, center.y())
                p1_transformed = transform.transform(p1)
                p2_transformed = transform.transform(p2)

                if map_crs.isGeographic():
                    # Raster is projected, map is geographic
                    resolution = abs(p2_transformed.x() - p1_transformed.x())
                    self.resln.setText(f"{resolution:.6f}")
                    self.unts.setText("°")
                    # We need to transform back to the raster's projected CRS to get meters
                    # This is a bit complex, let's use the native resolution for scale for now
                    if raster_crs.mapUnits() == QgsUnitTypes.DistanceUnit.Meters:
                        resolution_in_meters = native_res
                    elif raster_crs.mapUnits() == QgsUnitTypes.DistanceUnit.Feet:
                        resolution_in_meters = native_res * 0.3048
                else:
                    # Raster is geographic, map is projected
                    resolution = p1_transformed.distance(p2_transformed)
                    self.resln.setText(f"{resolution:.2f}")
                    self.unts.setText(QgsUnitTypes.toString(map_crs.mapUnits()))
                    if map_crs.mapUnits() == QgsUnitTypes.DistanceUnit.Meters:
                        resolution_in_meters = resolution
                    elif map_crs.mapUnits() == QgsUnitTypes.DistanceUnit.Feet:
                        resolution_in_meters = resolution * 0.3048

            # Calculate suggested scale only if resolution is valid
            if resolution_in_meters > 0:
                suggested_scale = round((resolution_in_meters * 2000) / 1000) * 1000
                self.scale.setText(str(suggested_scale))

        except Exception as e:
            self.messagebar.pushMessage("Error", str(e), level=Qgis.Critical)
