"""SecInterpDialog — A QGIS plugin.

 /***************************************************************************
 SecInterpDialog
                                 A QGIS plugin
 Data extraction for geological interpretation
 Generated by Plugin Builder: http://g-sherman.github.io/Qgis-Plugin-Builder/
                             -------------------
        begin                : 2025-11-15
        git sha              : $Format:%H$
        copyright            : (C) 2025 by Juan M Bernales
        email                : juanbernales@gmail.com
 ***************************************************************************/

/***************************************************************************
 *                                                                         *
 *   This program is free software; you can redistribute it and/or modify  *
 *   it under the terms of the GNU General Public License as published by  *
 *   the Free Software Foundation; either version 2 of the License, or     *
 *   (at your option) any later version.                                   *
 *                                                                         *
 ***************************************************************************/
"""

from typing import List
from pathlib import Path
import webbrowser

from qgis.core import (
    Qgis,
    QgsCoordinateTransform,
    QgsLayerTreeLayer,
    QgsMapLayer,
    QgsPointXY,
    QgsProject,
    QgsUnitTypes,
    QgsWkbTypes,
)
from qgis.gui import QgsMessageBar
from qgis.PyQt import QtCore
from qgis.PyQt.QtWidgets import QDialog, QFileDialog, QMessageBox, QDialogButtonBox
from qgis.PyQt.QtSvg import QSvgGenerator
from qgis.PyQt.QtPrintSupport import QPrinter
from qgis.PyQt.QtGui import QPainter, QImage, QColor, QPageSize
from qgis.PyQt.QtCore import QSize, QRectF, QSizeF, QMarginsF


from .ui_sec_interp_dialog_base import Ui_SecInterpDialogBase


class SecInterpDialog(QDialog, Ui_SecInterpDialogBase):
    """Dialog for the SecInterp QGIS plugin.

    This dialog provides the user interface and helper methods to populate
    combo boxes with layers from the current QGIS project (raster and vector
    layers filtered by geometry type). It also exposes the interface and
    plugin instance for interaction with the host application.

    Attributes:
    ----------
    iface :
        The QGIS interface instance passed by the host application.
    plugin_instance :
        The plugin instance that created this dialog.
    messagebar :
        The QGIS message bar widget used to display notifications and errors.
    """

    def __init__(self, iface=None, plugin_instance=None, parent=None):
        """Constructor."""
        super(SecInterpDialog, self).__init__(parent)
        # iface and plugin_instance are optional to make the dialog testable
        # in environments where a QGIS iface is not available.
        self.iface = iface
        self.plugin_instance = plugin_instance
        # Set up the user interface from Designer through the base class.
        self.setupUi(self)
        # Provide a safe, no-op messagebar when iface is not available (tests)
        if self.iface is None:

            class _NoOpMessageBar:
                def pushMessage(self, *_args, **_kwargs):
                    return None

            self.messagebar = _NoOpMessageBar()
        else:
            self.messagebar = self.iface.messageBar()
        self.scale.setText("50000")
        self.vertexag.setText("1")
        self.buffer_distance.setText("100")
        self.dip_scale_factor.setText("4")

        # Initialize zoom level tracking
        self.current_zoom_level = 1.0
        self.min_zoom = 0.1  # 10% of original size
        self.max_zoom = 10.0  # 10x original size

        # Store current preview data for re-rendering when checkboxes change
        self.current_topo_data = None
        self.current_geol_data = None
        self.current_struct_data = None

        self.populate_raster_layers()
        self.populate_outcrop_layers()
        self.populate_structural_layers()
        self.populate_crossline_layers()
        self.populate_band_combobox()
        # Update resolution only if iface or canvas isn't required (guarded in method)
        self.update_resolution_field()
        self.rasterdem.currentIndexChanged.connect(self.populate_band_combobox)
        self.rasterdem.currentIndexChanged.connect(self.update_resolution_field)
        self.rasterdem.currentIndexChanged.connect(self.update_button_state)
        self.crossline.currentIndexChanged.connect(self.update_button_state)
        self.outcrop.currentIndexChanged.connect(self.populate_outcrop_name_field)
        self.structural.currentIndexChanged.connect(self.populate_dip_name_field)
        self.structural.currentIndexChanged.connect(self.populate_strike_name_field)
        self.button_box.accepted.connect(self.accept_handler)
        self.button_box.rejected.connect(self.reject_handler)
        self.button_box.helpRequested.connect(self.open_help)
        self.dest_fold.textChanged.connect(self.update_button_state)
        self.preview_button.clicked.connect(self.preview_profile_handler)
        self.update_button_state()
        self.setup_preview_interactivity()
        self.current_scene = None

        # Connect preview checkboxes to update preview when toggled
        self.show_topo_cb.stateChanged.connect(self.update_preview_from_checkboxes)
        self.show_geol_cb.stateChanged.connect(self.update_preview_from_checkboxes)
        self.show_struct_cb.stateChanged.connect(self.update_preview_from_checkboxes)
        
        # Connect export button
        self.export_pre.clicked.connect(self.export_preview)

    def setup_preview_interactivity(self):
        """Configure preview widget for interactive zoom and pan."""
        from qgis.PyQt.QtWidgets import QGraphicsView

        # Enable drag mode for panning
        self.preview.setDragMode(QGraphicsView.ScrollHandDrag)

        # Set transformation anchors for smooth zooming
        self.preview.setTransformationAnchor(QGraphicsView.AnchorUnderMouse)
        self.preview.setResizeAnchor(QGraphicsView.AnchorUnderMouse)

        # Enable antialiasing for smoother graphics
        from qgis.PyQt.QtGui import QPainter

        self.preview.setRenderHint(QPainter.Antialiasing, True)

    def wheelEvent(self, event):
        """Handle mouse wheel for zooming in preview with limits."""
        # Check if mouse is over preview widget
        if self.preview.underMouse():
            zoom_factor = 1.15

            if event.angleDelta().y() > 0:
                # Zoom in - check if we're below max zoom
                new_zoom = self.current_zoom_level * zoom_factor
                if new_zoom <= self.max_zoom:
                    self.preview.scale(zoom_factor, zoom_factor)
                    self.current_zoom_level = new_zoom
            else:
                # Zoom out - check if we're above min zoom
                new_zoom = self.current_zoom_level / zoom_factor
                if new_zoom >= self.min_zoom:
                    self.preview.scale(1 / zoom_factor, 1 / zoom_factor)
                    self.current_zoom_level = new_zoom

            event.accept()
        else:
            super().wheelEvent(event)

    def open_help(self):
        """Open the help file in the default browser."""
        help_file = Path(__file__).parent / "help" / "html" / "index.html"
        if help_file.exists():
            webbrowser.open(help_file.as_uri())
        else:
            self.messagebar.pushMessage(
                "Error",
                "Help file not found. Please run 'make doc' to generate it.",
                level=Qgis.Warning,
            )

    def update_button_state(self):
        """Enable or disable buttons based on input validity.
        
        - Preview and Ok buttons require: DEM + Cross-section line
        - Save button requires: DEM + Cross-section line + Output path
        """
        has_output_path = bool(self.dest_fold.text())
        has_raster = bool(self.rasterdem.currentData())
        has_line = bool(self.crossline.currentData())
        
        # Preview and Ok require at least DEM and line
        can_preview = has_raster and has_line
        
        # Save requires DEM, line, and output path
        can_save = can_preview and has_output_path
        
        # Update Preview button
        preview_btn = self.preview_button
        if preview_btn:
            preview_btn.setEnabled(can_preview)
        
        # Update Ok button
        ok_btn = self.button_box.button(QDialogButtonBox.Ok)
        if ok_btn:
            ok_btn.setEnabled(can_preview)
        
        # Update Save button
        save_btn = self.button_box.button(QDialogButtonBox.Save)
        if save_btn:
            save_btn.setEnabled(can_save)

    def get_selected_values(self):
        """Get the selected values from the dialog with safe type conversion."""
        from . import validation_utils as vu

        # Safely convert numeric inputs with defaults
        _, _, scale = vu.validate_numeric_input(
            self.scale.text(), field_name="Scale", allow_empty=True
        )
        _, _, vertexag = vu.validate_numeric_input(
            self.vertexag.text(), field_name="Vertical exaggeration", allow_empty=True
        )
        _, _, buffer_dist = vu.validate_numeric_input(
            self.buffer_distance.text(), field_name="Buffer distance", allow_empty=True
        )
        _, _, dip_scale = vu.validate_numeric_input(
            self.dip_scale_factor.text(),
            field_name="Dip scale factor",
            allow_empty=True,
        )
        _, _, band_num = vu.validate_integer_input(
            self.band.currentText(), field_name="Band number", allow_empty=True
        )

        return {
            "raster_layer": self.rasterdem.currentData(),
            "outcrop_layer": self.outcrop.currentData(),
            "structural_layer": self.structural.currentData(),
            "crossline_layer": self.crossline.currentData(),
            "dip_field": self.dip.currentData(),
            "strike_field": self.strike.currentData(),
            "outcrop_name_field": self.ocropname.currentData(),
            "scale": scale if scale is not None else 50000,
            "vertexag": vertexag if vertexag is not None else 1.0,
            "selected_band": band_num if band_num is not None else 1,
            "buffer_distance": buffer_dist if buffer_dist is not None else 100.0,
            "dip_scale_factor": dip_scale if dip_scale is not None else 4.0,
            "output_path": self.dest_fold.text(),
        }

    def get_preview_options(self):
        """Return the state of preview layer checkboxes.

        Returns:
            dict: Keys 'show_topo', 'show_geol', 'show_struct' with boolean values.
        """
        return {
            "show_topo": bool(self.show_topo_cb.isChecked()),
            "show_geol": bool(self.show_geol_cb.isChecked()),
            "show_struct": bool(self.show_struct_cb.isChecked()),
        }

    def update_preview_from_checkboxes(self):
        """Update preview when checkboxes change.

        This method is called when any of the preview checkboxes are toggled.
        It re-renders the preview using the stored data and current checkbox states.
        """
        # Only update if we have data to display and a plugin instance capable
        # of drawing the preview (tests may construct the dialog without one).
        if (
            self.current_topo_data is not None
            and self.plugin_instance is not None
            and hasattr(self.plugin_instance, "draw_preview")
            and callable(getattr(self.plugin_instance, "draw_preview"))
        ):
            # Call the plugin's draw_preview method with stored data
            self.plugin_instance.draw_preview(
                self.current_topo_data, self.current_geol_data, self.current_struct_data
            )

    def preview_profile_handler(self):
        """Generate a quick preview with topographic, geological, and structural data.
        
        This method validates inputs and generates a preview with all available data layers
        without saving files to disk.
        """
        from . import validation_utils as vu
        from . import si_core_utils as scu
        
        # Skip if no plugin instance (e.g., in tests)
        if self.plugin_instance is None:
            return
        
        # Validate raster layer
        if not self.rasterdem.currentText():
            self.results.setPlainText("⚠ Please select a raster layer for preview.")
            return
        
        is_valid, error, raster_layer = vu.validate_layer_exists(
            self.rasterdem.currentData()
        )
        if not is_valid:
            self.results.setPlainText(f"⚠ Raster layer error: {error}")
            return
        
        # Validate crossline layer
        if not self.crossline.currentText():
            self.results.setPlainText("⚠ Please select a cross-section line for preview.")
            return
        
        is_valid, error, line_layer = vu.validate_layer_exists(
            self.crossline.currentData()
        )
        if not is_valid:
            self.results.setPlainText(f"⚠ Cross-section line error: {error}")
            return
        
        # Validate band number
        band_text = self.band.currentText()
        if not band_text:
            self.results.setPlainText("⚠ Please select a band number.")
            return
        
        is_valid, error, band_num = vu.validate_integer_input(
            band_text, min_val=1, field_name="Band number"
        )
        if not is_valid:
            self.results.setPlainText(f"⚠ Band number error: {error}")
            return
        
        # Validate band exists in raster
        is_valid, error = vu.validate_raster_band(raster_layer, band_num)
        if not is_valid:
            self.results.setPlainText(f"⚠ {error}")
            return
        
        try:
            from pathlib import Path
            import tempfile
            from qgis.core import QgsProject
            
            self.results.setPlainText("Generating preview...")
            
            # Generate topographic profile
            with tempfile.NamedTemporaryFile(mode='w', suffix='.csv', delete=False) as tmp:
                tmp_path = Path(tmp.name)
            
            profile_data = self.plugin_instance.topographic_profile(
                line_layer, raster_layer, tmp_path, band_num
            )
            tmp_path.unlink()
            
            if not profile_data or len(profile_data) < 2:
                self.results.setPlainText("⚠ No profile data generated. Check that the line intersects the raster.")
                return
            
            # Initialize result message
            result_msg = f"✓ Preview generated!\n\nTopography: {len(profile_data)} points\n"
            
            # Process geological data if outcrop layer is selected
            geol_data = None
            outcrop_layer_name = self.outcrop.currentData()
            if outcrop_layer_name:
                outcrop_layer = QgsProject.instance().mapLayersByName(outcrop_layer_name)[0]
                outcrop_name_field = self.ocropname.currentData()
                
                if outcrop_layer and outcrop_name_field:
                    with tempfile.NamedTemporaryFile(mode='w', suffix='.csv', delete=False) as tmp:
                        tmp_path = Path(tmp.name)
                    
                    geol_data = self.plugin_instance.geol_profile(
                        line_layer, raster_layer, outcrop_layer, 
                        outcrop_name_field, tmp_path, band_num
                    )
                    tmp_path.unlink()
                    
                    if geol_data:
                        result_msg += f"Geology: {len(geol_data)} points\n"
                    else:
                        result_msg += "Geology: No intersections\n"
            
            # Process structural data if structural layer is selected
            struct_data = None
            structural_layer_name = self.structural.currentData()
            if structural_layer_name:
                structural_layer = QgsProject.instance().mapLayersByName(structural_layer_name)[0]
                dip_field = self.dip.currentData()
                strike_field = self.strike.currentData()
                
                if structural_layer and dip_field and strike_field:
                    # Get buffer distance
                    _, _, buffer_dist = vu.validate_numeric_input(
                        self.buffer_distance.text(), 
                        field_name="Buffer distance", 
                        allow_empty=True
                    )
                    buffer_dist = buffer_dist if buffer_dist is not None else 100.0
                    
                    # Get line azimuth
                    line_feat = next(line_layer.getFeatures(), None)
                    if line_feat:
                        line_geom = line_feat.geometry()
                        line_azimuth = scu.calculate_line_azimuth(line_geom)
                        
                        with tempfile.NamedTemporaryFile(mode='w', suffix='.csv', delete=False) as tmp:
                            tmp_path = Path(tmp.name)
                        
                        struct_data = self.plugin_instance.project_structures(
                            line_layer, structural_layer, buffer_dist, line_azimuth,
                            dip_field, strike_field, tmp_path
                        )
                        tmp_path.unlink()
                        
                        if struct_data:
                            result_msg += f"Structures: {len(struct_data)} points\n"
                        else:
                            result_msg += f"Structures: None in {buffer_dist}m buffer\n"
            
            # Draw preview with all available data
            self.plugin_instance.draw_preview(profile_data, geol_data, struct_data)
            
            # Add distance and elevation ranges
            result_msg += f"\nDistance: {profile_data[0][0]:.1f} - {profile_data[-1][0]:.1f} m\n"
            result_msg += f"Elevation: {min(p[1] for p in profile_data):.1f} - {max(p[1] for p in profile_data):.1f} m\n\n"
            result_msg += "Adjust 'Vert. Exag.' and click Preview to update."
            
            self.results.setPlainText(result_msg)
            
        except Exception as e:
            import traceback
            error_details = traceback.format_exc()
            self.results.setPlainText(f"⚠ Error generating preview: {str(e)}\n\nDetails:\n{error_details}")


    def export_preview(self):
        """Export the current preview to a file (SVG, PDF, PNG, JPG)."""
        if not self.current_scene:
            self.results.setPlainText("⚠ No preview to export. Please generate a preview first.")
            return

        # Determine default directory: prioritize dest_fold, then last used dir, then home
        from qgis.core import QgsSettings
        settings = QgsSettings()
        
        # Use dest_fold if available, otherwise fall back to last used directory
        dest_folder = self.dest_fold.text().strip()
        if dest_folder:
            # Suggest a default filename in the destination folder
            default_path = str(Path(dest_folder) / "preview.svg")
        else:
            last_dir = settings.value("SecInterp/lastExportDir", "", type=str)
            default_path = str(Path(last_dir) / "preview.svg") if last_dir else "preview.svg"

        filename, selected_filter = QFileDialog.getSaveFileName(
            self,
            "Export Preview",
            default_path,
            "Scalable Vector Graphics (*.svg);;PDF Documents (*.pdf);;PNG Image (*.png);;JPEG Image (*.jpg)"
        )

        if not filename:
            return

        # Determine the correct extension based on the selected filter
        filter_ext_map = {
            "Scalable Vector Graphics (*.svg)": ".svg",
            "PDF Documents (*.pdf)": ".pdf",
            "PNG Image (*.png)": ".png",
            "JPEG Image (*.jpg)": ".jpg"
        }
        
        # Get current extension
        current_ext = Path(filename).suffix.lower()
        
        # Get expected extension from filter
        expected_ext = filter_ext_map.get(selected_filter, "")
        
        # If no extension or wrong extension, add/replace with the correct one
        if not current_ext or current_ext not in ['.svg', '.pdf', '.png', '.jpg', '.jpeg']:
            if expected_ext:
                filename = str(Path(filename).with_suffix(expected_ext))
        
        # Normalize extension
        ext = Path(filename).suffix.lower()
        
        # Save last used directory
        settings.setValue("SecInterp/lastExportDir", str(Path(filename).parent))

        try:
            self.results.setPlainText(f"Exporting preview to {filename}...")
            
            # Get scene rectangle - use sceneRect() to ensure we capture everything
            # itemsBoundingRect() might miss some elements
            source_rect = self.current_scene.sceneRect()
            
            # If sceneRect is empty, fall back to itemsBoundingRect with margin
            if source_rect.isEmpty():
                source_rect = self.current_scene.itemsBoundingRect()
                source_rect.adjust(-10, -10, 10, 10)
            
            # Ensure dimensions are valid
            width = max(int(source_rect.width()), 100)
            height = max(int(source_rect.height()), 100)
            
            if ext == '.svg':
                generator = QSvgGenerator()
                generator.setFileName(filename)
                generator.setSize(QSize(width, height))
                # ViewBox should start at (0, 0) for SVG, not at source_rect origin
                generator.setViewBox(QRectF(0, 0, width, height))
                generator.setTitle("Section Interpretation Preview")
                generator.setDescription("Generated by SecInterp QGIS Plugin")
                
                painter = QPainter()
                if painter.begin(generator):
                    painter.setRenderHint(QPainter.Antialiasing)
                    # Render the entire scene, translating from source_rect to (0,0)
                    self.current_scene.render(painter, QRectF(0, 0, width, height), source_rect)
                    painter.end()
                else:
                    raise Exception("Failed to initialize SVG painter")
                
            elif ext == '.pdf':
                printer = QPrinter(QPrinter.HighResolution)
                printer.setOutputFormat(QPrinter.PdfFormat)
                printer.setOutputFileName(filename)
                printer.setPageSize(QPageSize(QSizeF(width, height), QPageSize.Point))
                printer.setPageMargins(QMarginsF(0, 0, 0, 0))
                printer.setFullPage(True)
                
                painter = QPainter()
                if painter.begin(printer):
                    painter.setRenderHint(QPainter.Antialiasing)
                    # Render with proper coordinate translation
                    self.current_scene.render(painter, QRectF(0, 0, width, height), source_rect)
                    painter.end()
                else:
                    raise Exception("Failed to initialize PDF printer")
                
            elif ext in ['.png', '.jpg', '.jpeg']:
                # Create image with appropriate background
                image = QImage(width, height, QImage.Format_ARGB32)
                
                if ext in ['.jpg', '.jpeg']:
                    image.fill(QColor(255, 255, 255))  # White background for JPEG
                else:
                    image.fill(QColor(0, 0, 0, 0))  # Transparent for PNG
                
                painter = QPainter()
                if painter.begin(image):
                    painter.setRenderHint(QPainter.Antialiasing)
                    painter.setRenderHint(QPainter.SmoothPixmapTransform)
                    # Render with proper coordinate translation
                    self.current_scene.render(painter, QRectF(0, 0, width, height), source_rect)
                    painter.end()
                    
                    # Save with appropriate quality
                    quality = 95 if ext in ['.jpg', '.jpeg'] else -1
                    if not image.save(filename, None, quality):
                        raise Exception(f"Failed to save image to {filename}")
                else:
                    raise Exception("Failed to initialize image painter")
            else:
                raise Exception(f"Unsupported file extension: {ext}")
            
            self.results.setPlainText(f"✓ Preview exported successfully to:\n{filename}")
            
        except Exception as e:
            import traceback
            error_details = traceback.format_exc()
            self.results.setPlainText(f"⚠ Error exporting preview: {str(e)}\n\nDetails:\n{error_details}")


    def accept_handler(self):
        """Handle the accept button click event."""
        # When running without a QGIS iface (tests), skip strict validation
        if self.iface is None:
            self.accept()
            return

        if not self.validate_inputs():
            return

        self.accept()

    def reject_handler(self):
        """Handle the reject button click event."""
        self.close()

    def validate_inputs(self):
        """Validate the inputs from the dialog."""
        from . import validation_utils as vu
        from qgis.core import QgsWkbTypes, QVariant

        errors = []

        # Validate raster layer selection
        if not self.rasterdem.currentText():
            errors.append("No raster layer selected.")
        else:
            is_valid, error, raster_layer = vu.validate_layer_exists(
                self.rasterdem.currentData()
            )
            if not is_valid:
                errors.append(error)
            else:
                # Validate band number
                band_text = self.band.currentText()
                if band_text:
                    is_valid, error, band_num = vu.validate_integer_input(
                        band_text, min_val=1, field_name="Band number"
                    )
                    if is_valid:
                        is_valid, error = vu.validate_raster_band(
                            raster_layer, band_num
                        )
                        if not is_valid:
                            errors.append(error)
                    else:
                        errors.append(error)

        # Validate crossline layer selection
        if not self.crossline.currentText():
            errors.append("No crossline layer selected")
        else:
            is_valid, error, line_layer = vu.validate_layer_exists(
                self.crossline.currentData()
            )
            if not is_valid:
                errors.append(error)
            else:
                # Validate geometry type
                is_valid, error = vu.validate_layer_geometry(
                    line_layer, QgsWkbTypes.LineGeometry
                )
                if not is_valid:
                    errors.append(error)
                else:
                    # Validate has features
                    is_valid, error = vu.validate_layer_has_features(line_layer)
                    if not is_valid:
                        errors.append(error)

        # Validate output folder
        if not self.dest_fold.text():
            errors.append("Output folder is required")
        else:
            is_valid, error, _ = vu.validate_output_path(self.dest_fold.text())
            if not is_valid:
                errors.append(error)

        # Validate numeric inputs
        # Scale
        is_valid, error, _ = vu.validate_numeric_input(
            self.scale.text(), min_val=1, field_name="Scale"
        )
        if not is_valid:
            errors.append(error)

        # Vertical exaggeration
        is_valid, error, _ = vu.validate_numeric_input(
            self.vertexag.text(), min_val=0.1, field_name="Vertical exaggeration"
        )
        if not is_valid:
            errors.append(error)

        # Buffer distance
        is_valid, error, _ = vu.validate_numeric_input(
            self.buffer_distance.text(), min_val=0, field_name="Buffer distance"
        )
        if not is_valid:
            errors.append(error)

        # Dip scale factor
        is_valid, error, _ = vu.validate_numeric_input(
            self.dip_scale_factor.text(), min_val=0.1, field_name="Dip scale factor"
        )
        if not is_valid:
            errors.append(error)

        # Validate outcrop layer if selected
        if self.outcrop.currentData():
            is_valid, error, outcrop_layer = vu.validate_layer_exists(
                self.outcrop.currentData()
            )
            if not is_valid:
                errors.append(error)
            else:
                # Validate geometry type
                is_valid, error = vu.validate_layer_geometry(
                    outcrop_layer, QgsWkbTypes.PolygonGeometry
                )
                if not is_valid:
                    errors.append(error)
                else:
                    # Validate has features
                    is_valid, error = vu.validate_layer_has_features(outcrop_layer)
                    if not is_valid:
                        errors.append(error)

                    # Validate outcrop name field
                    if not self.ocropname.currentText():
                        errors.append(
                            "Outcrop name field is required when outcrop layer is selected"
                        )
                    else:
                        field_name = self.ocropname.currentData()
                        is_valid, error = vu.validate_field_exists(
                            outcrop_layer, field_name
                        )
                        if not is_valid:
                            errors.append(error)

        # Validate structural layer if selected
        if self.structural.currentData():
            is_valid, error, struct_layer = vu.validate_layer_exists(
                self.structural.currentData()
            )
            if not is_valid:
                errors.append(error)
            else:
                # Validate geometry type
                is_valid, error = vu.validate_layer_geometry(
                    struct_layer, QgsWkbTypes.PointGeometry
                )
                if not is_valid:
                    errors.append(error)
                else:
                    # Validate has features
                    is_valid, error = vu.validate_layer_has_features(struct_layer)
                    if not is_valid:
                        errors.append(error)

                    # Validate dip field
                    if not self.dip.currentText():
                        errors.append(
                            "Dip field is required when structural layer is selected"
                        )
                    else:
                        dip_field = self.dip.currentData()
                        is_valid, error = vu.validate_field_exists(
                            struct_layer, dip_field
                        )
                        if not is_valid:
                            errors.append(error)
                        else:
                            # Validate field type (should be numeric)
                            is_valid, error = vu.validate_field_type(
                                struct_layer, dip_field, [QVariant.Int, QVariant.Double]
                            )
                            if not is_valid:
                                errors.append(error)

                    # Validate strike field
                    if not self.strike.currentText():
                        errors.append(
                            "Strike field is required when structural layer is selected"
                        )
                    else:
                        strike_field = self.strike.currentData()
                        is_valid, error = vu.validate_field_exists(
                            struct_layer, strike_field
                        )
                        if not is_valid:
                            errors.append(error)
                        else:
                            # Validate field type (should be numeric)
                            is_valid, error = vu.validate_field_type(
                                struct_layer,
                                strike_field,
                                [QVariant.Int, QVariant.Double],
                            )
                            if not is_valid:
                                errors.append(error)

        if errors:
            QMessageBox.warning(self, "Validation Error", "\n".join(errors))
            return False

        return True

    def _populate_field_combobox(self, source_combobox, target_combobox):
        """Helper function to populate a combobox with field names from a selected vector layer."""
        try:
            selected_layer_name = source_combobox.currentData()
            target_combobox.clear()

            if not selected_layer_name:
                return

            layers = QgsProject.instance().mapLayersByName(selected_layer_name)
            if not layers:
                return

            vector_layer = layers[0]
            # Use addItem with data parameter so currentData() returns the field name
            for field in vector_layer.fields():
                target_combobox.addItem(field.name(), field.name())
        except Exception as e:
            self.messagebar.pushMessage("Error", str(e), level=Qgis.Critical)

    def populate_outcrop_name_field(self):
        """Populate the outcrop name combobox with the field names of the selected outcrop layer."""
        self._populate_field_combobox(self.outcrop, self.ocropname)

    def populate_dip_name_field(self):
        """Populate the dip name combobox with the field names the selected structural layer."""
        self._populate_field_combobox(self.structural, self.dip)

    def populate_strike_name_field(self):
        """Populate the strike name combobox with the field names the selected structural layer."""
        self._populate_field_combobox(self.structural, self.strike)

    def get_layer_names_by_type(self, layer_type) -> List[str]:
        """Get a list of layer names filtered by the specified layer type.

        This method scans all layers in the current project.

        Args:
              layer_type: The QgsMapLayer type to filter by (e.g.,
        QgsMapLayer.RasterLayer)

        Returns:
              A list of layer names matching the specified type
        """
        layers = QgsProject.instance().mapLayers().values()
        return [layer.name() for layer in layers if layer.type() == layer_type]

    def get_layer_names_by_geometry(self, geometry_type) -> List[str]:
        """Get a list of layer names filtered by the specified geometry type.

        This method scans all layers in the current project.

        Args:
              geometry_type: The QgsWkbTypes geometry type to filter by
        (e.g., QgsWkbTypes.LineGeometry)

        Returns:
              A list of layer names matching the specified geometry type
        """
        layers = QgsProject.instance().mapLayers().values()
        return [
            layer.name()
            for layer in layers
            if layer.type() == QgsMapLayer.VectorLayer
            and QgsWkbTypes.geometryType(layer.wkbType()) == geometry_type
        ]

    def populate_raster_layers(self) -> None:
        """Populate the raster layer combobox with raster layers from the current project."""
        try:
            raster_names = self.get_layer_names_by_type(QgsMapLayer.RasterLayer)
            self.rasterdem.clear()
            self.rasterdem.addItem("Select a raster layer", None)
            if raster_names:
                for name in raster_names:
                    self.rasterdem.addItem(name, name)
            else:
                self.rasterdem.addItem("No raster layers available", None)
        except Exception as e:
            self.messagebar.pushMessage("Error", str(e), level=Qgis.Critical)

    def populate_band_combobox(self):
        """Populate the band combobox with the band count of the selected raster layer."""
        try:
            selected_layer_name = self.rasterdem.currentText()
            if not selected_layer_name:
                self.band.clear()
                return

            layers = QgsProject.instance().mapLayersByName(selected_layer_name)
            if not layers:
                self.band.clear()
                return

            raster_layer = layers[0]
            band_count = raster_layer.bandCount()

            self.band.clear()
            self.band.addItems([str(i) for i in range(1, band_count + 1)])
        except Exception as e:
            self.messagebar.pushMessage("Error", str(e), level=Qgis.Critical)

    def update_resolution_field(self):
        """Calculate and update the resolution field for the selected raster layer."""
        try:
            selected_layer_name = self.rasterdem.currentData()
            if not selected_layer_name:
                self.resln.clear()
                self.unts.clear()
                return

            layers = QgsProject.instance().mapLayersByName(selected_layer_name)
            if not layers:
                self.resln.clear()
                self.unts.clear()
                return

            raster_layer = layers[0]
            raster_crs = raster_layer.crs()
            # Determine map CRS safely: if iface or canvas isn't available
            # (e.g., during unit tests), fall back to using raster CRS.
            try:
                if (
                    self.iface is None
                    or not hasattr(self.iface, "mapCanvas")
                    or self.iface.mapCanvas() is None
                ):
                    map_crs = raster_crs
                else:
                    map_crs = self.iface.mapCanvas().mapSettings().destinationCrs()
            except Exception:
                # Defensive: if anything goes wrong accessing iface/canvas, use raster CRS.
                map_crs = raster_crs
            native_res = raster_layer.rasterUnitsPerPixelX()

            # Validate native resolution
            if native_res <= 0:
                self.resln.setText("Invalid")
                self.unts.setText("")
                self.messagebar.pushMessage(
                    "Warning",
                    "Raster resolution is invalid or zero",
                    level=Qgis.Warning,
                )
                return

            resolution_in_meters = 0

            if raster_crs == map_crs:
                if raster_crs.isGeographic():
                    self.resln.setText(f"{native_res:.6f}")
                    self.unts.setText("°")
                    # For scale calculation, we need resolution in meters.
                    # This is an approximation at the equator.
                    resolution_in_meters = native_res * 111320
                else:
                    self.resln.setText(f"{native_res:.2f}")
                    self.unts.setText(QgsUnitTypes.toString(raster_crs.mapUnits()))
                    if raster_crs.mapUnits() == QgsUnitTypes.DistanceUnit.Meters:
                        resolution_in_meters = native_res
                    elif raster_crs.mapUnits() == QgsUnitTypes.DistanceUnit.Feet:
                        resolution_in_meters = native_res * 0.3048

            else:
                transform = QgsCoordinateTransform(
                    raster_crs, map_crs, QgsProject.instance()
                )
                center = raster_layer.extent().center()
                p1 = center
                p2 = QgsPointXY(center.x() + native_res, center.y())
                p1_transformed = transform.transform(p1)
                p2_transformed = transform.transform(p2)

                if map_crs.isGeographic():
                    # Raster is projected, map is geographic
                    resolution = abs(p2_transformed.x() - p1_transformed.x())
                    self.resln.setText(f"{resolution:.6f}")
                    self.unts.setText("°")
                    # We need to transform back to the raster's projected CRS to get meters
                    # This is a bit complex, let's use the native resolution for scale for now
                    if raster_crs.mapUnits() == QgsUnitTypes.DistanceUnit.Meters:
                        resolution_in_meters = native_res
                    elif raster_crs.mapUnits() == QgsUnitTypes.DistanceUnit.Feet:
                        resolution_in_meters = native_res * 0.3048
                else:
                    # Raster is geographic, map is projected
                    resolution = p1_transformed.distance(p2_transformed)
                    self.resln.setText(f"{resolution:.2f}")
                    self.unts.setText(QgsUnitTypes.toString(map_crs.mapUnits()))
                    if map_crs.mapUnits() == QgsUnitTypes.DistanceUnit.Meters:
                        resolution_in_meters = resolution
                    elif map_crs.mapUnits() == QgsUnitTypes.DistanceUnit.Feet:
                        resolution_in_meters = resolution * 0.3048

            # Calculate suggested scale only if resolution is valid
            if resolution_in_meters > 0:
                suggested_scale = round((resolution_in_meters * 2000) / 1000) * 1000
                self.scale.setText(str(suggested_scale))

        except Exception as e:
            self.messagebar.pushMessage("Error", str(e), level=Qgis.Critical)

    def populate_outcrop_layers(self) -> None:
        """Populate the outcrop layer combobox with polygon geometry vector layers
        from the current project.
        """
        try:
            polygon_names = self.get_layer_names_by_geometry(
                QgsWkbTypes.PolygonGeometry
            )
            self.outcrop.clear()
            self.outcrop.addItem("Do not use Outcrops Layer", None)
            for name in polygon_names:
                self.outcrop.addItem(name, name)
        except Exception as e:
            self.messagebar.pushMessage("Error", str(e), level=Qgis.Critical)

    def populate_structural_layers(self) -> None:
        """Populate the structural layer combobox with point geometry vector layers
        from the current project.
        """
        try:
            point_names = self.get_layer_names_by_geometry(QgsWkbTypes.PointGeometry)
            self.structural.clear()
            self.structural.addItem("Do not use Structural Layer", None)
            for name in point_names:
                self.structural.addItem(name, name)
        except Exception as e:
            self.messagebar.pushMessage("Error", str(e), level=Qgis.Critical)

    def populate_crossline_layers(self) -> None:
        """Populate the crossline layer combobox with line geometry vector layers
        from the current project.
        """
        try:
            line_names = self.get_layer_names_by_geometry(QgsWkbTypes.LineGeometry)
            self.crossline.clear()
            self.crossline.addItem("Select a crossline layer", None)
            if line_names:
                for name in line_names:
                    self.crossline.addItem(name, name)
            else:
                self.crossline.addItem("No line layers available", None)
        except Exception as e:
            self.messagebar.pushMessage("Error", str(e), level=Qgis.Critical)

    @QtCore.pyqtSlot(name="on_search_dest_clicked")
    def on_search_dest_clicked(self):
        """Open a directory dialog to select the destination folder."""
        folder = QFileDialog.getExistingDirectory(
            self,
            self.tr("Select Output Folder"),
            "",
            QFileDialog.ShowDirsOnly | QFileDialog.DontResolveSymlinks,
        )
        if not folder:
            return
        self.dest_fold.setText(folder)
