# -*- coding: utf-8 -*-

"""
/***************************************************************************
 SciPyFilters
                                 A QGIS plugin
 Filter collection implemented with SciPy
 Generated by Plugin Builder: http://g-sherman.github.io/Qgis-Plugin-Builder/
                              -------------------
        begin                : 2024-03-03
        copyright            : (C) 2024 by Florian Neukirchen
        email                : mail@riannek.de
 ***************************************************************************/

/***************************************************************************
 *                                                                         *
 *   This program is free software; you can redistribute it and/or modify  *
 *   it under the terms of the GNU General Public License as published by  *
 *   the Free Software Foundation; either version 2 of the License, or     *
 *   (at your option) any later version.                                   *
 *                                                                         *
 ***************************************************************************/
"""

__author__ = 'Florian Neukirchen'
__date__ = '2024-03-03'
__copyright__ = '(C) 2024 by Florian Neukirchen'

# This will get replaced with a git SHA1 when you do a git archive

__revision__ = '$Format:%H$'


from osgeo import gdal
import numpy as np
import json
from qgis.PyQt.QtCore import QCoreApplication
from qgis._core import QgsProcessingContext, QgsProcessingFeedback
from qgis.core import (QgsProcessingAlgorithm,
                       QgsProcessingParameterString,
                       QgsProcessingParameterDefinition,
                       QgsProcessingParameterNumber,
                       QgsProcessingException,
                       QgsProcessingLayerPostProcessorInterface,
                       QgsProcessingParameterBoolean,
                       QgsProcessingParameterRasterLayer,
                       QgsProcessingParameterEnum,
                       QgsProcessingParameterRasterDestination)

from processing.core.ProcessingConfig import ProcessingConfig


from scipy_filters.helpers import (str_to_array, 
                      convert_docstring_to_html,
                      bandmean,
                      minimumvalue,
                      maximumvalue,
                      centralvalue,
                      number_of_windows,
                      get_windows,
                      is_in_dtype_range,
                      get_np_dtype,
                      DEFAULTWINDOWSIZE)

from scipy_filters.ui.i18n import tr
from scipy_filters.scipy_algorithm_baseclasses import groups

class SciPyFilterNoDataMask(QgsProcessingAlgorithm):
    """
    Return mask of no data value, with 1 for no data, 0 for data.

    **Seperate** Return seperate mask for each band. Otherwise, 
    return one band, with value 1 if any band has no data.

    """
    SEPARATE = 'SEPARATE'
    OUTPUT = 'OUTPUT'
    INPUT = 'INPUT'

    _name = 'no_data_mask'
    _displayname = tr('No data mask')

    def initAlgorithm(self, config):

        try:
            self.windowsize = int(ProcessingConfig.getSetting('WINDOWSIZE'))
        except TypeError:
            self.windowsize = DEFAULTWINDOWSIZE
        if self.windowsize == 0:
            self.windowsize = None

        self.addParameter(
            QgsProcessingParameterRasterLayer(
                self.INPUT,
                tr('Input layer'),
            )
        )

        self.addParameter(QgsProcessingParameterBoolean(
            self.SEPARATE,
            tr('Each band seperate'),
            optional=True,
            defaultValue=False,
        ))

        self.addParameter(
            QgsProcessingParameterRasterDestination(
                self.OUTPUT,
                self.displayName(),
            ))
        
    def get_parameters(self, parameters, context):
        
        self.inputlayer = self.parameterAsRasterLayer(parameters, self.INPUT, context)
        self.output_raster = self.parameterAsOutputLayer(parameters, self.OUTPUT,context)
        self.separate = self.parameterAsBool(parameters, self.SEPARATE, context)


    def processAlgorithm(self, parameters, context, feedback):
        """
        Here is where the processing itself takes place.
        """
        feedback.setProgress(0)

        self.get_parameters(parameters, context)

        if not self.inputlayer.providerType() == "gdal":
            feedback.reportError(f"Raster provider {self.inputlayer.providerType()} is not supported, must be raster layer with a local file", fatalError = True)
            return {}

        self.ds = gdal.Open(self.inputlayer.source())

        if not self.ds:
            raise Exception("Failed to open Raster Layer")
        
        self.bandcount = self.ds.RasterCount

        if self.separate:
            outbands = self.bandcount
        else:
            outbands = 1

        # Prepare output
        driver = gdal.GetDriverByName('GTiff')

        self.out_ds = driver.Create(
            self.output_raster, 
            xsize = self.ds.RasterXSize,
            ysize = self.ds.RasterYSize,
            bands = outbands,
            eType = 1) # Data type is Byte (uint8)
        
        self.out_ds.SetGeoTransform(self.ds.GetGeoTransform())
        self.out_ds.SetProjection(self.ds.GetProjection())

        if feedback.isCanceled():
            return {}
        
        total = number_of_windows(self.ds.RasterXSize, self.ds.RasterYSize, windowsize=self.windowsize) + 1
        counter = 1
        pixelcount = 0

        windows = get_windows(self.ds.RasterXSize, self.ds.RasterYSize, windowsize=self.windowsize, margin=0)

        for win in windows:
            a = self.ds.ReadAsArray(*win.gdalin)
            nodata_mask = np.zeros((a.shape), dtype=bool)

            # Just in case get no data value of each band seperately
            for i in range(self.bandcount):
                nodata = self.ds.GetRasterBand(i+1).GetNoDataValue()
                if a.ndim == 2:
                    nodata_mask[a == nodata] = True
                elif a.ndim == 3:
                    nodata_mask[i][a[i] == nodata] = True
                    

            if self.separate or a.ndim == 2:
                self.out_ds.WriteArray(nodata_mask, *win.gdalout)
                pixelcount = pixelcount + np.sum(np.any(nodata_mask, axis=0).astype(int))
            else:
                self.out_ds.WriteArray(np.any(nodata_mask, axis=0), *win.gdalout)
                pixelcount = pixelcount + np.sum(np.any(nodata_mask, axis=0).astype(int))
                               
            feedback.setProgress(counter * 100 / total)
            counter += 1
            if feedback.isCanceled():
                return {}
                
        # Free some memory
        self.ds = None
        a = None

        # Close the dataset to write file to disk
        self.out_ds = None 

        feedback.pushInfo(tr("Number of pixels with no data value: {} ").format(pixelcount))
        feedback.setProgress(100)

        return {
            self.OUTPUT: self.output_raster, 
            'no data pixels': pixelcount,
            }
    
    def shortHelpString(self):
        """
        Returns the help string that is shown on the right side of the 
        user interface.
        """
        docstring = self.__doc__
        return convert_docstring_to_html(docstring)
    

    def name(self):
        return self._name
    
    def displayName(self):
        """
        Returns the translated algorithm name, which should be used for any
        user-visible display of the algorithm name.
        """
        return tr(self._displayname)
    
    def group(self):
        """
        Returns the name of the group this algorithm belongs to. This string
        should be localised.
        """
        s = groups.get(self.groupId())
        return tr(s)
    
    def groupId(self):
        """
        Returns the unique ID of the group this algorithm belongs to. This
        string should be fixed for the algorithm, and must not be localised.
        The group id should be unique within each provider. Group id should
        contain lowercase alphanumeric characters only and no spaces or other
        formatting characters.
        """
        return "no_data"
    
    def createInstance(self):
        return SciPyFilterNoDataMask()
    


class SciPyFilterApplyNoDataMask(QgsProcessingAlgorithm):
    """
    Apply no data mask to raster layer

    Set all cells where the mask layer is not 0 to no data.

    **No data mask** Raster layer, either with 1 band or same number of bands as input layer. 
    Must have same projection, geotransform and width / height as input layer. 
    Every cell with 0 in the mask layer is considered to be a data cell, 
    with 1 (or any non zero number) as no data value. 
    
    **No data value** Value to be used as proxy for no data. Must be in the range 
    of the data type. If data type is integer, decimal places are truncated.

    **Change** Optionally add or substract a value from each cell where the value is 
    equal to the no data value, to avoid to set the cell to no data. 
    If data type is integer, decimal places are truncated.

    """
    MASK = 'MASK'
    OUTPUT = 'OUTPUT'
    INPUT = 'INPUT'
    NODATA = 'NODATA'
    CHANGE = 'CHANGE'

    _name = 'apply_no_data_mask'
    _displayname = tr('Apply no data mask')

    def initAlgorithm(self, config):

        try:
            self.windowsize = int(ProcessingConfig.getSetting('WINDOWSIZE'))
        except TypeError:
            self.windowsize = DEFAULTWINDOWSIZE
        if self.windowsize == 0:
            self.windowsize = None

        self.addParameter(
            QgsProcessingParameterRasterLayer(
                self.INPUT,
                tr('Input layer'),
            )
        )

        self.addParameter(
            QgsProcessingParameterRasterLayer(
                self.MASK,
                tr('No data mask'),
            )
        )

        self.addParameter(QgsProcessingParameterNumber(
            self.NODATA,
            tr('No data value'),
            QgsProcessingParameterNumber.Type.Double,
            defaultValue=0, 
            optional=False, 
            minValue=0, 
            # maxValue=100
            ))  
        
        self.addParameter(QgsProcessingParameterNumber(
            self.CHANGE,
            tr('Change (+/-) value if same as no data value'),
            QgsProcessingParameterNumber.Type.Double,
            defaultValue=0, 
            optional=True, 
            ))  


        self.addParameter(
            QgsProcessingParameterRasterDestination(
                self.OUTPUT,
                self.displayName(),
            ))
        
    def checkParameterValues(self, parameters, context):
        inputlayer = self.parameterAsRasterLayer(parameters, self.INPUT, context)
        mask = self.parameterAsRasterLayer(parameters, self.MASK, context)
        nodata = self.parameterAsDouble(parameters, self.NODATA, context)
        change = self.parameterAsDouble(parameters, self.CHANGE, context)

        if mask == inputlayer:
            return False, tr("Mask layer and input layer can't be identical")

        if mask.bandCount() != 1 and mask.bandCount() != inputlayer.bandCount():
            return False, tr("Mask layer must have 1 band or same number of bands as input")

        if not inputlayer.providerType() == "gdal":
            return False, tr("Raster provider {} is not supported, must be raster layer with a local file".format(inputlayer.providerType()))
        
        if not mask.providerType() == "gdal":
            return False, tr("Mask: provider {} is not supported, must be raster layer with a local file".format(inputlayer.providerType()))

        ds = gdal.Open(inputlayer.source())
        if not ds:
            raise Exception("Failed to open Raster Layer")
        
        dtype = ds.GetRasterBand(1).DataType

        mask_ds = gdal.Open(mask.source())
        if not mask_ds:
            raise Exception("Failed to open Mask Layer")

        if not (mask_ds.GetProjection() == ds.GetProjection()
                    and mask_ds.RasterXSize == ds.RasterXSize
                    and mask_ds.RasterYSize == ds.RasterYSize
                    and mask_ds.GetGeoTransform() == ds.GetGeoTransform()):
                return False, tr("Mask layer does not match input layer")

        if dtype == 0:
            return False, tr("Unkown data type")
        if dtype <= 5:
            npdtype = get_np_dtype(dtype)
            # Integer
            if not is_in_dtype_range(int(nodata), npdtype):
                return False, tr("No data value is out of range of data type")
            if not is_in_dtype_range((int(nodata) + int(change)), npdtype):
                return False, tr("Changed value is out of range of data type")

        return super().checkParameterValues(parameters, context)
    
        
    def get_parameters(self, parameters, context):
        
        self.inputlayer = self.parameterAsRasterLayer(parameters, self.INPUT, context)
        self.mask = self.parameterAsRasterLayer(parameters, self.MASK, context)
        self.output_raster = self.parameterAsOutputLayer(parameters, self.OUTPUT,context)
        self.nodata = self.parameterAsDouble(parameters, self.NODATA, context)
        self.change = self.parameterAsDouble(parameters, self.CHANGE, context)



    def processAlgorithm(self, parameters, context, feedback):
        """
        Here is where the processing itself takes place.
        """
        feedback.setProgress(0)

        self.get_parameters(parameters, context)

        self.ds = gdal.Open(self.inputlayer.source())

        if not self.ds:
            raise Exception("Failed to open Raster Layer")
        
        self.maskds = gdal.Open(self.mask.source())
        if not self.maskds:
            raise Exception("Failed to open Mask Layer")
            
        self.bandcount = self.ds.RasterCount

        dtype = self.ds.GetRasterBand(1).DataType

        if dtype <= 5:
            # Integer
            self.nodata = int(self.nodata)
            self.change = int(self.nodata) + int(self.change)
        else:
            self.change = self.nodata + self.change

        feedback.pushInfo(tr("No data value: {} \nChanged value: {}").format(self.nodata, self.change))

        # Prepare output
        driver = gdal.GetDriverByName('GTiff')

        self.out_ds = driver.Create(
            self.output_raster, 
            xsize = self.ds.RasterXSize,
            ysize = self.ds.RasterYSize,
            bands = self.bandcount,
            eType = dtype) 
        
        self.out_ds.SetGeoTransform(self.ds.GetGeoTransform())
        self.out_ds.SetProjection(self.ds.GetProjection())

        if feedback.isCanceled():
            return {}
        
        total = number_of_windows(self.ds.RasterXSize, self.ds.RasterYSize, windowsize=self.windowsize) + 1
        counter = 1

        windows = get_windows(self.ds.RasterXSize, self.ds.RasterYSize, windowsize=self.windowsize, margin=0)

        for win in windows:
            a = self.ds.ReadAsArray(*win.gdalin)
            mask = self.maskds.ReadAsArray(*win.gdalin) 
            mask = mask.astype(bool)

            a[a == self.nodata] = self.change
            if mask.ndim == a.ndim:
                a[mask] = self.nodata
            else:
                # a is multiband, mask is single band
                for i in range(self.bandcount):
                    a[i][mask] = self.nodata

            self.out_ds.WriteArray(a, *win.gdalout)

            feedback.setProgress(counter * 100 / total)
            counter += 1
            if feedback.isCanceled():
                return {}
                
        # Free some memory
        self.ds = None
        a = None

        # Set no data on all bands
        for b in range(1, self.bandcount + 1):
            self.out_ds.GetRasterBand(b).SetNoDataValue(self.nodata) 
        

        # Close the dataset to write file to disk
        self.out_ds = None 


        feedback.setProgress(100)

        return {
            self.OUTPUT: self.output_raster
            }
    
    def shortHelpString(self):
        """
        Returns the help string that is shown on the right side of the 
        user interface.
        """
        docstring = self.__doc__
        return convert_docstring_to_html(docstring)
    

    def name(self):
        return self._name
    
    def displayName(self):
        """
        Returns the translated algorithm name, which should be used for any
        user-visible display of the algorithm name.
        """
        return tr(self._displayname)
    
    def group(self):
        """
        Returns the name of the group this algorithm belongs to. This string
        should be localised.
        """
        s = groups.get(self.groupId())
        return tr(s)
    
    def groupId(self):
        """
        Returns the unique ID of the group this algorithm belongs to. This
        string should be fixed for the algorithm, and must not be localised.
        The group id should be unique within each provider. Group id should
        contain lowercase alphanumeric characters only and no spaces or other
        formatting characters.
        """
        return "no_data"
    
    def createInstance(self):
        return SciPyFilterApplyNoDataMask()
    


class SciPyFilterFillNoData(QgsProcessingAlgorithm):
    """
    Fill no data cells (all bands)

    **Mode** Fill with 0, fill with value, fill with approximate band mean, \
    fill with exact band mean, fill with minimum of dtype range, \
    fill with maximum of dtype range, fill with central value of dtype range.

    **Value** Used for mode *fill with value*
    """
    
    VALUE = 'VALUE'
    MODE = 'MODE'
    OUTPUT = 'OUTPUT'
    INPUT = 'INPUT'

    _name = 'fill_no_data'
    _displayname = tr('Fill no data')

    def initAlgorithm(self, config):

        try:
            self.windowsize = int(ProcessingConfig.getSetting('WINDOWSIZE'))
        except TypeError:
            self.windowsize = DEFAULTWINDOWSIZE
        if self.windowsize == 0:
            self.windowsize = None

        self.addParameter(
            QgsProcessingParameterRasterLayer(
                self.INPUT,
                tr('Input layer'),
            )
        )

        mode_labels = [tr('Fill with 0'), tr('Fill with value'), tr('Fill with approximate band mean'), tr('Fill with exact band mean'), tr('Fill with minimum of dtype range'), tr('Fill with maximum of dtype range'), tr('Fill with central value of dtype range')]

        self.addParameter(QgsProcessingParameterEnum(
            self.MODE,
            tr('Filling Mode'),
            mode_labels,
            defaultValue=0)) 
        
        self.addParameter(QgsProcessingParameterNumber(
            self.VALUE,
            tr('Value'),
            QgsProcessingParameterNumber.Type.Double,
            defaultValue=0, 
            optional=False, 
            ))  

        self.addParameter(
            QgsProcessingParameterRasterDestination(
                self.OUTPUT,
                self.displayName(),
            ))
    
    
    def checkParameterValues(self, parameters, context):
        inputlayer = self.parameterAsRasterLayer(parameters, self.INPUT, context)
        mode = self.parameterAsInt(parameters, self.MODE, context)
        value = self.parameterAsDouble(parameters, self.VALUE, context)

        if not inputlayer.providerType() == "gdal":
            return False, tr("Raster provider {} is not supported, must be raster layer with a local file".format(inputlayer.providerType()))

        ds = gdal.Open(inputlayer.source())
        if not ds:
            raise Exception("Failed to open Raster Layer")
        
        
        if mode == 1: # Fill with value
            dtype = ds.GetRasterBand(1).DataType
            if dtype == 0:
                return False, tr("Unkown data type")
            if dtype <= 5:
                npdtype = get_np_dtype(dtype)
                # Integer
                if not is_in_dtype_range(int(value), npdtype):
                    return False, tr("No data value is out of range of data type")

        return super().checkParameterValues(parameters, context)
    

    def get_parameters(self, parameters, context):
        
        self.inputlayer = self.parameterAsRasterLayer(parameters, self.INPUT, context)
        self.output_raster = self.parameterAsOutputLayer(parameters, self.OUTPUT,context)
        self.mode = self.parameterAsInt(parameters, self.MODE, context)
        self.value = self.parameterAsDouble(parameters, self.VALUE, context)


    def processAlgorithm(self, parameters, context, feedback):
        """
        Here is where the processing itself takes place.
        """
        feedback.setProgress(0)

        self.get_parameters(parameters, context)

        self.ds = gdal.Open(self.inputlayer.source())

        if not self.ds:
            raise Exception("Failed to open Raster Layer")
        
        self.bandcount = self.ds.RasterCount
        self.dtype = self.ds.GetRasterBand(1).DataType
        npdtype = get_np_dtype(self.dtype)

        if 0 < self.dtype <= 5: # integer
            self.value = int(self.value)

        # Set fill no data values (np.array with shape == bandcount)
        if self.mode == 0:
            self.fillvalue = np.zeros(self.bandcount)
        elif self.mode == 1:
            self.fillvalue = np.zeros(self.bandcount)
            self.fillvalue[:] = self.value
        elif self.mode == 2:
            # approx band mean
            self.fillvalue = np.zeros(self.bandcount)
            for i in range(self.bandcount):
                self.fillvalue[i] = bandmean(self.ds, i+1, True)
        elif self.mode == 3:
            # exact band mean
            self.fillvalue = np.zeros(self.bandcount)
            for i in range(self.bandcount):
                self.fillvalue[i] = bandmean(self.ds, i+1, False)
        elif self.mode == 4:
            self.fillvalue = np.zeros(self.bandcount)
            self.fillvalue[:] = minimumvalue(npdtype)
        elif self.mode == 5:
            self.fillvalue = np.zeros(self.bandcount)
            self.fillvalue[:] = maximumvalue(npdtype)
        else:
            self.fillvalue = np.zeros(self.bandcount)
            self.fillvalue[:] = centralvalue(npdtype)

        feedback.pushInfo(tr("Fill value (each band): {} ").format(self.fillvalue))

        # Prepare output
        driver = gdal.GetDriverByName('GTiff')

        self.out_ds = driver.Create(
            self.output_raster, 
            xsize = self.ds.RasterXSize,
            ysize = self.ds.RasterYSize,
            bands = self.bandcount,
            eType = self.dtype) # Data type is Byte (uint8)
        
        self.out_ds.SetGeoTransform(self.ds.GetGeoTransform())
        self.out_ds.SetProjection(self.ds.GetProjection())

        if feedback.isCanceled():
            return {}
        
        total = self.bandcount * number_of_windows(self.ds.RasterXSize, self.ds.RasterYSize, windowsize=self.windowsize) + 1
        counter = 1

        windows = get_windows(self.ds.RasterXSize, self.ds.RasterYSize, windowsize=self.windowsize, margin=0)

        for win in windows:
            # Just in case get no data value of each band seperately
            for i in range(self.bandcount):
                band = self.ds.GetRasterBand(i+1)
                a = band.ReadAsArray(*win.gdalin)
                nodata = band.GetNoDataValue()

                a[a == nodata] = self.fillvalue[i]

                self.out_ds.GetRasterBand(i+1).WriteArray(a, *win.gdalout)
                               
                feedback.setProgress(counter * 100 / total)
                counter += 1
                if feedback.isCanceled():
                    return {}
                
        # Free some memory
        self.ds = None
        a = None

        # Close the dataset to write file to disk
        self.out_ds = None 

        feedback.setProgress(100)

        return {
            self.OUTPUT: self.output_raster, 
            "fill values": self.fillvalue.tolist(),
            }
    
    def shortHelpString(self):
        """
        Returns the help string that is shown on the right side of the 
        user interface.
        """
        docstring = self.__doc__
        return convert_docstring_to_html(docstring)
    

    def name(self):
        return self._name
    
    def displayName(self):
        """
        Returns the translated algorithm name, which should be used for any
        user-visible display of the algorithm name.
        """
        return tr(self._displayname)
    
    def group(self):
        """
        Returns the name of the group this algorithm belongs to. This string
        should be localised.
        """
        s = groups.get(self.groupId())
        return tr(s)
    
    def groupId(self):
        """
        Returns the unique ID of the group this algorithm belongs to. This
        string should be fixed for the algorithm, and must not be localised.
        The group id should be unique within each provider. Group id should
        contain lowercase alphanumeric characters only and no spaces or other
        formatting characters.
        """
        return "no_data"
    
    def createInstance(self):
        return SciPyFilterFillNoData()
    