# -*- coding: utf-8 -*-

"""
/***************************************************************************
 SciPyFilters
                                 A QGIS plugin
 Filter collection implemented with SciPy
 Generated by Plugin Builder: http://g-sherman.github.io/Qgis-Plugin-Builder/
                              -------------------
        begin                : 2024-03-03
        copyright            : (C) 2024 by Florian Neukirchen
        email                : mail@riannek.de
 ***************************************************************************/

/***************************************************************************
 *                                                                         *
 *   This program is free software; you can redistribute it and/or modify  *
 *   it under the terms of the GNU General Public License as published by  *
 *   the Free Software Foundation; either version 2 of the License, or     *
 *   (at your option) any later version.                                   *
 *                                                                         *
 ***************************************************************************/
"""

__author__ = 'Florian Neukirchen'
__date__ = '2024-03-03'
__copyright__ = '(C) 2024 by Florian Neukirchen'

# This will get replaced with a git SHA1 when you do a git archive

__revision__ = '$Format:%H$'



from scipy import ndimage

from qgis.core import (QgsProcessingParameterDefinition,
                       QgsProcessingException,
                        )

from scipy_filters.scipy_algorithm_baseclasses import SciPyAlgorithm

from scipy_filters.ui.structure_widget import (StructureWidgetWrapper, 
                                  SciPyParameterStructure,)

from scipy_filters.helpers import (str_to_int_or_list, 
                      check_structure, 
                      str_to_array, 
                      morphostructexamples)

from scipy_filters.ui.i18n import tr
from scipy_filters.ui.origin_widget import (OriginWidgetWrapper, 
                               SciPyParameterOrigin,)

class SciPyBinaryFillHolesAlgorithm(SciPyAlgorithm):
    """
    Fill holes in binary shapes by using binary dilations
    
    Calculated with binary_fill_holes from 
    `scipy.ndimage <https://docs.scipy.org/doc/scipy/reference/ndimage.html>`_.

    .. note:: No data cells within the filter radius are filled with 0.

    **Dimension** Calculate for each band separately (2D) 
    or use all bands as a 3D datacube and perform filter in 3D. 
    
    .. note:: bands will be the first axis of the datacube.
    
    **Structure** String representation of array. 
    Must have 2 dimensions if *dimension* is set to 2D. 
    Should have 3 dimensions if *dimension* is set to 3D, 
    but a 2D array is also excepted (a new axis is added as first 
    axis and the result is the same as calculating each band 
    seperately). Examples can be loaded with the load button. 
    For convenience (i.e. when calling from a script), 
    the following shortcuts are accepted as well: 
    "square", "cross", "cross3D", "ball", "cube".
    """

    STRUCTURE = 'STRUCTURE'
    ORIGIN = 'ORIGIN'

    # Overwrite constants of base class
    _name = 'fill_holes'
    _displayname = tr('Morphological (binary) fill holes')
    _outputname = tr("Morphology: Fill holes") # If set to None, the displayname is used 
    _groupid = 'morphological'
    
    # The function to be called
    def get_fct(self):
        return ndimage.binary_fill_holes

 
    def initAlgorithm(self, config):
        super().initAlgorithm(config)

        struct_param = SciPyParameterStructure(
            self.STRUCTURE,
            tr('Structure'),
            defaultValue="[[0, 1, 0],\n[1, 1, 1],\n[0, 1, 0]]",
            examples=morphostructexamples,
            multiLine=True,
            to_int=True,
            optional=True,
            )
        
        struct_param.setMetadata({
            'widget_wrapper': {
                'class': StructureWidgetWrapper
            }
        })

        self.addParameter(struct_param)

        origin_param = SciPyParameterOrigin(
            self.ORIGIN,
            tr('Origin'),
            defaultValue="0",
            optional=False,
            watch="STRUCTURE"
            )
        
        origin_param.setMetadata({
            'widget_wrapper': {
                'class': OriginWidgetWrapper
            }
        })

        origin_param.setFlags(origin_param.flags() | QgsProcessingParameterDefinition.Flag.FlagAdvanced)
        
        self.addParameter(origin_param)



    def get_parameters(self, parameters, context):
        kwargs = super().get_parameters(parameters, context)

        sizelist = [3]

        structure = self.parameterAsString(parameters, self.STRUCTURE, context)
        kwargs['structure'] = str_to_array(structure, self._ndim)

        sizelist.extend(kwargs['structure'].shape)
        self.margin = max(sizelist)

        return kwargs
    
    
    def checkParameterValues(self, parameters, context): 
        dims = self.getDimsForCheck(parameters, context)

        structure = self.parameterAsString(parameters, self.STRUCTURE, context)
        ok, s, shape = check_structure(structure, dims)
        if not ok:
            return (ok, s)
        

        origin = self.parameterAsString(parameters, self.ORIGIN, context)
        origin = str_to_int_or_list(origin)

        if isinstance(origin, list):          
            if len(origin) != dims:
                return (False, tr("Origin does not match number of dimensions"))
            for i in range(dims):
                if shape[i] != 0 and not (-(shape[i] // 2) <= origin[i] <= (shape[i]-1) // 2):
                    return (False, tr("Origin out of bounds of structure"))

        return super().checkParameterValues(parameters, context)
    
    def createInstance(self):
        return SciPyBinaryFillHolesAlgorithm()

