# -*- coding: utf-8 -*-

"""
/***************************************************************************
 SciPyFilters
                                 A QGIS plugin
 Filter collection implemented with SciPy
 Generated by Plugin Builder: http://g-sherman.github.io/Qgis-Plugin-Builder/
                              -------------------
        begin                : 2024-03-03
        copyright            : (C) 2024 by Florian Neukirchen
        email                : mail@riannek.de
 ***************************************************************************/

/***************************************************************************
 *                                                                         *
 *   This program is free software; you can redistribute it and/or modify  *
 *   it under the terms of the GNU General Public License as published by  *
 *   the Free Software Foundation; either version 2 of the License, or     *
 *   (at your option) any later version.                                   *
 *                                                                         *
 ***************************************************************************/
"""

__author__ = 'Florian Neukirchen'
__date__ = '2024-03-03'
__copyright__ = '(C) 2024 by Florian Neukirchen'

# This will get replaced with a git SHA1 when you do a git archive

__revision__ = '$Format:%H$'

from osgeo import gdal
from scipy import ndimage
import numpy as np
import enum
import os

from qgis.core import (QgsProcessingAlgorithm,
                       QgsProcessingParameterRasterLayer,
                       QgsProcessingParameterNumber,
                       QgsProcessingParameterRasterDestination,
                       QgsProcessingParameterEnum,
                       QgsProcessingParameterString,
                       QgsProcessingLayerPostProcessorInterface,
                       QgsProcessingParameterDefinition,
                       QgsProcessingParameterBoolean,
                       QgsProcessingException,
                        )

from processing.core.ProcessingConfig import ProcessingConfig

from scipy_filters.ui.sizes_widget import (SizesWidgetWrapper)
from scipy_filters.ui.dim_widget import (DimsWidgetWrapper, SciPyParameterDims)
from scipy_filters.ui.structure_widget import (StructureWidgetWrapper, 
                                  SciPyParameterStructure,)

from scipy_filters.ui.origin_widget import (OriginWidgetWrapper, 
                               SciPyParameterOrigin,)

from scipy_filters.helpers import (RasterWindow,
                      get_windows,
                      number_of_windows,
                      DEFAULTWINDOWSIZE,
                      wrap_margin,
                      str_to_int_or_list, 
                      check_structure,
                      str_to_array, 
                      footprintexamples,
                      dtype_options,
                      get_np_dtype,
                      is_in_dtype_range,
                      convert_docstring_to_html)

from scipy_filters.ui.i18n import tr

# Group IDs and group names
groups = {
    'edges': tr('Edges'),
    'morphological': tr("Morphological Filters"),
    'statistic': tr('Statistical Filters'),
    'blur': tr('Blur'),
    'convolution': tr('Convolution'),
    'enhance': tr('Enhance'),
    'pixel': tr('Pixel Based Filters'),
    'pca': tr('PCA'),
    'no_data': tr('No Data')
}

# Dimensions the algorithm is working on, more info see below
class Dimensions(enum.Enum):
    nD = 2         # users can decide between 1D, 2D, 3D
    twoD = 0       # Seperate for each band
    threeD = 1     # 3D filter in data cube


class SciPyAlgorithm(QgsProcessingAlgorithm):
    """
    Lowest level base class for algorithms based on SciPy.

    The function to be called is returned by the function
    get_fct(), needs to be overwritten by inheriting classes.
    Name, displayname are set as class variables, to be 
    overwritten by inheriting classes. Some of them must be
    put in tr() for translation.

    The shortHelpString is generated from the docstring of the classes.

    In inheriting classes, parameters can be added by:
    1) setting constants as class variables
    2) overwriting either insert_parameters (added below input parameter)
       or initAlgorithm (added above output parameter). 
       Don't forget to call the same function on super().
    3) overwriting the function get_parameters (don't forget to call
       the same function on super())

    Inheriting classes must implement createInstance, returning an
    instance of the class, e.g.:

    def createInstance(self):
        return SciPyGaussianAlgorithm()
    """

    # Constants used to refer to parameters and outputs. They will be
    # used when calling the algorithm from another algorithm, or when
    # calling from the QGIS console.

    OUTPUT = 'OUTPUT'
    INPUT = 'INPUT'
    DIMENSION = 'DIMENSION'
    DTYPE = 'DTYPE'
    BANDSTATS = 'BANDSTATS'
    
    # The following constants are supposed to be overwritten
    _name = 'name, short, lowercase without spaces'
    _displayname = 'User-visible name'
    # Output layer name: If set to None, the displayname is used 
    # Can be changed while getting the parameters.
    _outputname = None 
    _groupid = "" 

    
    _outbands = None # Optionally change the number of output bands
    _band_desc = None # Option to set description of bands (provide list with names)

    # Margin to be included in the window used for calculation
    # Must always be set according to the filter size when getting params
    margin = 0

    # Must be set to True if bordermode is "wrap" and we are using windows
    wrapping = False

    # output default data type

    # Note: the np.dtype of the output array is added as "output" 
    # to the kwargs. This works for ndimage filters,
    # otherwise it must be handled by the custom function

    _default_dtype = 0 # Optionally change default output dtype (value = idx of combobox)
    
    # Dimensions the algorithm is working on. 
    # The numbers match the index in the list of GUI strings (below).
    # Default during init is nD (for algorithms based on scipy.ndimage),
    # giving users a choice (and calling from a script without setting 
    # DIMENSIONS defaults to 2D in this case). This allows to write filters
    # that are not working in n dimensions.
    # Calling from a script, DIMENSION must be in (0,1,2) and match the 
    # values of the enum!

    _dimension = Dimensions.nD
    _ndim = None # to be set while getting parameters


    # Return the function to be called, to be overwritten
    def get_fct(self):
        return ndimage.laplace
    
    # Function to insert Parameters (overwrite in inherited classes)
    def insert_parameters(self, config):
        return

    # Init Algorithm
    def initAlgorithm(self, config):
        """
        Here we define the inputs and output of the algorithm, along
        with some other properties.
        """
        try:
            self.windowsize = int(ProcessingConfig.getSetting('WINDOWSIZE'))
        except TypeError:
            self.windowsize = DEFAULTWINDOWSIZE
        if self.windowsize == 0:
            self.windowsize = None

        # Border modes and labels (needs to be done here for translation to work)
        self.modes = ['reflect', 'constant', 'nearest', 'mirror', 'wrap']
        self.mode_labels = [tr('Reflect'), tr('Constant'), tr('Nearest'), tr('Mirror'), tr('Wrap')]

        # Some Algorithms will add a masklayer
        self.masklayer = None

        # Add parameters
        self.addParameter(
            QgsProcessingParameterRasterLayer(
                self.INPUT,
                tr('Input layer'),
            )
        )

        # Strings for the GUI
        dimension_options = [tr('2D (Separate for each band)'),
                            tr('3D (All bands as a 3D data cube)')]

        if self._dimension == Dimensions.nD:
            # Algorithms based on scipy.ndimage can have any number of dimensions
            dim_param = SciPyParameterDims(
                self.DIMENSION,
                tr('Dimension'),
                dimension_options,
                defaultValue=0,
                optional=False,)

            dim_param.setMetadata({
                'widget_wrapper': {
                    'class': DimsWidgetWrapper
                }
            })

            self.addParameter(dim_param)


        # Insert Parameters 
        self.insert_parameters(config)

        stats_param = QgsProcessingParameterBoolean(
            self.BANDSTATS,
            tr('Calculate band statistics'),
            optional=True,
            defaultValue=True,
        )

                
        stats_param.setFlags(stats_param.flags() | QgsProcessingParameterDefinition.Flag.FlagAdvanced)
      
        self.addParameter(stats_param)

        # Force translation of first dtype option ("same as input")
        dtype_options_tr = dtype_options
        dtype_options_tr[0] = tr(dtype_options_tr[0])
        
        dtype_param = QgsProcessingParameterEnum(
            self.DTYPE,
            tr('Output data type'),
            dtype_options_tr,
            defaultValue=self._default_dtype,
            optional=True)
        
        # Set as advanced parameter
        dtype_param.setFlags(dtype_param.flags() | QgsProcessingParameterDefinition.Flag.FlagAdvanced)
        self.addParameter(dtype_param)


        # Output

        if not self._outputname:
            self._outputname = self._displayname

        self.addParameter(
            QgsProcessingParameterRasterDestination(
                self.OUTPUT,
            tr(self._outputname)))
        
        
    def get_parameters(self, parameters, context):
        """
        Factoring this out of processAlgorithm allows to add Parameters
        in classes inheriting form this base class by overwriting this
        function. But always call kwargs = super().get_parameters(...)
        first!

        Returns kwargs dictionary and sets variables self.variable for 
        non-keyword arguments.

        This is the most basic base class and kwargs is empty {}.
        """
        self.inputlayer = self.parameterAsRasterLayer(parameters, self.INPUT, context)
        self.output_raster = self.parameterAsOutputLayer(parameters, self.OUTPUT, context)
        self.bandstats = self.parameterAsBool(parameters, self.BANDSTATS, context)
 
        self._outdtype = self.parameterAsInt(parameters, self.DTYPE, context)
        if not self._outdtype:
            # 0 will be changed to dtype of input layer
            self._outdtype = 0
        
        if self._dimension == Dimensions.nD:
            dimension = self.parameterAsInt(parameters, self.DIMENSION, context)
            if dimension == 1:
                self._dimension = Dimensions.threeD
                self._ndim = 3
            else:
                # Default to 2D
                self._dimension = Dimensions.twoD
                self._ndim = 2

        return {}


    def processAlgorithm(self, parameters, context, feedback):
        """
        Here is where the processing itself takes place.
        """

        feedback.setProgress(0)

        # Get Parameters
        kwargs = self.get_parameters(parameters, context)
        # print("kwargs\n", kwargs)

        self.fct = self.get_fct()

        if not self.inputlayer.providerType() == "gdal":
            feedback.reportError(f"Raster provider {self.inputlayer.providerType()} is not supported, must be raster layer with a local file", fatalError = True)
            return {}

        # Open Raster with GDAL
        self.ds = gdal.Open(self.inputlayer.source())

        if not self.ds:
            raise Exception("Failed to open Raster Layer")
        
        self.bandcount = self.ds.RasterCount

        self._indtype = self.ds.GetRasterBand(1).DataType
        if self._indtype in (8,9,10,11):
            feedback.reportError("Error: Complex is not supported", fatalError = True)
            return {}

        # Set to 2D if layer has only one band
        if self.bandcount == 1:
            self._dimension = Dimensions.twoD

        # Eventually open mask layer 
        if self.masklayer:
            if not self.masklayer.providerType() == "gdal":
                raise TypeError("Mask: provider {} is not supported, must be raster layer with a local file".format(self.masklayer.providerType()))

            self.mask_ds = gdal.Open(self.masklayer.source())
            if not self.mask_ds:
                raise Exception("Failed to open Mask Layer")
            
            # Mask must have same crs etc.
            if not (self.mask_ds.GetProjection() == self.ds.GetProjection()
                    and self.mask_ds.RasterXSize == self.ds.RasterXSize
                    and self.mask_ds.RasterYSize == self.ds.RasterYSize
                    and self.mask_ds.GetGeoTransform() == self.ds.GetGeoTransform()):
                feedback.pushInfo(tr("Mask layer does not match input layer, reprojecting mask."))

                geoTransform = self.ds.GetGeoTransform()

                kwargs_w = {"format": "GTiff", 'resampleAlg':'near'}
                kwargs_w["xRes"] = geoTransform[1]
                kwargs_w["yRes"] = abs(geoTransform[5])

                minx = geoTransform[0]
                maxy = geoTransform[3]
                maxx = minx + geoTransform[1] * self.ds.RasterXSize
                miny = maxy + geoTransform[5] * self.ds.RasterYSize

                kwargs_w["outputBounds"] = (minx, miny, maxx, maxy)

                warped_mask = gdal.Warp("/vsimem/tmpmask", self.mask_ds, **kwargs_w)
                kwargs['mask'] = warped_mask.GetRasterBand(1).ReadAsArray()
            else:
                feedback.pushInfo(tr("Mask layer does match input layer."))
                kwargs['mask'] = self.mask_ds.GetRasterBand(1).ReadAsArray()


        # Prepare output
        driver = gdal.GetDriverByName('GTiff')

        if not self._outbands:
            self._outbands = self.bandcount

        if self._outdtype == 0:
            # Set to dtype of input dataset
            self._outdtype = self._indtype

        # ndimage filters have a parameter for output dtype
        # For other function, it must be converted by hand
        kwargs['output'] = get_np_dtype(self._outdtype)

        # print(kwargs['output'])

        self.out_ds = driver.Create(
            self.output_raster, 
            xsize = self.ds.RasterXSize,
            ysize = self.ds.RasterYSize,
            bands = self._outbands,
            eType = self._outdtype)
        
        self.out_ds.SetGeoTransform(self.ds.GetGeoTransform())
        self.out_ds.SetProjection(self.ds.GetProjection())

        # output no data value
        self._nodata = self.ds.GetRasterBand(1).GetNoDataValue()
        if self._nodata is not None:
            if not is_in_dtype_range(self._nodata, kwargs['output']):
                self._nodata = np.iinfo(kwargs['output']).min

        if feedback.isCanceled():
            return {}
        
        total = number_of_windows(self.ds.RasterXSize, self.ds.RasterYSize, windowsize=self.windowsize) + 1
        if total > 2:
            feedback.pushInfo(tr("Calculating in {} windows with a margin of {} pixels.").format(total - 1, self.margin))

        # Start the actual work

        if self._dimension == Dimensions.twoD:
            # Iterate over bands and calculate 
            total = total * self.bandcount
            counter = 1
            for i in range(1, self.bandcount + 1):
                windows = get_windows(self.ds.RasterXSize, self.ds.RasterYSize, windowsize=self.windowsize, margin=self.margin)

                for win in windows:
                    a = self.ds.GetRasterBand(i).ReadAsArray(*win.gdalin)

                    if self.wrapping:
                        # Make shure that border mode "wrap" gets the far side of the complete array
                        wrap_margin(a, self.ds, win, band=i)


                    # Handle no data value
                    nodata = self.ds.GetRasterBand(i).GetNoDataValue()
                    nodata_mask = (a == nodata)
                    self.fill_nodata(a, nodata, i) 

                    # The actual function
                    filtered = self.fct(a, **kwargs)
                    
                    # Replace no data cells with output no data value
                    if self._nodata:
                        filtered[nodata_mask] = self._nodata

                    # chop off margin
                    slices = win.getslice(2)
                    filtered = filtered[slices]

                    self.out_ds.GetRasterBand(i).WriteArray(filtered, *win.gdalout)

                    feedback.setProgress(counter * 100 / total)
                    counter += 1
                    if feedback.isCanceled():
                        return {}
                
        elif self._dimension == Dimensions.threeD:
            counter = 1
            dims = 3
            if self._outbands == 1:
                dims = 2

            windows = get_windows(self.ds.RasterXSize, self.ds.RasterYSize, windowsize=self.windowsize, margin=self.margin)

            for win in windows:
                a = self.ds.ReadAsArray(*win.gdalin)       

                if self.wrapping:
                    # Make shure that border mode "wrap" gets the far side of the complete array
                    wrap_margin(a, self.ds, win)

                # Handle no data value 
                # (for each band seperately, some data formats support different values per band)
                nodata_mask = np.zeros((a.shape[-2], a.shape[-1]), dtype=bool)

                for i in range(self.bandcount):
                    nodata = self.ds.GetRasterBand(i+1).GetNoDataValue()
                    nodata_mask[a[i] == nodata] = True
                    self.fill_nodata(a[i], nodata, i+1)

                # The actual function
                filtered = self.fct(a, **kwargs)

                # chop off margin
                slices = win.getslice(dims)
                filtered = filtered[slices]

                # Replace no data cells with output no data value
                if self._nodata:
                    if dims == 3:
                        filtered[:,nodata_mask] = self._nodata 
                    else:
                        filtered[nodata_mask] = self._nodata 
                nodata_mask = None
                
                self.out_ds.WriteArray(filtered, *win.gdalout)

                feedback.setProgress(counter * 100 / total)
                counter += 1
                if feedback.isCanceled():
                    return {}

        # Set no data value on all bands
        if self._nodata is not None:
            for b in range(1, self._outbands + 1):
                self.out_ds.GetRasterBand(b).SetNoDataValue(self._nodata) 


        # Calculate and write band statistics (min, max, mean, std)
        if self.bandstats:
            for b in range(1, self._outbands + 1):
                band = self.out_ds.GetRasterBand(b)
                stats = band.GetStatistics(0,1)
                band.SetStatistics(*stats)

        # Check and feedback
        self.checkAndComplain(feedback)

        # Optionally change band desc
        if self._band_desc:
            for i in range(self._outbands):
                band = self.out_ds.GetRasterBand(i + 1)
                band.SetDescription(self._band_desc[i])

        # Free some memory
        self.ds = None
        a = None

        # Close the dataset to write file to disk
        self.out_ds = None 

        feedback.setProgress(100)
        
        # Optionally rename the output layer
        is_temporary = "OUTPUT" in os.path.basename(self.output_raster)

        if self._outputname and is_temporary:
            global renamer
            renamer = self.Renamer(tr(self._outputname))
            context.layerToLoadOnCompletionDetails(self.output_raster).setPostProcessor(renamer)


        return {self.OUTPUT: self.output_raster}


    def checkParameterValues(self, parameters, context):
        # Check parameters before starting processing, eventually giving feedback and stopping
        dim_option = self.parameterAsInt(parameters, self.DIMENSION, context)
        layer = self.parameterAsRasterLayer(parameters, self.INPUT, context)

        if not layer.providerType() == "gdal":
            return False, tr("Raster provider {} is not supported, must be raster layer with a local file".format(layer.providerType()))

        # 3D only possible with more than 1 bands
        if dim_option == 1 and layer.bandCount() == 1:
            return (False, tr("3D only possible if input layer has more than 1 bands"))
            
        return super().checkParameterValues(parameters, context)
    
    @property
    def dims(self):
        d = 2
        if self._dimension == Dimensions.threeD:
            d = 3
        return d

    def getDimsForCheck(self, parameters, context):
        dims = 2
        if self._dimension == Dimensions.nD:
            dim_option = self.parameterAsInt(parameters, self.DIMENSION, context)
            if dim_option == 1:
                dims = 3
        return dims
    
    def checkAndComplain(self, feedback):
        if 1 <= self._outdtype <= 5: # integer types
            info_in = np.iinfo(get_np_dtype(self._indtype))
            info_out = np.iinfo(get_np_dtype(self._outdtype))
            if info_in.min < info_out.min or info_in.max > info_out.max:
                msg = tr("Warning, the range of output datatype is not in the range of the input datatype. Clipping is likely.")
                feedback.reportError(msg, fatalError=False)


    def fill_nodata(self, array, nodata, band=None):
        """
        Replace nodata value with zero (inplace). 
        
        Function can be overwritten in inheriting classes to support 
        other fill value(s).

        Note: local filters not influenced by neighboring cells do not
        need no data to be filled, as no data values are masked anyway.
        This is only important for pixel in the neighborhood of NaN values,
        if the neighborhood is considered.
        """
        array[array == nodata] = 0


    class Renamer(QgsProcessingLayerPostProcessorInterface):
        """
        To rename output layer name in the postprocessing step.
        """
        def __init__(self, layer_name):
            self.name = layer_name
            super().__init__()
            
        def postProcessLayer(self, layer, context, feedback):
            layer.setName(self.name)

    def name(self):
        """
        Returns the algorithm name, used for identifying the algorithm. This
        string should be fixed for the algorithm, and must not be localised.
        The name should be unique within each provider. Names should contain
        lowercase alphanumeric characters only and no spaces or other
        formatting characters.
        """
        return self._name

    def displayName(self):
        """
        Returns the translated algorithm name, which should be used for any
        user-visible display of the algorithm name.
        """

        # string must be passed through tr(), even if the string was already marked with tr()
        return tr(self._displayname)

    def group(self):
        """
        Returns the name of the group this algorithm belongs to. This string
        should be localised.
        """
        if self._groupid == "":
            return ""
        s = groups.get(self._groupid)
        if not s:
            # If group ID is not in dictionary group, return error message for debugging
            return "Displayname of group must be set in groups dictionary"
        # s must be passed through tr(), even if the string was already marked with tr()
        return tr(s)


    def groupId(self):
        """
        Returns the unique ID of the group this algorithm belongs to. This
        string should be fixed for the algorithm, and must not be localised.
        The group id should be unique within each provider. Group id should
        contain lowercase alphanumeric characters only and no spaces or other
        formatting characters.
        """
        return self._groupid
    
    def shortHelpString(self):
        """
        Returns the help string that is shown on the right side of the 
        user interface.
        """
        docstring = self.__doc__
        return convert_docstring_to_html(docstring)




class SciPyAlgorithmWithMode(SciPyAlgorithm):
    """
    Base class with added mode and cval; used by laplace etc.
    """

    MODE = 'MODE'
    CVAL = 'CVAL'

    def initAlgorithm(self, config):
        # Call the super function first
        # (otherwise input is not the first parameter in the GUI)
        super().initAlgorithm(config)

        self.addParameter(QgsProcessingParameterEnum(
            self.MODE,
            tr('Border Mode'),
            self.mode_labels,
            defaultValue=0)) 
        
        self.addParameter(QgsProcessingParameterNumber(
            self.CVAL,
            tr('Constant value past edges for border mode "constant"'),
            QgsProcessingParameterNumber.Type.Double,
            defaultValue=0, 
            optional=True, 
            minValue=0, 
            # maxValue=100
            ))    

        self.margin = 1  
    
    def get_parameters(self, parameters, context):
        kwargs = super().get_parameters(parameters, context)

        mode = self.parameterAsInt(parameters, self.MODE, context) 
        kwargs['mode'] = self.modes[mode]

        if kwargs['mode'] == 'wrap':
            self.wrapping = True

        cval = self.parameterAsDouble(parameters, self.CVAL, context)
        if cval:
            kwargs['cval'] = cval

        return kwargs
    

class SciPyAlgorithmWithModeAxis(SciPyAlgorithmWithMode):
    """
    Base class with added mode and cval and axis; used by sobel etc.
    """

    AXIS = 'AXIS'
    axis_modes = ['Horizontal edges', 'Vertical edges', 'Band axis edges', 'Magnitude']

    def insert_parameters(self, config):
        
        self.addParameter(QgsProcessingParameterEnum(
            self.AXIS,
            tr('Axis'),
            self.axis_modes,
            defaultValue=0)) 
        
        super().insert_parameters(config)
           
    def get_parameters(self, parameters, context):
        """Axis parameter must be set in inheriting class to implement magnitude"""
        kwargs = super().get_parameters(parameters, context)

        self.axis_mode = self.parameterAsInt(parameters, self.AXIS, context) 

        self.axis = -1

        if self.axis_mode == 0:
            self.axis = -2
        if self.axis_mode == 1:
            self.axis = -1
        if self.axis_mode == 2 and self._dimension == Dimensions.threeD:
            self.axis = -3

        return kwargs
    
    def checkParameterValues(self, parameters, context):
        dim_option = self.parameterAsInt(parameters, self.DIMENSION, context)
        if dim_option == 0: # 2D
            axis_mode = self.parameterAsInt(parameters, self.AXIS, context)
            if axis_mode == 2: # Band (not in 2D)
                return (False, tr("Band axis not possible in 2D case"))
            
        return super().checkParameterValues(parameters, context)
    


class SciPyStatisticalAlgorithm(SciPyAlgorithmWithMode):
    """
    Base class for median, minimum etc.
    """
    SIZE = 'SIZE'
    SIZES = 'SIZES'
    FOOTPRINT = 'FOOTPRINT'
    ORIGIN = 'ORIGIN'

    def initAlgorithm(self, config):
        super().initAlgorithm(config)

        size_param = QgsProcessingParameterNumber(
            self.SIZE,
            tr('Size of flat structuring element (either size or footprint must be given, with footprint, size is ignored)'),
            QgsProcessingParameterNumber.Type.Integer,
            defaultValue=1, 
            optional=True, 
            minValue=1, 
            maxValue=20, # Large sizes are really slow
            )
        
        size_param.setFlags(size_param.flags() | QgsProcessingParameterDefinition.Flag.FlagHidden)

        self.addParameter(size_param)


        sizes_param = QgsProcessingParameterString(
            self.SIZES,
            tr('Size'),
            defaultValue="", 
            optional=True, 
            )
        
        sizes_param.setMetadata({
            'widget_wrapper': {
                'class': SizesWidgetWrapper
            }
        })

        self.addParameter(sizes_param)   

        footprint_param = SciPyParameterStructure(
            self.FOOTPRINT,
            tr('Footprint array'),
            defaultValue="",
            examples=footprintexamples,
            multiLine=True,
            optional=True,
            to_int=True,
            )
                
        footprint_param.setMetadata({
            'widget_wrapper': {
                'class': StructureWidgetWrapper
            }
        })

        self.addParameter(footprint_param)


        origin_param = SciPyParameterOrigin(
            self.ORIGIN,
            tr('Origin'),
            defaultValue="0",
            optional=False,
            watch="FOOTPRINT"
            )
        
        origin_param.setFlags(origin_param.flags() | QgsProcessingParameterDefinition.Flag.FlagAdvanced)


        origin_param.setMetadata({
            'widget_wrapper': {
                'class': OriginWidgetWrapper
            }
        })

        self.addParameter(origin_param)



    def checkParameterValues(self, parameters, context): 
        footprint = self.parameterAsString(parameters, self.FOOTPRINT, context)

        dims = self.getDimsForCheck(parameters, context)


        ok, s, shape = check_structure(footprint, dims)
        if not ok:
            s = 'Footprint: ' + s
            return (ok, tr(s))

        sizes = self.parameterAsString(parameters, self.SIZES, context)
        sizes = str_to_int_or_list(sizes)
        if isinstance(sizes, list):
            if len(sizes) != dims:
                return (False, tr("Sizes does not match number of dimensions"))

        origin = self.parameterAsString(parameters, self.ORIGIN, context)
        origin = str_to_int_or_list(origin)

        if isinstance(origin, list):          
            if len(origin) != dims:
                return (False, tr("Origin does not match number of dimensions"))
            
            for i in range(dims):
                if shape[i] != 0 and not (-(shape[i] // 2) <= origin[i] <= (shape[i]-1) // 2):
                    return (False, tr("Origin out of bounds of structure"))


        # Extra check for rank_filter: rank must be < as footprint size
        # It is easier to do it here as we already have the footprint checked
            
        from .algs.scipy_statistical_algorithms import SciPyRankAlgorithm 

        if isinstance(self, SciPyRankAlgorithm):
            rank = self.parameterAsInt(parameters, self.RANK, context)
            if footprint:
                footprint = str_to_array(footprint, dims=dims)
                footprintsize = footprint.size
            else:
                size = self.parameterAsInt(parameters, self.SIZE, context)
                footprintsize = np.power(size, dims)
            if rank >= footprintsize:
                return (False, tr('Rank must be smaller than the size of the footprint'))
            
        return super().checkParameterValues(parameters, context)
    

    def get_parameters(self, parameters, context):
        kwargs = super().get_parameters(parameters, context)

        sizes = self.parameterAsString(parameters, self.SIZES, context)

        if sizes:
            size = str_to_int_or_list(sizes)
            sizelist = size # used to calculate margin
        else:
            size = self.parameterAsInt(parameters, self.SIZE, context)
            sizelist = [size]

        if not size:
            # Just in case it is called from python and neither size or sizes or footprint is set
            size = 3
            sizelist = [3]
        kwargs['size'] = size


        footprint = self.parameterAsString(parameters, self.FOOTPRINT, context)
        if footprint:
            footprint = str_to_array(footprint, self._ndim) #converting footprint from str to array
            kwargs['footprint'] = footprint
            sizelist.extend(footprint.shape) #list is updated with the footprint shape


        origin = self.parameterAsString(parameters, self.ORIGIN, context)
        kwargs['origin'] = str_to_int_or_list(origin)

        cval = self.parameterAsDouble(parameters, self.CVAL, context)
        if cval:
            kwargs['cval'] = cval

        # Margin for moving window
        self.margin = int(np.ceil(max(sizelist) / 2) + np.abs(kwargs['origin']).max())

        return kwargs
    

