# -*- coding: utf-8 -*-
"""
/***************************************************************************
 SaveecobotLoader
                                 A QGIS plugin
 This plugins loads AQI data from Safeecobot
 Generated by Plugin Builder: http://g-sherman.github.io/Qgis-Plugin-Builder/
                              -------------------
        begin                : 2022-03-30
        git sha              : $Format:%H$
        copyright            : (C) 2022 by Bundesamt für Strahlenschutz
        email                : mlechner@bfs.de
 ***************************************************************************/

/***************************************************************************
 *                                                                         *
 *   This program is free software; you can redistribute it and/or modify  *
 *   it under the terms of the GNU General Public License as published by  *
 *   the Free Software Foundation; either version 2 of the License, or     *
 *   (at your option) any later version.                                   *
 *                                                                         *
 ***************************************************************************/
"""

from qgis.PyQt.QtCore import QSettings, QTranslator, QCoreApplication, QVariant, QDateTime, Qt, QUrlQuery, QUrl
from qgis.PyQt.QtNetwork import QNetworkRequest
from qgis.PyQt.QtGui import QIcon
from qgis.PyQt.QtWidgets import QAction, QProgressDialog
from qgis.core import Qgis, QgsVectorLayer, QgsProject, QgsFeature, QgsPointXY, QgsGeometry, QgsField, QgsCoordinateReferenceSystem, QgsNetworkAccessManager, QgsNetworkReplyContent
from qgis.utils import iface

# Initialize Qt resources from file resources.py
from .resources import *
# Import the code for the dialog
from .saveecobot_loader_dialog import SaveecobotLoaderDialog
import os.path
import json
from datetime import datetime


class SaveecobotLoader:
    """QGIS Plugin Implementation."""

    def __init__(self, iface):
        """Constructor.

        :param iface: An interface instance that will be passed to this class
            which provides the hook by which you can manipulate the QGIS
            application at run time.
        :type iface: QgsInterface
        """
        # Save reference to the QGIS interface
        self.iface = iface
        # initialize plugin directory
        self.plugin_dir = os.path.dirname(__file__)
        # initialize locale
        locale = QSettings().value('locale/userLocale')[0:2]
        locale_path = os.path.join(
            self.plugin_dir,
            'i18n',
            'SaveecobotLoader_{}.qm'.format(locale))

        if os.path.exists(locale_path):
            self.translator = QTranslator()
            self.translator.load(locale_path)
            QCoreApplication.installTranslator(self.translator)

        # Declare instance attributes
        self.actions = []
        self.menu = self.tr(u'&Saveecobot Loader')

        # Check if plugin was started the first time in current QGIS session
        # Must be set in initGui() to survive plugin reloads
        self.first_start = None

    # noinspection PyMethodMayBeStatic
    def tr(self, message):
        """Get the translation for a string using Qt translation API.

        We implement this ourselves since we do not inherit QObject.

        :param message: String for translation.
        :type message: str, QString

        :returns: Translated version of message.
        :rtype: QString
        """
        # noinspection PyTypeChecker,PyArgumentList,PyCallByClass
        return QCoreApplication.translate('SaveecobotLoader', message)


    def add_action(
        self,
        icon_path,
        text,
        callback,
        enabled_flag=True,
        add_to_menu=True,
        add_to_toolbar=True,
        status_tip=None,
        whats_this=None,
        parent=None):
        """Add a toolbar icon to the toolbar.

        :param icon_path: Path to the icon for this action. Can be a resource
            path (e.g. ':/plugins/foo/bar.png') or a normal file system path.
        :type icon_path: str

        :param text: Text that should be shown in menu items for this action.
        :type text: str

        :param callback: Function to be called when the action is triggered.
        :type callback: function

        :param enabled_flag: A flag indicating if the action should be enabled
            by default. Defaults to True.
        :type enabled_flag: bool

        :param add_to_menu: Flag indicating whether the action should also
            be added to the menu. Defaults to True.
        :type add_to_menu: bool

        :param add_to_toolbar: Flag indicating whether the action should also
            be added to the toolbar. Defaults to True.
        :type add_to_toolbar: bool

        :param status_tip: Optional text to show in a popup when mouse pointer
            hovers over the action.
        :type status_tip: str

        :param parent: Parent widget for the new action. Defaults None.
        :type parent: QWidget

        :param whats_this: Optional text to show in the status bar when the
            mouse pointer hovers over the action.

        :returns: The action that was created. Note that the action is also
            added to self.actions list.
        :rtype: QAction
        """

        icon = QIcon(icon_path)
        action = QAction(icon, text, parent)
        action.triggered.connect(callback)
        action.setEnabled(enabled_flag)

        if status_tip is not None:
            action.setStatusTip(status_tip)

        if whats_this is not None:
            action.setWhatsThis(whats_this)

        if add_to_toolbar:
            # Adds plugin icon to Plugins toolbar
            self.iface.addToolBarIcon(action)

        if add_to_menu:
            self.iface.addPluginToVectorMenu(
                self.menu,
                action)

        self.actions.append(action)

        return action

    def initGui(self):
        """Create the menu entries and toolbar icons inside the QGIS GUI."""

        icon_path = ':/plugins/saveecobot_loader/icon.png'
        self.add_action(
            icon_path,
            text=self.tr(u'Saveecobot Loader'),
            callback=self.run,
            parent=self.iface.mainWindow())

        # will be set False in run()
        self.first_start = True


    def unload(self):
        """Removes the plugin menu item and icon from QGIS GUI."""
        for action in self.actions:
            self.iface.removePluginVectorMenu(
                self.tr(u'&Saveecobot Loader'),
                action)
            self.iface.removeToolBarIcon(action)


    def run(self):
        """Run method that performs all the real work"""

        # Create the dialog with elements (after translation) and keep reference
        # Only create GUI ONCE in callback, so that it will only load when the plugin is started
        if self.first_start == True:
            self.first_start = False
            self.dlg = SaveecobotLoaderDialog()
            self.dlg.sebDataUrlLineEdit.setText("https://www.saveecobot.com/storage/maps_data.js")
            self.dlg.sebMarkerDataUrlLineEdit.setText("https://www.saveecobot.com/en/maps/marker.json")

        # set default values for dlg
        self.dlg.dateTimeEdit.setDateTime(datetime.utcnow())
        self.dlg.lineEditLayerName.setText(datetime.strftime(datetime.utcnow(), '%Y%m%d-%H%M_saveecobot'))
        # show the dialog
        self.dlg.show()
        # Run the dialog event loop
        result = self.dlg.exec_()
        # See if OK was pressed
        if result:
            requesttime = self.dlg.dateTimeEdit.dateTime()
            sebtimestring = requesttime.toString('yyyy-MM-ddThhmm:ss')
            markertimestring = requesttime.toString('yyyy-MM-ddThh-mm:ss')
            sebkey = 'gamma'
            
            # Set up the first GET Request to SaveEcoBot
            sebquery = QUrlQuery()
            sebquery.addQueryItem('date', sebtimestring)
            seburl = QUrl(self.dlg.sebDataUrlLineEdit.text())
            seburl.setQuery(sebquery)
            self.prog = QProgressDialog('Requesting data. This may take a while. Please be patient ...', 'Cancel', 0, 100)
            self.prog.setWindowTitle('SaveEcoBot data loader. Requesting locations.')
            self.prog.setWindowModality(Qt.WindowModal)
            self.prog.setMinimumWidth(500)
            self.prog.show()

            manager = QgsNetworkAccessManager()

            sebrequest = QNetworkRequest(seburl)
            response: QgsNetworkReplyContent = manager.blockingGet(sebrequest)
            status_code = response.attribute(QNetworkRequest.HttpStatusCodeAttribute)
            if status_code == 200:
                # Get the content of the response and process it
                sebdata = json.loads(bytes(response.content()))
            else:
                self.iface.messageBar().pushMessage("SaveEcoBot loader error", "Couldn't load data from " + seburl, level=Qgis.Critical)

            # create layer
            vl_name = self.dlg.lineEditLayerName.text() if (self.dlg.lineEditLayerName.text() and len(self.dlg.lineEditLayerName.text()) > 0) else "temporary_saveecobot"
            vl = QgsVectorLayer("Point", vl_name, "memory")
            vl.setCrs(QgsCoordinateReferenceSystem('EPSG:4326'))

            pr = vl.dataProvider()
            # add fields default fields
            pr.addAttributes([QgsField("id", QVariant.Int),
                    QgsField("lon",  QVariant.Double),
                    QgsField("lat", QVariant.Double)])
            # add all other fields i a generic way
            keylist = []
            if 'devices' in sebdata.keys() and isinstance(sebdata['devices'], list):
                for sebdatarow in sebdata['devices']:
                    for key in sebdatarow.keys():
                        if (key not in keylist):
                            keylist.append(key)
                            try:
                                float(sebdatarow[key])
                                pr.addAttributes([QgsField(key, QVariant.Double)])
                            except ValueError:
                                pr.addAttributes([QgsField(key, QVariant.String)])
            else:
                self.iface.messageBar().pushMessage("SaveEcoBot loader error", "No devices list found in data from " + seburl, level=Qgis.Critical)
                exit
                    
            vl.updateFields() # tell the vector layer to fetch changes from the provider
            # add detailfields

            if 'devices' in sebdata.keys() and isinstance(sebdata['devices'], list):
                for sebdatarow in sebdata['devices']:
                    if sebkey in sebdatarow:
                        # add a feature
                        feat = QgsFeature()
                        feat.setFields(vl.fields())
                        feat.setGeometry(QgsGeometry.fromPointXY(QgsPointXY(float(sebdatarow["n"]),float(sebdatarow["a"]))))
                        featattributes = []
                        for key in feat.fields().names():
                            if key == "id":
                                featattributes.append(int(sebdatarow["i"]))
                            if key == "lon":
                                featattributes.append(float(sebdatarow["a"]))
                            if key == "lat":
                                featattributes.append(float(sebdatarow["n"]))
                        for key in feat.fields().names():
                            if key not in ("id", "device_id", "lon", "lat"):
                                if key in sebdatarow:
                                    if vl.fields().field(key).type() == QVariant.Int:
                                        try:
                                            featattributes.append(int(sebdatarow[key]))
                                        except:
                                            featattributes.append(None)
                                            self.iface.messageBar().pushMessage("Warning", "wrong data type at id " + str(featattributes[0]) + " for attribute " + key + ". value " + str(sebdatarow[key]) + " not imported.", level=Qgis.Warning)
                                    elif vl.fields().field(key).type() == QVariant.Double:
                                        try:
                                            featattributes.append(float(sebdatarow[key]))
                                        except:
                                            featattributes.append(None)
                                            self.iface.messageBar().pushMessage("Warning", "wrong data type at id " + str(featattributes[0]) + " for attribute " + key + ". value " + str(sebdatarow[key]) + " not imported.", level=Qgis.Warning)
                                    elif vl.fields().field(key).type() == QVariant.String:
                                        try:
                                            featattributes.append(str(sebdatarow[key]))
                                        except:
                                            featattributes.append(None)
                                            self.iface.messageBar().pushMessage("Warning", "wrong data type at id " + str(featattributes[0]) + " for attribute " + key + ". value " + str(sebdatarow[key]) + " not imported.", level=Qgis.Warning)
                                    else:
                                        featattributes.append(None)
                                else:
                                    featattributes.append(None)
                        if len(featattributes) == len(vl.fields()):
                            feat.setAttributes(featattributes)
                            pr.addFeatures([feat])
                        else:
                            self.iface.messageBar().pushMessage("Error", "featattributes len for id " + str(featattributes[0]) + " is " + str(len(featattributes)), level=Qgis.Critical)

            # update layer's extent when new features have been added
            # because change of extent in provider is not propagated to the layer
            vl.updateExtents()
            QgsProject.instance().addMapLayer(vl)
            # add last_updated_at / gamma to layer abstract
            if 'last_updated_at' in sebdata.keys() and 'gamma' in sebdata['last_updated_at']:
                vlm = vl.metadata()
                vlm.setAbstract('gamma latest updated (Ukraine local time): ' + sebdata['last_updated_at']['gamma'])
                vl.setMetadata(vlm)
            # refresh
            self.iface.mapCanvas().refresh()

            pr.addAttributes([QgsField("device_id", QVariant.String),
                    QgsField("latest", QVariant.DateTime),
                    QgsField("history", QVariant.String),
                    QgsField("history_hours", QVariant.Int),
                    QgsField("content", QVariant.String)])
            vl.updateFields() # tell the vector layer to fetch changes from the provider
            vl.startEditing()
            markerquery = QUrlQuery()
            markerquery.addQueryItem('marker_type', 'device')
            markerquery.addQueryItem('pollutant', str(sebkey))
            markerquery.addQueryItem('is_wide', str(1))
            markerquery.addQueryItem('is_iframe', str(0))
            markerquery.addQueryItem('is_widget', str(0))
            markerquery.addQueryItem('rand', markertimestring)
            markerurl = QUrl(self.dlg.sebMarkerDataUrlLineEdit.text())

            # load detail data for features
            count = vl.featureCount()
            self.prog.setWindowTitle('SaveEcoBot details loader. Requesting ' + str(count) + ' details.')

            for current, feature in enumerate(vl.getFeatures()):
                markerdata = json.loads('{}')
                sfid = str(feature.attribute("id"))
                percent = current / float(count) * 100
                self.prog.setValue(int(percent))
                self.prog.setLabelText('Requesting details for device ID ' + sfid + '. Please be patient.')
                markerquery.addQueryItem('marker_id', sfid)
                markerurl.setQuery(markerquery)
                markerrequest = QNetworkRequest(markerurl)
                response: QgsNetworkReplyContent = manager.blockingGet(markerrequest)
                status_code = response.attribute(QNetworkRequest.HttpStatusCodeAttribute)
                if status_code == 200:
                    # Get the content of the response and process it
                    markerdata = json.loads(bytes(response.content()))
                else:
                    self.iface.messageBar().pushMessage("SaveEcoBot loader error", "Could not load details for " + sfid, level=Qgis.Critical)
                markerquery.removeQueryItem('marker_id')
                if markerdata != json.loads('{}'):
                    feature.setAttribute("device_id", str(markerdata["marker_data"]["id"]))
                    if (len(markerdata["history"]) > 0):
                        feature.setAttribute("latest", QDateTime.fromString(sorted(markerdata["history"].keys()).pop(), "yyyy-MM-dd hh:mm:ss"))
                        feature.setAttribute("history", str(markerdata["history"]))
                    feature.setAttribute("history_hours", int(markerdata["history_hours"]))
                    feature.setAttribute("content", str(markerdata["content"]))
                    vl.updateFeature(feature)
                percent = current / float(count) * 100
                self.prog.setValue(int(percent))
                self.prog.setLabelText('Requesting details for device ID ' + sfid + '. Please be patient.')
                if self.prog.wasCanceled():
                    break
            vl.commitChanges()
            # activate show feature count
            ltroot = QgsProject.instance().layerTreeRoot()
            ltlayer = ltroot.findLayer(vl.id())
            ltlayer.setCustomProperty("showFeatureCount", True)

            self.prog.close()
            self.iface.mapCanvas().refresh()
