# -*- coding: utf-8 -*-
from qgis.core import (
    QgsProcessing, QgsProcessingAlgorithm,
    QgsProcessingParameterRasterLayer, QgsProcessingParameterVectorLayer,
    QgsProcessingParameterFile, QgsProcessingParameterField,
    QgsProcessingParameterString, QgsProcessingParameterNumber,
    QgsProcessingParameterBand, QgsProcessingParameterFolderDestination,
    QgsProcessingParameterBoolean, QgsRasterLayer, QgsProject, QgsProcessingUtils
)
import os, time, numpy as np, pandas as pd, processing, rasterio, shutil
try:
    from scipy.ndimage import median_filter, binary_opening, binary_closing
    scipy_installed = True
except ImportError:
    scipy_installed = False

class MasterWorkflowAlgorithm(QgsProcessingAlgorithm):
    INPUT_RASTER = 'INPUT_RASTER'; INPUT_TRAINING_VEC = 'INPUT_TRAINING_VEC'; INPUT_TESTING_VEC = 'INPUT_TESTING_VEC'
    DEPTH_FIELD_TRAINING = 'DEPTH_FIELD_TRAINING'; DEPTH_FIELD_TESTING = 'DEPTH_FIELD_TESTING'; N_ITERATIONS = 'N_ITERATIONS'
    OUTPUT_FOLDER = 'OUTPUT_FOLDER'; APPLY_WATER_MASK = 'APPLY_WATER_MASK'; GREEN_BAND = 'GREEN_BAND'; NIR_BAND = 'NIR_BAND'; SWIR_BAND = 'SWIR_BAND'
    WATER_INDEX_THRESHOLD = 'WATER_INDEX_THRESHOLD'; RUN_RF = 'RUN_RF'; RUN_GB = 'RUN_GB'; RUN_ET = 'RUN_ET'
    RUN_MLP = 'RUN_MLP'; RUN_SVR = 'RUN_SVR'; RUN_KNN = 'RUN_KNN'; RUN_DT = 'RUN_DT'; RUN_ELASTIC = 'RUN_ELASTIC'
    RUN_RIDGE = 'RUN_RIDGE'; RUN_LASSO = 'RUN_LASSO'; RUN_LINEAR = 'RUN_LINEAR'; RUN_BANDRATIO = 'RUN_BANDRATIO'
    MEDIAN_FILTER_SIZE = 'MEDIAN_FILTER_SIZE'; SAVE_BEST_MODEL = 'SAVE_BEST_MODEL'

    def initAlgorithm(self, config=None):
        self.addParameter(QgsProcessingParameterRasterLayer(self.INPUT_RASTER, 'Input Satellite Image'))
        self.addParameter(QgsProcessingParameterVectorLayer(self.INPUT_TRAINING_VEC, 'Training Points'))
        self.addParameter(QgsProcessingParameterField(self.DEPTH_FIELD_TRAINING, 'Depth Field (Training)', parentLayerParameterName=self.INPUT_TRAINING_VEC, type=QgsProcessingParameterField.Numeric))
        self.addParameter(QgsProcessingParameterVectorLayer(self.INPUT_TESTING_VEC, 'Unseen Testing Points'))
        self.addParameter(QgsProcessingParameterField(self.DEPTH_FIELD_TESTING, 'Depth Field (Testing)', parentLayerParameterName=self.INPUT_TESTING_VEC, type=QgsProcessingParameterField.Numeric))
        self.addParameter(QgsProcessingParameterBoolean(self.APPLY_WATER_MASK, 'Apply Water Masking', defaultValue=True))
        self.addParameter(QgsProcessingParameterBand(self.GREEN_BAND, 'Water Masking - Green Band', parentLayerParameterName=self.INPUT_RASTER, defaultValue=3, optional=True))
        self.addParameter(QgsProcessingParameterBand(self.NIR_BAND, 'Water Masking - NIR Band (for NDWI)', parentLayerParameterName=self.INPUT_RASTER, optional=True))
        self.addParameter(QgsProcessingParameterBand(self.SWIR_BAND, 'Water Masking - SWIR Band (for MNDWI)', parentLayerParameterName=self.INPUT_RASTER, optional=True))
        self.addParameter(QgsProcessingParameterNumber(self.WATER_INDEX_THRESHOLD, 'Water Index Threshold', type=QgsProcessingParameterNumber.Double, defaultValue=0.0, optional=True))
        self.addParameter(QgsProcessingParameterBoolean(self.RUN_RF, 'Run: RandomForest', defaultValue=True)); self.addParameter(QgsProcessingParameterBoolean(self.RUN_GB, 'Run: Gradient Boosting', defaultValue=True))
        self.addParameter(QgsProcessingParameterBoolean(self.RUN_ET, 'Run: Extra Trees', defaultValue=True)); self.addParameter(QgsProcessingParameterBoolean(self.RUN_MLP, 'Run: MLP', defaultValue=False))
        self.addParameter(QgsProcessingParameterBoolean(self.RUN_SVR, 'Run: SVR (slow)', defaultValue=False)); self.addParameter(QgsProcessingParameterBoolean(self.RUN_KNN, 'Run: KNN', defaultValue=True))
        self.addParameter(QgsProcessingParameterBoolean(self.RUN_DT, 'Run: Decision Tree', defaultValue=False)); self.addParameter(QgsProcessingParameterBoolean(self.RUN_ELASTIC, 'Run: ElasticNet', defaultValue=False))
        self.addParameter(QgsProcessingParameterBoolean(self.RUN_RIDGE, 'Run: Ridge', defaultValue=False)); self.addParameter(QgsProcessingParameterBoolean(self.RUN_LASSO, 'Run: Lasso', defaultValue=False))
        self.addParameter(QgsProcessingParameterBoolean(self.RUN_LINEAR, 'Run: Linear Regression', defaultValue=True))
        self.addParameter(QgsProcessingParameterBoolean(self.RUN_BANDRATIO, 'Run: Band Ratio (Auto-Search)', defaultValue=True))
        self.addParameter(QgsProcessingParameterNumber(self.N_ITERATIONS, 'Search Iterations (for complex models)', type=QgsProcessingParameterNumber.Integer, defaultValue=20, minValue=10))
        self.addParameter(QgsProcessingParameterNumber(self.MEDIAN_FILTER_SIZE, 'Apply Median Filter to ALL Results (0 to disable)', type=QgsProcessingParameterNumber.Integer, defaultValue=3, minValue=0))
        self.addParameter(QgsProcessingParameterBoolean(self.SAVE_BEST_MODEL, 'Save the Best Performing Model', defaultValue=True, optional=True))
        self.addParameter(QgsProcessingParameterFolderDestination(self.OUTPUT_FOLDER, 'Main Output Folder'))

    def name(self): return 'sdb_master_workflow'
    def displayName(self): return 'SDB Master Workflow (Full Comparison)'
    def group(self): return 'SDB Tools'
    def groupId(self): return 'sdb_tools'
    def createInstance(self): return MasterWorkflowAlgorithm()
    def shortHelpString(self): return "This master tool automates the entire SDB process by running and comparing a selection of algorithms."
            
    def processAlgorithm(self, parameters, context, feedback):
        input_raster = self.parameterAsRasterLayer(parameters, self.INPUT_RASTER, context).source()
        training_points = self.parameterAsVectorLayer(parameters, self.INPUT_TRAINING_VEC, context).source()
        testing_points = self.parameterAsVectorLayer(parameters, self.INPUT_TESTING_VEC, context).source()
        depth_field_training = self.parameterAsString(parameters, self.DEPTH_FIELD_TRAINING, context)
        depth_field_testing = self.parameterAsString(parameters, self.DEPTH_FIELD_TESTING, context)
        n_iterations = self.parameterAsInt(parameters, self.N_ITERATIONS, context)
        main_output_folder = self.parameterAsString(parameters, self.OUTPUT_FOLDER, context)
        apply_water_mask = self.parameterAsBool(parameters, self.APPLY_WATER_MASK, context)
        filter_size = self.parameterAsInt(parameters, self.MEDIAN_FILTER_SIZE, context)
        save_best_model = self.parameterAsBool(parameters, self.SAVE_BEST_MODEL, context)
        start_time = time.time()

        if (filter_size > 1 or apply_water_mask) and not scipy_installed:
            raise RuntimeError("Required library 'scipy' is not installed.")
        os.makedirs(main_output_folder, exist_ok=True)
        
        path_to_use_for_processing = input_raster
        feedback.setProgress(0)
        
        if apply_water_mask:
            # ... Water Masking Code ... (No changes here)
            pass

        feedback.setProgress(10)

        all_algorithms = {
            'sdb_autopredict_rf': (self.RUN_RF, 'rf', 'RandomForest'),
            'sdb_autopredict_gb': (self.RUN_GB, 'gb', 'Gradient Boosting'),
            'sdb_autopredict_extratrees': (self.RUN_ET, 'extratrees', 'Extra Trees'),
            'sdb_autopredict_mlp': (self.RUN_MLP, 'mlp', 'MLP'),
            'sdb_autopredict_svr': (self.RUN_SVR, 'svr', 'SVR'),
            'sdb_autopredict_knn': (self.RUN_KNN, 'knn', 'KNN'),
            'sdb_autopredict_decisiontree': (self.RUN_DT, 'dt', 'Decision Tree'),
            'sdb_autopredict_elasticnet': (self.RUN_ELASTIC, 'elasticnet', 'ElasticNet'),
            'sdb_autopredict_ridge': (self.RUN_RIDGE, 'ridge', 'Ridge'),
            'sdb_autopredict_lasso': (self.RUN_LASSO, 'lasso', 'Lasso'),
            'ml_linear': (self.RUN_LINEAR, 'linear', 'Linear Regression'),
            'sdb_autopredict_bandratio': (self.RUN_BANDRATIO, 'bandratio', 'Band Ratio')
        }
        
        selected_algs_to_run = []
        for alg_id, (param, short_name, display_name) in all_algorithms.items():
            if self.parameterAsBool(parameters, param, context):
                selected_algs_to_run.append((alg_id, short_name, display_name))

        final_summary = []; total_algs = len(selected_algs_to_run)
        if total_algs == 0: raise RuntimeError("No algorithms were selected to run.")
        progress_per_alg = 85.0 / total_algs

        for i, (alg_id, short_name, display_name) in enumerate(selected_algs_to_run):
            if feedback.isCanceled(): break
            feedback.pushInfo(f"\n--- Running Algorithm {i+1}/{total_algs}: {display_name.upper()} ---")
            feedback.setProgress(int(10 + (i * progress_per_alg)))
            alg_output_folder = os.path.join(main_output_folder, short_name)
            os.makedirs(alg_output_folder, exist_ok=True)
            
            params = {
                'INPUT_RASTER': path_to_use_for_processing,
                'INPUT_SAMPLES_VEC': training_points,
                'DEPTH_FIELD_VEC': depth_field_training,
                'OUTPUT_FOLDER': alg_output_folder,
                'SAVE_MODEL': True, # Always tell the child to save its model
                'N_ITERATIONS': n_iterations
            }
            
            predicted_raster_filename = f"autopredict_{short_name}_depth.tif"
            internal_report_filename = f"autopredict_{short_name}_report.txt"

            try:
                processing.run(f'sdb_tools:{alg_id}', params, context=context, feedback=feedback, is_child_algorithm=True)
            except Exception as e:
                feedback.pushWarning(f"Algorithm {display_name} failed. Skipping. Error: {e}")
                continue
            
            # ... Evaluation code ... (No changes here)
            path_for_evaluation = os.path.join(alg_output_folder, predicted_raster_filename)
            summary_entry = {'Algorithm': display_name.upper(), 'Alg_Short_Name': short_name, 'Unseen_Data_R2': np.nan, 'Unseen_Data_RMSE': np.nan, 'Raster_Path': path_for_evaluation}
            if os.path.exists(path_for_evaluation):
                eval_report_path = os.path.join(alg_output_folder, "FINAL_EVALUATION_REPORT.txt")
                eval_params = {'INPUT_PREDICTED_RASTER': path_for_evaluation, 'INPUT_UNSEEN_VEC': testing_points, 'DEPTH_FIELD_VEC': depth_field_testing, 'OUTPUT_FILE': eval_report_path}
                try:
                    processing.run('sdb_tools:sdb_evaluate_model', eval_params, context=context, feedback=feedback, is_child_algorithm=True)
                    if os.path.exists(eval_report_path):
                        with open(eval_report_path, 'r', encoding='utf-8') as f:
                            for line in f:
                                if "R-squared (R2):" in line: summary_entry['Unseen_Data_R2'] = float(line.split(':')[1].strip())
                                if "Root Mean Squared Error (RMSE):" in line: summary_entry['Unseen_Data_RMSE'] = float(line.split(':')[1].strip())
                except Exception as e:
                    feedback.pushWarning(f"Evaluation failed for {display_name}. Error: {e}")
            else:
                feedback.pushWarning(f"Predicted raster not found for {display_name}. Skipping evaluation.")
            
            final_summary.append(summary_entry)

        feedback.setProgress(95)
        feedback.pushInfo("\n--- WORKFLOW COMPLETE ---")
        summary_report_path = os.path.join(main_output_folder, "MASTER_SUMMARY_REPORT.txt")
        if not final_summary:
            feedback.pushWarning("No algorithms were successfully run.")
            return {}
        
        summary_df = pd.DataFrame(final_summary).dropna(subset=['Unseen_Data_R2', 'Unseen_Data_RMSE'])
        
        if not summary_df.empty:
            # ... Summary Report Generation ... (No changes here)
            r2_norm = (summary_df['Unseen_Data_R2'] - summary_df['Unseen_Data_R2'].min()) / (summary_df['Unseen_Data_R2'].max() - summary_df['Unseen_Data_R2'].min() + 1e-9)
            rmse_norm = 1 - ((summary_df['Unseen_Data_RMSE'] - summary_df['Unseen_Data_RMSE'].min()) / (summary_df['Unseen_Data_RMSE'].max() - summary_df['Unseen_Data_RMSE'].min() + 1e-9))
            summary_df['Final_Score'] = 0.7 * r2_norm + 0.3 * rmse_norm
            summary_df = summary_df.sort_values(by='Final_Score', ascending=False)
            with open(summary_report_path, 'w', encoding='utf-8') as f:
                 f.write(summary_df.to_string(index=False))
            feedback.pushInfo(f"All processes finished. Master summary report saved to: {summary_report_path}")

            best_result = summary_df.iloc[0]

            if save_best_model:
                best_alg_short_name = best_result['Alg_Short_Name']
                source_model_path = os.path.join(main_output_folder, best_alg_short_name, 'trained_model.joblib')
                if os.path.exists(source_model_path):
                    destination_model_path = os.path.join(main_output_folder, f"BEST_MODEL_{best_alg_short_name}.joblib")
                    try:
                        shutil.copy(source_model_path, destination_model_path)
                        # <<< تم التعديل: استبدال pushSuccess بـ pushInfo
                        feedback.pushInfo(f"Best performing model ('{best_result['Algorithm']}') was successfully saved to: {destination_model_path}")
                    except Exception as e:
                        feedback.pushWarning(f"Could not copy the best model. Error: {e}")
                else:
                    feedback.pushWarning(f"Could not find the saved model file for the best algorithm at: {source_model_path}.")
            
            best_raster_path = best_result['Raster_Path']
            if os.path.exists(best_raster_path):
                feedback.pushInfo(f"\nLoading the best result into QGIS: {best_result['Algorithm']} (Final Score = {best_result['Final_Score']:.4f})")
                QgsProject.instance().addMapLayer(QgsRasterLayer(best_raster_path, f"Best Result - {best_result['Algorithm']}"))
            else:
                feedback.pushWarning("Could not find the best result raster to load.")
        else:
            feedback.pushWarning("No models were successfully evaluated.")

        feedback.setProgress(100)
        return {}