from qgis.PyQt.QtCore import QSettings, QTranslator, QCoreApplication, QVariant,Qt
from qgis.PyQt.QtGui import QIcon, QFont
from qgis.PyQt.QtWidgets import QAction, QFileDialog, QMessageBox, QProgressDialog
from qgis.core import (
    QgsProcessingFeatureSourceDefinition,
    QgsVectorLayer,
    QgsRasterLayer,
    QgsWkbTypes,
    QgsProject,
    QgsRectangle,
    QgsVectorFileWriter,
    QgsField,
    QgsFeature,
    QgsGeometry,
    QgsProcessingFeedback,
    QgsSpatialIndex, 
    QgsPalLayerSettings, 
    QgsTextFormat, 
    QgsVectorLayerSimpleLabeling,
    QgsCoordinateReferenceSystem,
    QgsCoordinateTransform,QgsFeatureRequest,
    QgsMessageLog,
    Qgis

)
import processing
import os
import time
from datetime import datetime
import csv
import shutil
import json
from . import export_rename_tiles as tiles
from ..utils.errors import get_error_message, get_error_title
from ..resume.resume import Resume
from ..logger.logger import Logger

logger = Logger()
resume = Resume.get_instance()


def ensure_grid_ids(layer: QgsVectorLayer, start_value: int = None):
    """
    Ensure all features in the given vector layer have a valid 'grid_id' attribute.

    - Adds 'grid_id' field if it does not exist.
    - Assigns sequential IDs to features where 'grid_id' is missing or NULL.
    - If start_value is None, numbering starts from (max existing grid_id + 1) or 1 if none.

    :param layer: QgsVectorLayer to check/update.
    :param start_value: Optional integer to manually start numbering from.
    :return: Number of features updated.
    """
    if layer is None or not layer.isValid():
        logger.info("Invalid or missing layer passed to ensure_grid_ids()")
        return 0

    if layer.type() != QgsVectorLayer.VectorLayer:
        logger.info(f"Layer '{layer.name()}' is not a vector layer.")
        return 0

    provider = layer.dataProvider()

    # Ensure 'grid_id' field exists
    idx = layer.fields().indexOf('grid_id')
    if idx < 0:
        provider.addAttributes([QgsField('grid_id', QVariant.Int)])
        layer.updateFields()
        idx = layer.fields().indexOf('grid_id')
        logger.info(f"Field 'grid_id' created in layer '{layer.name()}'.")

    # Collect missing grid_ids and find maximum existing value
    missing_ids = []
    max_existing = None

    for feat in layer.getFeatures():
        gid_val = feat.attribute(idx)

        if gid_val in [None, ""]:
            missing_ids.append(feat.id())
            continue

        try:
            n = int(gid_val)
            if max_existing is None or n > max_existing:
                max_existing = n
        except Exception:
            # Non-integer or invalid values ignored
            pass

    # Nothing to update
    if not missing_ids:
        logger.info(f"Layer '{layer.name()}': all features already have 'grid_id'.")
        return 0

    # Determine starting number
    if start_value is not None:
        current_id = start_value
    else:
        current_id = (max_existing + 1) if max_existing is not None else 1

    # Prepare changes
    change_dict = {fid: {idx: i} for i, fid in enumerate(missing_ids, start=current_id)}

    # Apply changes
    try:
        layer.startEditing()
        provider.changeAttributeValues(change_dict)
        layer.commitChanges()
        logger.info(
            f"Layer '{layer.name()}': assigned grid_id to {len(missing_ids)} features (starting from {current_id}).",

        )
        return len(missing_ids)

    except Exception as e:
        try:
            layer.rollBack()
        except Exception:
            pass
        logger.error(f"Failed to update grid_id for layer '{layer.name()}': {e}")
        return 0


def ensure_feature_ids(layer: QgsVectorLayer, start_value: int = None):
    """
    Ensure all features in `layer` have a non-null 'feature_id' attribute.
    - If field missing: it will be created.
    - If some features are NULL/None/empty: only those will be assigned IDs.
    - If start_value is None, numbering continues from max existing feature_id + 1 (or 1 if none).
    Returns number of features updated.
    """
    if layer is None or not layer.isValid():
        logger.error("Invalid or missing layer passed to ensure_feature_ids()")
        return 0

    if layer.type() != QgsVectorLayer.VectorLayer:
        logger.error(f"Layer '{layer.name()}' is not a vector layer.")
        return 0

    provider = layer.dataProvider()
    # Ensure field exists
    index_of_layer = layer.fields().indexOf('layer')

    if index_of_layer >= 0:  # Check if layer field already exists
        # Get the index of the 'layer' field and remove it
        provider().deleteAttributes([index_of_layer])
        layer.updateFields()  # Update fields in the layer

    idx = layer.fields().indexOf('feature_id')
    if idx < 0:
        provider.addAttributes([QgsField('feature_id', QVariant.Int)])
        layer.updateFields()
        idx = layer.fields().indexOf('feature_id')

    # Find missing features and highest existing feature_id
    missing_feature_ids = []
    max_existing = None

    for feat in layer.getFeatures():
        # Use feature.isNull(index) to detect Qgs NULL
        if feat.attribute(idx) in [None]:
            missing_feature_ids.append(feat.id())
            continue

        val = feat.attribute(idx)
        # Treat None or empty string as missing
        if val is None or val == "":
            missing_feature_ids.append(feat.id())
            continue

        # Try parse int to detect maximum
        try:
            n = int(val)
            if max_existing is None or n > max_existing:
                max_existing = n
        except Exception:
            # Non-integer values are ignored for max computation
            pass

    # Nothing to do
    if not missing_feature_ids:
        logger.info(f"Layer '{layer.name()}': all features already have 'feature_id'.")
        return 0

    # Determine starting number
    if start_value is not None:
        current_id = start_value
    else:
        current_id = (max_existing + 1) if max_existing is not None else 1

    # Prepare bulk update dict: {featureId: {fieldIndex: value}}
    change_dict = {}
    for fid in missing_feature_ids:
        change_dict[fid] = {idx: current_id}
        current_id += 1

    # Apply changes
    try:
        layer.startEditing()
        provider.changeAttributeValues(change_dict)
        layer.commitChanges()
        logger.info(f"Layer '{layer.name()}': updated {len(missing_feature_ids)} feature_id(s).")
        return len(missing_feature_ids)
    except Exception as e:
        # If commit fails, roll back editing session if possible
        try:
            layer.rollBack()
        except Exception:
            pass
        logger.error(f"Failed to update feature_id for layer '{layer.name()}': {e}")
        return 0


def clip_layers_to_grid(grid_layer, layers, output_base_dir, progress_signal):
    """Clip all layers by grid cells -> clip, merge and archive."""
    feedback = QgsProcessingFeedback()
    resume.layers_to_clip = get_layer_names(layers)
    resume.destination = output_base_dir
    resume.grid_layer = get_layer_name(grid_layer)



    # Define the output directory inside the selected directory
    output_base_dir = os.path.join(output_base_dir, "Grid Output")

    # If the directory exists, remove it
    if not os.path.exists(output_base_dir):
        try:
            os.makedirs(output_base_dir)
        except Exception as e:
            error_msg = get_error_message('DIRECTORY_CREATION_ERROR', error=str(e))
            QMessageBox.warning(None, get_error_title('FILE_ERROR'), error_msg)
            return



    csv_file_path = os.path.join(output_base_dir, "grid_data.csv")

    try:
        with open(csv_file_path, mode='w', newline='', encoding='utf-8') as csv_file:
            csv_writer = csv.writer(csv_file)
            csv_writer.writerow(["grid_name", "creation_date", "assigned_to_surveyor", "submission_date"])
    except Exception as e:
        error_msg = get_error_message('FILE_SAVE_ERROR', error=str(e))
        QMessageBox.warning(None, get_error_title('FILE_ERROR'), error_msg)
        return


    #Check for feature ids
    for layer in layers:
        if layer.type() == QgsVectorLayer.VectorLayer:
            ensure_feature_ids(layer)
    #Check for grid ids
    ensure_grid_ids(grid_layer)

    # Define the request to order features by their 'id' attribute
    grid_to_start_from = resume.last_grid_processed
    request = (QgsFeatureRequest()
        .setFilterExpression(f'"grid_id" > {grid_to_start_from}').setOrderBy(
        QgsFeatureRequest.OrderBy([QgsFeatureRequest.OrderByClause('grid_id', ascending=True)])
    ))
    grids = grid_layer.getFeatures(request)
    for i, grid in enumerate(grids) :
        logger.info(f'Processing from grid : {i}')
        grid_cell_id = grid["grid_id"]
        # Skip features that have already been processed

        current_step = i + grid_to_start_from

        layers_name = []
        grid_cell_geom = grid.geometry()

        if not grid_cell_geom or not grid_cell_geom.isGeosValid():
            error_msg = get_error_message('INVALID_GEOMETRY', grid_cell_id=grid_cell_id)
            QMessageBox.warning(None, get_error_title('GEOMETRY_ERROR'), error_msg)
            continue

        grid_dir = os.path.join(output_base_dir, f"grid_{grid_cell_id}")
        try:
            if not os.path.exists(grid_dir):
                os.makedirs(grid_dir)
        except Exception as e:
            error_msg = get_error_message('DIRECTORY_CREATION_ERROR', error=str(e))
            QMessageBox.warning(None, get_error_title('FILE_ERROR'), error_msg)
            continue

        temp_layer = QgsVectorLayer(
            "Polygon?crs={}".format(grid_layer.crs().authid()), 
            f"grid_cell_{grid_cell_id}", "memory"
        )
        temp_layer_data = temp_layer.dataProvider()
        temp_layer_data.addAttributes(grid_layer.fields())
        temp_layer.updateFields()

        temp_feature = QgsFeature()
        temp_feature.setGeometry(grid_cell_geom)
        temp_feature.setAttributes(grid.attributes())
        temp_layer_data.addFeatures([temp_feature])
        temp_layer.updateExtents()
        
        try:
            create_html_file(temp_layer,grid_dir, grid_layer.crs())
            create_kml_file(temp_layer,grid_dir, grid_layer.crs())
        except Exception as e:
            error_msg = get_error_message('PROCESSING_ERROR', error=str(e))
            logger.error(error_msg)
            QMessageBox.warning(None, get_error_title('PROCESSING_ERROR'), error_msg)

            continue
            
        clipped_layers = {
            "Point": [],
            "Line": [],
            "Polygon": []
        }

        for layer in layers:
            logger.info(f'Clipping &Exporting : Processing layer : {layer.name()}, type : {layer.type()}')
            if layer.type() == QgsVectorLayer.VectorLayer:  # Handle vector layers
                geometry_type = QgsWkbTypes.flatType(layer.wkbType())
                output_path = os.path.join(grid_dir, f"{layer.name()}.geojson")
                clip_params = {
                    'INPUT': layer,
                    'OVERLAY': temp_layer,
                    'OUTPUT': output_path
                }
                try:
                    processing.run("qgis:clip", clip_params, feedback=feedback)
                    if geometry_type == QgsWkbTypes.MultiPoint or geometry_type == QgsWkbTypes.Point:
                        clipped_layers["Point"].append(output_path)
                        layers_name.append(f"{{{layer.name()} : Point}}")
                    elif geometry_type == QgsWkbTypes.MultiLineString or geometry_type == QgsWkbTypes.LineString:
                        clipped_layers["Line"].append(output_path)
                        layers_name.append(f"{{{layer.name()} : Line}}")
                    elif geometry_type == QgsWkbTypes.MultiPolygon or geometry_type == QgsWkbTypes.Polygon:
                        clipped_layers["Polygon"].append(output_path)
                        layers_name.append(f"{{{layer.name()} : Polygon}}")
                except Exception as e:
                    error_msg = get_error_message('VECTOR_CLIP_ERROR', layer_name=layer.name(), grid_cell_id=grid_cell_id, error=str(e))
                    QMessageBox.warning(None, get_error_title('PROCESSING_ERROR'), error_msg)
        
            elif layer.type() == QgsRasterLayer.RasterLayer:  # Handle raster layers
                output_path = os.path.join(grid_dir, f"{layer.name()}_clipped.tif")
                tile_output_dir = os.path.join(grid_dir, "tiles")
                reprojected_raster = os.path.join(grid_dir, f"{layer.name()}_reproject.tif")

                clip_params = {
                    'INPUT': layer.source(),
                    'MASK': temp_layer,
                    'OUTPUT': output_path,
                    'NODATA': -9999  # Define nodata value if needed
                }
                try:
                    processing.run("gdal:cliprasterbymasklayer", clip_params, feedback=feedback)
                    # Check if the output file was actually created
                    if not os.path.exists(output_path):
                        continue
                    if not os.path.exists(tile_output_dir):
                        os.makedirs(tile_output_dir)
                    params = {
                        'INPUT': output_path,
                        'TARGET_CRS': 'EPSG:3857',
                        'RESOLUTION': 0.0001,
                        'OUTPUT': reprojected_raster
                    }
                    processing.run("gdal:warpreproject", params, feedback = feedback)
                    params = {
                        'INPUT': reprojected_raster,  # Input raster file path
                        'OUTPUT': tile_output_dir,  # Output tile directory
                        'ZOOM' : '16-22',
                        'TILE_FORMAT': 'png',  # Adjust format if needed
                        'RESAMPLING': 0,  # Default is nearest neighbor (adjust if needed)
                        'TMS_CONVENTION': True,  # Use TMS-compatible tiles (flipped Y-coordinate)
                        'PROFILE': 0,  # Mercator profile
                        'WEB_VIEWER': 'none',  # Generates OpenLayers web viewer files
                    }

                    processing.run("gdal:gdal2tiles", params, feedback=feedback)

                    tiles.rename_tiles(tile_output_dir)
                    remove_files([output_path, reprojected_raster])
                except Exception as e:
                    error_msg = get_error_message('RASTER_CLIP_ERROR', layer_name=layer.name(), grid_cell_id=grid_cell_id, error=str(e))
                    QMessageBox.warning(None, get_error_title('PROCESSING_ERROR'), error_msg)

        #Merging Geometries
        geometry_output_files = {
            "Point": os.path.join(grid_dir, "point.geojson"),
            "Line": os.path.join(grid_dir, "line.geojson"),
            "Polygon": os.path.join(grid_dir, "polygon.geojson")
        }
        for geometry_type, layer_paths in clipped_layers.items():
            if layer_paths :
                merge_clipped_layers(layer_paths, geometry_output_files[geometry_type], geometry_type, "EPSG:4326", grid_cell_id)
        for geometry_type, layer_paths in clipped_layers.items() :
            remove_files(layer_paths)


        try:
            with open(csv_file_path, mode='a', newline='', encoding='utf-8') as csv_file:
                csv_writer = csv.writer(csv_file)
                csv_writer.writerow([
                    f"grid_{grid_cell_id}",
                    datetime.now().strftime('%Y-%m-%d %H:%M:%S'),
                    "",  # Assigned to surveyor (empty)
                    ""  # Submission date (empty)
                ])
        except Exception as e:
            error_msg = get_error_message('FILE_SAVE_ERROR', error=str(e))
            logger.error(error_msg)
            QMessageBox.warning(None, get_error_title('FILE_ERROR'), error_msg)


        try:
            create_metadata(grid_name=f"grid_{grid_cell_id}", grid_layer=temp_layer, grid_dir=grid_dir,
                            layers_name=layers_name, crs=grid_layer.crs())
        except Exception as e:
            error_msg = get_error_message('PROCESSING_ERROR', error=str(e))

            QMessageBox.warning(None, get_error_title('PROCESSING_ERROR'), error_msg)

            
        #Archiving
        archive_name = os.path.join(grid_dir, f"grid_{grid_cell_id}")
        create_archive(grid_dir,archive_name)

        progress_signal.emit(current_step)
        del temp_layer
        resume.last_grid_processed = grid_cell_id

    resume.completed = True

def create_kml_file(layer, grid_dir, crs) :
    try:
        kml_file_path = os.path.join(grid_dir, f"location.kml")
        options = QgsVectorFileWriter.SaveVectorOptions()
        QgsVectorFileWriter.writeAsVectorFormat(
            layer,
            kml_file_path,
            "utf-8",
            crs,
            "KML"
        )
    except Exception as e:
        error_msg = get_error_message('FILE_SAVE_ERROR', error=str(e))
        logger.error(error_msg)
        QMessageBox.warning(None, get_error_title('FILE_ERROR'), error_msg)
        raise


def create_html_file(layer, grid_dir, crs):
    """
    Creates an HTML file with a script that redirects to Google Maps for the polygon feature.
    """
    try:
        html_file_path = os.path.join(grid_dir, f"location.html")
        target_crs = QgsCoordinateReferenceSystem("EPSG:4326")
        transform = QgsCoordinateTransform(crs, target_crs, QgsProject.instance())
        # Get the geometry of the first feature (assuming only one feature per layer)
        layer.startEditing()
        feature = next(layer.getFeatures(), None)
        layer.commitChanges()

        if not feature:
            error_msg = get_error_message('NO_FEATURE_FOUND')
            raise ValueError(error_msg)

        geometry = feature.geometry()

        if geometry.isEmpty():
            error_msg = get_error_message('EMPTY_GEOMETRY')
            raise ValueError(error_msg)

        # Extract the centroid of the polygon as latitude and longitude
        geometry.transform(transform)

        # Extract the centroid of the polygon as latitude and longitude
        centroid = geometry.centroid().asPoint()
        latitude, longitude = centroid.y(), centroid.x()

        # Write the HTML file
        html_content = f"""
            <!DOCTYPE html>
            <html lang="en">
            <head>
                <meta charset="UTF-8">
                <meta name="viewport" content="width=device-width, initial-scale=1.0">
                <title>Polygon Location</title>
                <script>
                    // Google Maps URL with latitude and longitude
                    const googleMapsUrl = "https://www.google.com/maps?q={latitude},{longitude}";
        
                    // Redirect to the Google Maps URL when the page loads
                    window.onload = () => {{
                        window.location.href = googleMapsUrl;
                    }};
                </script>
            </head>
            <body>
                <p>If you are not redirected automatically, click <a href="https://www.google.com/maps?q={latitude},{longitude}">here</a>.</p>
            </body>
            </html>
            """

        # Save the HTML file
        with open(html_file_path, 'w', encoding='utf-8') as html_file:
            html_file.write(html_content)
            
    except Exception as e:
        error_msg = get_error_message('FILE_SAVE_ERROR', error=str(e))
        QMessageBox.warning(None, get_error_title('FILE_ERROR'), error_msg)
        raise

def create_metadata(grid_name,grid_layer, grid_dir, layers_name, crs) :
    """
        Creates an JSON file that contains the information about clipped grid
        Extent of grid
        Layers clipped and
        Grid cell id.
        """
    try:
        json_file_path = os.path.join(grid_dir, "metadata.json")
        target_crs = QgsCoordinateReferenceSystem("EPSG:4326")
        transform = QgsCoordinateTransform(crs, target_crs, QgsProject.instance())

        # Get the geometry of the first feature (assuming only one feature per layer)
        grid_layer.startEditing()
        feature = next(grid_layer.getFeatures(), None)
        grid_layer.commitChanges()

        if not feature:
            error_msg = get_error_message('NO_FEATURE_FOUND')
            raise ValueError(error_msg)

        geometry = feature.geometry()
        geometry.transform(transform)
        bounding_box = geometry.boundingBox()
        metadata = {
            "north": bounding_box.yMaximum(),
            "south": bounding_box.yMinimum(),
            "east": bounding_box.xMaximum(),
            "west": bounding_box.xMinimum(),
            "layers" : layers_name,
            "grid" : grid_name
        }
        with open(json_file_path, 'w', encoding='utf-8') as json_file:
            json.dump(metadata, json_file, indent=4)
    except Exception as e:
        error_msg = get_error_message('FILE_SAVE_ERROR', error=str(e))
        QMessageBox.warning(None, get_error_title('FILE_ERROR'), error_msg)
        raise
        

def merge_clipped_layers (layers_path, merged_layer_path, geometry_type,crs, grid_cell_id) :
    feedback = QgsProcessingFeedback()
    valid_layers = []
    for path in layers_path:
        layer = QgsVectorLayer(path, "temp_layer", "ogr")
        if not layer.isValid():
            continue
        if layer.featureCount() > 0:
            valid_layers.append(path)

    # If no valid layers exist, exit
    if not valid_layers:
        return

    merge_params = {
            'LAYERS': valid_layers,
            'CRS': crs,
            'OUTPUT': merged_layer_path
    }
    try:
            processing.run("qgis:mergevectorlayers", merge_params, feedback = feedback)
    except Exception as e:
            error_msg = get_error_message('MERGE_LAYERS_ERROR', geometry_type=geometry_type, error=str(e))
            QMessageBox.warning(None, get_error_title('PROCESSING_ERROR'), error_msg)

def close_files (file_paths) :
    for file_path in file_paths :
        file = open(file_path, "r+")
        file.close()

def remove_files(file_paths) :
    for file_path in file_paths:
        retries = 5  # Number of retries to delete the file
        while retries > 0:
            try:
                os.remove(file_path)
                break
            except PermissionError:
                retries -= 1
                time.sleep(0.5)  # Wait for 500ms before retrying
            except Exception as e:
                error_msg = get_error_message('FILE_DELETION_ERROR', file_path=file_path, error=str(e))
                raise Exception(error_msg)


def check_geometries_and_extents(layers):
    """Check for invalid geometries and ensure layer extents overlap."""
    try:
        invalid_geometries = []
        all_extents = []

        for i, layer in enumerate(layers):
            if layer.type() == QgsVectorLayer.VectorLayer:
                for feature in layer.getFeatures():
                    if not feature.geometry().isGeosValid():
                        invalid_geometries.append((layer.name(), feature.id()))
                all_extents.append(layer.extent())

        combined_extent = QgsRectangle()
        for extent in all_extents:
            combined_extent.combineExtentWith(extent)

        if combined_extent.isEmpty():
            error_msg = get_error_message('NO_OVERLAPPING_EXTENTS')
            raise Exception(error_msg)

        if invalid_geometries:
            details = "\n".join([f"Layer: {layer}, Feature ID: {fid}" for layer, fid in invalid_geometries])
            error_msg = get_error_message('INVALID_GEOMETRIES_DETECTED', details=details)
            raise Exception(error_msg)

        return combined_extent
    except Exception as e:
        error_msg = get_error_message('VALIDATION_ERROR', details=str(e))
        QMessageBox.warning(None, get_error_title('VALIDATION_ERROR'), error_msg)
        raise

def create_archive (grid_dir, archive_name) :
    try :
        logger.info(f"Archiving for grid directory: {grid_dir}")
        files_to_compress = []
        tiles_dir = None
        for root, dirs, files in os.walk(grid_dir):
            for file in files:
                if file not in ["location.html", "location.kml"]:
                    files_to_compress.append(os.path.join(root, file))
            if "tiles" in dirs:
                tiles_dir = os.path.join(root, "tiles")
            # Create a temporary directory for archiving
        temp_archive_dir = os.path.join(grid_dir, f"temp_archive")
        if not os.path.exists(temp_archive_dir):
            os.makedirs(temp_archive_dir)

        for file_path in files_to_compress:
            relative_path = os.path.relpath(file_path, grid_dir)
            dest_path = os.path.join(temp_archive_dir, relative_path)
            os.makedirs(os.path.dirname(dest_path), exist_ok=True)
            shutil.copy(file_path, dest_path)

        # Compress files to .zip
        shutil.make_archive(archive_name, 'zip', temp_archive_dir)

        # Rename .zip to .amrut
        amrut_file = f"{archive_name}.amrut"
        os.rename(f"{archive_name}.zip", amrut_file)

        # Cleanup temporary archive directory
        shutil.rmtree(temp_archive_dir)

        # Remove original files (except location.html)
        remove_files(files_to_compress)

        # Remove tiles directory
        if tiles_dir and os.path.exists(tiles_dir):
            shutil.rmtree(tiles_dir)

    except Exception as e:
        error_msg = get_error_message('ARCHIVE_CREATION_ERROR', error=str(e))
        QMessageBox.warning(None, get_error_title('ARCHIVE_ERROR'), error_msg)

def merge_feature_collection(geojson_files, output_file) :
    polygon_path = geojson_files["Polygon"]
    line_path = geojson_files["Line"]
    point_path = geojson_files["Point"]

    polygon_data = read_features(polygon_path)
    line_data = read_features(line_path)
    point_data = read_features(point_path)

    all_features = []
    for data in (polygon_data, line_data, point_data):
        if "features" in data:
            all_features.extend(data["features"])

    merged_geojson = {
        "type": "FeatureCollection",
        "name": "merged",
        "crs": polygon_data.get("crs", None),
        "features": all_features
    }
    with open(output_file, "w", encoding='utf-8') as f:
        json.dump(merged_geojson, f, ensure_ascii=False, indent=2)

def read_features(path) :
    if not os.path.exists(path):
        return {"features": []}
    with open(path, 'r', encoding='utf-8') as f:
        geojson = json.load(f)
        return geojson


def get_layer_names(layers):
    """
    Extract layer names from a list of QGIS layers and return as a string array.

    Args:
        layers: List of QgsVectorLayer or QgsRasterLayer objects

    Returns:
        List[str]: Array of layer names
    """
    layer_names = []
    for layer in layers:
        if layer and hasattr(layer, 'name'):
            layer_names.append(layer.name())
    return layer_names
def get_layer_name(layer):
    """
    Extract layer names from a list of QGIS layers and return as a string array.

    Args:
        layer: QgsVectorLayer or QgsRasterLayer objects

    Returns:
        String: Array of layer names
    """
    if layer and hasattr(layer, 'name'):
        return layer.name()
    return None