import os
import datetime
from qgis.core import Qgis, QgsMessageLog
from .utils import get_qgis_project_root

class Logger:
    """
        Handles logging messages to 'sankalan.log' in the current QGIS project root directory.
        Each entry includes a timestamp and log level.
        """

    def __init__(self, plugin_name="Sankalan"):
        self.plugin_name = plugin_name
        self.log_file_path = os.path.join(get_qgis_project_root(), 'sankalan.log')

        # Ensure the log file exists
        if not os.path.exists(self.log_file_path):
            with open(self.log_file_path, 'w', encoding='utf-8') as f:
                f.write(f"{plugin_name} Log File\n")
                f.write("=" * 50 + "\n\n")

    def _write_log(self, level: str, message: str):
        timestamp = datetime.datetime.now().strftime("%Y-%m-%d %H:%M:%S")
        log_entry = f"[{timestamp}] [{level}] {message}\n"

        # Append to the log file
        with open(self.log_file_path, 'a', encoding='utf-8') as f:
            f.write(log_entry)

        # Also push to QGIS Message Log panel
        if level == "INFO":
            QgsMessageLog.logMessage(message, self.plugin_name, Qgis.Info)
        elif level == "WARNING":
            QgsMessageLog.logMessage(message, self.plugin_name, Qgis.Warning)
        elif level == "ERROR":
            QgsMessageLog.logMessage(message, self.plugin_name, Qgis.Critical)
        else:
            QgsMessageLog.logMessage(message, self.plugin_name, Qgis.Info)

    def info(self, message: str):
        self._write_log("INFO", message)

    def warning(self, message: str):
        self._write_log("WARNING", message)

    def error(self, message: str):
        self._write_log("ERROR", message)
