from qgis.core import (
    QgsProcessingFeatureSourceDefinition,
    QgsVectorLayer,
    QgsRasterLayer,
    QgsWkbTypes,
    QgsProject,
    QgsRectangle,
    QgsField,
    QgsFeature,
    QgsGeometry,
    QgsVectorFileWriter,
    QgsProcessingFeedback,
    QgsSpatialIndex, 
    QgsPalLayerSettings, 
    QgsTextFormat, 
    QgsVectorLayerSimpleLabeling,
    QgsVectorDataProvider,
    QgsRectangle,
    QgsCoordinateReferenceSystem,
    QgsCoordinateTransform,
    QgsVectorLayerUtils,
    QgsMessageLog,
    Qgis
)
from qgis.PyQt.QtGui import QIcon, QFont
from qgis.PyQt.QtCore import QSettings, QTranslator, QCoreApplication, QVariant,Qt
from qgis.PyQt.QtWidgets import QAction, QFileDialog, QMessageBox, QProgressDialog
from .import export_geometry as geometry
from ..utils.errors import get_error_message, get_error_title
from ..logger.logger import Logger
import os
import uuid

logger = Logger()

def create_grid_layer(bbox, grid_size, crs):
    """
    Create a grid layer covering the given bounding box
    
    Args:
        bbox (list): [xmin, ymin, xmax, ymax] coordinates
        grid_size (float): Size of each grid cell
        crs (str): Coordinate reference system ID (e.g., 'EPSG:4326')
    
    Returns:
        QgsVectorLayer: The created grid layer
    """
    xmin, ymin, xmax, ymax = bbox
    grid_layer = QgsVectorLayer(f"Polygon?crs={crs}", "Grid", "memory")
    
    if not grid_layer.isValid():
        logger.error(f"Layer not valid : {crs}")
        raise ValueError(f"Could not create grid layer with CRS: {crs}")
        
    provider = grid_layer.dataProvider()

    # Define the fields for the layer
    provider.addAttributes([QgsField("id", QVariant.Int)])
    grid_layer.updateFields()

    grid_cells = []
    cell_id = 0
    x = xmin
    while x < xmax:
        y = ymin
        while y < ymax:
            # Create the grid cell as a rectangle
            rect = QgsRectangle(x, y, x + grid_size, y + grid_size)
            feature = QgsFeature()
            feature.setGeometry(QgsGeometry.fromRect(rect))
            feature.setAttributes([cell_id])
            grid_cells.append(feature)
            y += grid_size
            cell_id += 1
        x += grid_size

    logger.info('Number of Grid Cells created: ' + str(cell_id +1),)
    # Add the features (grid cells) to the layer
    provider.addFeatures(grid_cells)
    grid_layer.updateExtents()
    #add_grid_labels(grid_layer)

    return grid_layer

def add_grid_labels(grid_layer):
    """Add labels to the grid layer, showing grid cell IDs."""
    try:
        if not isinstance(grid_layer, QgsVectorLayer):
            QMessageBox.warning(
                None,
                get_error_title('LAYER_ERROR'),
                get_error_message('LAYER_TYPE_ERROR')
            )
            return

        # Step 1: Initialize label settings
        label_settings = QgsPalLayerSettings()
        label_settings.fieldName = "id"  # Field to use for labeling (e.g., grid cell ID)
        label_settings.placement = QgsPalLayerSettings.OverPoint  # Label placement over the grid cells

        # Step 2: Configure text format (font, size, etc.)
        text_format = QgsTextFormat()
        text_format.setFont(QFont("Arial", 10))  # Use desired font and size
        text_format.setSize(10)  # Font size for labels
        label_settings.setFormat(text_format)

        # Step 3: Assign the labeling settings to the layer
        labeling = QgsVectorLayerSimpleLabeling(label_settings)  # Create a labeling object
        grid_layer.setLabeling(labeling)  # Assign labeling configuration to the layer
        grid_layer.setLabelsEnabled(True)  # Enable labels for the grid layer

        # Step 4: Refresh the layer to apply the changes
        grid_layer.triggerRepaint()
        
    except Exception as e:
        logger.error(f"Error adding labels to grid layer: {e}")
        QMessageBox.warning(
            None,
            get_error_title('LAYER_ERROR'),
            get_error_message('PROCESSING_ERROR', error=str(e))
        )

def create_grid_within_single_polygon(selectedLayers,polygon_layer, grid_size, crs = 'EPSG:32644'):
    """
    Create a grid covering only a single polygon geometry.

    :param polygon_layer: QgsVectorLayer containing one polygon geometry.
    :param grid_size: Size of the grid cells (e.g., 500 for 500m x 500m).
    :param crs: CRS for the output grid layer (default EPSG:32644).
    :return: QgsVectorLayer with the generated grid.
    """
    try:
        error = geometry.validate_layer(polygon_layer)
        if error:
            msg = get_error_message('INVALID_GEOMETRIES_DETECTED', details="\n".join([f"{error_msg}" for error_msg in error]))
            logger.error(f"Invalid : ".join([f"{error_msg}" for error_msg in error]))
            QMessageBox.warning(
                None,
                get_error_title('VALIDATION_ERROR'),
                msg
            )
            raise Exception(msg)
        
        if not polygon_layer.isValid():
            error_msg = get_error_message('INVALID_POLYGON_LAYER')
            logger.error(f"Layer is not valid : {error_msg}")
            QMessageBox.warning(
                None,
                get_error_title('LAYER_ERROR'),
                error_msg
            )
            raise ValueError(error_msg)

        # Extract the polygon geometry
        features = list(polygon_layer.getFeatures())
        if len(features) != 1:
            error_msg = get_error_message('SINGLE_POLYGON_REQUIRED')
            logger.error(f"Layer is not valid : {error_msg}")
            QMessageBox.warning(
                None,
                get_error_title('LAYER_ERROR'),
                error_msg
            )
            raise ValueError(error_msg)
            
        polygon_geom = features[0].geometry()

        if polygon_geom is None or polygon_geom.isEmpty():
            error_msg = get_error_message('EMPTY_POLYGON_GEOMETRY')
            logger.error(f"Layer is not valid : {error_msg}")
            QMessageBox.warning(
                None,
                get_error_title('GEOMETRY_ERROR'),
                error_msg
            )
            raise ValueError(error_msg)

        # Generate the grid cells and clip them to the polygon geometry
        combined_extent_of_selected_layers = geometry.getExtent(selectedLayers)
        extent_of_polygon_layer = geometry.getExtent([polygon_layer])
        logger.info('Grid Creation : AOI Layer BBOX: ' + str(extent_of_polygon_layer.toRectF().getCoords()))
        logger.info('Grid Creation : All Selected Layers BBOX: ' + str(combined_extent_of_selected_layers.toRectF().getCoords()))
        if not extent_of_polygon_layer.contains(combined_extent_of_selected_layers):
            error_msg = get_error_message('LAYER_EXTENT_ERROR')
            logger.error(f"Layer is not valid : {error_msg}")
            QMessageBox.warning(
                None,
                get_error_title('VALIDATION_ERROR'),
                error_msg
            )
            raise Exception(error_msg)

        grid_layer = create_grid_layer(extent_of_polygon_layer.toRectF().getCoords(),grid_size, 'EPSG:32644')

        polygon_crs = polygon_layer.crs().authid()
        grid_crs = grid_layer.crs().authid()
        logger.info(
            'AOI CRS ' + str(polygon_crs) + ' Grid CRS ' + str(grid_crs),)
        if polygon_crs != grid_crs:
            transform = QgsCoordinateTransform( polygon_layer.crs(), grid_layer.crs(), QgsProject.instance())
            polygon_geom.transform(transform)
        
        unique_id = str(uuid.uuid4())
        clipped_grid_layer = QgsVectorLayer("Polygon?crs={}".format('EPSG:32644'), unique_id, "memory")
        logger.info('Grid Creation : AOI Layer Geom: ' + str(polygon_geom.asWkt()))
        provider = clipped_grid_layer.dataProvider()
        provider.addAttributes(grid_layer.fields())
        clipped_grid_layer.updateFields()

        for grid_feature in grid_layer.getFeatures():
            grid_geom = grid_feature.geometry()
            if grid_geom.intersects(polygon_geom):
                clipped_geom = grid_geom.intersection(polygon_geom)
                if not clipped_geom.isEmpty():
                    clipped_feature = QgsFeature()
                    clipped_feature.setGeometry(clipped_geom)
                    clipped_feature.setAttributes(grid_feature.attributes())
                    provider.addFeature(clipped_feature)

        clipped_grid_layer.updateExtents()
        error = geometry.validate_layer(clipped_grid_layer)
       
        if not error:
            file_path = getFilePath(unique_id)
            save_file_to_disk(layer = clipped_grid_layer, file_path=file_path)
            saved_grid_layer =  QgsVectorLayer(file_path + ".gpkg", unique_id, "ogr")
            QgsProject.instance().addMapLayer(saved_grid_layer)
            return unique_id
        else:
            msg = get_error_message('GRID_VALIDATION_ERROR', details="\n".join([f"{error_msg}" for error_msg in error]))
            logger.error(f"Grid Creation Error: {msg}")
            QMessageBox.warning(
                None,
                get_error_title('VALIDATION_ERROR'),
                msg
            )
            raise Exception(msg)

    except Exception as e:
        error_msg = get_error_message('GRID_CREATION_ERROR', error=str(e))
        logger.error(f"Grid Creation Error: {error_msg}")
        QMessageBox.warning(
            None,
            get_error_title('GRID_ERROR'),
            error_msg
        )
        raise Exception(error_msg)

def getFilePath(file_name):
    try:
        project = QgsProject.instance()

        # Get the project file path (if it's saved)
        project_file_path = project.fileName()

        # Get the directory where the project file is located
        project_directory = os.path.dirname(project_file_path) if project_file_path else os.path.expanduser("~")
        
        # Ensure the directory exists
        if not os.path.exists(project_directory):
            os.makedirs(project_directory)

        return os.path.join(project_directory, file_name)
        
    except Exception as e:
        error_msg = get_error_message('DIRECTORY_CREATION_ERROR', error=str(e))
        logger.error(f"Grid Creation Error: {error_msg}")
        QMessageBox.warning(
            None,
            get_error_title('FILE_ERROR'),
            error_msg
        )
        raise Exception(error_msg)

def save_file_to_disk(file_path, layer):
    try:
        options = QgsVectorFileWriter.SaveVectorOptions()

        # Set the CRS transformation context (optional)
        error = QgsVectorFileWriter.writeAsVectorFormatV2(
            layer=layer,
            fileName = file_path,
            transformContext= QgsProject.instance().transformContext(),
            options =options,  
        )
        
        if error[0] != QgsVectorFileWriter.NoError:
            error_msg = get_error_message('FILE_SAVE_ERROR', error=str(error[1]))
            logger.error(f"Grid Creation Error: {error_msg}")
            QMessageBox.warning(
                None,
                get_error_title('FILE_ERROR'),
                error_msg
            )
            raise Exception(error_msg)
            
    except Exception as e:
        error_msg = get_error_message('FILE_SAVE_ERROR', error=str(e))
        logger.error(f"Grid Creation Error: {error_msg}")
        QMessageBox.warning(
            None,
            get_error_title('FILE_ERROR'),
            error_msg
        )
        raise Exception(error_msg)


