from qgis.core import QgsProject, QgsMessageLog, Qgis
from ..utils.errors import get_error_message
from ..logger.logger import Logger
import os
import json
import zipfile

logger = Logger()

def validate_project():
    """
    Validates if a QGIS project is open and valid
    Returns: True if project is valid, raises ValueError otherwise
    """
    project = QgsProject.instance()
    
    if not project:
        error_msg = get_error_message('NO_PROJECT_LOADED_ERROR')
        logger.error(error_msg)
        raise ValueError(error_msg)
    
    if not project.fileName():
        error_msg = get_error_message('PROJECT_NOT_SAVED_ERROR')
        logger.error(error_msg)
        raise ValueError(error_msg)
    
    if not project.homePath():
        error_msg = get_error_message('PROJECT_STATUS_CHECK_ERROR', error="Project home path not set")
        logger.error(error_msg)
        raise ValueError(error_msg)
    
    return True

def validate_amrut_file(file_path):
    """
    Validates structure and content of AMRUT file
    Returns: (is_valid: bool, error_message: str)
    """
    if not os.path.exists(file_path):
        error_msg = get_error_message('FILE_NOT_FOUND_ERROR')
        logger.error(error_msg)
        return False, error_msg
    
    try:
        with zipfile.ZipFile(file_path, 'r') as zip_ref:
            if 'metadata.json' not in zip_ref.namelist():
                error_msg = get_error_message('MISSING_METADATA_FILE')
                logger.error(error_msg)
                return False, error_msg
            
            try:
                metadata_content = zip_ref.read('metadata.json').decode('utf-8')
                metadata = json.loads(metadata_content)
                
                if not isinstance(metadata, dict):
                    error_msg = get_error_message('INVALID_METADATA_FORMAT')
                    logger.error(error_msg)
                    return False, error_msg
                
                if 'layers' not in metadata:
                    error_msg = get_error_message('INVALID_LAYERS_ARRAY')
                    logger.error(error_msg)
                    return False, error_msg
                
                if not metadata['layers']:
                    error_msg = get_error_message('LAYER_CONSTRUCTION_ERROR', error="No layers defined in metadata")
                    logger.error(error_msg)
                    return False, error_msg
                
                # for layer in metadata['layers']:
                #     logger.error(layer, 'AMRUT', Qgis.Info)
                #
                #     # Parse layer string to dict and extract name
                #     layer_str = layer.strip('{}').strip()
                #     layer_name = layer_str.split(':', 1)[0].strip()
                #     layer_name = layer_name.strip('"\'')
                #     # First key
                #     layer_file = f"{layer_name}.geojson"
                #     logger.Info(f"Layer file :{layer_file}")                #     logger.Info(f"Available Files :{zip_ref.namelist()}")                #     if layer_file not in zip_ref.namelist():
                #         error_msg = get_error_message('FEATURE_NOT_FOUND_IN_LAYER', feature_id=layer_name)
                #         logger.error(error_msg)
                #         return False, error_msg
                
                return True, None
                
            except json.JSONDecodeError:
                error_msg = get_error_message('INVALID_METADATA_FORMAT')
                logger.error(error_msg)
                return False, error_msg
                
    except zipfile.BadZipFile:
        error_msg = get_error_message('AMRUT_FILE_VALIDATION_ERROR', error="Invalid or corrupted file")
        logger.error(error_msg)
        return False, error_msg
    except Exception as e:
        error_msg = get_error_message('UNKNOWN_ERROR', error=str(e))
        logger.error(error_msg)
        return False, error_msg

def get_amrut_files(directory):
    """Returns a list of all files in the directory ending with .amrut extension."""
    if not os.path.isdir(directory):
        error_msg = get_error_message('FILE_ACCESS_ERROR', error="Invalid directory path")
        logger.error(error_msg)
        return []
    
    return [f for f in os.listdir(directory) if f.endswith(".amrut")]

def validate_amrut_files(directory):
    """Returns a Pair of <Boolean and Message/Data>"""
    amrut_files = get_amrut_files(directory)
    
    if len(amrut_files) == 0:
        error_msg = get_error_message('NO_DIRECTORY_SELECTED')
        logger.error(error_msg)
        return False, error_msg

    layer_map = {}

    for amrut_file in amrut_files:
        amrut_path = os.path.join(directory, amrut_file)
        with zipfile.ZipFile(amrut_path, 'r') as zip_ref:
            if "metadata.json" not in zip_ref.namelist():
                error_msg = get_error_message('MISSING_METADATA_FILE')
                logger.error(error_msg)
                return False, error_msg

            try:
                with zip_ref.open("metadata.json") as metadata_file:
                    metadata = json.load(metadata_file)
                    if "resurvey" in metadata:
                        error_msg = get_error_message('FILE_MARKED_FOR_RESURVEY')
                        logger.error(error_msg)
                        return False, error_msg
                    if metadata.get("qc_status") != "verified":
                        error_msg = get_error_message('NOT_VERIFIED_FILE', file_name=amrut_file)
                        logger.error(error_msg)
                        return False, error_msg

                    layers_qc_completed = metadata.get("layers_qc_completed", [])
                    if not layers_qc_completed:
                        error_msg = get_error_message('ALL_LAYERS_VERIFIED')
                        logger.error(error_msg)
                        return False, error_msg

                    for layer_name in layers_qc_completed:
                        layer_map[layer_name] = "Processed"

            except json.JSONDecodeError:
                error_msg = get_error_message('INVALID_METADATA_FORMAT')
                logger.error(error_msg)
                return False, error_msg
            except Exception as e:
                error_msg = get_error_message('UNKNOWN_ERROR', error=str(e))
                logger.error(error_msg)
                return False, error_msg

    return True, (amrut_files, layer_map)
