from PyQt5.QtCore import QRunnable, QThreadPool, pyqtSignal, QObject
from . import export_geometry as geometry, export_grid as grid
from . import export_clip as clip
from qgis.core import QgsMessageLog, Qgis
from qgis.core import QgsVectorLayer
from qgis.PyQt.QtWidgets import QMessageBox
from ..utils.errors import get_error_message
from ..logger.logger import Logger
logger = Logger()
class LayerValidationWorker(QObject):
    """Worker to validate geometries and extents in a background thread."""

    result_signal = pyqtSignal(bool, str)  # Signal to send results back
    finished = pyqtSignal()
    
    def __init__(self, layers):
        super().__init__()
        self.layers = layers

    def run(self):
        """Run validation in the background."""
        try:
            # Example: Validate layers (replace with your own validation code)
            logger.info('Validation started for selected layers.')  # Check if the task is started
            valid = geometry.check_geometries_and_extents(self.layers)
            logger.info('Validation Completed for selected layers.', )  # Check result
            self.result_signal.emit(valid, "") 
            self.finished.emit() # Emit result back to the main thread
        except Exception as e:
            error_msg = get_error_message('VALIDATION_ERROR', details=str(e))
            logger.error(error_msg)
            logger.error(f'Validation error: {error_msg}')
            self.result_signal.emit(False, error_msg)
            self.finished.emit() # Emit error message

class GridLayerValidationWorker(QObject):
    """Worker to validate geometries and extents in a background thread."""

    result_signal = pyqtSignal(bool, str)  # Signal to send results back
    finished = pyqtSignal()
    
    def __init__(self, layer):
        super().__init__()
        self.layer = layer

    def run(self):
        """Run validation in the background."""
        try:
            # Example: Validate layers (replace with your own validation code)
            logger.info('Validation started for selected grid layers.')  # Check if the task is started
            valid = geometry.check_polygon_in_a_layer(self.layer)
            logger.info('Validation Completed for selected grid layers.')  # Check result
            self.result_signal.emit(valid, "") 
            self.finished.emit() # Emit result back to the main thread
        except Exception as e:
            error_msg = get_error_message('VALIDATION_ERROR', details=str(e))
            logger.error(f'Validation error: {error_msg}')
            self.result_signal.emit(False, error_msg)
            self.finished.emit() # Emit error message

class GridLayerCreationWorker(QObject):
    """Worker to create grid layer in a background thread."""
    
    result_signal = pyqtSignal(str)  # Signal to send result layer ID back
    error_signal = pyqtSignal(str)   # Signal for error messages
    finished = pyqtSignal()
    
    def __init__(self, selected_layers, boundary_layer, grid_size):
        super().__init__()
        self.selected_layers = selected_layers
        self.boundary_layer = boundary_layer
        self.grid_size = grid_size

    def run(self):
        try:
            logger.info('Creating Grid Layer')
            layer_id = grid.create_grid_within_single_polygon(
                self.selected_layers,
                self.boundary_layer,
                self.grid_size, 
                self.boundary_layer.crs().authid()
            )
            self.result_signal.emit(layer_id)
            self.finished.emit()
        except Exception as e:
            error_msg = get_error_message('GRID_CREATION_ERROR', details=str(e))
            logger.error(f'Grid creation error: {error_msg}')
            self.error_signal.emit(error_msg)
            self.finished.emit()

class ClippingWorker(QObject):

    success_signal = pyqtSignal(bool)  # Signal to send results back
    error_signal = pyqtSignal(str)
    progress_signal = pyqtSignal(int)
    finished = pyqtSignal()
    
    def __init__(self, gridLayer, selectedLayers, output_dir):
        super().__init__()
        self.gridLayer = gridLayer
        self.selectedLayers = selectedLayers
        self.output_dir = output_dir

    def run(self):
        try:
            # Example: Validate layers (replace with your own validation code)
            logger.info('Starting clipping operation')  # Check if the task is started
            clip.clip_layers_to_grid(grid_layer=self.gridLayer, layers=self.selectedLayers, progress_signal=self.progress_signal, output_base_dir=self.output_dir)
            self.success_signal.emit(True) 
            self.finished.emit() # Emit result back to the main thread
        except Exception as e:
            error_msg = get_error_message('CLIPPING_ERROR', error=str(e))
            logger.error(f'Clipping error: {error_msg}')
            self.error_signal.emit(error_msg)
            self.finished.emit() # Emit error message