import os
from qgis.core import QgsProject



def get_qgis_project_root():
    """
    Returns the root directory of the currently open QGIS project as a string.
    """
    return QgsProject.instance().homePath()

def create_sankalan_file(params: dict):
    """
    Creates a file named 'sankalan' in the project root directory and writes params in .env style, utf-8 encoded.
    :param params: Dict of key-value pairs to write
    """
    root_dir = get_qgis_project_root()
    file_path = os.path.join(root_dir, 'sankalan')
    lines = [f"{key}={value}" for key, value in params.items()]
    content = '\n'.join(lines)
    with open(file_path, 'w', encoding='utf-8') as f:
        f.write(content)

def read_sankalan_file():
    """
    Reads the 'sankalan' file from the QGIS project root and returns a dictionary of key-value pairs.
    """
    root_dir = get_qgis_project_root()
    file_path = os.path.join(root_dir, 'sankalan')
    params = {}
    if not os.path.isfile(file_path):
        return params
    with open(file_path, 'r', encoding='utf-8') as f:
        for line in f:
            line = line.strip()
            if not line or line.startswith('#'):
                continue
            if '=' in line:
                key, value = line.split('=', 1)
                params[key.strip()] = value.strip()
    return params


def update_sankalan_file(updates: dict, overwrite: bool = False):
    """
    Updates or replaces the 'sankalan' file in the QGIS project root.

    :param updates: Dict of key-value pairs to write.
    :param overwrite: If True, replaces the entire file with updates.
                      If False, merges updates into existing content.
    """
    if overwrite:
        # Completely rewrite the file
        create_sankalan_file(updates)
    else:
        # Merge existing + new
        params = read_sankalan_file()
        params.update(updates)
        create_sankalan_file(params)

def delete_sankalan_file():
    """
    Deletes any existing 'sankalan' file in the QGIS project root

    """
    root_dir = get_qgis_project_root()
    file_path = os.path.join(root_dir, 'sankalan')

    # Remove the existing file if it exists
    if os.path.exists(file_path):
        try:
            os.remove(file_path)
        except Exception as e:
            raise IOError(f"Failed to delete existing sankalan file: {e}")

