import json, threading
from typing import List, Optional
from .utils import read_sankalan_file, update_sankalan_file, delete_sankalan_file


class Resume:
    """
    Resume object that manages state persistence through the Sankalan file.
    Attributes are automatically read from the file on creation and written when updated.
    """
    _instance = None
    _lock = threading.Lock()

    # (your existing __init__ and methods)

    @classmethod
    def get_instance(cls):
        """
        Returns a single shared instance of Resume.
        Creates it if it doesn't exist.
        Thread-safe.
        """
        if cls._instance is None:
            with cls._lock:
                if cls._instance is None:
                    cls._instance = cls()
        return cls._instance

    @classmethod
    def destroy_instance(cls):
        """
        Destroys the existing Resume instance (if any) and its file.
        """
        if cls._instance:
            try:
                cls._instance.destroy()  # your cleanup logic
            except Exception:
                pass
            cls._instance = None

    def __init__(self):
        """
        Initialize Resume object by reading values from the Sankalan file.
        """
        self._layers_to_clip: List[str] = []
        self._grid_layer: str = ""
        self._last_grid_processed: int = -1
        self._is_server_connected: bool = False
        self._destination: str = ""
        self._completed : bool = False
        
        # Load values from Sankalan file
        self._load_from_sankalan()
    
    def _load_from_sankalan(self):
        """
        Load values from the Sankalan file.
        """
        params = read_sankalan_file()
        
        # Parse layers_to_clip from JSON string
        layers_str = params.get('layers_to_clip', '[]')
        try:
            self._layers_to_clip = json.loads(layers_str) if layers_str else []
        except json.JSONDecodeError:
            self._layers_to_clip = []
        
        # Load other attributes
        self._grid_layer = params.get('grid_layer', '')
        
        # Parse last_grid_processed as integer
        try:
            self._last_grid_processed = int(params.get('last_grid_processed', '0'))
        except (ValueError, TypeError):
            self._last_grid_processed = -1
        
        # Parse is_server_connected as boolean
        server_connected_str = params.get('isServerConnected', 'False').lower()
        completed_str = params.get('completed', 'False').lower()
        self._is_server_connected = server_connected_str in ('true', '1', 'yes', 'on')
        self._completed = completed_str in ('true', '1', 'yes', 'on')
        
        self._destination = params.get('destination', '')
    
    def _save_to_sankalan(self):
        """
        Save current values to the Sankalan file.
        """
        updates = {
            'layers_to_clip': json.dumps(self._layers_to_clip),
            'grid_layer': self._grid_layer,
            'last_grid_processed': str(self._last_grid_processed),
            'isServerConnected': str(self._is_server_connected),
            'completed': str(self._completed),
            'destination': self._destination
        }
        update_sankalan_file(updates)
    
    @property
    def layers_to_clip(self) -> List[str]:
        """Get the list of layers to clip."""
        return self._layers_to_clip.copy()
    
    @layers_to_clip.setter
    def layers_to_clip(self, value: List[str]):
        """Set the list of layers to clip and save to Sankalan file."""
        self._layers_to_clip = value.copy() if value else []
        self._save_to_sankalan()
    
    @property
    def grid_layer(self) -> str:
        """Get the grid layer name."""
        return self._grid_layer
    
    @grid_layer.setter
    def grid_layer(self, value: str):
        """Set the grid layer name and save to Sankalan file."""
        self._grid_layer = value or ""
        self._save_to_sankalan()
    
    @property
    def last_grid_processed(self) -> int:
        """Get the last grid processed index."""
        return self._last_grid_processed
    
    @last_grid_processed.setter
    def last_grid_processed(self, value: int):
        """Set the last grid processed index and save to Sankalan file."""
        self._last_grid_processed = value if isinstance(value, int) else -1
        self._save_to_sankalan()
    
    @property
    def is_server_connected(self) -> bool:
        """Get the server connection status."""
        return self._is_server_connected
    
    @is_server_connected.setter
    def is_server_connected(self, value: bool):
        """Set the server connection status and save to Sankalan file."""
        self._is_server_connected = bool(value)
        self._save_to_sankalan()

    @property
    def completed(self) -> bool:
        """Get the server connection status."""
        return self._completed

    @completed.setter
    def completed(self, value: bool):
        """Set the server connection status and save to Sankalan file."""
        self._completed = bool(value)
        self._save_to_sankalan()
    @property
    def destination(self) -> str:
        """Get the destination path."""
        return self._destination
    
    @destination.setter
    def destination(self, value: str):
        """Set the destination path and save to Sankalan file."""
        self._destination = value or ""
        self._save_to_sankalan()
    
    def reset(self):
        """
        Reset all values to their defaults and save to Sankalan file.
        """
        self._layers_to_clip = []
        self._grid_layer = ""
        self._last_grid_processed = -1
        self._is_server_connected = False
        self._completed = False
        self._destination = ""
        self._save_to_sankalan()
        delete_sankalan_file()

    def is_resumable(self) -> bool:
        """
        Check if the
        task is resumable.
        A
        task is resumable if it
        has
        values in layers_to_clip, grid_layer,
        destination, and completed is False.

        Returns:
        bool: True if the
        task is resumable, False
        otherwise.
        """

        return (
                bool(self._layers_to_clip) and  # layers_to_clip has values
                bool(self._grid_layer.strip()) and  # grid_layer is not empty
                bool(self._destination.strip()) and  # destination is not empty
                not self._completed  # completed is False
        )

    def destroy(self):
        self.__dict__.clear()

    def __str__(self) -> str:
        """String representation of the Resume object."""
        return (f"Resume(layers_to_clip={self.layers_to_clip}, "
                f"grid_layer='{self.grid_layer}', "
                f"last_grid_processed={self.last_grid_processed}, "
                f"is_server_connected={self.is_server_connected}, "
                f"destination='{self.destination}')")
    
    def __repr__(self) -> str:
        """Detailed representation of the Resume object."""
        return self.__str__()
