from .import_validation import validate_project, validate_amrut_file
from ..utils.errors import get_error_message
from ..logger.logger import Logger
from qgis.core import (
    QgsVectorFileWriter,
    QgsVectorLayer,
    QgsProject,
    QgsFeature,
    QgsGeometry,
    QgsFields,
    QgsField,
    QgsWkbTypes,
    QgsCoordinateReferenceSystem,
    QgsMessageLog,
    Qgis
)
from PyQt5.QtCore import QVariant
import os
import zipfile
import tempfile
import processing
import traceback
import uuid

merged_layer = None
logger = Logger()

def construct_layer(directory, amrut_files, layer_name):
    """Constructs layer from AMRUT files with proper error handling"""
    try:
        # Validate project first
        validate_project()
        
        global merged_layer
        init_merged_layer(layer_name)
        layers_to_merge = []

        # Validate each AMRUT file
        for amrut_file in amrut_files:
            logger.info(f"Processing file {amrut_file}")
            amrut_path = os.path.join(directory, amrut_file)
            is_valid, error_msg = validate_amrut_file(amrut_path)
            
            if not is_valid:
                QgsMessageLog.logMessage(error_msg, 'Sankalan 2.0', Qgis.Critical)
                return False, error_msg

            # Process valid file
            try:
                with zipfile.ZipFile(amrut_path, 'r') as zip_ref:
                    layer_file_name = f"{layer_name}.geojson"
                    if layer_file_name not in zip_ref.namelist():
                       continue

                    # Extract and process layer
                    geojson_layer = process_layer_file(zip_ref, layer_file_name, layer_name)
                    if geojson_layer and geojson_layer.isValid():
                        layers_to_merge.append(geojson_layer)
                    else:
                        error_msg = get_error_message('LAYER_CONSTRUCTION_ERROR', error=f"Failed to create valid layer from {amrut_file}")
                        QgsMessageLog.logMessage(error_msg, 'Sankalan 2.0', Qgis.Critical)
                        return False, error_msg

            except Exception as e:
                error_msg = get_error_message('IMPORT_HANDLER_ERROR', error=str(e))
                QgsMessageLog.logMessage(error_msg, 'Sankalan 2.0', Qgis.Critical)
                return False, error_msg

        # Merge and save layers
        if not layers_to_merge:
            error_msg = get_error_message('LAYER_CONSTRUCTION_ERROR', error="No valid layers found to merge")
            return False, error_msg

        try:
            merge_layers(layers_to_merge)
            saved_layer_path = save_temporary_layer(layer_name)
            return True, saved_layer_path
        except Exception as e:
            error_msg = get_error_message('LAYER_MERGE_ERROR', error=str(e))
            QgsMessageLog.logMessage(error_msg, 'Sankalan 2.0', Qgis.Critical)
            return False, error_msg

    except ValueError as ve:
        error_msg = get_error_message('VALIDATION_ERROR', details=str(ve))
        QgsMessageLog.logMessage(error_msg, 'Sankalan 2.0', Qgis.Critical)
        return False, error_msg
    except Exception as e:
        tb = traceback.format_exc()
        error_msg = get_error_message('UNKNOWN_ERROR', error=f"{e}\n{tb}")
        QgsMessageLog.logMessage(error_msg, 'Sankalan 2.0', Qgis.Critical)
        return False, error_msg

def geometry_string_for_memory_layer(wkb_type: int) -> str:
    # Get a display string or default
    disp = QgsWkbTypes.displayString(wkb_type)
    if not disp:
        # Fallback to a sensible default
        return "Polygon"

    # Ensure disp is a string
    disp = str(disp)

    # Normalize: remove Z/M markers and take first token only
    # Examples: "MultiPolygon ZM" -> "MultiPolygon", "PolygonZ" -> "Polygon"
    # Handle both space-separated and concatenated suffixes across QGIS versions
    token = disp.split()[0]  # string, not a slice
    # Remove trailing Z/M if concatenated (e.g., "PolygonZ", "MultiLineStringM", "PointZM")
    for suffix in ("ZM", "Z", "M"):
        if token.endswith(suffix):
            token = token[: -len(suffix)]
            break

    geom = token  # should be a plain string like "Polygon", "MultiPolygon", "LineString", "Point"

    # Map generic types to memory provider-friendly names
    mapping = {
        "MultiSurface": "MultiPolygon",
        "MultiCurve": "MultiLineString",
        "CurvePolygon": "Polygon",
        "GeometryCollection": "MultiPolygon",  # choose a default if collections appear
        "UnknownGeometry": "Polygon",
    }

    return mapping.get(geom, geom)

def init_merged_layer(layer_name):
    """Initialize merged layer for processing"""
    global merged_layer
    print(f"Init merged layer for {layer_name}")

    matches = QgsProject.instance().mapLayersByName(layer_name)
    if not matches:
        error_msg = get_error_message(
            'LAYER_RETRIEVAL_ERROR',
            layer_name=layer_name,
            error="Layer not found in project"
        )
        QgsMessageLog.logMessage(error_msg, 'Sankalan 2.0', Qgis.Critical)
        raise ValueError(error_msg)

    if len(matches) > 1:
        logger.warning(f"Multiple layers named '{layer_name}' found; using the first match.")
    active_layer = matches[0]

    # Determine geometry type string safely
    geometry_type = active_layer.wkbType()
    geometry_str = geometry_string_for_memory_layer(geometry_type)
    crs = QgsProject.instance().crs()
    layer_uri = f"{geometry_str}?crs={crs.authid()}"

    merged_layer = QgsVectorLayer(layer_uri, f"Temporary_{layer_name}", "memory")
    if not merged_layer or not merged_layer.isValid():
        error_msg = get_error_message(
            'LAYER_CONSTRUCTION_ERROR',
            error=f"Failed to create memory layer with uri '{layer_uri}'"
        )
        QgsMessageLog.logMessage(error_msg, 'Sankalan 2.0', Qgis.Critical)
        raise ValueError(error_msg)

    return merged_layer


def merge_layers(layers_to_merge):
    """Merge multiple layers into one"""
    for layers in layers_to_merge :
        logger.info(f"Merging layers : {layers}.")
        global merged_layer
    parameters = {
        'LAYERS': layers_to_merge,
        'CRS': QgsProject.instance().crs(),
        'OUTPUT': 'memory:'
    }
    try:
        merged_layer = processing.run("native:mergevectorlayers", parameters)['OUTPUT']
        if not merged_layer or not merged_layer.isValid():
            error_msg = get_error_message('LAYER_MERGE_ERROR', error="Failed to create valid merged layer")
            QgsMessageLog.logMessage(error_msg, 'Sankalan 2.0', Qgis.Critical)
            raise ValueError(error_msg)
    except Exception as e:
        error_msg = get_error_message('LAYER_MERGE_ERROR', error=str(e))
        QgsMessageLog.logMessage(error_msg, 'Sankalan 2.0', Qgis.Critical)
        raise

def save_temporary_layer(layer_name):
    """Save merged layer to disk"""
    global merged_layer
    try:
        project_path = QgsProject.instance().homePath()
        temporary_layer_path = os.path.join(project_path, f"{layer_name}_vetted.gpkg")
        save_file_to_disk(temporary_layer_path, merged_layer)
        return temporary_layer_path
    except Exception as e:
        error_msg = get_error_message('FILE_SAVE_ERROR', error=str(e))
        QgsMessageLog.logMessage(error_msg, 'Sankalan 2.0', Qgis.Critical)
        raise ValueError(error_msg)

def save_file_to_disk(file_path, layer):
    """Save layer to disk with error handling"""
    try:
        options = QgsVectorFileWriter.SaveVectorOptions()
        error = QgsVectorFileWriter.writeAsVectorFormatV2(
            layer=layer,
            fileName=file_path,
            transformContext=QgsProject.instance().transformContext(),
            options=options
        )
        if error[0] != QgsVectorFileWriter.NoError:
            error_msg = get_error_message('FILE_SAVE_ERROR', error=error[1])
            QgsMessageLog.logMessage(error_msg, 'Sankalan 2.0', Qgis.Critical)
            raise ValueError(error_msg)
    except Exception as e:
        error_msg = get_error_message('FILE_SAVE_ERROR', error=str(e))
        QgsMessageLog.logMessage(error_msg, 'Sankalan 2.0', Qgis.Critical)
        raise

def process_layer_file(zip_ref, layer_file_name, layer_name):
    """Process individual layer file from ZIP archive"""
    try:
        geojson_data = zip_ref.read(layer_file_name).decode('utf-8')
        temp_dir = tempfile.gettempdir()
        unique_id = uuid.uuid4().hex
        temp_geojson_file_path = os.path.join(temp_dir, f"Temporary_{unique_id}_{os.path.basename(layer_file_name)}")
        
        with open(temp_geojson_file_path, 'w', encoding='utf-8') as temp_file:
            temp_file.write(geojson_data)
            
        layer = QgsVectorLayer(temp_geojson_file_path, layer_name, "ogr")
        if not layer.isValid():
            error_msg = get_error_message('LAYER_CONSTRUCTION_ERROR', error=f"Failed to create valid layer from {layer_file_name}")
            QgsMessageLog.logMessage(error_msg, 'Sankalan 2.0', Qgis.Critical)
            return None
            
        return layer
        
    except Exception as e:
        error_msg = get_error_message('IMPORT_HANDLER_ERROR', error=str(e))
        QgsMessageLog.logMessage(error_msg, 'Sankalan 2.0', Qgis.Critical)
        return None


