from PyQt5.QtCore import QRunnable, QThreadPool, pyqtSignal, QObject
from . import export_geometry as geometry, export_clip as clip, export_grid as grid
from qgis.core import QgsMessageLog, Qgis
from qgis.core import QgsVectorLayer
from qgis.PyQt.QtWidgets import QMessageBox
from .errors import get_error_message, get_error_title

class LayerValidationWorker(QObject):
    """Worker to validate geometries and extents in a background thread."""

    result_signal = pyqtSignal(bool, str)  # Signal to send results back
    finished = pyqtSignal()
    
    def __init__(self, layers):
        super().__init__()
        self.layers = layers

    def run(self):
        """Run validation in the background."""
        try:
            # Example: Validate layers (replace with your own validation code)
            QgsMessageLog.logMessage('Validation started.', 'AMRUT_Export', Qgis.Info)  # Check if the task is started
            valid = geometry.check_geometries_and_extents(self.layers)
            QgsMessageLog.logMessage('Validation Completed.', 'AMRUT_Export', Qgis.Info)  # Check result
            self.result_signal.emit(valid, "") 
            self.finished.emit() # Emit result back to the main thread
        except Exception as e:
            error_msg = get_error_message('VALIDATION_ERROR', details=str(e))
            QgsMessageLog.logMessage(f'Validation error: {error_msg}', 'AMRUT_Export', Qgis.Critical)
            self.result_signal.emit(False, error_msg)
            self.finished.emit() # Emit error message

class GridLayerValidationWorker(QObject):
    """Worker to validate geometries and extents in a background thread."""

    result_signal = pyqtSignal(bool, str)  # Signal to send results back
    finished = pyqtSignal()
    
    def __init__(self, layer):
        super().__init__()
        self.layer = layer

    def run(self):
        """Run validation in the background."""
        try:
            # Example: Validate layers (replace with your own validation code)
            QgsMessageLog.logMessage('Validation started.', 'AMRUT_Export', Qgis.Info)  # Check if the task is started
            valid = geometry.check_polygon_in_a_layer(self.layer)
            QgsMessageLog.logMessage('Validation Completed.', 'AMRUT_Export', Qgis.Info)  # Check result
            self.result_signal.emit(valid, "") 
            self.finished.emit() # Emit result back to the main thread
        except Exception as e:
            error_msg = get_error_message('VALIDATION_ERROR', details=str(e))
            QgsMessageLog.logMessage(f'Validation error: {error_msg}', 'AMRUT_Export', Qgis.Critical)
            self.result_signal.emit(False, error_msg)
            self.finished.emit() # Emit error message

class GridLayerCreationWorker(QObject):
    """Worker to create grid layer in a background thread."""
    
    result_signal = pyqtSignal(str)  # Signal to send result layer ID back
    error_signal = pyqtSignal(str)   # Signal for error messages
    finished = pyqtSignal()
    
    def __init__(self, selected_layers, boundary_layer, grid_size):
        super().__init__()
        self.selected_layers = selected_layers
        self.boundary_layer = boundary_layer
        self.grid_size = grid_size

    def run(self):
        try:
            QgsMessageLog.logMessage('Creating Grid Layer', 'AMRUT_Export', Qgis.Info)
            layer_id = grid.create_grid_within_single_polygon(
                self.selected_layers,
                self.boundary_layer,
                self.grid_size, 
                self.boundary_layer.crs().authid()
            )
            self.result_signal.emit(layer_id)
            self.finished.emit()
        except Exception as e:
            error_msg = get_error_message('GRID_CREATION_ERROR', details=str(e))
            QgsMessageLog.logMessage(f'Grid creation error: {error_msg}', 'AMRUT_Export', Qgis.Critical)
            self.error_signal.emit(error_msg)
            self.finished.emit()

class ClippingWorker(QObject):

    success_signal = pyqtSignal(bool)  # Signal to send results back
    error_signal = pyqtSignal(str)
    progress_signal = pyqtSignal(int)
    finished = pyqtSignal()
    
    def __init__(self, gridLayer, selectedLayers, output_dir):
        super().__init__()
        self.gridLayer = gridLayer
        self.selectedLayers = selectedLayers
        self.output_dir = output_dir

    def run(self):
        try:
            # Example: Validate layers (replace with your own validation code)
            QgsMessageLog.logMessage('Starting clipping operation', 'AMRUT_Export', Qgis.Info)  # Check if the task is started
            clip.clip_layers_to_grid(grid_layer=self.gridLayer, layers=self.selectedLayers, progress_signal=self.progress_signal, output_base_dir=self.output_dir)
            self.success_signal.emit(True) 
            self.finished.emit() # Emit result back to the main thread
        except Exception as e:
            error_msg = get_error_message('CLIPPING_ERROR', error=str(e))
            QgsMessageLog.logMessage(f'Clipping error: {error_msg}', 'AMRUT_Export', Qgis.Critical)
            self.error_signal.emit(error_msg)
            self.finished.emit() # Emit error message