# -*- coding: utf-8 -*-
"""
erd_scene.py
Scene QGraphics pour afficher un diagramme ER interactif :
tables déplaçables, courbes entre PK/FK, zoom/pan.
"""

from typing import Dict, List

from qgis.PyQt.QtCore import Qt, QRectF, QPointF
from qgis.PyQt.QtGui import (
    QPainter, QPen, QBrush, QColor, QFont, QPainterPath, QTransform
)
from qgis.PyQt.QtWidgets import (
    QGraphicsItem, QGraphicsPathItem, QGraphicsScene,
    QGraphicsView, QGraphicsSimpleTextItem
)

from .ddl_parser import Table, ForeignKey


class TableItem(QGraphicsItem):
    HEADER_HEIGHT = 24
    ROW_HEIGHT = 18
    MARGIN = 6
    WIDTH = 260

    def __init__(self, table: Table):
        super().__init__()
        self.table = table
        self.edges: List["RelationItem"] = []
        self._row_y: Dict[str, float] = {}

        self.setFlag(QGraphicsItem.ItemIsMovable, True)
        self.setFlag(QGraphicsItem.ItemIsSelectable, True)
        self.setFlag(QGraphicsItem.ItemSendsGeometryChanges, True)

    # ------------------------------------------------------------------ bounding rect

    def boundingRect(self) -> QRectF:
        n_rows = max(1, len(self.table.columns))
        h = (
            self.MARGIN * 2
            + self.HEADER_HEIGHT
            + n_rows * self.ROW_HEIGHT
        )
        return QRectF(0, 0, self.WIDTH, h)

    # ------------------------------------------------------------------ paint

    def paint(self, painter: QPainter, option, widget=None):
        r = self.boundingRect()
        painter.setRenderHint(QPainter.Antialiasing, True)

        # Ombre légère
        shadow_rect = r.translated(2, 2)
        painter.setPen(Qt.NoPen)
        painter.setBrush(QColor(0, 0, 0, 30))
        painter.drawRoundedRect(shadow_rect, 6, 6)

        # Fond carte
        bg_color = QColor(250, 250, 252)
        border_color = QColor(120, 120, 160) if self.isSelected() else QColor(180, 180, 200)
        painter.setBrush(bg_color)
        painter.setPen(QPen(border_color, 1.2))
        painter.drawRoundedRect(r, 6, 6)

        # Header
        header_rect = QRectF(
            r.left() + 1,
            r.top() + 1,
            r.width() - 2,
            self.HEADER_HEIGHT
        )
        painter.setBrush(QColor(220, 220, 245))
        painter.setPen(Qt.NoPen)
        painter.drawRoundedRect(header_rect, 6, 6)

        painter.setPen(QColor(50, 50, 80))
        font = painter.font()
        font.setBold(True)
        painter.setFont(font)
        painter.drawText(
            header_rect.adjusted(8, 0, -8, 0),
            Qt.AlignVCenter | Qt.AlignLeft,
            self.table.name
        )

        # Séparateur header / corps
        painter.setPen(QPen(QColor(200, 200, 220), 1))
        painter.drawLine(
            int(header_rect.left()),
            int(header_rect.bottom()),
            int(header_rect.right()),
            int(header_rect.bottom())
        )

        # Corps : colonnes
        font.setBold(False)
        painter.setFont(font)

        icon_x = r.left() + self.MARGIN
        name_x = icon_x + 16
        type_x = name_x + 110
        detail_x = type_x + 60

        self._row_y.clear()
        y = r.top() + self.MARGIN + self.HEADER_HEIGHT

        for col in self.table.columns.values():
            ...
            # icône PK / FK (plus de "abc"/"123")
            flags = []
            # 🔑 pour la clé primaire
            if col.primary_key:
                flags.append("🔑")
            # # pour la clé étrangère
            if col.foreign_key:
                flags.append("#")
            icon = " ".join(flags)

            painter.setPen(QColor(90, 90, 120))
            painter.drawText(
                QRectF(icon_x, y, 30, self.ROW_HEIGHT),
                Qt.AlignVCenter | Qt.AlignLeft,
                icon
            )

            # nom
            painter.setPen(QColor(40, 40, 60))
            painter.drawText(
                QRectF(name_x, y, type_x - name_x - 4, self.ROW_HEIGHT),
                Qt.AlignVCenter | Qt.AlignLeft,
                col.name
            )

            # type
            painter.setPen(QColor(120, 120, 150))
            painter.drawText(
                QRectF(type_x, y, detail_x - type_x - 4, self.ROW_HEIGHT),
                Qt.AlignVCenter | Qt.AlignLeft,
                col.type
            )

            # détails
            detail_tokens = []
            if col.not_null:
                detail_tokens.append("not null")
            if col.unique and not col.primary_key:
                detail_tokens.append("unique")
            if col.default:
                detail_tokens.append(f"default {col.default}")

            painter.setPen(QColor(140, 140, 160))
            painter.drawText(
                QRectF(detail_x, y, r.right() - detail_x - self.MARGIN, self.ROW_HEIGHT),
                Qt.AlignVCenter | Qt.AlignLeft,
                ", ".join(detail_tokens)
            )

            center_y = y + self.ROW_HEIGHT / 2.0
            self._row_y[col.name] = center_y
            y += self.ROW_HEIGHT

    # ------------------------------------------------------------------ helpers

    def anchor_for_column(self, col_name: str, side: str) -> QPointF:
        y = self._row_y.get(col_name, self.boundingRect().center().y())
        if side == "left":
            x = self.boundingRect().left()
        else:
            x = self.boundingRect().right()
        return self.mapToScene(QPointF(x, y))

    def itemChange(self, change, value):
        if change == QGraphicsItem.ItemPositionHasChanged:
            for edge in self.edges:
                edge.update_path()
        return super().itemChange(change, value)


class RelationItem(QGraphicsPathItem):
    def __init__(self, src: TableItem, src_col: str,
                 ref: TableItem, ref_col: str,
                 cardinality: str):
        super().__init__()
        self.src = src
        self.src_col = src_col
        self.ref = ref
        self.ref_col = ref_col
        self.cardinality = cardinality

        pen = QPen(QColor(160, 160, 190))
        pen.setWidthF(1.2)
        self.setPen(pen)
        self.setZValue(-1)

        self.label = QGraphicsSimpleTextItem(cardinality, self)
        self.label.setBrush(QColor(110, 110, 140))

        self.update_path()

    def update_path(self):
        p1 = self.ref.anchor_for_column(self.ref_col, side="right")
        p2 = self.src.anchor_for_column(self.src_col, side="left")

        path = QPainterPath(p1)
        mid_x = (p1.x() + p2.x()) / 2.0
        path.cubicTo(mid_x, p1.y(), mid_x, p2.y(), p2.x(), p2.y())
        self.setPath(path)

        mid = path.pointAtPercent(0.5)
        self.label.setPos(mid + QPointF(4, -4))


class ErdScene(QGraphicsScene):
    def __init__(self, tables: Dict[str, Table], fks: List[ForeignKey]):
        super().__init__()
        self.setBackgroundBrush(QColor(245, 246, 250))

        self.table_items: Dict[str, TableItem] = {}

        # placer les tables en grille grossière
        cols = max(1, int(len(tables) ** 0.5))
        spacing_x = 320
        spacing_y = 200

        for idx, (tname, table) in enumerate(tables.items()):
            item = TableItem(table)
            row = idx // cols
            col = idx % cols
            item.setPos(col * spacing_x, row * spacing_y)
            self.addItem(item)
            self.table_items[tname] = item

        # relations
        for fk in fks:
            src_item = self.table_items.get(fk.src_table)
            ref_item = self.table_items.get(fk.ref_table)
            if not src_item or not ref_item or not fk.src_cols or not fk.ref_cols:
                continue
            src_col = fk.src_cols[0]
            ref_col = fk.ref_cols[0]

            if fk.src_unique and fk.src_not_null:
                card = "1-1"
            elif fk.src_unique and not fk.src_not_null:
                card = "0-1"
            elif fk.src_not_null:
                card = "1-N"
            else:
                card = "0-N"

            edge = RelationItem(src_item, src_col, ref_item, ref_col, card)
            self.addItem(edge)
            src_item.edges.append(edge)
            ref_item.edges.append(edge)

        self.setSceneRect(self.itemsBoundingRect().adjusted(-40, -40, 40, 40))


class ErdView(QGraphicsView):
    def __init__(self, parent=None):
        super().__init__(parent)
        self._zoom = 1.0
        self.setRenderHints(
            QPainter.Antialiasing | QPainter.TextAntialiasing
        )
        self.setDragMode(QGraphicsView.ScrollHandDrag)

    def wheelEvent(self, event):
        angle = event.angleDelta().y()
        if angle == 0:
            return
        factor = 1.2 if angle > 0 else 0.8
        self._zoom *= factor
        self.scale(factor, factor)

    def reset_view(self):
        self._zoom = 1.0
        self.setTransform(QTransform())
        if self.scene():
            self.fitInView(self.sceneRect(), Qt.KeepAspectRatio)