# -*- coding: utf-8 -*-

from PyQt4.QtCore import *
from PyQt4.QtGui import *

from qgis.gui import *
from qgis.core import *

from DatabaseModel import TableItem, SchemaItem, SqlItem, DatabaseModel
import postgis_utils

from Utils import QueryUtils
from ui.WizPage1_ui import Ui_WizardPage

class WizardPage(QWizardPage, Ui_WizardPage):
	
	def __init__(self, iface, wizState):
		QWizardPage.__init__(self)
		self.iface = iface
		self.canvas = self.iface.mapCanvas()
		self.wizState = wizState

		self.setupUi(self)

		self.drawer = PolygonDrawer(self.canvas)

		self.tablesSelector.setMode(self.tablesSelector.SingleSelection, postgis_utils.Table.GEOM_POLYGON, False)
		self.connect( self.tablesSelector, SIGNAL("currentTableChanged(PyQt_PyObject)"), self.fillDbColsCombo )

		self.connect(self.cuttingArchiveTypeCombo, SIGNAL("currentIndexChanged(int)"), SIGNAL("completeChanged()") )

		self.connect(self.dbColsCombo, SIGNAL("currentIndexChanged(int)"), SIGNAL( "completeChanged()" ) )

		self.connect(self.layerNamesCombo, SIGNAL("currentIndexChanged(int)"), self.fillLayerAttrsCombo)
		self.connect(self.layerAttrsCombo, SIGNAL("currentIndexChanged(int)"), SIGNAL("completeChanged()") )

		self.connect(self.drawPolygonButton, SIGNAL("clicked()"), self.startDrawPolygon)
		self.connect(self.drawer, SIGNAL("polygonDrawingEnded()"), self.endDrawPolygon)
		self.connect(self.cutTitleEdit, SIGNAL("textChanged(const QString &)"), SIGNAL("completeChanged()") )

	def initializePage(self):
		# fill defaults
		self.tablesSelector.fillConnectionCombo()
		self.fillLayerNamesCombo()

		# but first let the wizard chance to show the page
		QTimer.singleShot(50, self.enableDrawPolygonOnFly)

	def validatePage(self):
		index = self.cuttingArchiveTypeCombo.currentIndex()
		db = self.tablesSelector.db

		self.wizState.cuttingArchiveMode = index
		self.wizState.cuttingArchiveModeName = self.cuttingArchiveTypeCombo.currentText()
		self.wizState.cuttingArchiveDb = (self.tablesSelector.currentConnection, db) if index == 0 else (None, None)

		if index == 0:
			if isinstance(self.selectedTable, TableItem):
				table = self.selectedTable
				self.wizState.cuttingArchiveLayer = ( table.schema().name, table.name, table.geom_column, None )
			elif isinstance(self.selectedTable, SqlItem):
				sql = self.selectedTable
				self.wizState.cuttingArchiveLayer = ( None, None, sql.geom_column, sql.name )
			else:
				return False
			self.wizState.cuttingArchiveTitle = self.dbColsCombo.currentText().mid(4)

			schema, table, geom, query = self.wizState.cuttingArchiveLayer
			if query == None:
				query = db._table_name( schema, table )

			srid, geomType, dim = QueryUtils.getSridAndGeomType( db, query, geom )
			if srid > 0:
				self.wizState.srid = srid

		elif index == 1:
			self.wizState.cuttingArchiveLayer = self.layers[self.layerNamesCombo.currentIndex()]
			self.wizState.cuttingArchiveTitle = self.layerAttrsCombo.currentText().mid(4)
			self.wizState.srid = self.wizState.cuttingArchiveLayer.srs().postgisSrid()
		elif index == 2:
			self.wizState.cuttingArchiveLayer = self.drawer.getPolygon()
			self.wizState.cuttingArchiveTitle = self.cutTitleEdit.text()
			self.wizState.srid = self.canvas.mapRenderer().destinationSrs().postgisSrid()
		return True

	def isComplete(self):
		index = self.cuttingArchiveTypeCombo.currentIndex()
		if index == 0:
			return self.dbColsCombo.currentIndex() >= 0
		elif index == 1:
			return not (self.layerNamesCombo.currentIndex() < 0 or self.layerAttrsCombo.currentIndex() < 0)
		elif index == 2:
			return self.drawer.isValid() and not self.cutTitleEdit.text().isEmpty()
		return False

	def enableDrawPolygonOnFly(self):
		if not self.canvas.mapRenderer().destinationSrs().geographicFlag():
			return
		self.cuttingArchiveTypeCombo.removeItem(2)
		QMessageBox.information(self, self.tr( "Feature disabled" ), self.tr( "Defining a clip polygon by digitising it is disabled. \nTo enable it use a Projected Coordinate System." ) )

	def onClosing(self):
		self.canvas.unsetMapTool(self.drawer)
		self.drawer.reset()

		# close the connection to the database
		#self.tablesSelector.setDatabase(None)

	def startDrawPolygon(self):
		self.wizard().hide()
		self.drawer.reset()
		self.canvas.setMapTool(self.drawer)

	def endDrawPolygon(self):
		self.canvas.unsetMapTool(self.drawer)
		self.wizard().setModal( True )
		self.wizard().show()
		self.emit( SIGNAL( "completeChanged()" ) )

	def fillDbColsCombo(self, item):
		self.selectedTable = item

		self.dbColsCombo.clear()
		if item == None:
			return

		if isinstance(item, TableItem):
			table = self.tablesSelector.db._table_name(item.schema().name, item.name)
		elif isinstance(item, SqlItem):
			table = item.name

		fields = QueryUtils.getFields( self.tablesSelector.db, table, None )
		self.dbColsCombo.addItems( self.getFieldNames4Combo(fields) )


	def getFieldNames4Combo(self, fields):
		names = []

		for f in fields:
			if f.typeName() == 'bool':
				continue
			elif f.typeName() == 'geometry':
				continue

			typeicon = "(?)"
			if f.type() == QVariant.Int:
				typeicon = "(I)"
			elif f.type() == QVariant.Double or f.type() == QVariant.LongLong:
				typeicon = "(R)"
			elif f.type() == QVariant.String:
				typeicon = "(S)"
			names.append( typeicon + " " + unicode(f.name()) )

		return names
		

	def fillLayerNamesCombo(self):
		self.layerNamesCombo.clear()

		layers = dict()
		names = []
		layerMap = QgsMapLayerRegistry.instance().mapLayers()
		for name, layer in layerMap.iteritems():
			if layer.type() == layer.VectorLayer and layer.geometryType() == QGis.Polygon:
				layers[ len(layers) ] = layer
				names.append(layer.name())

		self.layers = layers
		self.layerNamesCombo.addItems( names )

	def fillLayerAttrsCombo(self):
		self.layerAttrsCombo.clear()

		# get the layer file name
		index = self.layerNamesCombo.currentIndex()
		if index < 0 or index >= len(self.layers):
			return

		layer = self.layers[index]
		if layer == None:
			return

		names = []
		pr = layer.dataProvider()
		flds = pr.fields()

		self.layerAttrsCombo.addItems(  self.getFieldNames4Combo(flds.values()) )


class PolygonDrawer(QgsMapToolEmitPoint):
	def __init__(self, canvas):
		self.canvas = canvas
		QgsMapToolEmitPoint.__init__(self, self.canvas)

		self.rubberBand = QgsRubberBand( self.canvas, True )	# true, its a polygon
		self.rubberBand.setColor( Qt.red )
		self.rubberBand.setWidth( 1 )

		self.isEmittingPoints = False

	def reset(self):
		self.isEmittingPoints = False
		self.rubberBand.reset( True )	# true, its a polygon

	def canvasPressEvent(self, e):
		if e.button() == Qt.RightButton:
			self.isEmittingPoints = False
			self.emit( SIGNAL("polygonDrawingEnded()") )
			return

		if e.button() == Qt.LeftButton:
			self.isEmittingPoints = True
		else:
			return
		point = self.toMapCoordinates( e.pos() )
		self.rubberBand.addPoint( point, True )	# true to update canvas
		self.rubberBand.show()

	def canvasMoveEvent(self, e):
		if not self.isEmittingPoints:
			return

		point = self.toMapCoordinates( e.pos() )
		self.rubberBand.movePoint( point )

	def isValid(self):
		return self.rubberBand.numberOfVertices() > 0

	def getPolygon(self):
		points = []
		for i in range( self.rubberBand.numberOfVertices() ):
			points.append( self.rubberBand.getPoint(0, i) )
		# close the ring by adding the first point also at the end of the list
		points.append(points[0])

		return QgsGeometry.fromPolygon( [points] )

	def deactivate(self):
		QgsMapTool.deactivate(self)
		self.emit(SIGNAL("deactivated()"))

