# -*- coding: utf-8 -*-

from PyQt4.QtCore import *
from PyQt4.QtGui import *

from qgis.gui import *
from qgis.core import *

from DatabaseModel import TableItem, SchemaItem, SqlItem, DatabaseModel
import postgis_utils

from ui.TablesSelector_ui import Ui_TablesSelector

class TablesSelector(QWidget, Ui_TablesSelector):

	SingleSelection, MultiSelection = range(2)
	
	def __init__(self, parent=None):
		QWidget.__init__(self, parent)
		self.db = None
		self.currentConnection = None
		self.queries = dict()
		self.selectedTables = []

		self.setupUi(self)

		self.dbTreeView.setModel( DatabaseModel(self) )
		self.setMode( self.SingleSelection, postgis_utils.Table.FILTER_ALL, True )

		self.connect(self.connectButton, SIGNAL("clicked()"), self.connectToDatabase)

		self.connect(self.dbTreeView.model(), SIGNAL("dataChanged(const QModelIndex &, const QModelIndex &)"), self.refreshDbTree)
		self.connect(self.dbTreeView.selectionModel(), SIGNAL("currentChanged(const QModelIndex&, const QModelIndex&)"), self.currentTableChanged)
		self.connect(self.dbTreeView.selectionModel(), SIGNAL("selectionChanged(const QItemSelection &, const QItemSelection &)"), self.selectionChanged)

		self.connect(self.customQuery, SIGNAL("completed(bool)"), self.addQueryBtn.setEnabled)
		self.connect(self.addQueryBtn, SIGNAL("clicked()"), self.addQueryToList)
		self.connect(self.runSqlLayerBtn, SIGNAL("clicked()"), self.runSqlLayer)

		self.fillConnectionCombo()

	def setMode(self, selectionMode, showTablesMode, showUniqueInQuery):
		if selectionMode != None:
			if selectionMode == self.SingleSelection:
				self.dbTreeView.setSelectionMode(QAbstractItemView.SingleSelection)
			else:
				self.dbTreeView.setSelectionMode(QAbstractItemView.MultiSelection)

			self.dbTreeView.clearSelection()
			self.selectedTables = []

		if showUniqueInQuery != None:
			self.customQuery.showUniqueColumn(showUniqueInQuery)

		if showTablesMode != None:
			model = self.dbTreeView.model()
			model.setReadOnlyMode(True)
			model.showOnlyReadableTables(True)
			model.setGeometryFilter(showTablesMode)
			self.refreshDbTree()

	def setCurrentConnection(self, name):
		index = self.connectionsCombo.findText(name)
		if index < 0:
			return False

		self.connectionsCombo.setCurrentIndex(index)
		return True

	def setDatabase(self, db):
		self.db = db
		self.refreshDbTree()

	def runSqlLayer(self):
		from qgis.utils import plugins, updateAvailablePlugins

		updateAvailablePlugins()
		for plugin in plugins.keys():
			if plugin == "rt_sql_layer" and plugins.has_key(plugin):
				plugins[plugin].runWithSettings(self)
				return

		QMessageBox.critical(self, self.tr( "Plugin not found" ), self.tr( "Unable to find the RT Sql-Layer plugin. \nMake sure it's installed." ) )


	def fillConnectionCombo(self):
		self.connectionsCombo.clear()

		conns = []
		settings = QSettings()
		settings.beginGroup("/PostgreSQL/connections")
		keys = settings.childGroups()
		for key in keys:
			conns.append(key)
		settings.endGroup()
		
		self.connectionsCombo.addItems( conns )

		settings = QSettings()
		selected = unicode(settings.value("/PostgreSQL/connections/selected").toString())
		self.setCurrentConnection(selected)

	def connectToDatabase(self):
		# if there's open database already, get rid of it
		if self.db:
			self.dbDisconnect()

		if self.connectionsCombo.currentIndex() < 0:
			return

		selected = self.connectionsCombo.currentText()
		
		# get connection details from QSettings
		settings = QSettings()
		settings.beginGroup( u"/PostgreSQL/connections/" + selected )
		if not settings.contains("database"): # non-existent entry?
			QMessageBox.critical(self, "Error", "Unable to connect: there is no defined database connection \"%s\"." % selected)
			return
		
		get_value_str = lambda x: unicode(settings.value(x).toString())
		host, database, username, password = map(get_value_str, ["host", "database", "username", "password"])
		port = settings.value("port").toInt()[0]

		# qgis1.5 use 'savePassword' instead of 'save' setting
		if not ( settings.value("save").toBool() or settings.value("savePassword").toBool() ):
			(password, ok) = QInputDialog.getText(self, "Enter password", "Enter password for connection \"%s\":" % selected, QLineEdit.Password)
			if not ok: return
		settings.endGroup()
		
		# connect to DB
		try:
			self.db = postgis_utils.GeoDB(host=host, port=port, dbname=database, user=username, passwd=password)
		except postgis_utils.DbError, e:
			QMessageBox.critical(self, "Error", "Couldn't connect to database:\n"+e.msg)
			return
		
		# set as default in QSettings
		settings.setValue("/PostgreSQL/connections/selected", QVariant(selected))
		self.currentConnection = selected

		self.refreshDbTree()

	def dbDisconnect(self):	
		self.db = None
		self.currentConnection = None
		self.refreshDbTree()

	def refreshDbTree(self):
		model = self.dbTreeView.model()
		model.loadFromDb(self.db)
		if self.queries.has_key( self.currentConnection ):
			model.loadQueries( self.queries[self.currentConnection] )
		model.reset()
				
		# only expand when there are not too many tables
		if model.tree.tableCount < 10:
			self.dbTreeView.expandAll()

		self.selectedTables = []
		self.customQuery.setDatabase(self.db)


	def addQueryToList(self):
		if not self.queries.has_key( self.currentConnection ):
			self.queries[self.currentConnection] = []

		queryAndCols = self.customQuery.getQueryAndCols()
		if queryAndCols == None:
			return

		self.queries[self.currentConnection].append( queryAndCols )
		self.refreshDbTree()


	def currentTableChanged(self, index, indexOld):
		item = index.internalPointer()
		
		if isinstance(item, SchemaItem):
			item = None

		self.emit( SIGNAL("currentTableChanged(PyQt_PyObject)"), item )


	def selectionChanged(self, selection, selectionOld):
		for index in selectionOld.indexes():
			if index in selection:
				continue

			item = index.internalPointer()
			if self.selectedTables.count(item) > 0:
				self.selectedTables.remove(item)

		for index in selection.indexes():
			if index in selectionOld:
				continue

			item = index.internalPointer()
			if self.selectedTables.count(item) <= 0:
				self.selectedTables.append(item)

		self.emit( SIGNAL("tablesSelectionChanged()") )
