"""
/***************************************************************************
 Roll, a QGIS plugin for design and analysis of 3D seismic survey geometries
 Generated by Plugin Builder: http://g-sherman.github.io/Qgis-Plugin-Builder/

        begin                : 2022-10-09
        git sha              : $Format:%H$
        copyright            : (C) 2022 by Duijndam.Dev
        email                : bart.duijndam@ziggo.nl
 ***************************************************************************/

/***************************************************************************
 *                                                                         *
 *   This program is free software; you can redistribute it and/or modify  *
 *   it under the terms of the GNU General Public License as published by  *
 *   the Free Software Foundation; either version 2 of the License, or     *
 *   (at your option) any later version.                                   *
 *                                                                         *
 ***************************************************************************/

The following PyQt example shows the framework that was used as a reference, to build this AddIn
https://doc.qt.io/qtforpython-5/overviews/qtwidgets-mainwindows-application-example.html#application-example

To help Python development, I installed some VS Code plugins as recommended here:
https://www.youtube.com/watch?v=glXGae6TsY8

Note: To find out where libraries reside, use 'inspect':

>>>import inspect
>>>inspect.getfile(qgis.PyQt.QtCore)
'C:\\Program Files\\QGIS 3.28.1\\apps\\qgis\\python\\qgis\\PyQt\\QtCore.py'
"""
# Currently I am using PyLint to check for errors, which causes a few issues on its own.
# See: https://stackoverflow.com/questions/52123470/how-do-i-disable-pylint-unused-import-error-messages-in-vs-code
# See: https://gist.github.com/xen/6334976
# See: https://pylint.pycqa.org/en/latest/user_guide/messages/index.html
# See: http://pylint-messages.wikidot.com/all-codes for more codes. see also: https://manpages.org/pylint
# See: https://pylint.pycqa.org/en/latest/user_guide/messages/messages_overview.html#messages-overview for the official list
# See: https://gispofinland.medium.com/cooking-with-gispo-qgis-plugin-development-in-vs-code-19f95efb1977 IMPORTANT FOR A FULLY FUNCTIONING SETUP

# Note: Currently the only one language server supporting autocompletion for qgis and PyQt5 libraries is the Jedi language server
# Qgis and PyQt5 packages are using compiled python code and other language servers are having troubles parsing the API from those libraries (Actually this is for security reasons).
# In the following QGIS releases python stub files that describes the API are included in the qgis package so also much better Pylance language server can be then used
# If you have *.pyi files in C:/OSGeo4W64/apps/qgis-ltr/python/qgis go with the Pylance language server.

# This file is getting TOO BIG (> 3000 lines). Its main class has been (will further be) split into multiple documents.
# See: https://www.reddit.com/r/Python/comments/91wbhc/how_do_i_split_a_huge_class_over_multiple_files/?rdt=44096

# As of 3.32 scaling issues have popped up in QGIS. !
# See: https://github.com/qgis/QGIS/issues/53898
# Solution:
# 1) Right click 'qgis-bin.exe' in folder 'C:\Program Files\QGIS 3.36.3\bin'
# 2) Select the Compatibility tab
# 3) Select 'change high DPI settings'
# 4) Set the tickmark before 'Override high DPI ...'
# 5) Have scaling performed by 'Application'
# 6) In the same folder edit the file 'qgis-bin.env'
# 7) Add one line at the end:
# 8) QT_SCALE_FACTOR_ROUNDING_POLICY=Floor
# 9) Save the file in a different (user) folder as C:\Program Files is protected
# 10) Drag the edited file to the C:\Program Files\QGIS 3.36.3\bin folder
# 11) You'll be asked to confirm you want to overwrite the *.env file
# that solved my problems ! I use font size 9.0 and Icon size 24

# Extra toolbars have been added to be able to close the Display pane, use a toolbar instead.
# But in doing so, I don't want to duplicate all signals and slots !
# See: https://stackoverflow.com/questions/16703039/pyqt-can-a-qpushbutton-be-assigned-a-qaction
# See: https://stackoverflow.com/questions/4149117/how-can-i-implement-the-button-that-holds-an-action-qaction-and-can-connect-w
# See: https://stackoverflow.com/questions/38576380/difference-between-qpushbutton-and-qtoolbutton
# See: https://stackoverflow.com/questions/10368947/how-to-make-qmenu-item-checkable-pyqt4-python
# See: https://stackoverflow.com/questions/23388754/two-shortcuts-for-one-action
# See: https://stackoverflow.com/questions/53936403/two-shortcuts-for-one-button-in-pyqt
# See: https://doc.qt.io/qtforpython-5/PySide2/QtWidgets/QActionGroup.html#PySide2.QtWidgets.PySide2.QtWidgets.QActionGroup


import contextlib
import gc
import os
import os.path
import sys
import traceback
import typing
import winsound  # make a sound when an exception ocurs
from datetime import timedelta
from enum import Enum
from math import atan2, ceil, degrees
from timeit import default_timer as timer

# PyQtGraph related imports
import numpy as np  # Numpy functions needed for plot creation
import pyqtgraph as pg
from console import console
from numpy.compat import asstr
from qgis.PyQt import uic
from qgis.PyQt.QtCore import QDateTime, QEvent, QFile, QFileInfo, QIODevice, QItemSelection, QItemSelectionModel, QModelIndex, QPoint, QSettings, Qt, QTextStream, QThread
from qgis.PyQt.QtGui import QBrush, QColor, QFont, QIcon, QKeySequence, QTextCursor, QTextOption, QTransform
from qgis.PyQt.QtPrintSupport import QPrintDialog, QPrinter, QPrintPreviewDialog
from qgis.PyQt.QtWidgets import (
    QAction,
    QActionGroup,
    QApplication,
    QButtonGroup,
    QCheckBox,
    QDialogButtonBox,
    QDockWidget,
    QFileDialog,
    QFrame,
    QGraphicsEllipseItem,
    QGroupBox,
    QHBoxLayout,
    QHeaderView,
    QLabel,
    QMainWindow,
    QMessageBox,
    QPlainTextEdit,
    QProgressBar,
    QPushButton,
    QTabWidget,
    QToolButton,
    QVBoxLayout,
    QWidget,
)
from qgis.PyQt.QtXml import QDomDocument

from . import config  # used to pass initial settings

# from .event_lookup import event_lookup
from .find import Find
from .functions import aboutText, exampleSurveyXmlText, highDpiText, licenseText, rawcount
from .functions_numba import numbaAziInline, numbaAziX_line, numbaFilterSlice2D, numbaNdft_1D, numbaNdft_2D, numbaOffInline, numbaOffsetBin, numbaOffX_line, numbaSlice3D, numbaSliceStats, numbaSpiderBin
from .land_wizard import LandSurveyWizard
from .my_parameters import registerAllParameterTypes
from .qgis_interface import CreateQgisRasterLayer, ExportRasterLayerToQgis, exportPointLayerToQgis, exportSurveyOutlineToQgis, identifyQgisPointLayer, readQgisPointLayer
from .roll_binning import BinningType
from .roll_main_window_create_geom_tab import createGeomTab
from .roll_main_window_create_layout_tab import createLayoutTab
from .roll_main_window_create_pattern_tab import createPatternTab
from .roll_main_window_create_sps_tab import createSpsTab
from .roll_main_window_create_stack_response_tab import createStackResponseTab
from .roll_main_window_create_trace_table_tab import createTraceTableTab
from .roll_output import RollOutput
from .roll_survey import RollSurvey, SurveyType
from .settings import SettingsDialog, readSettings, writeSettings
from .sps_io_and_qc import (
    calcMaxXPStraces,
    calculateLineStakeTransform,
    deletePntDuplicates,
    deletePntOrphans,
    deleteRelDuplicates,
    deleteRelOrphans,
    fileExportAsR01,
    fileExportAsS01,
    fileExportAsX01,
    findRecOrphans,
    findSrcOrphans,
    getGeometry,
    markUniqueRPSrecords,
    markUniqueSPSrecords,
    markUniqueXPSrecords,
    pntType1,
    readRPSFiles,
    readSPSFiles,
    readXPSFiles,
    relType2,
)
from .worker_threads import BinFromGeometryWorker, BinningWorker, GeometryWorker
from .xml_code_editor import QCodeEditor, XMLHighlighter


class ImagType(Enum):
    NoIm = 0
    Fold = 1
    MinO = 2
    MaxO = 3


class MsgType(Enum):
    Info = 0
    Binning = 1
    Geometry = 2
    Debug = 3
    Error = 4
    Exception = 5


class Direction(Enum):
    NA = 0
    Up = 1
    Dn = 2
    Lt = 3
    Rt = 4


# This loads your .ui file so that PyQt can populate your plugin with the elements from Qt Designer
FORM_CLASS, _ = uic.loadUiType(os.path.join(os.path.dirname(__file__), 'roll_main_window_base.ui'))

# See: https://gist.github.com/mistic100/dcbffbd9e9c15271dd14
class QButtonGroupEx(QButtonGroup):
    def setCheckedId(self, id_) -> int:
        for button in self.buttons():
            if self.id(button) == id_:
                button.setChecked(True)
                return id_
        return None


# See: https://groups.google.com/g/pyqtgraph/c/V01QJKvrUio/m/iUBp5NePCQAJ
class LineROI(pg.LineSegmentROI):
    def __init__(self, *args, **kwargs):
        super().__init__(*args, **kwargs)

    def addHandle(self, *args, **kwargs):
        # Larger handle for improved visibility
        self.handleSize = 8
        super().addHandle(*args, **kwargs)

    def checkPointMove(self, handle, pos, modifiers):
        # needed to prevent 'eternal' range-jitter preventing the plot to complete
        self.getViewBox().disableAutoRange(axis='xy')
        return True

    def generateSvg(self, nodes):
        pass                                                                    # for the time being don't do anything; just to keep PyLint happy


class QHLine(QFrame):
    def __init__(self, parent=None):
        super().__init__(parent)
        self.setFrameShape(QFrame.HLine)
        self.setFrameShadow(QFrame.Sunken)

    def generateSvg(self, nodes):
        pass                                                                    # for the time being don't do anything; just to keep PyLint happy


def silentPrint(*_, **__):
    pass


class RollMainWindow(QMainWindow, FORM_CLASS):
    def __init__(self, parent=None):
        """Constructor."""
        super(RollMainWindow, self).__init__(parent)
        # Set up the user interface from Designer through FORM_CLASS.
        # After self.setupUi() you can access any designer object by doing self.<objectname>,
        # and you can use autoconnect slots - see http://qt-project.org/doc/qt-4.8/designer-using-a-ui-file.html
        # widgets-and-dialogs-with-auto-connect
        # See also: https://doc.qt.io/qt-6/designer-using-a-ui-file-python.html
        # See: https://docs.qgis.org/3.22/en/docs/documentation_guidelines/substitutions.html#toolbar-button-icons for QGIS Icons
        self.setupUi(self)

        # reset GUI when the plugin is restarted (not when restarted from minimized state on windows)
        self.killMe = False

        # GQIS interface
        self.iface = None                                                       # access to QGis interface

        # toolbar parameters
        self.debug = False                                                      # use debug settings
        self.XisY = True                                                        # equal x / y scaling
        self.rect = False                                                       # zoom using a rectangle
        self.glob = False                                                       # global coordinates
        self.gridX = True                                                       # use grid lines
        self.gridY = True                                                       # use grid lines
        self.antiA = [False for i in range(12)]                                 # anti-alias painting
        self.ruler = False                                                      # show a ruler to measure distances

        # exception handling
        # self.oldExceptHook = sys.excepthook                                     # make a copy before changing it
        # sys.excepthook = self.exceptionHook                                     # deal with uncaught exceptions using hook

        # print handling
        # self.oldPrint = builtins.print                                          # need to be able to get back to 'normal'

        # list with most recently used [mru] file actions
        self.recentFileActions = []
        self.recentFileList = []

        # workerTread parameters
        self.worker = None                                                      # 'moveToThread' object
        self.thread = None                                                      # corresponding worker thread
        self.startTime = None                                                   # thread start time

        # statusbar widgets
        self.posWidgetStatusbar = QLabel('(x, y): (0.00, 0.00)')                # mouse' position label, in bottom right corner
        self.progressLabel = QLabel('doing a lot of stuff in the background')   # label next to progressbar indicating background process
        self.progressBar = QProgressBar()                                       # progressbar in statusbar
        self.progressBar.setMaximumWidth(500)                                   # to avoid 'jitter' when the mouse moves and posWidgetStatusbar changes width
        height = self.posWidgetStatusbar.height()                               # needed to avoid statusbar 'growing' vertically by adding the progressbar
        self.progressBar.setMaximumHeight(height)                               # to avoid ugly appearance on statusbar

        # binning analysis
        self.output = RollOutput()                                              # contains result arrays and min/max values
        self.binAreaChanged = False                                             # set when binning area changes in property tree

        # display parameters in Layout tab
        self.imageType = 0                                                      # 1 = fold map
        self.layoutMax = 0.0                                                    # max value for image's colorbar (minimum is always 0)
        self.layoutImg = None                                                   # numpy array to be displayed; binOutput / minOffset / maxOffset / rmsOffset

        # analysis numpy arrays
        self.inlineStk = None                                                   # numpy array with inline Kr stack reponse
        self.x_lineStk = None                                                   # numpy array with x_line Kr stack reponse
        self.xyCellStk = None                                                   # numpy array with cell's KxKy stack response
        self.xyPatResp = None                                                   # numpy array with pattern's KxKy response

        # layout and analysis image-items
        self.layoutImItem = None                                                # pg ImageItems showing analysis result
        self.stkTrkImItem = None
        self.stkBinImItem = None
        self.stkCelImItem = None
        self.offAziImItem = None
        self.kxyPatImItem = None

        # corresponding color bars
        self.layoutColorBar = None                                              # colorBars, added to imageItem
        self.stkTrkColorBar = None
        self.stkBinColorBar = None
        self.stkCelColorBar = None
        self.offAziColorBar = None
        self.kxyPatColorBar = None

        # rps, sps, xps input arrays
        self.rpsImport = None                                                   # numpy array with list of RPS records
        self.spsImport = None                                                   # numpy array with list of SPS records
        self.xpsImport = None                                                   # numpy array with list of XPS records

        self.rpsLiveE = None                                                    # numpy array with list of live RPS coordinates
        self.rpsLiveN = None                                                    # numpy array with list of live RPS coordinates
        self.rpsDeadE = None                                                    # numpy array with list of dead RPS coordinates
        self.rpsDeadN = None                                                    # numpy array with list of dead RPS coordinates

        self.spsLiveE = None                                                    # numpy array with list of live SPS coordinates
        self.spsLiveN = None                                                    # numpy array with list of live SPS coordinates
        self.spsDeadE = None                                                    # numpy array with list of dead SPS coordinates
        self.spsDeadN = None                                                    # numpy array with list of dead SPS coordinates

        # rel, src, rel input arrays
        self.recGeom = None                                                     # numpy array with list of REC records
        self.srcGeom = None                                                     # numpy array with list of SRC records
        self.relGeom = None                                                     # numpy array with list of REL records

        self.recLiveE = None                                                    # numpy array with list of live REC coordinates
        self.recLiveN = None                                                    # numpy array with list of live REC coordinates
        self.recDeadE = None                                                    # numpy array with list of dead REC coordinates
        self.recDeadN = None                                                    # numpy array with list of dead REC coordinates

        self.srcLiveE = None                                                    # numpy array with list of live SRC coordinates
        self.srcLiveN = None                                                    # numpy array with list of live SRC coordinates
        self.srcDeadE = None                                                    # numpy array with list of dead SRC coordinates
        self.srcDeadN = None                                                    # numpy array with list of dead SRC coordinates

        # spider plot settings
        self.spiderPoint = QPoint(-1, -1)                                       # spider point 'out of scope'
        self.spiderSrcX = None                                                  # numpy array with list of SRC part of spider plot
        self.spiderSrcY = None                                                  # numpy array with list of SRC part of spider plot
        self.spiderRecX = None                                                  # numpy array with list of REC part of spider plot
        self.spiderRecY = None                                                  # numpy array with list of REC part of spider plot
        self.spiderText = None                                                  # text label describing spider bin, stake, fold
        self.actionSpider.setChecked(False)                                     # reset spider plot to 'off'

        # export layers to QGIS
        self.spsLayer = None                                                    # QGIS layer for sps point I/O
        self.rpsLayer = None                                                    # QGIS layer for rpr point I/O
        self.srcLayer = None                                                    # QGIS layer for src point I/O
        self.recLayer = None                                                    # QGIS layer for rec point I/O
        self.spsField = None                                                    # QGIS field for sps point selection I/O
        self.rpsField = None                                                    # QGIS field for rps point selection I/O
        self.srcField = None                                                    # QGIS field for src point selection I/O
        self.recField = None                                                    # QGIS field for rec point selection I/O

        # ruler settings
        self.lineROI = None                                                     # the ruler's dotted line
        self.roiLabels = None                                                   # the ruler's three labels
        self.rulerState = None                                                  # ruler's state, used to redisplay ruler at last used location

        # warning dialogs that can be hidden
        self.hideSpsCrsWarning = False                                          # warning message: sps crs should be identical to project crs

        # pattern information plotting parameters
        self.patternLayout = True                                               # True shows geometry (layout). False shows kxky response

        icon_path = ':/plugins/roll/icon.png'
        icon = QIcon(icon_path)
        self.setWindowIcon(icon)

        # See: https://gist.github.com/dgovil/d83e7ddc8f3fb4a28832ccc6f9c7f07b dealing with settings
        # See also : https://doc.qt.io/qtforpython-5/PySide2/QtCore/QSettings.html
        # QCoreApplication.setOrganizationName('Duijndam.Dev')
        # QCoreApplication.setApplicationName('Roll')
        # self.settings = QSettings()   ## doesn't work as expected with QCoreApplication.setXXX

        self.settings = QSettings(config.organization, config.application)
        self.fileName = ''
        self.workingDirectory = ''
        self.importDirectory = ''

        # # # # # # # # # # # # # # # # # # # # # # # # # # # # # # # # # # # # # # # # # # # # # # # # # # # # # # #

        # first docking pane, used to display geometry and analysis results
        self.dockDisplay = QDockWidget('Display pane')
        self.dockDisplay.setAllowedAreas(Qt.LeftDockWidgetArea | Qt.RightDockWidgetArea)

        # See: https://www.geeksforgeeks.org/pyqt5-qdockwidget-setting-style-sheet/ for styling
        self.dockDisplay.setStyleSheet('QDockWidget::title {background : lightblue;}')

        self.geometryChoice = QGroupBox('Geometry to display')                  # create display widget(s)
        self.analysisChoice = QGroupBox('Analysis to display')                  # create display widget(s)
        self.analysisToQgis = QGroupBox('Export to QGIS')                       # create display widget(s)

        self.geometryChoice.setMinimumWidth(140)
        self.analysisChoice.setMinimumWidth(140)
        self.analysisToQgis.setMinimumWidth(140)

        self.geometryChoice.setAlignment(Qt.AlignHCenter)
        self.analysisChoice.setAlignment(Qt.AlignHCenter)
        self.analysisToQgis.setAlignment(Qt.AlignHCenter)

        # display pane
        vbox0 = QVBoxLayout()
        self.displayLayout = QHBoxLayout()                                      # required layout

        self.displayLayout.addStretch()                                         # add some stretch to main center widget(s)
        self.displayLayout.addLayout(vbox0)
        self.displayLayout.addStretch()                                         # add some stretch to main center widget(s)

        vbox0.addStretch()                                                      # add some stretch to main center widget(s)
        vbox0.addWidget(self.geometryChoice)                                    # add main widget(s)
        vbox0.addStretch()                                                      # add some stretch to main center widget(s)
        vbox0.addWidget(self.analysisChoice)                                    # add main widget(s)
        vbox0.addStretch()                                                      # add some stretch to main center widget(s)
        vbox0.addWidget(self.analysisToQgis)                                    # add main widget(s)
        vbox0.addStretch()                                                      # add some stretch to main center widget(s)

        self.tbTemplat = QToolButton()
        self.tbRecList = QToolButton()
        self.tbSrcList = QToolButton()
        self.tbRpsList = QToolButton()
        self.tbSpsList = QToolButton()
        self.tbAllList = QToolButton()

        self.tbTemplat.setMinimumWidth(110)
        self.tbRecList.setMinimumWidth(110)
        self.tbSrcList.setMinimumWidth(110)
        self.tbRpsList.setMinimumWidth(110)
        self.tbSpsList.setMinimumWidth(110)
        self.tbAllList.setMinimumWidth(110)

        self.tbTemplat.setStyleSheet('QToolButton { selection-background-color: blue } QToolButton:checked { background-color: lightblue } QToolButton:pressed { background-color: red }')
        self.tbRecList.setStyleSheet('QToolButton { selection-background-color: blue } QToolButton:checked { background-color: lightblue } QToolButton:pressed { background-color: red }')
        self.tbSrcList.setStyleSheet('QToolButton { selection-background-color: blue } QToolButton:checked { background-color: lightblue } QToolButton:pressed { background-color: red }')
        self.tbTemplat.setStyleSheet('QToolButton { selection-background-color: blue } QToolButton:checked { background-color: lightblue } QToolButton:pressed { background-color: red }')
        self.tbRpsList.setStyleSheet('QToolButton { selection-background-color: blue } QToolButton:checked { background-color: lightblue } QToolButton:pressed { background-color: red }')
        self.tbSpsList.setStyleSheet('QToolButton { selection-background-color: blue } QToolButton:checked { background-color: lightblue } QToolButton:pressed { background-color: red }')
        self.tbAllList.setStyleSheet('QToolButton { selection-background-color: blue } QToolButton:checked { background-color: lightblue } QToolButton:pressed { background-color: red }')

        self.tbTemplat.setDefaultAction(self.actionTemplates)
        self.tbRecList.setDefaultAction(self.actionRecPoints)
        self.tbSrcList.setDefaultAction(self.actionSrcPoints)
        self.tbRpsList.setDefaultAction(self.actionRpsPoints)
        self.tbSpsList.setDefaultAction(self.actionSpsPoints)
        self.tbAllList.setDefaultAction(self.actionAllPoints)

        self.actionTemplates.setChecked(True)
        self.actionRecPoints.setEnabled(False)
        self.actionSrcPoints.setEnabled(False)
        self.actionRpsPoints.setEnabled(False)
        self.actionSpsPoints.setEnabled(False)
        self.actionAllPoints.setEnabled(False)
        self.actionAllPoints.setChecked(True)

        vbox1 = QVBoxLayout()
        vbox1.addWidget(self.tbTemplat)
        vbox1.addWidget(self.tbRecList)
        vbox1.addWidget(self.tbSrcList)
        vbox1.addWidget(self.tbRpsList)
        vbox1.addWidget(self.tbSpsList)
        vbox1.addWidget(QHLine())
        vbox1.addWidget(self.tbAllList)

        self.geometryChoice.setLayout(vbox1)

        self.tbNone = QToolButton()
        self.tbFold = QToolButton()
        self.tbMinO = QToolButton()
        self.tbMaxO = QToolButton()
        self.tbRmsO = QToolButton()

        self.tbNone.setMinimumWidth(110)
        self.tbFold.setMinimumWidth(110)
        self.tbMinO.setMinimumWidth(110)
        self.tbMaxO.setMinimumWidth(110)
        self.tbRmsO.setMinimumWidth(110)

        self.tbNone.setStyleSheet('QToolButton { selection-background-color: blue } QToolButton:checked { background-color: lightblue } QToolButton:pressed { background-color: red }')
        self.tbFold.setStyleSheet('QToolButton { selection-background-color: blue } QToolButton:checked { background-color: lightblue } QToolButton:pressed { background-color: red }')
        self.tbMinO.setStyleSheet('QToolButton { selection-background-color: blue } QToolButton:checked { background-color: lightblue } QToolButton:pressed { background-color: red }')
        self.tbMaxO.setStyleSheet('QToolButton { selection-background-color: blue } QToolButton:checked { background-color: lightblue } QToolButton:pressed { background-color: red }')
        self.tbRmsO.setStyleSheet('QToolButton { selection-background-color: blue } QToolButton:checked { background-color: lightblue } QToolButton:pressed { background-color: red }')

        self.actionNone.setChecked(True)                                        # action coupled to tbNone
        self.tbNone.setDefaultAction(self.actionNone)                           # coupling done here
        self.tbFold.setDefaultAction(self.actionFold)
        self.tbMinO.setDefaultAction(self.actionMinO)
        self.tbMaxO.setDefaultAction(self.actionMaxO)
        self.tbRmsO.setDefaultAction(self.actionRmsO)

        vbox2 = QVBoxLayout()
        vbox2.addWidget(self.tbNone)
        vbox2.addWidget(self.tbFold)
        vbox2.addWidget(self.tbMinO)
        vbox2.addWidget(self.tbMaxO)
        vbox2.addWidget(self.tbRmsO)

        vbox2.addWidget(QHLine())
        self.actionSpider.triggered.connect(self.handleSpiderPlot)
        self.actionSpider.setEnabled(False)

        self.tbSpider = QToolButton()
        self.tbSpider.setMinimumWidth(110)
        self.tbSpider.setStyleSheet('QToolButton { selection-background-color: blue } QToolButton:checked { background-color: lightblue } QToolButton:pressed { background-color: red }')
        self.tbSpider.setDefaultAction(self.actionSpider)
        vbox2.addWidget(self.tbSpider)

        self.btnSpiderLt = QToolButton()
        self.btnSpiderRt = QToolButton()
        self.btnSpiderUp = QToolButton()
        self.btnSpiderDn = QToolButton()

        self.btnSpiderLt.setDefaultAction(self.actionMoveLt)
        self.btnSpiderRt.setDefaultAction(self.actionMoveRt)
        self.btnSpiderUp.setDefaultAction(self.actionMoveUp)
        self.btnSpiderDn.setDefaultAction(self.actionMoveDn)

        # Note: to use a stylesheet on buttons (=actions) in a toolbar, you needt to use the toolbar's stylesheet and select individual actions to 'style'
        # See: https://stackoverflow.com/questions/32460193/how-to-change-qaction-background-color-using-stylesheets-css
        self.btnSpiderLt.setStyleSheet('QToolButton { selection-background-color: blue } QToolButton:checked { background-color: lightblue } QToolButton:pressed { background-color: red }')
        self.btnSpiderRt.setStyleSheet('QToolButton { selection-background-color: blue } QToolButton:checked { background-color: lightblue } QToolButton:pressed { background-color: red }')
        self.btnSpiderUp.setStyleSheet('QToolButton { selection-background-color: blue } QToolButton:checked { background-color: lightblue } QToolButton:pressed { background-color: red }')
        self.btnSpiderDn.setStyleSheet('QToolButton { selection-background-color: blue } QToolButton:checked { background-color: lightblue } QToolButton:pressed { background-color: red }')

        self.actionMoveLt.triggered.connect(self.spiderGoLt)
        self.actionMoveRt.triggered.connect(self.spiderGoRt)
        self.actionMoveUp.triggered.connect(self.spiderGoUp)
        self.actionMoveDn.triggered.connect(self.spiderGoDn)

        self.actionMoveLt.setShortcuts(['Alt+Left', 'Alt+Shift+Left', 'Alt+Ctrl+Left', 'Alt+Shift+Ctrl+Left'])
        self.actionMoveRt.setShortcuts(['Alt+Right', 'Alt+Shift+Right', 'Alt+Ctrl+Right', 'Alt+Shift+Ctrl+Right'])
        self.actionMoveUp.setShortcuts(['Alt+Up', 'Alt+Shift+Up', 'Alt+Ctrl+Up', 'Alt+Shift+Ctrl+Up'])
        self.actionMoveDn.setShortcuts(['Alt+Down', 'Alt+Shift+Down', 'Alt+Ctrl+Down', 'Alt+Shift+Ctrl+Down'])

        hbox1 = QHBoxLayout()
        hbox1.addStretch()
        hbox1.addWidget(self.btnSpiderLt)
        hbox1.addWidget(self.btnSpiderRt)
        hbox1.addWidget(self.btnSpiderUp)
        hbox1.addWidget(self.btnSpiderDn)
        hbox1.addStretch()
        vbox2.addLayout(hbox1)

        self.analysisChoice.setLayout(vbox2)

        self.analysisActionGroup = QActionGroup(self)
        self.analysisActionGroup.addAction(self.actionNone)
        self.analysisActionGroup.addAction(self.actionFold)
        self.analysisActionGroup.addAction(self.actionMinO)
        self.analysisActionGroup.addAction(self.actionMaxO)
        self.analysisActionGroup.addAction(self.actionRmsO)
        self.actionNone.setChecked(True)

        self.actionNone.triggered.connect(self.onActionNoneTriggered)
        self.actionFold.triggered.connect(self.onActionFoldTriggered)
        self.actionMinO.triggered.connect(self.onActionMinOTriggered)
        self.actionMaxO.triggered.connect(self.onActionMaxOTriggered)
        self.actionRmsO.triggered.connect(self.onActionRmsOTriggered)

        self.btnBinToQGIS = QPushButton('Fold Map')
        self.btnMinToQGIS = QPushButton('Min Offset')
        self.btnMaxToQGIS = QPushButton('Max Offset')
        self.btnRmsToQGIS = QPushButton('Rms Offset')

        self.btnBinToQGIS.setMinimumWidth(110)
        self.btnMinToQGIS.setMinimumWidth(110)
        self.btnMaxToQGIS.setMinimumWidth(110)
        self.btnRmsToQGIS.setMinimumWidth(110)

        self.btnBinToQGIS.setStyleSheet('background-color:lightgoldenrodyellow; font-weight:bold;')
        self.btnMinToQGIS.setStyleSheet('background-color:lightgoldenrodyellow; font-weight:bold;')
        self.btnMaxToQGIS.setStyleSheet('background-color:lightgoldenrodyellow; font-weight:bold;')
        self.btnRmsToQGIS.setStyleSheet('background-color:lightgoldenrodyellow; font-weight:bold;')

        vbox3 = QVBoxLayout()
        vbox3.addWidget(self.btnBinToQGIS)
        vbox3.addWidget(self.btnMinToQGIS)
        vbox3.addWidget(self.btnMaxToQGIS)
        vbox3.addWidget(self.btnRmsToQGIS)
        self.analysisToQgis.setLayout(vbox3)

        self.displayWidget = QWidget()                                          # placeholder widget to generate a layout
        self.displayWidget.setLayout(self.displayLayout)                        # add layout to widget
        self.dockDisplay.setWidget(self.displayWidget)                          # set widget as main widget in docking panel

        self.addDockWidget(Qt.LeftDockWidgetArea, self.dockDisplay)             # add docking panel to main window
        self.dockDisplay.toggleViewAction().setShortcut(QKeySequence('Ctrl+Alt+d'))
        self.menu_View.addAction(self.dockDisplay.toggleViewAction())           # show/hide as requested

        # # # # # # # # # # # # # # # # # # # # # # # # # # # # # # # # # # # # # # # # # # # # # # # # # # # # # # #

        # second docking pane
        self.dockLogging = QDockWidget('Logging pane')
        self.dockLogging.setAllowedAreas(Qt.BottomDockWidgetArea | Qt.TopDockWidgetArea)

        # See: https://www.geeksforgeeks.org/pyqt5-qdockwidget-setting-style-sheet/ for styling
        self.dockLogging.setStyleSheet('QDockWidget::title {background : lightblue;}')

        # See: https://wiki.python.org/moin/PyQt/Handling%20context%20menus to change context menu, very informative
        # See: https://stackoverflow.com/questions/32053072/pyqt-extend-existing-contextual-menu-when-editing-qstandarditems-text-in-a-qtr
        # See: https://stackoverflow.com/questions/8676597/customising-location-sensitive-context-menu-in-qtextedit
        # See: https://forum.qt.io/topic/60790/add-actions-from-qtextedit-to-edit-menu/10

        self.logEdit = QPlainTextEdit()
        self.logEdit.clear()                                                    # Should not be necessary between sessions
        self.logEdit.setUndoRedoEnabled(False)                                  # Don't allow undo on the logging pane
        # self.logEdit.setReadOnly(True)                                        # if we set this 'True' the context menu no longer allows 'delete', just 'select all' and 'copy'
        self.logEdit.setLineWrapMode(QPlainTextEdit.NoWrap)
        self.logEdit.setWordWrapMode(QTextOption.NoWrap)
        self.logEdit.setStyleSheet('QPlainTextEdit { font-family: Courier New; font-weight: bold; font-size: 12px;}')
        # self.logEdit.setFont(QFont("Ubuntu Mono", 8, QFont.Normal))           # Does not line up columns properly !

        self.dockLogging.setWidget(self.logEdit)
        self.addDockWidget(Qt.BottomDockWidgetArea, self.dockLogging)
        self.dockLogging.toggleViewAction().setShortcut(QKeySequence('Ctrl+Alt+l'))
        self.menu_View.addAction(self.dockLogging.toggleViewAction())

        # # # # # # # # # # # # # # # # # # # # # # # # # # # # # # # # # # # # # # # # # # # # # # # # # # # # # # #

        self.survey = RollSurvey()                                              # (re)set the survey object; needed in property pane

        self.mainTabWidget = QTabWidget()
        self.mainTabWidget.setTabPosition(QTabWidget.South)
        self.mainTabWidget.setTabShape(QTabWidget.Rounded)
        self.mainTabWidget.setDocumentMode(False)                               # has only effect on OSX ?!
        self.mainTabWidget.resize(300, 200)

        self.analysisTabWidget = QTabWidget()
        self.analysisTabWidget.setTabPosition(QTabWidget.South)
        self.analysisTabWidget.setTabShape(QTabWidget.Rounded)
        self.analysisTabWidget.setDocumentMode(False)                           # has only effect on OSX ?!
        self.analysisTabWidget.resize(300, 200)

        # See: https://stackoverflow.com/questions/69152935/adding-the-same-object-to-a-qtabwidget
        # See: pyqtgraph/examples/RemoteSpeedTest.py to keep gui responsive when updating a plot (uses multiprocessing)

        self.plotTitles = [
            'New survey',
            'Offsets for inline direction',
            'Offsets for x-line direction',
            'Azimuth for inline direction',
            'Azimuth for x-line direction',
            'Stack response for inline direction',
            'Stack response for x-line direction',
            'Kx-Ky single bin stack response',
            '|Offset| distribution in binning area',
            'Offset/azimuth distribution in binning area',
            'Pattern information',
        ]

        # these plotting widgets have "installEventFilter()" applied to catch the window 'Show' event in "eventFilter()"
        # this makes it possible to reroute commands and status from the plotting toolbar buttons to the active plot
        self.offTrkWidget = self.createPlotWidget(self.plotTitles[1], 'inline', 'offset', 'm', 'm')                         # False -> no fixed aspect ratio
        self.offBinWidget = self.createPlotWidget(self.plotTitles[2], 'x-line', 'offset', 'm', 'm')
        self.aziTrkWidget = self.createPlotWidget(self.plotTitles[3], 'inline', 'angle of incidence', 'm', 'deg', False)
        self.aziBinWidget = self.createPlotWidget(self.plotTitles[4], 'x-line', 'angle of incidence', 'm', 'deg', False)    # no fixed aspect ratio
        self.stkTrkWidget = self.createPlotWidget(self.plotTitles[5], 'inline', '|Kr|', 'm', ' 1/km', False)
        self.stkBinWidget = self.createPlotWidget(self.plotTitles[6], 'x-line', '|Kr|', 'm', ' 1/km', False)
        self.stkCelWidget = self.createPlotWidget(self.plotTitles[7], 'Kx', 'Ky', '1/km', '1/km')
        self.offsetWidget = self.createPlotWidget(self.plotTitles[8], '|offset|', 'frequency', 'm', ' #', False)
        self.offAziWidget = self.createPlotWidget(self.plotTitles[9], 'azimuth', '|offset|', 'deg', 'm', False)
        self.arraysWidget = self.createPlotWidget(self.plotTitles[10], 'inline', 'x-line', 'm', 'm')

        # Create the various views (tabs) on the data
        # Use QCodeEditor with a XmlHighlighter instead of a 'plain' QPlainTextEdit
        # See: https://github.com/luchko/QCodeEditor/blob/master/QCodeEditor.py

        self.textEdit = QCodeEditor(SyntaxHighlighter=XMLHighlighter)           # only one widget on Xml-tab; add directly
        self.textEdit.document().setModified(False)
        self.textEdit.installEventFilter(self)                                  # catch the 'Show' event to connect to toolbar buttons

        # The following tabs have multiple widgets per page, start by giving them a simple QWidget
        self.tabPatterns = QWidget()
        self.tabGeom = QWidget()
        self.tabSps = QWidget()
        self.tabTraces = QWidget()
        self.tabKxKyStack = QWidget()

        self.tabGeom.installEventFilter(self)                                   # catch the 'Show' event to connect to toolbar buttons
        self.tabSps.installEventFilter(self)                                    # catch the 'Show' event to connect to toolbar buttons
        self.tabTraces.installEventFilter(self)                                 # catch the 'Show' event to connect to toolbar buttons

        # The following functions have been removed from this file's class definition, to reduce the size of 'roll_main_window.py'
        # They now reside in separate source files. Therefore self.createLayoutTab() is now called as createLayoutTab(self) instead.
        createLayoutTab(self)
        createPatternTab(self)
        createGeomTab(self)
        createSpsTab(self)
        createTraceTableTab(self)
        createStackResponseTab(self)

        # Add tabs to main tab widget
        self.mainTabWidget.addTab(self.layoutWidget, 'Layout')
        self.mainTabWidget.addTab(self.tabPatterns, 'Patterns')
        self.mainTabWidget.addTab(self.textEdit, 'Xml')
        self.mainTabWidget.addTab(self.tabGeom, 'Geometry')
        self.mainTabWidget.addTab(self.tabSps, 'SPS import')
        self.mainTabWidget.addTab(self.analysisTabWidget, 'Analysis')
        self.mainTabWidget.currentChanged.connect(self.onMainTabChange)         # active tab changed!

        # Add tabs to analysis tab widget
        self.analysisTabWidget.addTab(self.tabTraces, 'Trace table')
        self.analysisTabWidget.addTab(self.offTrkWidget, 'Offset Inline')
        self.analysisTabWidget.addTab(self.offBinWidget, 'Offset X-line')
        self.analysisTabWidget.addTab(self.aziTrkWidget, 'Azi Inline')
        self.analysisTabWidget.addTab(self.aziBinWidget, 'Azi X-line')
        self.analysisTabWidget.addTab(self.stkTrkWidget, 'Stack Inline')
        self.analysisTabWidget.addTab(self.stkBinWidget, 'Stack X-line')
        self.analysisTabWidget.addTab(self.tabKxKyStack, 'Kx-Ky Stack')
        self.analysisTabWidget.addTab(self.offsetWidget, '|O| Histogram')
        self.analysisTabWidget.addTab(self.offAziWidget, 'O/A Histogram')
        # self.arraysWidget is embedded in the layout of the 'pattern' tab
        # self.analysisTabWidget.addTab(self.stkCelWidget, 'Kx-Ky Stack')
        # self.analysisTabWidget.currentChanged.connect(self.onAnalysisTabChange)   # active tab changed!

        self.setCurrentFileName()

        # connect actions
        self.textEdit.document().modificationChanged.connect(self.setWindowModified)    # forward signal to myself, and make some changes
        self.setWindowModified(self.textEdit.document().isModified())                   # update window status based on document status
        self.textEdit.cursorPositionChanged.connect(self.cursorPositionChanged)         # to show cursor position in statusbar

        self.layoutWidget.scene().sigMouseMoved.connect(self.MouseMovedInPlot)
        self.layoutWidget.getViewBox().sigRangeChangedManually.connect(self.mouseBeingDragged)  # essential to find plotting state for LOD plotting
        self.layoutWidget.plotItem.sigRangeChanged.connect(self.layoutRangeChanged)     # to handle changes in tickmarks when zooming

        self.actionDebug.setCheckable(True)
        self.actionDebug.setChecked(self.debug)
        self.actionDebug.setStatusTip('Show debug information in QGIS Python console')
        self.actionDebug.triggered.connect(self.viewDebug)

        # the following actions are related to the plotWidget
        self.actionZoomAll.triggered.connect(self.layoutWidget.autoRange)
        self.actionZoomRect.setCheckable(True)
        self.actionZoomRect.setChecked(self.rect)
        self.actionZoomRect.triggered.connect(self.plotZoomRect)

        self.actionAspectRatio.setCheckable(True)
        self.actionAspectRatio.setChecked(self.XisY)
        self.actionAspectRatio.triggered.connect(self.plotAspectRatio)

        self.actionAntiAlias.setCheckable(True)
        self.actionAntiAlias.setChecked(self.antiA[0])
        self.actionAntiAlias.triggered.connect(self.plotAntiAlias)

        self.actionPlotGridX.setCheckable(True)
        self.actionPlotGridX.setChecked(self.gridX)
        self.actionPlotGridX.triggered.connect(self.plotGridX)

        self.actionPlotGridY.setCheckable(True)
        self.actionPlotGridY.setChecked(self.gridY)
        self.actionPlotGridY.triggered.connect(self.plotGridY)

        self.actionProjected.setCheckable(True)
        self.actionProjected.setChecked(self.glob)
        self.actionProjected.triggered.connect(self.plotProjected)

        self.actionRuler.setCheckable(True)
        self.actionRuler.setChecked(self.ruler)
        self.actionRuler.triggered.connect(self.showRuler)

        # actions related to the file menu
        for i in range(config.maxRecentFiles):
            self.recentFileActions.append(QAction(self, visible=False, triggered=self.fileOpenRecent))
            self.menuOpenRecent.addAction(self.recentFileActions[i])

        self.actionNew.triggered.connect(self.newFile)
        self.actionNewLandSurvey.triggered.connect(self.fileNewLandSurvey)
        self.actionNewMarineSurvey.triggered.connect(self.fileNewMarineSurvey)
        self.actionOpen.triggered.connect(self.fileOpen)
        self.actionImportSPS.triggered.connect(self.fileImportSPS)
        self.actionPrint.triggered.connect(self.filePrint)
        self.actionSave.triggered.connect(self.fileSave)
        self.actionSaveAs.triggered.connect(self.fileSaveAs)
        self.actionSettings.triggered.connect(self.fileSettings)

        self.textEdit.document().modificationChanged.connect(self.actionSave.setEnabled)

        # actions related to file -> export
        self.actionExportFoldMap.triggered.connect(self.fileExportFoldMap)
        self.actionExportMinOffsets.triggered.connect(self.fileExportMinOffsets)
        self.actionExportMaxOffsets.triggered.connect(self.fileExportMaxOffsets)

        self.actionExportAnaAsCsv.triggered.connect(self.fileExportAnaAsCsv)

        self.actionExportRecAsCsv.triggered.connect(self.fileExportRecAsCsv)
        self.actionExportSrcAsCsv.triggered.connect(self.fileExportSrcAsCsv)
        self.actionExportRelAsCsv.triggered.connect(self.fileExportRelAsCsv)
        self.actionExportRecAsR01.triggered.connect(self.fileExportRecAsR01)
        self.actionExportSrcAsS01.triggered.connect(self.fileExportSrcAsS01)
        self.actionExportRelAsX01.triggered.connect(self.fileExportRelAsX01)

        self.actionExportRpsAsCsv.triggered.connect(self.fileExportRpsAsCsv)
        self.actionExportSpsAsCsv.triggered.connect(self.fileExportSpsAsCsv)
        self.actionExportXpsAsCsv.triggered.connect(self.fileExportXpsAsCsv)
        self.actionExportRpsAsR01.triggered.connect(self.fileExportRpsAsR01)
        self.actionExportSpsAsS01.triggered.connect(self.fileExportSpsAsS01)
        self.actionExportXpsAsX01.triggered.connect(self.fileExportXpsAsX01)

        self.actionExportFoldMapToQGIS.triggered.connect(self.exportBinToQGIS)
        self.actionExportMinOffsetsToQGIS.triggered.connect(self.exportMinToQGIS)
        self.actionExportMaxOffsetsToQGIS.triggered.connect(self.exportMaxToQGIS)
        self.actionExportRmsOffsetsToQGIS.triggered.connect(self.exportRmsToQGIS)

        self.actionQuit.triggered.connect(self.close)                           # closes the window and arrives at CloseEvent()

        # actions related to the edit menu
        # undo and redo are solely associated with the main xml textEdit
        self.actionUndo.triggered.connect(self.textEdit.undo)
        self.actionRedo.triggered.connect(self.textEdit.redo)
        self.textEdit.document().undoAvailable.connect(self.actionUndo.setEnabled)
        self.textEdit.document().redoAvailable.connect(self.actionRedo.setEnabled)

        # copy, cut, paste and select-all must be managed by all active widgets
        # See: https://stackoverflow.com/questions/40041131/pyqt-global-copy-paste-actions-for-custom-widgets
        self.actionCut.triggered.connect(self.cut)
        self.actionCopy.triggered.connect(self.copy)
        self.actionFind.triggered.connect(self.find)
        self.actionPaste.triggered.connect(self.paste)
        self.actionSelectAll.triggered.connect(self.selectAll)

        # the following setEnabled items need to be re-wired, they are still connected to the textEdit
        self.textEdit.copyAvailable.connect(self.actionCut.setEnabled)
        self.textEdit.copyAvailable.connect(self.actionCopy.setEnabled)

        # actions related to the view menu
        self.actionRefresh.triggered.connect(self.UpdateAllViews)
        self.actionAbout.triggered.connect(self.OnAbout)
        self.actionLicense.triggered.connect(self.OnLicense)
        self.actionHighDpi.triggered.connect(self.OnHighDpi)

        # actions related to the processing menu
        self.actionBasicBinFromTemplates.triggered.connect(self.basicBinFromTemplates)
        self.actionFullBinFromTemplates.triggered.connect(self.fullBinFromTemplates)
        self.actionBasicBinFromGeometry.triggered.connect(self.basicBinFromGeometry)
        self.actionFullBinFromGeometry.triggered.connect(self.fullBinFromGeometry)
        self.actionBasicBinFromSps.triggered.connect(self.basicBinFromSps)
        self.actionFullBinFromSps.triggered.connect(self.fullBinFromSps)
        self.actionGeometryFromTemplates.triggered.connect(self.createGeometryFromTemplates)

        self.actionStopThread.triggered.connect(self.stopWorkerThread)
        self.enableProcessingMenuItems()                                        # enables processing menu items except 'stop processing thread'

        # actions related to geometry items to be displayed
        self.actionTemplates.triggered.connect(self.plotLayout)
        self.actionRecPoints.triggered.connect(self.plotLayout)
        self.actionSrcPoints.triggered.connect(self.plotLayout)
        self.actionRpsPoints.triggered.connect(self.plotLayout)
        self.actionSpsPoints.triggered.connect(self.plotLayout)
        self.actionAllPoints.triggered.connect(self.plotLayout)

        # enable/disable various actions
        self.actionClose.setEnabled(False)
        self.actionSave.setEnabled(self.textEdit.document().isModified())
        # self.actionSaveAs.setEnabled((self.textEdit.document().blockCount() > 1))        # need at least one line of text to save the document
        self.actionSaveAs.setEnabled((True))                                    # need at least one line of text to save the document

        self.actionUndo.setEnabled(self.textEdit.document().isUndoAvailable())
        self.actionRedo.setEnabled(self.textEdit.document().isRedoAvailable())
        self.actionCut.setEnabled(False)
        self.actionCopy.setEnabled(False)
        self.actionPaste.setEnabled(self.clipboardHasText())

        self.updateMenuStatus(True)                                             # keep menu status in sync with program's state

        # make the main tab widget the central widget
        self.setCentralWidget(self.mainTabWidget)

        self.posWidgetStatusbar = QLabel('(x, y): (0.00, 0.00)')
        self.statusbar.addPermanentWidget(self.posWidgetStatusbar, stretch=0)   # widget in bottomright corner of statusbar

        self.parseText(exampleSurveyXmlText())
        self.textEdit.setPlainText(exampleSurveyXmlText())
        self.textEdit.moveCursor(QTextCursor.Start)

        # # # # # # # # # # # # # # # # # # # # # # # # # # # # # # # # # # # # # # # # # # # # # # # # # # # # # # #

        # third docking pane, used to display survey properties
        # defined late, as it needs access the loaded survey object
        self.dockProperty = QDockWidget('Property pane')
        self.dockProperty.setAllowedAreas(Qt.LeftDockWidgetArea | Qt.RightDockWidgetArea)
        self.dockProperty.setStyleSheet('QDockWidget::title {background : lightblue;}')

        # setup the ParameterTree object
        self.paramTree = pg.parametertree.ParameterTree(showHeader=True)        # define parameter tree widget
        self.paramTree.header().setSectionResizeMode(QHeaderView.ResizeMode.Interactive)
        self.paramTree.header().resizeSection(0, 280)
        self.registerParameters()
        self.resetSurveyProperties()                                            # get the parameters into the parameter tree

        self.propertyWidget = QWidget()                                         # placeholder widget to generate a layout
        self.propertyLayout = QVBoxLayout()                                     # required vertical layout

        buttons = QDialogButtonBox.Ok | QDialogButtonBox.Cancel | QDialogButtonBox.Apply
        self.propertyButtonBox = QDialogButtonBox(buttons)                      # define 3 buttons to handle property changes

        # connect 3 buttons (signals) to their event handlers (slots)
        self.propertyButtonBox.accepted.connect(self.applyPropertyChangesAndHide)
        self.propertyButtonBox.rejected.connect(self.resetSurveyProperties)
        self.propertyButtonBox.button(QDialogButtonBox.Apply).clicked.connect(self.applyPropertyChanges)

        self.propertyLayout.addWidget(self.paramTree)                           # add parameter tree to layout
        self.propertyLayout.addStretch()                                        # add some stretch towards 3 buttons
        self.propertyLayout.addWidget(self.propertyButtonBox)                   # add 3 buttons

        self.propertyWidget.setLayout(self.propertyLayout)                      # add layout to widget
        self.dockProperty.setWidget(self.propertyWidget)

        self.addDockWidget(Qt.RightDockWidgetArea, self.dockProperty)           # add docking panel to main window

        self.dockProperty.toggleViewAction().setShortcut(QKeySequence('Ctrl+Alt+p'))
        self.menu_View.addAction(self.dockProperty.toggleViewAction())          # show/hide as requested

        self.menu_View.addSeparator()
        self.menu_View.addAction(self.fileBar.toggleViewAction())
        self.menu_View.addAction(self.editBar.toggleViewAction())
        self.menu_View.addAction(self.graphBar.toggleViewAction())
        self.menu_View.addAction(self.moveBar.toggleViewAction())

        # # # # # # # # # # # # # # # # # # # # # # # # # # # # # # # # # # # # # # # # # # # # # # # # # # # # # # #

        self.plotLayout()

        readSettings(self)
        self.updateRecentFileActions()                                          # update the MRU file menu actions, with info from readSettings()

        self.appendLogMessage('Plugin : Started')
        self.statusbar.showMessage('Ready', 3000)

    # deal with pattern selection for display & kxky plotting
    def onPattern1IndexChanged(self):
        self.plotPatterns()

    def onPattern2IndexChanged(self):
        self.plotPatterns()

    def onActionPatternLayoutTriggered(self):
        self.patternLayout = True
        self.plotPatterns()

    def onActionPattern_kx_kyTriggered(self):
        self.patternLayout = False
        self.plotPatterns()

    # deal with pattern selection for bin stack response
    def onStackPatternIndexChanged(self):
        nX = self.spiderPoint.x()                                               # get x, y indices into bin array
        nY = self.spiderPoint.y()

        if self.spiderPoint.x() < 0:
            return

        if self.spiderPoint.y() < 0:
            return

        if self.survey.binTransform is None:
            return

        invBinTransform, _ = self.survey.binTransform.inverted()                # need to go from bin nr's to cmp(x, y)
        cmpX, cmpY = invBinTransform.map(nX, nY)                                # get local coordinates from line and point indices
        stkX, stkY = self.survey.st2Transform.map(cmpX, cmpY)                   # get the corresponding bin and stake numbers

        self.plotStkCel(nX, nY, stkX, stkY)

    def eventFilter(self, source, event):
        if event.type() == QEvent.Show:                                             # do 'cheap' test first
            if isinstance(source, pg.PlotWidget):                                   # do 'expensive' test next

                with contextlib.suppress(RuntimeError):                             # rewire zoomAll button
                    self.actionZoomAll.triggered.disconnect()
                self.actionZoomAll.triggered.connect(source.autoRange)

                plotIndex = self.getVisiblePlotIndex(source)                        # update toolbar status
                if plotIndex is not None:
                    self.actionZoomAll.setEnabled(True)                             # useful for all plots
                    self.actionZoomRect.setEnabled(True)                            # useful for all plots
                    self.actionAspectRatio.setEnabled(True)                         # useful for all plots
                    self.actionAntiAlias.setEnabled(True)                           # useful for plots only
                    self.actionRuler.setEnabled(plotIndex == 0)                     # useful for 1st plot only
                    self.actionProjected.setEnabled(plotIndex == 0)                 # useful for 1st plot only

                    self.actionAntiAlias.setChecked(self.antiA[plotIndex])          # useful for all plots

                    plotItem = source.getPlotItem()
                    self.gridX = plotItem.saveState()['xGridCheck']                 # update x-gridline status
                    self.actionPlotGridX.setChecked(self.gridX)

                    self.gridY = plotItem.saveState()['yGridCheck']                 # update y-gridline status
                    self.actionPlotGridY.setChecked(self.gridY)

                    self.XisY = plotItem.saveState()['view']['aspectLocked']        # update XisY status
                    self.actionAspectRatio.setChecked(self.XisY)

                    viewBox = plotItem.getViewBox()
                    self.rect = viewBox.getState()['mouseMode'] == pg.ViewBox.RectMode  # update rect status
                    self.actionZoomRect.setChecked(self.rect)
                    self.updateVisiblePlotWidget(plotIndex)
                    return True
            else:                                                                   # QEvent.Show; but for different widgets
                self.actionZoomAll.setEnabled(False)                                # useful for plots only
                self.actionZoomRect.setEnabled(False)                               # useful for plots only
                self.actionAspectRatio.setEnabled(False)                            # useful for plots only
                self.actionAntiAlias.setEnabled(False)                              # useful for plots only
                self.actionRuler.setEnabled(False)                                  # useful for 1st plot only
                self.actionProjected.setEnabled(False)                              # useful for 1st plot only
                return True

        return super().eventFilter(source, event)

    def applyPropertyChangesAndHide(self):
        self.applyPropertyChanges()
        self.dockProperty.hide()

    def registerParameters(self):
        registerAllParameterTypes()

    def resetSurveyProperties(self):
        self.paramTree.clear()

        # set the survey object in the property pane using current survey properties
        copy = self.survey.deepcopy()

        # first (globally) define the patterns to choose from
        config.patternList = ['<no pattern>']
        for p in copy.patternList:
            config.patternList.append(p.name)

        # first copy the crs for global access (need to fix this later)
        config.surveyCrs = copy.crs

        # brush color for main parameter categories
        brush = '#add8e6'

        surveyParams = [
            dict(brush=brush, name='Survey configuration', type='myConfiguration', value=copy),
            dict(brush=brush, name='Survey analysis', type='myAnalysis', value=copy),
            dict(brush=brush, name='Survey reflectors', type='myReflectors', value=copy),
            dict(brush=brush, name='Survey grid', type='myGrid', value=copy.grid),
            dict(brush=brush, name='Block list', type='myBlockList', value=copy.blockList),
            dict(brush=brush, name='Pattern list', type='myPatternList', value=copy.patternList),
        ]

        # surveyParams = dict(name='Survey configuration',type='mySurvey', value=copy, brush='#add8e6')
        # self.parameters = pg.parametertree.Parameter.create(name='Survey configuration', type='mySurvey', value=copy, brush='#add8e6')

        self.parameters = pg.parametertree.Parameter.create(name='Survey Properties', type='group', children=surveyParams)
        self.parameters.sigTreeStateChanged.connect(self.propertyTreeStateChanged)
        self.paramTree.setParameters(self.parameters, showTop=False)

        # Make sure we get a notification, when the binning area or the survey grid has changed, to ditch the analysis files
        self.anaChild = self.parameters.child('Survey analysis')
        self.binChild = self.anaChild.child('Binning area')
        self.binChild.sigTreeStateChanged.connect(self.binningSettingsHaveChanged)

        self.grdChild = self.parameters.child('Survey grid')
        self.grdChild.sigTreeStateChanged.connect(self.binningSettingsHaveChanged)

        # deal with a bug, not showing tooltip information in the list of parameterItems
        for item in self.paramTree.listAllItems():                              # Bug. See: https://github.com/pyqtgraph/pyqtgraph/issues/2744
            p = item.param                                                      # get parameter belonging to parameterItem
            p.setToDefault()                                                    # set all parameters to their default value
            if hasattr(item, 'updateDefaultBtn'):                               # note: not all parameterItems have this method
                item.updateDefaultBtn()                                         # reset the default-buttons to their grey value
            if 'tip' in p.opts:                                                 # this solves the above mentioned bug
                item.setToolTip(0, p.opts['tip'])                               # the widgets now get their tooltips

        # for pattern response display and kxky response
        self.pattern1.clear()
        self.pattern1.addItem('<no pattern>')                                   # setup first pattern list in pattern tab
        for item in self.survey.patternList:
            self.pattern1.addItem(item.name)

        self.pattern2.clear()
        self.pattern2.addItem('<no pattern>')                                   # setup second pattern list in pattern tab
        for item in self.survey.patternList:
            self.pattern2.addItem(item.name)

        listSize = len(self.survey.patternList)                                 # show the first two paterns (if available)
        self.pattern1.setCurrentIndex(min(listSize, 1))                         # select first pattern in list
        self.pattern2.setCurrentIndex(min(listSize, 2))                         # select first pattern in list

        # for convolution of stack response with pattern response
        self.pattern3.clear()
        self.pattern3.addItem('<no pattern>')                                   # setup first pattern list in pattern tab
        for item in self.survey.patternList:
            self.pattern3.addItem(item.name)

        self.pattern4.clear()
        self.pattern4.addItem('<no pattern>')                                   # setup second pattern list in pattern tab
        for item in self.survey.patternList:
            self.pattern4.addItem(item.name)

        self.pattern3.setCurrentIndex(min(listSize, 1))                         # select first pattern in list
        self.pattern4.setCurrentIndex(min(listSize, 2))                         # select first pattern in list

    def applyPropertyChanges(self):
        # build new survey object from scratch, and start adding to it
        copy = RollSurvey()

        CFG = self.parameters.child('Survey configuration')
        # copy.crs = CFG.child('Survey CRS').value()
        # copy.type = SurveyType[CFG.child('Survey type').value()]
        # copy.name = CFG.child('Survey name').value()

        copy.crs, surType, copy.name = CFG.value()                              # get tuple of data from parameter
        copy.type = SurveyType[surType]                                         # SurveyType is an enum
        config.surveyCrs = copy.crs                                             # needed for global access to crs

        ANA = self.parameters.child('Survey analysis')
        copy.output.rctOutput, copy.angles, copy.binning, copy.offset, copy.unique = ANA.value()

        REF = self.parameters.child('Survey reflectors')
        copy.globalPlane, copy.globalSphere = REF.value()

        GRD = self.parameters.child('Survey grid')
        copy.grid = GRD.value()

        BLK = self.parameters.child('Block list')
        copy.blockList = BLK.value()

        PAT = self.parameters.child('Pattern list')
        copy.patternList = PAT.value()

        # first check survey integrity before committing to it.
        if copy.checkIntegrity() is False:
            return

        self.survey = copy.deepcopy()                                           # start using the updated survey object

        # update the survey object with the necessary steps
        self.survey.calcTransforms()                                            # (re)calculate the transforms being used
        self.survey.calcSeedData()                                              # needed for circles, spirals & well-seeds; may affect bounding box
        self.survey.calcBoundingRect()                                          # (re)calculate the boundingBox as part of parsing the data
        self.survey.calcNoShotPoints()                                          # (re)calculate nr of shot points

        plainText = self.survey.toXmlString()                                   # convert the survey object itself to an Xml string
        self.textEdit.setTextViaCursor(plainText)                               # get text into the textEdit, NOT resetting its doc status
        self.textEdit.document().setModified(True)                              # we edited the document; so it's been modified

        if self.binAreaChanged:                                                 # we need to throw away the analysis results
            self.binAreaChanged = False                                         # reset this flag

            self.inlineStk = None                                               # numpy array with inline Kr stack reponse
            self.x_lineStk = None                                               # numpy array with x_line Kr stack reponse
            self.xyCellStk = None                                               # numpy array with cell's KxKy stack response
            self.xyPatResp = None                                               # numpy array with pattern's KxKy response

            # the following arrays are calculated in a separate binning thread and stored under the 'output' object
            self.output.binOutput = None                                        # numpy array with foldmap
            self.output.minOffset = None                                        # numpy array with minimum offset
            self.output.maxOffset = None                                        # numpy array with maximum offset
            self.output.rmsOffset = None                                        # numpy array with rms delta offset
            self.output.ofAziHist = None                                        # numpy array with offset/azimuth distribution
            self.output.offstHist = None                                        # numpy array with offset distribution

            if self.output.anaOutput is not None:                               # close memory mapped file, as well
                self.appendLogMessage(f"Edited : Closing memory mapped file {self.fileName + '.ana.npy'}")

                self.anaModel.setData(None)                                     # first remove reference to self.output.anaOutput
                self.output.D2_Output = None                                    # flattened reference to self.output.anaOutput

                # Note  numpy.load can also be used to access a file from disk, in a memory-mapped mode
                # See: https://numpy.org/doc/stable/reference/generated/numpy.load.html

                # alternatively, use numpy.lib.format.open_memmap
                # See; https://numpy.org/doc/stable/reference/generated/numpy.lib.format.open_memmap.html#numpy.lib.format.open_memmap
                # note that this approach is incompatible with using numpy.memmap
                # the reason is that unlike numpy.memmap, lib.format.open_memmap uses an additional file header
                # this means that metadata such as the shape of the array is serialized with the data itself
                # this approach woull be preferred over numpy.memmap, as it is less prone to errors
                # BUT; it does not allow resizing any already created memory mapped files. So this is a no-go.

                # I had serious difficulties in closing/deleting a memory mapped file
                # see: https://stackoverflow.com/questions/6481378/cant-delete-file-for-memorymappedfile
                # see: https://stackoverflow.com/questions/50460461/python-mmap-what-if-i-dont-call-mmap-close-manually
                # See: https://stackoverflow.com/questions/6397495/unmap-of-numpy-memmap
                # See: https://stackoverflow.com/questions/39953501/i-cant-remove-file-created-by-memmap
                # In the end the solution appeared to be:
                # 1) Flush the data to make sure everything has been copied to disk
                # 2) remove all references (self.output.D2_Output) to the memory mapped object
                # 3) delete the self.output.anaOutput object
                # 4) reinstate the object as 'None'
                # 5) force the garbage collector to do its thing (probably not equired, but let's get some memory space anyhow)
                # this removed all references to the memory mapped file, and the file can now be deleted

                self.anaModel.setData(None)                                     # show empty trace table
                self.output.D2_Output = None                                    # remove reference to self.output.anaOutput
                del self.output.anaOutput                                       # try to delete the object
                self.output.anaOutput = None                                    # the object was deleted; reinstate the None version
                gc.collect()                                                    # get the garbage collector going

            binFileName = self.fileName + '.bin.npy'                            # file names for analysis files
            minFileName = self.fileName + '.min.npy'
            maxFileName = self.fileName + '.max.npy'
            rmsFileName = self.fileName + '.rms.npy'
            anaFileName = self.fileName + '.ana.npy'

            try:
                if os.path.exists(binFileName):
                    os.remove(binFileName)                                      # remove file names, if possible
                if os.path.exists(minFileName):
                    os.remove(minFileName)
                if os.path.exists(maxFileName):
                    os.remove(maxFileName)
                if os.path.exists(rmsFileName):
                    os.remove(rmsFileName)
                if os.path.exists(anaFileName):
                    os.remove(anaFileName)
            except OSError as e:
                self.appendLogMessage(f"Can't delete file, {e}")
            self.updateMenuStatus(True)                                         # keep menu status in sync with program's state; analysis files have been deleted !
        else:
            self.updateMenuStatus(False)                                        # keep menu status in sync with program's state; analysis files have not been deleted

        self.appendLogMessage(f'Edited : {self.fileName} survey object updated')
        self.plotLayout()

    def binningSettingsHaveChanged(self, *_):                                   # param, changes unused; replaced by *_
        self.binAreaChanged = True

    ## If anything changes in the tree, print a message
    def propertyTreeStateChanged(self, param, changes):
        # self.propertyButtonBox.button(QDialogButtonBox.Apply).setEnabled(True)
        print('┌── sigTreeStateChanged --> tree changes:')
        for param, change, data in changes:
            path = self.parameters.childPath(param)
            if path is not None:
                childName = '.'.join(path)
            else:
                childName = param.name()
            print(f'│     parameter: {childName}')
            print(f'│     change:    {change}')
            print(f'│     data:      {str(data)}')
            print('└───────────────────────────────────────')

    def onMainTabChange(self, index):                                           # manage focus when active tab is changed; doesn't work 100% yet !
        if index == 0:                                                          # main plotting widget
            self.handleSpiderPlot()
        else:
            widget = self.mainTabWidget.currentWidget()
            if isinstance(widget, QCodeEditor):
                widget.setFocus()

    def cut(self):
        currentWidget = QApplication.focusWidget()
        try:
            currentWidget.cut()
        except AttributeError as e:                                             # current widget does not support cut(), so ignore command
            print('Exception occurred: ', e)
        self.actionPaste.setEnabled(self.clipboardHasText())

    def copy(self):
        currentWidget = QApplication.focusWidget()
        try:
            currentWidget.copy()
        except AttributeError as e:                                             # current widget does not support copy(), so ignore command
            print('Exception occurred: ', e)
        self.actionPaste.setEnabled(self.clipboardHasText())

    def paste(self):
        currentWidget = QApplication.focusWidget()
        try:
            currentWidget.paste()
        except AttributeError as e:                                             # current widget does not support paste(), so ignore command
            print('Exception occurred: ', e)
        return

    def selectAll(self):
        currentWidget = QApplication.focusWidget()
        try:
            currentWidget.selectAll()
        except AttributeError as e:                                             # current widget does not support selectAll(), so ignore command
            print('Exception occurred: ', e)
        return

    def find(self):
        # find only operates on the xml-text edit
        self.mainTabWidget.setCurrentIndex(1)                                   # make sure we display the 'xml' tab
        Find(self).show()                                                       # show find and replace dialog

    def createPlotWidget(self, plotTitle='', xAxisTitle='', yAxisTitle='', unitX='', unitY='', aspectLocked=True):
        """Create a plot widget for first usage"""
        w = pg.PlotWidget(background='w')
        w.setAspectLocked(True)                                                 # setting can be changed through a toolbar
        w.showGrid(x=True, y=True, alpha=0.75)                                  # shows the grey grid lines
        w.setMinimumSize(150, 150)                                              # prevent excessive widget shrinking
        # See: https://stackoverflow.com/questions/44402399/how-to-disable-the-default-context-menu-of-pyqtgraph for context menu options
        w.setContextMenuActionVisible('Transforms', False)
        w.setContextMenuActionVisible('Downsample', False)
        w.setContextMenuActionVisible('Average', False)
        w.setContextMenuActionVisible('Alpha', False)
        w.setContextMenuActionVisible('Points', False)

        # w.setMenuEnabled(False, enableViewBoxMenu=None)                       # get rid of context menu but keep ViewBox menu
        # w.ctrlMenu = None                                                     # get rid of 'Plot Options' in context menu
        # w.scene().contextMenu = None                                          # get rid of 'Export' in context menu

        # set up plot title
        w.setTitle(plotTitle, color='b', size='16pt')

        # setup axes
        styles = {'color': '#000', 'font-size': '10pt'}
        w.showAxes(True, showValues=(True, False, False, True))                 # show values at the left and at the bottom
        w.setLabel('bottom', xAxisTitle, units=unitX, **styles)                 # shows axis at the bottom, and shows the units label
        w.setLabel('left', yAxisTitle, units=unitY, **styles)                   # shows axis at the left, and shows the units label
        w.setLabel('top', ' ', **styles)                                        # shows axis at the top, no label, no tickmarks
        w.setLabel('right', ' ', **styles)                                      # shows axis at the right, no label, no tickmarks
        w.setAspectLocked(aspectLocked)

        w.installEventFilter(self)                                              # filter the 'Show' event to connect to toolbar buttons
        return w

    def resetPlotWidget(self, w, plotTitle):
        w.plotItem.clear()
        w.setTitle(plotTitle, color='b', size='16pt')

    # deal with the spider navigation
    # See: https://stackoverflow.com/questions/49316067/how-get-pressed-keys-in-mousepressevent-method-with-qt
    def spiderGoRt(self, *_, direction=Direction.Rt):
        self.navigateSpider(direction=direction)

    def spiderGoLt(self, *_, direction=Direction.Lt):
        self.navigateSpider(direction=direction)

    def spiderGoUp(self, *_, direction=Direction.Up):
        self.navigateSpider(direction=direction)

    def spiderGoDn(self, *_, direction=Direction.Dn):
        self.navigateSpider(direction=direction)

    # See: https://groups.google.com/g/pyqtgraph/c/kewFL4LkoYE?pli=1 to add pg.TextItem to indicate fold & location
    def handleSpiderPlot(self):
        if self.tbSpider.isChecked():
            self.navigateSpider(Direction.NA)
        else:
            self.plotLayout()

    def navigateSpider(self, direction):
        if self.output.anaOutput is None or self.output.binOutput is None:
            return

        step = 1
        # See: https://doc.qt.io/archives/qt-4.8/qapplication.html#queryKeyboardModifiers
        modifierPressed = QApplication.keyboardModifiers()
        if (modifierPressed & Qt.ControlModifier) == Qt.ControlModifier:
            step = 10
        if (modifierPressed & Qt.ShiftModifier) == Qt.ShiftModifier:
            step *= 5

        xAnaSize = self.output.anaOutput.shape[0]                        # spider only available when anaOutput is available
        yAnaSize = self.output.anaOutput.shape[1]
        zAnaFold = self.output.anaOutput.shape[2]                        # max allowable fold
        wAnaCols = self.output.anaOutput.shape[3]                        # need 13 columns here

        assert wAnaCols == 13, 'there need to be 13 fields in the analysis array'

        xBinSize = self.output.binOutput.shape[0]                               # The x, y sizes need to match
        yBinSize = self.output.binOutput.shape[1]

        if xAnaSize != xBinSize or yAnaSize != yBinSize:
            QMessageBox.warning(self, 'Misaligned analysis arrays', 'Binning file and extended analysis file have dissimilar sizes. Please rerun analysis')
            return

        if self.spiderPoint == QPoint(-1, -1):                                  # no valid position yet; move to center
            self.spiderPoint = QPoint(xAnaSize // 2, yAnaSize // 2)
        elif direction == Direction.Rt:                                         # valid position, so move spider around
            self.spiderPoint += QPoint(1, 0) * step
        elif direction == Direction.Lt:
            self.spiderPoint -= QPoint(1, 0) * step
        elif direction == Direction.Up:
            self.spiderPoint += QPoint(0, 1) * step
        elif direction == Direction.Dn:
            self.spiderPoint -= QPoint(0, 1) * step

        if self.spiderPoint.x() < 0:
            self.spiderPoint.setX(0)

        if self.spiderPoint.y() < 0:
            self.spiderPoint.setY(0)

        if self.spiderPoint.x() >= xAnaSize:
            self.spiderPoint.setX(xAnaSize - 1)

        if self.spiderPoint.y() >= yAnaSize:
            self.spiderPoint.setY(yAnaSize - 1)

        nX = self.spiderPoint.x()                                               # get x, y indices into bin array
        nY = self.spiderPoint.y()

        try:                                                                    # protect against potential index errors
            fold = self.output.binOutput[nX, nY]                         # max fold; accounting for unique fold
            fold = min(fold, zAnaFold)                                          # 3rd dimension in analysis file reflects available records per cmp
        except IndexError:                                                      # in case array is resized during FileNew()
            return                                                              # something went wrong accessing binning array

        plotIndex = self.getVisiblePlotWidget()[1]                              # get the active plot widget nr
        if plotIndex == 0:                                                      # update the layout plot

            if fold > 0:                                                        # create the spider legs
                self.spiderSrcX, self.spiderSrcY, self.spiderRecX, self.spiderRecY = numbaSpiderBin(self.output.anaOutput[nX, nY, 0:fold, :])
            else:                                                               # nothing to show
                self.spiderSrcX, self.spiderSrcY, self.spiderRecX, self.spiderRecY = (None, None, None, None)

            invBinTransform, _ = self.survey.binTransform.inverted()            # need to go from bin nr's to cmp(x, y)
            cmpX, cmpY = invBinTransform.map(nX, nY)                            # get local coordinates from line and point indices
            stkX, stkY = self.survey.st2Transform.map(cmpX, cmpY)               # get the corresponding bin and stake numbers

            if fold > 0:
                labelX = cmpX
                labelY = max(self.spiderRecY.max(), self.spiderSrcY.max())
            else:
                labelX = cmpX
                labelY = cmpY

            if self.glob:                                                       # we need to convert local to global coords
                labelX, labelY = self.survey.glbTransform.map(labelX, labelY)   # get global position from local version

            if self.spiderText is None:                                         # first time around
                self.spiderText = pg.TextItem(anchor=(0.5, 1.3), border='b', color='b', fill=(130, 255, 255, 200), text='spiderLabel')
                self.spiderText.setZValue(1000)                                 # make sure it is visible

            self.spiderText.setPos(labelX, labelY)                                  # move the label above spider's cmp position
            self.spiderText.setText(f'S({int(stkX)},{int(stkY)}), fold = {fold}')   # update the label text accordingly
            self.plotLayout()

        else:
            self.updateVisiblePlotWidget(plotIndex)                             # update one of the analysis plots

        # now sync the trace table selection with the spider position
        sizeY = self.output.anaOutput.shape[1]                           # x-line size of analysis array
        maxFld = self.output.anaOutput.shape[2]                          # max fold from analysis file
        offset = (nX * sizeY + nY) * maxFld                                     # calculate offset for self.output.D2_Output array
        index = self.anaView.model().index(offset, 0)                           # turn offset into index object
        self.anaView.scrollTo(index)                                            # scroll to index
        self.anaView.selectRow(offset)                                          # for the time being, *only* select first row of traces in a bin

        fold = max(fold, 1)                                                     # only highlight one line when fold = 0
        TL = QModelIndex(self.anaView.model().index(offset, 0))
        BR = QModelIndex(self.anaView.model().index(offset + fold - 1, 0))
        selection = QItemSelection(TL, BR)

        sm = self.anaView.selectionModel()                                      # select corresponding rows in self.anaView table
        sm.select(selection, QItemSelectionModel.ClearAndSelect | QItemSelectionModel.Rows)

    # define several sps, rps, xps button functions

    def sortXpsData(self, index):
        if self.xpsImport is None:
            return

        self.xpsModel.setSort(index)

        if index < 3:
            self.xpsImport.sort(order=['SrcInd', 'SrcLin', 'SrcPnt', 'RecInd', 'RecLin', 'RecMin', 'RecMax'])
        elif index == 3:
            self.xpsImport.sort(order=['Record', 'SrcInd', 'SrcLin', 'SrcPnt', 'RecInd', 'RecLin', 'RecMin', 'RecMax'])
        else:
            self.xpsImport.sort(order=['RecInd', 'RecLin', 'RecMin', 'RecMax'])
        self.xpsModel.setData(self.xpsImport)

    def removeRpsDuplicates(self):
        if self.rpsImport is None:
            return

        self.rpsImport, before, after = deletePntDuplicates(self.rpsImport)
        self.rpsModel.setData(self.rpsImport)                                   # update the model's data
        if after < before:                                                      # need to update the (x, y) points as well
            self.rpsLiveE, self.rpsLiveN, self.rpsDeadE, self.rpsDeadN = getGeometry(self.rpsImport)
            self.updateMenuStatus(False)                                        # keep menu status in sync with program's state; don't reset analysis figure
            self.plotLayout()
        self.appendLogMessage(f'Filter : Filtered {before:,} records. Removed {(before - after):,} rps-duplicates')

    def removeSpsDuplicates(self):
        if self.spsImport is None:
            return

        self.spsImport, before, after = deletePntDuplicates(self.spsImport)
        self.spsModel.setData(self.spsImport)
        if after < before:
            self.spsLiveE, self.spsLiveN, self.spsDeadE, self.spsDeadN = getGeometry(self.spsImport)
            self.updateMenuStatus(False)                                        # keep menu status in sync with program's state; don't reset analysis figure
            self.plotLayout()
        self.appendLogMessage(f'Filter : Filtered {before:,} records. Removed {(before - after):,} sps-duplicates')

    def removeRpsOrphans(self):
        if self.rpsImport is None:
            return

        self.rpsImport, before, after = deletePntOrphans(self.rpsImport)
        self.rpsModel.setData(self.rpsImport)
        if after < before:
            self.rpsLiveE, self.rpsLiveN, self.rpsDeadE, self.rpsDeadN = getGeometry(self.rpsImport)
            self.updateMenuStatus(False)                                        # keep menu status in sync with program's state; don't reset analysis figure
            self.plotLayout()
        self.appendLogMessage(f'Filter : Filtered {before:,} records. Removed {(before - after):,} rps/xps-orphans')

    def removeSpsOrphans(self):
        if self.spsImport is None:
            return

        self.spsImport, before, after = deletePntOrphans(self.spsImport)
        self.spsModel.setData(self.spsImport)
        if after < before:
            self.spsLiveE, self.spsLiveN, self.spsDeadE, self.spsDeadN = getGeometry(self.spsImport)
            self.updateMenuStatus(False)                                        # keep menu status in sync with program's state; don't reset analysis figure
            self.plotLayout()
        self.appendLogMessage(f'Filter : Filtered {before:,} records. Removed {(before - after):,} sps/xps-orphans')

    def removeXpsDuplicates(self):
        if self.xpsImport is None:
            return

        self.xpsImport, before, after = deleteRelDuplicates(self.xpsImport)
        self.xpsModel.setData(self.xpsImport)
        self.appendLogMessage(f'Filter : Filtered {before:,} records. Removed {(before - after):,} xps-duplicates')

    def removeXpsSpsOrphans(self):
        if self.xpsImport is None:
            return

        self.xpsImport, before, after = deleteRelOrphans(self.xpsImport, True)
        self.xpsModel.setData(self.xpsImport)
        self.appendLogMessage(f'Filter : Filtered {before:,} records. Removed {(before - after):,} xps/sps-orphans')

    def removeXpsRpsOrphans(self):
        if self.xpsImport is None:
            return

        self.xpsImport, before, after = deleteRelOrphans(self.xpsImport, False)
        self.xpsModel.setData(self.xpsImport)
        self.appendLogMessage(f'Filter : Filtered {before:,} records. Removed {(before - after):,} xps/rps-orphans')

    # define src, rec, rel button functions
    def sortRelData(self, index):
        if self.relGeom is None:
            return

        if index < 3:
            self.relGeom.sort(order=['SrcInd', 'SrcLin', 'SrcPnt', 'RecInd', 'RecLin', 'RecMin', 'RecMax'])
        elif index == 3:
            self.relGeom.sort(order=['Record', 'SrcInd', 'SrcLin', 'SrcPnt', 'RecInd', 'RecLin', 'RecMin', 'RecMax'])
        else:
            self.relGeom.sort(order=['RecInd', 'RecLin', 'RecMin', 'RecMax'])
        self.relModel.setData(self.relGeom)

    def removeRecDuplicates(self):
        if self.recGeom is None:
            return

        self.recGeom, before, after = deletePntDuplicates(self.recGeom)
        self.recModel.setData(self.recGeom)                                     # update the model's data
        if after < before:                                                      # need to update the (x, y) points as well
            self.recLiveE, self.recLiveN, self.recDeadE, self.recDeadN = getGeometry(self.recGeom)
            self.updateMenuStatus(False)                                        # keep menu status in sync with program's state; don't reset analysis figure
            self.plotLayout()
        self.appendLogMessage(f'Filter : Filtered {before:,} records. Removed {(before - after):,} rec-duplicates')

    def removeSrcDuplicates(self):
        if self.srcGeom is None:
            return

        self.srcGeom, before, after = deletePntDuplicates(self.srcGeom)
        self.srcModel.setData(self.srcGeom)
        if after < before:
            self.srcLiveE, self.srcLiveN, self.srcDeadE, self.srcDeadN = getGeometry(self.srcGeom)
            self.updateMenuStatus(False)                                        # keep menu status in sync with program's state; don't reset analysis figure
            self.plotLayout()
        self.appendLogMessage(f'Filter : Filtered {before:,} records. Removed {(before - after):,} src-duplicates')

    def removeRecOrphans(self):
        if self.recGeom is None:
            return

        self.recGeom, before, after = deletePntOrphans(self.recGeom)
        self.recModel.setData(self.recGeom)
        if after < before:
            self.recLiveE, self.recLiveN, self.recDeadE, self.recDeadN = getGeometry(self.recGeom)
            self.updateMenuStatus(False)                                        # keep menu status in sync with program's state; don't reset analysis figure
            self.plotLayout()
        self.appendLogMessage(f'Filter : Filtered {before:,} records. Removed {(before - after):,} rec/rel-orphans')

    def removeSrcOrphans(self):
        if self.srcGeom is None:
            return

        self.srcGeom, before, after = deletePntOrphans(self.srcGeom)
        self.srcModel.setData(self.srcGeom)
        if after < before:
            self.srcLiveE, self.srcLiveN, self.srcDeadE, self.srcDeadN = getGeometry(self.srcGeom)
            self.updateMenuStatus(False)                                        # keep menu status in sync with program's state; don't reset analysis figure
            self.plotLayout()
        self.appendLogMessage(f'Filter : Filtered {before:,} records. Removed {(before - after):,} src/rel-orphans')

    def removeRelDuplicates(self):
        if self.relGeom is None:
            return

        self.relGeom, before, after = deleteRelDuplicates(self.relGeom)
        self.relModel.setData(self.relGeom)
        self.appendLogMessage(f'Filter : Filtered {before:,} records. Removed {(before - after):,} rel-duplicates')

    def removeRelSrcOrphans(self):
        if self.relGeom is None:
            return

        self.relGeom, before, after = deleteRelOrphans(self.relGeom, True)
        self.relModel.setData(self.relGeom)
        self.appendLogMessage(f'Filter : Filtered {before:,} records. Removed {(before - after):,} rel/src-orphans')

    def removeRelRecOrphans(self):
        if self.relGeom is None:
            return

        self.relGeom, before, after = deleteRelOrphans(self.relGeom, False)
        self.relModel.setData(self.relGeom)
        self.appendLogMessage(f'Filter : Filtered {before:,} records. Removed {(before - after):,} rel/rec-orphans')

    # define file export functions
    def fileExportFoldMap(self):
        if self.survey is not None and self.output.binOutput is not None and self.survey.crs is not None:
            fileName = self.fileName + '.bin.tif'
            fileName = CreateQgisRasterLayer(fileName, self.output.binOutput, self.survey)
            if fileName:
                self.appendLogMessage(f'Export : exported fold map to {fileName}')

    def fileExportMinOffsets(self):
        if self.survey is not None and self.output.minOffset is not None and self.survey.crs is not None:
            fileName = self.fileName + '.min.tif'
            fileName = CreateQgisRasterLayer(fileName, self.output.minOffset, self.survey)
            if fileName:
                self.appendLogMessage(f'Export : exported min-offsets to {fileName}')

    def fileExportMaxOffsets(self):
        if self.survey is not None and self.output.maxOffset is not None and self.survey.crs is not None:
            fileName = self.fileName + '.max.tif'
            fileName = CreateQgisRasterLayer(fileName, self.output.maxOffset, self.survey)
            if fileName:
                self.appendLogMessage(f'Export : exported max-offsets to {fileName}')

    def fileExportRmsOffsets(self):
        if self.survey is not None and self.output.rmsOffset is not None and self.survey.crs is not None:
            fileName = self.fileName + '.rms.tif'
            fileName = CreateQgisRasterLayer(fileName, self.output.rmsOffset, self.survey)
            if fileName:
                self.appendLogMessage(f'Export : exported rms-offsets to {fileName}')

    def exportBinToQGIS(self):
        if self.survey is not None and self.output.binOutput is not None and self.survey.crs is not None:
            fileName = self.fileName + '.bin.tif'
            fileName = ExportRasterLayerToQgis(fileName, self.output.binOutput, self.survey)
            if fileName:
                self.appendLogMessage('Export : incorporated fold map in QGIS')

    def exportMinToQGIS(self):
        if self.survey is not None and self.output.minOffset is not None and self.survey.crs is not None:
            fileName = self.fileName + '.min.tif'
            fileName = ExportRasterLayerToQgis(fileName, self.output.minOffset, self.survey)
            if fileName:
                self.appendLogMessage('Export : incorporated min-offset map in QGIS')

    def exportMaxToQGIS(self):
        if self.survey is not None and self.output.maxOffset is not None and self.survey.crs is not None:
            fileName = self.fileName + '.max.tif'
            fileName = ExportRasterLayerToQgis(fileName, self.output.maxOffset, self.survey)
            if fileName:
                self.appendLogMessage('Export : incorporated max-offset map in QGIS')

    def exportRmsToQGIS(self):
        if self.survey is not None and self.output.rmsOffset is not None and self.survey.crs is not None:
            fileName = self.fileName + '.rms.tif'
            fileName = ExportRasterLayerToQgis(fileName, self.output.rmsOffset, self.survey)
            if fileName:
                self.appendLogMessage('Export : incorporated rms-offset map in QGIS')

    def exportRpsToQgis(self):
        if self.rpsImport is not None and self.survey is not None and self.survey.crs is not None:
            if not self.fileName:                                               # filename ="" normally indicates working with 'new' file !
                layerName = self.survey.name
            else:
                layerName = QFileInfo(self.fileName).baseName()
            layerName += '-rps-data'
            self.rpsLayer = exportPointLayerToQgis(layerName, self.rpsImport, self.survey.crs, source=False)

    def exportSpsToQgis(self):
        if self.spsImport is not None and self.survey is not None and self.survey.crs is not None:
            if not self.fileName:                                               # filename ="" normally indicates working with 'new' file !
                layerName = self.survey.name
            else:
                layerName = QFileInfo(self.fileName).baseName()
            layerName += '-sps-data'
            self.spsLayer = exportPointLayerToQgis(layerName, self.spsImport, self.survey.crs, source=True)

    def exportRecToQgis(self):
        if self.recGeom is not None and self.survey is not None and self.survey.crs is not None:
            if not self.fileName:                                               # filename ="" normally indicates working with 'new' file !
                layerName = self.survey.name
            else:
                layerName = QFileInfo(self.fileName).baseName()
            layerName += '-rec-data'
            self.recLayer = exportPointLayerToQgis(layerName, self.recGeom, self.survey.crs, source=False)

    def exportSrcToQgis(self):
        if self.srcGeom is not None and self.survey is not None and self.survey.crs is not None:
            if not self.fileName:                                               # filename ="" normally indicates working with 'new' file !
                layerName = self.survey.name
            else:
                layerName = QFileInfo(self.fileName).baseName()
            layerName += '-src-data'
            self.srcLayer = exportPointLayerToQgis(layerName, self.srcGeom, self.survey.crs, source=True)

    def importSrcFromQgis(self):
        self.srcLayer, self.srcField = identifyQgisPointLayer(self.iface, self.srcLayer, self.srcField, 'Src')

        if self.srcLayer is None:
            return

        with pg.BusyCursor():
            self.srcGeom = readQgisPointLayer(self.srcLayer.id(), self.srcField)

        if self.srcGeom is None:
            QMessageBox.information(None, 'No features found', 'No valid features found in QGIS layer', QMessageBox.Cancel)
            return

        # nSrcOrphans, nRelOrphans = findSrcOrphans(self.srcGeom, self.relGeom)
        # self.appendLogMessage(f'Import : . . . src-records contain {nRelOrphans:,} xps-orphans')
        # self.appendLogMessage(f'Import : . . . rel-records contain {nSrcOrphans:,} sps-orphans')

        self.srcLiveE, self.srcLiveN, self.srcDeadE, self.srcDeadN = getGeometry(self.srcGeom)

        self.srcModel.setData(self.srcGeom)
        self.textEdit.document().setModified(True)                              # set modified flag; so we'll save src data as numpy arrays upon saving the file
        self.updateMenuStatus(False)                                            # keep menu status in sync with program's state; don't reset analysis figure
        self.plotLayout()

    def importRecFromQgis(self):
        self.recLayer, self.recField = identifyQgisPointLayer(self.iface, self.recLayer, self.recField, 'Rec')

        if self.recLayer is None:
            return

        with pg.BusyCursor():
            self.recGeom = readQgisPointLayer(self.recLayer.id(), self.recField)

        if self.recGeom is None:
            QMessageBox.information(None, 'No features found', 'No valid features found in QGIS layer', QMessageBox.Cancel)
            return

        # nRecOrphans, nRelOrphans = findRecOrphans(self.recGeom, self.relGeom)
        # self.appendLogMessage(f'Import : . . . rps-records contain {nRelOrphans:,} rel-orphans')
        # self.appendLogMessage(f'Import : . . . xps-records contain {nRecOrphans:,} rec-orphans')

        self.recLiveE, self.recLiveN, self.recDeadE, self.recDeadN = getGeometry(self.recGeom)

        self.recModel.setData(self.recGeom)
        self.textEdit.document().setModified(True)                              # set modified flag; so we'll save rec data as numpy arrays upon saving the file
        self.updateMenuStatus(False)                                            # keep menu status in sync with program's state; don't reset analysis figure
        self.plotLayout()

    def exportOutToQgis(self):
        layerName = QFileInfo(self.fileName).baseName()
        exportSurveyOutlineToQgis(layerName, self.survey)

    def updateMenuStatus(self, resetAnalysis=True):
        if resetAnalysis:
            self.actionNone.setChecked(True)                                    # coupled with tbNone; reset analysis figure
            self.imageType = 0                                                  # reset analysis type to zero
            self.handleImageSelection()                                         # change image (if available) and finally plot survey layout

        self.actionExportFoldMap.setEnabled(self.output.binOutput is not None)
        self.actionExportMinOffsets.setEnabled(self.output.minOffset is not None)
        self.actionExportMaxOffsets.setEnabled(self.output.maxOffset is not None)
        self.actionExportRmsOffsets.setEnabled(self.output.rmsOffset is not None)
        self.actionExportAnaAsCsv.setEnabled(self.output.anaOutput is not None)

        self.actionExportRecAsCsv.setEnabled(self.recGeom is not None)
        self.actionExportSrcAsCsv.setEnabled(self.srcGeom is not None)
        self.actionExportRelAsCsv.setEnabled(self.relGeom is not None)
        self.actionExportRecAsR01.setEnabled(self.recGeom is not None)
        self.actionExportSrcAsS01.setEnabled(self.srcGeom is not None)
        self.actionExportRelAsX01.setEnabled(self.relGeom is not None)
        self.actionExportSrcToQGIS.setEnabled(self.srcGeom is not None)
        self.actionExportRecToQGIS.setEnabled(self.recGeom is not None)

        self.actionExportRpsAsCsv.setEnabled(self.rpsImport is not None)
        self.actionExportSpsAsCsv.setEnabled(self.spsImport is not None)
        self.actionExportXpsAsCsv.setEnabled(self.xpsImport is not None)
        self.actionExportRpsAsR01.setEnabled(self.rpsImport is not None)
        self.actionExportSpsAsS01.setEnabled(self.spsImport is not None)
        self.actionExportXpsAsX01.setEnabled(self.xpsImport is not None)
        self.actionExportSpsToQGIS.setEnabled(self.spsImport is not None)
        self.actionExportRpsToQGIS.setEnabled(self.rpsImport is not None)

        self.btnSrcRemoveDuplicates.setEnabled(self.srcGeom is not None)
        self.btnSrcRemoveOrphans.setEnabled(self.srcGeom is not None)
        self.btnSrcExportToQGIS.setEnabled(self.srcGeom is not None)

        self.btnRecRemoveDuplicates.setEnabled(self.recGeom is not None)
        self.btnRecRemoveOrphans.setEnabled(self.recGeom is not None)
        self.btnRecExportToQGIS.setEnabled(self.recGeom is not None)

        self.btnRelRemoveSrcOrphans.setEnabled(self.relGeom is not None)
        self.btnRelRemoveDuplicates.setEnabled(self.relGeom is not None)
        self.btnRelRemoveRecOrphans.setEnabled(self.relGeom is not None)

        self.actionExportAreasToQGIS.setEnabled(len(self.fileName) > 0)         # test if file name isn't empty
        self.btnRelExportToQGIS.setEnabled(len(self.fileName) > 0)              # test if file name isn't empty

        self.btnSpsExportToQGIS.setEnabled(self.spsImport is not None)
        self.btnRpsExportToQGIS.setEnabled(self.rpsImport is not None)

        self.actionFold.setEnabled(self.output.binOutput is not None)
        self.actionMinO.setEnabled(self.output.minOffset is not None)
        self.actionMaxO.setEnabled(self.output.maxOffset is not None)
        self.actionRmsO.setEnabled(self.output.rmsOffset is not None)

        self.actionSpider.setEnabled(self.output.anaOutput is not None and self.output.binOutput is not None)  # the spider button in the display pane
        self.actionMoveLt.setEnabled(self.output.anaOutput is not None)  # the navigation buttons in the Display pane AND on toolbar (moveBar)
        self.actionMoveRt.setEnabled(self.output.anaOutput is not None)
        self.actionMoveUp.setEnabled(self.output.anaOutput is not None)
        self.actionMoveDn.setEnabled(self.output.anaOutput is not None)

        self.btnBinToQGIS.setEnabled(self.output.binOutput is not None)
        self.btnMinToQGIS.setEnabled(self.output.minOffset is not None)
        self.btnMaxToQGIS.setEnabled(self.output.maxOffset is not None)
        self.btnRmsToQGIS.setEnabled(self.output.rmsOffset is not None)

        self.actionExportFoldMapToQGIS.setEnabled(self.output.binOutput is not None)
        self.actionExportMinOffsetsToQGIS.setEnabled(self.output.minOffset is not None)
        self.actionExportMaxOffsetsToQGIS.setEnabled(self.output.maxOffset is not None)
        self.actionExportRmsOffsetsToQGIS.setEnabled(self.output.rmsOffset is not None)

        self.actionRecPoints.setEnabled(self.recGeom is not None)
        self.actionSrcPoints.setEnabled(self.srcGeom is not None)
        self.actionRpsPoints.setEnabled(self.rpsImport is not None)
        self.actionSpsPoints.setEnabled(self.spsImport is not None)
        self.actionAllPoints.setEnabled(self.recGeom is not None or self.srcGeom is not None or self.rpsImport is not None or self.spsImport is not None)

    def setColorbarLabel(self, label):                                          # I should really subclass colorbarItem to properly set the text label
        if label is not None:
            if self.layoutColorBar.horizontal:
                self.layoutColorBar.getAxis('bottom').setLabel(label)
            else:
                self.layoutColorBar.getAxis('left').setLabel(label)

    def onActionNoneTriggered(self):
        self.imageType = 0
        self.handleImageSelection()

    def onActionFoldTriggered(self):
        self.imageType = 1
        self.handleImageSelection()

    def onActionMinOTriggered(self):
        self.imageType = 2
        self.handleImageSelection()

    def onActionMaxOTriggered(self):
        self.imageType = 3
        self.handleImageSelection()

    def onActionRmsOTriggered(self):
        self.imageType = 4
        self.handleImageSelection()

    def handleImageSelection(self):                                             # change image (if available) and finally plot survey layout

        colorMap = config.fold_OffCmap                                          # default fold & offset color map
        if self.imageType == 0:                                                 # now deal with all image types
            self.layoutImg = None                                               # no image to show
            label = 'N/A'
            self.layoutMax = 10
            colorMap = config.inActiveCmap                                      # grey color map
        elif self.imageType == 1:
            self.layoutImg = self.output.binOutput                              # don't make a copy, create a view
            self.layoutMax = self.output.maximumFold
            label = 'fold'
        elif self.imageType == 2:
            self.layoutImg = self.output.minOffset
            self.layoutMax = self.output.maxMinOffset
            label = 'minimum offset'
        elif self.imageType == 3:
            self.layoutImg = self.output.maxOffset
            self.layoutMax = self.output.maxMaxOffset
            label = 'maximum offset'
        elif self.imageType == 4:
            self.layoutImg = self.output.rmsOffset
            self.layoutMax = self.output.maxRmsOffset
            label = 'rms offset increments'
        else:
            raise NotImplementedError('selected analysis type currently not implemented.')

        self.layoutImItem = pg.ImageItem()                                          # create a PyqtGraph image item
        self.layoutImItem.setImage(self.layoutImg, levels=(0.0, self.layoutMax))    # set image and its range limits

        if self.layoutColorBar is None:                                             # create colorbar with default values
            self.layoutColorBar = self.layoutWidget.plotItem.addColorBar(self.layoutImItem, colorMap=config.inActiveCmap, label='N/A', limits=(0, None), rounding=10.0, values=(0, 10))

        if self.layoutColorBar is not None:
            self.layoutColorBar.setImageItem(self.layoutImItem)                     # couple imageItem to the colorbar
            self.layoutColorBar.setLevels(low=0.0, high=self.layoutMax)
            self.layoutColorBar.setColorMap(colorMap)
            self.setColorbarLabel(label)

        self.plotLayout()

    def mouseBeingDragged(self):                                                # essential for LOD plotting whilst moving the survey object around
        self.survey.mouseGrabbed = True

    def exceptionHook(self, eType, eValue, eTraceback):
        """Function handling uncaught exceptions. It is triggered each time an uncaught exception occurs."""
        if issubclass(eType, KeyboardInterrupt):
            # ignore keyboard interrupt to support ctrl+C on console applications
            sys.__excepthook__(eType, eValue, eTraceback)
        else:

            # use the next string for a messagebox if "debug is on"
            # See: https://waylonwalker.com/python-sys-excepthook/
            # traceback_details = "\n".join(traceback.extract_tb(eTraceback).format())

            traceback_str = ''
            # for file_name, line_number, func_name, text in traceback.extract_tb(eTraceback, limit=2)[1:]:
            for file_name, line_number, func_name, _ in traceback.extract_tb(eTraceback, limit=2):
                file_name = os.path.basename(file_name)
                traceback_str += f' File "{file_name}", line {line_number}, in function "{func_name}".'

            if traceback_str != '':
                traceback_str = f'Traceback: {traceback_str}'

            exceptionMsg = f'Error&nbsp;&nbsp;:&nbsp;{eType.__name__}: {eValue} {traceback_str}'
            self.appendLogMessage(exceptionMsg, MsgType.Exception)

    def cursorPositionChanged(self):
        line = self.textEdit.textCursor().blockNumber() + 1
        col = self.textEdit.textCursor().columnNumber() + 1
        self.posWidgetStatusbar.setText(f'Line: {line} Col: {col}')

    def MouseMovedInPlot(self, pos):                                            # See: https://stackoverflow.com/questions/46166205/display-coordinates-in-pyqtgraph
        if self.layoutWidget.sceneBoundingRect().contains(pos):                 # is mouse moved within the scene area ?
            mousePoint = self.layoutWidget.plotItem.vb.mapSceneToView(pos)      # get scene coordinates

            if self.glob:                                                       # plot is using global coordinates
                toLocTransform, _ = self.survey.glbTransform.inverted()
                globalPoint = mousePoint
                localPoint = toLocTransform.map(globalPoint)
            else:                                                               # plot is using local coordinates
                localPoint = mousePoint
                globalPoint = self.survey.glbTransform.map(localPoint)

            lx = localPoint.x()
            ly = localPoint.y()
            gx = globalPoint.x()
            gy = globalPoint.y()

            if self.survey.binning.method == BinningType.cmp:                   # calculate reflector depth at cursor
                gz = 0.0
                lz = 0.0
            elif self.survey.binning.method == BinningType.plane:
                gz = self.survey.globalPlane.depthAt(globalPoint)               # get global depth from defined plane
                lz = self.survey.localPlane.depthAt(localPoint)                 # get local depth from transformed plane
            elif self.survey.binning.method == BinningType.sphere:
                gz = self.survey.globalSphere.depthAt(globalPoint)              # get global depth from defined sphere
                lz = self.survey.localSphere.depthAt(localPoint)                # get local depth from transformed sphere
            else:
                raise ValueError('wrong binning method selected')

            if self.survey.binTransform is not None:
                binPoint = self.survey.binTransform.map(localPoint)
                bx = int(binPoint.x())
                by = int(binPoint.y())
            else:
                bx = 0
                by = 0

            if self.survey.st2Transform is not None:
                stkPoint = self.survey.st2Transform.map(localPoint)
                sx = int(stkPoint.x())
                sy = int(stkPoint.y())
            else:
                sx = 0
                sy = 0

            if self.layoutImg is not None and bx >= 0 and by >= 0 and bx < self.layoutImg.shape[0] and by < self.layoutImg.shape[1]:
                # provide statusbar information within the analysis area
                if self.imageType == 0:
                    self.posWidgetStatusbar.setText(f'S:({sx:,d}, {sy:,d}), L:({lx:,.2f}, {ly:,.2f}, {lz:,.2f}), W:({gx:,.2f}, {gy:,.2f}, {gz:,.2f}) ')
                elif self.imageType == 1:
                    fold = self.layoutImg[bx, by]
                    self.posWidgetStatusbar.setText(f'fold: {fold:,d}, S:({sx:,d}, {sy:,d}), L:({lx:,.2f}, {ly:,.2f}, {lz:,.2f}), W:({gx:,.2f}, {gy:,.2f}, {gz:,.2f}) ')
                elif self.imageType == 2:
                    offset = float(self.layoutImg[bx, by])
                    self.posWidgetStatusbar.setText(f'|min offset|: {offset:.2f}, S:({sx:,d}, {sy:,d}), L:({lx:,.2f}, {ly:,.2f}, {lz:,.2f}), W:({gx:,.2f}, {gy:,.2f}, {gz:,.2f}) ')
                elif self.imageType == 3:
                    offset = float(self.layoutImg[bx, by])
                    self.posWidgetStatusbar.setText(f'|max offset|: {offset:.2f}, S:({sx:,d}, {sy:,d}), L:({lx:,.2f}, {ly:,.2f}, {lz:,.2f}), W:({gx:,.2f}, {gy:,.2f}, {gz:,.2f}) ')
                elif self.imageType == 4:
                    offset = float(self.layoutImg[bx, by])
                    self.posWidgetStatusbar.setText(f'rms offset inc: {offset:.2f}, S:({sx:,d}, {sy:,d}), L:({lx:,.2f}, {ly:,.2f}, {lz:,.2f}), W:({gx:,.2f}, {gy:,.2f}, {gz:,.2f}) ')
            else:
                # provide statusbar information outside the analysis area
                self.posWidgetStatusbar.setText(f'S:({sx:,d}, {sy:,d}), L:({lx:,.2f}, {ly:,.2f}, {lz:,.2f}), W:({gx:,.2f}, {gy:,.2f}, {gz:,.2f}) ')

    def getVisiblePlotIndex(self, plotWidget):
        if plotWidget == self.layoutWidget:
            return 0
        elif plotWidget == self.offTrkWidget:
            return 1
        elif plotWidget == self.offBinWidget:
            return 2
        elif plotWidget == self.aziTrkWidget:
            return 3
        elif plotWidget == self.aziBinWidget:
            return 4
        elif plotWidget == self.stkTrkWidget:
            return 5
        elif plotWidget == self.stkBinWidget:
            return 6
        elif plotWidget == self.stkCelWidget:
            return 7
        elif plotWidget == self.offsetWidget:
            return 8
        elif plotWidget == self.offAziWidget:
            return 9
        elif plotWidget == self.arraysWidget:
            return 10

        return None

    def getVisiblePlotWidget(self):
        if self.layoutWidget.isVisible():
            return (self.layoutWidget, 0)
        if self.offTrkWidget.isVisible():
            return (self.offTrkWidget, 1)
        if self.offBinWidget.isVisible():
            return (self.offBinWidget, 2)
        if self.aziTrkWidget.isVisible():
            return (self.aziTrkWidget, 3)
        if self.aziBinWidget.isVisible():
            return (self.aziBinWidget, 4)
        if self.stkTrkWidget.isVisible():
            return (self.stkTrkWidget, 5)
        if self.stkBinWidget.isVisible():
            return (self.stkBinWidget, 6)
        if self.stkCelWidget.isVisible():
            return (self.stkCelWidget, 7)
        if self.offsetWidget.isVisible():
            return (self.offsetWidget, 8)
        if self.offAziWidget.isVisible():
            return (self.offAziWidget, 9)
        if self.arraysWidget.isVisible():
            return (self.arraysWidget, 10)

        return (None, None)

    def updateVisiblePlotWidget(self, index: int) -> None:
        if index == 0:
            self.plotLayout()                                                   # no conditions to plot main layout plot
            return

        if index == 10:                                                         # no condition to plot patterns either
            self.plotPatterns()
            return

        if self.output.anaOutput is None:                                       # we need self.output.anaOutput to display meaningful ANALYSIS information
            return

        xAnaSize = self.output.anaOutput.shape[0]                               # make sure we have a valid self.spiderPoint, hence valid nx, ny
        yAnaSize = self.output.anaOutput.shape[1]

        if self.spiderPoint == QPoint(-1, -1):                                  # no valid position yet; move to center
            self.spiderPoint = QPoint(xAnaSize // 2, yAnaSize // 2)

        if self.spiderPoint.x() < 0:                                            # build in some safety settings
            self.spiderPoint.setX(0)

        if self.spiderPoint.y() < 0:
            self.spiderPoint.setY(0)

        if self.spiderPoint.x() >= xAnaSize:
            self.spiderPoint.setX(xAnaSize - 1)

        if self.spiderPoint.y() >= yAnaSize:
            self.spiderPoint.setY(yAnaSize - 1)

        nX = self.spiderPoint.x()                                               # get x, y indices into bin array
        nY = self.spiderPoint.y()

        invBinTransform, _ = self.survey.binTransform.inverted()                # need to go from bin nr's to cmp(x, y)
        cmpX, cmpY = invBinTransform.map(nX, nY)                                # get local coordinates from line and point indices
        stkX, stkY = self.survey.st2Transform.map(cmpX, cmpY)                   # get the corresponding bin and stake numbers

        x0 = self.survey.output.rctOutput.left()                                # x origin of binning area
        y0 = self.survey.output.rctOutput.top()                                 # y origin of binning area

        dx = self.survey.grid.binSize.x()                                       # x bin size
        dy = self.survey.grid.binSize.y()                                       # y bin size
        ox = 0.5 * dx                                                           # half the x bin size
        oy = 0.5 * dy                                                           # half the y bin size

        if index == 1:
            self.plotOffTrk(nY, stkY, ox)
        elif index == 2:
            self.plotOffBin(nX, stkX, oy)
        elif index == 3:
            self.plotAziTrk(nY, stkY, ox)
        elif index == 4:
            self.plotAziBin(nX, stkX, oy)
        elif index == 5:
            self.plotStkTrk(nY, stkY, x0, dx)
        elif index == 6:
            self.plotStkBin(nX, stkX, y0, dy)
        elif index == 7:
            self.plotStkCel(nX, nY, stkX, stkY)
        elif index == 8:
            self.plotOffset()
        elif index == 9:
            self.plotOffAzi()

    def plotZoomRect(self):
        visiblePlot = self.getVisiblePlotWidget()[0]
        if visiblePlot is not None:
            viewBox = visiblePlot.getViewBox()
            self.rect = viewBox.getState()['mouseMode'] == pg.ViewBox.RectMode   # get rect status
            if self.rect:
                viewBox.setMouseMode(pg.ViewBox.PanMode)
            else:
                viewBox.setMouseMode(pg.ViewBox.RectMode)

    def plotAspectRatio(self):
        visiblePlot = self.getVisiblePlotWidget()[0]
        if visiblePlot is not None:
            plotItem = visiblePlot.getPlotItem()
            self.XisY = not plotItem.saveState()['view']['aspectLocked']        # get XisY status
            visiblePlot.setAspectLocked(self.XisY)

    def plotAntiAlias(self):
        visiblePlot, index = self.getVisiblePlotWidget()
        if visiblePlot is not None:                                             # there's no internal AA state
            self.antiA[index] = not self.antiA[index]                           # maintain status externally
            visiblePlot.setAntialiasing(self.antiA[index])                      # enable/disable aa plotting

    def plotGridX(self):
        visiblePlot = self.getVisiblePlotWidget()[0]
        if visiblePlot is not None:
            plotItem = visiblePlot.getPlotItem()
            self.gridX = not plotItem.saveState()['xGridCheck']                 # update x-gridline status
            if self.gridX:
                visiblePlot.showGrid(x=True, alpha=0.75)                        # show the grey grid lines
            else:
                visiblePlot.showGrid(x=False)                                   # don't show the grey grid lines

    def plotGridY(self):
        visiblePlot = self.getVisiblePlotWidget()[0]
        if visiblePlot is not None:
            plotItem = visiblePlot.getPlotItem()
            self.gridY = not plotItem.saveState()['yGridCheck']                 # update y-gridline status
            if self.gridY:
                visiblePlot.showGrid(y=True, alpha=0.75)                        # show the grey grid lines
            else:
                visiblePlot.showGrid(y=False)                                   # don't show the grey grid lines

    def plotProjected(self):
        self.glob = self.actionProjected.isChecked()

        if self.ruler:
            self.actionRuler.setChecked(False)
            self.showRuler(False)

        self.rulerState = None

        if self.debug:                                                          # provide some debugging output on the applied transform
            # Get the transform that maps from local coordinates to the item's ViewBox coordinates
            transform = self.survey.glbTransform                                # GraphicsItem method
            if transform is not None:
                s1 = f'm11 ={transform.m11():12.6f},   m12 ={transform.m12():12.6f},   m13 ={transform.m13():12.6f} » [A1, B1, ...]'
                s2 = f'm21 ={transform.m21():12.6f},   m22 ={transform.m22():12.6f},   m23 ={transform.m23():12.6f} » [A2, B2, ...]'
                s3 = f'm31 ={transform.m31():12.6f},   m32 ={transform.m32():12.6f},   m33 ={transform.m33():12.6f} » [A0, B0, ...]<br>'

                self.appendLogMessage('plotProjected(). Showing transform parameters before changing view', MsgType.Debug)
                self.appendLogMessage(s1, MsgType.Debug)
                self.appendLogMessage(s2, MsgType.Debug)
                self.appendLogMessage(s3, MsgType.Debug)

                if not transform.isIdentity():
                    i_trans, _ = transform.inverted()                           # inverted_transform, invertable = transform.inverted()
                    s1 = f'm11 ={i_trans.m11():12.6f},   m12 ={i_trans.m12():12.6f},   m13 ={i_trans.m13():12.6f} » [A1, B1, ...]'
                    s2 = f'm21 ={i_trans.m21():12.6f},   m22 ={i_trans.m22():12.6f},   m23 ={i_trans.m23():12.6f} » [A2, B2, ...]'
                    s3 = f'm31 ={i_trans.m31():12.6f},   m32 ={i_trans.m32():12.6f},   m33 ={i_trans.m33():12.6f} » [A0, B0, ...]<br>'

                    self.appendLogMessage('plotProjected(). Showing inverted-transform parameters before changing view', MsgType.Debug)
                    self.appendLogMessage(s1, MsgType.Debug)
                    self.appendLogMessage(s2, MsgType.Debug)
                    self.appendLogMessage(s3, MsgType.Debug)

        self.handleSpiderPlot()                                                 # spider label should move depending on local/global coords
        self.layoutWidget.autoRange()                                           # show the full range of objects when changing local vs global coordinates
        self.plotLayout()

    def showRuler(self, checked):
        self.ruler = checked
        self.plotLayout()

    def UpdateAllViews(self):
        plainText = self.textEdit.getTextViaCursor()                            # read complete file content, not affecting doc status
        success = self.parseText(plainText)                                     # parse the string & check if it went okay...

        if success:
            plainText = self.survey.toXmlString()                               # convert the survey object itself to an xml string
            self.textEdit.setTextViaCursor(plainText)                           # get text into the textEdit, NOT resetting its doc status
            self.textEdit.document().setModified(True)                          # we edited the document; so it's been modified
            self.resetSurveyProperties()                                        # update property pane accordingly

        self.plotLayout()
        # self.layoutWidget.enableAutoRange()                                     # makes the plot 'fit' the survey outline.

    def layoutRangeChanged(self):
        """handle resizing of plot in view of bin-aligned gridlines"""
        axLft = self.layoutWidget.plotItem.getAxis('left')                      # get y-axis # 1
        axBot = self.layoutWidget.plotItem.getAxis('bottom')                    # get x-axis # 1
        axTop = self.layoutWidget.plotItem.getAxis('top')                       # get x-axis # 2
        axRht = self.layoutWidget.plotItem.getAxis('right')                     # get y-axis # 2

        vb = self.layoutWidget.getViewBox().viewRect()                          # view area in world coords
        dx = self.survey.grid.binSize.x()                                       # x bin size
        dy = self.survey.grid.binSize.y()                                       # y bin size

        if vb.width() > dx and vb.height() > dy:                                # area must be > a single bin to do something
            if not self.glob and (vb.width() < 30.0 * dx or vb.height() < 30.0 * dy):   # scale grid towards bin size

                # In this case it would be nice to have 3 decimals at the tick marks, not two
                # See: https://stackoverflow.com/questions/47500216/pyqtgraph-force-axis-labels-to-have-decimal-points

                xTicks = [dx, 0.2 * dx]                                         # no tickmarks smaller than bin size
                yTicks = [dy, 0.2 * dy]                                         # no tickmarks smaller than bin size
                axBot.setTickSpacing(xTicks[0], xTicks[1])                      # set x ticks (major and minor)
                axLft.setTickSpacing(yTicks[0], yTicks[1])                      # set x ticks (major and minor)
                axTop.setTickSpacing(xTicks[0], xTicks[1])                      # set x ticks (major and minor)
                axRht.setTickSpacing(yTicks[0], yTicks[1])                      # set x ticks (major and minor)
            else:
                axBot.setTickSpacing()                                          # set to default values
                axLft.setTickSpacing()                                          # set to default values
                axTop.setTickSpacing()                                          # set to default values
                axRht.setTickSpacing()                                          # set to default values

    def plotLayout(self):
        # first we are going to see how large the survey area is, to establish a boundingbox
        # See: https://www.geeksforgeeks.org/pyqtgraph-removing-item-from-plot-window/
        # self.layoutWidget.plotItem.removeItem(self.legend)
        # self.layoutWidget.plotItem.removeItem(self.srcLines)
        # self.layoutWidget.plotItem.removeItem(self.recLines)
        # See also: https://groups.google.com/g/pyqtgraph/c/tlryVLCDmmQ when the view does not refresh

        self.layoutWidget.plotItem.clear()
        self.layoutWidget.setTitle(self.survey.name, color='b', size='16pt')
        self.layoutWidget.showAxes(True, showValues=(True, False, False, True))   # show values at the left and at the bottom

        transform = QTransform()                                                # empty (unit) transform

        # setup axes first
        styles = {'color': '#000', 'font-size': '10pt'}
        if self.glob:                                                           # global -> easting & westing
            self.layoutWidget.setLabel('bottom', 'Easting', units='m', **styles)  # shows axis at the bottom, and shows the units label
            self.layoutWidget.setLabel('left', 'Northing', units='m', **styles)   # shows axis at the left, and shows the units label
            self.layoutWidget.setLabel('top', ' ', **styles)                    # shows axis at the top, no label, no tickmarks
            self.layoutWidget.setLabel('right', ' ', **styles)                  # shows axis at the right, no label, no tickmarks
            transform = self.survey.glbTransform                                # get global coordinate conversion transform
        else:                                                                   # local -> inline & crossline
            self.layoutWidget.setLabel('bottom', 'inline', units='m', **styles)   # shows axis at the bottom, and shows the units label
            self.layoutWidget.setLabel('left', 'crossline', units='m', **styles)  # shows axis at the left, and shows the units label
            self.layoutWidget.setLabel('top', ' ', **styles)                    # shows axis at the top, no label, no tickmarks
            self.layoutWidget.setLabel('right', ' ', **styles)                  # shows axis at the right, no label, no tickmarks

        # add image, if available and required
        if self.layoutImItem is not None and self.imageType > 0:
            self.layoutImItem.setTransform(self.survey.cmpTransform * transform)   # combine two transforms
            self.layoutWidget.plotItem.addItem(self.layoutImItem)

        # add survey geometry if templates are to be displayed (controlled by checkbox)
        if self.tbTemplat.isChecked():
            surveyItem = self.survey
            surveyItem.setTransform(transform)                                  # always do this; will reset transform for 'local' plot
            self.layoutWidget.plotItem.addItem(surveyItem)                      # this plots the survey geometry

        # to add SPS data, i.e. point lists, please have a look at:
        # https://pyqtgraph.readthedocs.io/en/latest/api_reference/graphicsItems/plotitem.html#pyqtgraph.PlotItem.plot and :
        # https://pyqtgraph.readthedocs.io/en/latest/api_reference/graphicsItems/plotdataitem.html#pyqtgraph.PlotDataItem.__init__
        # https://pyqtgraph.readthedocs.io/en/latest/api_reference/graphicsItems/scatterplotitem.html#pyqtgraph.ScatterPlotItem.setSymbol
        # https://pyqtgraph.readthedocs.io/en/latest/api_reference/graphicsItems/plotdataitem.html

        # addItem(item, *args, **kargs,)
        # [source] https://pyqtgraph.readthedocs.io/en/latest/_modules/pyqtgraph/graphicsItems/PlotItem/PlotItem.html#PlotItem.addItem
        # Add a graphics item to the view box. If the item has plot data (PlotDataItem , PlotCurveItem , ScatterPlotItem ), it may be included in analysis performed by the PlotItem.

        if self.tbSpsList.isChecked() and self.spsLiveE is not None and self.spsLiveN is not None:
            spsTransform = QTransform()                                         # empty (unit) transform
            if not self.glob and self.survey.glbTransform is not None:          # global -> easting & westing
                spsTransform, _ = self.survey.glbTransform.inverted()

            spsLive = self.layoutWidget.plot(
                x=self.spsLiveE,
                y=self.spsLiveN,
                connect='all',
                pxMode=False,
                pen=None,
                symbol=config.spsPointSymbol,
                symbolPen=pg.mkPen('k'),
                symbolSize=config.spsSymbolSize,
                symbolBrush=QColor(config.spsBrushColor),
            )
            spsLive.setTransform(spsTransform)

        if self.tbSpsList.isChecked() and self.tbAllList.isChecked() and self.spsDeadE is not None and self.spsDeadN is not None:
            spsTransform = QTransform()                                         # empty (unit) transform
            if not self.glob and self.survey.glbTransform is not None:          # global -> easting & westing
                spsTransform, _ = self.survey.glbTransform.inverted()

            spsDead = self.layoutWidget.plot(
                x=self.spsDeadE,
                y=self.spsDeadN,
                connect='all',
                pxMode=False,
                pen=None,
                symbol=config.spsPointSymbol,
                symbolPen=pg.mkPen('k'),
                symbolSize=config.spsSymbolSize,
                symbolBrush=QColor(config.spsBrushColor),
            )
            spsDead.setTransform(spsTransform)

        if self.tbRpsList.isChecked() and self.rpsLiveE is not None and self.rpsLiveN is not None:
            rpsTransform = QTransform()                                         # empty (unit) transform
            if not self.glob and self.survey.glbTransform is not None:          # global -> easting & westing
                rpsTransform, _ = self.survey.glbTransform.inverted()

            rpsLive = self.layoutWidget.plot(
                x=self.rpsLiveE,
                y=self.rpsLiveN,
                connect='all',
                pxMode=False,
                pen=None,
                symbol=config.rpsPointSymbol,
                symbolPen=pg.mkPen('k'),
                symbolSize=config.rpsSymbolSize,
                symbolBrush=QColor(config.rpsBrushColor),
            )
            rpsLive.setTransform(rpsTransform)

        if self.tbRpsList.isChecked() and self.tbAllList.isChecked() and self.rpsDeadE is not None and self.rpsDeadN is not None:
            rpsTransform = QTransform()                                         # empty (unit) transform
            if not self.glob and self.survey.glbTransform is not None:          # global -> easting & westing
                rpsTransform, _ = self.survey.glbTransform.inverted()

            rpsDead = self.layoutWidget.plot(
                x=self.rpsDeadE,
                y=self.rpsDeadN,
                connect='all',
                pxMode=False,
                pen=None,
                symbol=config.rpsPointSymbol,
                symbolPen=pg.mkPen('k'),
                symbolSize=config.rpsSymbolSize,
                symbolBrush=QColor(config.rpsBrushColor),
            )
            rpsDead.setTransform(rpsTransform)

        if self.tbSrcList.isChecked() and self.srcLiveE is not None and self.srcLiveN is not None:
            srcTransform = QTransform()                                         # empty (unit) transform
            if not self.glob and self.survey.glbTransform is not None:          # global -> easting & westing
                srcTransform, _ = self.survey.glbTransform.inverted()

            srcLive = self.layoutWidget.plot(
                x=self.srcLiveE,
                y=self.srcLiveN,
                connect='all',
                pxMode=False,
                pen=None,
                symbol=config.srcPointSymbol,
                symbolPen=pg.mkPen('#bdbdbd'),
                symbolSize=config.srcSymbolSize,
                symbolBrush=QColor(config.srcBrushColor),
            )
            srcLive.setTransform(srcTransform)

        if self.tbSrcList.isChecked() and self.tbAllList.isChecked() and self.srcDeadE is not None and self.srcDeadN is not None:
            srcTransform = QTransform()                                         # empty (unit) transform
            if not self.glob and self.survey.glbTransform is not None:          # global -> easting & westing
                srcTransform, _ = self.survey.glbTransform.inverted()

            srcDead = self.layoutWidget.plot(
                x=self.srcDeadE,
                y=self.srcDeadN,
                connect='all',
                pxMode=False,
                pen=None,
                symbol=config.srcPointSymbol,
                symbolPen=pg.mkPen('#bdbdbd'),
                symbolSize=config.srcSymbolSize,
                symbolBrush=QColor(config.srcBrushGrey),
            )
            srcDead.setTransform(srcTransform)

        if self.tbRecList.isChecked() and self.recLiveE is not None and self.recLiveN is not None:
            recTransform = QTransform()                                         # empty (unit) transform
            if not self.glob and self.survey.glbTransform is not None:          # global -> easting & westing
                recTransform, _ = self.survey.glbTransform.inverted()

            recLive = self.layoutWidget.plot(
                x=self.recLiveE,
                y=self.recLiveN,
                connect='all',
                pxMode=False,
                pen=None,
                symbol=config.recPointSymbol,
                symbolPen=pg.mkPen('#bdbdbd'),
                symbolSize=config.recSymbolSize,
                symbolBrush=QColor(config.recBrushColor),
            )
            recLive.setTransform(recTransform)

        if self.tbRecList.isChecked() and self.tbAllList.isChecked() and self.recDeadE is not None and self.recDeadN is not None:
            recTransform = QTransform()                                         # empty (unit) transform
            if not self.glob and self.survey.glbTransform is not None:          # global -> easting & westing
                recTransform, _ = self.survey.glbTransform.inverted()

            recDead = self.layoutWidget.plot(
                x=self.recDeadE,
                y=self.recDeadN,
                connect='all',
                pxMode=False,
                pen=None,
                symbol=config.recPointSymbol,
                symbolPen=pg.mkPen('#bdbdbd'),
                symbolSize=config.recSymbolSize,
                symbolBrush=QColor(config.recBrushGrey),
            )
            recDead.setTransform(recTransform)

        if self.tbSpider.isChecked() and self.output.anaOutput is not None and self.output.binOutput is not None:
            if self.spiderSrcX is not None:                                     # if we have data to show, plot it

                src = self.layoutWidget.plot(
                    x=self.spiderSrcX, y=self.spiderSrcY, connect='pairs', symbol='o', pen=pg.mkPen('r', width=2), symbolSize=5, pxMode=False, symbolPen=pg.mkPen('r'), symbolBrush=QColor('#77FF2929')
                )
                src.setTransform(transform)

                rec = self.layoutWidget.plot(
                    x=self.spiderRecX, y=self.spiderRecY, connect='pairs', symbol='o', pen=pg.mkPen('b', width=2), symbolSize=5, pxMode=False, symbolPen=pg.mkPen('b'), symbolBrush=QColor('#772929FF')
                )
                rec.setTransform(transform)

            if self.spiderText is not None:
                self.layoutWidget.plotItem.addItem(self.spiderText)                 # show the spider label anyhow

        if self.tbTemplat.isChecked():
            # Add a marker for the origin
            oriX = [0.0]
            oriY = [0.0]
            orig = self.layoutWidget.plot(x=oriX, y=oriY, symbol='o', symbolSize=16, symbolPen=(0, 0, 0, 100), symbolBrush=(180, 180, 180, 100))
            orig.setTransform(transform)

        if self.survey.binning.method == BinningType.sphere:
            # Draw sphere as a circle in the plot for guidance when binning against a sphere
            # See: https://stackoverflow.com/questions/33525279/pyqtgraph-how-do-i-plot-an-ellipse-or-a-circle
            # See: https://doc.qt.io/qtforpython-5/PySide2/QtWidgets/QGraphicsEllipseItem.html
            r = self.survey.localSphere.radius
            x = self.survey.localSphere.origin.x() - r
            y = self.survey.localSphere.origin.y() - r
            w = r * 2.0
            h = r * 2.0
            sphereArea = QGraphicsEllipseItem(x, y, w, h)
            sphereArea.setPen(pg.mkPen(100, 100, 100))
            sphereArea.setBrush(QBrush(QColor(config.binAreaColor)))            # use same color as binning region
            sphereArea.setTransform(transform)
            self.layoutWidget.plotItem.addItem(sphereArea)

        if self.ruler:
            # add ruler if required
            p1 = pg.mkPen('r', style=Qt.DashLine)
            p2 = pg.mkPen('r', style=Qt.DashLine, width=2)
            p3 = pg.mkPen('b')
            p4 = pg.mkPen('b', width=3)

            # get default location for ruler, dependent on current viewRect
            viewRect = self.layoutWidget.plotItem.vb.viewRect()
            ptCenter = viewRect.center()
            pt1 = (ptCenter + viewRect.topLeft()) / 2.0
            pt2 = (ptCenter + viewRect.bottomRight()) / 2.0

            self.lineROI = LineROI([[pt1.x(), pt1.y()], [pt2.x(), pt2.y()]], pen=p1, hoverPen=p2, handlePen=p3, handleHoverPen=p4)
            if self.rulerState is not None:                                     # restore state, if possible
                self.lineROI.setState(self.rulerState)

            self.layoutWidget.plotItem.addItem(self.lineROI)
            self.lineROI.sigRegionChanged.connect(self.roiChanged)

            length = len(self.lineROI.getHandles()) + 1
            self.roiLabels = [pg.TextItem(anchor=(0.5, 1.3), border='b', color='b', fill=(130, 255, 255, 200), text='label') for _ in range(length)]

            for label in self.roiLabels:
                self.layoutWidget.plotItem.addItem(label)
                label.setZValue(1000)
            self.roiChanged()

    def plotOffTrk(self, nY: int, stkY: int, ox: float):
        with pg.BusyCursor():
            plotTitle = f'{self.plotTitles[1]} [line={stkY}]'

            slice3D = self.output.anaOutput[:, nY, :, :]
            slice2D = slice3D.reshape(slice3D.shape[0] * slice3D.shape[1], slice3D.shape[2])           # convert to 2D
            slice2D = numbaFilterSlice2D(slice2D, self.survey.unique.apply)

            self.offTrkWidget.plotItem.clear()
            self.offTrkWidget.setTitle(plotTitle, color='b', size='16pt')
            if slice2D.shape[0] == 0:                                           # empty array
                return

            x, y = numbaOffInline(slice2D, ox)
            self.offTrkWidget.plot(x=x, y=y, connect='pairs', pen=pg.mkPen('k', width=2))

    def plotOffBin(self, nX: int, stkX: int, oy: float):
        with pg.BusyCursor():
            self.offBinWidget.plotItem.clear()

            slice3D = self.output.anaOutput[nX, :, :, :]
            slice2D = slice3D.reshape(slice3D.shape[0] * slice3D.shape[1], slice3D.shape[2])           # convert to 2D
            slice2D = numbaFilterSlice2D(slice2D, self.survey.unique.apply)

            plotTitle = f'{self.plotTitles[2]} [stake={stkX}]'
            self.offBinWidget.setTitle(plotTitle, color='b', size='16pt')

            if slice2D.shape[0] == 0:                                           # empty array; nothing to see here...
                return

            x, y = numbaOffX_line(slice2D, oy)
            self.offBinWidget.plot(x=x, y=y, connect='pairs', pen=pg.mkPen('k', width=2))

    def plotAziTrk(self, nY: int, stkY: int, ox: float):
        with pg.BusyCursor():
            self.aziTrkWidget.plotItem.clear()

            slice3D = self.output.anaOutput[:, nY, :, :]
            slice2D = slice3D.reshape(slice3D.shape[0] * slice3D.shape[1], slice3D.shape[2])           # convert to 2D
            slice2D = numbaFilterSlice2D(slice2D, self.survey.unique.apply)

            plotTitle = f'{self.plotTitles[3]} [line={stkY}]'
            self.aziTrkWidget.setTitle(plotTitle, color='b', size='16pt')

            if slice2D.shape[0] == 0:                                           # empty array; nothing to see here...
                return

            x, y = numbaAziInline(slice2D, ox)
            self.aziTrkWidget.plot(x=x, y=y, connect='pairs', pen=pg.mkPen('k', width=2))

    def plotAziBin(self, nX: int, stkX: int, oy: float):
        with pg.BusyCursor():
            self.aziBinWidget.plotItem.clear()

            slice3D = self.output.anaOutput[nX, :, :, :]
            slice2D = slice3D.reshape(slice3D.shape[0] * slice3D.shape[1], slice3D.shape[2])           # convert to 2D
            slice2D = numbaFilterSlice2D(slice2D, self.survey.unique.apply)

            plotTitle = f'{self.plotTitles[4]} [stake={stkX}]'
            self.aziBinWidget.setTitle(plotTitle, color='b', size='16pt')
            if slice2D.shape[0] == 0:                                           # empty array; nothing to see here...
                return

            x, y = numbaAziX_line(slice2D, oy)
            self.aziBinWidget.plot(x=x, y=y, connect='pairs', pen=pg.mkPen('k', width=2))

    def plotStkTrk(self, nY: int, stkY: int, x0: float, dx: float):
        with pg.BusyCursor():
            dK = 0.001 * config.kr_Stack.z()
            kMax = 0.001 * config.kr_Stack.y() + dK
            kStart = 1000.0 * (0.0 - 0.5 * dK)                                  # scale by factor 1000 as we want to show [1/km] on scale
            kDelta = 1000.0 * dK                                                # same here

            slice3D, I = numbaSlice3D(self.output.anaOutput[:, nY, :, :], self.survey.unique.apply)
            if slice3D.shape[0] == 0:                                           # empty array; nothing to see here...
                return

            self.inlineStk = numbaNdft_1D(kMax, dK, slice3D, I)

            tr = QTransform()                                                   # prepare ImageItem transformation:
            tr.translate(x0, kStart)                                            # move image to correct location
            tr.scale(dx, kDelta)                                                # scale horizontal and vertical axes

            self.stkTrkImItem = pg.ImageItem()                                  # create PyqtGraph image item
            self.stkTrkImItem.setImage(self.inlineStk, levels=(-50.0, 0.0))     # plot with log scale from -50 to 0
            self.stkTrkImItem.setTransform(tr)

            if self.stkTrkColorBar is None:
                self.stkTrkColorBar = self.stkTrkWidget.plotItem.addColorBar(self.stkTrkImItem, colorMap=config.analysisCmap, label='dB attenuation', limits=(-100.0, 0.0), rounding=10.0, values=(-50.0, 0.0))
                self.stkTrkColorBar.setLevels(low=-50.0, high=0.0)
            else:
                self.stkTrkColorBar.setImageItem(self.stkTrkImItem)
                self.stkTrkColorBar.setColorMap(config.analysisCmap)            # in case the colorbar has been changed

            self.stkTrkWidget.plotItem.clear()
            self.stkTrkWidget.plotItem.addItem(self.stkTrkImItem)

            plotTitle = f'{self.plotTitles[5]} [line={stkY}]'
            self.stkTrkWidget.setTitle(plotTitle, color='b', size='16pt')

    def plotStkBin(self, nX: int, stkX: int, y0: float, dy: float):
        with pg.BusyCursor():
            dK = 0.001 * config.kr_Stack.z()
            kMax = 0.001 * config.kr_Stack.y() + dK
            kStart = 1000.0 * (0.0 - 0.5 * dK)                                  # scale by factor 1000 as we want to show [1/km] on scale
            kDelta = 1000.0 * dK                                                # same here

            slice3D, I = numbaSlice3D(self.output.anaOutput[nX, :, :, :], self.survey.unique.apply)
            if slice3D.shape[0] == 0:                                           # empty array; nothing to see here...
                return

            self.x_lineStk = numbaNdft_1D(kMax, dK, slice3D, I)

            tr = QTransform()                                                   # prepare ImageItem transformation:
            tr.translate(y0, kStart)                                            # move image to correct location
            tr.scale(dy, kDelta)                                                # scale horizontal and vertical axes

            self.stkBinImItem = pg.ImageItem()                                  # create PyqtGraph image item
            self.stkBinImItem.setImage(self.x_lineStk, levels=(-50.0, 0.0))     # plot with log scale from -50 to 0
            self.stkBinImItem.setTransform(tr)

            if self.stkBinColorBar is None:
                self.stkBinColorBar = self.stkBinWidget.plotItem.addColorBar(self.stkBinImItem, colorMap=config.analysisCmap, label='dB attenuation', limits=(-100.0, 0.0), rounding=10.0, values=(-50.0, 0.0))
                self.stkBinColorBar.setLevels(low=-50.0, high=0.0)
            else:
                self.stkBinColorBar.setImageItem(self.stkBinImItem)
                self.stkBinColorBar.setColorMap(config.analysisCmap)            # in case the colorbar has been changed

            self.stkBinWidget.plotItem.clear()
            self.stkBinWidget.plotItem.addItem(self.stkBinImItem)

            plotTitle = f'{self.plotTitles[6]} [stake={stkX}]'
            self.stkBinWidget.setTitle(plotTitle, color='b', size='16pt')

    def plotStkCel(self, nX: int, nY: int, stkX: int, stkY: int):
        with pg.BusyCursor():
            kMin = 0.001 * config.kxyStack.x()
            kMax = 0.001 * config.kxyStack.y()
            dK = 0.001 * config.kxyStack.z()
            kMax = kMax + dK

            kStart = 1000.0 * (kMin - 0.5 * dK)                                 # scale by factor 1000 as we want to show [1/km] on scale
            kDelta = 1000.0 * dK                                                # same here

            offsetX, offsetY, noData = numbaOffsetBin(self.output.anaOutput[nX, nY, :, :], self.survey.unique.apply)
            if noData:
                fold = 0
            else:
                fold = offsetX.shape[0]

            if offsetX is None:
                kX = np.arange(kMin, kMax, dK)
                nX = kX.shape[0]
                self.xyCellStk = np.ones(shape=(nX, nX), dtype=np.float32) * -50.0           # create -50 dB array of the right size and type
            else:
                self.xyCellStk = numbaNdft_2D(kMin, kMax, dK, offsetX, offsetY)

            i3 = self.pattern3.currentIndex() - 1                               # turn <no pattern> into -1
            i4 = self.pattern4.currentIndex() - 1
            imax = len(self.survey.patternList)

            if self.tbStackPatterns.isChecked() and i3 >= 0 and i3 < imax:
                x3, y3 = self.survey.patternList[i3].calcPatternPointArrays()
                self.xyCellStk = self.xyCellStk + numbaNdft_2D(kMin, kMax, dK, x3, y3)

            if self.tbStackPatterns.isChecked() and i4 >= 0 and i4 < imax:
                x4, y4 = self.survey.patternList[i4].calcPatternPointArrays()
                self.xyCellStk = self.xyCellStk + numbaNdft_2D(kMin, kMax, dK, x4, y4)

            tr = QTransform()                                               # prepare ImageItem transformation:
            tr.translate(kStart, kStart)                                    # move image to correct location
            tr.scale(kDelta, kDelta)                                        # scale horizontal and vertical axes

            self.stkCelImItem = pg.ImageItem()                              # create PyqtGraph image item
            self.stkCelImItem.setImage(self.xyCellStk, levels=(-50.0, 0.0))   # plot with log scale from -50 to 0
            self.stkCelImItem.setTransform(tr)
            if self.stkCelColorBar is None:
                self.stkCelColorBar = self.stkCelWidget.plotItem.addColorBar(self.stkCelImItem, colorMap=config.analysisCmap, label='dB attenuation', limits=(-100.0, 0.0), rounding=10.0, values=(-50.0, 0.0))
                self.stkCelColorBar.setLevels(low=-50.0, high=0.0)
            else:
                self.stkCelColorBar.setImageItem(self.stkCelImItem)
                self.stkCelColorBar.setColorMap(config.analysisCmap)        # in case the colorbar has been changed

            self.stkCelWidget.plotItem.clear()
            self.stkCelWidget.plotItem.addItem(self.stkCelImItem)

            plotTitle = f'{self.plotTitles[7]} [stake={stkX}, line={stkY}, fold={fold}]'
            self.stkCelWidget.setTitle(plotTitle, color='b', size='16pt')

    def plotOffset(self):
        with pg.BusyCursor():

            dO = 50.0                                                           # offsets increments
            oMax = ceil(self.output.maxMaxOffset / dO) * dO + dO                # max y-scale; make sure end value is included
            oR = np.arange(0, oMax, dO)                                         # numpy array with values [0 ... oMax]

            if self.output.offstHist is None:
                offsets, _, noData = numbaSliceStats(self.output.anaOutput, self.survey.unique.apply)
                if noData:
                    return

                y, x = np.histogram(offsets, bins=oR)                           # create a histogram with 100m offset increments

                y1 = np.append(y, 0)                                            # add a dummy value to make x- and y-arrays equal size
                self.output.offstHist = np.stack((x, y1))                       # See: https://numpy.org/doc/stable/reference/generated/numpy.stack.html#numpy.stack

            x2 = self.output.offstHist[0, :]                                    # x in the top row
            y2 = self.output.offstHist[1, :-1]                                  # y in the bottom row, minus the phony last value

            count = np.sum(self.output.binOutput)                               # available traces
            plotTitle = f'{self.plotTitles[8]} [{count:,} traces]'
            self.offsetWidget.setTitle(plotTitle, color='b', size='16pt')
            self.offsetWidget.plotItem.clear()
            self.offsetWidget.plot(x2, y2, stepMode='center', fillLevel=0, fillOutline=True, brush=(0, 0, 255, 150), pen=pg.mkPen('k', width=1))

    def plotOffAzi(self):
        with pg.BusyCursor():
            dA = 5.0                                                            # azimuth increments
            dO = 100.0                                                          # offsets increments

            aMin = -180.0                                                       # max x-scale
            aMax = 180.0                                                        # max x-scale
            aMax += dA                                                          # make sure end value is included

            oMax = ceil(self.output.maxMaxOffset / dO) * dO + dO                # max y-scale; make sure end value is included

            if self.output.ofAziHist is None:                                   # calculate offset/azimuth distribution
                offsets, azimuth, noData = numbaSliceStats(self.output.anaOutput, self.survey.unique.apply)
                if noData:
                    return

                aR = np.arange(aMin, aMax, dA)                                  # numpy array with values [0 ... fMax]
                oR = np.arange(0, oMax, dO)                                     # numpy array with values [0 ... oMax]
                self.output.ofAziHist = np.histogram2d(x=azimuth, y=offsets, bins=[aR, oR], range=None, density=None, weights=None)[0]

            tr = QTransform()                                                   # prepare ImageItem transformation:
            tr.translate(aMin, 0)                                               # move image to correct location
            tr.scale(dA, dO)                                                    # scale horizontal and vertical axes

            self.offAziImItem = pg.ImageItem()                                  # create PyqtGraph image item
            self.offAziImItem.setImage(self.output.ofAziHist)
            self.offAziImItem.setTransform(tr)
            if self.offAziColorBar is None:
                self.offAziColorBar = self.offAziWidget.plotItem.addColorBar(self.offAziImItem, colorMap=config.analysisCmap, label='frequency', rounding=10.0)
                self.offAziColorBar.setLevels(low=0.0)                          # , high=0.0
            else:
                self.offAziColorBar.setImageItem(self.offAziImItem)
                self.offAziColorBar.setColorMap(config.analysisCmap)            # in case the colorbar has been changed

            self.offAziWidget.plotItem.clear()
            self.offAziWidget.plotItem.addItem(self.offAziImItem)

            count = np.sum(self.output.binOutput)                               # available traces
            plotTitle = f'{self.plotTitles[9]} [{count:,} traces]'
            self.offAziWidget.setTitle(plotTitle, color='b', size='16pt')

        # For Polar Coordinates, see:
        # See: https://stackoverflow.com/questions/57174173/polar-coordinate-system-in-pyqtgraph
        # See: https://groups.google.com/g/pyqtgraph/c/9Vv1kJdxE6U/m/FuCsSg182jUJ
        # See: https://doc.qt.io/qtforpython-6/PySide6/QtCharts/QPolarChart.html
        # See: https://www.youtube.com/watch?v=DyPjsj6azY4
        # See: https://stackoverflow.com/questions/50720719/how-to-create-a-color-circle-in-pyqt
        # See: https://stackoverflow.com/questions/70471687/pyqt-creating-color-circle

    def plotPatterns(self):

        self.arraysWidget.plotItem.clear()
        self.arraysWidget.setTitle(self.plotTitles[10], color='b', size='16pt')
        self.arraysWidget.showAxes(True, showValues=(True, False, False, True))   # show values at the left and at the bottom

        styles = {'color': '#000', 'font-size': '10pt'}
        self.arraysWidget.setLabel('top', ' ', **styles)                        # shows axis at the top, no label, no tickmarks
        self.arraysWidget.setLabel('right', ' ', **styles)                      # shows axis at the right, no label, no tickmarks

        i1 = self.pattern1.currentIndex() - 1                                   # turn <no pattern> into -1
        i2 = self.pattern2.currentIndex() - 1
        imax = len(self.survey.patternList)

        if self.patternLayout:                                                  # display the layout
            self.arraysWidget.setLabel('bottom', 'inline', units='m', **styles)   # shows axis at the bottom, and shows the units label
            self.arraysWidget.setLabel('left', 'crossline', units='m', **styles)  # shows axis at the left, and shows the units label

            if i1 >= 0 and i1 < imax:
                self.arraysWidget.plotItem.addItem(self.survey.patternList[i1])

            if i2 >= 0 and i2 < imax:
                self.arraysWidget.plotItem.addItem(self.survey.patternList[i2])

        else:                                                                   # calculate kxky pattern response of selected patterns
            self.arraysWidget.setLabel('bottom', 'Kx', units='1/km', **styles)  # shows axis at the bottom, and shows the units label
            self.arraysWidget.setLabel('left', 'Ky', units='1/km', **styles)    # shows axis at the left, and shows the units label

            with pg.BusyCursor():                                               # now do the real work
                kMin = 0.001 * config.kxyArray.x()
                kMax = 0.001 * config.kxyArray.y()
                dK = 0.001 * config.kxyArray.z()
                kMax = kMax + dK

                kStart = 1000.0 * (kMin - 0.5 * dK)                             # scale by factor 1000 as we want to show [1/km] on scale
                kDelta = 1000.0 * dK                                            # same here

                x1 = y1 = x2 = y2 = None                                        # avoid variables from being undefined
                kX = np.arange(kMin, kMax, dK)
                nX = kX.shape[0]

                if (x1 is None or len(x1) == 0) and (x2 is None or len(x2) == 0):
                    self.xyPatResp = np.ones(shape=(nX, nX), dtype=np.float32) * -50.0  # create -50 dB array of the right size and type
                else:
                    self.xyPatResp = np.zeros(shape=(nX, nX), dtype=np.float32)   # create zero array of the right size and type

                    if i1 >= 0 and i1 < imax:
                        x1, y1 = self.survey.patternList[i1].calcPatternPointArrays()
                        self.xyPatResp = self.xyPatResp + numbaNdft_2D(kMin, kMax, dK, x1, y1)

                    if i2 >= 0 and i2 < imax:
                        x2, y2 = self.survey.patternList[i2].calcPatternPointArrays()
                        self.xyPatResp = self.xyPatResp + numbaNdft_2D(kMin, kMax, dK, x2, y2)

                tr = QTransform()                                               # prepare ImageItem transformation:
                tr.translate(kStart, kStart)                                    # move image to correct location
                tr.scale(kDelta, kDelta)                                        # scale horizontal and vertical axes

                self.kxyPatImItem = pg.ImageItem()                              # create PyqtGraph image item
                self.kxyPatImItem.setImage(self.xyPatResp, levels=(-50.0, 0.0))   # plot with log scale from -50 to 0
                self.kxyPatImItem.setTransform(tr)
                if self.kxyPatColorBar is None:
                    self.kxyPatColorBar = self.arraysWidget.plotItem.addColorBar(
                        self.kxyPatImItem, colorMap=config.analysisCmap, label='dB attenuation', limits=(-100.0, 0.0), rounding=10.0, values=(-50.0, 0.0)
                    )
                    self.kxyPatColorBar.setLevels(low=-50.0, high=0.0)
                else:
                    self.kxyPatColorBar.setImageItem(self.kxyPatImItem)
                    self.kxyPatColorBar.setColorMap(config.analysisCmap)        # in case the colorbar has been changed

                self.arraysWidget.plotItem.clear()
                self.arraysWidget.plotItem.addItem(self.kxyPatImItem)

        plotTitle = f'{self.plotTitles[10]} [{self.pattern1.currentText()} * {self.pattern2.currentText()}]'
        plotTitle = plotTitle.replace('<', '&lt;')                              # bummer; plotTitle is an html string
        plotTitle = plotTitle.replace('>', '&gt;')                              # we need to escape the angle brackets

        self.arraysWidget.setTitle(plotTitle, color='b', size='16pt')

    def roiChanged(self):
        pos = []
        for i, handle in enumerate(self.lineROI.getHandles()):
            handlePos = self.lineROI.pos() + handle.pos()
            self.roiLabels[i].setPos(handlePos)
            pos.append(handlePos)
            self.roiLabels[i].setText(f'({handlePos[0]:.2f}, {handlePos[1]:.2f})')

        # put label in the middle of the line
        pos2 = (pos[0] + pos[1]) / 2.0
        diff = pos[1] - pos[0]
        self.roiLabels[2].setPos(pos2)
        self.roiLabels[2].setText(f'|r|={diff.length():.2f}, Ø={degrees(atan2(diff.y(),diff.x())):.2f}°')
        self.rulerState = self.lineROI.saveState()

    def closeEvent(self, e):  # main window about to be closed event
        # See: https://doc.qt.io/qt-6/qwidget.html#closeEvent
        # See: https://stackoverflow.com/questions/22460003/pyqts-qmainwindow-closeevent-is-never-called

        if self.fileNew():                                                      # file (maybe) saved and cancel NOT used
            self.dockLogging.setFloating(False)                                 # don't keep floating docking widgets hanging araound once closed
            self.dockDisplay.setFloating(False)                                 # don't keep floating docking widgets hanging araound once closed
            self.dockProperty.setFloating(False)                                # don't keep floating docking widgets hanging araound once closed
            # self.writeSettings()                                              # save geometry and state of window(s)
            writeSettings(self)                                                 # save geometry and state of window(s)

            if self.workingDirectory:                                           # append information to log file in working directory
                logFile = os.path.join(self.workingDirectory, '.roll.log')      # join directory & log file name
                with open(logFile, 'a', encoding='utf-8') as file:              # append information to logfile
                    file.write(self.logEdit.toPlainText())                      # get text from logEdit
                    file.write('+++\n\n')                                       # closing remarks

            # builtins.print = self.oldPrint                                    # restore builtins.print
            # sys.excepthook = self.oldExceptHook                               # restore sys.excepthook back to the original
            e.accept()                                                          # finally accep the event

            self.killMe = True                                                  # to restart the GUI from scratch when the plugin is activated again

        else:
            e.ignore()                                                          # ignore the event and stay active

        # See: https://stackoverflow.com/questions/26114034/defining-a-wx-panel-destructor-in-wxpython/73972953#73972953
        # See: http://enki-editor.org/2014/08/23/Pyqt_mem_mgmt.html

        # The following code is already done in self.fileNew()
        # if self.thread is not None and self.thread.isRunning():
        #     reply = QMessageBox.question(self, 'Please confirm',
        #         "Cancel work in progress and close Roll ?",
        #         QMessageBox.Yes, QMessageBox.Cancel)

        #     if reply == QMessageBox.Yes:
        #         self.thread.requestInterruption()
        #         self.thread.quit()
        #         self.thread.wait()
        #         # force thread termination
        #         # self.thread.terminate()
        #         self.thread.deleteLater()
        #         e.accept()
        #     else:
        #         e.ignore()                                                      # ignore the event and stay active

    def newFile(self):                                                          # wrapper around fileNew; used to create a log message
        if self.fileNew():
            self.appendLogMessage('Created: new file')
            self.plotLayout()                                                   # update the survey, but not the xml-tab

    def resetNumpyArraysAndModels(self):
        """reset various analysis arrays"""

        # numpy binning arrays
        self.layoutImg = None                                                   # numpy array to be displayed; binOutput / minOffset / maxOffset

        # analysis numpy arrays
        self.inlineStk = None                                                   # numpy array with inline Kr stack reponse
        self.x_lineStk = None                                                   # numpy array with x_line Kr stack reponse
        self.xyCellStk = None                                                   # numpy array with cell's KxKy stack response
        self.xyPatResp = None                                                   # numpy array with pattern's KxKy response

        # layout and analysis image-items
        self.layoutImItem = None                                                # pg ImageItems showing analysis result
        self.stkTrkImItem = None
        self.stkBinImItem = None
        self.stkCelImItem = None
        self.offAziImItem = None
        self.kxyPatImItem = None

        # corresponding color bars
        # self.layoutColorBar = None                                              # DON'T reset these; adjust the existing ones
        # self.stkTrkColorBar = None
        # self.stkBinColorBar = None
        # self.stkCelColorBar = None
        # self.offAziColorBar = None

        # rps, sps, xps input arrays
        self.rpsImport = None                                                   # numpy array with list of RPS records
        self.spsImport = None                                                   # numpy array with list of SPS records
        self.xpsImport = None                                                   # numpy array with list of XPS records

        self.rpsLiveE = None                                                    # numpy array with list of live RPS coordinates
        self.rpsLiveN = None                                                    # numpy array with list of live RPS coordinates
        self.rpsDeadE = None                                                    # numpy array with list of dead RPS coordinates
        self.rpsDeadN = None                                                    # numpy array with list of dead RPS coordinates

        self.spsLiveE = None                                                    # numpy array with list of live SPS coordinates
        self.spsLiveN = None                                                    # numpy array with list of live SPS coordinates
        self.spsDeadE = None                                                    # numpy array with list of dead SPS coordinates
        self.spsDeadN = None                                                    # numpy array with list of dead SPS coordinates

        # rel, src, rel input arrays
        self.recGeom = None                                                     # numpy array with list of REC records
        self.srcGeom = None                                                     # numpy array with list of SRC records
        self.relGeom = None                                                     # numpy array with list of REL records

        self.recLiveE = None                                                    # numpy array with list of live REC coordinates
        self.recLiveN = None                                                    # numpy array with list of live REC coordinates
        self.recDeadE = None                                                    # numpy array with list of dead REC coordinates
        self.recDeadN = None                                                    # numpy array with list of dead REC coordinates

        self.srcLiveE = None                                                    # numpy array with list of live SRC coordinates
        self.srcLiveN = None                                                    # numpy array with list of live SRC coordinates
        self.srcDeadE = None                                                    # numpy array with list of dead SRC coordinates
        self.srcDeadN = None                                                    # numpy array with list of dead SRC coordinates

        # spider plot settings
        self.spiderPoint = QPoint(-1, -1)                                       # spider point 'out of scope'
        self.spiderSrcX = None                                                  # numpy array with list of SRC part of spider plot
        self.spiderSrcY = None                                                  # numpy array with list of SRC part of spider plot
        self.spiderRecX = None                                                  # numpy array with list of REC part of spider plot
        self.spiderRecY = None                                                  # numpy array with list of REC part of spider plot
        self.spiderText = None                                                  # text label describing spider bin, stake, fold
        self.actionSpider.setChecked(False)                                     # reset spider plot to 'off'

        # export layers to QGIS
        self.spsLayer = None                                                    # QGIS layer for sps point I/O
        self.rpsLayer = None                                                    # QGIS layer for rpr point I/O
        self.srcLayer = None                                                    # QGIS layer for src point I/O
        self.recLayer = None                                                    # QGIS layer for rec point I/O
        self.spsField = None                                                    # QGIS field for sps point selection I/O
        self.rpsField = None                                                    # QGIS field for rps point selection I/O
        self.srcField = None                                                    # QGIS field for src point selection I/O
        self.recField = None                                                    # QGIS field for rec point selection I/O

        # ruler settings
        self.lineROI = None                                                     # the ruler's dotted line
        self.roiLabels = None                                                   # the ruler's three labels
        self.rulerState = None                                                  # ruler's state, used to redisplay ruler at last used location

        self.anaModel.setData(None)                                             # update the trace table model

        self.rpsModel.setData(self.rpsImport)                                   # update the three rps/sps/xps models
        self.spsModel.setData(self.spsImport)
        self.xpsModel.setData(self.xpsImport)

        self.recModel.setData(self.recGeom)                                     # update the three rec/rel/src models
        self.relModel.setData(self.relGeom)
        self.srcModel.setData(self.srcGeom)

        self.output.binOutput = None
        self.output.minOffset = None
        self.output.maxOffset = None
        self.output.rmsOffset = None
        self.output.ofAziHist = None
        self.output.offstHist = None

        if self.output.anaOutput is not None:                                   # remove memory mapped file, as well
            self.output.D2_Output = None                                        # flattened reference to self.output.anaOutput

            self.output.anaOutput.flush()                                       # make sure all data is written to disk
            del self.output.anaOutput                                           # try to delete the object
            self.output.anaOutput = None                                        # the object was deleted; reinstate the None version
            gc.collect()                                                        # get the garbage collector going

        self.resetPlotWidget(self.offTrkWidget, self.plotTitles[1])             # clear all analysis plots
        self.resetPlotWidget(self.offBinWidget, self.plotTitles[2])
        self.resetPlotWidget(self.aziTrkWidget, self.plotTitles[3])
        self.resetPlotWidget(self.aziBinWidget, self.plotTitles[4])
        self.resetPlotWidget(self.stkTrkWidget, self.plotTitles[5])
        self.resetPlotWidget(self.stkBinWidget, self.plotTitles[6])
        self.resetPlotWidget(self.stkCelWidget, self.plotTitles[7])
        self.resetPlotWidget(self.offsetWidget, self.plotTitles[8])
        self.resetPlotWidget(self.offAziWidget, self.plotTitles[9])
        self.resetPlotWidget(self.arraysWidget, self.plotTitles[10])

        self.updateMenuStatus(True)                                             # keep menu status in sync with program's state; and reset analysis figure

    def fileNew(self):                                                          # better create new file created through a wizard
        if self.maybeKillThread() and self.maybeSave():                         # make sure thread is killed AND current file  is saved (all only when needed)
            self.resetNumpyArraysAndModels()                                    # empty all arrays and reset plot titles

            # start defining new survey
            self.parseText(exampleSurveyXmlText())                              # read & parse xml string and create new survey object
            self.textEdit.setPlainText(exampleSurveyXmlText())                  # copy xml content to text edit control
            self.resetSurveyProperties()                                        # get the new parameters into the parameter tree
            self.textEdit.moveCursor(QTextCursor.Start)                         # move cursor to front
            self.survey.calcTransforms()                                        # (re)calculate the transforms being used
            self.survey.calcSeedData()                                          # needed for circles, spirals & well-seeds; may affect bounding box
            self.survey.calcBoundingRect()                                      # (re)calculate the boundingBox as part of parsing the data
            self.survey.calcNoShotPoints()                                      # (re)calculate nr of SPs

            self.setCurrentFileName()                                           # update self.fileName, set textEditModified(False) and setWindowModified(False)

            self.actionProjected.setChecked(False)                              # set to 'local' plotting (not global)
            self.plotProjected()                                                # enforce 'local' plotting and plotLayout()

            return True                                                         # we emptied the document, and reset the survey object
        else:
            return False                                                        # user had 2nd thoughts and did not close the document

    def fileNewLandSurvey(self):
        if not self.fileNew():                                                  # user had 2nd thoughts and did not close the document; return False
            return False

        dlg = LandSurveyWizard(self)

        if dlg.exec():                                                          # Run the dialog event loop, and obtain survey object
            self.survey = dlg.survey                                            # get survey from dialog

            plainText = self.survey.toXmlString()                               # convert the survey object to an Xml string
            self.textEdit.highlighter = XMLHighlighter(self.textEdit.document())  # we want some color highlighteded text
            self.textEdit.setFont(QFont('Ubuntu Mono', 9, QFont.Normal))        # the font may have been messed up by the initial html text

            self.textEdit.setTextViaCursor(plainText)                           # get text into the textEdit, NOT resetting its doc status
            self.UpdateAllViews()                                               # parse the textEdit; show the corresponding plot
            self.resetSurveyProperties()                                        # get the new parameters into the parameter tree

            self.appendLogMessage(f'Wizard : created land survey: {self.survey.name}')
            config.surveyNumber += 1                                            # update global counter

            if self.debug:
                self.appendLogMessage('Land Survey Wizard profiling information', MsgType.Debug)
                i = 0
                while i < len(self.dlg.survey.timerTmin):                       # log some debug messages
                    tMin = self.dlg.survey.timerTmin[i] * 1000.0 if self.worker.survey.timerTmin[i] != float('Inf') else 0.0
                    tMax = self.dlg.survey.timerTmax[i] * 1000.0
                    tTot = self.dlg.survey.timerTtot[i] * 1000.0
                    freq = self.dlg.survey.timerFreq[i]
                    tAvr = tTot / freq if freq > 0 else 0.0
                    message = f'{i:02d}: min:{tMin:011.3f}, max:{tMax:011.3f}, tot:{tTot:011.3f}, avr:{tAvr:011.3f}, freq:{freq:07d}'
                    self.appendLogMessage(message, MsgType.Debug)
                    i += 1

    def fileNewMarineSurvey(self):
        QMessageBox.information(self, 'Not implemented', 'The marine wizard has not yet been implemented', QMessageBox.Cancel)
        return

    def maybeSave(self):
        if not self.textEdit.document().isModified():                           # no need to do anything, as the doc wasn't modified
            return True

        ret = QMessageBox.warning(self, 'Roll', 'The document has been modified.\nDo you want to save your changes?', QMessageBox.Save | QMessageBox.Discard | QMessageBox.Cancel)        # want to save changes ?

        if ret == QMessageBox.Save:                                             # yes please; try to save changes
            return self.fileSave()                                              # if not succesfull, return False

        if ret == QMessageBox.Cancel:                                           # user 2nd thoughts ? return False
            return False

        return True                                                             # we're done dealing with the current document

    def maybeKillThread(self) -> bool:
        if self.thread is not None and self.thread.isRunning():
            reply = QMessageBox.question(self, 'Please confirm', 'Cancel work in progress and lose results ?', QMessageBox.Yes, QMessageBox.Cancel)

            if reply == QMessageBox.Cancel:
                return False
            else:
                self.thread.requestInterruption()
                self.thread.quit()
                self.thread.wait()

        # by now the thread has finished, so clean up and return 'True'
        self.worker = None                                                      # moveToThread object
        self.thread = None                                                      # corresponding worker thread

        self.hideStatusbarWidgets()                                             # remove temporary widgets from statusbar (don't kill 'm)

        self.layoutImg = None                                                   # numpy array to be displayed
        self.layoutImItem = None                                                # pg ImageItem showing analysis result

        self.updateMenuStatus(True)                                             # keep menu status in sync with program's state; and reset analysis figure
        self.handleImageSelection()                                             # update the colorbar accordingly

        return True

    def setCurrentFileName(self, fileName=''):                                  # update self.fileName, set textEditModified(False) and setWindowModified(False)
        self.fileName = fileName
        self.textEdit.document().setModified(False)
        # print(f'called: {whoamI()}() from: {callerName()}(), line {lineNo()}, filename = "{self.fileName}", isWindowModified = {self.isWindowModified()}')

        if not self.fileName:                                                   # filename ="" normally indicates working with 'new' file !
            shownName = self.survey.name
        else:
            shownName = QFileInfo(fileName).fileName()

            try:                                                                # if it is already somewhere in the MRU list, remove it
                self.recentFileList.remove(fileName)
            except ValueError:
                pass

            self.recentFileList.insert(0, fileName)                             # insert it at the top
            del self.recentFileList[config.maxRecentFiles :]                    # make sure the list does not overgrow

            self.updateRecentFileActions()

        self.setWindowTitle(self.tr(f'{shownName}[*] - Roll Survey'))           # update window name, with optional * for modified status
        self.setWindowModified(False)                                           # reset document status

    def updateRecentFileActions(self):                                          # update the MRU file menu actions
        numRecentFiles = min(len(self.recentFileList), config.maxRecentFiles)   # get actual number of recent files

        for i in range(numRecentFiles):
            fileName = self.recentFileList[i]
            showName = QFileInfo(fileName).fileName()
            text = f'&{i + 1} {showName}'
            self.recentFileActions[i].setText(text)
            self.recentFileActions[i].setData(self.recentFileList[i])
            self.recentFileActions[i].setVisible(True)

        for j in range(numRecentFiles, config.maxRecentFiles):
            self.recentFileActions[j].setVisible(False)

    def fileLoad(self, fileName):

        file = QFile(fileName)
        if not file.open(QFile.ReadOnly | QFile.Text):                          # report status message and return False
            try:                                                                # remove from MRU in case of errors
                self.recentFileList.remove(fileName)
            except ValueError:
                pass

            self.appendLogMessage(f'Open&nbsp;&nbsp;&nbsp;: Cannot open file:{fileName}. Error:{file.errorString()}', MsgType.Error)
            return False
        self.survey = RollSurvey()                                              # reset the survey object; get rid of all blocks in the list !
        self.appendLogMessage(f'Opened : {fileName}')                           # send status message
        self.setCurrentFileName(fileName)                                       # update self.fileName, set textEditModified(False) and setWindowModified(False)

        stream = QTextStream(file)                                              # create a stream to read all the data
        plainText = stream.readAll()                                            # load text in a string
        file.close()                                                            # file object no longer needed

        # Xml tab
        success = self.parseText(plainText)                                     # parse the string; load the textEdit even if parsing fails !
        self.textEdit.setPlainText(plainText)                                   # update plainText widget, and reset undo/redo & modified status
        self.resetNumpyArraysAndModels()                                        # empty all arrays and reset plot titles

        if success:                                                             # read the corresponding analysis files
            # continue loading the anaysis files that belong to this project

            w = self.survey.output.rctOutput.width()                            # expected dimensions of analysis files
            h = self.survey.output.rctOutput.height()
            dx = self.survey.grid.binSize.x()
            dy = self.survey.grid.binSize.y()
            nx = ceil(w / dx)
            ny = ceil(h / dy)

            if os.path.exists(self.fileName + '.bin.npy'):                      # open the existing foldmap file
                self.output.binOutput = np.load(self.fileName + '.bin.npy')
                nX = self.output.binOutput.shape[0]                             # check against nx
                nY = self.output.binOutput.shape[1]                             # check against ny

                if nx != nX or ny != nY:
                    self.appendLogMessage('Loaded : . . . Fold map&nbsp; : Wrong dimensions, compared to analysis area - file ignored')
                    self.output.binOutput = None
                else:
                    self.output.maximumFold = self.output.binOutput.max()           # calc min/max fold is straightforward
                    self.output.minimumFold = self.output.binOutput.min()

                    # todo: clean this up, so image handling isn't required here !
                    self.actionFold.setChecked(True)

                    self.imageType = 1                                              # set analysis type to one (fold)
                    self.layoutImg = self.survey.output.binOutput                   # use fold map for image data np-array
                    self.layoutMax = self.output.maximumFold                        # use appropriate maximum
                    self.layoutImItem = pg.ImageItem()                              # create PyqtGraph image item
                    self.layoutImItem.setImage(self.layoutImg, levels=(0.0, self.layoutMax))

                    label = 'fold'
                    if self.layoutColorBar is None:
                        self.layoutColorBar = self.layoutWidget.plotItem.addColorBar(self.layoutImItem, colorMap=config.fold_OffCmap, label=label, limits=(0, None), rounding=10.0, values=(0.0, self.layoutMax))
                    else:
                        self.layoutColorBar.setImageItem(self.layoutImItem)
                        self.layoutColorBar.setLevels(low=0.0, high=self.layoutMax)
                        self.layoutColorBar.setColorMap(config.fold_OffCmap)
                        self.setColorbarLabel(label)

                    self.appendLogMessage(f'Loaded : . . . Fold map&nbsp; : Min:{self.output.minimumFold} - Max:{self.output.maximumFold} ')
            else:
                self.output.binOutput = None
                # todo: cleanup next 2 lines as well
                self.actionNone.setChecked(True)
                self.imageType = 0                                              # set analysis type to zero (no analysis)

            if os.path.exists(self.fileName + '.min.npy'):                      # load the existing min-offsets file
                self.output.minOffset = np.load(self.fileName + '.min.npy')
                nX = self.output.minOffset.shape[0]                             # check against nx, ny
                nY = self.output.minOffset.shape[1]                             # check against nx, ny

                if nx != nX or ny != nY:
                    self.appendLogMessage('Loaded : . . . Min-offset: Wrong dimensions, compared to analysis area - file ignored')
                    self.output.minOffset = None
                else:
                    self.output.minOffset[self.output.minOffset == np.NINF] = np.Inf    # replace (-inf) by (inf) for min values
                    self.output.minMinOffset = self.output.minOffset.min()          # calc min offset against max (inf) values

                    self.output.minOffset[self.output.minOffset == np.Inf] = np.NINF  # replace (inf) by (-inf) for max values
                    self.output.maxMinOffset = self.output.minOffset.max()          # calc max values against (-inf) minimum
                    self.output.maxMinOffset = max(self.output.maxMinOffset, 0)     # avoid -inf as maximum

                    self.appendLogMessage(f'Loaded : . . . Min-offset: Min:{self.output.minMinOffset:.2f}m - Max:{self.output.maxMinOffset:.2f}m ')
            else:
                self.output.minOffset = None

            if os.path.exists(self.fileName + '.max.npy'):                      # load the existing max-offsets file
                self.output.maxOffset = np.load(self.fileName + '.max.npy')
                nX = self.output.maxOffset.shape[0]                             # check against nx, ny
                nY = self.output.maxOffset.shape[1]                             # check against nx, ny

                if nx != nX or ny != nY:
                    self.appendLogMessage('Loaded : . . . Max-offset: Wrong dimensions, compared to analysis area - file ignored')
                    self.output.maxOffset = None
                else:
                    self.output.maxMaxOffset = self.output.maxOffset.max()          # calc max offset against max (-inf) values
                    self.output.maxMaxOffset = max(self.output.maxMaxOffset, 0)     # avoid -inf as maximum
                    self.output.maxOffset[self.output.maxOffset == np.NINF] = np.inf   # replace (-inf) by (inf) for min values

                    self.output.minMaxOffset = self.output.maxOffset.min()          # calc min offset against min (inf) values
                    self.output.maxOffset[self.output.maxOffset == np.Inf] = np.NINF   # replace (inf) by (-inf) for max values
                    self.appendLogMessage(f'Loaded : . . . Max-offset: Min:{self.output.minMaxOffset:.2f}m - Max:{self.output.maxMaxOffset:.2f}m ')
            else:
                self.output.maxOffset = None

            if os.path.exists(self.fileName + '.rms.npy'):                      # load the existing max-offsets file
                self.output.rmsOffset = np.load(self.fileName + '.rms.npy')
                nX = self.output.rmsOffset.shape[0]                             # check against nx, ny
                nY = self.output.rmsOffset.shape[1]                             # check against nx, ny

                if nx != nX or ny != nY:
                    self.appendLogMessage('Loaded : . . . Rms-offset: Wrong dimensions, compared to analysis area - file ignored')
                    self.output.rmsOffset = None
                else:
                    self.output.maxRmsOffset = self.output.rmsOffset.max()          # calc max offset against max (-inf) values
                    self.output.minRmsOffset = self.output.rmsOffset.min()          # calc min offset against min (inf) values
                    self.output.minRmsOffset = max(self.output.minRmsOffset, 0)     # avoid -inf as maximum
                    self.appendLogMessage(f'Loaded : . . . Rms-offset: Min:{self.output.minRmsOffset:.2f}m - Max:{self.output.maxRmsOffset:.2f}m ')
            else:
                self.output.rmsOffset = None

            ##### 4/9/24

            if os.path.exists(self.fileName + '.off.npy'):                      # load the existing azimuth/offset histogram file
                self.output.offstHist = np.load(self.fileName + '.off.npy')
                nX = self.output.offstHist.shape[0]                             # check against nx
                # nY = self.output.offstHist.shape[1]                             # check against ny

                if nX != 2:
                    self.appendLogMessage('Loaded : . . . offset: Wrong dimensions of histogram - file ignored')
                    self.output.offstHist = None
                else:
                    self.appendLogMessage('Loaded : . . . offset histogram')
            else:
                self.output.offstHist = None

            if os.path.exists(self.fileName + '.azi.npy'):                      # load the existing azimuth/offset histogram file
                self.output.ofAziHist = np.load(self.fileName + '.azi.npy')
                nX = self.output.ofAziHist.shape[0]                             # check against nx
                # nY = self.output.ofAziHist.shape[1]                             # check against ny

                if nX != 360 // 5:
                    self.appendLogMessage('Loaded : . . . azi-offset: Wrong dimensions of histogram - file ignored')
                    self.output.ofAziHist = None
                else:
                    self.appendLogMessage('Loaded : . . . azi-offset histogram')
            else:
                self.output.ofAziHist = None

            if self.output.binOutput is not None and os.path.exists(self.fileName + '.ana.npy'):   # only open the analysis file if binning file exists
                try:

                    if self.survey.grid.fold > 0:
                        fold = self.survey.grid.fold                            # fold is defined by the grid's fold (preferred)
                    else:
                        fold = self.maximumFold                                 # fold is defined by observed maxfold in bin file

                    ### if we had a large memmap file open earlier; close it and call the garbage collector
                    if self.output.anaOutput is not None:
                        self.output.D2_Output = None                            # remove reference to self.output.anaOutput
                        del self.output.anaOutput                               # delete self.output.anaOutput array with detailed results
                        self.output.anaOutput = None                            # remove self.output.anaOutput itself
                        gc.collect()                                            # start the garbage collector to free up some space

                    # self.output.anaOutput = np.lib.format.open_memmap(self.fileName + '.ana.npy', mode='r+', dtype=np.float32, shape=None)
                    self.output.anaOutput = np.memmap(self.fileName + '.ana.npy', dtype=np.float32, mode='r+', shape=(nx, ny, fold, 13))
                    nT = self.output.anaOutput.size                             # total size of array in Nr of elements

                    # we know the (supposed) size of the binning area, and nr of columns in the file. Therefore
                    delta = nT - (nx * ny * fold * 13)

                    if delta != 0:
                        self.appendLogMessage(f'Loaded : . . . Analysis &nbsp;: mismatch in trace table compared to fold {fold:,} x-size {nx}, and y-size {ny}. Please rerun extended analysis', MsgType.Error)
                        self.output.D2_Output = None                            # remove reference to self.output.anaOutput
                        self.output.anaOutput = None                            # remove self.output.anaOutput itself
                        self.anaModel.setData(None)                             # use this as the model data
                    else:
                        self.output.D2_Output = self.output.anaOutput.reshape(nx * ny * fold, 13)   # create a 2 dim array for table access
                        self.anaModel.setData(self.output.D2_Output)                    # use this as the model data

                    if self.output.maximumFold > fold:
                        self.appendLogMessage(
                            f'Loaded : . . . Analysis &nbsp;: observed fold in in binning file {self.output.maximumFold:,} larger than allowed in trace table {fold:,} missing traces in spider plot !'
                        )

                    self.appendLogMessage(f'Loaded : . . . Analysis &nbsp;: {self.output.D2_Output.shape[0]:,} traces (reserved space)')
                except ValueError:
                    self.appendLogMessage(f"Loaded : . . . Analysis &nbsp;: read error {self.fileName + '.ana.npy'}")
                    self.anaModel.setData(None)
                    self.output.D2_Output = None
                    self.output.anaOutput = None
            else:
                self.anaModel.setData(None)
                self.output.D2_Output = None
                self.output.anaOutput = None

            # unfortunately the QTableView widget does not like too large a dataset, so we need to impose a limit to the allowable number of trace records
            if self.output.D2_Output is not None and self.output.D2_Output.shape[0] > config.maxAnalysisRows:
                self.appendLogMessage(f'Loaded : . . . Analysis &nbsp;: {self.output.D2_Output.shape[0]:,} traces; too large to display in Trace Table', MsgType.Error)
                self.anaModel.setData(None)                                     # we can still use self.output.D2_Output and self.output.anaOutput; we just can't display the trace table
            else:
                self.anaModel.setData(self.output.D2_Output)                    # use this as the model data

            if os.path.exists(self.fileName + '.rps.npy'):                      # open the existing rps-file
                self.rpsImport = np.load(self.fileName + '.rps.npy')
                self.rpsLiveE, self.rpsLiveN, self.rpsDeadE, self.rpsDeadN = getGeometry(self.rpsImport)

                nImport = self.rpsImport.shape[0]
                self.appendLogMessage(f'Loaded : . . . read {nImport:,} rps-records')
            else:
                self.rpsImport = None
                self.actionRpsPoints.setChecked(False)
                self.actionRpsPoints.setEnabled(False)

            if os.path.exists(self.fileName + '.sps.npy'):                      # open the existing sps-file
                self.spsImport = np.load(self.fileName + '.sps.npy')
                self.spsLiveE, self.spsLiveN, self.spsDeadE, self.spsDeadN = getGeometry(self.spsImport)

                nImport = self.spsImport.shape[0]
                self.appendLogMessage(f'Loaded : . . . read {nImport:,} sps-records')
            else:
                self.spsImport = None
                self.actionSpsPoints.setChecked(False)
                self.actionSpsPoints.setEnabled(False)

            if os.path.exists(self.fileName + '.xps.npy'):                      # open the existing xps-file
                self.xpsImport = np.load(self.fileName + '.xps.npy')

                nImport = self.xpsImport.shape[0]
                self.appendLogMessage(f'Loaded : . . . read {nImport:,} xps-records')
            else:
                self.xpsImport = None

            if os.path.exists(self.fileName + '.rec.npy'):                      # open the existing rps-file
                self.recGeom = np.load(self.fileName + '.rec.npy')
                self.recLiveE, self.recLiveN, self.recDeadE, self.recDeadN = getGeometry(self.recGeom)

                nImport = self.recGeom.shape[0]
                self.appendLogMessage(f'Loaded : . . . read {nImport:,} rec-records')
            else:
                self.recGeom = None
                self.actionRecPoints.setChecked(False)
                self.actionRecPoints.setEnabled(False)

            if os.path.exists(self.fileName + '.src.npy'):                      # open the existing rps-file
                self.srcGeom = np.load(self.fileName + '.src.npy')
                self.srcLiveE, self.srcLiveN, self.srcDeadE, self.srcDeadN = getGeometry(self.srcGeom)

                nImport = self.srcGeom.shape[0]
                self.appendLogMessage(f'Loaded : . . . read {nImport:,} src-records')
            else:
                self.srcGeom = None
                self.actionSrcPoints.setChecked(False)
                self.actionSrcPoints.setEnabled(False)

            if os.path.exists(self.fileName + '.rel.npy'):                      # open the existing xps-file
                self.relGeom = np.load(self.fileName + '.rel.npy')

                nImport = self.relGeom.shape[0]
                self.appendLogMessage(f'Loaded : . . . read {nImport:,} rel-records')
            else:
                self.relGeom = None

            self.rpsModel.setData(self.rpsImport)                               # update the three rps/sps/xps models
            self.spsModel.setData(self.spsImport)
            self.xpsModel.setData(self.xpsImport)

            self.recModel.setData(self.recGeom)                                 # update the three rec/rel/src models
            self.relModel.setData(self.relGeom)
            self.srcModel.setData(self.srcGeom)

            self.handleImageSelection()                                         # change selection and plot survey

        self.spiderPoint = QPoint(-1, -1)                                       # reset the spider location
        index = self.anaView.model().index(0, 0)                                # turn offset into index
        self.anaView.scrollTo(index)                                            # scroll to the first trace in the trace table
        self.anaView.selectRow(0)                                               # for the time being, *only* select first row of traces in a bin

        self.updateMenuStatus(True)                                             # keep menu status in sync with program's state; and reset analysis figure
        self.enableProcessingMenuItems(True)                                    # enable processing menu items; disable 'stop processing thread'
        self.layoutWidget.enableAutoRange()                                     # make the layout plot 'fit' the survey outline
        self.mainTabWidget.setCurrentIndex(0)                                   # make sure we display the Layout tab

        # self.plotLayout()                                                     # plot the survey object
        self.resetSurveyProperties()                                            # get the new parameters into the parameter tree
        self.survey.checkIntegrity()                                            # check for survey integrity; in particular well file validity
        return success

    def fileImportSPS(self) -> bool:
        if not self.hideSpsCrsWarning:
            mb = QMessageBox()
            mb.setWindowTitle('Please note')
            mb.setText(
                "'SPS Import' requires the SPS data to have the same CRS as the current Roll project.  Continue importing SPS data?\n",
            )
            cb = QCheckBox("Don't show this message again")
            mb.setCheckBox(cb)
            mb.setStandardButtons(mb.Ok | mb.Cancel)
            ret = mb.exec()
            self.hideSpsCrsWarning = cb.isChecked()
            if ret == mb.Cancel:
                return False

        if not self.fileName:
            reply = QMessageBox.question(
                self,
                'Please confirm',
                "'SPS Import' requires saving this file first, to obtain a valid filename in a directory with write access.\n\nSave survey file and continue ?",
                QMessageBox.Yes,
                QMessageBox.Cancel,
            )
            if reply == QMessageBox.Cancel:
                return False

            if self.fileSaveAs() is False:
                return False

        fileNames, _ = QFileDialog.getOpenFileNames(  # filetype variable not used
            self,  # self; that's me
            'Import SPS data...',  # caption
            self.importDirectory,  # start directory + filename
            'SPS triplets (*.s01 *.r01 *.x01);;'
            'SPS triplets (*.sps *.rps *.xps);;'
            'SPS triplets (*.sp1 *.rp1 *.xp1);;'
            'Source   files (*.sps *.s01 *.sp1);;'
            'Receiver files (*.rps *.r01 *.rp1);;'
            'Relation files (*.xps *.x01 *.xp1);;'
            ' All files (*.*)',
        )                                             # file extensions
        # options -> not being used
        if not fileNames:
            return False

        self.importDirectory = os.path.dirname(fileNames[0])                # retrieve the directory name from first file found

        spsFiles = []
        rpsFiles = []
        xpsFiles = []

        nSps = 0
        nRps = 0
        nXps = 0

        for fileName in fileNames:
            suffix = QFileInfo(fileName).suffix().lower()
            if suffix.startswith('s'):
                spsFiles.append(fileName)
                nSps += rawcount(fileName)
            elif suffix.startswith('r'):
                rpsFiles.append(fileName)
                nRps += rawcount(fileName)
            elif suffix.startswith('x'):
                xpsFiles.append(fileName)
                nXps += rawcount(fileName)
            else:
                baseName = QFileInfo(fileName).completeBaseName()
                QMessageBox.information(None, 'Import error', f"Unsupported file extension in selected file(s):\n\n'{baseName}.{suffix}'\n")
                return False

        self.appendLogMessage(f"Import : importing SPS-data using the '{config.spsDialect}' SPS-dialect")
        self.appendLogMessage(f'Import : importing {len(rpsFiles)} rps-file(s), {len(spsFiles)} sps-file(s) and {len(xpsFiles)} xps-file(s)')

        with pg.BusyCursor():                                               # this may take a while; start wait cursor
            # get receiver data
            if rpsFiles:
                self.rpsImport = np.zeros(shape=nRps, dtype=pntType1)

                spsFormat = next((item for item in config.spsFormatList if item['name'] == config.spsDialect), None)
                assert spsFormat is not None, f'No valid SPS dialect with name {config.spsDialect}'
                nRps = readRPSFiles(rpsFiles, self.rpsImport, spsFormat)

                if nRps <= 0:                                              # no records found
                    self.rpsImport = None
                    self.appendLogMessage('Import : . . . unexpectedly no rps-records found', MsgType.Error)
                else:
                    self.appendLogMessage(f'Import : . . . read {nRps:,} rps-records')
            else:
                self.rpsImport = None

            # get source data
            if spsFiles:
                self.spsImport = np.zeros(shape=nSps, dtype=pntType1)

                spsFormat = next((item for item in config.spsFormatList if item['name'] == config.spsDialect), None)
                assert spsFormat is not None, f'No valid SPS dialect with name {config.spsDialect}'
                nSps = readSPSFiles(spsFiles, self.spsImport, spsFormat)

                if nSps <= 0:                                              # no records found
                    self.spsImport = None
                    self.appendLogMessage('Import : . . . unexpectedly no sps-records found', MsgType.Error)
                else:
                    self.appendLogMessage(f'Import : . . . read {nSps:,} sps-records ')
            else:
                self.spsImport = None

            # get relational  data
            if xpsFiles:
                self.xpsImport = np.zeros(shape=nXps, dtype=relType2)

                xpsFormat = next((item for item in config.xpsFormatList if item['name'] == config.spsDialect), None)
                assert xpsFormat is not None, f'No valid XPS dialect with name {config.spsDialect}'
                nXps = readXPSFiles(xpsFiles, self.xpsImport, xpsFormat)

                if nXps <= 0:                                              # no records found
                    self.xpsImport = None
                    self.appendLogMessage('Import : . . . unexpectedly no xps-records found', MsgType.Error)
                else:
                    self.appendLogMessage(f'Import : . . . read {nXps:,} xps-records')
            else:
                self.xpsImport = None

        # See: for KAP input: D:\WorkStuff\G\RIJKES-L-84599\APPS\OMNI studies\2015\Kapuni\Kapuni-files\SPS
        # See: for NAM input: D:\WorkStuff\G\RIJKES-L-84599\APPS\OMNI studies\2015\Dollard\Dollard-files\SPS\Groningen

        # sort and analyse imported arrays
        with pg.BusyCursor():
            if self.rpsImport is not None:
                nImport = self.rpsImport.shape[0]
                nUnique = markUniqueRPSrecords(self.rpsImport, sort=True)
                self.appendLogMessage(f'Import : . . . analysed rps-records; found {nUnique:,} unique records and {(nImport - nUnique):,} duplicates')

                X = calculateLineStakeTransform(self.rpsImport)
                #               A0                         + A1 * Xs                 + A2 * Ys              = Xt
                #                               B0                      + B1 * Xs                 + B2 * Ys = Yt  ==>
                self.appendLogMessage(f'Import : . . . from rps info: X = {X[0]:.2f} + {X[2]:.2f} x Nx + {X[4]:.2f} x Ny')
                self.appendLogMessage(f'Import : . . . from rps info: Y = {X[1]:.2f} + {X[3]:.2f} x Nx + {X[5]:.2f} x Ny')

                self.rpsLiveE, self.rpsLiveN, self.rpsDeadE, self.rpsDeadN = getGeometry(self.rpsImport)
                self.tbRpsList.setChecked(True)

            if self.spsImport is not None:
                nImport = self.spsImport.shape[0]
                nUnique = markUniqueSPSrecords(self.spsImport, sort=True)
                self.appendLogMessage(f'Import : . . . analysed sps-records; found {nUnique:,} unique records and {(nImport - nUnique):,} duplicates')

                X = calculateLineStakeTransform(self.spsImport)
                #               A0                         + A1 * Xs                 + A2 * Ys              = Xt
                #                               B0                      + B1 * Xs                 + B2 * Ys = Yt  ==>
                self.appendLogMessage(f'Import : . . . from sps info: X = {X[0]:.2f} + {X[2]:.2f} x Nx + {X[4]:.2f} x Ny')
                self.appendLogMessage(f'Import : . . . from sps info: Y = {X[1]:.2f} + {X[3]:.2f} x Nx + {X[5]:.2f} x Ny')

                self.spsLiveE, self.spsLiveN, self.spsDeadE, self.spsDeadN = getGeometry(self.spsImport)
                self.tbSpsList.setChecked(True)

            if self.xpsImport is not None:
                nImport = self.xpsImport.shape[0]
                nUnique = markUniqueXPSrecords(self.xpsImport, sort=True)
                self.appendLogMessage(f'Import : . . . analysed xps-records; found {nUnique:,} unique records and {(nImport - nUnique):,} duplicates')

                traces = calcMaxXPStraces(self.xpsImport)
                self.appendLogMessage(f'Import : . . . xps-records allow for maximum {traces:,} traces')

            # handle doublets  of RPS / XPS and SPS / XPS-files
            if nSps > 0 and nXps > 0:
                nSpsOrphans, nXpsOrphans = findSrcOrphans(self.spsImport, self.xpsImport)
                self.appendLogMessage(f'Import : . . . sps-records contain {nXpsOrphans:,} xps-orphans')
                self.appendLogMessage(f'Import : . . . xps-records contain {nSpsOrphans:,} sps-orphans')

            if nRps > 0 and nXps > 0:
                nRpsOrphans, nXpsOrphans = findRecOrphans(self.rpsImport, self.xpsImport)
                self.appendLogMessage(f'Import : . . . rps-records contain {nXpsOrphans:,} xps-orphans')
                self.appendLogMessage(f'Import : . . . xps-records contain {nRpsOrphans:,} rps-orphans')

        self.spsModel.setData(self.spsImport)                                   # update the three sps/rps/xps models
        self.xpsModel.setData(self.xpsImport)
        self.rpsModel.setData(self.rpsImport)

        self.mainTabWidget.setCurrentIndex(4)                                   # make sure we display the 'SPS import' tab

        self.actionRpsPoints.setEnabled(nRps > 0)
        self.actionSpsPoints.setEnabled(nSps > 0)

        self.textEdit.document().setModified(True)                              # set modified flag; so we'll save sps data as numpy arrays upon saving the file

        return True

    def fileOpen(self):
        if self.maybeKillThread() and self.maybeSave():                         # current file may be modified; save it or discard edits
            fn, _ = QFileDialog.getOpenFileName(
                self,  # self; that's me
                'Open File...',  # caption
                self.workingDirectory,  # start directory + filename
                'Survey files (*.roll);; All files (*.*)'  # file extensions
                # options                                                       # not being used
            )
            if fn:
                self.workingDirectory = os.path.dirname(fn)                     # retrieve the directory name

                # sync workingDirectory to settings, so it is available outside of RollMainWindow
                self.settings.setValue('settings/workingDirectory', self.workingDirectory)

                self.fileLoad(fn)                                               # load() does all the hard work

    def fileOpenRecent(self):
        action = self.sender()
        if action:
            self.fileLoad(action.data())

    def fileSave(self):
        if not self.fileName:                                                   # need to have a valid filename first, and set the workingDirectory
            return self.fileSaveAs()

        plainText = self.textEdit.toPlainText()
        plainText = plainText.replace('\t', '    ')                             # replace any tabs (that might be caused by editing) by 4 spaces

        file = QFile(self.fileName)
        success = file.open(QIODevice.WriteOnly | QIODevice.Truncate)

        if success:
            _ = QTextStream(file) << plainText                                  # unused stream replaced by _ to make PyLint happy
            self.appendLogMessage(f'Saved&nbsp;&nbsp;: {self.fileName}')
            self.textEdit.document().setModified(False)
            file.close()

            # try to save the analysis files as well
            if self.output.binOutput is not None:
                np.save(self.fileName + '.bin.npy', self.output.binOutput)      # numpy array with fold map

            if self.output.minOffset is not None:
                np.save(self.fileName + '.min.npy', self.output.minOffset)      # numpy array with min-offset map

            if self.output.maxOffset is not None:
                np.save(self.fileName + '.max.npy', self.output.maxOffset)      # numpy array with max-offset map

            if self.output.rmsOffset is not None:
                np.save(self.fileName + '.rms.npy', self.output.rmsOffset)      # numpy array with max-offset map

            if self.output.offstHist is not None:
                np.save(self.fileName + '.off.npy', self.output.offstHist)      # numpy array with offset histogram

            if self.output.ofAziHist is not None:
                np.save(self.fileName + '.azi.npy', self.output.ofAziHist)      # numpy array with offset/azimuth histogram

            if self.rpsImport is not None:
                np.save(self.fileName + '.rps.npy', self.rpsImport)             # numpy array with list of RPS records

            if self.spsImport is not None:
                np.save(self.fileName + '.sps.npy', self.spsImport)             # numpy array with list of SPS records

            if self.xpsImport is not None:
                np.save(self.fileName + '.xps.npy', self.xpsImport)             # numpy array with list of XPS records

            if self.recGeom is not None:
                np.save(self.fileName + '.rec.npy', self.recGeom)               # numpy array with list of REC records

            if self.relGeom is not None:
                np.save(self.fileName + '.rel.npy', self.relGeom)               # numpy array with list of REL records

            if self.srcGeom is not None:
                np.save(self.fileName + '.src.npy', self.srcGeom)               # numpy array with list of SRC records
        else:
            self.appendLogMessage(f'saving : Cannot save file: {self.fileName}', MsgType.Error)
            QMessageBox.information(self, 'Write error', f'Cannot save file:\n{self.fileName}')

        self.updateMenuStatus(False)                                            # keep menu status in sync with program's state; don't reset analysis figure

        return success

    def fileSaveAs(self):
        fileName = os.path.join(self.workingDirectory, self.survey.name)        # join dir & survey name, as proposed file path
        fn, _ = QFileDialog.getSaveFileName(
            self,  # that's me
            'Save as...',  # dialog caption
            fileName,  # start directory + filename
            'Survey files (*.roll);; All files (*.*)'  # file extensions
            # options                                                           # options not used
        )
        if not fn:
            return False

        if not fn.lower().endswith('.roll'):                                    # make sure file extension is okay
            fn += '.roll'                                                       # just add the file extension

        self.setCurrentFileName(fn)                                             # update self.fileName, set textEditModified(False) and setWindowModified(False)
        return self.fileSave()

    def fileSettings(self):                                                     # dialog implementation modeled after https://github.com/dglent/meteo-qt/blob/master/meteo_qt/settings.py
        dlg = SettingsDialog(self)
        dlg.appliedSignal.connect(self.updateSettings)

        if dlg.exec():                                                          # Run the dialog event loop, and obtain survey object
            self.updateSettings()

    def updateSettings(self):
        self.handleImageSelection()
        self.plotLayout()

    def fileExportAsCsv(self, fileName, extension):
        fn, selectedFilter = QFileDialog.getSaveFileName(
            self,  # that's me
            'Save as...',  # dialog caption
            fileName + extension,  # start directory + filename
            'comma separated file (*.csv);;semicolumn separated file (*.csv);;space separated file (*.csv);;tab separated file (*.csv);;All files (*.*)',  # file extensions
            # options                                                           # options not used
        )

        delimiter = ','                                                         # default delimiter value
        if fn:                                                                  # we have a valid file name
            if selectedFilter == 'semicolumn separated file (*.csv)':           # select appropriate delimiter
                delimiter = ';'
            elif selectedFilter == 'space separated file (*.csv)':
                delimiter = ' '
            elif selectedFilter == 'tab separated file (*.csv)':
                delimiter = '\t'

            if not fn.lower().endswith(extension):                              # make sure file extension is okay
                fn += extension                                                 # just add the file extension

        return (fn, delimiter)

    def fileExportAnaAsCsv(self):

        fn, delimiter = self.fileExportAsCsv(self.fileName, '.ana.csv')

        if not fn:
            return False

        hdr = asstr(delimiter).join(map(asstr, self.anaModel.getHeader()))     # one-liner (from numpy) to assemble header
        fmt = self.anaModel.getFormat()
        data = self.output.D2_Output

        with pg.BusyCursor():
            np.savetxt(fn, data, delimiter=delimiter, fmt=fmt, comments='', header=hdr)
            self.appendLogMessage(f"Export : exported {self.output.D2_Output.shape[0]:,} lines to '{fn}'")

    def fileExportRecAsCsv(self):

        fn, delimiter = self.fileExportAsCsv(self.fileName, '.rec.csv')

        if not fn:
            return False

        hdr = asstr(delimiter).join(map(asstr, self.recModel.getHeader()))     # one-liner (from numpy) to assemble header
        fmt = self.recModel.getFormat()
        data = self.recModel.getData()

        with pg.BusyCursor():
            np.savetxt(fn, data, delimiter=delimiter, fmt=fmt, comments='', header=hdr)
            self.appendLogMessage(f"Export : exported {data.shape[0]:,} lines to '{fn}'")

    def fileExportRpsAsR01(self):

        fileName = self.fileName + '.sps.r01'                                   # start directory + filename
        data = self.rpsModel.getData()

        fn, records = fileExportAsR01(self, fileName, data, self.survey.crs)
        if records > 0:
            self.appendLogMessage(f"Export : exported {records:,} lines to '{fn}'")

    def fileExportRecAsR01(self):

        fileName = self.fileName + '.rec.r01'                                   # start directory + filename
        data = self.recModel.getData()

        fn, records = fileExportAsR01(self, fileName, data, self.survey.crs)
        if records > 0:
            self.appendLogMessage(f"Export : exported {records:,} lines to '{fn}'")

    def fileExportSpsAsS01(self):

        fileName = self.fileName + '.sps.s01'                                   # start directory + filename
        data = self.spsModel.getData()

        fn, records = fileExportAsS01(self, fileName, data, self.survey.crs)
        if records > 0:
            self.appendLogMessage(f"Export : exported {records:,} lines to '{fn}'")

    def fileExportSrcAsS01(self):

        fileName = self.fileName + '.src.s01'                                   # start directory + filename
        data = self.srcModel.getData()

        fn, records = fileExportAsS01(self, fileName, data, self.survey.crs)
        if records > 0:
            self.appendLogMessage(f"Export : exported {records:,} lines to '{fn}'")

    def fileExportXpsAsX01(self):

        fileName = self.fileName + '.sps.x01'                                   # start directory + filename
        data = self.xpsModel.getData()

        fn, records = fileExportAsX01(self, fileName, data, self.survey.crs)
        if records > 0:
            self.appendLogMessage(f"Export : exported {records:,} lines to '{fn}'")

    def fileExportRelAsX01(self):

        fileName = self.fileName + '.rel.x01'                                   # start directory + filename
        data = self.relModel.getData()

        fn, records = fileExportAsX01(self, fileName, data, self.survey.crs)
        if records > 0:
            self.appendLogMessage(f"Export : exported {records:,} lines to '{fn}'")

    def fileExportSrcAsCsv(self):

        fn, delimiter = self.fileExportAsCsv(self.fileName, '.src.csv')

        if not fn:
            return False

        hdr = asstr(delimiter).join(map(asstr, self.srcModel.getHeader()))     # one-liner (from numpy) to assemble header
        fmt = self.srcModel.getFormat()
        data = self.srcModel.getData()

        with pg.BusyCursor():
            np.savetxt(fn, data, delimiter=delimiter, fmt=fmt, comments='', header=hdr)
            self.appendLogMessage(f"Export : exported {data.shape[0]:,} lines to '{fn}'")

    def fileExportRelAsCsv(self):

        fn, delimiter = self.fileExportAsCsv(self.fileName, '.rel.csv')

        if not fn:
            return False

        hdr = asstr(delimiter).join(map(asstr, self.relModel.getHeader()))     # one-liner (from numpy) to assemble header
        fmt = self.relModel.getFormat()
        data = self.relModel.getData()

        with pg.BusyCursor():
            np.savetxt(fn, data, delimiter=delimiter, fmt=fmt, comments='', header=hdr)
            self.appendLogMessage(f"Export : exported {data.shape[0]:,} lines to '{fn}'")

    def fileExportRpsAsCsv(self):

        fn, delimiter = self.fileExportAsCsv(self.fileName, '.rps.csv')

        if not fn:
            return False

        hdr = asstr(delimiter).join(map(asstr, self.rpsModel.getHeader()))     # one-liner (from numpy) to assemble header
        fmt = self.rpsModel.getFormat()
        data = self.rpsModel.getData()

        with pg.BusyCursor():
            np.savetxt(fn, data, delimiter=delimiter, fmt=fmt, comments='', header=hdr)
            self.appendLogMessage(f"Export : exported {data.shape[0]:,} lines to '{fn}'")

    def fileExportSpsAsCsv(self):

        fn, delimiter = self.fileExportAsCsv(self.fileName, '.sps.csv')

        if not fn:
            return False

        hdr = asstr(delimiter).join(map(asstr, self.spsModel.getHeader()))     # one-liner (from numpy) to assemble header
        fmt = self.spsModel.getFormat()
        data = self.spsModel.getData()

        with pg.BusyCursor():
            np.savetxt(fn, data, delimiter=delimiter, fmt=fmt, comments='', header=hdr)
            self.appendLogMessage(f"Export : exported {data.shape[0]:,} lines to '{fn}'")

    def fileExportXpsAsCsv(self):

        fn, delimiter = self.fileExportAsCsv(self.fileName, '.xps.csv')

        if not fn:
            return False

        hdr = asstr(delimiter).join(map(asstr, self.xpsModel.getHeader()))     # one-liner (from numpy) to assemble header
        fmt = self.xpsModel.getFormat()
        data = self.xpsModel.getData()

        with pg.BusyCursor():
            np.savetxt(fn, data, delimiter=delimiter, fmt=fmt, comments='', header=hdr)
            self.appendLogMessage(f"Export : exported {data.shape[0]:,} lines to '{fn}'")

    def filePrint(self):
        printer = QPrinter(QPrinter.HighResolution)
        dlg = QPrintDialog(printer, self)

        if self.textEdit.textCursor().hasSelection():
            dlg.addEnabledOption(QPrintDialog.PrintSelection)

        dlg.setWindowTitle('Print Document')

        if dlg.exec_() == QPrintDialog.Accepted:
            self.textEdit.print_(printer)

        del dlg

    def filePrintPreview(self):
        printer = QPrinter(QPrinter.HighResolution)
        preview = QPrintPreviewDialog(printer, self)
        preview.paintRequested.connect(self.printPreview)
        preview.exec_()

    def printPreview(self, printer):
        self.textEdit.print_(printer)

    def filePrintPdf(self):
        fn, _ = QFileDialog.getSaveFileName(self, 'Export PDF', None, 'PDF files (*.pdf);;All Files (*)')

        if fn:
            if QFileInfo(fn).suffix().isEmpty():
                fn += '.pdf'

            printer = QPrinter(QPrinter.HighResolution)
            printer.setOutputFormat(QPrinter.PdfFormat)
            printer.setOutputFileName(fn)
            self.textEdit.document().print_(printer)

    def viewDebug(self):
        self.debug = not self.debug                                             # toggle status
        self.actionDebug.setChecked(self.debug)                                 # update status in button
        # See also: https://stackoverflow.com/questions/8391411/how-to-block-calls-to-print

        if self.debug:
            # builtins.print = self.oldPrint                                    # use/restore builtins.print
            if console._console is None:                                        # pylint: disable=W0212 # unfortunately, need to access protected members
                console.show_console()
            else:
                console._console.setUserVisible(True)                           # pylint: disable=W0212 # unfortunately, need to access protected members
            print('print() to Python console has been enabled; Python console is opened')             # this message should always be printed
            self.appendLogMessage('Debug&nbsp;&nbsp;: print() to Python console has been enabled')
        else:
            print('print() to Python console has been disabled; Python console is closed')            # this message is the last one to be printed
            # builtins.print = silentPrint                                        # suppress print output by calling 'dummy' routine
            print('this print message should not show')                         # this message is the last one to be printed

            if console._console is not None:                                    # pylint: disable=W0212 # unfortunately, need to access protected members
                console._console.setUserVisible(False)                          # pylint: disable=W0212 # unfortunately, need to access protected members
            self.appendLogMessage('Debug&nbsp;&nbsp;: print() to Python console has been disabled')

    def appendLogMessage(self, message: str = 'test', index: MsgType = MsgType.Info):
        # dateTime = QDateTime.currentDateTime().toString("dd-MM-yyyy hh:mm:ss")
        dateTime = QDateTime.currentDateTime().toString('yyyy-MM-ddTHH:mm:ss')  # UTC time; same format as is used in QGis

        if index == MsgType.Debug and not self.debug:                         # debug message, which needs to be suppressed
            return

        # use &nbsp; (non-breaking-space) to prevent html eating up subsequent spaces !
        switch = {  # see: https://doc.qt.io/qt-6/qcolor.html
            MsgType.Info: f'<p>{dateTime}&nbsp;&nbsp;&nbsp;&nbsp;&nbsp;info&nbsp;&nbsp;&nbsp;&nbsp;&nbsp;&nbsp;&nbsp;{message}</p>',  # info     = black
            MsgType.Binning: f'<p style="color:blue" >{dateTime}&nbsp;&nbsp;&nbsp;&nbsp;&nbsp;binning&nbsp;&nbsp;&nbsp;&nbsp;{message}</p>',  # Binning  = blue
            MsgType.Geometry: f'<p style="color:green">{dateTime}&nbsp;&nbsp;&nbsp;&nbsp;&nbsp;geometry&nbsp;&nbsp;&nbsp;{message}</p>',  # Geometry = green
            MsgType.Debug: f'<p style="color:darkCyan">{dateTime}&nbsp;&nbsp;&nbsp;&nbsp;&nbsp;debug&nbsp;&nbsp;&nbsp;&nbsp;&nbsp;&nbsp;Called : {message}</p>',  # Debug    = darkCyan
            MsgType.Error: f'<p style="color:red">{dateTime}&nbsp;&nbsp;&nbsp;&nbsp;&nbsp;error&nbsp;&nbsp;&nbsp;&nbsp;&nbsp;&nbsp;{message}</p>',  # eRror    = red
            MsgType.Exception: f'<p style="color:red">{dateTime}&nbsp;&nbsp;&nbsp;&nbsp;&nbsp;exception&nbsp;&nbsp;{message}</p>',  # exception= bold red
        }

        self.logEdit.appendHtml(switch.get(index, 'Unknown Message type'))      # Adds the message to the widget
        self.logEdit.moveCursor(QTextCursor.End)                                # scroll the edit control

        if index == MsgType.Exception:                                          # play a sound to notify end user of exception
            # SystemAsterisk, SystemExclamation, SystemExit, SystemHand, SystemQuestion are common sounds; use asyync to avoid waiting on sound to finish
            winsound.PlaySound('SystemHand', winsound.SND_ALIAS | winsound.SND_ASYNC)

    def parseText(self, plainText):
        if len(plainText) == 0:                                                 # no text available; return False
            self.appendLogMessage('Parse&nbsp;&nbsp;: error occurred while parsing empty string', MsgType.Error)
            return False
        # print (plainText)
        doc = QDomDocument()                                                    # first get a QDomDocument to work with
        success, errorMsg, errorLine, errorColumn = doc.setContent(plainText)

        if success:                                                             # parsing went ok, start with a new survey object
            self.survey = RollSurvey()                                          # only reset the survey object upon succesful parse
            self.survey.readXml(doc)                                            # build the RollSurvey object tree
            self.survey.calcTransforms()                                        # (re)calculate the transforms being used
            self.survey.calcSeedData()                                          # needed for circles, spirals & well-seeds; may affect bounding box
            self.survey.calcBoundingRect()                                      # (re)calculate the boundingBox as part of parsing the data
            self.survey.calcNoShotPoints()                                      # (re)calculate nr of SPs

            self.appendLogMessage(f'Parsed : {self.fileName} survey object updated')
            return True
        else:  # an error occurred
            self.appendLogMessage(
                f'Parse&nbsp;&nbsp;: {errorMsg}, at line: {errorLine} col:{errorColumn}; survey object not updated',
                MsgType.Error,
            )
            return False

    def OnAbout(self):
        QMessageBox.about(self, 'About Roll', aboutText())

    def OnLicense(self):
        QMessageBox.about(self, 'License conditions', licenseText())

    def OnHighDpi(self):
        QMessageBox.about(self, 'High DPI UI scaling issues', highDpiText())

    def clipboardHasText(self):
        return len(QApplication.clipboard().text()) != 0

    def enableProcessingMenuItems(self, enable=True):
        self.actionBasicBinFromTemplates.setEnabled(enable)
        self.actionFullBinFromTemplates.setEnabled(enable)
        if enable is True and self.srcGeom is not None and self.relGeom is not None and self.recGeom is not None:
            self.actionBasicBinFromGeometry.setEnabled(enable)
            self.actionFullBinFromGeometry.setEnabled(enable)
        else:
            self.actionBasicBinFromGeometry.setEnabled(False)
            self.actionFullBinFromGeometry.setEnabled(False)

        if enable is True and self.spsImport is not None and self.xpsImport is not None and self.rpsImport is not None:
            self.actionBasicBinFromSps.setEnabled(enable)
            self.actionFullBinFromSps.setEnabled(enable)
        else:
            self.actionBasicBinFromSps.setEnabled(False)
            self.actionFullBinFromSps.setEnabled(False)

        self.actionGeometryFromTemplates.setEnabled(enable)
        self.actionStopThread.setEnabled(not enable)

    def testBasicBinningConditions(self) -> bool:
        if self.survey.unique.apply:                                            # can't do unique fold with basic binning
            QMessageBox.information(
                self,
                'Please adjust',
                "Applying 'Unique offsets' requires using a 'Full Binning' process, as it is implemented as a post-processing step on the trace table",
                QMessageBox.Cancel,
            )
            return False
        else:
            return True

    def testFullBinningConditions(self) -> bool:
        if self.survey.grid.fold <= 0:                                          # the size of the analysis file is defined by the grid's fold
            QMessageBox.information(
                self,
                'Please adjust',
                "'Full Binning' requires selecting a max fold value > 0 in the 'local grid' settings, to allocate space for a memory mapped file.\n\n"
                "You can determine the required max fold value by first running 'Basic Binning'",
                QMessageBox.Cancel,
            )
            return False

        if not self.fileName:
            QMessageBox.information(self, 'Please adjust', "'Full Binning' requires saving this file first, to obtain a valid filename in a directory with write access.", QMessageBox.Cancel)
            return False

        return True

    def prepFullBinningConditions(self):
        if self.output.anaOutput is not None:                                   # get rid of current memory mapped array first
            self.anaModel.setData(None)                                         # first remove reference to self.output.anaOutput
            self.output.D2_Output = None                                        # flattened reference to self.output.anaOutput

            self.output.anaOutput.flush()                                       # make sure all data is written to disk
            del self.output.anaOutput                                           # try to delete the object
            self.output.anaOutput = None                                        # the object was deleted; reinstate the None version
            gc.collect()                                                        # get the garbage collector going

        # prepare for a new memory mapped object; calculate memmap size
        w = self.survey.output.rctOutput.width()
        h = self.survey.output.rctOutput.height()
        dx = self.survey.grid.binSize.x()
        dy = self.survey.grid.binSize.y()
        nx = ceil(w / dx)
        ny = ceil(h / dy)
        fold = self.survey.grid.fold
        n = nx * ny * fold
        self.appendLogMessage(f'Thread : prepare memory mapped file for {n:,} traces, with nx={nx}, ny={ny}, fold={fold:,}', MsgType.Binning)
        # See: https://stackoverflow.com/questions/22385801/best-practices-with-reading-and-operating-on-fortran-ordered-arrays-with-numpy
        # The order='F' in the allocation of the array could impact reading/writing data to the mmap array
        # This depends on how he binning process traverses through the binning area

        try:
            anaFileName = self.fileName + '.ana.npy'
            if os.path.exists(anaFileName):
                # self.output.anaOutput = np.lib.format.open_memmap(anaFileName, mode='r+', dtype=np.float32, shape=(nx, ny, fold, 13))
                self.output.anaOutput = np.memmap(anaFileName, shape=(nx, ny, fold, 13), dtype=np.float32, mode='r+')

            else:
                # self.output.anaOutput = np.lib.format.open_memmap(anaFileName, mode='w+', dtype=np.float32, shape=(nx, ny, fold, 13))
                self.output.anaOutput = np.memmap(anaFileName, shape=(nx, ny, fold, 13), dtype=np.float32, mode='w+')
            self.output.anaOutput.fill(0.0)                                            # enforce zero values for all elements

            nX = self.output.anaOutput.shape[0]                             # check against nx, ny
            nY = self.output.anaOutput.shape[1]                             # check against nx, ny
            nZ = self.output.anaOutput.shape[2]                             # fold
            nC = self.output.anaOutput.shape[3]                             # columns

            if nx != nX or ny != nY or nZ != fold or nC != 13:
                self.appendLogMessage('Thread : memory mapped file size error while allocating memory', MsgType.Error)
                return False

            # self.D2_output = self.output.anaOutput.reshape(nx * ny * fold, 13)
            # self.D2_output[:,:] = 0.0

        except MemoryError as e:
            self.appendLogMessage(f'Thread : memory error {e}', MsgType.Error)
            return False

        return True

    def basicBinFromTemplates(self):
        if self.testBasicBinningConditions():
            self.binFromTemplates(False)

    def fullBinFromTemplates(self):
        if self.testFullBinningConditions():
            self.binFromTemplates(True)

    def basicBinFromGeometry(self):
        if self.testBasicBinningConditions():
            self.binFromGeometry(False)

    def fullBinFromGeometry(self):
        if self.testFullBinningConditions():
            self.binFromGeometry(True)

    def basicBinFromSps(self):
        if self.testBasicBinningConditions():
            self.binFromSps(False)

    def fullBinFromSps(self):
        if self.testFullBinningConditions():
            self.binFromSps(True)

    def binFromTemplates(self, fullAnalysis):

        if fullAnalysis:
            success = self.prepFullBinningConditions()
            if not success:
                return
            self.progressLabel.setText('Bin from Templates - full analysis')
        else:
            self.progressLabel.setText('Bin from Templates - basic analysis')

        self.showStatusbarWidgets()                                             # show two temporary progress widgets
        self.enableProcessingMenuItems(False)                                   # disable processing menu items

        self.appendLogMessage(f"Thread : started 'Bin from templates', using {self.survey.nShotPoints:,} shot points", MsgType.Binning)

        # Step 1: Create a worker class
        # done in file 'worker_threads.py'

        # Step 2: Create a QThread object
        self.thread = QThread()

        # Step 3: Create a worker object
        xmlString = self.survey.toXmlString()
        self.worker = BinningWorker(xmlString)

        # and define the binning mode
        self.worker.setExtended(fullAnalysis)

        # pass the memory mapped file to worker
        self.worker.setMemMappedFile(self.output.anaOutput)

        # Step 4: Move worker to the thread
        self.worker.moveToThread(self.thread)

        # Step 5: Connect signals and slots
        self.thread.started.connect(self.worker.run)                            # start thread
        self.worker.survey.progress.connect(self.threadProgress)                # report thread progress from the survey object in the worker
        self.worker.survey.message.connect(self.threadMessage)                  # update thread task-description in statusbar
        self.worker.finished.connect(self.binningThreadFinished)                # stop thread;
        self.worker.finished.connect(self.thread.quit)
        # self.worker.finished.connect(self.worker.deleteLater)                 # See: http://qt-project.org/forums/viewthread/19848

        # Step 6: Start the thread
        self.startTime = timer()
        self.thread.start(QThread.NormalPriority)

    def binFromGeometry(self, fullAnalysis):
        if self.srcGeom is None or self.relGeom is None or self.recGeom is None:
            self.appendLogMessage('Thread : one or more of the geometry files have not been defined', MsgType.Error)
            return

        if fullAnalysis:
            success = self.prepFullBinningConditions()
            if not success:
                return
            self.progressLabel.setText('Bin from Geometry - full analysis')
        else:
            self.progressLabel.setText('Bin from Geometry - basic analysis')

        self.showStatusbarWidgets()                                             # show two temporary progress widgets
        self.enableProcessingMenuItems(False)                                   # disable processing menu items

        self.appendLogMessage(f"Thread : started 'Bin from geometry', using {self.srcGeom.shape[0]:,} shot points", MsgType.Binning)

        # Step 1: Create a worker class
        # done in file 'worker_threads.py'

        # Step 2: Create a QThread object
        self.thread = QThread()

        # Step 3: Create a worker object
        xmlString = self.survey.toXmlString()
        self.worker = BinFromGeometryWorker(xmlString)

        # and define the binning mode
        self.worker.setExtended(fullAnalysis)

        # pass the memory mapped file to worker
        self.worker.setMemMappedFile(self.output.anaOutput)

        # Pass the geometry tables, in same order as used in the Geometry tab
        self.worker.setGeometryArrays(self.srcGeom, self.relGeom, self.recGeom)

        # Step 4: Move worker to the thread
        self.worker.moveToThread(self.thread)

        # Step 5: Connect signals and slots
        self.thread.started.connect(self.worker.run)                            # start thread
        self.worker.survey.progress.connect(self.threadProgress)                # report thread progress from the survey object in the worker
        self.worker.survey.message.connect(self.threadMessage)                  # update thread task-description in statusbar
        self.worker.finished.connect(self.binningThreadFinished)                # stop thread;
        self.worker.finished.connect(self.thread.quit)
        # self.worker.finished.connect(self.worker.deleteLater)                 # See: http://qt-project.org/forums/viewthread/19848

        # Step 6: Start the thread
        self.startTime = timer()
        self.thread.start(QThread.NormalPriority)

    def binFromSps(self, fullAnalysis):
        if self.spsImport is None or self.xpsImport is None or self.rpsImport is None:
            self.appendLogMessage('Thread : one or more of the sps files have not been defined', MsgType.Error)
            return

        if fullAnalysis:
            success = self.prepFullBinningConditions()
            if not success:
                return
            self.progressLabel.setText('Bin from imported SPS - full analysis')
        else:
            self.progressLabel.setText('Bin from imported SPS - basic analysis')

        self.showStatusbarWidgets()                                             # show two temporary progress widgets
        self.enableProcessingMenuItems(False)                                   # disable processing menu items

        self.appendLogMessage(f"Thread : started 'Bin from Imported SPS', using {self.spsImport.shape[0]:,} shot points", MsgType.Binning)

        # Step 1: Create a worker class
        # done in file 'worker_threads.py'

        # Step 2: Create a QThread object
        self.thread = QThread()

        # Step 3: Create a worker object
        xmlString = self.survey.toXmlString()
        self.worker = BinFromGeometryWorker(xmlString)

        # and define the binning mode
        self.worker.setExtended(fullAnalysis)

        # pass the memory mapped file to worker
        self.worker.setMemMappedFile(self.output.anaOutput)

        # Pass the geometry tables, in same order as used in the Geometry tab
        self.worker.setGeometryArrays(self.spsImport, self.xpsImport, self.rpsImport)

        # Step 4: Move worker to the thread
        self.worker.moveToThread(self.thread)

        # Step 5: Connect signals and slots
        self.thread.started.connect(self.worker.run)                            # start thread
        self.worker.survey.progress.connect(self.threadProgress)                # report thread progress from the survey object in the worker
        self.worker.survey.message.connect(self.threadMessage)                  # update thread task-description in statusbar
        self.worker.finished.connect(self.binningThreadFinished)                # stop thread;
        self.worker.finished.connect(self.thread.quit)
        # self.worker.finished.connect(self.worker.deleteLater)                 # See: http://qt-project.org/forums/viewthread/19848

        # Step 6: Start the thread
        self.startTime = timer()
        self.thread.start(QThread.NormalPriority)

    def createGeometryFromTemplates(self):
        self.progressLabel.setText('Create Geometry from Templates')

        self.showStatusbarWidgets()                                             # show two temporary progress widgets
        self.enableProcessingMenuItems(False)                                   # disable processing menu items

        self.appendLogMessage(f"Thread : started 'Create Geometry from Templates', from {self.survey.nShotPoints:,} shot points", MsgType.Geometry)

        # Step 1: Create a worker class
        # done in file 'worker_threads.py'

        # Step 2: Create a QThread object
        self.thread = QThread()

        # Step 3: Create a worker object
        xmlString = self.survey.toXmlString()
        self.worker = GeometryWorker(xmlString)

        # Step 4: Move worker to the thread
        self.worker.moveToThread(self.thread)

        # Step 5: Connect signals and slots
        self.thread.started.connect(self.worker.run)                            # start thread
        self.worker.survey.progress.connect(self.threadProgress)                # report thread progress from the survey object in the worker
        self.worker.survey.message.connect(self.threadMessage)                  # update thread task-description in statusbar
        self.worker.finished.connect(self.geometryThreadFinished)               # stop thread;
        self.worker.finished.connect(self.thread.quit)
        # self.worker.finished.connect(self.worker.deleteLater)                 # See: http://qt-project.org/forums/viewthread/19848

        # Step 6: Start the thread
        self.startTime = timer()
        self.thread.start(QThread.NormalPriority)

    def threadProgress(self, val):
        if self.progressBar is not None:                                        # in case thread is rumming before GUI set up
            self.progressBar.setValue(val)

    def threadMessage(self, val):
        if self.progressLabel is not None:                                      # in case thread is rumming before GUI set up
            self.progressLabel.setText(val)

    def stopWorkerThread(self):
        if self.thread is not None and self.thread.isRunning():
            self.thread.requestInterruption()

    def binningThreadFinished(self, success):                                   # extra argument only available in BinningWorker class

        if not success:
            self.layoutImg = None                                               # numpy array to be displayed
            self.layoutImItem = None                                            # pg ImageItem showing analysis result
            self.handleImageSelection()                                         # change selection and plot survey

            self.appendLogMessage('Thread : . . . aborted binning operation', MsgType.Error)
            self.appendLogMessage(f'Thread : . . . {self.worker.survey.errorText}', MsgType.Error)
            QMessageBox.information(self, 'Interrupted', 'Worker thread aborted')
        else:
            # copy analysis arrays from worker
            self.output.binOutput = self.worker.survey.output.binOutput.copy()  # create a copy; not a view of the array(s)
            self.output.minOffset = self.worker.survey.output.minOffset.copy()
            self.output.maxOffset = self.worker.survey.output.maxOffset.copy()

            if self.worker.survey.output.anaOutput is not None:                             # extended binning
                self.output.rmsOffset = self.worker.survey.output.rmsOffset.copy()   # only defined in extended binning
                self.output.anaOutput = self.worker.survey.output.anaOutput.copy()   # results from full binning analysis
                self.output.ofAziHist = self.worker.survey.output.ofAziHist.copy()
                self.output.offstHist = self.worker.survey.output.offstHist.copy()

                shape = self.output.anaOutput.shape
                self.output.D2_Output = self.output.anaOutput.reshape(shape[0] * shape[1] * shape[2], shape[3])
                self.anaModel.setData(self.output.D2_Output)
                # self.anaView.resizeColumnsToContents()

            # copy limits from worker; avoid -inf values
            self.output.minimumFold = max(self.worker.survey.output.minimumFold, 0)
            self.output.maximumFold = max(self.worker.survey.output.maximumFold, 0)
            self.output.minMinOffset = max(self.worker.survey.output.minMinOffset, 0)
            self.output.maxMinOffset = max(self.worker.survey.output.maxMinOffset, 0)
            self.output.minMaxOffset = max(self.worker.survey.output.minMaxOffset, 0)
            self.output.maxMaxOffset = max(self.worker.survey.output.maxMaxOffset, 0)
            self.output.minRmsOffset = max(self.worker.survey.output.minRmsOffset, 0)
            self.output.maxRmsOffset = max(self.worker.survey.output.maxRmsOffset, 0)

            if self.survey.grid.fold <= 0:
                self.survey.grid.fold = self.output.maximumFold                        # make sure we have a workable fold value next time around

                # now we need to update the fold value in the text edit as well
                plainText = self.survey.toXmlString()                           # convert the survey object itself to an xml string
                self.textEdit.setTextViaCursor(plainText)                       # get text into the textEdit, NOT resetting its doc status
                self.textEdit.document().setModified(True)                      # we edited the document; so it's been modified

            if self.imageType == 0:                                             # if no image was selected before
                self.actionFold.setChecked(True)
                self.imageType = 1                                              # set analysis type to one (fold)

            if self.imageType == 1:
                self.layoutImg = self.output.binOutput
                self.layoutMax = self.output.maximumFold
                label = 'fold'
            elif self.imageType == 2:
                self.layoutImg = self.output.minOffset
                self.layoutMax = self.output.maxMinOffset
                label = 'minimum offset'
            elif self.imageType == 3:
                self.layoutImg = self.output.maxOffset
                self.layoutMax = self.output.maxMaxOffset
                label = 'maximum offset'
            elif self.imageType == 4:
                self.layoutImg = self.output.rmsOffset
                self.layoutMax = self.output.maxRmsOffset
                label = 'rms delta-offset'
            else:
                raise NotImplementedError('selected analysis type currently not implemented.')

            self.layoutImItem = pg.ImageItem()                                  # create PyqtGraph image item
            self.layoutImItem.setImage(self.layoutImg, levels=(0.0, self.layoutMax))

            # just to be sure; copy cmpTransform back from worker's survey object
            self.survey.cmpTransform = self.worker.survey.cmpTransform

            if self.layoutColorBar is None:
                self.layoutColorBar = self.layoutWidget.plotItem.addColorBar(self.layoutImItem, colorMap=config.fold_OffCmap, label=label, limits=(0, None), rounding=10.0, values=(0.0, self.layoutMax))
            else:
                self.layoutColorBar.setImageItem(self.layoutImItem)
                self.layoutColorBar.setLevels(low=0.0, high=self.layoutMax)
                self.layoutColorBar.setColorMap(config.fold_OffCmap)
                self.setColorbarLabel(label)

            self.plotLayout()                                                   # plot survey with colorbar

            endTime = timer()
            elapsed = timedelta(seconds=endTime - self.startTime)               # get the elapsed time for binning
            elapsed = timedelta(seconds=ceil(elapsed.total_seconds()))          # round up to nearest second

            self.appendLogMessage(f'Thread : Binning completed. Elapsed time:{elapsed} ', MsgType.Binning)
            self.appendLogMessage(f'Thread : . . . Fold&nbsp; &nbsp; &nbsp; &nbsp;: Min:{self.output.minimumFold} - Max:{self.output.maximumFold} ', MsgType.Binning)
            self.appendLogMessage(f'Thread : . . . Min-offsets: Min:{self.output.minMinOffset:.2f}m - Max:{self.output.maxMinOffset:.2f}m ', MsgType.Binning)
            self.appendLogMessage(f'Thread : . . . Max-offsets: Min:{self.output.minMaxOffset:.2f}m - Max:{self.output.maxMaxOffset:.2f}m ', MsgType.Binning)

            if self.output.rmsOffset is not None:                        # only do this for "Full binning"
                self.appendLogMessage(f'Thread : . . . Rms-offsets: Min:{self.output.minRmsOffset:.2f}m - Max:{self.output.maxRmsOffset:.2f}m ', MsgType.Binning)

            info = ''
            if not self.fileName:
                # we created an analysis for a yet to be saved file; set modified True, to force saving results when name has been defined
                self.textEdit.document().setModified(True)
                info = 'Analysis results are yet to be saved.'
            else:
                # Save the analysis results
                np.save(self.fileName + '.bin.npy', self.output.binOutput)
                np.save(self.fileName + '.min.npy', self.output.minOffset)
                np.save(self.fileName + '.max.npy', self.output.maxOffset)

                # Note, the rms offset results are only available with extended binning
                if self.output.rmsOffset is not None:                    # only do this for "Full binning"
                    np.save(self.fileName + '.rms.npy', self.output.rmsOffset)

                info = 'Analysis results have been saved.'

            QMessageBox.information(self, 'Done', f'Worker thread completed. {info} ')

        self.updateMenuStatus(False)                                            # keep menu status in sync with program's state; don't reset analysis figure
        self.enableProcessingMenuItems()                                        # enable processing menu items (again)
        self.hideStatusbarWidgets()                                             # remove temporary widgets from statusbar (don't kill 'm)

    def geometryThreadFinished(self, success):

        if self.debug:
            self.appendLogMessage('geometryFromTemplates() profiling information', MsgType.Debug)
            i = 0
            while i < len(self.worker.survey.timerTmin):                        # log some debug messages
                tMin = self.worker.survey.timerTmin[i] * 1000.0 if self.worker.survey.timerTmin[i] != float('Inf') else 0.0
                tMax = self.worker.survey.timerTmax[i] * 1000.0
                tTot = self.worker.survey.timerTtot[i] * 1000.0
                freq = self.worker.survey.timerFreq[i]
                tAvr = tTot / freq if freq > 0 else 0.0
                message = f'{i:02d}: min:{tMin:011.3f}, max:{tMax:011.3f}, tot:{tTot:011.3f}, avr:{tAvr:011.3f}, freq:{freq:07d}'
                self.appendLogMessage(message, MsgType.Debug)
                i += 1

        if not success:
            self.appendLogMessage('Thread : . . . aborted geometry creation', MsgType.Error)
            self.appendLogMessage(f'Thread : . . . {self.worker.survey.errorText}', MsgType.Error)
            QMessageBox.information(self, 'Interrupted', 'Worker thread aborted')
        else:

            # copy analysis arrays from worker
            self.recGeom = self.worker.survey.output.recGeom.copy()             # create a copy; not a view of the arrays
            self.relGeom = self.worker.survey.output.relGeom.copy()
            self.srcGeom = self.worker.survey.output.srcGeom.copy()

            # QMessageBox.information(self, 'Done', f'Worker thread completed. No crash!!!')
            # self.updateMenuStatus(False)                                            # keep menu status in sync with program's state; don't reset analysis figure
            # self.enableProcessingMenuItems()                                        # enable processing menu items (again)
            # self.mainTabWidget.setCurrentIndex(2)                                   # make sure we display the 'Geometry' tab
            # self.hideStatusbarWidgets()                                             # remove temporary widgets from statusbar (don't kill 'm)
            # return

            self.recModel.setData(self.recGeom)                                 # update the three rec/rel/src models
            self.relModel.setData(self.relGeom)
            self.srcModel.setData(self.srcGeom)

            self.recLiveE, self.recLiveN, self.recDeadE, self.recDeadN = getGeometry(self.recGeom)
            self.srcLiveE, self.srcLiveN, self.srcDeadE, self.srcDeadN = getGeometry(self.srcGeom)

            endTime = timer()
            elapsed = timedelta(seconds=endTime - self.startTime)               # get the elapsed time for geometry creation
            elapsed = timedelta(seconds=ceil(elapsed.total_seconds()))          # round up to nearest second

            self.appendLogMessage(f"Thread : completed 'Create Geometry from Templates'. Elapsed time:{elapsed} ", MsgType.Geometry)

            info = ''
            if not self.fileName:
                # we created an analysis for a yet to be saved file; set modified True, to force saving results when name has been defined
                self.textEdit.document().setModified(True)
                info = 'Analysis results are yet to be saved.'
            else:
                # Save the analysis results
                np.save(self.fileName + '.rec.npy', self.recGeom)
                np.save(self.fileName + '.rel.npy', self.relGeom)
                np.save(self.fileName + '.src.npy', self.srcGeom)
                info = 'Analysis results have been saved.'
            QMessageBox.information(self, 'Done', f'Worker thread completed. {info} ')

        self.updateMenuStatus(False)                                            # keep menu status in sync with program's state; don't reset analysis figure
        self.enableProcessingMenuItems()                                        # enable processing menu items (again)
        self.mainTabWidget.setCurrentIndex(2)                                   # make sure we display the 'Geometry' tab
        self.hideStatusbarWidgets()                                             # remove temporary widgets from statusbar (don't kill 'm)

    def showStatusbarWidgets(self):
        self.progressBar.setValue(0)                                            # first reset to zero, to avoid future glitches in progress shown
        self.statusbar.addWidget(self.progressBar)                              # add temporary widget to statusbar (again)
        self.progressBar.show()                                                 # forces showing progressbar (again)
        self.statusbar.addWidget(self.progressLabel)                            # add temporary widget to statusbar (again)
        self.progressLabel.show()                                               # forces showing progressLabel (again)

    def hideStatusbarWidgets(self):
        self.statusbar.removeWidget(self.progressBar)                           # remove widget from statusbar (don't kill it)
        self.progressBar.setValue(0)                                            # reset progressbar to zero, when out of sight
        self.statusbar.removeWidget(self.progressLabel)                         # remove progress label as well


###  From the PyQGIS Programmer's Guide
# CHAPTER 12. WRITING A STANDALONE APPLICATION
# 12.2 Creating a Minimal PyQGIS Application
# To begin, we can turn our simple application into a minimal PyQGIS app that displays a shapefile by adding a QGIS map canvas instead of the text editor control:

# def _main_():
#     # Listing 12.2: interactive_qgis.py
#     from PyQt4 import QtGui
#     from qgis.core import *
#     from qgis.gui import *

#     app = QtGui.QApplication([])
#     QgsApplication.setPrefixPath("/dev1/apps/qgis", True)
#     QgsApplication.initQgis()

#     main_win = QtGui.QMainWindow()
#     frame = QtGui.QFrame(main_win)
#     main_win.setCentralWidget(frame)
#     grid_layout= QtGui.QGridLayout(frame)

#     map_canvas = QgsMapCanvas()
#     grid_layout.addWidget (map_canvas)
#     map_canvas.setCanvasColor(QtGui.QColor(255, 255, 255))
#     layer = QgsVectorLayer(
#         '/dev1/gis_data/qgis_sample_data/shapefiles/alaska.shp',
#         'alaska',
#         'ogr')
#     QgsMapLayerRegistry.instance().addMapLayer(layer)
#     canvas_layer = QgsMapCanvasLayer(layer)
#     map_canvas.setLayerSet([canvas_layer])
#     map_canvas.zoomToFullExtent()

#     main_win.show()

#     # Need the following statement if running as a script
#     app.exec( )

# This gives us the little application shown in Figure 12.2, on the facing page.
# It isn't much to look at no toolbars, menus, map controls, or legend, just a map canvas with a single layer.
# Let's take a deeper look at the code required to get the app up and running.
# We started out with the basic text editor app and substituted it with a QgsMap- Canvas.
# To get that to work, we have to do a bit of setup first.
# (End of page)

# To create standalone applications, see also:
# See: https://subscription.packtpub.com/book/programming/9781783984985/1/ch01lvl1sec18/creating-a-standalone-application

# See: https://forum.qt.io/topic/84824/accessing-the-main-window-from-python/8
def findMainWindow() -> typing.Union[RollMainWindow, None]:
    # Global function to find the (open) RollMainWindow in application
    app = QApplication.instance()
    for widget in app.topLevelWidgets():
        if isinstance(widget, RollMainWindow):
            return widget
    return None


def main():
    sys.path.append(PARENT_DIR)

    app = QApplication(sys.argv)
    window = RollMainWindow()
    window.show()
    app.exec_()


if __name__ == '__main__':
    # attempt to run roll outside of QGIS
    # does not work yet: "Importerror attempted relative import with no known parent package"
    # See also: https://stackoverflow.com/questions/76262177/packing-qgis-standalone-application-with-qgis-gui
    PARENT_DIR = os.path.dirname(os.path.dirname(os.path.abspath(__file__)))
    sys.path.append(PARENT_DIR)
    os.chdir(PARENT_DIR)

    print('package ' + __package__)
    main()
