"""
/***************************************************************************
 Roll
                                 A QGIS plugin
 Design and analysis of 3D seismic survey geometry
 Generated by Plugin Builder: http://g-sherman.github.io/Qgis-Plugin-Builder/
                             -------------------
        begin                : 2022-10-09
        git sha              : $Format:%H$
        copyright            : (C) 2022 by Duijndam.Dev
        email                : bart.duijndam@ziggo.nl
 ***************************************************************************/

/***************************************************************************
 *                                                                         *
 *   This program is free software; you can redistribute it and/or modify  *
 *   it under the terms of the GNU General Public License as published by  *
 *   the Free Software Foundation; either version 2 of the License, or     *
 *   (at your option) any later version.                                   *
 *                                                                         *
 ***************************************************************************/

The following PyQt example shows the framework that was used to build this AddIn
https://doc.qt.io/qtforpython-5/overviews/qtwidgets-mainwindows-application-example.html#application-example

I installed some VS Code plugins as recommended here:
https://www.youtube.com/watch?v=glXGae6TsY8

To find out where libraries reside, use 'inspect':

>>>import inspect
>>>inspect.getfile(qgis.PyQt.QtCore)
'C:\\Program Files\\QGIS 3.28.1\\apps\\qgis\\python\\qgis\\PyQt\\QtCore.py'
"""
# Currently using PyLint to check for errors, which causes a few issues on its own.
# See: https://stackoverflow.com/questions/52123470/how-do-i-disable-pylint-unused-import-error-messages-in-vs-code
# See: https://gist.github.com/xen/6334976
# See: https://pylint.pycqa.org/en/latest/user_guide/messages/index.html
# See: http://pylint-messages.wikidot.com/all-codes for more codes. see also: https://manpages.org/pylint
# See: https://pylint.pycqa.org/en/latest/user_guide/messages/messages_overview.html#messages-overview for the official list
# See: https://gispofinland.medium.com/cooking-with-gispo-qgis-plugin-development-in-vs-code-19f95efb1977 IMPORTANT FOR A FULLY FUNCTIONING SETUP
# Note 1: Currently the only one language server supporting autocompletion for qgis and PyQt5 libraries is the Jedi language server
# Qgis and PyQt5 packages are using compiled python code and other language servers are having troubles parsing the API from those libraries (Actually this is for security reasons).
# In the following QGIS releases python stub files that describes the API are included in the qgis package so also much better Pylance language server can be then used
# If you have *.pyi files in C:/OSGeo4W64/apps/qgis-ltr/python/qgis go with the Pylance language server.

# This file is getting TOO BIG. Need to split class into multiple documents.
# See: https://www.reddit.com/r/Python/comments/91wbhc/how_do_i_split_a_huge_class_over_multiple_files/?rdt=44096


# As of 3.32 scaling issues have popped up in QGIS. !
# See: https://github.com/qgis/QGIS/issues/53898
# Solution:
# 1) Right click 'qgis-bin.exe' in folder 'C:\Program Files\QGIS 3.36.3\bin'
# 2) Select the Compatibility tab
# 3) Select 'change high DPI settings'
# 4) Set the tickmark before 'Override high DPI ...'
# 5) Have scaling performed by 'Application'
# 6) In the same folder edit the file 'qgis-bin.env'
# 7) Add one line at the end:
# 8) QT_SCALE_FACTOR_ROUNDING_POLICY=Floor
# 9) Save the file in a different (user) folder as C:\Program Files is protected
# 10) Drag the edited file to the C:\Program Files\QGIS 3.36.3\bin folder
# 11) You'll be asked to confirm you want to overwrite the *.env file
# that solved my problems ! I use font size 9.0 and Icon size 24

# Now creating extra toolbars to be able to close the Display pane, and work from a toolbar instead.
# But I don't want to duplicate all signals and slots !
# See: https://stackoverflow.com/questions/16703039/pyqt-can-a-qpushbutton-be-assigned-a-qaction
# See: https://stackoverflow.com/questions/4149117/how-can-i-implement-the-button-that-holds-an-action-qaction-and-can-connect-w
# See: https://stackoverflow.com/questions/38576380/difference-between-qpushbutton-and-qtoolbutton
# See: https://stackoverflow.com/questions/10368947/how-to-make-qmenu-item-checkable-pyqt4-python
# See: https://stackoverflow.com/questions/23388754/two-shortcuts-for-one-action
# See: https://stackoverflow.com/questions/53936403/two-shortcuts-for-one-button-in-pyqt
# See: https://doc.qt.io/qtforpython-5/PySide2/QtWidgets/QActionGroup.html#PySide2.QtWidgets.PySide2.QtWidgets.QActionGroup


import contextlib
import gc
import os
import os.path
import sys
import traceback
import typing
import winsound  # make a sound when an exception ocurs
from datetime import timedelta
from enum import Enum
from math import atan2, ceil, degrees
from timeit import default_timer as timer

# PyQtGraph related imports
import numpy as np  # Numpy functions needed for plot creation
import pyqtgraph as pg
from console import console
from numpy.compat import asstr
from qgis.PyQt import uic
from qgis.PyQt.QtCore import QDateTime, QEvent, QFile, QFileInfo, QIODevice, QItemSelection, QItemSelectionModel, QModelIndex, QPoint, QSettings, Qt, QTextStream, QThread
from qgis.PyQt.QtGui import QBrush, QColor, QFont, QIcon, QKeySequence, QTextCursor, QTextOption, QTransform
from qgis.PyQt.QtPrintSupport import QPrintDialog, QPrinter, QPrintPreviewDialog
from qgis.PyQt.QtWidgets import (
    QAction,
    QActionGroup,
    QApplication,
    QButtonGroup,
    QCheckBox,
    QDialogButtonBox,
    QDockWidget,
    QFileDialog,
    QFrame,
    QGraphicsEllipseItem,
    QGridLayout,
    QGroupBox,
    QHBoxLayout,
    QHeaderView,
    QLabel,
    QMainWindow,
    QMessageBox,
    QPlainTextEdit,
    QProgressBar,
    QPushButton,
    QSplitter,
    QTabWidget,
    QToolButton,
    QVBoxLayout,
    QWidget,
)
from qgis.PyQt.QtXml import QDomDocument

from . import config  # used to pass initial settings

# from .event_lookup import event_lookup
from .find import Find
from .functions import aboutText, exampleSurveyXmlText, highDpiText, licenseText, rawcount
from .functions_numba import numbaAziInline, numbaAziX_line, numbaFilterSlice2D, numbaNdft_1D, numbaNdft_2D, numbaOffInline, numbaOffsetBin, numbaOffX_line, numbaSlice3D, numbaSliceStats, numbaSpiderBin
from .land_wizard import LandSurveyWizard
from .my_parameters import registerAllParameterTypes
from .qgis_interface import CreateQgisRasterLayer, ExportRasterLayerToQgis, exportPointLayerToQgis, exportSurveyOutlineToQgis, identifyQgisPointLayer, readQgisPointLayer, updateQgisPointLayer
from .roll_binning import BinningType
from .roll_survey import RollSurvey, SurveyType
from .settings import SettingsDialog, readSettings, writeSettings
from .sps_io_and_qc import (
    calcMaxXPStraces,
    calculateLineStakeTransform,
    deletePntDuplicates,
    deletePntOrphans,
    deleteRelDuplicates,
    deleteRelOrphans,
    fileExportAsR01,
    fileExportAsS01,
    fileExportAsX01,
    findRecOrphans,
    findSrcOrphans,
    getRecGeometry,
    getSrcGeometry,
    markUniqueRPSrecords,
    markUniqueSPSrecords,
    markUniqueXPSrecords,
    pntType1,
    readRPSFiles,
    readSPSFiles,
    readXPSFiles,
    relType2,
)
from .table_model_view import AnaTableModel, RpsTableModel, SpsTableModel, TableView, XpsTableModel
from .worker_threads import BinFromGeometryWorker, BinningWorker, GeometryWorker
from .xml_code_editor import QCodeEditor, XMLHighlighter


class ImagType(Enum):
    NoIm = 0
    Fold = 1
    MinO = 2
    MaxO = 3


class MsgType(Enum):
    Info = 0
    Binning = 1
    Geometry = 2
    Debug = 3
    Error = 4
    Exception = 5


class Direction(Enum):
    NA = 0
    Up = 1
    Dn = 2
    Lt = 3
    Rt = 4


# This loads your .ui file so that PyQt can populate your plugin with the elements from Qt Designer
FORM_CLASS, _ = uic.loadUiType(os.path.join(os.path.dirname(__file__), 'roll_main_window_base.ui'))

# See: https://gist.github.com/mistic100/dcbffbd9e9c15271dd14
class QButtonGroupEx(QButtonGroup):
    def setCheckedId(self, id_) -> int:
        for button in self.buttons():
            if self.id(button) == id_:
                button.setChecked(True)
                return id_
        return None


# See: https://groups.google.com/g/pyqtgraph/c/V01QJKvrUio/m/iUBp5NePCQAJ
class LineROI(pg.LineSegmentROI):
    def __init__(self, *args, **kwargs):
        super().__init__(*args, **kwargs)

    def addHandle(self, *args, **kwargs):
        # Larger handle for improved visibility
        self.handleSize = 8
        super().addHandle(*args, **kwargs)

    def checkPointMove(self, handle, pos, modifiers):
        # needed to prevent 'eternal' range-jitter preventing the plot to complete
        self.getViewBox().disableAutoRange(axis='xy')
        return True

    def generateSvg(self, nodes):
        pass                                                                    # for the time being don't do anything; just to keep PyLint happy


class QHLine(QFrame):
    def __init__(self, parent=None):
        super().__init__(parent)
        self.setFrameShape(QFrame.HLine)
        self.setFrameShadow(QFrame.Sunken)

    def generateSvg(self, nodes):
        pass                                                                    # for the time being don't do anything; just to keep PyLint happy


def silentPrint(*_, **__):
    pass


class RollMainWindow(QMainWindow, FORM_CLASS):
    def __init__(self, parent=None):
        """Constructor."""
        super(RollMainWindow, self).__init__(parent)
        # Set up the user interface from Designer through FORM_CLASS.
        # After self.setupUi() you can access any designer object by doing self.<objectname>,
        # and you can use autoconnect slots - see http://qt-project.org/doc/qt-4.8/designer-using-a-ui-file.html
        # widgets-and-dialogs-with-auto-connect
        # See also: https://doc.qt.io/qt-6/designer-using-a-ui-file-python.html
        # See: https://docs.qgis.org/3.22/en/docs/documentation_guidelines/substitutions.html#toolbar-button-icons for QGIS Icons
        self.setupUi(self)

        # reset GUI when the plugin is restarted (not when restarted from minimized state on windows)
        self.killMe = False

        # GQIS interface
        self.iface = None                                                       # access to QGis interface

        # toolbar parameters
        self.debug = False                                                      # use debug settings
        self.XisY = True                                                        # equal x / y scaling
        self.rect = False                                                       # zoom using a rectangle
        self.glob = False                                                       # global coordinates
        self.gridX = True                                                       # use grid lines
        self.gridY = True                                                       # use grid lines
        self.antiA = [False for i in range(10)]                                 # anti-alias painting
        self.ruler = False                                                      # show a ruler to measure distances

        # exception handling
        # self.oldExceptHook = sys.excepthook                                     # make a copy before changing it
        # sys.excepthook = self.exceptionHook                                     # deal with uncaught exceptions using hook

        # print handling
        # self.oldPrint = builtins.print                                          # need to be able to get back to 'normal'

        # list with most recently used [mru] file actions
        self.recentFileActions = []
        self.recentFileList = []

        # workerTread parameters
        self.worker = None                                                      # 'moveToThread' object
        self.thread = None                                                      # corresponding worker thread
        self.startTime = None                                                   # thread start time

        # statusbar widgets
        self.posWidgetStatusbar = QLabel('(x, y): (0.00, 0.00)')                # mouse' position label, in bottom right corner
        self.progressLabel = QLabel('doing a lot of stuff in the background')   # label next to progressbar indicating background process
        self.progressBar = QProgressBar()                                       # progressbar in statusbar
        self.progressBar.setMaximumWidth(500)                                   # to avoid 'jitter' when the mouse moves and posWidgetStatusbar changes width
        height = self.posWidgetStatusbar.height()                               # needed to avoid statusbar 'growing' vertically by adding the progressbar
        self.progressBar.setMaximumHeight(height)                               # to avoid ugly appearance on statusbar

        # binning analysis
        self.imageType = 0                                                      # 1 = fold map
        self.layoutMax = 0.0                                                    # max value for image's colorbar (minimum is always 0)
        self.minimumFold = 0                                                    # fold minimum
        self.maximumFold = 0                                                    # fold maximum
        self.minMinOffset = 0.0                                                 # min-offset minimum
        self.maxMinOffset = 0.0                                                 # min-offset maximum
        self.minMaxOffset = 0.0                                                 # max-offset minimum
        self.maxMaxOffset = 0.0                                                 # max-offset maximum
        self.minRmsOffset = 0.0                                                 # rms-offset minimum
        self.maxRmsOffset = 0.0                                                 # rms-offset maximum

        self.binAreaChanged = False                                             # set when binning area changes in property tree

        # numpy binning arrays
        self.layoutImg = None                                                   # numpy array to be displayed; binOutput / minOffset / maxOffset

        self.D2_Output = None                                                   # flattened 2D-version of self.survey.output.anaOutput

        # analysis numpy arrays
        self.inlineStk = None                                                   # numpy array with inline Kr stack reponse
        self.x_lineStk = None                                                   # numpy array with x_line Kr stack reponse
        self.xyCellStk = None                                                   # numpy array with cell's KxKy stack response
        self.xyPatResp = None                                                   # numpy array with pattern's KxKy response
        self.offAziPlt = None                                                   # numpy array with offset distribution

        # layout and analysis image-items
        self.layoutImItem = None                                                # pg ImageItems showing analysis result
        self.stkTrkImItem = None
        self.stkBinImItem = None
        self.stkCelImItem = None
        self.offAziImItem = None

        # corresponding color bars
        self.layoutColorBar = None                                              # colorBars, added to imageItem
        self.stkTrkColorBar = None
        self.stkBinColorBar = None
        self.stkCelColorBar = None
        self.offAziColorBar = None

        # rps, sps, xps input arrays
        self.rpsImport = None                                                   # numpy array with list of RPS records
        self.spsImport = None                                                   # numpy array with list of SPS records
        self.xpsImport = None                                                   # numpy array with list of XPS records

        self.rpsCoordE = None                                                   # numpy array with list of RPS coordinates
        self.rpsCoordN = None                                                   # numpy array with list of RPS coordinates
        self.rpsCoordI = None                                                   # numpy array with list of RPS connections

        self.spsCoordE = None                                                   # numpy array with list of SPS coordinates
        self.spsCoordN = None                                                   # numpy array with list of SPS coordinates
        self.spsCoordI = None                                                   # numpy array with list of SPS connections

        # rel, src, rel input arrays
        self.recGeom = None                                                     # numpy array with list of REC records
        self.srcGeom = None                                                     # numpy array with list of SRC records
        self.relGeom = None                                                     # numpy array with list of REL records

        self.recCoordE = None                                                   # numpy array with list of REC coordinates
        self.recCoordN = None                                                   # numpy array with list of REC coordinates
        self.recCoordI = None                                                   # numpy array with list of REC connections

        self.srcCoordE = None                                                   # numpy array with list of SRC coordinates
        self.srcCoordN = None                                                   # numpy array with list of SRC coordinates
        self.srcCoordI = None                                                   # numpy array with list of SRC connections

        # spider plot settings
        self.spiderPoint = QPoint(-1, -1)                                       # spider point 'out of scope'
        self.spiderSrcX = None                                                  # numpy array with list of SRC part of spider plot
        self.spiderSrcY = None                                                  # numpy array with list of SRC part of spider plot
        self.spiderRecX = None                                                  # numpy array with list of REC part of spider plot
        self.spiderRecY = None                                                  # numpy array with list of REC part of spider plot
        self.spiderText = None                                                  # text label describing spider bin, stake, fold

        # export layers to QGIS
        self.rpsLayerId = None                                                  # QGIS layerId to be checked when updating a QgisVectorLayer
        self.spsLayerId = None                                                  # QGIS layerId to be checked when updating a QgisVectorLayer
        self.recLayerId = None                                                  # QGIS layerId to be checked when updating a QgisVectorLayer
        self.srcLayerId = None                                                  # QGIS layerId to be checked when updating a QgisVectorLayer

        # ruler settings
        self.lineROI = None                                                     # the ruler's dotted line
        self.roiLabels = None                                                   # the ruler's three labels
        self.rulerState = None                                                  # ruler's state, used to redisplay ruler at last used location

        # warning dialogs that can be hidden
        self.hideSpsCrsWarning = False                                          # warning message: sps crs should be identical to project crs

        icon_path = ':/plugins/roll/icon.png'
        icon = QIcon(icon_path)
        self.setWindowIcon(icon)

        # See: https://gist.github.com/dgovil/d83e7ddc8f3fb4a28832ccc6f9c7f07b dealing with settings
        # See also : https://doc.qt.io/qtforpython-5/PySide2/QtCore/QSettings.html
        # QCoreApplication.setOrganizationName('Duijndam.Dev')
        # QCoreApplication.setApplicationName('Roll')
        # self.settings = QSettings()   ## doesn't work as expected with QCoreApplication.setXXX

        self.settings = QSettings(config.organization, config.application)
        self.fileName = ''
        self.workingDirectory = ''
        self.importDirectory = ''

        # first docking pane, used to display geometry and analysis results
        self.dockDisplay = QDockWidget('Display pane')
        self.dockDisplay.setAllowedAreas(Qt.LeftDockWidgetArea | Qt.RightDockWidgetArea)

        # See: https://www.geeksforgeeks.org/pyqt5-qdockwidget-setting-style-sheet/ for styling
        self.dockDisplay.setStyleSheet('QDockWidget::title {background : lightblue;}')

        self.geometryChoice = QGroupBox('Geometry to display')                  # create display widget(s)
        self.analysisChoice = QGroupBox('Analysis to display')                  # create display widget(s)
        self.analysisToQgis = QGroupBox('Export to QGIS')                       # create display widget(s)

        self.geometryChoice.setMinimumWidth(140)
        self.analysisChoice.setMinimumWidth(140)
        self.analysisToQgis.setMinimumWidth(140)

        self.geometryChoice.setAlignment(Qt.AlignHCenter)
        self.analysisChoice.setAlignment(Qt.AlignHCenter)
        self.analysisToQgis.setAlignment(Qt.AlignHCenter)

        # display pane
        vbox0 = QVBoxLayout()
        # self.displayLayout = QVBoxLayout()                                      # required layout
        self.displayLayout = QHBoxLayout()                                      # required layout

        self.displayLayout.addStretch()                                         # add some stretch to main center widget(s)
        self.displayLayout.addLayout(vbox0)
        self.displayLayout.addStretch()                                         # add some stretch to main center widget(s)

        vbox0.addStretch()                                                      # add some stretch to main center widget(s)
        vbox0.addWidget(self.geometryChoice)                                    # add main widget(s)
        vbox0.addStretch()                                                      # add some stretch to main center widget(s)
        vbox0.addWidget(self.analysisChoice)                                    # add main widget(s)
        vbox0.addStretch()                                                      # add some stretch to main center widget(s)
        vbox0.addWidget(self.analysisToQgis)                                    # add main widget(s)
        vbox0.addStretch()                                                      # add some stretch to main center widget(s)

        # self.displayLayout.addStretch()                                         # add some stretch to main center widget(s)
        # self.displayLayout.addWidget(self.geometryChoice)                       # add main widget(s)
        # self.displayLayout.addStretch()                                         # add some stretch to main center widget(s)
        # self.displayLayout.addWidget(self.analysisChoice)                       # add main widget(s)
        # self.displayLayout.addStretch()                                         # add some stretch to main center widget(s)
        # self.displayLayout.addWidget(self.analysisToQgis)                       # add main widget(s)
        # self.displayLayout.addStretch()                                         # add some stretch to main center widget(s)

        self.tbTemplat = QToolButton()
        self.tbRecList = QToolButton()
        self.tbSrcList = QToolButton()
        self.tbRpsList = QToolButton()
        self.tbSpsList = QToolButton()

        self.tbTemplat.setMinimumWidth(110)
        self.tbRecList.setMinimumWidth(110)
        self.tbSrcList.setMinimumWidth(110)
        self.tbRpsList.setMinimumWidth(110)
        self.tbSpsList.setMinimumWidth(110)

        self.tbTemplat.setStyleSheet('QToolButton { selection-background-color: blue } QToolButton:checked { background-color: lightblue } QToolButton:pressed { background-color: red }')
        self.tbRecList.setStyleSheet('QToolButton { selection-background-color: blue } QToolButton:checked { background-color: lightblue } QToolButton:pressed { background-color: red }')
        self.tbSrcList.setStyleSheet('QToolButton { selection-background-color: blue } QToolButton:checked { background-color: lightblue } QToolButton:pressed { background-color: red }')
        self.tbTemplat.setStyleSheet('QToolButton { selection-background-color: blue } QToolButton:checked { background-color: lightblue } QToolButton:pressed { background-color: red }')
        self.tbRpsList.setStyleSheet('QToolButton { selection-background-color: blue } QToolButton:checked { background-color: lightblue } QToolButton:pressed { background-color: red }')
        self.tbSpsList.setStyleSheet('QToolButton { selection-background-color: blue } QToolButton:checked { background-color: lightblue } QToolButton:pressed { background-color: red }')

        self.tbTemplat.setDefaultAction(self.actionTemplates)
        self.tbRecList.setDefaultAction(self.actionRecPoints)
        self.tbSrcList.setDefaultAction(self.actionSrcPoints)
        self.tbRpsList.setDefaultAction(self.actionRpsPoints)
        self.tbSpsList.setDefaultAction(self.actionSpsPoints)

        self.actionTemplates.setChecked(True)
        self.actionRecPoints.setEnabled(False)
        self.actionSrcPoints.setEnabled(False)
        self.actionRpsPoints.setEnabled(False)
        self.actionSpsPoints.setEnabled(False)

        vbox1 = QVBoxLayout()
        vbox1.addWidget(self.tbTemplat)
        vbox1.addWidget(self.tbRecList)
        vbox1.addWidget(self.tbSrcList)
        vbox1.addWidget(self.tbRpsList)
        vbox1.addWidget(self.tbSpsList)
        self.geometryChoice.setLayout(vbox1)

        vbox2 = QVBoxLayout()

        self.tbNone = QToolButton()
        self.tbFold = QToolButton()
        self.tbMinO = QToolButton()
        self.tbMaxO = QToolButton()
        self.tbRmsO = QToolButton()

        self.tbNone.setMinimumWidth(110)
        self.tbFold.setMinimumWidth(110)
        self.tbMinO.setMinimumWidth(110)
        self.tbMaxO.setMinimumWidth(110)
        self.tbRmsO.setMinimumWidth(110)

        self.tbNone.setStyleSheet('QToolButton { selection-background-color: blue } QToolButton:checked { background-color: lightblue } QToolButton:pressed { background-color: red }')
        self.tbFold.setStyleSheet('QToolButton { selection-background-color: blue } QToolButton:checked { background-color: lightblue } QToolButton:pressed { background-color: red }')
        self.tbMinO.setStyleSheet('QToolButton { selection-background-color: blue } QToolButton:checked { background-color: lightblue } QToolButton:pressed { background-color: red }')
        self.tbMaxO.setStyleSheet('QToolButton { selection-background-color: blue } QToolButton:checked { background-color: lightblue } QToolButton:pressed { background-color: red }')
        self.tbRmsO.setStyleSheet('QToolButton { selection-background-color: blue } QToolButton:checked { background-color: lightblue } QToolButton:pressed { background-color: red }')

        self.actionNone.setChecked(True)                                        # action coupled to tbNone
        self.tbNone.setDefaultAction(self.actionNone)                           # coupling done here
        self.tbFold.setDefaultAction(self.actionFold)
        self.tbMinO.setDefaultAction(self.actionMinO)
        self.tbMaxO.setDefaultAction(self.actionMaxO)
        self.tbRmsO.setDefaultAction(self.actionRmsO)

        vbox2.addWidget(self.tbNone)
        vbox2.addWidget(self.tbFold)
        vbox2.addWidget(self.tbMinO)
        vbox2.addWidget(self.tbMaxO)
        vbox2.addWidget(self.tbRmsO)

        vbox2.addWidget(QHLine())
        self.actionSpider.triggered.connect(self.handleSpiderPlot)
        self.actionSpider.setEnabled(False)

        self.tbSpider = QToolButton()
        self.tbSpider.setMinimumWidth(110)
        self.tbSpider.setStyleSheet('QToolButton { selection-background-color: blue } QToolButton:checked { background-color: lightblue } QToolButton:pressed { background-color: red }')
        self.tbSpider.setDefaultAction(self.actionSpider)
        vbox2.addWidget(self.tbSpider)

        self.btnSpiderLt = QToolButton()
        self.btnSpiderRt = QToolButton()
        self.btnSpiderUp = QToolButton()
        self.btnSpiderDn = QToolButton()

        self.btnSpiderLt.setDefaultAction(self.actionMoveLt)
        self.btnSpiderRt.setDefaultAction(self.actionMoveRt)
        self.btnSpiderUp.setDefaultAction(self.actionMoveUp)
        self.btnSpiderDn.setDefaultAction(self.actionMoveDn)

        # Note: to use a stylesheet on buttons (=actions) in a toolbar, you needt to use the toolbar's stylesheet and select individual actions to 'style'
        # See: https://stackoverflow.com/questions/32460193/how-to-change-qaction-background-color-using-stylesheets-css
        self.btnSpiderLt.setStyleSheet('QToolButton { selection-background-color: blue } QToolButton:checked { background-color: lightblue } QToolButton:pressed { background-color: red }')
        self.btnSpiderRt.setStyleSheet('QToolButton { selection-background-color: blue } QToolButton:checked { background-color: lightblue } QToolButton:pressed { background-color: red }')
        self.btnSpiderUp.setStyleSheet('QToolButton { selection-background-color: blue } QToolButton:checked { background-color: lightblue } QToolButton:pressed { background-color: red }')
        self.btnSpiderDn.setStyleSheet('QToolButton { selection-background-color: blue } QToolButton:checked { background-color: lightblue } QToolButton:pressed { background-color: red }')

        self.actionMoveLt.triggered.connect(self.spiderGoLt)
        self.actionMoveRt.triggered.connect(self.spiderGoRt)
        self.actionMoveUp.triggered.connect(self.spiderGoUp)
        self.actionMoveDn.triggered.connect(self.spiderGoDn)

        self.actionMoveLt.setShortcuts(['Alt+Left', 'Alt+Shift+Left', 'Alt+Ctrl+Left', 'Alt+Shift+Ctrl+Left'])
        self.actionMoveRt.setShortcuts(['Alt+Right', 'Alt+Shift+Right', 'Alt+Ctrl+Right', 'Alt+Shift+Ctrl+Right'])
        self.actionMoveUp.setShortcuts(['Alt+Up', 'Alt+Shift+Up', 'Alt+Ctrl+Up', 'Alt+Shift+Ctrl+Up'])
        self.actionMoveDn.setShortcuts(['Alt+Down', 'Alt+Shift+Down', 'Alt+Ctrl+Down', 'Alt+Shift+Ctrl+Down'])

        hbox1 = QHBoxLayout()
        hbox1.addStretch()
        hbox1.addWidget(self.btnSpiderLt)
        hbox1.addWidget(self.btnSpiderRt)
        hbox1.addWidget(self.btnSpiderUp)
        hbox1.addWidget(self.btnSpiderDn)
        hbox1.addStretch()
        vbox2.addLayout(hbox1)

        # grid1 = QGridLayout()
        # grid1.addWidget(self.btnSpiderLt, 0, 0)
        # grid1.addWidget(self.btnSpiderRt, 0, 1)
        # grid1.addWidget(self.btnSpiderUp, 1, 0)
        # grid1.addWidget(self.btnSpiderDn, 1, 1)
        # vbox2.addLayout(grid1)

        self.analysisChoice.setLayout(vbox2)

        self.anActionGroup = QActionGroup(self)
        self.anActionGroup.addAction(self.actionNone)
        self.anActionGroup.addAction(self.actionFold)
        self.anActionGroup.addAction(self.actionMinO)
        self.anActionGroup.addAction(self.actionMaxO)
        self.anActionGroup.addAction(self.actionRmsO)
        self.actionNone.setChecked(True)

        self.actionNone.triggered.connect(self.onActionNoneTriggered)
        self.actionFold.triggered.connect(self.onActionFoldTriggered)
        self.actionMinO.triggered.connect(self.onActionMinOTriggered)
        self.actionMaxO.triggered.connect(self.onActionMaxOTriggered)
        self.actionRmsO.triggered.connect(self.onActionRmsOTriggered)

        self.btnBinToQGIS = QPushButton('Fold Map')
        self.btnMinToQGIS = QPushButton('Min Offset')
        self.btnMaxToQGIS = QPushButton('Max Offset')
        self.btnRmsToQGIS = QPushButton('Rms Offset')

        self.btnBinToQGIS.setMinimumWidth(110)
        self.btnMinToQGIS.setMinimumWidth(110)
        self.btnMaxToQGIS.setMinimumWidth(110)
        self.btnRmsToQGIS.setMinimumWidth(110)

        self.btnBinToQGIS.setMaximumWidth(5)
        self.btnMinToQGIS.setMaximumWidth(5)
        self.btnMaxToQGIS.setMaximumWidth(5)
        self.btnRmsToQGIS.setMaximumWidth(5)

        self.btnBinToQGIS.setStyleSheet('background-color:lightgoldenrodyellow; font-weight:bold;')
        self.btnMinToQGIS.setStyleSheet('background-color:lightgoldenrodyellow; font-weight:bold;')
        self.btnMaxToQGIS.setStyleSheet('background-color:lightgoldenrodyellow; font-weight:bold;')
        self.btnRmsToQGIS.setStyleSheet('background-color:lightgoldenrodyellow; font-weight:bold;')

        vbox3 = QVBoxLayout()
        vbox3.addWidget(self.btnBinToQGIS)
        vbox3.addWidget(self.btnMinToQGIS)
        vbox3.addWidget(self.btnMaxToQGIS)
        vbox3.addWidget(self.btnRmsToQGIS)
        self.analysisToQgis.setLayout(vbox3)

        self.displayWidget = QWidget()                                          # placeholder widget to generate a layout
        self.displayWidget.setLayout(self.displayLayout)                        # add layout to widget
        self.dockDisplay.setWidget(self.displayWidget)                          # set widget as main widget in docking panel

        self.addDockWidget(Qt.LeftDockWidgetArea, self.dockDisplay)             # add docking panel to main window
        self.dockDisplay.toggleViewAction().setShortcut(QKeySequence('Ctrl+Alt+d'))
        self.menu_View.addAction(self.dockDisplay.toggleViewAction())           # show/hide as requested

        # # # # # # # # # # # # # # # # # # # # # # # # # # # # # # # # # # # # # # # # # # # # # # # # # # # # # # #

        # second docking pane
        self.dockLogging = QDockWidget('Logging pane')
        self.dockLogging.setAllowedAreas(Qt.BottomDockWidgetArea | Qt.TopDockWidgetArea)

        # See: https://www.geeksforgeeks.org/pyqt5-qdockwidget-setting-style-sheet/ for styling
        self.dockLogging.setStyleSheet('QDockWidget::title {background : lightblue;}')

        # See: https://wiki.python.org/moin/PyQt/Handling%20context%20menus to change context menu, very informative
        # See: https://stackoverflow.com/questions/32053072/pyqt-extend-existing-contextual-menu-when-editing-qstandarditems-text-in-a-qtr
        # See: https://stackoverflow.com/questions/8676597/customising-location-sensitive-context-menu-in-qtextedit
        # See: https://forum.qt.io/topic/60790/add-actions-from-qtextedit-to-edit-menu/10

        self.logEdit = QPlainTextEdit()
        self.logEdit.clear()                                                    # Should not be necessary between sessions
        self.logEdit.setUndoRedoEnabled(False)                                  # Don't allow undo on the logging pane
        # self.logEdit.setReadOnly(True)                                        # if we set this 'True' the context menu no longer allows 'delete', just 'select all' and 'copy'
        self.logEdit.setLineWrapMode(QPlainTextEdit.NoWrap)
        self.logEdit.setWordWrapMode(QTextOption.NoWrap)
        self.logEdit.setStyleSheet('QPlainTextEdit { font-family: Courier New; font-weight: bold; font-size: 12px;}')
        # self.logEdit.setFont(QFont("Ubuntu Mono", 8, QFont.Normal))           # Does not line up columns properly !

        self.dockLogging.setWidget(self.logEdit)
        self.addDockWidget(Qt.BottomDockWidgetArea, self.dockLogging)
        self.dockLogging.toggleViewAction().setShortcut(QKeySequence('Ctrl+Alt+l'))
        self.menu_View.addAction(self.dockLogging.toggleViewAction())

        # # # # # # # # # # # # # # # # # # # # # # # # # # # # # # # # # # # # # # # # # # # # # # # # # # # # # # #

        self.survey = RollSurvey()                                              # (re)set the survey object; needed in property pane

        self.mainTabWidget = QTabWidget()
        self.mainTabWidget.setTabPosition(QTabWidget.South)
        self.mainTabWidget.setTabShape(QTabWidget.Rounded)
        self.mainTabWidget.setDocumentMode(False)                               # has only effect on OSX ?!
        self.mainTabWidget.resize(300, 200)

        self.analysisTabWidget = QTabWidget()
        self.analysisTabWidget.setTabPosition(QTabWidget.South)
        self.analysisTabWidget.setTabShape(QTabWidget.Rounded)
        self.analysisTabWidget.setDocumentMode(False)                           # has only effect on OSX ?!
        self.analysisTabWidget.resize(300, 200)

        # See: https://stackoverflow.com/questions/69152935/adding-the-same-object-to-a-qtabwidget
        # See: pyqtgraph/examples/RemoteSpeedTest.py to keep gui responsive when updating a plot (uses multiprocessing)

        self.plotTitles = [
            'New survey',
            'Offsets for inline direction',
            'Offsets for x-line direction',
            'Azimuth for inline direction',
            'Azimuth for x-line direction',
            'Stack response for inline direction',
            'Stack response for x-line direction',
            'Kx-Ky strack response for a single bin',
            '|Offset| distribution in binning area',
            'Offset/azimuth distribution in binning area',
        ]

        # these plotting widgets have "installEventFilter()" applied to catch the window 'Show' event in "eventFilter()"
        # this makes it possile to reroute commands and status from the plotting toolbar buttons to the active plot
        self.offTrkWidget = self.createPlotWidget(self.plotTitles[1], 'inline', 'offset', 'm', 'm')                         # False -> no fixed aspect ratio
        self.offBinWidget = self.createPlotWidget(self.plotTitles[2], 'x-line', 'offset', 'm', 'm')
        self.aziTrkWidget = self.createPlotWidget(self.plotTitles[3], 'inline', 'angle of incidence', 'm', 'deg', False)
        self.aziBinWidget = self.createPlotWidget(self.plotTitles[4], 'x-line', 'angle of incidence', 'm', 'deg', False)    # no fixed aspect ratio
        self.stkTrkWidget = self.createPlotWidget(self.plotTitles[5], 'inline', '|Kr|', 'm', ' 1/km', False)
        self.stkBinWidget = self.createPlotWidget(self.plotTitles[6], 'x-line', '|Kr|', 'm', ' 1/km', False)
        self.stkCelWidget = self.createPlotWidget(self.plotTitles[7], 'Kx', 'Ky', '1/km', '1/km')
        self.offsetWidget = self.createPlotWidget(self.plotTitles[8], '|offset|', 'frequency', 'm', ' #', False)
        self.offAziWidget = self.createPlotWidget(self.plotTitles[9], 'azimuth', '|offset|', 'deg', 'm', False)

        # Create the various views (tabs) on the data
        # Use QCodeEditor with a XmlHighlighter instead of a 'plain' QPlainTextEdit
        # See: https://github.com/luchko/QCodeEditor/blob/master/QCodeEditor.py

        self.textEdit = QCodeEditor(SyntaxHighlighter=XMLHighlighter)           # only one widget on Xml-tab; add directly
        self.textEdit.document().setModified(False)
        self.textEdit.installEventFilter(self)                                  # catch the 'Show' event to connect to toolbar buttons

        # The following tabs have multiple widgets per page, start by giving them a simple QWidget
        self.tabGeom = QWidget()
        self.tabSps = QWidget()
        self.tabTraces = QWidget()

        self.tabGeom.installEventFilter(self)                                   # catch the 'Show' event to connect to toolbar buttons
        self.tabSps.installEventFilter(self)                                    # catch the 'Show' event to connect to toolbar buttons
        self.tabTraces.installEventFilter(self)                                 # catch the 'Show' event to connect to toolbar buttons

        self.createLayoutTab()
        self.createGeomTab()
        self.createSpsTab()
        self.createTraceTableTab()

        # Add tabs to main tab widget
        self.mainTabWidget.addTab(self.layoutWidget, 'Layout')
        self.mainTabWidget.addTab(self.textEdit, 'Xml')
        self.mainTabWidget.addTab(self.tabGeom, 'Geometry')
        self.mainTabWidget.addTab(self.tabSps, 'SPS import')
        self.mainTabWidget.addTab(self.analysisTabWidget, 'Analysis')
        self.mainTabWidget.currentChanged.connect(self.onMainTabChange)         # active tab changed!

        # Add tabs to analysis tab widget
        self.analysisTabWidget.addTab(self.tabTraces, 'Trace table')
        self.analysisTabWidget.addTab(self.offTrkWidget, 'Offset Inline')
        self.analysisTabWidget.addTab(self.offBinWidget, 'Offset X-line')
        self.analysisTabWidget.addTab(self.aziTrkWidget, 'Azi Inline')
        self.analysisTabWidget.addTab(self.aziBinWidget, 'Azi X-line')
        self.analysisTabWidget.addTab(self.stkTrkWidget, 'Stack Inline')
        self.analysisTabWidget.addTab(self.stkBinWidget, 'Stack X-line')
        self.analysisTabWidget.addTab(self.stkCelWidget, 'Kx-Ky Stack')
        self.analysisTabWidget.addTab(self.offsetWidget, '|O| Histogram')
        self.analysisTabWidget.addTab(self.offAziWidget, 'O/A Histogram')
        # self.analysisTabWidget.currentChanged.connect(self.onAnalysisTabChange)   # active tab changed!

        self.setCurrentFileName()

        # connect actions
        self.textEdit.document().modificationChanged.connect(self.setWindowModified)    # forward signal to myself, and make some changes
        self.setWindowModified(self.textEdit.document().isModified())                   # update window status based on document status
        self.textEdit.cursorPositionChanged.connect(self.cursorPositionChanged)         # to show cursor position in statusbar

        self.layoutWidget.scene().sigMouseMoved.connect(self.MouseMovedInPlot)
        self.layoutWidget.getViewBox().sigRangeChangedManually.connect(self.mouseBeingDragged)  # essential to find plotting state for LOD plotting
        self.layoutWidget.plotItem.sigRangeChanged.connect(self.layoutRangeChanged)     # to handle changes in tickmarks when zooming

        self.actionDebug.setCheckable(True)
        self.actionDebug.setChecked(self.debug)
        self.actionDebug.setStatusTip('Show debug information in QGIS Python console')
        self.actionDebug.triggered.connect(self.viewDebug)

        # the following actions are related to the plotWidget
        self.actionZoomAll.triggered.connect(self.layoutWidget.autoRange)
        self.actionZoomRect.setCheckable(True)
        self.actionZoomRect.setChecked(self.rect)
        self.actionZoomRect.triggered.connect(self.plotZoomRect)

        self.actionAspectRatio.setCheckable(True)
        self.actionAspectRatio.setChecked(self.XisY)
        self.actionAspectRatio.triggered.connect(self.plotAspectRatio)

        self.actionAntiAlias.setCheckable(True)
        self.actionAntiAlias.setChecked(self.antiA[0])
        self.actionAntiAlias.triggered.connect(self.plotAntiAlias)

        self.actionPlotGridX.setCheckable(True)
        self.actionPlotGridX.setChecked(self.gridX)
        self.actionPlotGridX.triggered.connect(self.plotGridX)

        self.actionPlotGridY.setCheckable(True)
        self.actionPlotGridY.setChecked(self.gridY)
        self.actionPlotGridY.triggered.connect(self.plotGridY)

        self.actionProjected.setCheckable(True)
        self.actionProjected.setChecked(self.glob)
        self.actionProjected.triggered.connect(self.plotProjected)

        self.actionRuler.setCheckable(True)
        self.actionRuler.setChecked(self.ruler)
        self.actionRuler.triggered.connect(self.plotRuler)

        # actions related to the file menu
        for i in range(config.maxRecentFiles):
            self.recentFileActions.append(QAction(self, visible=False, triggered=self.fileOpenRecent))
            self.menuOpenRecent.addAction(self.recentFileActions[i])

        self.actionNew.triggered.connect(self.newFile)
        self.actionNewLandSurvey.triggered.connect(self.fileNewLandSurvey)
        self.actionNewMarineSurvey.triggered.connect(self.fileNewMarineSurvey)
        self.actionOpen.triggered.connect(self.fileOpen)
        self.actionImportSPS.triggered.connect(self.fileImportSPS)
        self.actionPrint.triggered.connect(self.filePrint)
        self.actionSave.triggered.connect(self.fileSave)
        self.actionSaveAs.triggered.connect(self.fileSaveAs)
        self.actionSettings.triggered.connect(self.fileSettings)

        self.textEdit.document().modificationChanged.connect(self.actionSave.setEnabled)

        # actions related to file -> export
        self.actionExportFoldMap.triggered.connect(self.fileExportFoldMap)
        self.actionExportMinOffsets.triggered.connect(self.fileExportMinOffsets)
        self.actionExportMaxOffsets.triggered.connect(self.fileExportMaxOffsets)

        self.actionExportAnaAsCsv.triggered.connect(self.fileExportAnaAsCsv)

        self.actionExportRecAsCsv.triggered.connect(self.fileExportRecAsCsv)
        self.actionExportSrcAsCsv.triggered.connect(self.fileExportSrcAsCsv)
        self.actionExportRelAsCsv.triggered.connect(self.fileExportRelAsCsv)
        self.actionExportRecAsR01.triggered.connect(self.fileExportRecAsR01)
        self.actionExportSrcAsS01.triggered.connect(self.fileExportSrcAsS01)
        self.actionExportRelAsX01.triggered.connect(self.fileExportRelAsX01)

        self.actionExportRpsAsCsv.triggered.connect(self.fileExportRpsAsCsv)
        self.actionExportSpsAsCsv.triggered.connect(self.fileExportSpsAsCsv)
        self.actionExportXpsAsCsv.triggered.connect(self.fileExportXpsAsCsv)
        self.actionExportRpsAsR01.triggered.connect(self.fileExportRpsAsR01)
        self.actionExportSpsAsS01.triggered.connect(self.fileExportSpsAsS01)
        self.actionExportXpsAsX01.triggered.connect(self.fileExportXpsAsX01)

        self.actionExportFoldMapToQGIS.triggered.connect(self.exportBinToQGIS)
        self.actionExportMinOffsetsToQGIS.triggered.connect(self.exportMinToQGIS)
        self.actionExportMaxOffsetsToQGIS.triggered.connect(self.exportMaxToQGIS)
        self.actionExportRmsOffsetsToQGIS.triggered.connect(self.exportRmsToQGIS)

        self.actionQuit.triggered.connect(self.close)                           # closes the window and arrives at CloseEvent()

        # actions related to the edit menu
        # undo and redo are solely associated with the main xml textEdit
        self.actionUndo.triggered.connect(self.textEdit.undo)
        self.actionRedo.triggered.connect(self.textEdit.redo)
        self.textEdit.document().undoAvailable.connect(self.actionUndo.setEnabled)
        self.textEdit.document().redoAvailable.connect(self.actionRedo.setEnabled)

        # copy, cut, paste and select-all must be managed by all active widgets
        # See: https://stackoverflow.com/questions/40041131/pyqt-global-copy-paste-actions-for-custom-widgets
        self.actionCut.triggered.connect(self.cut)
        self.actionCopy.triggered.connect(self.copy)
        self.actionFind.triggered.connect(self.find)
        self.actionPaste.triggered.connect(self.paste)
        self.actionSelectAll.triggered.connect(self.selectAll)

        # the following setEnabled items need to be re-wired, they are still connected to the textEdit
        self.textEdit.copyAvailable.connect(self.actionCut.setEnabled)
        self.textEdit.copyAvailable.connect(self.actionCopy.setEnabled)

        # actions related to the view menu
        self.actionRefresh.triggered.connect(self.UpdateAllViews)
        self.actionAbout.triggered.connect(self.OnAbout)
        self.actionLicense.triggered.connect(self.OnLicense)
        self.actionHighDpi.triggered.connect(self.OnHighDpi)

        # actions related to the processing menu
        self.actionBasicBinFromTemplates.triggered.connect(self.basicBinFromTemplates)
        self.actionFullBinFromTemplates.triggered.connect(self.fullBinFromTemplates)
        self.actionBasicBinFromGeometry.triggered.connect(self.basicBinFromGeometry)
        self.actionFullBinFromGeometry.triggered.connect(self.fullBinFromGeometry)
        self.actionBasicBinFromSps.triggered.connect(self.basicBinFromSps)
        self.actionFullBinFromSps.triggered.connect(self.fullBinFromSps)
        self.actionGeometryFromTemplates.triggered.connect(self.createGeometryFromTemplates)

        self.actionStopThread.triggered.connect(self.stopWorkerThread)
        self.enableProcessingMenuItems()                                        # enables processing menu items except 'stop processing thread'

        # actions related to geometry items to be displayed
        self.actionTemplates.triggered.connect(self.plotLayout)
        self.actionRecPoints.triggered.connect(self.plotLayout)
        self.actionSrcPoints.triggered.connect(self.plotLayout)
        self.actionRpsPoints.triggered.connect(self.plotLayout)
        self.actionSpsPoints.triggered.connect(self.plotLayout)

        # enable/disable various actions
        self.actionClose.setEnabled(False)
        self.actionSave.setEnabled(self.textEdit.document().isModified())
        # self.actionSaveAs.setEnabled((self.textEdit.document().blockCount() > 1))        # need at least one line of text to save the document
        self.actionSaveAs.setEnabled((True))                                    # need at least one line of text to save the document

        self.actionUndo.setEnabled(self.textEdit.document().isUndoAvailable())
        self.actionRedo.setEnabled(self.textEdit.document().isRedoAvailable())
        self.actionCut.setEnabled(False)
        self.actionCopy.setEnabled(False)
        self.actionPaste.setEnabled(self.clipboardHasText())

        self.updateMenuStatus(True)                                             # keep menu status in sync with program's state

        # make the main tab widget the central widget
        self.setCentralWidget(self.mainTabWidget)

        self.posWidgetStatusbar = QLabel('(x, y): (0.00, 0.00)')
        self.statusbar.addPermanentWidget(self.posWidgetStatusbar, stretch=0)   # widget in bottomright corner of statusbar

        self.parseText(exampleSurveyXmlText())
        self.textEdit.setPlainText(exampleSurveyXmlText())
        self.textEdit.moveCursor(QTextCursor.Start)

        # # # # # # # # # # # # # # # # # # # # # # # # # # # # # # # # # # # # # # # # # # # # # # # # # # # # # # #

        # third docking pane, used to display survey properties
        # defined late, as it needs access the loaded survey object
        self.dockProperty = QDockWidget('Property pane')
        self.dockProperty.setAllowedAreas(Qt.LeftDockWidgetArea | Qt.RightDockWidgetArea)
        self.dockProperty.setStyleSheet('QDockWidget::title {background : lightblue;}')

        # setup the ParameterTree object
        self.paramTree = pg.parametertree.ParameterTree(showHeader=True)        # define parameter tree widget
        self.paramTree.header().setSectionResizeMode(QHeaderView.ResizeMode.Interactive)
        self.paramTree.header().resizeSection(0, 280)
        self.registerParameters()
        self.resetSurveyProperties()                                            # get the parameters into the parameter tree

        self.propertyWidget = QWidget()                                         # placeholder widget to generate a layout
        self.propertyLayout = QVBoxLayout()                                     # required vertical layout

        buttons = QDialogButtonBox.Ok | QDialogButtonBox.Cancel | QDialogButtonBox.Apply
        self.propertyButtonBox = QDialogButtonBox(buttons)                      # define 3 buttons to handle property changes

        # connect 3 buttons (signals) to their event handlers (slots)
        self.propertyButtonBox.accepted.connect(self.applyPropertyChangesAndHide)
        self.propertyButtonBox.rejected.connect(self.resetSurveyProperties)
        self.propertyButtonBox.button(QDialogButtonBox.Apply).clicked.connect(self.applyPropertyChanges)

        self.propertyLayout.addWidget(self.paramTree)                           # add parameter tree to layout
        self.propertyLayout.addStretch()                                        # add some stretch towards 3 buttons
        self.propertyLayout.addWidget(self.propertyButtonBox)                   # add 3 buttons

        self.propertyWidget.setLayout(self.propertyLayout)                      # add layout to widget
        self.dockProperty.setWidget(self.propertyWidget)

        self.addDockWidget(Qt.RightDockWidgetArea, self.dockProperty)           # add docking panel to main window

        self.dockProperty.toggleViewAction().setShortcut(QKeySequence('Ctrl+Alt+p'))
        self.menu_View.addAction(self.dockProperty.toggleViewAction())          # show/hide as requested

        self.menu_View.addSeparator()
        self.menu_View.addAction(self.fileBar.toggleViewAction())
        self.menu_View.addAction(self.editBar.toggleViewAction())
        self.menu_View.addAction(self.graphBar.toggleViewAction())
        self.menu_View.addAction(self.moveBar.toggleViewAction())

        # # # # # # # # # # # # # # # # # # # # # # # # # # # # # # # # # # # # # # # # # # # # # # # # # # # # # # #

        self.plotLayout()

        readSettings(self)
        self.updateRecentFileActions()                                          # update the MRU file menu actions, with info from readSettings()

        self.appendLogMessage('Plugin : Started')
        self.statusbar.showMessage('Ready', 3000)

    def eventFilter(self, source, event):
        if event.type() == QEvent.Show:                                             # do 'cheap' test first
            if isinstance(source, pg.PlotWidget):                                   # do 'expensive' test next

                with contextlib.suppress(RuntimeError):                             # rewire zoomAll button
                    self.actionZoomAll.triggered.disconnect()
                self.actionZoomAll.triggered.connect(source.autoRange)

                plotIndex = self.getVisiblePlotIndex(source)                        # update toolbar status
                if plotIndex is not None:
                    self.actionZoomAll.setEnabled(True)                             # useful for all plots
                    self.actionZoomRect.setEnabled(True)                            # useful for all plots
                    self.actionAspectRatio.setEnabled(True)                         # useful for all plots
                    self.actionAntiAlias.setEnabled(True)                           # useful for plots only
                    self.actionRuler.setEnabled(plotIndex == 0)                     # useful for 1st plot only
                    self.actionProjected.setEnabled(plotIndex == 0)                 # useful for 1st plot only

                    self.actionAntiAlias.setChecked(self.antiA[plotIndex])          # useful for all plots

                    plotItem = source.getPlotItem()
                    self.gridX = plotItem.saveState()['xGridCheck']                 # update x-gridline status
                    self.actionPlotGridX.setChecked(self.gridX)

                    self.gridY = plotItem.saveState()['yGridCheck']                 # update y-gridline status
                    self.actionPlotGridY.setChecked(self.gridY)

                    self.XisY = plotItem.saveState()['view']['aspectLocked']        # update XisY status
                    self.actionAspectRatio.setChecked(self.XisY)

                    viewBox = plotItem.getViewBox()
                    self.rect = viewBox.getState()['mouseMode'] == pg.ViewBox.RectMode  # update rect status
                    self.actionZoomRect.setChecked(self.rect)
                    self.updateVisiblePlotWidget(plotIndex)
                    return True
            else:                                                                   # QEvent.Show; but for different widgets
                self.actionZoomAll.setEnabled(False)                                # useful for plots only
                self.actionZoomRect.setEnabled(False)                               # useful for plots only
                self.actionAspectRatio.setEnabled(False)                            # useful for plots only
                self.actionAntiAlias.setEnabled(False)                              # useful for plots only
                self.actionRuler.setEnabled(False)                                  # useful for 1st plot only
                self.actionProjected.setEnabled(False)                              # useful for 1st plot only
                return True

        return super().eventFilter(source, event)

    def applyPropertyChangesAndHide(self):
        self.applyPropertyChanges()
        self.dockProperty.hide()

    def registerParameters(self):
        registerAllParameterTypes()

    def resetSurveyProperties(self):
        self.paramTree.clear()

        # set the survey object in the property pane using current survey properties
        copy = self.survey.deepcopy()

        # first (globally) define the patterns to choose from
        config.patternList = ['<no pattern>']
        for p in copy.patternList:
            config.patternList.append(p.name)

        # first copy the crs for global access (need to fix this later)
        config.surveyCrs = copy.crs

        # brush color for main parameter categories
        brush = '#add8e6'

        surveyParams = [
            dict(brush=brush, name='Survey configuration', type='myConfiguration', value=copy),
            dict(brush=brush, name='Survey analysis', type='myAnalysis', value=copy),
            dict(brush=brush, name='Survey reflectors', type='myReflectors', value=copy),
            dict(brush=brush, name='Survey grid', type='myGrid', value=copy.grid),
            dict(brush=brush, name='Block list', type='myBlockList', value=copy.blockList),
            dict(brush=brush, name='Pattern list', type='myPatternList', value=copy.patternList),
        ]

        # surveyParams = dict(name='Survey configuration',type='mySurvey', value=copy, brush='#add8e6')
        # self.parameters = pg.parametertree.Parameter.create(name='Survey configuration', type='mySurvey', value=copy, brush='#add8e6')

        self.parameters = pg.parametertree.Parameter.create(name='Survey Properties', type='group', children=surveyParams)
        self.parameters.sigTreeStateChanged.connect(self.propertyTreeStateChanged)
        self.paramTree.setParameters(self.parameters, showTop=False)

        # Make sure we get a notification, when the binning area has changed, to ditch the analysis files
        self.anaChild = self.parameters.child('Survey analysis')
        self.binChild = self.anaChild.child('Binning area')
        self.binChild.sigTreeStateChanged.connect(self.binAreaHasChanged)

        # deal with a bug, not showing tooltip information in the list of parameterItems
        for item in self.paramTree.listAllItems():                              # Bug. See: https://github.com/pyqtgraph/pyqtgraph/issues/2744
            p = item.param                                                      # get parameter belonging to parameterItem
            p.setToDefault()                                                    # set all parameters to their default value
            if hasattr(item, 'updateDefaultBtn'):                               # note: not all parameterItems have this method
                item.updateDefaultBtn()                                         # reset the default-buttons to their grey value
            if 'tip' in p.opts:                                                 # this solves the above mentioned bug
                item.setToolTip(0, p.opts['tip'])                               # the widgets now get their tooltips

    def applyPropertyChanges(self):
        # build new survey object from scratch, and start adding to it
        copy = RollSurvey()

        CFG = self.parameters.child('Survey configuration')
        # copy.crs = CFG.child('Survey CRS').value()
        # copy.type = SurveyType[CFG.child('Survey type').value()]
        # copy.name = CFG.child('Survey name').value()

        copy.crs, surType, copy.name = CFG.value()                              # get tuple of data from parameter
        copy.type = SurveyType[surType]                                         # SurveyType is an enum
        config.surveyCrs = copy.crs                                             # needed for global access to crs

        ANA = self.parameters.child('Survey analysis')
        copy.output.rctOutput, copy.angles, copy.binning, copy.offset, copy.unique = ANA.value()

        REF = self.parameters.child('Survey reflectors')
        copy.globalPlane, copy.globalSphere = REF.value()

        GRD = self.parameters.child('Survey grid')
        copy.grid = GRD.value()

        BLK = self.parameters.child('Block list')
        copy.blockList = BLK.value()

        PAT = self.parameters.child('Pattern list')
        copy.patternList = PAT.value()

        # first check survey integrity before committing to it.
        if copy.checkIntegrity() is False:
            return

        self.survey = copy.deepcopy()                                           # start using the updated survey object

        # update the survey object with the necessary steps
        self.survey.calcTransforms()                                            # (re)calculate the transforms being used
        self.survey.calcSeedData()                                              # needed for circles, spirals & well-seeds; may affect bounding box
        self.survey.calcBoundingRect()                                          # (re)calculate the boundingBox as part of parsing the data
        self.survey.calcNoShotPoints()                                          # (re)calculate nr of SPs

        plainText = self.survey.toXmlString()                                   # convert the survey object itself to an Xml string
        self.textEdit.setTextViaCursor(plainText)                               # get text into the textEdit, NOT resetting its doc status
        self.textEdit.document().setModified(True)                              # we edited the document; so it's been modified

        if self.binAreaChanged:                                                 # we need to throw away the analysis results
            self.binAreaChanged = False                                         # reset this flag

            self.inlineStk = None                                               # numpy array with inline Kr stack reponse
            self.x_lineStk = None                                               # numpy array with x_line Kr stack reponse
            self.xyCellStk = None                                               # numpy array with cell's KxKy stack response
            self.xyPatResp = None                                               # numpy array with pattern's KxKy response
            self.offAziPlt = None                                               # numpy array with offset distribution

            self.survey.output.binOutput = None                                 # numpy array with foldmap
            self.survey.output.minOffset = None                                 # numpy array with minimum offset
            self.survey.output.maxOffset = None                                 # numpy array with maximum offset
            self.survey.output.rmsOffset = None                                 # numpy array with rms delta offset

            binFileName = self.fileName + '.bin.npy'                            # file names for analysis files
            minFileName = self.fileName + '.min.npy'
            maxFileName = self.fileName + '.max.npy'

            try:
                os.remove(binFileName)                                          # remove file names, if possible
                os.remove(minFileName)
                os.remove(maxFileName)
            except OSError:
                print('cant delete analysis file')

            if self.survey.output.anaOutput is not None:                        # remove memory mapped file, as well
                self.survey.output.anaOutput.flush()
                # self.survey.output.anaOutput._mmap.close()                    # See: https://stackoverflow.com/questions/6397495/unmap-of-numpy-memmap
                del self.survey.output.anaOutput
                self.D2_Output = None                                           # first remove reference to self.survey.output.anaOutput
                self.survey.output.anaOutput = None                             # then remove self.survey.output.anaOutput itself
                gc.collect()                                                    # get the garbage collector going

            self.updateMenuStatus(True)                                         # keep menu status in sync with program's state

        self.appendLogMessage(f'Edited : {self.fileName} survey object updated')
        self.plotLayout()

    def binAreaHasChanged(self, *_):                                               # param, changes unused; replaced by *_
        self.binAreaChanged = True

    ## If anything changes in the tree, print a message
    def propertyTreeStateChanged(self, param, changes):
        # self.propertyButtonBox.button(QDialogButtonBox.Apply).setEnabled(True)
        print('┌── sigTreeStateChanged --> tree changes:')
        for param, change, data in changes:
            path = self.parameters.childPath(param)
            if path is not None:
                childName = '.'.join(path)
            else:
                childName = param.name()
            print(f'│     parameter: {childName}')
            print(f'│     change:    {change}')
            print(f'│     data:      {str(data)}')
            print('└───────────────────────────────────────')

    def onMainTabChange(self, index):                                           # manage focus when active tab is changed; doesn't work 100% yet !
        if index == 0:                                                          # main plotting widget
            self.handleSpiderPlot()
        else:
            widget = self.mainTabWidget.currentWidget()
            if isinstance(widget, QCodeEditor):
                widget.setFocus()

    def cut(self):
        currentWidget = QApplication.focusWidget()
        try:
            currentWidget.cut()
        except AttributeError as e:                                             # current widget does not support cut(), so ignore command
            print('Exception occurred: ', e)
        self.actionPaste.setEnabled(self.clipboardHasText())

    def copy(self):
        currentWidget = QApplication.focusWidget()
        try:
            currentWidget.copy()
        except AttributeError as e:                                             # current widget does not support copy(), so ignore command
            print('Exception occurred: ', e)
        self.actionPaste.setEnabled(self.clipboardHasText())

    def paste(self):
        currentWidget = QApplication.focusWidget()
        try:
            currentWidget.paste()
        except AttributeError as e:                                             # current widget does not support paste(), so ignore command
            print('Exception occurred: ', e)
        return

    def selectAll(self):
        currentWidget = QApplication.focusWidget()
        try:
            currentWidget.selectAll()
        except AttributeError as e:                                             # current widget does not support selectAll(), so ignore command
            print('Exception occurred: ', e)
        return

    def find(self):
        # find only operates on the xml-text edit
        self.mainTabWidget.setCurrentIndex(1)                                   # make sure we display the 'xml' tab
        Find(self).show()                                                       # show find and replace dialog

    def createPlotWidget(self, plotTitle='', xAxisTitle='', yAxisTitle='', unitX='', unitY='', aspectLocked=True):
        """Create a plot widget for first usage"""
        w = pg.PlotWidget(background='w')
        w.setAspectLocked(True)                                                 # setting can be changed through a toolbar
        w.showGrid(x=True, y=True, alpha=0.75)                                  # shows the grey grid lines
        w.setMinimumSize(150, 150)                                              # prevent excessive widget shrinking
        # See: https://stackoverflow.com/questions/44402399/how-to-disable-the-default-context-menu-of-pyqtgraph for context menu options
        w.setContextMenuActionVisible('Transforms', False)
        w.setContextMenuActionVisible('Downsample', False)
        w.setContextMenuActionVisible('Average', False)
        w.setContextMenuActionVisible('Alpha', False)
        w.setContextMenuActionVisible('Points', False)

        # w.setMenuEnabled(False, enableViewBoxMenu=None)                       # get rid of context menu but keep ViewBox menu
        # w.ctrlMenu = None                                                     # get rid of 'Plot Options' in context menu
        # w.scene().contextMenu = None                                          # get rid of 'Export' in context menu

        # set up plot title
        w.setTitle(plotTitle, color='b', size='16pt')

        # setup axes
        styles = {'color': '#000', 'font-size': '10pt'}
        w.showAxes(True, showValues=(True, False, False, True))                 # show values at the left and at the bottom
        w.setLabel('bottom', xAxisTitle, units=unitX, **styles)                 # shows axis at the bottom, and shows the units label
        w.setLabel('left', yAxisTitle, units=unitY, **styles)                   # shows axis at the left, and shows the units label
        w.setLabel('top', ' ', **styles)                                        # shows axis at the top, no label, no tickmarks
        w.setLabel('right', ' ', **styles)                                      # shows axis at the right, no label, no tickmarks
        w.setAspectLocked(aspectLocked)

        w.installEventFilter(self)                                              # filter the 'Show' event to connect to toolbar buttons
        return w

    def resetPlotWidget(self, w, plotTitle):
        w.plotItem.clear()
        w.setTitle(plotTitle, color='b', size='16pt')

    # create tabbed pages
    def createLayoutTab(self):
        self.layoutWidget = self.createPlotWidget()

    def createTraceTableTab(self):
        # analysis table; to copy data to clipboard, create a subclassed QTableView, see bottom of following article:
        # See: https://stackoverflow.com/questions/40225270/copy-paste-multiple-items-from-qtableview-in-pyqt4
        self.anaModel = AnaTableModel(self.D2_Output)

        # See: https://stackoverflow.com/questions/7840325/change-the-selection-color-of-a-qtablewidget
        table_style = 'QTableView::item:selected{background-color : #add8e6;selection-color : #000000;}'

        # first create the widget(s)
        self.anaView = TableView()
        self.anaView.setModel(self.anaModel)
        self.anaView.resizeColumnsToContents()
        self.anaView.setStyleSheet(table_style)                                 # define selection colors

        label_style = 'font-family: Arial; font-weight: bold; font-size: 16px;'
        self.anaLabel = QLabel('\nANALYSIS records')
        self.anaLabel.setAlignment(Qt.AlignCenter)
        self.anaLabel.setStyleSheet(label_style)

        # then create the layout
        self.tabTraces.layout = QVBoxLayout(self)
        self.tabTraces.layout.setContentsMargins(1, 1, 1, 1)
        self.tabTraces.layout.addWidget(self.anaLabel)
        self.tabTraces.layout.addWidget(self.anaView)

        # put table on traces tab
        self.tabTraces.setLayout(self.tabTraces.layout)

    def createGeomTab(self):
        table_style = 'QTableView::item:selected{background-color : #add8e6;selection-color : #000000;}'
        label_style = 'font-family: Arial; font-weight: bold; font-size: 16px;'

        # first create the main widgets
        self.srcView = TableView()                                              # create src view
        self.srcModel = SpsTableModel(self.srcGeom)                             # create src model
        self.srcView.setModel(self.srcModel)                                    # add the model to the view
        self.srcView.setStyleSheet(table_style)                                 # define selection colors

        self.relView = TableView()                                              # create rel view
        self.relModel = XpsTableModel(self.relGeom)                             # create rel model
        self.relView.setModel(self.relModel)                                    # add the model to the view
        self.relHdrView = self.relView.horizontalHeader()                       # to detect button clicks here
        self.relHdrView.sectionClicked.connect(self.sortRelData)                # handle the section-clicked signal
        self.relView.setStyleSheet(table_style)                                 # define selection colors

        self.recView = TableView()                                              # create rec view
        self.recModel = RpsTableModel(self.recGeom)                             # create rec model
        self.recView.setModel(self.recModel)                                    # add the model to the view
        self.recView.setStyleSheet(table_style)                                 # define selection colors

        self.srcLabel = QLabel('SRC records')
        self.srcLabel.setAlignment(Qt.AlignCenter)
        self.srcLabel.setStyleSheet(label_style)

        self.relLabel = QLabel('REL records')
        self.relLabel.setAlignment(Qt.AlignCenter)
        self.relLabel.setStyleSheet(label_style)

        self.recLabel = QLabel('REC records')
        self.recLabel.setAlignment(Qt.AlignCenter)
        self.recLabel.setStyleSheet(label_style)

        # then create widget containers for the layout
        self.srcPane = QWidget()
        self.relPane = QWidget()
        self.recPane = QWidget()

        # then create button layout
        self.btnSrcRemoveDuplicates = QPushButton('Remove &Duplicates')
        self.btnSrcRemoveOrphans = QPushButton('Remove &REL-orphins')

        self.btnSrcExportToQGIS = QPushButton('Export to QGIS')
        self.btnSrcUpdateToQGIS = QPushButton('Update QGIS')
        self.btnSrcReadFromQGIS = QPushButton('Read from QGIS')

        self.btnRelRemoveSrcOrphans = QPushButton('Remove &SRC-orphins')
        self.btnRelRemoveDuplicates = QPushButton('Remove &Duplicates')
        self.btnRelRemoveRecOrphans = QPushButton('Remove &REC-orphins')

        self.btnRecRemoveDuplicates = QPushButton('Remove &Duplicates')
        self.btnRecRemoveOrphans = QPushButton('Remove &REL-orphins')

        self.btnRecExportToQGIS = QPushButton('Export to QGIS')
        self.btnRecUpdateToQGIS = QPushButton('Update QGIS')
        self.btnRecReadFromQGIS = QPushButton('Read from QGIS')

        self.btnRelExportToQGIS = QPushButton('Export Src, Cmp, Rec && Binning &Boundaries to QGIS')
        self.btnRelExportToQGIS.setToolTip('This button is enabled once you have saved the project')

        # make the buttons stand out a bit
        # See: https://www.webucator.com/article/python-color-constants-module/
        self.btnSrcRemoveDuplicates.setStyleSheet('background-color:lavender; font-weight:bold;')
        self.btnSrcRemoveOrphans.setStyleSheet('background-color:lavender; font-weight:bold;')

        self.btnRelRemoveSrcOrphans.setStyleSheet('background-color:lavender; font-weight:bold;')
        self.btnRelRemoveDuplicates.setStyleSheet('background-color:lavender; font-weight:bold;')
        self.btnRelRemoveRecOrphans.setStyleSheet('background-color:lavender; font-weight:bold;')

        self.btnRecRemoveDuplicates.setStyleSheet('background-color:lavender; font-weight:bold;')
        self.btnRecRemoveOrphans.setStyleSheet('background-color:lavender; font-weight:bold;')

        self.btnSrcExportToQGIS.setStyleSheet('background-color:lightgoldenrodyellow; font-weight:bold;')
        self.btnSrcUpdateToQGIS.setStyleSheet('background-color:lightgoldenrodyellow; font-weight:bold;')
        self.btnSrcReadFromQGIS.setStyleSheet('background-color:lightgoldenrodyellow; font-weight:bold;')

        self.btnRecExportToQGIS.setStyleSheet('background-color:lightgoldenrodyellow; font-weight:bold;')
        self.btnRecUpdateToQGIS.setStyleSheet('background-color:lightgoldenrodyellow; font-weight:bold;')
        self.btnRecReadFromQGIS.setStyleSheet('background-color:lightgoldenrodyellow; font-weight:bold;')

        self.btnRelExportToQGIS.setStyleSheet('background-color:lightgoldenrodyellow; font-weight:bold;')

        # these buttons have signals
        self.btnSrcRemoveDuplicates.pressed.connect(self.removeSrcDuplicates)   # src buttons & actions
        self.btnSrcRemoveOrphans.pressed.connect(self.removeSrcOrphans)
        self.actionExportSrcToQGIS.triggered.connect(self.exportSrcToQgis)      # export src records to QGIS
        self.btnSrcExportToQGIS.pressed.connect(self.exportSrcToQgis)           # export src records to QGIS
        self.btnSrcUpdateToQGIS.pressed.connect(self.updateSrcToQGIS)
        self.btnSrcReadFromQGIS.pressed.connect(self.importSrcFromQgis)

        self.btnRelRemoveDuplicates.pressed.connect(self.removeRelDuplicates)   # rel buttons & actions
        self.btnRelRemoveSrcOrphans.pressed.connect(self.removeRelSrcOrphans)
        self.btnRelRemoveRecOrphans.pressed.connect(self.removeRelRecOrphans)
        self.actionExportAreasToQGIS.triggered.connect(self.exportOutToQgis)    # export survey outline to QGIS
        self.btnRelExportToQGIS.pressed.connect(self.exportOutToQgis)           # export survey outline to QGIS

        self.btnRecRemoveDuplicates.pressed.connect(self.removeRecDuplicates)   # rec buttons & actions
        self.btnRecRemoveOrphans.pressed.connect(self.removeRecOrphans)
        self.actionExportRecToQGIS.triggered.connect(self.exportRecToQgis)      # export rec records to QGIS
        self.btnRecExportToQGIS.pressed.connect(self.exportRecToQgis)           # export rec records to QGIS
        self.btnRecUpdateToQGIS.pressed.connect(self.updateRecToQGIS)
        self.btnRecReadFromQGIS.pressed.connect(self.importRecFromQgis)

        self.btnBinToQGIS.pressed.connect(self.exportBinToQGIS)                 # figures
        self.btnMinToQGIS.pressed.connect(self.exportMinToQGIS)
        self.btnMaxToQGIS.pressed.connect(self.exportMaxToQGIS)
        self.btnRmsToQGIS.pressed.connect(self.exportRmsToQGIS)

        label1 = QLabel('«-Cleanup table-»')
        label1.setStyleSheet('border: 1px solid black;background-color:lavender')
        label1.setAlignment(Qt.AlignCenter)

        label2 = QLabel('«-Cleanup table-»')
        label2.setStyleSheet('border: 1px solid black;background-color:lavender')
        label2.setAlignment(Qt.AlignCenter)

        grid1 = QGridLayout()
        grid1.addWidget(self.btnSrcRemoveDuplicates, 0, 0)
        grid1.addWidget(label1, 0, 1)
        grid1.addWidget(self.btnSrcRemoveOrphans, 0, 2)

        grid1.addWidget(self.btnSrcExportToQGIS, 1, 0)
        grid1.addWidget(self.btnSrcUpdateToQGIS, 1, 1)
        grid1.addWidget(self.btnSrcReadFromQGIS, 1, 2)

        grid2 = QGridLayout()
        grid2.addWidget(self.btnRelRemoveSrcOrphans, 0, 0)
        grid2.addWidget(self.btnRelRemoveDuplicates, 0, 1)
        grid2.addWidget(self.btnRelRemoveRecOrphans, 0, 2)
        grid2.addWidget(self.btnRelExportToQGIS, 1, 0, 1, 3)

        grid3 = QGridLayout()
        grid3.addWidget(self.btnRecRemoveOrphans, 0, 0)
        grid3.addWidget(label2, 0, 1)
        grid3.addWidget(self.btnRecRemoveDuplicates, 0, 2)

        grid3.addWidget(self.btnRecExportToQGIS, 1, 0)
        grid3.addWidget(self.btnRecUpdateToQGIS, 1, 1)
        grid3.addWidget(self.btnRecReadFromQGIS, 1, 2)

        # then create the three vertical layouts
        vbox1 = QVBoxLayout()
        vbox1.addWidget(self.srcLabel)
        vbox1.addWidget(self.srcView)
        vbox1.addLayout(grid1)

        vbox2 = QVBoxLayout()
        vbox2.addWidget(self.relLabel)
        vbox2.addWidget(self.relView)
        vbox2.addLayout(grid2)

        vbox3 = QVBoxLayout()
        vbox3.addWidget(self.recLabel)
        vbox3.addWidget(self.recView)
        vbox3.addLayout(grid3)

        # set the layout for the three panes
        self.srcPane.setLayout(vbox1)
        self.relPane.setLayout(vbox2)
        self.recPane.setLayout(vbox3)

        # Create the widgets for the bottom pane
        self.geomBottom = QPlainTextEdit()
        self.geomBottom.appendHtml('<b>Navigation:</b>')
        self.geomBottom.appendHtml('Use <b>Ctrl + Page-Up / Page-Down</b> to find next duplicate record.')
        self.geomBottom.appendHtml('Use <b>Ctrl + Up-arrow / Down-arrow</b> to find next source orphan.')
        self.geomBottom.appendHtml('Use <b>Ctrl + Left-arrow / Right-arrow</b> to find next receiver orphan.')
        self.geomBottom.appendHtml('The <b>XPS records</b> are only tested for valid rec-station values in the <b>rec min</b> and <b>rec max</b> columns (and not for any stations in between).')
        self.geomBottom.setReadOnly(True)                                        # if we set this 'True' the context menu no longer allows 'delete', just 'select all' and 'copy'

        self.geomBottom.setFrameShape(QFrame.StyledPanel)
        self.geomBottom.setStyleSheet('background-color:lavender')   # See: https://www.w3.org/TR/SVG11/types.html#ColorKeywords

        # use splitters to be able to rearrange the layout
        splitter1 = QSplitter(Qt.Horizontal)
        splitter1.addWidget(self.srcPane)
        splitter1.addWidget(self.relPane)
        splitter1.addWidget(self.recPane)
        splitter1.setSizes([200, 200, 200])

        splitter2 = QSplitter(Qt.Vertical)
        splitter2.addWidget(splitter1)
        splitter2.addWidget(self.geomBottom)
        splitter2.setSizes([900, 100])

        # ceate the main layout for the SPS tab
        hbox = QHBoxLayout()
        hbox.addWidget(splitter2)

        self.tabGeom.setLayout(hbox)

    def createSpsTab(self):
        table_style = 'QTableView::item:selected{background-color : #add8e6;selection-color : #000000;}'
        label_style = 'font-family: Arial; font-weight: bold; font-size: 16px;'

        # first create the main widgets
        self.spsView = TableView()                                              # create sps view
        self.spsModel = SpsTableModel(self.spsImport)                          # create sps model
        self.spsView.setModel(self.spsModel)                                    # add the model to the view
        self.spsView.setStyleSheet(table_style)                                 # define selection colors

        self.xpsView = TableView()                                              # create xps view
        self.xpsModel = XpsTableModel(self.xpsImport)                           # create xps model
        self.xpsView.setModel(self.xpsModel)                                    # add the model to the view
        self.xpsHdrView = self.xpsView.horizontalHeader()                       # to detect button clicks here
        self.xpsHdrView.sectionClicked.connect(self.sortXpsData)                # handle the section-clicked signal
        self.xpsView.setStyleSheet(table_style)                                 # define selection colors

        self.rpsView = TableView()                                              # create rps view
        self.rpsModel = RpsTableModel(self.rpsImport)                           # create xps model
        self.rpsView.setModel(self.rpsModel)                                    # add the model to the view
        self.rpsView.setStyleSheet(table_style)                                 # define selection colors

        self.spsLabel = QLabel('SPS records')
        self.spsLabel.setAlignment(Qt.AlignCenter)
        self.spsLabel.setStyleSheet(label_style)

        self.xpsLabel = QLabel('XPS records')
        self.xpsLabel.setAlignment(Qt.AlignCenter)
        self.xpsLabel.setStyleSheet(label_style)

        self.rpsLabel = QLabel('RPS records')
        self.rpsLabel.setAlignment(Qt.AlignCenter)
        self.rpsLabel.setStyleSheet(label_style)

        # then create widget containers for the layout
        self.spsPane = QWidget()
        self.xpsPane = QWidget()
        self.rpsPane = QWidget()

        # then create button layout
        self.btnSpsRemoveDuplicates = QPushButton('Remove Duplicates')
        self.btnSpsExportToQGIS = QPushButton('Export to QGIS')
        self.btnSpsRemoveOrphans = QPushButton('Remove &XPS-orphins')

        self.btnXpsRemoveSpsOrphans = QPushButton('Remove SPS-orphins')
        self.btnXpsRemoveDuplicates = QPushButton('Remove Duplicates')
        self.btnXpsRemoveRpsOrphans = QPushButton('Remove RPS-orphins')

        self.btnRpsRemoveDuplicates = QPushButton('Remove &Duplicates')
        self.btnRpsExportToQGIS = QPushButton('Export to QGIS')
        self.btnRpsRemoveOrphans = QPushButton('Remove &XPS-orphins')

        # make the export buttons stand out a bit
        self.btnSpsExportToQGIS.setStyleSheet('background-color:lightgoldenrodyellow; font-weight:bold;')
        self.btnRpsExportToQGIS.setStyleSheet('background-color:lightgoldenrodyellow; font-weight:bold;')

        # these buttons have signals
        self.btnSpsRemoveDuplicates.pressed.connect(self.removeSpsDuplicates)

        self.actionExportSpsToQGIS.triggered.connect(self.exportSpsToQgis)      # export sps records to QGIS
        self.btnSpsExportToQGIS.pressed.connect(self.exportSpsToQgis)           # export sps records to QGIS
        self.btnSpsRemoveOrphans.pressed.connect(self.removeSpsOrphans)

        self.btnXpsRemoveSpsOrphans.pressed.connect(self.removeXpsSpsOrphans)
        self.btnXpsRemoveDuplicates.pressed.connect(self.removeXpsDuplicates)
        self.btnXpsRemoveRpsOrphans.pressed.connect(self.removeXpsRpsOrphans)

        self.btnRpsRemoveDuplicates.pressed.connect(self.removeRpsDuplicates)
        self.actionExportRpsToQGIS.triggered.connect(self.exportRpsToQgis)      # export rps records to QGIS
        self.btnRpsExportToQGIS.pressed.connect(self.exportRpsToQgis)           # export rps records to QGIS
        self.btnRpsRemoveOrphans.pressed.connect(self.removeRpsOrphans)

        grid1 = QGridLayout()
        grid1.addWidget(self.btnSpsRemoveDuplicates, 0, 0)
        grid1.addWidget(self.btnSpsExportToQGIS, 0, 1)
        grid1.addWidget(self.btnSpsRemoveOrphans, 0, 2)

        grid2 = QGridLayout()
        grid2.addWidget(self.btnXpsRemoveSpsOrphans, 0, 0)
        grid2.addWidget(self.btnXpsRemoveDuplicates, 0, 1)
        grid2.addWidget(self.btnXpsRemoveRpsOrphans, 0, 2)

        grid3 = QGridLayout()
        grid3.addWidget(self.btnRpsRemoveOrphans, 0, 0)
        grid3.addWidget(self.btnRpsExportToQGIS, 0, 1)
        grid3.addWidget(self.btnRpsRemoveDuplicates, 0, 2)

        # then create the three vertical layouts
        vbox1 = QVBoxLayout()
        vbox1.addWidget(self.spsLabel)
        vbox1.addWidget(self.spsView)
        vbox1.addLayout(grid1)

        vbox2 = QVBoxLayout()
        vbox2.addWidget(self.xpsLabel)
        vbox2.addWidget(self.xpsView)
        vbox2.addLayout(grid2)

        vbox3 = QVBoxLayout()
        vbox3.addWidget(self.rpsLabel)
        vbox3.addWidget(self.rpsView)
        vbox3.addLayout(grid3)

        # set the layout for the three panes
        self.spsPane.setLayout(vbox1)
        self.xpsPane.setLayout(vbox2)
        self.rpsPane.setLayout(vbox3)

        # Create the widgets for the bottom pane
        self.spsBottom = QPlainTextEdit()
        self.spsBottom.appendHtml('<b>Navigation:</b>')
        self.spsBottom.appendHtml('Use <b>Ctrl + Page-Up / Page-Down</b> to find next duplicate record.')
        self.spsBottom.appendHtml('Use <b>Ctrl + Up-arrow / Down-arrow</b> to find next source orphan.')
        self.spsBottom.appendHtml('Use <b>Ctrl + Left-arrow / Right-arrow</b> to find next receiver orphan.')
        self.spsBottom.appendHtml('The <b>XPS records</b> are only tested for valid rec-station values in the <b>rec min</b> and <b>rec max</b> columns (and not for any stations in between).')
        self.spsBottom.setReadOnly(True)                                        # if we set this 'True' the context menu no longer allows 'delete', just 'select all' and 'copy'

        self.spsBottom.setFrameShape(QFrame.StyledPanel)
        self.spsBottom.setStyleSheet('background-color:lightgoldenrodyellow')   # See: https://www.w3.org/TR/SVG11/types.html#ColorKeywords

        # use splitters to be able to rearrange the layout
        splitter1 = QSplitter(Qt.Horizontal)
        splitter1.addWidget(self.spsPane)
        splitter1.addWidget(self.xpsPane)
        splitter1.addWidget(self.rpsPane)
        splitter1.setSizes([200, 200, 200])

        splitter2 = QSplitter(Qt.Vertical)
        splitter2.addWidget(splitter1)
        splitter2.addWidget(self.spsBottom)
        splitter2.setSizes([900, 100])

        # ceate the main layout for the SPS tab
        hbox = QHBoxLayout()
        hbox.addWidget(splitter2)

        self.tabSps.setLayout(hbox)

    # deal with the spider navigation
    # See: https://stackoverflow.com/questions/49316067/how-get-pressed-keys-in-mousepressevent-method-with-qt
    def spiderGoRt(self, *_, direction=Direction.Rt):
        self.navigateSpider(direction=direction)

    def spiderGoLt(self, *_, direction=Direction.Lt):
        self.navigateSpider(direction=direction)

    def spiderGoUp(self, *_, direction=Direction.Up):
        self.navigateSpider(direction=direction)

    def spiderGoDn(self, *_, direction=Direction.Dn):
        self.navigateSpider(direction=direction)

    # See: https://groups.google.com/g/pyqtgraph/c/kewFL4LkoYE?pli=1 to add pg.TextItem to indicate fold & location
    def handleSpiderPlot(self):
        if self.tbSpider.isChecked():
            self.navigateSpider(Direction.NA)
        else:
            self.plotLayout()

    def navigateSpider(self, direction):
        if self.survey.output.anaOutput is None or self.survey.output.binOutput is None:
            return

        step = 1
        # See: https://doc.qt.io/archives/qt-4.8/qapplication.html#queryKeyboardModifiers
        modifierPressed = QApplication.keyboardModifiers()
        if (modifierPressed & Qt.ControlModifier) == Qt.ControlModifier:
            step = 10
        if (modifierPressed & Qt.ShiftModifier) == Qt.ShiftModifier:
            step *= 5

        xSize = self.survey.output.anaOutput.shape[0]
        ySize = self.survey.output.anaOutput.shape[1]

        if self.spiderPoint == QPoint(-1, -1):                                  # no valid position yet; move to center
            self.spiderPoint = QPoint(xSize // 2, ySize // 2)
        elif direction == Direction.Rt:                                         # valid position, so move spider around
            self.spiderPoint += QPoint(1, 0) * step
        elif direction == Direction.Lt:
            self.spiderPoint -= QPoint(1, 0) * step
        elif direction == Direction.Up:
            self.spiderPoint += QPoint(0, 1) * step
        elif direction == Direction.Dn:
            self.spiderPoint -= QPoint(0, 1) * step

        if self.spiderPoint.x() < 0:
            self.spiderPoint.setX(0)

        if self.spiderPoint.y() < 0:
            self.spiderPoint.setY(0)

        if self.spiderPoint.x() >= xSize:
            self.spiderPoint.setX(xSize - 1)

        if self.spiderPoint.y() >= ySize:
            self.spiderPoint.setY(ySize - 1)

        nX = self.spiderPoint.x()                                               # get x, y indices into bin array
        nY = self.spiderPoint.y()

        try:                                                                    # protect against potential index errors
            fold = self.survey.output.binOutput[nX, nY]                         # max fold; accounting for unique fold
            fold = min(fold, self.survey.output.anaOutput.shape[2])             # 3rd dimension in analysis file reflects available records in cmp
        except IndexError:                                                      # in case array is resized during FileNew()
            return                                                              # something went wrong accessing binning array

        plotIndex = self.getVisiblePlotWidget()[1]                              # get the active plot widget nr
        if plotIndex == 0:                                                      # update the layout plot

            self.spiderSrcX, self.spiderSrcY, self.spiderRecX, self.spiderRecY = numbaSpiderBin(self.survey.output.anaOutput[nX, nY, 0:fold, :])

            invBinTransform, _ = self.survey.binTransform.inverted()            # need to go from bin nr's to cmp(x, y)
            cmpX, cmpY = invBinTransform.map(nX, nY)                            # get local coordinates from line and point indices
            stkX, stkY = self.survey.st2Transform.map(cmpX, cmpY)               # get the corresponding bin and stake numbers

            if fold > 0:
                labelX = cmpX
                labelY = max(self.spiderRecY.max(), self.spiderSrcY.max())
            else:
                labelX = cmpX
                labelY = cmpY

            if self.glob:                                                       # we need to convert local to global coords
                labelX, labelY = self.survey.glbTransform.map(labelX, labelY)   # get global position from local version

            if self.spiderText is None:
                self.spiderText = pg.TextItem(anchor=(0.5, 1.3), border='b', color='b', fill=(130, 255, 255, 200), text='spiderLabel')
                self.spiderText.setZValue(1000)

            self.spiderText.setPos(labelX, labelY)                                  # move the label above spider's cmp position
            self.spiderText.setText(f'S({int(stkX)},{int(stkY)}), fold = {fold}')   # update the label text accordingly
            self.plotLayout()

        else:
            self.updateVisiblePlotWidget(plotIndex)                             # update one of the analysis plots

        # now sync the trace table with the spider position
        sizeY = self.survey.output.anaOutput.shape[1]                           # x-line size of analysis array
        maxFld = self.survey.output.anaOutput.shape[2]                          # max fold from analysis file
        offset = (nX * sizeY + nY) * maxFld                                     # calculate offset for self.D2_Output array
        index = self.anaView.model().index(offset, 0)                           # turn offset into index
        self.anaView.scrollTo(index)                                            # scroll to index
        self.anaView.selectRow(offset)                                          # for the time being, *only* select first row of traces in a bin

        fold = max(fold, 1)                                                     # only highlight one line when fold = 0
        TL = QModelIndex(self.anaView.model().index(offset, 0))
        BR = QModelIndex(self.anaView.model().index(offset + fold - 1, 0))
        selection = QItemSelection(TL, BR)

        sm = self.anaView.selectionModel()                                      # select corresponding rows in self.anaView table
        sm.select(selection, QItemSelectionModel.ClearAndSelect | QItemSelectionModel.Rows)

    # define several sps, rps, xps button functions

    def sortXpsData(self, index):
        if self.xpsImport is None:
            return

        self.xpsModel.setSort(index)

        if index < 3:
            self.xpsImport.sort(order=['SrcInd', 'SrcLin', 'SrcPnt', 'RecInd', 'RecLin', 'RecMin', 'RecMax'])
        elif index == 3:
            self.xpsImport.sort(order=['RecNum', 'SrcInd', 'SrcLin', 'SrcPnt', 'RecInd', 'RecLin', 'RecMin', 'RecMax'])
        else:
            self.xpsImport.sort(order=['RecInd', 'RecLin', 'RecMin', 'RecMax'])
        self.xpsModel.setData(self.xpsImport)

    def removeRpsDuplicates(self):
        if self.rpsImport is None:
            return

        self.rpsImport, before, after = deletePntDuplicates(self.rpsImport)
        self.rpsModel.setData(self.rpsImport)                                   # update the model's data
        if after < before:                                                      # need to update the (x, y) points as well
            self.rpsCoordE, self.rpsCoordN, self.rpsCoordI = getRecGeometry(self.rpsImport, connect=False)
            self.updateMenuStatus(False)                                        # keep menu status in sync with program's state; don't reset analysis figure
            self.plotLayout()
        self.appendLogMessage(f'Filter : Filtered {before:,} records. Removed {(before - after):,} rps-duplicates')

    def removeSpsDuplicates(self):
        if self.spsImport is None:
            return

        self.spsImport, before, after = deletePntDuplicates(self.spsImport)
        self.spsModel.setData(self.spsImport)
        if after < before:
            self.spsCoordE, self.spsCoordN, self.spsCoordI = getSrcGeometry(self.spsImport, connect=False)
            self.updateMenuStatus(False)                                        # keep menu status in sync with program's state; don't reset analysis figure
            self.plotLayout()
        self.appendLogMessage(f'Filter : Filtered {before:,} records. Removed {(before - after):,} sps-duplicates')

    def removeRpsOrphans(self):
        if self.rpsImport is None:
            return

        self.rpsImport, before, after = deletePntOrphans(self.rpsImport)
        self.rpsModel.setData(self.rpsImport)
        if after < before:
            self.rpsCoordE, self.rpsCoordN, self.rpsCoordI = getRecGeometry(self.rpsImport, connect=False)
            self.updateMenuStatus(False)                                        # keep menu status in sync with program's state; don't reset analysis figure
            self.plotLayout()
        self.appendLogMessage(f'Filter : Filtered {before:,} records. Removed {(before - after):,} rps/xps-orphans')

    def removeSpsOrphans(self):
        if self.spsImport is None:
            return

        self.spsImport, before, after = deletePntOrphans(self.spsImport)
        self.spsModel.setData(self.spsImport)
        if after < before:
            self.spsCoordE, self.spsCoordN, self.spsCoordI = getSrcGeometry(self.spsImport, connect=False)
            self.updateMenuStatus(False)                                        # keep menu status in sync with program's state; don't reset analysis figure
            self.plotLayout()
        self.appendLogMessage(f'Filter : Filtered {before:,} records. Removed {(before - after):,} sps/xps-orphans')

    def removeXpsDuplicates(self):
        if self.xpsImport is None:
            return

        self.xpsImport, before, after = deleteRelDuplicates(self.xpsImport)
        self.xpsModel.setData(self.xpsImport)
        self.appendLogMessage(f'Filter : Filtered {before:,} records. Removed {(before - after):,} xps-duplicates')

    def removeXpsSpsOrphans(self):
        if self.xpsImport is None:
            return

        self.xpsImport, before, after = deleteRelOrphans(self.xpsImport, True)
        self.xpsModel.setData(self.xpsImport)
        self.appendLogMessage(f'Filter : Filtered {before:,} records. Removed {(before - after):,} xps/sps-orphans')

    def removeXpsRpsOrphans(self):
        if self.xpsImport is None:
            return

        self.xpsImport, before, after = deleteRelOrphans(self.xpsImport, False)
        self.xpsModel.setData(self.xpsImport)
        self.appendLogMessage(f'Filter : Filtered {before:,} records. Removed {(before - after):,} xps/rps-orphans')

    # define src, rec, rel button functions
    def sortRelData(self, index):
        if self.relGeom is None:
            return

        if index < 3:
            self.relGeom.sort(order=['SrcInd', 'SrcLin', 'SrcPnt', 'RecInd', 'RecLin', 'RecMin', 'RecMax'])
        elif index == 3:
            self.relGeom.sort(order=['RecNum', 'SrcInd', 'SrcLin', 'SrcPnt', 'RecInd', 'RecLin', 'RecMin', 'RecMax'])
        else:
            self.relGeom.sort(order=['RecInd', 'RecLin', 'RecMin', 'RecMax'])
        self.relModel.setData(self.relGeom)

    def removeRecDuplicates(self):
        if self.recGeom is None:
            return

        self.recGeom, before, after = deletePntDuplicates(self.recGeom)
        self.recModel.setData(self.recGeom)                                     # update the model's data
        if after < before:                                                      # need to update the (x, y) points as well
            self.recCoordE, self.recCoordN, self.recCoordI = getRecGeometry(self.recGeom, connect=False)
            self.updateMenuStatus(False)                                        # keep menu status in sync with program's state; don't reset analysis figure
            self.plotLayout()
        self.appendLogMessage(f'Filter : Filtered {before:,} records. Removed {(before - after):,} rec-duplicates')

    def removeSrcDuplicates(self):
        if self.srcGeom is None:
            return

        self.srcGeom, before, after = deletePntDuplicates(self.srcGeom)
        self.srcModel.setData(self.srcGeom)
        if after < before:
            self.srcCoordE, self.srcCoordN, self.srcCoordI = getSrcGeometry(self.srcGeom, connect=False)
            self.updateMenuStatus(False)                                        # keep menu status in sync with program's state; don't reset analysis figure
            self.plotLayout()
        self.appendLogMessage(f'Filter : Filtered {before:,} records. Removed {(before - after):,} src-duplicates')

    def removeRecOrphans(self):
        if self.recGeom is None:
            return

        self.recGeom, before, after = deletePntOrphans(self.recGeom)
        self.recModel.setData(self.recGeom)
        if after < before:
            self.recCoordE, self.recCoordN, self.recCoordI = getRecGeometry(self.recGeom, connect=False)
            self.updateMenuStatus(False)                                        # keep menu status in sync with program's state; don't reset analysis figure
            self.plotLayout()
        self.appendLogMessage(f'Filter : Filtered {before:,} records. Removed {(before - after):,} rec/rel-orphans')

    def removeSrcOrphans(self):
        if self.srcGeom is None:
            return

        self.srcGeom, before, after = deletePntOrphans(self.srcGeom)
        self.srcModel.setData(self.srcGeom)
        if after < before:
            self.srcCoordE, self.srcCoordN, self.srcCoordI = getSrcGeometry(self.srcGeom, connect=False)
            self.updateMenuStatus(False)                                        # keep menu status in sync with program's state; don't reset analysis figure
            self.plotLayout()
        self.appendLogMessage(f'Filter : Filtered {before:,} records. Removed {(before - after):,} src/rel-orphans')

    def removeRelDuplicates(self):
        if self.relGeom is None:
            return

        self.relGeom, before, after = deleteRelDuplicates(self.relGeom)
        self.relModel.setData(self.relGeom)
        self.appendLogMessage(f'Filter : Filtered {before:,} records. Removed {(before - after):,} rel-duplicates')

    def removeRelSrcOrphans(self):
        if self.relGeom is None:
            return

        self.relGeom, before, after = deleteRelOrphans(self.relGeom, True)
        self.relModel.setData(self.relGeom)
        self.appendLogMessage(f'Filter : Filtered {before:,} records. Removed {(before - after):,} rel/src-orphans')

    def removeRelRecOrphans(self):
        if self.relGeom is None:
            return

        self.relGeom, before, after = deleteRelOrphans(self.relGeom, False)
        self.relModel.setData(self.relGeom)
        self.appendLogMessage(f'Filter : Filtered {before:,} records. Removed {(before - after):,} rel/rec-orphans')

    # define file export functions
    def fileExportFoldMap(self):
        if self.survey is not None and self.survey.output.binOutput is not None and self.survey.crs is not None:
            fileName = self.fileName + '.bin.tif'
            fileName = CreateQgisRasterLayer(fileName, self.survey.output.binOutput, self.survey)
            if fileName:
                self.appendLogMessage(f'Export : exported fold map to {fileName}')

    def fileExportMinOffsets(self):
        if self.survey is not None and self.survey.output.minOffset is not None and self.survey.crs is not None:
            fileName = self.fileName + '.min.tif'
            fileName = CreateQgisRasterLayer(fileName, self.survey.output.minOffset, self.survey)
            if fileName:
                self.appendLogMessage(f'Export : exported min-offsets to {fileName}')

    def fileExportMaxOffsets(self):
        if self.survey is not None and self.survey.output.maxOffset is not None and self.survey.crs is not None:
            fileName = self.fileName + '.max.tif'
            fileName = CreateQgisRasterLayer(fileName, self.survey.output.maxOffset, self.survey)
            if fileName:
                self.appendLogMessage(f'Export : exported max-offsets to {fileName}')

    def fileExportRmsOffsets(self):
        if self.survey is not None and self.survey.output.rmsOffset is not None and self.survey.crs is not None:
            fileName = self.fileName + '.rms.tif'
            fileName = CreateQgisRasterLayer(fileName, self.survey.output.rmsOffset, self.survey)
            if fileName:
                self.appendLogMessage(f'Export : exported rms-offsets to {fileName}')

    def exportBinToQGIS(self):
        if self.survey is not None and self.survey.output.binOutput is not None and self.survey.crs is not None:
            fileName = self.fileName + '.bin.tif'
            fileName = ExportRasterLayerToQgis(fileName, self.survey.output.binOutput, self.survey)
            if fileName:
                self.appendLogMessage('Export : incorporated fold map in QGIS')

    def exportMinToQGIS(self):
        if self.survey is not None and self.survey.output.minOffset is not None and self.survey.crs is not None:
            fileName = self.fileName + '.min.tif'
            fileName = ExportRasterLayerToQgis(fileName, self.survey.output.minOffset, self.survey)
            if fileName:
                self.appendLogMessage('Export : incorporated min-offset map in QGIS')

    def exportMaxToQGIS(self):
        if self.survey is not None and self.survey.output.maxOffset is not None and self.survey.crs is not None:
            fileName = self.fileName + '.max.tif'
            fileName = ExportRasterLayerToQgis(fileName, self.survey.output.maxOffset, self.survey)
            if fileName:
                self.appendLogMessage('Export : incorporated max-offset map in QGIS')

    def exportRmsToQGIS(self):
        if self.survey is not None and self.survey.output.rmsOffset is not None and self.survey.crs is not None:
            fileName = self.fileName + '.rms.tif'
            fileName = ExportRasterLayerToQgis(fileName, self.survey.output.rmsOffset, self.survey)
            if fileName:
                self.appendLogMessage('Export : incorporated max-offset map in QGIS')

    def exportRpsToQgis(self):
        if self.rpsImport is not None and self.survey is not None and self.survey.crs is not None:
            layerName = QFileInfo(self.fileName).baseName() + '-rps-data'
            self.rpsLayerId = exportPointLayerToQgis(layerName, self.rpsImport, self.survey.crs, source=False)

    def exportSpsToQgis(self):
        if self.spsImport is not None and self.survey is not None and self.survey.crs is not None:
            layerName = QFileInfo(self.fileName).baseName() + '-sps-data'
            self.spsLayerId = exportPointLayerToQgis(layerName, self.spsImport, self.survey.crs, source=True)

    def exportRecToQgis(self):
        if self.recGeom is not None and self.survey is not None and self.survey.crs is not None:
            layerName = QFileInfo(self.fileName).baseName() + '-rec-data'
            self.recLayerId = exportPointLayerToQgis(layerName, self.recGeom, self.survey.crs, source=False)

    def exportSrcToQgis(self):
        if self.srcGeom is not None and self.survey is not None and self.survey.crs is not None:
            layerName = QFileInfo(self.fileName).baseName() + '-src-data'
            self.srcLayerId = exportPointLayerToQgis(layerName, self.srcGeom, self.survey.crs, source=True)

    def updateRecToQGIS(self):
        if self.recLayerId is None:
            self.recLayerId = identifyQgisPointLayer(self.iface, 'Rec')
            if self.recLayerId is None:
                return False

        if self.recGeom is not None and self.survey is not None and self.survey.crs is not None:
            success = updateQgisPointLayer(self.recLayerId, self.recGeom, self.survey.crs, source=False)
            return success

    def updateSrcToQGIS(self):
        if self.srcLayerId is None:
            self.srcLayerId = identifyQgisPointLayer(self.iface, 'Src')
            if self.srcLayerId is None:
                return False

        if self.srcGeom is not None and self.survey is not None and self.survey.crs is not None:
            success = updateQgisPointLayer(self.srcLayerId, self.srcGeom, self.survey.crs, source=True)
            return success

    def importSrcFromQgis(self):

        if self.srcLayerId is None:
            self.srcLayerId = identifyQgisPointLayer(self.iface, 'Src')
            if self.srcLayerId is None:
                return

        self.srcGeom = readQgisPointLayer(self.srcLayerId)

        nSrcOrphans, nRelOrphans = findSrcOrphans(self.srcGeom, self.relGeom)
        self.appendLogMessage(f'Import : . . . src-records contain {nRelOrphans:,} xps-orphans')
        self.appendLogMessage(f'Import : . . . rel-records contain {nSrcOrphans:,} sps-orphans')

        self.srcCoordE, self.srcCoordN, self.srcCoordI = self.srcGeom
        self.srcModel.setData(self.srcGeom)
        self.updateMenuStatus(False)                                            # keep menu status in sync with program's state; don't reset analysis figure
        self.plotLayout()

    def importRecFromQgis(self):

        if self.recLayerId is None:
            self.recLayerId = identifyQgisPointLayer(self.iface, 'Rec')
            if self.recLayerId is None:
                return

        self.recGeom = readQgisPointLayer(self.recLayerId)

        nRecOrphans, nRelOrphans = findRecOrphans(self.recGeom, self.relGeom)
        self.appendLogMessage(f'Import : . . . rps-records contain {nRelOrphans:,} rel-orphans')
        self.appendLogMessage(f'Import : . . . xps-records contain {nRecOrphans:,} rec-orphans')

        self.recCoordE, self.recCoordN, self.recCoordI = getRecGeometry(self.recGeom, connect=False)
        self.recModel.setData(self.recGeom)
        self.updateMenuStatus(False)                                            # keep menu status in sync with program's state; don't reset analysis figure
        self.plotLayout()

    def exportOutToQgis(self):
        layerName = QFileInfo(self.fileName).baseName()
        exportSurveyOutlineToQgis(layerName, self.survey)

    def updateMenuStatus(self, resetAnalysis=True):
        if resetAnalysis:
            self.actionNone.setChecked(True)                                    # coupled with tbNone
            self.imageType = 0                                                  # reset analysis type to zero

        self.actionExportFoldMap.setEnabled(self.survey.output.binOutput is not None)
        self.actionExportMinOffsets.setEnabled(self.survey.output.minOffset is not None)
        self.actionExportMaxOffsets.setEnabled(self.survey.output.maxOffset is not None)
        self.actionExportRmsOffsets.setEnabled(self.survey.output.rmsOffset is not None)
        self.actionExportAnaAsCsv.setEnabled(self.survey.output.anaOutput is not None)

        self.actionExportRecAsCsv.setEnabled(self.recGeom is not None)
        self.actionExportSrcAsCsv.setEnabled(self.srcGeom is not None)
        self.actionExportRelAsCsv.setEnabled(self.relGeom is not None)
        self.actionExportRecAsR01.setEnabled(self.recGeom is not None)
        self.actionExportSrcAsS01.setEnabled(self.srcGeom is not None)
        self.actionExportRelAsX01.setEnabled(self.relGeom is not None)
        self.actionExportSrcToQGIS.setEnabled(self.srcGeom is not None)
        self.actionExportRecToQGIS.setEnabled(self.recGeom is not None)

        self.actionExportRpsAsCsv.setEnabled(self.rpsImport is not None)
        self.actionExportSpsAsCsv.setEnabled(self.spsImport is not None)
        self.actionExportXpsAsCsv.setEnabled(self.xpsImport is not None)
        self.actionExportRpsAsR01.setEnabled(self.rpsImport is not None)
        self.actionExportSpsAsS01.setEnabled(self.spsImport is not None)
        self.actionExportXpsAsX01.setEnabled(self.xpsImport is not None)
        self.actionExportSpsToQGIS.setEnabled(self.spsImport is not None)
        self.actionExportRpsToQGIS.setEnabled(self.rpsImport is not None)

        self.btnSrcRemoveDuplicates.setEnabled(self.srcGeom is not None)
        self.btnSrcRemoveOrphans.setEnabled(self.srcGeom is not None)
        self.btnSrcExportToQGIS.setEnabled(self.srcGeom is not None)
        self.btnSrcUpdateToQGIS.setEnabled(self.srcGeom is not None)

        self.btnRecRemoveDuplicates.setEnabled(self.recGeom is not None)
        self.btnRecRemoveOrphans.setEnabled(self.recGeom is not None)
        self.btnRecExportToQGIS.setEnabled(self.recGeom is not None)
        self.btnRecUpdateToQGIS.setEnabled(self.recGeom is not None)

        self.btnRelRemoveSrcOrphans.setEnabled(self.relGeom is not None)
        self.btnRelRemoveDuplicates.setEnabled(self.relGeom is not None)
        self.btnRelRemoveRecOrphans.setEnabled(self.relGeom is not None)

        self.actionExportAreasToQGIS.setEnabled(len(self.fileName) > 0)         # test if file name isn't empty
        self.btnRelExportToQGIS.setEnabled(len(self.fileName) > 0)              # test if file name isn't empty

        self.btnSpsExportToQGIS.setEnabled(self.spsImport is not None)
        self.btnRpsExportToQGIS.setEnabled(self.rpsImport is not None)

        self.actionFold.setEnabled(self.survey.output.binOutput is not None)
        self.actionMinO.setEnabled(self.survey.output.minOffset is not None)
        self.actionMaxO.setEnabled(self.survey.output.maxOffset is not None)
        self.actionRmsO.setEnabled(self.survey.output.rmsOffset is not None)

        self.actionSpider.setEnabled(self.survey.output.anaOutput is not None)  # the spider button in the display pane
        self.actionMoveLt.setEnabled(self.survey.output.anaOutput is not None)  # the navigation buttons in the Display pane AND on toolbar (moveBar)
        self.actionMoveRt.setEnabled(self.survey.output.anaOutput is not None)
        self.actionMoveUp.setEnabled(self.survey.output.anaOutput is not None)
        self.actionMoveDn.setEnabled(self.survey.output.anaOutput is not None)

        self.btnBinToQGIS.setEnabled(self.survey.output.binOutput is not None)
        self.btnMinToQGIS.setEnabled(self.survey.output.minOffset is not None)
        self.btnMaxToQGIS.setEnabled(self.survey.output.maxOffset is not None)
        self.btnRmsToQGIS.setEnabled(self.survey.output.rmsOffset is not None)

        self.actionExportFoldMapToQGIS.setEnabled(self.survey.output.binOutput is not None)
        self.actionExportMinOffsetsToQGIS.setEnabled(self.survey.output.minOffset is not None)
        self.actionExportMaxOffsetsToQGIS.setEnabled(self.survey.output.maxOffset is not None)
        self.actionExportRmsOffsetsToQGIS.setEnabled(self.survey.output.rmsOffset is not None)

        self.actionRecPoints.setEnabled(self.recGeom is not None)
        self.actionSrcPoints.setEnabled(self.srcGeom is not None)
        self.actionRpsPoints.setEnabled(self.rpsImport is not None)
        self.actionSpsPoints.setEnabled(self.spsImport is not None)

    def setColorbarLabel(self, label):                                          # I should really subclass colorbarItem to properly set the text label
        if label is not None:
            if self.layoutColorBar.horizontal:
                self.layoutColorBar.getAxis('bottom').setLabel(label)
            else:
                self.layoutColorBar.getAxis('left').setLabel(label)

    def onActionNoneTriggered(self):
        self.imageType = 0
        self.handleImageSelection()

    def onActionFoldTriggered(self):
        self.imageType = 1
        self.handleImageSelection()

    def onActionMinOTriggered(self):
        self.imageType = 2
        self.handleImageSelection()

    def onActionMaxOTriggered(self):
        self.imageType = 3
        self.handleImageSelection()

    def onActionRmsOTriggered(self):
        self.imageType = 4
        self.handleImageSelection()

    def handleImageSelection(self):                                             # change image (if available) and finally plot survey layout

        if self.layoutImItem is not None and self.layoutColorBar is None:
            self.layoutColorBar = self.layoutWidget.plotItem.addColorBar(self.layoutImItem, colorMap=config.inActiveCmap, label='N/A', limits=(0, None), rounding=10.0, values=(0, 10))

        if self.layoutImg is None or self.imageType == 0:
            self.layoutImItem = None
            label = 'N/A'
            self.layoutMax = 10
            colorMap = config.inActiveCmap
        else:
            if self.imageType == 1:
                self.layoutImg = self.survey.output.binOutput                   # don't make a copy, just a view
                self.layoutMax = self.maximumFold
                label = 'fold'
            elif self.imageType == 2:
                self.layoutImg = self.survey.output.minOffset
                self.layoutMax = self.maxMinOffset
                label = 'minimum offset'
            elif self.imageType == 3:
                self.layoutImg = self.survey.output.maxOffset
                self.layoutMax = self.maxMaxOffset
                label = 'maximum offset'
            elif self.imageType == 4:
                self.layoutImg = self.survey.output.rmsOffset
                self.layoutMax = self.maxRmsOffset
                label = 'rms offset increments'
            else:
                raise NotImplementedError('selected analysis type currently not implemented.')

            colorMap = config.fold_OffCmap
            self.layoutImItem = pg.ImageItem()                                     # create PyqtGraph image item
            self.layoutImItem.setImage(self.layoutImg, levels=(0.0, self.layoutMax))
            self.layoutColorBar.setImageItem(self.layoutImItem)

        if self.layoutColorBar is not None:
            self.layoutColorBar.setLevels(low=0.0, high=self.layoutMax)
            self.layoutColorBar.setColorMap(colorMap)
            self.setColorbarLabel(label)

        self.plotLayout()

    def mouseBeingDragged(self):                                                # essential for LOD plotting whilst moving the survey object around
        self.survey.mouseGrabbed = True

    def exceptionHook(self, eType, eValue, eTraceback):
        """Function handling uncaught exceptions. It is triggered each time an uncaught exception occurs."""
        if issubclass(eType, KeyboardInterrupt):
            # ignore keyboard interrupt to support ctrl+C on console applications
            sys.__excepthook__(eType, eValue, eTraceback)
        else:

            # use the next string for a messagebox if "debug is on"
            # See: https://waylonwalker.com/python-sys-excepthook/
            # traceback_details = "\n".join(traceback.extract_tb(eTraceback).format())

            traceback_str = ''
            # for file_name, line_number, func_name, text in traceback.extract_tb(eTraceback, limit=2)[1:]:
            for file_name, line_number, func_name, _ in traceback.extract_tb(eTraceback, limit=2):
                file_name = os.path.basename(file_name)
                traceback_str += f' File "{file_name}", line {line_number}, in function "{func_name}".'

            if traceback_str != '':
                traceback_str = f'Traceback: {traceback_str}'

            exceptionMsg = f'Error&nbsp;&nbsp;:&nbsp;{eType.__name__}: {eValue} {traceback_str}'
            self.appendLogMessage(exceptionMsg, MsgType.Exception)

    def cursorPositionChanged(self):
        line = self.textEdit.textCursor().blockNumber() + 1
        col = self.textEdit.textCursor().columnNumber() + 1
        self.posWidgetStatusbar.setText(f'Line: {line} Col: {col}')

    def MouseMovedInPlot(self, pos):                                            # See: https://stackoverflow.com/questions/46166205/display-coordinates-in-pyqtgraph
        if self.layoutWidget.sceneBoundingRect().contains(pos):                 # is mouse moved within the scene area ?
            mousePoint = self.layoutWidget.plotItem.vb.mapSceneToView(pos)      # get scene coordinates

            if self.glob:                                                       # plot is using global coordinates
                toLocTransform, _ = self.survey.glbTransform.inverted()
                globalPoint = mousePoint
                localPoint = toLocTransform.map(globalPoint)
            else:                                                               # plot is using local coordinates
                localPoint = mousePoint
                globalPoint = self.survey.glbTransform.map(localPoint)

            lx = localPoint.x()
            ly = localPoint.y()
            gx = globalPoint.x()
            gy = globalPoint.y()

            if self.survey.binning.method == BinningType.cmp:                   # calculate reflector depth at cursor
                gz = 0.0
                lz = 0.0
            elif self.survey.binning.method == BinningType.plane:
                gz = self.survey.globalPlane.depthAt(globalPoint)               # get global depth from defined plane
                lz = self.survey.localPlane.depthAt(localPoint)                 # get local depth from transformed plane
            elif self.survey.binning.method == BinningType.sphere:
                gz = self.survey.globalSphere.depthAt(globalPoint)              # get global depth from defined sphere
                lz = self.survey.localSphere.depthAt(localPoint)                # get local depth from transformed sphere
            else:
                raise ValueError('wrong binning method selected')

            if self.survey.binTransform is not None:
                binPoint = self.survey.binTransform.map(localPoint)
                bx = int(binPoint.x())
                by = int(binPoint.y())
            else:
                bx = 0
                by = 0

            if self.survey.st2Transform is not None:
                stkPoint = self.survey.st2Transform.map(localPoint)
                sx = int(stkPoint.x())
                sy = int(stkPoint.y())
            else:
                sx = 0
                sy = 0

            if self.layoutImg is not None and bx >= 0 and by >= 0 and bx < self.layoutImg.shape[0] and by < self.layoutImg.shape[1]:
                # provide statusbar information within the analysis area
                if self.imageType == 0:
                    self.posWidgetStatusbar.setText(f'S:({sx:,d}, {sy:,d}), L:({lx:,.2f}, {ly:,.2f}, {lz:,.2f}), W:({gx:,.2f}, {gy:,.2f}, {gz:,.2f}) ')
                elif self.imageType == 1:
                    fold = self.layoutImg[bx, by]
                    self.posWidgetStatusbar.setText(f'fold: {fold:,d}, S:({sx:,d}, {sy:,d}), L:({lx:,.2f}, {ly:,.2f}, {lz:,.2f}), W:({gx:,.2f}, {gy:,.2f}, {gz:,.2f}) ')
                elif self.imageType == 2:
                    offset = float(self.layoutImg[bx, by])
                    self.posWidgetStatusbar.setText(f'|min offset|: {offset:.2f}, S:({sx:,d}, {sy:,d}), L:({lx:,.2f}, {ly:,.2f}, {lz:,.2f}), W:({gx:,.2f}, {gy:,.2f}, {gz:,.2f}) ')
                elif self.imageType == 3:
                    offset = float(self.layoutImg[bx, by])
                    self.posWidgetStatusbar.setText(f'|max offset|: {offset:.2f}, S:({sx:,d}, {sy:,d}), L:({lx:,.2f}, {ly:,.2f}, {lz:,.2f}), W:({gx:,.2f}, {gy:,.2f}, {gz:,.2f}) ')
                elif self.imageType == 4:
                    offset = float(self.layoutImg[bx, by])
                    self.posWidgetStatusbar.setText(f'rms offset inc: {offset:.2f}, S:({sx:,d}, {sy:,d}), L:({lx:,.2f}, {ly:,.2f}, {lz:,.2f}), W:({gx:,.2f}, {gy:,.2f}, {gz:,.2f}) ')
            else:
                # provide statusbar information outside the analysis area
                self.posWidgetStatusbar.setText(f'S:({sx:,d}, {sy:,d}), L:({lx:,.2f}, {ly:,.2f}, {lz:,.2f}), W:({gx:,.2f}, {gy:,.2f}, {gz:,.2f}) ')

    def getVisiblePlotIndex(self, plotWidget):
        if plotWidget == self.layoutWidget:
            return 0
        elif plotWidget == self.offTrkWidget:
            return 1
        elif plotWidget == self.offBinWidget:
            return 2
        elif plotWidget == self.aziTrkWidget:
            return 3
        elif plotWidget == self.aziBinWidget:
            return 4
        elif plotWidget == self.stkTrkWidget:
            return 5
        elif plotWidget == self.stkBinWidget:
            return 6
        elif plotWidget == self.stkCelWidget:
            return 7
        elif plotWidget == self.offsetWidget:
            return 8
        elif plotWidget == self.offAziWidget:
            return 9

        return None

    def getVisiblePlotWidget(self):
        if self.layoutWidget.isVisible():
            return (self.layoutWidget, 0)
        if self.offTrkWidget.isVisible():
            return (self.offTrkWidget, 1)
        if self.offBinWidget.isVisible():
            return (self.offBinWidget, 2)
        if self.aziTrkWidget.isVisible():
            return (self.aziTrkWidget, 3)
        if self.aziBinWidget.isVisible():
            return (self.aziBinWidget, 4)
        if self.stkTrkWidget.isVisible():
            return (self.stkTrkWidget, 5)
        if self.stkBinWidget.isVisible():
            return (self.stkBinWidget, 6)
        if self.stkCelWidget.isVisible():
            return (self.stkCelWidget, 7)
        if self.offsetWidget.isVisible():
            return (self.offsetWidget, 8)
        if self.offAziWidget.isVisible():
            return (self.offAziWidget, 9)

        return (None, None)

    def updateVisiblePlotWidget(self, index: int):
        if index == 0:
            self.plotLayout()                                                   # no conditions to plot main layout plot
            return

        if self.survey.output.anaOutput is None:                                # we need self.survey.output.anaOutput to display meaningful ANALYSIS information
            return

        xSize = self.survey.output.anaOutput.shape[0]                           # make sure we have a valid self.spiderPoint, hence valid nx, ny
        ySize = self.survey.output.anaOutput.shape[1]

        if self.spiderPoint == QPoint(-1, -1):                                  # no valid position yet; move to center
            self.spiderPoint = QPoint(xSize // 2, ySize // 2)

        nX = self.spiderPoint.x()                                               # create nx, ny from self.spiderPoint
        nY = self.spiderPoint.y()

        invBinTransform, _ = self.survey.binTransform.inverted()                # need to go from bin nr's to cmp(x, y)
        cmpX, cmpY = invBinTransform.map(nX, nY)                                # get local coordinates from line and point indices
        stkX, stkY = self.survey.st2Transform.map(cmpX, cmpY)                   # get the corresponding bin and stake numbers

        x0 = self.survey.output.rctOutput.left()                                # x origin of binning area
        y0 = self.survey.output.rctOutput.top()                                 # y origin of binning area

        dx = self.survey.grid.binSize.x()                                       # x bin size
        dy = self.survey.grid.binSize.y()                                       # y bin size
        ox = 0.5 * dx                                                           # half the x bin size
        oy = 0.5 * dy                                                           # half the y bin size

        if index == 1:
            self.plotOffTrk(nY, stkY, ox)
        elif index == 2:
            self.plotOffBin(nX, stkX, oy)
        elif index == 3:
            self.plotAziTrk(nY, stkY, ox)
        elif index == 4:
            self.plotAziBin(nX, stkX, oy)
        elif index == 5:
            self.plotStkTrk(nY, stkY, x0, dx)
        elif index == 6:
            self.plotStkBin(nX, stkX, y0, dy)
        elif index == 7:
            self.plotStkCel(nX, nY, stkX, stkY)
        elif index == 8:
            self.plotOffset()
        elif index == 9:
            self.plotOffAzi()

    def plotZoomRect(self):
        visiblePlot = self.getVisiblePlotWidget()[0]
        if visiblePlot is not None:
            viewBox = visiblePlot.getViewBox()
            self.rect = viewBox.getState()['mouseMode'] == pg.ViewBox.RectMode   # get rect status
            if self.rect:
                viewBox.setMouseMode(pg.ViewBox.PanMode)
            else:
                viewBox.setMouseMode(pg.ViewBox.RectMode)

    def plotAspectRatio(self):
        visiblePlot = self.getVisiblePlotWidget()[0]
        if visiblePlot is not None:
            plotItem = visiblePlot.getPlotItem()
            self.XisY = not plotItem.saveState()['view']['aspectLocked']        # get XisY status
            visiblePlot.setAspectLocked(self.XisY)

    def plotAntiAlias(self):
        visiblePlot, index = self.getVisiblePlotWidget()
        if visiblePlot is not None:                                             # there's no internal AA state
            self.antiA[index] = not self.antiA[index]                           # maintain status externally
            visiblePlot.setAntialiasing(self.antiA[index])                      # enable/disable aa plotting

    def plotGridX(self):
        visiblePlot = self.getVisiblePlotWidget()[0]
        if visiblePlot is not None:
            plotItem = visiblePlot.getPlotItem()
            self.gridX = not plotItem.saveState()['xGridCheck']                 # update x-gridline status
            if self.gridX:
                visiblePlot.showGrid(x=True, alpha=0.75)                        # show the grey grid lines
            else:
                visiblePlot.showGrid(x=False)                                   # don't show the grey grid lines

    def plotGridY(self):
        visiblePlot = self.getVisiblePlotWidget()[0]
        if visiblePlot is not None:
            plotItem = visiblePlot.getPlotItem()
            self.gridY = not plotItem.saveState()['yGridCheck']                 # update y-gridline status
            if self.gridY:
                visiblePlot.showGrid(y=True, alpha=0.75)                        # show the grey grid lines
            else:
                visiblePlot.showGrid(y=False)                                   # don't show the grey grid lines

    def plotProjected(self):
        self.glob = self.actionProjected.isChecked()

        if self.ruler:
            self.actionRuler.setChecked(False)
            self.plotRuler()

        self.rulerState = None

        if self.debug:                                                          # provide some debugging output on the applied transform
            # Get the transform that maps from local coordinates to the item's ViewBox coordinates
            transform = self.survey.glbTransform                                # GraphicsItem method
            if transform is not None:
                s1 = f'm11 ={transform.m11():12.6f},   m12 ={transform.m12():12.6f},   m13 ={transform.m13():12.6f} » [A1, B1, ...]'
                s2 = f'm21 ={transform.m21():12.6f},   m22 ={transform.m22():12.6f},   m23 ={transform.m23():12.6f} » [A2, B2, ...]'
                s3 = f'm31 ={transform.m31():12.6f},   m32 ={transform.m32():12.6f},   m33 ={transform.m33():12.6f} » [A0, B0, ...]<br>'

                self.appendLogMessage('plotProjected(). Showing transform parameters before changing view', MsgType.Debug)
                self.appendLogMessage(s1, MsgType.Debug)
                self.appendLogMessage(s2, MsgType.Debug)
                self.appendLogMessage(s3, MsgType.Debug)

                if not transform.isIdentity():
                    i_trans, _ = transform.inverted()                           # inverted_transform, invertable = transform.inverted()
                    s1 = f'm11 ={i_trans.m11():12.6f},   m12 ={i_trans.m12():12.6f},   m13 ={i_trans.m13():12.6f} » [A1, B1, ...]'
                    s2 = f'm21 ={i_trans.m21():12.6f},   m22 ={i_trans.m22():12.6f},   m23 ={i_trans.m23():12.6f} » [A2, B2, ...]'
                    s3 = f'm31 ={i_trans.m31():12.6f},   m32 ={i_trans.m32():12.6f},   m33 ={i_trans.m33():12.6f} » [A0, B0, ...]<br>'

                    self.appendLogMessage('plotProjected(). Showing inverted-transform parameters before changing view', MsgType.Debug)
                    self.appendLogMessage(s1, MsgType.Debug)
                    self.appendLogMessage(s2, MsgType.Debug)
                    self.appendLogMessage(s3, MsgType.Debug)

        self.handleSpiderPlot()                                                 # spider label should move depending on local/global coords
        self.layoutWidget.autoRange()                                           # show the full range of objects when changing local vs global coordinates
        self.plotLayout()

    def plotRuler(self):
        self.ruler = not self.ruler
        self.plotLayout()

    def UpdateAllViews(self):
        plainText = self.textEdit.getTextViaCursor()                            # read complete file content, not affecting doc status
        success = self.parseText(plainText)                                     # parse the string & check if it went okay...

        if success:
            plainText = self.survey.toXmlString()                               # convert the survey object itself to an xml string
            self.textEdit.setTextViaCursor(plainText)                           # get text into the textEdit, NOT resetting its doc status
            self.textEdit.document().setModified(True)                          # we edited the document; so it's been modified
            self.resetSurveyProperties()                                        # update property pane accordingly

        self.plotLayout()
        # self.layoutWidget.enableAutoRange()                                     # makes the plot 'fit' the survey outline.

    def layoutRangeChanged(self):
        """handle resizing of plot in view of bin-aligned gridlines"""
        axLft = self.layoutWidget.plotItem.getAxis('left')                      # get y-axis # 1
        axBot = self.layoutWidget.plotItem.getAxis('bottom')                    # get x-axis # 1
        axTop = self.layoutWidget.plotItem.getAxis('top')                       # get x-axis # 2
        axRht = self.layoutWidget.plotItem.getAxis('right')                     # get y-axis # 2

        vb = self.layoutWidget.getViewBox().viewRect()                          # view area in world coords
        dx = self.survey.grid.binSize.x()                                       # x bin size
        dy = self.survey.grid.binSize.y()                                       # y bin size

        if vb.width() > dx and vb.height() > dy:                                # area must be > a single bin to do something
            if not self.glob and (vb.width() < 30.0 * dx or vb.height() < 30.0 * dy):   # scale grid towards bin size

                # In this case it would be nice to have 3 decimals at the tick marks, not two
                # See: https://stackoverflow.com/questions/47500216/pyqtgraph-force-axis-labels-to-have-decimal-points

                xTicks = [dx, 0.2 * dx]                                         # no tickmarks smaller than bin size
                yTicks = [dy, 0.2 * dy]                                         # no tickmarks smaller than bin size
                axBot.setTickSpacing(xTicks[0], xTicks[1])                      # set x ticks (major and minor)
                axLft.setTickSpacing(yTicks[0], yTicks[1])                      # set x ticks (major and minor)
                axTop.setTickSpacing(xTicks[0], xTicks[1])                      # set x ticks (major and minor)
                axRht.setTickSpacing(yTicks[0], yTicks[1])                      # set x ticks (major and minor)
            else:
                axBot.setTickSpacing()                                          # set to default values
                axLft.setTickSpacing()                                          # set to default values
                axTop.setTickSpacing()                                          # set to default values
                axRht.setTickSpacing()                                          # set to default values

    def plotLayout(self):
        # first we are going to see how large the survey area is, to establish a boundingbox
        # See: https://www.geeksforgeeks.org/pyqtgraph-removing-item-from-plot-window/
        # self.layoutWidget.plotItem.removeItem(self.legend)
        # self.layoutWidget.plotItem.removeItem(self.srcLines)
        # self.layoutWidget.plotItem.removeItem(self.recLines)
        # See also: https://groups.google.com/g/pyqtgraph/c/tlryVLCDmmQ when the view does not refresh

        self.layoutWidget.plotItem.clear()
        self.layoutWidget.setTitle(self.survey.name, color='b', size='16pt')
        self.layoutWidget.showAxes(True, showValues=(True, False, False, True))   # show values at the left and at the bottom

        transform = QTransform()                                                # empty (unit) transform

        # setup axes first
        styles = {'color': '#000', 'font-size': '10pt'}
        if self.glob:                                                           # global -> easting & westing
            self.layoutWidget.setLabel('bottom', 'Easting', units='m', **styles)  # shows axis at the bottom, and shows the units label
            self.layoutWidget.setLabel('left', 'Northing', units='m', **styles)   # shows axis at the left, and shows the units label
            self.layoutWidget.setLabel('top', ' ', **styles)                    # shows axis at the top, no label, no tickmarks
            self.layoutWidget.setLabel('right', ' ', **styles)                  # shows axis at the right, no label, no tickmarks
            transform = self.survey.glbTransform                                # get global coordinate conversion transform
        else:                                                                   # local -> inline & crossline
            self.layoutWidget.setLabel('bottom', 'inline', units='m', **styles)   # shows axis at the bottom, and shows the units label
            self.layoutWidget.setLabel('left', 'crossline', units='m', **styles)  # shows axis at the left, and shows the units label
            self.layoutWidget.setLabel('top', ' ', **styles)                    # shows axis at the top, no label, no tickmarks
            self.layoutWidget.setLabel('right', ' ', **styles)                  # shows axis at the right, no label, no tickmarks

        # add image, if available and required
        if self.layoutImItem is not None and self.imageType > 0:
            self.layoutImItem.setTransform(self.survey.cmpTransform * transform)   # combine two transforms
            self.layoutWidget.plotItem.addItem(self.layoutImItem)

        # add survey geometry if templates are to be displayed (controlled by checkbox)
        if self.tbTemplat.isChecked():
            surveyItem = self.survey
            surveyItem.setTransform(transform)                                  # always do this; will reset transform for 'local' plot
            self.layoutWidget.plotItem.addItem(surveyItem)                      # this plots the survey geometry

        # to add SPS data, i.e. point lists, please have a look at:
        # https://pyqtgraph.readthedocs.io/en/latest/api_reference/graphicsItems/plotitem.html#pyqtgraph.PlotItem.plot and :
        # https://pyqtgraph.readthedocs.io/en/latest/api_reference/graphicsItems/plotdataitem.html#pyqtgraph.PlotDataItem.__init__
        # https://pyqtgraph.readthedocs.io/en/latest/api_reference/graphicsItems/scatterplotitem.html#pyqtgraph.ScatterPlotItem.setSymbol
        # https://pyqtgraph.readthedocs.io/en/latest/api_reference/graphicsItems/plotdataitem.html

        if self.tbSpsList.isChecked() and self.spsCoordE is not None and self.spsCoordN is not None:
            spsTransform = QTransform()                                         # empty (unit) transform
            if not self.glob and self.survey.glbTransform is not None:          # global -> easting & westing
                spsTransform, _ = self.survey.glbTransform.inverted()

            # sps = self.layoutWidget.plot( x=self.spsCoordE, y=self.spsCoordN, connect=self.spsCoordI, pxMode=False,
            #                             symbol=config.spsPointSymbol,
            #                             symbolPen=pg.mkPen("k"),
            #                             symbolSize=config.spsSymbolSize,
            #                             symbolBrush=QColor(config.spsBrushColor))

            sps = self.layoutWidget.plot(
                x=self.spsCoordE,
                y=self.spsCoordN,
                connect='all',
                pxMode=False,
                pen=None,
                symbol=config.spsPointSymbol,
                symbolPen=pg.mkPen('k'),
                symbolSize=config.spsSymbolSize,
                symbolBrush=QColor(config.spsBrushColor),
            )
            sps.setTransform(spsTransform)

        if self.tbRpsList.isChecked() and self.rpsCoordE is not None and self.rpsCoordN is not None:
            rpsTransform = QTransform()                                         # empty (unit) transform
            if not self.glob and self.survey.glbTransform is not None:          # global -> easting & westing
                rpsTransform, _ = self.survey.glbTransform.inverted()

            # rps = self.layoutWidget.plot( x=self.rpsCoordE, y=self.rpsCoordN, connect=self.rpsCoordI, pxMode=False,
            #                             symbol=config.rpsPointSymbol,
            #                             symbolPen=pg.mkPen("k"),
            #                             symbolSize=config.rpsSymbolSize,
            #                             symbolBrush=QColor(config.rpsBrushColor ))

            rps = self.layoutWidget.plot(
                x=self.rpsCoordE,
                y=self.rpsCoordN,
                connect='all',
                pxMode=False,
                pen=None,
                symbol=config.rpsPointSymbol,
                symbolPen=pg.mkPen('k'),
                symbolSize=config.rpsSymbolSize,
                symbolBrush=QColor(config.rpsBrushColor),
            )
            rps.setTransform(rpsTransform)

        if self.tbSrcList.isChecked() and self.srcCoordE is not None and self.srcCoordN is not None:
            srcTransform = QTransform()                                         # empty (unit) transform
            if not self.glob and self.survey.glbTransform is not None:          # global -> easting & westing
                srcTransform, _ = self.survey.glbTransform.inverted()

            src = self.layoutWidget.plot(
                x=self.srcCoordE,
                y=self.srcCoordN,
                connect=self.srcCoordI,
                pxMode=False,
                symbol=config.srcPointSymbol,
                symbolPen=pg.mkPen('k'),
                symbolSize=config.srcSymbolSize,
                symbolBrush=QColor(config.srcBrushColor),
            )
            src.setTransform(srcTransform)

        if self.tbRecList.isChecked() and self.recCoordE is not None and self.recCoordN is not None:
            recTransform = QTransform()                                         # empty (unit) transform
            if not self.glob and self.survey.glbTransform is not None:          # global -> easting & westing
                recTransform, _ = self.survey.glbTransform.inverted()

            # rec = self.layoutWidget.plot( x=self.recCoordE, y=self.recCoordN, connect=self.recCoordI, pxMode=False,
            #                             symbol=config.recPointSymbol,
            #                             symbolPen=pg.mkPen("k"),
            #                             symbolSize=config.recSymbolSize,
            #                             symbolBrush=QColor(config.recBrushColor ))

            rec = self.layoutWidget.plot(
                x=self.recCoordE,
                y=self.recCoordN,
                connect='all',
                pxMode=False,
                pen=None,
                symbol=config.recPointSymbol,
                symbolPen=pg.mkPen('k'),
                symbolSize=config.recSymbolSize,
                symbolBrush=QColor(config.recBrushColor),
            )
            rec.setTransform(recTransform)

        if self.tbSpider.isChecked() and self.survey.output.anaOutput is not None and self.survey.output.binOutput is not None and self.spiderSrcX is not None:

            src = self.layoutWidget.plot(
                x=self.spiderSrcX, y=self.spiderSrcY, connect='pairs', symbol='o', pen=pg.mkPen('r', width=2), symbolSize=5, pxMode=False, symbolPen=pg.mkPen('r'), symbolBrush=QColor('#77FF2929')
            )
            src.setTransform(transform)

            rec = self.layoutWidget.plot(
                x=self.spiderRecX, y=self.spiderRecY, connect='pairs', symbol='o', pen=pg.mkPen('b', width=2), symbolSize=5, pxMode=False, symbolPen=pg.mkPen('b'), symbolBrush=QColor('#772929FF')
            )
            rec.setTransform(transform)

            self.layoutWidget.plotItem.addItem(self.spiderText)

        if self.tbTemplat.isChecked():
            # Add a marker for the origin
            oriX = [0.0]
            oriY = [0.0]
            orig = self.layoutWidget.plot(x=oriX, y=oriY, symbol='o', symbolSize=16, symbolPen=(0, 0, 0, 100), symbolBrush=(180, 180, 180, 100))
            orig.setTransform(transform)

        if self.survey.binning.method == BinningType.sphere:
            # Draw sphere as a circle in the plot for guidance when binning against a sphere
            # See: https://stackoverflow.com/questions/33525279/pyqtgraph-how-do-i-plot-an-ellipse-or-a-circle
            # See: https://doc.qt.io/qtforpython-5/PySide2/QtWidgets/QGraphicsEllipseItem.html
            r = self.survey.localSphere.radius
            x = self.survey.localSphere.origin.x() - r
            y = self.survey.localSphere.origin.y() - r
            w = r * 2.0
            h = r * 2.0
            sphereArea = QGraphicsEllipseItem(x, y, w, h)
            sphereArea.setPen(pg.mkPen(100, 100, 100))
            sphereArea.setBrush(QBrush(QColor(config.binAreaColor)))            # use same color as binning region
            sphereArea.setTransform(transform)
            self.layoutWidget.plotItem.addItem(sphereArea)

        if self.ruler:
            # add ruler if required
            p1 = pg.mkPen('r', style=Qt.DashLine)
            p2 = pg.mkPen('r', style=Qt.DashLine, width=2)
            p3 = pg.mkPen('b')
            p4 = pg.mkPen('b', width=3)

            # get default location for ruler, dependent on current viewRect
            viewRect = self.layoutWidget.plotItem.vb.viewRect()
            ptCenter = viewRect.center()
            pt1 = (ptCenter + viewRect.topLeft()) / 2.0
            pt2 = (ptCenter + viewRect.bottomRight()) / 2.0

            self.lineROI = LineROI([[pt1.x(), pt1.y()], [pt2.x(), pt2.y()]], pen=p1, hoverPen=p2, handlePen=p3, handleHoverPen=p4)
            if self.rulerState is not None:                                     # restore state, if possible
                self.lineROI.setState(self.rulerState)

            self.layoutWidget.plotItem.addItem(self.lineROI)
            self.lineROI.sigRegionChanged.connect(self.roiChanged)

            length = len(self.lineROI.getHandles()) + 1
            self.roiLabels = [pg.TextItem(anchor=(0.5, 1.3), border='b', color='b', fill=(130, 255, 255, 200), text='label') for _ in range(length)]

            for label in self.roiLabels:
                self.layoutWidget.plotItem.addItem(label)
                label.setZValue(1000)
            self.roiChanged()

    def plotOffTrk(self, nY: int, stkY: int, ox: float):
        with pg.BusyCursor():
            slice3D = self.survey.output.anaOutput[:, nY, :, :]
            slice2D = slice3D.reshape(slice3D.shape[0] * slice3D.shape[1], slice3D.shape[2])           # convert to 2D

            slice2D = numbaFilterSlice2D(slice2D, self.survey.unique.apply)
            if slice2D.shape[0] == 0:                                           # empty array; nothing to see here...
                return

            x, y = numbaOffInline(slice2D, ox)

            plotTitle = f'{self.plotTitles[1]} [line={stkY}]'
            self.offTrkWidget.setTitle(plotTitle, color='b', size='16pt')
            self.offTrkWidget.plotItem.clear()
            self.offTrkWidget.plot(x=x, y=y, connect='pairs', pen=pg.mkPen('k', width=2))

    def plotOffBin(self, nX: int, stkX: int, oy: float):
        with pg.BusyCursor():
            slice3D = self.survey.output.anaOutput[nX, :, :, :]
            slice2D = slice3D.reshape(slice3D.shape[0] * slice3D.shape[1], slice3D.shape[2])           # convert to 2D

            slice2D = numbaFilterSlice2D(slice2D, self.survey.unique.apply)
            if slice2D.shape[0] == 0:                                           # empty array; nothing to see here...
                return

            x, y = numbaOffX_line(slice2D, oy)

            plotTitle = f'{self.plotTitles[2]} [stake={stkX}]'
            self.offBinWidget.setTitle(plotTitle, color='b', size='16pt')
            self.offBinWidget.plotItem.clear()
            self.offBinWidget.plot(x=x, y=y, connect='pairs', pen=pg.mkPen('k', width=2))

    def plotAziTrk(self, nY: int, stkY: int, ox: float):
        with pg.BusyCursor():
            slice3D = self.survey.output.anaOutput[:, nY, :, :]
            slice2D = slice3D.reshape(slice3D.shape[0] * slice3D.shape[1], slice3D.shape[2])           # convert to 2D

            slice2D = numbaFilterSlice2D(slice2D, self.survey.unique.apply)
            if slice2D.shape[0] == 0:                                           # empty array; nothing to see here...
                return

            x, y = numbaAziInline(slice2D, ox)

            plotTitle = f'{self.plotTitles[3]} [line={stkY}]'
            self.aziTrkWidget.setTitle(plotTitle, color='b', size='16pt')
            self.aziTrkWidget.plotItem.clear()
            self.aziTrkWidget.plot(x=x, y=y, connect='pairs', pen=pg.mkPen('k', width=2))

    def plotAziBin(self, nX: int, stkX: int, oy: float):
        with pg.BusyCursor():
            slice3D = self.survey.output.anaOutput[nX, :, :, :]
            slice2D = slice3D.reshape(slice3D.shape[0] * slice3D.shape[1], slice3D.shape[2])           # convert to 2D

            slice2D = numbaFilterSlice2D(slice2D, self.survey.unique.apply)
            if slice2D.shape[0] == 0:                                           # empty array; nothing to see here...
                return

            x, y = numbaAziX_line(slice2D, oy)

            plotTitle = f'{self.plotTitles[4]} [stake={stkX}]'
            self.aziBinWidget.setTitle(plotTitle, color='b', size='16pt')
            self.aziBinWidget.plotItem.clear()
            self.aziBinWidget.plot(x=x, y=y, connect='pairs', pen=pg.mkPen('k', width=2))

    def plotStkTrk(self, nY: int, stkY: int, x0: float, dx: float):
        with pg.BusyCursor():
            dK = 0.001 * config.kr_Range.z()
            kMax = 0.001 * config.kr_Range.y() + dK
            kStart = 1000.0 * (0.0 - 0.5 * dK)                                  # scale by factor 1000 as we want to show [1/km] on scale
            kDelta = 1000.0 * dK                                                # same here

            slice3D, I = numbaSlice3D(self.survey.output.anaOutput[:, nY, :, :], self.survey.unique.apply)
            if slice3D.shape[0] == 0:                                           # empty array; nothing to see here...
                return

            self.inlineStk = numbaNdft_1D(kMax, dK, slice3D, I)

            tr = QTransform()                                                   # prepare ImageItem transformation:
            tr.translate(x0, kStart)                                            # move image to correct location
            tr.scale(dx, kDelta)                                                # scale horizontal and vertical axes

            self.stkTrkImItem = pg.ImageItem()                                  # create PyqtGraph image item
            self.stkTrkImItem.setImage(self.inlineStk, levels=(-50.0, 0.0))     # plot with log scale from -50 to 0
            self.stkTrkImItem.setTransform(tr)

            if self.stkTrkColorBar is None:
                self.stkTrkColorBar = self.stkTrkWidget.plotItem.addColorBar(self.stkTrkImItem, colorMap=config.analysisCmap, label='dB attenuation', limits=(-100.0, 0.0), rounding=10.0, values=(-50.0, 0.0))
                self.stkTrkColorBar.setLevels(low=-50.0, high=0.0)
            else:
                self.stkTrkColorBar.setImageItem(self.stkTrkImItem)
                self.stkTrkColorBar.setColorMap(config.analysisCmap)            # in case the colorbar has been changed

            self.stkTrkWidget.plotItem.clear()
            self.stkTrkWidget.plotItem.addItem(self.stkTrkImItem)

            plotTitle = f'{self.plotTitles[5]} [line={stkY}]'
            self.stkTrkWidget.setTitle(plotTitle, color='b', size='16pt')

    def plotStkBin(self, nX: int, stkX: int, y0: float, dy: float):
        with pg.BusyCursor():
            dK = 0.001 * config.kr_Range.z()
            kMax = 0.001 * config.kr_Range.y() + dK
            kStart = 1000.0 * (0.0 - 0.5 * dK)                                  # scale by factor 1000 as we want to show [1/km] on scale
            kDelta = 1000.0 * dK                                                # same here

            slice3D, I = numbaSlice3D(self.survey.output.anaOutput[nX, :, :, :], self.survey.unique.apply)
            if slice3D.shape[0] == 0:                                           # empty array; nothing to see here...
                return

            self.x_lineStk = numbaNdft_1D(kMax, dK, slice3D, I)

            tr = QTransform()                                                   # prepare ImageItem transformation:
            tr.translate(y0, kStart)                                            # move image to correct location
            tr.scale(dy, kDelta)                                                # scale horizontal and vertical axes

            self.stkBinImItem = pg.ImageItem()                                  # create PyqtGraph image item
            self.stkBinImItem.setImage(self.x_lineStk, levels=(-50.0, 0.0))     # plot with log scale from -50 to 0
            self.stkBinImItem.setTransform(tr)

            if self.stkBinColorBar is None:
                self.stkBinColorBar = self.stkBinWidget.plotItem.addColorBar(self.stkBinImItem, colorMap=config.analysisCmap, label='dB attenuation', limits=(-100.0, 0.0), rounding=10.0, values=(-50.0, 0.0))
                self.stkBinColorBar.setLevels(low=-50.0, high=0.0)
            else:
                self.stkBinColorBar.setImageItem(self.stkBinImItem)
                self.stkBinColorBar.setColorMap(config.analysisCmap)            # in case the colorbar has been changed

            self.stkBinWidget.plotItem.clear()
            self.stkBinWidget.plotItem.addItem(self.stkBinImItem)

            plotTitle = f'{self.plotTitles[6]} [stake={stkX}]'
            self.stkBinWidget.setTitle(plotTitle, color='b', size='16pt')

    def plotStkCel(self, nX: int, nY: int, stkX: int, stkY: int):
        with pg.BusyCursor():
            kMin = 0.001 * config.kxyRange.x()
            kMax = 0.001 * config.kxyRange.y()
            dK = 0.001 * config.kxyRange.z()
            kMax = kMax + dK

            kStart = 1000.0 * (kMin - 0.5 * dK)                             # scale by factor 1000 as we want to show [1/km] on scale
            kDelta = 1000.0 * dK                                            # same here

            offsetX, offsetY, noData = numbaOffsetBin(self.survey.output.anaOutput[nX, nY, :, :], self.survey.unique.apply)
            if noData:
                return

            self.xyCellStk = numbaNdft_2D(kMin, kMax, dK, offsetX, offsetY)

            tr = QTransform()                                               # prepare ImageItem transformation:
            tr.translate(kStart, kStart)                                    # move image to correct location
            tr.scale(kDelta, kDelta)                                        # scale horizontal and vertical axes

            self.stkCelImItem = pg.ImageItem()                              # create PyqtGraph image item
            self.stkCelImItem.setImage(self.xyCellStk, levels=(-50.0, 0.0))   # plot with log scale from -50 to 0
            self.stkCelImItem.setTransform(tr)
            if self.stkCelColorBar is None:
                self.stkCelColorBar = self.stkCelWidget.plotItem.addColorBar(self.stkCelImItem, colorMap=config.analysisCmap, label='dB attenuation', limits=(-100.0, 0.0), rounding=10.0, values=(-50.0, 0.0))
                self.stkCelColorBar.setLevels(low=-50.0, high=0.0)
            else:
                self.stkCelColorBar.setImageItem(self.stkCelImItem)
                self.stkCelColorBar.setColorMap(config.analysisCmap)        # in case the colorbar has been changed

            self.stkCelWidget.plotItem.clear()
            self.stkCelWidget.plotItem.addItem(self.stkCelImItem)

            plotTitle = f'{self.plotTitles[7]} [stake={stkX}, line={stkY}, fold={offsetX.shape[0]}]'
            self.stkCelWidget.setTitle(plotTitle, color='b', size='16pt')

    def plotOffset(self):
        with pg.BusyCursor():
            offsets, _, noData = numbaSliceStats(self.survey.output.anaOutput, self.survey.unique.apply)
            if noData:
                return

            y, x = np.histogram(offsets, bins='auto')                       # create a histogram the easy way

            count = offsets.shape[0]                                        # nr available traces
            plotTitle = f'{self.plotTitles[8]} [{count:,} traces]'
            self.offsetWidget.setTitle(plotTitle, color='b', size='16pt')
            self.offsetWidget.plotItem.clear()
            self.offsetWidget.plot(x, y, stepMode='center', fillLevel=0, fillOutline=True, brush=(0, 0, 255, 150), pen=pg.mkPen('k', width=1))

    def plotOffAzi(self):
        with pg.BusyCursor():
            dA = 5.0                                                        # azimuth increments
            aMin = -180.0                                                   # max x-scale
            aMax = 180.0                                                    # max x-scale
            aMax += dA                                                      # make sure end value is included

            dO = 100.0                                                      # offsets increments
            oMax = ceil(self.maxMaxOffset / dO) * dO + dO                   # max y-scale; make sure end value is included

            aR = np.arange(aMin, aMax, dA)                                  # numpy array with values [0 ... fMax]
            oR = np.arange(0, oMax, dO)                                     # numpy array with values [0 ... oMax]

            offsets, azimuth, noData = numbaSliceStats(self.survey.output.anaOutput, self.survey.unique.apply)
            if noData:
                return

            self.offAziPlt = np.histogram2d(x=azimuth, y=offsets, bins=[aR, oR], range=None, density=None, weights=None)[0]

            tr = QTransform()                                               # prepare ImageItem transformation:
            tr.translate(aMin, 0)                                           # move image to correct location
            tr.scale(dA, dO)                                                # scale horizontal and vertical axes

            self.offAziImItem = pg.ImageItem()                              # create PyqtGraph image item
            self.offAziImItem.setImage(self.offAziPlt)                      #
            self.offAziImItem.setTransform(tr)
            if self.offAziColorBar is None:
                self.offAziColorBar = self.offAziWidget.plotItem.addColorBar(self.offAziImItem, colorMap=config.analysisCmap, label='frequency', rounding=10.0)
                self.offAziColorBar.setLevels(low=0.0)                      # , high=0.0
            else:
                self.offAziColorBar.setImageItem(self.offAziImItem)
                self.offAziColorBar.setColorMap(config.analysisCmap)        # in case the colorbar has been changed

            self.offAziWidget.plotItem.clear()
            self.offAziWidget.plotItem.addItem(self.offAziImItem)

            count = offsets.shape[0]                                        # nr available traces
            plotTitle = f'{self.plotTitles[9]} [{count:,} traces]'
            self.offAziWidget.setTitle(plotTitle, color='b', size='16pt')

        # For Polar Coordinates, see:
        # See: https://stackoverflow.com/questions/57174173/polar-coordinate-system-in-pyqtgraph
        # See: https://groups.google.com/g/pyqtgraph/c/9Vv1kJdxE6U/m/FuCsSg182jUJ
        # See: https://doc.qt.io/qtforpython-6/PySide6/QtCharts/QPolarChart.html
        # See: https://www.youtube.com/watch?v=DyPjsj6azY4
        # See: https://stackoverflow.com/questions/50720719/how-to-create-a-color-circle-in-pyqt
        # See: https://stackoverflow.com/questions/70471687/pyqt-creating-color-circle

    def roiChanged(self):
        pos = []
        for i, handle in enumerate(self.lineROI.getHandles()):
            handlePos = self.lineROI.pos() + handle.pos()
            self.roiLabels[i].setPos(handlePos)
            pos.append(handlePos)
            self.roiLabels[i].setText(f'({handlePos[0]:.2f}, {handlePos[1]:.2f})')

        # put label in the middle of the line
        pos2 = (pos[0] + pos[1]) / 2.0
        diff = pos[1] - pos[0]
        self.roiLabels[2].setPos(pos2)
        self.roiLabels[2].setText(f'|r|={diff.length():.2f}, Ø={degrees(atan2(diff.y(),diff.x())):.2f}°')
        self.rulerState = self.lineROI.saveState()

    def closeEvent(self, e):  # main window about to be closed event
        # See: https://doc.qt.io/qt-6/qwidget.html#closeEvent
        # See: https://stackoverflow.com/questions/22460003/pyqts-qmainwindow-closeevent-is-never-called

        if self.fileNew():                                                      # file (maybe) saved and cancel NOT used
            self.dockLogging.setFloating(False)                                 # don't keep floating docking widgets hanging araound once closed
            self.dockDisplay.setFloating(False)                                 # don't keep floating docking widgets hanging araound once closed
            self.dockProperty.setFloating(False)                                # don't keep floating docking widgets hanging araound once closed
            # self.writeSettings()                                              # save geometry and state of window(s)
            writeSettings(self)                                                 # save geometry and state of window(s)

            if self.workingDirectory:                                           # append information to log file in working directory
                logFile = os.path.join(self.workingDirectory, '.roll.log')      # join directory & log file name
                with open(logFile, 'a', encoding='utf-8') as file:              # append information to logfile
                    file.write(self.logEdit.toPlainText())                      # get text from logEdit
                    file.write('+++\n\n')                                       # closing remarks

            # builtins.print = self.oldPrint                                    # restore builtins.print
            # sys.excepthook = self.oldExceptHook                               # restore sys.excepthook back to the original
            e.accept()                                                          # finally accep the event

            self.killMe = True                                                  # to restart the GUI from scratch when the plugin is activated again

        else:
            e.ignore()                                                          # ignore the event and stay active

        # See: https://stackoverflow.com/questions/26114034/defining-a-wx-panel-destructor-in-wxpython/73972953#73972953
        # See: http://enki-editor.org/2014/08/23/Pyqt_mem_mgmt.html

        # The following code is already done in self.fileNew()
        # if self.thread is not None and self.thread.isRunning():
        #     reply = QMessageBox.question(self, 'Please confirm',
        #         "Cancel work in progress and close Roll ?",
        #         QMessageBox.Yes, QMessageBox.Cancel)

        #     if reply == QMessageBox.Yes:
        #         self.thread.requestInterruption()
        #         self.thread.quit()
        #         self.thread.wait()
        #         # force thread termination
        #         # self.thread.terminate()
        #         self.thread.deleteLater()
        #         e.accept()
        #     else:
        #         e.ignore()                                                      # ignore the event and stay active

    def newFile(self):                                                          # wrapper around fileNew; used to create a log message
        if self.fileNew():
            self.appendLogMessage('Created: new file')
            self.plotLayout()                                                   # update the survey, but not the xml-tab

    def resetNumpyArraysAndModels(self):
        """reset various analysis arrays"""

        # numpy binning arrays
        self.layoutImg = None                                                   # numpy array to be displayed; binOutput / minOffset / maxOffset

        # analysis numpy arrays
        self.inlineStk = None                                                   # numpy array with inline Kr stack reponse
        self.x_lineStk = None                                                   # numpy array with x_line Kr stack reponse
        self.xyCellStk = None                                                   # numpy array with cell's KxKy stack response
        self.xyPatResp = None                                                   # numpy array with pattern's KxKy response
        self.offAziPlt = None                                                   # numpy array with offset distribution

        # layout and analysis image-items
        self.layoutImItem = None                                                # pg ImageItems showing analysis result
        self.stkTrkImItem = None
        self.stkBinImItem = None
        self.stkCelImItem = None
        self.offAziImItem = None

        # corresponding color bars
        # self.layoutColorBar = None                                              # DON'T reset these; adjust the existing ones
        # self.stkTrkColorBar = None
        # self.stkBinColorBar = None
        # self.stkCelColorBar = None
        # self.offAziColorBar = None

        # rps, sps, xps input arrays
        self.rpsImport = None                                                   # numpy array with list of RPS records
        self.spsImport = None                                                   # numpy array with list of SPS records
        self.xpsImport = None                                                   # numpy array with list of XPS records

        self.rpsCoordE = None                                                   # numpy array with list of RPS coordinates
        self.rpsCoordN = None                                                   # numpy array with list of RPS coordinates
        self.rpsCoordI = None                                                   # numpy array with list of RPS connections

        self.spsCoordE = None                                                   # numpy array with list of SPS coordinates
        self.spsCoordN = None                                                   # numpy array with list of SPS coordinates
        self.spsCoordI = None                                                   # numpy array with list of SPS connections

        # rel, src, rel input arrays
        self.recGeom = None                                                     # numpy array with list of REC records
        self.srcGeom = None                                                     # numpy array with list of SRC records
        self.relGeom = None                                                     # numpy array with list of REL records

        self.recCoordE = None                                                   # numpy array with list of REC coordinates
        self.recCoordN = None                                                   # numpy array with list of REC coordinates
        self.recCoordI = None                                                   # numpy array with list of REC connections

        self.srcCoordE = None                                                   # numpy array with list of SRC coordinates
        self.srcCoordN = None                                                   # numpy array with list of SRC coordinates
        self.srcCoordI = None                                                   # numpy array with list of SRC connections

        # spider plot settings
        self.spiderPoint = QPoint(-1, -1)                                       # spider point 'out of scope'
        self.spiderSrcX = None                                                  # numpy array with list of SRC part of spider plot
        self.spiderSrcY = None                                                  # numpy array with list of SRC part of spider plot
        self.spiderRecX = None                                                  # numpy array with list of REC part of spider plot
        self.spiderRecY = None                                                  # numpy array with list of REC part of spider plot
        self.spiderText = None                                                  # text label describing spider bin, stake, fold
        self.actionSpider.setChecked(False)                                     # reset spider plot to 'off'

        # export layers to QGIS
        self.rpsLayerId = None                                                  # QGIS layerId to be checked when updating a QgisVectorLayer
        self.spsLayerId = None                                                  # QGIS layerId to be checked when updating a QgisVectorLayer
        self.recLayerId = None                                                  # QGIS layerId to be checked when updating a QgisVectorLayer
        self.srcLayerId = None                                                  # QGIS layerId to be checked when updating a QgisVectorLayer

        # ruler settings
        self.lineROI = None                                                     # the ruler's dotted line
        self.roiLabels = None                                                   # the ruler's three labels
        self.rulerState = None                                                  # ruler's state, used to redisplay ruler at last used location

        self.rpsModel.setData(self.rpsImport)                                   # update the three rps/sps/xps models
        self.spsModel.setData(self.spsImport)
        self.xpsModel.setData(self.xpsImport)

        self.recModel.setData(self.recGeom)                                     # update the three rec/rel/src models
        self.relModel.setData(self.relGeom)
        self.srcModel.setData(self.srcGeom)

        self.survey.output.binOutput = None
        self.survey.output.minOffset = None
        self.survey.output.maxOffset = None
        self.survey.output.rmsOffset = None

        if self.survey.output.anaOutput is not None:                            # remove memory mapped file, as well
            self.survey.output.anaOutput.flush()
            # self.survey.output.anaOutput._mmap.close()                        # See: https://stackoverflow.com/questions/6397495/unmap-of-numpy-memmap
            del self.survey.output.anaOutput
            self.D2_Output = None                                               # first remove reference to self.survey.output.anaOutput
            self.survey.output.anaOutput = None                                 # then remove self.survey.output.anaOutput itself
            gc.collect()                                                        # get the garbage collector going
        self.anaModel.setData(self.D2_Output)                                   # use this as the model data

        self.resetPlotWidget(self.offTrkWidget, self.plotTitles[1])             # clear all analysis plots
        self.resetPlotWidget(self.offBinWidget, self.plotTitles[2])
        self.resetPlotWidget(self.aziTrkWidget, self.plotTitles[3])
        self.resetPlotWidget(self.aziBinWidget, self.plotTitles[4])
        self.resetPlotWidget(self.stkTrkWidget, self.plotTitles[5])
        self.resetPlotWidget(self.stkBinWidget, self.plotTitles[6])
        self.resetPlotWidget(self.stkCelWidget, self.plotTitles[7])
        self.resetPlotWidget(self.offsetWidget, self.plotTitles[8])
        self.resetPlotWidget(self.offAziWidget, self.plotTitles[9])

        self.updateMenuStatus(True)                                             # keep menu status in sync with program's state; and reset analysis figure

    def fileNew(self):                                                          # better create new file created through a wizard
        if self.maybeKillThread() and self.maybeSave():                         # make sure thread is killed AND current file  is saved (all only when needed)
            self.resetNumpyArraysAndModels()                                    # empty all arrays and reset plot titles

            # start defining new survey
            self.parseText(exampleSurveyXmlText())                              # read & parse xml string and create new survey object
            self.textEdit.setPlainText(exampleSurveyXmlText())                  # copy xml content to text edit control
            self.resetSurveyProperties()                                        # get the new parameters into the parameter tree
            self.textEdit.moveCursor(QTextCursor.Start)                         # move cursor to front
            self.survey.calcTransforms()                                        # (re)calculate the transforms being used
            self.survey.calcSeedData()                                          # needed for circles, spirals & well-seeds; may affect bounding box
            self.survey.calcBoundingRect()                                      # (re)calculate the boundingBox as part of parsing the data
            self.survey.calcNoShotPoints()                                      # (re)calculate nr of SPs

            self.setCurrentFileName()                                           # update self.fileName, set textEditModified(False) and setWindowModified(False)

            self.actionProjected.setChecked(False)                              # set to 'local' plotting (not global)
            self.plotProjected()                                                # enforce 'local' plotting and plotLayout()

            return True                                                         # we emptied the document, and reset the survey object
        else:
            return False                                                        # user had 2nd thoughts and did not close the document

    def fileNewLandSurvey(self):
        if not self.fileNew():                                                  # user had 2nd thoughts and did not close the document; return False
            return False

        dlg = LandSurveyWizard(self)

        if dlg.exec():                                                          # Run the dialog event loop, and obtain survey object
            self.survey = dlg.survey                                            # get survey from dialog

            plainText = self.survey.toXmlString()                               # convert the survey object to an Xml string
            self.textEdit.highlighter = XMLHighlighter(self.textEdit.document())  # we want some color highlighteded text
            self.textEdit.setFont(QFont('Ubuntu Mono', 9, QFont.Normal))        # the font may have been messed up by the initial html text

            self.textEdit.setTextViaCursor(plainText)                           # get text into the textEdit, NOT resetting its doc status
            self.UpdateAllViews()                                               # parse the textEdit; show the corresponding plot
            self.resetSurveyProperties()                                        # get the new parameters into the parameter tree

            self.appendLogMessage(f'Wizard : created land survey: {self.survey.name}')
            config.surveyNumber += 1                                            # update global counter

            if self.debug:
                self.appendLogMessage('Land Survey Wizard profiling information', MsgType.Debug)
                i = 0
                while i < len(self.dlg.survey.timerTmin):                       # log some debug messages
                    tMin = self.dlg.survey.timerTmin[i] * 1000.0 if self.worker.survey.timerTmin[i] != float('Inf') else 0.0
                    tMax = self.dlg.survey.timerTmax[i] * 1000.0
                    tTot = self.dlg.survey.timerTtot[i] * 1000.0
                    freq = self.dlg.survey.timerFreq[i]
                    tAvr = tTot / freq if freq > 0 else 0.0
                    message = f'{i:02d}: min:{tMin:011.3f}, max:{tMax:011.3f}, tot:{tTot:011.3f}, avr:{tAvr:011.3f}, freq:{freq:07d}'
                    self.appendLogMessage(message, MsgType.Debug)
                    i += 1

    def fileNewMarineSurvey(self):
        QMessageBox.information(self, 'Not implemented', 'The marine wizard has not yet been implemented', QMessageBox.Cancel)
        return

    def maybeSave(self):
        if not self.textEdit.document().isModified():                           # no need to do anything, as the doc wasn't modified
            return True

        ret = QMessageBox.warning(self, 'Roll', 'The document has been modified.\nDo you want to save your changes?', QMessageBox.Save | QMessageBox.Discard | QMessageBox.Cancel)        # want to save changes ?

        if ret == QMessageBox.Save:                                             # yes please; try to save changes
            return self.fileSave()                                              # if not succesfull, return False

        if ret == QMessageBox.Cancel:                                           # user 2nd thoughts ? return False
            return False

        return True                                                             # we're done dealing with the current document

    def maybeKillThread(self) -> bool:
        if self.thread is not None and self.thread.isRunning():
            reply = QMessageBox.question(self, 'Please confirm', 'Cancel work in progress and lose results ?', QMessageBox.Yes, QMessageBox.Cancel)

            if reply == QMessageBox.Cancel:
                return False
            else:
                self.thread.requestInterruption()
                self.thread.quit()
                self.thread.wait()

        # by now the thread has finished, so clean up and return 'True'
        self.worker = None                                                      # moveToThread object
        self.thread = None                                                      # corresponding worker thread

        self.hideStatusbarWidgets()                                             # remove temporary widgets from statusbar (don't kill 'm)

        self.layoutImg = None                                                   # numpy array to be displayed
        self.layoutImItem = None                                                # pg ImageItem showing analysis result

        self.updateMenuStatus(True)                                             # keep menu status in sync with program's state; and reset analysis figure
        self.handleImageSelection()                                             # update the colorbar accordingly

        return True

    def setCurrentFileName(self, fileName=''):                                  # update self.fileName, set textEditModified(False) and setWindowModified(False)
        self.fileName = fileName
        self.textEdit.document().setModified(False)
        # print(f'called: {whoamI()}() from: {callerName()}(), line {lineNo()}, filename = "{self.fileName}", isWindowModified = {self.isWindowModified()}')

        if not self.fileName:                                                   # filename ="" normally indicates working with 'new' file !
            shownName = self.survey.name
        else:
            shownName = QFileInfo(fileName).fileName()

            try:                                                                # if it is already somewhere in the MRU list, remove it
                self.recentFileList.remove(fileName)
            except ValueError:
                pass

            self.recentFileList.insert(0, fileName)                             # insert it at the top
            del self.recentFileList[config.maxRecentFiles :]                     # make sure the list does not overgrow

            self.updateRecentFileActions()

        self.setWindowTitle(self.tr(f'{shownName}[*] - Roll Survey'))           # update window name, with optional * for modified status
        self.setWindowModified(False)                                           # reset document status

    def updateRecentFileActions(self):                                          # update the MRU file menu actions
        numRecentFiles = min(len(self.recentFileList), config.maxRecentFiles)   # get actual number of recent files

        for i in range(numRecentFiles):
            fileName = self.recentFileList[i]
            showName = QFileInfo(fileName).fileName()
            text = f'&{i + 1} {showName}'
            self.recentFileActions[i].setText(text)
            self.recentFileActions[i].setData(self.recentFileList[i])
            self.recentFileActions[i].setVisible(True)

        for j in range(numRecentFiles, config.maxRecentFiles):
            self.recentFileActions[j].setVisible(False)

    def fileLoad(self, fileName):

        file = QFile(fileName)
        if not file.open(QFile.ReadOnly | QFile.Text):                          # report status message and return False
            try:                                                                # remove from MRU in case of errors
                self.recentFileList.remove(fileName)
            except ValueError:
                pass

            self.appendLogMessage(f'Open&nbsp;&nbsp;&nbsp;: Cannot open file:{fileName}. Error:{file.errorString()}', MsgType.Error)
            return False
        self.survey = RollSurvey()                                              # reset the survey object; get rid of all blocks in the list !
        self.appendLogMessage(f'Opened : {fileName}')                           # send status message
        self.setCurrentFileName(fileName)                                       # update self.fileName, set textEditModified(False) and setWindowModified(False)

        stream = QTextStream(file)                                              # create a stream to read all the data
        plainText = stream.readAll()                                            # load text in a string
        file.close()                                                            # file object no longer needed

        # Xml tab
        success = self.parseText(plainText)                                     # parse the string; load the textEdit even if parsing fails !
        self.textEdit.setPlainText(plainText)                                   # update plainText widget, and reset undo/redo & modified status

        if success:                                                             # read the corresponding analysis files
            self.resetNumpyArraysAndModels()                                    # empty all arrays and reset plot titles

            if os.path.exists(self.fileName + '.bin.npy'):                      # open the existing foldmap file
                self.survey.output.binOutput = np.load(self.fileName + '.bin.npy')
                self.maximumFold = self.survey.output.binOutput.max()           # calc min/max fold is straightforward
                self.minimumFold = self.survey.output.binOutput.min()

                self.actionFold.setChecked(True)
                self.imageType = 1                                              # set analysis type to one (fold)

                self.layoutImg = self.survey.output.binOutput                   # use fold map for image data np-array
                self.layoutMax = self.maximumFold                               # use appropriate maximum
                self.layoutImItem = pg.ImageItem()                              # create PyqtGraph image item
                self.layoutImItem.setImage(self.layoutImg, levels=(0.0, self.layoutMax))

                self.appendLogMessage(f'Loaded : . . . Fold map&nbsp; : Min:{self.minimumFold} - Max:{self.maximumFold} ')
            else:
                self.survey.output.binOutput = None
                self.actionFold.setChecked(False)
                self.imageType = 0                                              # set analysis type to zero (no analysis)

            if os.path.exists(self.fileName + '.min.npy'):                      # load the existing min-offsets file
                self.survey.output.minOffset = np.load(self.fileName + '.min.npy')

                self.survey.output.minOffset[self.survey.output.minOffset == np.NINF] = np.Inf    # replace (-inf) by (inf) for min values
                self.minMinOffset = self.survey.output.minOffset.min()          # calc min offset against max (inf) values

                self.survey.output.minOffset[self.survey.output.minOffset == np.Inf] = np.NINF  # replace (inf) by (-inf) for max values
                self.maxMinOffset = self.survey.output.minOffset.max()          # calc max values against (-inf) minimum
                self.maxMinOffset = max(self.maxMinOffset, 0)                   # avoid -inf as maximum
                self.appendLogMessage(f'Loaded : . . . Min-offset: Min:{self.minMinOffset:.2f}m - Max:{self.maxMinOffset:.2f}m ')
            else:
                self.survey.output.minOffset = None

            if os.path.exists(self.fileName + '.max.npy'):                      # load the existing max-offsets file
                self.survey.output.maxOffset = np.load(self.fileName + '.max.npy')

                self.maxMaxOffset = self.survey.output.maxOffset.max()          # calc max offset against max (-inf) values
                self.maxMaxOffset = max(self.maxMaxOffset, 0)                   # avoid -inf as maximum
                self.survey.output.maxOffset[self.survey.output.maxOffset == np.NINF] = np.inf              # replace (-inf) by (inf) for min values

                self.minMaxOffset = self.survey.output.maxOffset.min()          # calc min offset against min (inf) values
                self.survey.output.maxOffset[self.survey.output.maxOffset == np.Inf] = np.NINF              # replace (inf) by (-inf) for max values
                self.appendLogMessage(f'Loaded : . . . Max-offset: Min:{self.minMaxOffset:.2f}m - Max:{self.maxMaxOffset:.2f}m ')
            else:
                self.survey.output.maxOffset = None

            if os.path.exists(self.fileName + '.rms.npy'):                      # load the existing max-offsets file
                self.survey.output.rmsOffset = np.load(self.fileName + '.rms.npy')
                self.maxRmsOffset = self.survey.output.rmsOffset.max()          # calc max offset against max (-inf) values
                self.minRmsOffset = self.survey.output.rmsOffset.min()                        # calc min offset against min (inf) values
                self.minRmsOffset = max(self.minRmsOffset, 0)                   # avoid -inf as maximum
                self.appendLogMessage(f'Loaded : . . . Rms-offset: Min:{self.minRmsOffset:.2f}m - Max:{self.maxRmsOffset:.2f}m ')
            else:
                self.survey.output.rmsOffset = None

            if os.path.exists(self.fileName + '.ana.npy'):                      # open the existing analysis file

                if self.survey.grid.fold > 0:
                    fold = self.survey.grid.fold                                # fold is defined by the grid's fold
                else:
                    fold = self.maximumFold                                     # fold is defined by observed maxfold in bin file

                if fold > 0:
                    ### if we had a large memmap file open earlier; close it and call the garbage collector
                    if self.survey.output.anaOutput is not None:
                        del self.survey.output.anaOutput                        # delete self.survey.output.anaOutput array with detailed results
                        self.D2_Output = None                                   # remove reference to self.survey.output.anaOutput
                        self.survey.output.anaOutput = None                     # remove self.survey.output.anaOutput itself
                        gc.collect()                                            # start the garbage collector to free up some space

                    w = self.survey.output.rctOutput.width()
                    h = self.survey.output.rctOutput.height()
                    dx = self.survey.grid.binSize.x()
                    dy = self.survey.grid.binSize.y()

                    nx = round(w / dx)
                    ny = round(h / dy)

                    # Note also numpy.load can be used to access a file from disk, in a memory-mapped mode
                    # See: https://numpy.org/doc/stable/reference/generated/numpy.load.html
                    anaFileName = self.fileName + '.ana.npy'
                    self.survey.output.anaOutput = np.memmap(anaFileName, dtype=np.float32, mode='r+', shape=(nx, ny, fold, 13))

                    self.D2_Output = self.survey.output.anaOutput.reshape(nx * ny * fold, 13)   # create a 2 dim array for table access
                    self.appendLogMessage(f'Loaded : . . . Analysis &nbsp;: {self.D2_Output.shape[0]:,} traces (reserved space)')
                    # print(self.survey.output.anaOutput)
                    # for i in range(nx):
                    #     for j in range(ny):
                    #         for k in range(fold):
                    #             print (self.survey.output.anaOutput[i, j, k])
                else:
                    self.anaModel.setData(None)
                    self.D2_Output = None
                    self.survey.output.anaOutput = None
            else:
                self.anaModel.setData(None)
                self.D2_Output = None
                self.survey.output.anaOutput = None

            self.anaModel.setData(self.D2_Output)                               # use this as the model data
            self.anaView.resizeColumnsToContents()                              # resize the columns of the view

            if os.path.exists(self.fileName + '.rps.npy'):                      # open the existing rps-file
                self.rpsImport = np.load(self.fileName + '.rps.npy')
                self.rpsCoordE, self.rpsCoordN, self.rpsCoordI = getRecGeometry(self.rpsImport, connect=False)

                nImport = self.rpsImport.shape[0]
                self.appendLogMessage(f'Loaded : . . . read {nImport:,} rps-records')
            else:
                self.rpsImport = None
                self.actionRpsPoints.setChecked(False)
                self.actionRpsPoints.setEnabled(False)

            if os.path.exists(self.fileName + '.sps.npy'):                      # open the existing sps-file
                self.spsImport = np.load(self.fileName + '.sps.npy')
                self.spsCoordE, self.spsCoordN, self.spsCoordI = getSrcGeometry(self.spsImport, connect=False)

                nImport = self.spsImport.shape[0]
                self.appendLogMessage(f'Loaded : . . . read {nImport:,} sps-records')
            else:
                self.spsImport = None
                self.actionSpsPoints.setChecked(False)
                self.actionSpsPoints.setEnabled(False)

            if os.path.exists(self.fileName + '.xps.npy'):                      # open the existing xps-file
                self.xpsImport = np.load(self.fileName + '.xps.npy')

                nImport = self.xpsImport.shape[0]
                self.appendLogMessage(f'Loaded : . . . read {nImport:,} xps-records')
            else:
                self.xpsImport = None

            if os.path.exists(self.fileName + '.rec.npy'):                      # open the existing rps-file
                self.recGeom = np.load(self.fileName + '.rec.npy')
                self.recCoordE, self.recCoordN, self.recCoordI = getRecGeometry(self.recGeom, connect=False)

                nImport = self.recGeom.shape[0]
                self.appendLogMessage(f'Loaded : . . . read {nImport:,} rec-records')
            else:
                self.recGeom = None
                self.actionRecPoints.setChecked(False)
                self.actionRecPoints.setEnabled(False)

            if os.path.exists(self.fileName + '.src.npy'):                      # open the existing rps-file
                self.srcGeom = np.load(self.fileName + '.src.npy')
                self.srcCoordE, self.srcCoordN, self.srcCoordI = getSrcGeometry(self.srcGeom, connect=False)

                nImport = self.srcGeom.shape[0]
                self.appendLogMessage(f'Loaded : . . . read {nImport:,} src-records')
            else:
                self.srcGeom = None
                self.actionSrcPoints.setChecked(False)
                self.actionSrcPoints.setEnabled(False)

            if os.path.exists(self.fileName + '.rel.npy'):                      # open the existing xps-file
                self.relGeom = np.load(self.fileName + '.rel.npy')

                nImport = self.relGeom.shape[0]
                self.appendLogMessage(f'Loaded : . . . read {nImport:,} rel-records')
            else:
                self.relGeom = None

            self.rpsModel.setData(self.rpsImport)                               # update the three rps/sps/xps models
            self.spsModel.setData(self.spsImport)
            self.xpsModel.setData(self.xpsImport)

            self.recModel.setData(self.recGeom)                                 # update the three rec/rel/src models
            self.relModel.setData(self.relGeom)
            self.srcModel.setData(self.srcGeom)

            self.handleImageSelection()                                         # change selection and plot survey

        self.spiderPoint = QPoint(-1, -1)                                       # reset the spider location
        index = self.anaView.model().index(0, 0)                                # turn offset into index
        self.anaView.scrollTo(index)                                            # scroll to the first trace in the trace table
        self.anaView.selectRow(0)                                               # for the time being, *only* select first row of traces in a bin
        self.updateMenuStatus(True)                                             # keep menu status in sync with program's state; and reset analysis figure
        self.enableProcessingMenuItems(True)                                    # enable processing menu items; disable 'stop processing thread'
        self.layoutWidget.enableAutoRange()                                     # make the layout plot 'fit' the survey outline
        self.mainTabWidget.setCurrentIndex(0)                                   # make sure we display the 'xml' tab

        # self.plotLayout()                                                     # plot the survey object
        self.resetSurveyProperties()                                            # get the new parameters into the parameter tree
        self.survey.checkIntegrity()                                            # check for survey integrity; in particular well file validity
        return success

    def fileImportSPS(self) -> bool:
        if not self.hideSpsCrsWarning:
            mb = QMessageBox()
            mb.setWindowTitle('Please note')
            mb.setText(
                "'SPS Import' requires the SPS data to have the same CRS as the current Roll project.  Continue importing SPS data?\n",
            )
            cb = QCheckBox("Don't show this message again")
            mb.setCheckBox(cb)
            mb.setStandardButtons(mb.Ok | mb.Cancel)
            ret = mb.exec()
            self.hideSpsCrsWarning = cb.isChecked()
            if ret == mb.Cancel:
                return False

        if not self.fileName:
            reply = QMessageBox.question(
                self,
                'Please confirm',
                "'SPS Import' requires saving this file first, to obtain a valid filename in a directory with write access.\n\nSave survey file and continue ?",
                QMessageBox.Yes,
                QMessageBox.Cancel,
            )
            if reply == QMessageBox.Cancel:
                return False

            if self.fileSaveAs() is False:
                return False

        fileNames, _ = QFileDialog.getOpenFileNames(  # filetype variable not used
            self,  # self; that's me
            'Import SPS data...',  # caption
            self.importDirectory,  # start directory + filename
            'SPS triplets (*.s01 *.r01 *.x01);;'
            'SPS triplets (*.sps *.rps *.xps);;'
            'SPS triplets (*.sp1 *.rp1 *.xp1);;'
            'Source   files (*.sps *.s01 *.sp1);;'
            'Receiver files (*.rps *.r01 *.rp1);;'
            'Relation files (*.xps *.x01 *.xp1);;'
            ' All files (*.*)',
        )                                             # file extensions
        # options -> not being used
        if not fileNames:
            return False

        self.importDirectory = os.path.dirname(fileNames[0])                # retrieve the directory name from first file found

        spsFiles = []
        rpsFiles = []
        xpsFiles = []

        nSps = 0
        nRps = 0
        nXps = 0

        for fileName in fileNames:
            suffix = QFileInfo(fileName).suffix().lower()
            if suffix.startswith('s'):
                spsFiles.append(fileName)
                nSps += rawcount(fileName)
            elif suffix.startswith('r'):
                rpsFiles.append(fileName)
                nRps += rawcount(fileName)
            elif suffix.startswith('x'):
                xpsFiles.append(fileName)
                nXps += rawcount(fileName)
            else:
                baseName = QFileInfo(fileName).completeBaseName()
                QMessageBox.information(None, 'Import error', f"Unsupported file extension in selected file(s):\n\n'{baseName}.{suffix}'\n")
                return False

        self.appendLogMessage(f"Import : importing SPS-data using the '{config.spsDialect}' SPS-dialect")
        self.appendLogMessage(f'Import : importing {len(rpsFiles)} rps-file(s), {len(spsFiles)} sps-file(s) and {len(xpsFiles)} xps-file(s)')

        with pg.BusyCursor():                                               # this may take a while; start wait cursor
            # get receiver data
            if rpsFiles:
                self.rpsImport = np.zeros(shape=nRps, dtype=pntType1)

                spsFormat = next((item for item in config.spsFormatList if item['name'] == config.spsDialect), None)
                assert spsFormat is not None, f'No valid SPS dialect with name {config.spsDialect}'
                nRps = readRPSFiles(rpsFiles, self.rpsImport, spsFormat)

                if nRps <= 0:                                              # no records found
                    self.rpsImport = None
                    self.appendLogMessage('Import : . . . unexpectedly no rps-records found', MsgType.Error)
                else:
                    self.appendLogMessage(f'Import : . . . read {nRps:,} rps-records')
            else:
                self.rpsImport = None

            # get source data
            if spsFiles:
                self.spsImport = np.zeros(shape=nSps, dtype=pntType1)

                spsFormat = next((item for item in config.spsFormatList if item['name'] == config.spsDialect), None)
                assert spsFormat is not None, f'No valid SPS dialect with name {config.spsDialect}'
                nSps = readSPSFiles(spsFiles, self.spsImport, spsFormat)

                if nSps <= 0:                                              # no records found
                    self.spsImport = None
                    self.appendLogMessage('Import : . . . unexpectedly no sps-records found', MsgType.Error)
                else:
                    self.appendLogMessage(f'Import : . . . read {nSps:,} sps-records ')
            else:
                self.spsImport = None

            # get relational  data
            if xpsFiles:
                self.xpsImport = np.zeros(shape=nXps, dtype=relType2)

                xpsFormat = next((item for item in config.xpsFormatList if item['name'] == config.spsDialect), None)
                assert xpsFormat is not None, f'No valid XPS dialect with name {config.spsDialect}'
                nXps = readXPSFiles(xpsFiles, self.xpsImport, xpsFormat)

                if nXps <= 0:                                              # no records found
                    self.xpsImport = None
                    self.appendLogMessage('Import : . . . unexpectedly no xps-records found', MsgType.Error)
                else:
                    self.appendLogMessage(f'Import : . . . read {nXps:,} xps-records')
            else:
                self.xpsImport = None

        # See: for KAP input: D:\WorkStuff\G\RIJKES-L-84599\APPS\OMNI studies\2015\Kapuni\Kapuni-files\SPS
        # See: for NAM input: D:\WorkStuff\G\RIJKES-L-84599\APPS\OMNI studies\2015\Dollard\Dollard-files\SPS\Groningen

        # sort and analyse imported arrays
        with pg.BusyCursor():
            if self.rpsImport is not None:
                nImport = self.rpsImport.shape[0]
                nUnique = markUniqueRPSrecords(self.rpsImport, sort=True)
                self.appendLogMessage(f'Import : . . . analysed rps-records; found {nUnique:,} unique records and {(nImport - nUnique):,} duplicates')

                X = calculateLineStakeTransform(self.rpsImport)
                #               A0                         + A1 * Xs                 + A2 * Ys              = Xt
                #                               B0                      + B1 * Xs                 + B2 * Ys = Yt  ==>
                self.appendLogMessage(f'Import : . . . from rps info: X = {X[0]:.2f} + {X[2]:.2f} x Nx + {X[4]:.2f} x Ny')
                self.appendLogMessage(f'Import : . . . from rps info: Y = {X[1]:.2f} + {X[3]:.2f} x Nx + {X[5]:.2f} x Ny')

                self.rpsCoordE, self.rpsCoordN, self.rpsCoordI = getRecGeometry(self.rpsImport, connect=False)
                self.tbRpsList.setChecked(True)

            if self.spsImport is not None:
                nImport = self.spsImport.shape[0]
                nUnique = markUniqueSPSrecords(self.spsImport, sort=True)
                self.appendLogMessage(f'Import : . . . analysed sps-records; found {nUnique:,} unique records and {(nImport - nUnique):,} duplicates')

                X = calculateLineStakeTransform(self.spsImport)
                #               A0                         + A1 * Xs                 + A2 * Ys              = Xt
                #                               B0                      + B1 * Xs                 + B2 * Ys = Yt  ==>
                self.appendLogMessage(f'Import : . . . from sps info: X = {X[0]:.2f} + {X[2]:.2f} x Nx + {X[4]:.2f} x Ny')
                self.appendLogMessage(f'Import : . . . from sps info: Y = {X[1]:.2f} + {X[3]:.2f} x Nx + {X[5]:.2f} x Ny')

                self.spsCoordE, self.spsCoordN, self.spsCoordI = getSrcGeometry(self.spsImport, connect=False)
                self.tbSpsList.setChecked(True)

            if self.xpsImport is not None:
                nImport = self.xpsImport.shape[0]
                nUnique = markUniqueXPSrecords(self.xpsImport, sort=True)
                self.appendLogMessage(f'Import : . . . analysed xps-records; found {nUnique:,} unique records and {(nImport - nUnique):,} duplicates')

                traces = calcMaxXPStraces(self.xpsImport)
                self.appendLogMessage(f'Import : . . . xps-records allow for maximum {traces:,} traces')

            # handle doublets  of RPS / XPS and SPS / XPS-files
            if nSps > 0 and nXps > 0:
                nSpsOrphans, nXpsOrphans = findSrcOrphans(self.spsImport, self.xpsImport)
                self.appendLogMessage(f'Import : . . . sps-records contain {nXpsOrphans:,} xps-orphans')
                self.appendLogMessage(f'Import : . . . xps-records contain {nSpsOrphans:,} sps-orphans')

            if nRps > 0 and nXps > 0:
                nRpsOrphans, nXpsOrphans = findRecOrphans(self.rpsImport, self.xpsImport)
                self.appendLogMessage(f'Import : . . . rps-records contain {nXpsOrphans:,} xps-orphans')
                self.appendLogMessage(f'Import : . . . xps-records contain {nRpsOrphans:,} rps-orphans')

        self.spsModel.setData(self.spsImport)                                   # update the three sps/rps/xps models
        self.xpsModel.setData(self.xpsImport)
        self.rpsModel.setData(self.rpsImport)

        self.mainTabWidget.setCurrentIndex(3)                                   # make sure we display the 'SPS import' tab

        self.actionRpsPoints.setEnabled(nRps > 0)
        self.actionSpsPoints.setEnabled(nSps > 0)

        self.textEdit.document().setModified(True)                              # set modified flag; so we'll save sps data as numpy arrays upon saving the file

        return True

    def fileOpen(self):
        if self.maybeKillThread() and self.maybeSave():                         # current file may be modified; save it or discard edits
            fn, _ = QFileDialog.getOpenFileName(
                self,  # self; that's me
                'Open File...',  # caption
                self.workingDirectory,  # start directory + filename
                'Survey files (*.roll);; All files (*.*)'  # file extensions
                # options                                                       # not being used
            )
            if fn:
                self.workingDirectory = os.path.dirname(fn)                     # retrieve the directory name

                # sync workingDirectory to settings, so it is available outside of RollMainWindow
                self.settings.setValue('settings/workingDirectory', self.workingDirectory)

                self.fileLoad(fn)                                               # load() does all the hard work

    def fileOpenRecent(self):
        action = self.sender()
        if action:
            self.fileLoad(action.data())

    def fileSave(self):
        if not self.fileName:                                                   # need to have a valid filename first, and set the workingDirectory
            return self.fileSaveAs()

        plainText = self.textEdit.toPlainText()
        plainText = plainText.replace('\t', '    ')                             # replace any tabs (that might be caused by editing) by 4 spaces

        file = QFile(self.fileName)
        success = file.open(QIODevice.WriteOnly | QIODevice.Truncate)

        if success:
            _ = QTextStream(file) << plainText                                  # unused stream replaced by _ to make PyLint happy
            self.appendLogMessage(f'Saved&nbsp;&nbsp;: {self.fileName}')
            self.textEdit.document().setModified(False)
            file.close()

            # try to save the analysis files as well
            if self.survey.output.binOutput is not None:
                np.save(self.fileName + '.bin.npy', self.survey.output.binOutput)   # numpy array with fold map

            if self.survey.output.minOffset is not None:
                np.save(self.fileName + '.min.npy', self.survey.output.minOffset)   # numpy array with min-offset map

            if self.survey.output.maxOffset is not None:
                np.save(self.fileName + '.max.npy', self.survey.output.maxOffset)   # numpy array with max-offset map

            if self.survey.output.rmsOffset is not None:
                np.save(self.fileName + '.rms.npy', self.survey.output.rmsOffset)   # numpy array with max-offset map

            if self.rpsImport is not None:
                np.save(self.fileName + '.rps.npy', self.rpsImport)             # numpy array with list of RPS records

            if self.spsImport is not None:
                np.save(self.fileName + '.sps.npy', self.spsImport)             # numpy array with list of SPS records

            if self.xpsImport is not None:
                np.save(self.fileName + '.xps.npy', self.xpsImport)             # numpy array with list of XPS records

            if self.recGeom is not None:
                np.save(self.fileName + '.rec.npy', self.recGeom)               # numpy array with list of REC records

            if self.relGeom is not None:
                np.save(self.fileName + '.rel.npy', self.relGeom)               # numpy array with list of REL records

            if self.srcGeom is not None:
                np.save(self.fileName + '.src.npy', self.srcGeom)               # numpy array with list of SRC records
        else:
            self.appendLogMessage(f'saving : Cannot save file: {self.fileName}', MsgType.Error)
            QMessageBox.information(self, 'Write error', f'Cannot save file:\n{self.fileName}')

        self.updateMenuStatus(False)                                            # keep menu status in sync with program's state; don't reset analysis figure

        return success

    def fileSaveAs(self):
        fileName = os.path.join(self.workingDirectory, self.survey.name)        # join dir & survey name, as proposed file path
        fn, _ = QFileDialog.getSaveFileName(
            self,  # that's me
            'Save as...',  # dialog caption
            fileName,  # start directory + filename
            'Survey files (*.roll);; All files (*.*)'  # file extensions
            # options                                                           # options not used
        )
        if not fn:
            return False

        if not fn.lower().endswith('.roll'):                                    # make sure file extension is okay
            fn += '.roll'                                                       # just add the file extension

        self.setCurrentFileName(fn)                                             # update self.fileName, set textEditModified(False) and setWindowModified(False)
        return self.fileSave()

    def fileSettings(self):                                                     # dialog implementation modeled after https://github.com/dglent/meteo-qt/blob/master/meteo_qt/settings.py
        dlg = SettingsDialog(self)
        dlg.appliedSignal.connect(self.updateSettings)

        if dlg.exec():                                                          # Run the dialog event loop, and obtain survey object
            self.updateSettings()

    def updateSettings(self):
        self.handleImageSelection()
        self.plotLayout()

    def fileExportAsCsv(self, fileName, extension):
        fn, selectedFilter = QFileDialog.getSaveFileName(
            self,  # that's me
            'Save as...',  # dialog caption
            fileName + extension,  # start directory + filename
            'comma separated file (*.csv);;semicolumn separated file (*.csv);;space separated file (*.csv);;tab separated file (*.csv);;All files (*.*)',  # file extensions
            # options                                                           # options not used
        )

        delimiter = ','                                                         # default delimiter value
        if fn:                                                                  # we have a valid file name
            if selectedFilter == 'semicolumn separated file (*.csv)':           # select appropriate delimiter
                delimiter = ';'
            elif selectedFilter == 'space separated file (*.csv)':
                delimiter = ' '
            elif selectedFilter == 'tab separated file (*.csv)':
                delimiter = '\t'

            if not fn.lower().endswith(extension):                              # make sure file extension is okay
                fn += extension                                                 # just add the file extension

        return (fn, delimiter)

    def fileExportAnaAsCsv(self):

        fn, delimiter = self.fileExportAsCsv(self.fileName, '.ana.csv')

        if not fn:
            return False

        hdr = asstr(delimiter).join(map(asstr, self.anaModel.getHeader()))     # one-liner (from numpy) to assemble header
        fmt = self.anaModel.getFormat()
        data = self.D2_Output

        with pg.BusyCursor():
            np.savetxt(fn, data, delimiter=delimiter, fmt=fmt, comments='', header=hdr)
            self.appendLogMessage(f"Export : exported {self.D2_Output.shape[0]:,} lines to '{fn}'")

    def fileExportRecAsCsv(self):

        fn, delimiter = self.fileExportAsCsv(self.fileName, '.rec.csv')

        if not fn:
            return False

        hdr = asstr(delimiter).join(map(asstr, self.recModel.getHeader()))     # one-liner (from numpy) to assemble header
        fmt = self.recModel.getFormat()
        data = self.recModel.getData()

        with pg.BusyCursor():
            np.savetxt(fn, data, delimiter=delimiter, fmt=fmt, comments='', header=hdr)
            self.appendLogMessage(f"Export : exported {data.shape[0]:,} lines to '{fn}'")

    def fileExportRpsAsR01(self):

        fileName = self.fileName + '.sps.r01'                                   # start directory + filename
        data = self.rpsModel.getData()

        fn, records = fileExportAsR01(self, fileName, data, self.survey.crs)
        if records > 0:
            self.appendLogMessage(f"Export : exported {records:,} lines to '{fn}'")

    def fileExportRecAsR01(self):

        fileName = self.fileName + '.rec.r01'                                   # start directory + filename
        data = self.recModel.getData()

        fn, records = fileExportAsR01(self, fileName, data, self.survey.crs)
        if records > 0:
            self.appendLogMessage(f"Export : exported {records:,} lines to '{fn}'")

    def fileExportSpsAsS01(self):

        fileName = self.fileName + '.sps.s01'                                   # start directory + filename
        data = self.spsModel.getData()

        fn, records = fileExportAsS01(self, fileName, data, self.survey.crs)
        if records > 0:
            self.appendLogMessage(f"Export : exported {records:,} lines to '{fn}'")

    def fileExportSrcAsS01(self):

        fileName = self.fileName + '.src.s01'                                   # start directory + filename
        data = self.srcModel.getData()

        fn, records = fileExportAsS01(self, fileName, data, self.survey.crs)
        if records > 0:
            self.appendLogMessage(f"Export : exported {records:,} lines to '{fn}'")

    def fileExportXpsAsX01(self):

        fileName = self.fileName + '.sps.x01'                                   # start directory + filename
        data = self.xpsModel.getData()

        fn, records = fileExportAsX01(self, fileName, data, self.survey.crs)
        if records > 0:
            self.appendLogMessage(f"Export : exported {records:,} lines to '{fn}'")

    def fileExportRelAsX01(self):

        fileName = self.fileName + '.rel.x01'                                   # start directory + filename
        data = self.relModel.getData()

        fn, records = fileExportAsX01(self, fileName, data, self.survey.crs)
        if records > 0:
            self.appendLogMessage(f"Export : exported {records:,} lines to '{fn}'")

    def fileExportSrcAsCsv(self):

        fn, delimiter = self.fileExportAsCsv(self.fileName, '.src.csv')

        if not fn:
            return False

        hdr = asstr(delimiter).join(map(asstr, self.srcModel.getHeader()))     # one-liner (from numpy) to assemble header
        fmt = self.srcModel.getFormat()
        data = self.srcModel.getData()

        with pg.BusyCursor():
            np.savetxt(fn, data, delimiter=delimiter, fmt=fmt, comments='', header=hdr)
            self.appendLogMessage(f"Export : exported {data.shape[0]:,} lines to '{fn}'")

    def fileExportRelAsCsv(self):

        fn, delimiter = self.fileExportAsCsv(self.fileName, '.rel.csv')

        if not fn:
            return False

        hdr = asstr(delimiter).join(map(asstr, self.relModel.getHeader()))     # one-liner (from numpy) to assemble header
        fmt = self.relModel.getFormat()
        data = self.relModel.getData()

        with pg.BusyCursor():
            np.savetxt(fn, data, delimiter=delimiter, fmt=fmt, comments='', header=hdr)
            self.appendLogMessage(f"Export : exported {data.shape[0]:,} lines to '{fn}'")

    def fileExportRpsAsCsv(self):

        fn, delimiter = self.fileExportAsCsv(self.fileName, '.rps.csv')

        if not fn:
            return False

        hdr = asstr(delimiter).join(map(asstr, self.rpsModel.getHeader()))     # one-liner (from numpy) to assemble header
        fmt = self.rpsModel.getFormat()
        data = self.rpsModel.getData()

        with pg.BusyCursor():
            np.savetxt(fn, data, delimiter=delimiter, fmt=fmt, comments='', header=hdr)
            self.appendLogMessage(f"Export : exported {data.shape[0]:,} lines to '{fn}'")

    def fileExportSpsAsCsv(self):

        fn, delimiter = self.fileExportAsCsv(self.fileName, '.sps.csv')

        if not fn:
            return False

        hdr = asstr(delimiter).join(map(asstr, self.spsModel.getHeader()))     # one-liner (from numpy) to assemble header
        fmt = self.spsModel.getFormat()
        data = self.spsModel.getData()

        with pg.BusyCursor():
            np.savetxt(fn, data, delimiter=delimiter, fmt=fmt, comments='', header=hdr)
            self.appendLogMessage(f"Export : exported {data.shape[0]:,} lines to '{fn}'")

    def fileExportXpsAsCsv(self):

        fn, delimiter = self.fileExportAsCsv(self.fileName, '.xps.csv')

        if not fn:
            return False

        hdr = asstr(delimiter).join(map(asstr, self.xpsModel.getHeader()))     # one-liner (from numpy) to assemble header
        fmt = self.xpsModel.getFormat()
        data = self.xpsModel.getData()

        with pg.BusyCursor():
            np.savetxt(fn, data, delimiter=delimiter, fmt=fmt, comments='', header=hdr)
            self.appendLogMessage(f"Export : exported {data.shape[0]:,} lines to '{fn}'")

    def filePrint(self):
        printer = QPrinter(QPrinter.HighResolution)
        dlg = QPrintDialog(printer, self)

        if self.textEdit.textCursor().hasSelection():
            dlg.addEnabledOption(QPrintDialog.PrintSelection)

        dlg.setWindowTitle('Print Document')

        if dlg.exec_() == QPrintDialog.Accepted:
            self.textEdit.print_(printer)

        del dlg

    def filePrintPreview(self):
        printer = QPrinter(QPrinter.HighResolution)
        preview = QPrintPreviewDialog(printer, self)
        preview.paintRequested.connect(self.printPreview)
        preview.exec_()

    def printPreview(self, printer):
        self.textEdit.print_(printer)

    def filePrintPdf(self):
        fn, _ = QFileDialog.getSaveFileName(self, 'Export PDF', None, 'PDF files (*.pdf);;All Files (*)')

        if fn:
            if QFileInfo(fn).suffix().isEmpty():
                fn += '.pdf'

            printer = QPrinter(QPrinter.HighResolution)
            printer.setOutputFormat(QPrinter.PdfFormat)
            printer.setOutputFileName(fn)
            self.textEdit.document().print_(printer)

    def viewDebug(self):
        self.debug = not self.debug                                             # toggle status
        self.actionDebug.setChecked(self.debug)                                 # update status in button
        # See also: https://stackoverflow.com/questions/8391411/how-to-block-calls-to-print

        if self.debug:
            # builtins.print = self.oldPrint                                    # use/restore builtins.print
            if console._console is None:                                        # pylint: disable=W0212 # unfortunately, need to access protected members
                console.show_console()
            else:
                console._console.setUserVisible(True)                           # pylint: disable=W0212 # unfortunately, need to access protected members
            print('print() to Python console has been enabled; Python console is opened')             # this message should always be printed
            self.appendLogMessage('Debug&nbsp;&nbsp;: print() to Python console has been enabled')
        else:
            print('print() to Python console has been disabled; Python console is closed')            # this message is the last one to be printed
            # builtins.print = silentPrint                                        # suppress print output by calling 'dummy' routine
            print('this print message should not show')                         # this message is the last one to be printed

            if console._console is not None:                                    # pylint: disable=W0212 # unfortunately, need to access protected members
                console._console.setUserVisible(False)                          # pylint: disable=W0212 # unfortunately, need to access protected members
            self.appendLogMessage('Debug&nbsp;&nbsp;: print() to Python console has been disabled')

    def appendLogMessage(self, message: str = 'test', index: MsgType = MsgType.Info):
        # dateTime = QDateTime.currentDateTime().toString("dd-MM-yyyy hh:mm:ss")
        dateTime = QDateTime.currentDateTime().toString('yyyy-MM-ddTHH:mm:ss')  # UTC time; same format as is used in QGis

        if index == MsgType.Debug and not self.debug:                         # debug message, which needs to be suppressed
            return

        # use &nbsp; (non-breaking-space) to prevent html eating up subsequent spaces !
        switch = {  # see: https://doc.qt.io/qt-6/qcolor.html
            MsgType.Info: f'<p>{dateTime}&nbsp;&nbsp;&nbsp;&nbsp;&nbsp;info&nbsp;&nbsp;&nbsp;&nbsp;&nbsp;&nbsp;&nbsp;{message}</p>',  # info     = black
            MsgType.Binning: f'<p style="color:blue" >{dateTime}&nbsp;&nbsp;&nbsp;&nbsp;&nbsp;binning&nbsp;&nbsp;&nbsp;&nbsp;{message}</p>',  # Binning  = blue
            MsgType.Geometry: f'<p style="color:green">{dateTime}&nbsp;&nbsp;&nbsp;&nbsp;&nbsp;geometry&nbsp;&nbsp;&nbsp;{message}</p>',  # Geometry = green
            MsgType.Debug: f'<p style="color:darkCyan">{dateTime}&nbsp;&nbsp;&nbsp;&nbsp;&nbsp;debug&nbsp;&nbsp;&nbsp;&nbsp;&nbsp;&nbsp;Called : {message}</p>',  # Debug    = darkCyan
            MsgType.Error: f'<p style="color:red">{dateTime}&nbsp;&nbsp;&nbsp;&nbsp;&nbsp;error&nbsp;&nbsp;&nbsp;&nbsp;&nbsp;&nbsp;{message}</p>',  # eRror    = red
            MsgType.Exception: f'<p style="color:red">{dateTime}&nbsp;&nbsp;&nbsp;&nbsp;&nbsp;exception&nbsp;&nbsp;{message}</p>',  # exception= bold red
        }

        self.logEdit.appendHtml(switch.get(index, 'Unknown Message type'))      # Adds the message to the widget
        self.logEdit.moveCursor(QTextCursor.End)                                # scroll the edit control

        if index == MsgType.Exception:                                          # play a sound to notify end user of exception
            # SystemAsterisk, SystemExclamation, SystemExit, SystemHand, SystemQuestion are common sounds; use asyync to avoid waiting on sound to finish
            winsound.PlaySound('SystemHand', winsound.SND_ALIAS | winsound.SND_ASYNC)

    def parseText(self, plainText):
        if len(plainText) == 0:                                                 # no text available; return False
            self.appendLogMessage('Parse&nbsp;&nbsp;: error occurred while parsing empty string', MsgType.Error)
            return False
        # print (plainText)
        doc = QDomDocument()                                                    # first get a QDomDocument to work with
        success, errorMsg, errorLine, errorColumn = doc.setContent(plainText)

        if success:                                                             # parsing went ok, start with a new survey object
            self.survey = RollSurvey()                                          # only reset the survey object upon succesful parse
            self.survey.readXml(doc)                                            # build the RollSurvey object tree
            self.survey.calcTransforms()                                        # (re)calculate the transforms being used
            self.survey.calcSeedData()                                          # needed for circles, spirals & well-seeds; may affect bounding box
            self.survey.calcBoundingRect()                                      # (re)calculate the boundingBox as part of parsing the data
            self.survey.calcNoShotPoints()                                      # (re)calculate nr of SPs

            self.appendLogMessage(f'Parsed : {self.fileName} survey object updated')
            return True
        else:  # an error occurred
            self.appendLogMessage(
                f'Parse&nbsp;&nbsp;: {errorMsg}, at line: {errorLine} col:{errorColumn}; survey object not updated',
                MsgType.Error,
            )
            return False

    def OnAbout(self):
        QMessageBox.about(self, 'About Roll', aboutText())

    def OnLicense(self):
        QMessageBox.about(self, 'License conditions', licenseText())

    def OnHighDpi(self):
        QMessageBox.about(self, 'High DPI UI scaling issues', highDpiText())

    def clipboardHasText(self):
        return len(QApplication.clipboard().text()) != 0

    def enableProcessingMenuItems(self, enable=True):
        self.actionBasicBinFromTemplates.setEnabled(enable)
        self.actionFullBinFromTemplates.setEnabled(enable)
        if enable is True and self.srcGeom is not None and self.relGeom is not None and self.recGeom is not None:
            self.actionBasicBinFromGeometry.setEnabled(enable)
            self.actionFullBinFromGeometry.setEnabled(enable)
        else:
            self.actionBasicBinFromGeometry.setEnabled(False)
            self.actionFullBinFromGeometry.setEnabled(False)

        if enable is True and self.spsImport is not None and self.xpsImport is not None and self.rpsImport is not None:
            self.actionBasicBinFromSps.setEnabled(enable)
            self.actionFullBinFromSps.setEnabled(enable)
        else:
            self.actionBasicBinFromSps.setEnabled(False)
            self.actionFullBinFromSps.setEnabled(False)

        self.actionGeometryFromTemplates.setEnabled(enable)
        self.actionStopThread.setEnabled(not enable)

    def testBasicBinningConditions(self) -> bool:
        if self.survey.unique.apply:                                            # can't do unique fold with basic binning
            QMessageBox.information(
                self,
                'Please adjust',
                "Applying 'Unique offsets' requires using a 'Full Binning' process, as it is implemented as a post-processing step on the trace table",
                QMessageBox.Cancel,
            )
            return False
        else:
            return True

    def testFullBinningConditions(self) -> bool:
        if self.survey.grid.fold <= 0:                                          # the size of the analysis file is defined by the grid's fold
            QMessageBox.information(
                self,
                'Please adjust',
                "'Full Binning' requires selecting a max fold value > 0 in the 'local grid' settings, to allocate space for a memory mapped file.\n\n"
                "You can determine the required max fold value by first running 'Basic Binning'",
                QMessageBox.Cancel,
            )
            return False

        if not self.fileName:
            QMessageBox.information(self, 'Please adjust', "'Full Binning' requires saving this file first, to obtain a valid filename in a directory with write access.", QMessageBox.Cancel)
            return False

        return True

    def prepFullBinningConditions(self):
        if self.survey.output.anaOutput is not None:                                          # get rid of current memory mapped array first
            self.survey.output.anaOutput.flush()                                              # write all stuff to disk
            del self.survey.output.anaOutput                                                  # delete object
            self.D2_Output = None                                               # remove reference to self.survey.output.anaOutput
            self.survey.output.anaOutput = None                                               # remove self.survey.output.anaOutput itself
            gc.collect()                                                        # free up memory as much as possible

        # prepare for a new memory mapped object; calculate memmap size
        w = self.survey.output.rctOutput.width()
        h = self.survey.output.rctOutput.height()
        dx = self.survey.grid.binSize.x()
        dy = self.survey.grid.binSize.y()
        nx = ceil(w / dx)
        ny = ceil(h / dy)
        fold = self.survey.grid.fold
        n = nx * ny * fold
        self.appendLogMessage(f'Thread : prepare memory mapped file for {n:,} traces, with nx={nx}, ny={ny}, fold={fold:,}', MsgType.Binning)
        # See: https://stackoverflow.com/questions/22385801/best-practices-with-reading-and-operating-on-fortran-ordered-arrays-with-numpy
        # The order='F' in the allocation of the array could impact reading/writing data to the mmap array
        # This depends on how he binning process traverses through the binning area

        try:
            anaFileName = self.fileName + '.ana.npy'
            if os.path.exists(anaFileName):
                self.survey.output.anaOutput = np.memmap(anaFileName, shape=(nx, ny, fold, 13), dtype=np.float32, mode='r+')
            else:
                self.survey.output.anaOutput = np.memmap(anaFileName, shape=(nx, ny, fold, 13), dtype=np.float32, mode='w+')
            self.survey.output.anaOutput.fill(0.0)                                            # enforce zero values for all elements

            # self.D2_output = self.survey.output.anaOutput.reshape(nx * ny * fold, 13)
            # self.D2_output[:,:] = 0.0

        except MemoryError as e:
            self.appendLogMessage(f'Thread : memory error {e}', MsgType.Error)
            return False

        return True

    def basicBinFromTemplates(self):
        if self.testBasicBinningConditions():
            self.binFromTemplates(False)

    def fullBinFromTemplates(self):
        if self.testFullBinningConditions():
            self.binFromTemplates(True)

    def basicBinFromGeometry(self):
        if self.testBasicBinningConditions():
            self.binFromGeometry(False)

    def fullBinFromGeometry(self):
        if self.testFullBinningConditions():
            self.binFromGeometry(True)

    def basicBinFromSps(self):
        if self.testBasicBinningConditions():
            self.binFromSps(False)

    def fullBinFromSps(self):
        if self.testFullBinningConditions():
            self.binFromSps(True)

    def binFromTemplates(self, fullAnalysis):

        if fullAnalysis:
            success = self.prepFullBinningConditions()
            if not success:
                return
            self.progressLabel.setText('Bin from Templates - full analysis')
        else:
            self.progressLabel.setText('Bin from Templates - basic analysis')

        self.showStatusbarWidgets()                                             # show two temporary progress widgets
        self.enableProcessingMenuItems(False)                                   # disable processing menu items

        self.appendLogMessage(f"Thread : started 'Bin from templates', using {self.survey.nShotPoints:,} shot points", MsgType.Binning)

        # Step 1: Create a worker class
        # done in file 'worker_threads.py'

        # Step 2: Create a QThread object
        self.thread = QThread()

        # Step 3: Create a worker object
        xmlString = self.survey.toXmlString()
        self.worker = BinningWorker(xmlString)

        # and define the binning mode
        self.worker.setExtended(fullAnalysis)

        # pass the memory mapped file to worker
        self.worker.setMemMappedFile(self.survey.output.anaOutput)

        # Step 4: Move worker to the thread
        self.worker.moveToThread(self.thread)

        # Step 5: Connect signals and slots
        self.thread.started.connect(self.worker.run)                            # start thread
        self.worker.survey.progress.connect(self.threadProgress)                # report thread progress from the survey object in the worker
        self.worker.survey.message.connect(self.threadMessage)                  # update thread task-description in statusbar
        self.worker.finished.connect(self.binningThreadFinished)                # stop thread;
        self.worker.finished.connect(self.thread.quit)
        # self.worker.finished.connect(self.worker.deleteLater)                 # See: http://qt-project.org/forums/viewthread/19848

        # Step 6: Start the thread
        self.startTime = timer()
        self.thread.start(QThread.NormalPriority)

    def binFromGeometry(self, fullAnalysis):
        if self.srcGeom is None or self.relGeom is None or self.recGeom is None:
            self.appendLogMessage('Thread : one or more of the geometry files have not been defined', MsgType.Error)
            return

        if fullAnalysis:
            success = self.prepFullBinningConditions()
            if not success:
                return
            self.progressLabel.setText('Bin from Geometry - full analysis')
        else:
            self.progressLabel.setText('Bin from Geometry - basic analysis')

        self.showStatusbarWidgets()                                             # show two temporary progress widgets
        self.enableProcessingMenuItems(False)                                   # disable processing menu items

        self.appendLogMessage(f"Thread : started 'Bin from geometry', using {self.srcGeom.shape[0]:,} shot points", MsgType.Binning)

        # Step 1: Create a worker class
        # done in file 'worker_threads.py'

        # Step 2: Create a QThread object
        self.thread = QThread()

        # Step 3: Create a worker object
        xmlString = self.survey.toXmlString()
        self.worker = BinFromGeometryWorker(xmlString)

        # and define the binning mode
        self.worker.setExtended(fullAnalysis)

        # pass the memory mapped file to worker
        self.worker.setMemMappedFile(self.survey.output.anaOutput)

        # Pass the geometry tables, in same order as used in the Geometry tab
        self.worker.setGeometryArrays(self.srcGeom, self.relGeom, self.recGeom)

        # Step 4: Move worker to the thread
        self.worker.moveToThread(self.thread)

        # Step 5: Connect signals and slots
        self.thread.started.connect(self.worker.run)                            # start thread
        self.worker.survey.progress.connect(self.threadProgress)                # report thread progress from the survey object in the worker
        self.worker.survey.message.connect(self.threadMessage)                  # update thread task-description in statusbar
        self.worker.finished.connect(self.binningThreadFinished)                # stop thread;
        self.worker.finished.connect(self.thread.quit)
        # self.worker.finished.connect(self.worker.deleteLater)                 # See: http://qt-project.org/forums/viewthread/19848

        # Step 6: Start the thread
        self.startTime = timer()
        self.thread.start(QThread.NormalPriority)

    def binFromSps(self, fullAnalysis):
        if self.spsImport is None or self.xpsImport is None or self.rpsImport is None:
            self.appendLogMessage('Thread : one or more of the sps files have not been defined', MsgType.Error)
            return

        if fullAnalysis:
            success = self.prepFullBinningConditions()
            if not success:
                return
            self.progressLabel.setText('Bin from imported SPS - full analysis')
        else:
            self.progressLabel.setText('Bin from imported SPS - basic analysis')

        self.showStatusbarWidgets()                                             # show two temporary progress widgets
        self.enableProcessingMenuItems(False)                                   # disable processing menu items

        self.appendLogMessage(f"Thread : started 'Bin from Imported SPS', using {self.spsImport.shape[0]:,} shot points", MsgType.Binning)

        # Step 1: Create a worker class
        # done in file 'worker_threads.py'

        # Step 2: Create a QThread object
        self.thread = QThread()

        # Step 3: Create a worker object
        xmlString = self.survey.toXmlString()
        self.worker = BinFromGeometryWorker(xmlString)

        # and define the binning mode
        self.worker.setExtended(fullAnalysis)

        # pass the memory mapped file to worker
        self.worker.setMemMappedFile(self.survey.output.anaOutput)

        # Pass the geometry tables, in same order as used in the Geometry tab
        self.worker.setGeometryArrays(self.spsImport, self.xpsImport, self.rpsImport)

        # Step 4: Move worker to the thread
        self.worker.moveToThread(self.thread)

        # Step 5: Connect signals and slots
        self.thread.started.connect(self.worker.run)                            # start thread
        self.worker.survey.progress.connect(self.threadProgress)                # report thread progress from the survey object in the worker
        self.worker.survey.message.connect(self.threadMessage)                  # update thread task-description in statusbar
        self.worker.finished.connect(self.binningThreadFinished)                # stop thread;
        self.worker.finished.connect(self.thread.quit)
        # self.worker.finished.connect(self.worker.deleteLater)                 # See: http://qt-project.org/forums/viewthread/19848

        # Step 6: Start the thread
        self.startTime = timer()
        self.thread.start(QThread.NormalPriority)

    def createGeometryFromTemplates(self):
        self.progressLabel.setText('Create Geometry from Templates')

        self.showStatusbarWidgets()                                             # show two temporary progress widgets
        self.enableProcessingMenuItems(False)                                   # disable processing menu items

        self.appendLogMessage(f"Thread : started 'Create Geometry from Templates', from {self.survey.nShotPoints:,} shot points", MsgType.Geometry)

        # Step 1: Create a worker class
        # done in file 'worker_threads.py'

        # Step 2: Create a QThread object
        self.thread = QThread()

        # Step 3: Create a worker object
        xmlString = self.survey.toXmlString()
        self.worker = GeometryWorker(xmlString)

        # Step 4: Move worker to the thread
        self.worker.moveToThread(self.thread)

        # Step 5: Connect signals and slots
        self.thread.started.connect(self.worker.run)                            # start thread
        self.worker.survey.progress.connect(self.threadProgress)                # report thread progress from the survey object in the worker
        self.worker.survey.message.connect(self.threadMessage)                  # update thread task-description in statusbar
        self.worker.finished.connect(self.geometryThreadFinished)               # stop thread;
        self.worker.finished.connect(self.thread.quit)
        # self.worker.finished.connect(self.worker.deleteLater)                 # See: http://qt-project.org/forums/viewthread/19848

        # Step 6: Start the thread
        self.startTime = timer()
        self.thread.start(QThread.NormalPriority)

    def threadProgress(self, val):
        if self.progressBar is not None:                                        # in case thread is rumming before GUI set up
            self.progressBar.setValue(val)

    def threadMessage(self, val):
        if self.progressLabel is not None:                                      # in case thread is rumming before GUI set up
            self.progressLabel.setText(val)

    def stopWorkerThread(self):
        if self.thread is not None and self.thread.isRunning():
            self.thread.requestInterruption()

    def binningThreadFinished(self, success):                                   # extra argument only available in BinningWorker class

        if not success:
            self.layoutImg = None                                               # numpy array to be displayed
            self.layoutImItem = None                                            # pg ImageItem showing analysis result
            self.handleImageSelection()                                         # change selection and plot survey

            self.appendLogMessage('Thread : . . . aborted binning operation', MsgType.Error)
            self.appendLogMessage(f'Thread : . . . {self.worker.survey.errorText}', MsgType.Error)
            QMessageBox.information(self, 'Interrupted', 'Worker thread aborted')
        else:
            # copy analysis arrays from worker
            self.survey.output.binOutput = self.worker.survey.output.binOutput.copy()       # create a copy; not a view of the array(s)
            self.survey.output.minOffset = self.worker.survey.output.minOffset.copy()
            self.survey.output.maxOffset = self.worker.survey.output.maxOffset.copy()

            if self.worker.survey.output.anaOutput is not None:                             # extended binning
                self.survey.output.rmsOffset = self.worker.survey.output.rmsOffset.copy()   # only defined in extended binning
                self.survey.output.anaOutput = self.worker.survey.output.anaOutput.copy()   # results from full binning analysis
                shape = self.survey.output.anaOutput.shape
                self.D2_Output = self.survey.output.anaOutput.reshape(shape[0] * shape[1] * shape[2], shape[3])
                self.anaModel.setData(self.D2_Output)
                self.anaView.resizeColumnsToContents()

            # copy limits from worker; avoid -inf values
            self.minimumFold = max(self.worker.survey.output.minimumFold, 0)
            self.maximumFold = max(self.worker.survey.output.maximumFold, 0)
            self.minMinOffset = max(self.worker.survey.output.minMinOffset, 0)
            self.maxMinOffset = max(self.worker.survey.output.maxMinOffset, 0)
            self.minMaxOffset = max(self.worker.survey.output.minMaxOffset, 0)
            self.maxMaxOffset = max(self.worker.survey.output.maxMaxOffset, 0)
            self.minRmsOffset = max(self.worker.survey.output.minRmsOffset, 0)
            self.maxRmsOffset = max(self.worker.survey.output.maxRmsOffset, 0)

            if self.survey.grid.fold <= 0:
                self.survey.grid.fold = self.maximumFold                        # make sure we have a workable fold value next time around

                # now we need to update the fold value in the text edit as well
                plainText = self.survey.toXmlString()                           # convert the survey object itself to an xml string
                self.textEdit.setTextViaCursor(plainText)                       # get text into the textEdit, NOT resetting its doc status
                self.textEdit.document().setModified(True)                      # we edited the document; so it's been modified

            if self.imageType == 0:                                             # if no image was selected before
                self.actionFold.setChecked(True)
                self.imageType = 1                                              # set analysis type to one (fold)

            if self.imageType == 1:
                self.layoutImg = self.survey.output.binOutput
                self.layoutMax = self.maximumFold
                label = 'fold'
            elif self.imageType == 2:
                self.layoutImg = self.survey.output.minOffset
                self.layoutMax = self.maxMinOffset
                label = 'minimum offset'
            elif self.imageType == 3:
                self.layoutImg = self.survey.output.maxOffset
                self.layoutMax = self.maxMaxOffset
                label = 'maximum offset'
            elif self.imageType == 4:
                self.layoutImg = self.survey.output.rmsOffset
                self.layoutMax = self.maxRmsOffset
                label = 'rms delta-offset'
            else:
                raise NotImplementedError('selected analysis type currently not implemented.')

            self.layoutImItem = pg.ImageItem()                                  # create PyqtGraph image item
            self.layoutImItem.setImage(self.layoutImg, levels=(0.0, self.layoutMax))

            # just to be sure; copy cmpTransform back from worker's survey object
            self.survey.cmpTransform = self.worker.survey.cmpTransform

            if self.layoutColorBar is None:
                self.layoutColorBar = self.layoutWidget.plotItem.addColorBar(self.layoutImItem, colorMap=config.fold_OffCmap, label=label, limits=(0, None), rounding=10.0, values=(0.0, self.layoutMax))
            else:
                self.layoutColorBar.setImageItem(self.layoutImItem)
                self.layoutColorBar.setLevels(low=0.0, high=self.layoutMax)
                self.layoutColorBar.setColorMap(config.fold_OffCmap)
                self.setColorbarLabel(label)

            self.plotLayout()                                                   # plot survey with colorbar

            endTime = timer()
            elapsed = timedelta(seconds=endTime - self.startTime)               # get the elapsed time for binning
            elapsed = timedelta(seconds=ceil(elapsed.total_seconds()))          # round up to nearest second

            self.appendLogMessage(f'Thread : Binning completed. Elapsed time:{elapsed} ', MsgType.Binning)
            self.appendLogMessage(f'Thread : . . . Fold&nbsp; &nbsp; &nbsp; &nbsp;: Min:{self.minimumFold} - Max:{self.maximumFold} ', MsgType.Binning)
            self.appendLogMessage(f'Thread : . . . Min-offsets: Min:{self.minMinOffset:.2f}m - Max:{self.maxMinOffset:.2f}m ', MsgType.Binning)
            self.appendLogMessage(f'Thread : . . . Max-offsets: Min:{self.minMaxOffset:.2f}m - Max:{self.maxMaxOffset:.2f}m ', MsgType.Binning)

            if self.survey.output.rmsOffset is not None:                        # only do this for "Full binning"
                self.appendLogMessage(f'Thread : . . . Rms-offsets: Min:{self.minRmsOffset:.2f}m - Max:{self.maxRmsOffset:.2f}m ', MsgType.Binning)

            info = ''
            if not self.fileName:
                # we created an analysis for a yet to be saved file; set modified True, to force saving results when name has been defined
                self.textEdit.document().setModified(True)
                info = 'Analysis results are yet to be saved.'
            else:
                # Save the analysis results
                np.save(self.fileName + '.bin.npy', self.survey.output.binOutput)
                np.save(self.fileName + '.min.npy', self.survey.output.minOffset)
                np.save(self.fileName + '.max.npy', self.survey.output.maxOffset)

                # Note, the rms offset results are only available with extended binning
                if self.survey.output.rmsOffset is not None:                    # only do this for "Full binning"
                    np.save(self.fileName + '.rms.npy', self.survey.output.rmsOffset)

                info = 'Analysis results have been saved.'

            QMessageBox.information(self, 'Done', f'Worker thread completed. {info} ')

        self.updateMenuStatus(False)                                            # keep menu status in sync with program's state; don't reset analysis figure
        self.enableProcessingMenuItems()                                        # enable processing menu items (again)
        self.hideStatusbarWidgets()                                             # remove temporary widgets from statusbar (don't kill 'm)

    def geometryThreadFinished(self, success):

        if self.debug:
            self.appendLogMessage('geometryFromTemplates() profiling information', MsgType.Debug)
            i = 0
            while i < len(self.worker.survey.timerTmin):                        # log some debug messages
                tMin = self.worker.survey.timerTmin[i] * 1000.0 if self.worker.survey.timerTmin[i] != float('Inf') else 0.0
                tMax = self.worker.survey.timerTmax[i] * 1000.0
                tTot = self.worker.survey.timerTtot[i] * 1000.0
                freq = self.worker.survey.timerFreq[i]
                tAvr = tTot / freq if freq > 0 else 0.0
                message = f'{i:02d}: min:{tMin:011.3f}, max:{tMax:011.3f}, tot:{tTot:011.3f}, avr:{tAvr:011.3f}, freq:{freq:07d}'
                self.appendLogMessage(message, MsgType.Debug)
                i += 1

        if not success:
            self.appendLogMessage('Thread : . . . aborted geometry creation', MsgType.Error)
            self.appendLogMessage(f'Thread : . . . {self.worker.survey.errorText}', MsgType.Error)
            QMessageBox.information(self, 'Interrupted', 'Worker thread aborted')
        else:

            # copy analysis arrays from worker
            self.recGeom = self.worker.survey.output.recGeom.copy()             # create a copy; not a view of the arrays
            self.relGeom = self.worker.survey.output.relGeom.copy()
            self.srcGeom = self.worker.survey.output.srcGeom.copy()

            # QMessageBox.information(self, 'Done', f'Worker thread completed. No crash!!!')
            # self.updateMenuStatus(False)                                            # keep menu status in sync with program's state; don't reset analysis figure
            # self.enableProcessingMenuItems()                                        # enable processing menu items (again)
            # self.mainTabWidget.setCurrentIndex(2)                                   # make sure we display the 'Geometry' tab
            # self.hideStatusbarWidgets()                                             # remove temporary widgets from statusbar (don't kill 'm)
            # return

            self.recModel.setData(self.recGeom)                                 # update the three rec/rel/src models
            self.relModel.setData(self.relGeom)
            self.srcModel.setData(self.srcGeom)

            self.recCoordE, self.recCoordN, self.recCoordI = getRecGeometry(self.recGeom, connect=False)
            self.srcCoordE, self.srcCoordN, self.srcCoordI = getSrcGeometry(self.srcGeom, connect=False)

            endTime = timer()
            elapsed = timedelta(seconds=endTime - self.startTime)               # get the elapsed time for geometry creation
            elapsed = timedelta(seconds=ceil(elapsed.total_seconds()))          # round up to nearest second

            self.appendLogMessage(f"Thread : completed 'Create Geometry from Templates'. Elapsed time:{elapsed} ", MsgType.Geometry)

            info = ''
            if not self.fileName:
                # we created an analysis for a yet to be saved file; set modified True, to force saving results when name has been defined
                self.textEdit.document().setModified(True)
                info = 'Analysis results are yet to be saved.'
            else:
                # Save the analysis results
                np.save(self.fileName + '.rec.npy', self.recGeom)
                np.save(self.fileName + '.rel.npy', self.relGeom)
                np.save(self.fileName + '.src.npy', self.srcGeom)
                info = 'Analysis results have been saved.'
            QMessageBox.information(self, 'Done', f'Worker thread completed. {info} ')

        self.updateMenuStatus(False)                                            # keep menu status in sync with program's state; don't reset analysis figure
        self.enableProcessingMenuItems()                                        # enable processing menu items (again)
        self.mainTabWidget.setCurrentIndex(2)                                   # make sure we display the 'Geometry' tab
        self.hideStatusbarWidgets()                                             # remove temporary widgets from statusbar (don't kill 'm)

    def showStatusbarWidgets(self):
        self.progressBar.setValue(0)                                            # first reset to zero, to avoid future glitches in progress shown
        self.statusbar.addWidget(self.progressBar)                              # add temporary widget to statusbar (again)
        self.progressBar.show()                                                 # forces showing progressbar (again)
        self.statusbar.addWidget(self.progressLabel)                            # add temporary widget to statusbar (again)
        self.progressLabel.show()                                               # forces showing progressLabel (again)

    def hideStatusbarWidgets(self):
        self.statusbar.removeWidget(self.progressBar)                           # remove widget from statusbar (don't kill it)
        self.progressBar.setValue(0)                                            # reset progressbar to zero, when out of sight
        self.statusbar.removeWidget(self.progressLabel)                         # remove progress label as well


# See: https://forum.qt.io/topic/84824/accessing-the-main-window-from-python/8
def findMainWindow() -> typing.Union[RollMainWindow, None]:
    # Global function to find the (open) RollMainWindow in application
    app = QApplication.instance()
    for widget in app.topLevelWidgets():
        if isinstance(widget, RollMainWindow):
            return widget
    return None


def main():
    sys.path.append(PARENT_DIR)

    app = QApplication(sys.argv)
    window = RollMainWindow()
    window.show()
    app.exec_()


if __name__ == '__main__':
    # attempt to run roll outside of QGIS
    # does not work yet: "Importerror attempted relative import with no known parent package"
    PARENT_DIR = os.path.dirname(os.path.dirname(os.path.abspath(__file__)))
    sys.path.append(PARENT_DIR)
    os.chdir(PARENT_DIR)

    print('package ' + __package__)
    main()
