"""
***************************************************************************
*                                                                         *
*   This program is free software; you can redistribute it and/or modify  *
*   it under the terms of the GNU General Public License as published by  *
*   the Free Software Foundation; either version 2 of the License, or     *
*   (at your option) any later version.                                   *
*                                                                         *
***************************************************************************
"""
from qgis.PyQt.QtCore import QCoreApplication
from qgis.core import (
    QgsProcessingAlgorithm,
    QgsProcessingParameterEnum,
    QgsProcessingParameterString,
    QgsProcessingParameterFeatureSource,
    QgsProcessingParameterFileDestination,
    QgsProcessing,
    QgsProcessingException
)
import osmnx as ox
import numpy as np
import matplotlib.pyplot as plt
import geopandas as gpd
import math
from shapely.geometry import MultiLineString

class RoadNetworkOrientationOSMnxs(QgsProcessingAlgorithm):
    """
    Uses either OSMnx place name(s) or a QGIS line/polygon layer to compute and plot
    network subplots and orientation rose diagrams with Shannon entropy.

    Inputs:
        - INPUT_TYPE: 'OSMnx Place(s)' or 'QGIS Layer'
        - PLACE_NAME: semicolon-separated place queries (for OSMnx)
        - LAYER_INPUT: vector line or polygon layer (for QGIS mode)
    Outputs:
        - OUTPUT_NETWORK: PNG of network plots
        - OUTPUT_ROSE: PNG of orientation rose diagrams
    """
    INPUT_TYPE = 'INPUT_TYPE'
    PLACE_NAME = 'PLACE_NAME'
    LAYER_INPUT = 'LAYER_INPUT'
    OUTPUT_NETWORK = 'OUTPUT_NETWORK'
    OUTPUT_ROSE = 'OUTPUT_ROSE'

    def tr(self, text):
        return QCoreApplication.translate('RoadNetworkOrientationOSMnxs', text)

    def createInstance(self):
        return RoadNetworkOrientationOSMnxs()

    def name(self):
        return 'roadnetwork_orientation_osmnxs'

    def displayName(self):
        return self.tr('Road Network Orientation using OSMnx (Rose Diagram)')

    def shortHelpString(self):
        return self.tr(
            'Computes and plots street network and orientation rose with entropy, '
            'using OSMnx or a QGIS line/polygon layer.\n'
            'For OSMnx mode, place names can be sourced via Nominatim UI: https://nominatim.openstreetmap.org/ui/.\n'
            'This algorithm was developed during the <b>PT. SAGAMARTHA ULTIMA</b> SkillShare in 2025 and created by <b>FIRMAN AFRIANTO</b>.'
        )

    def initAlgorithm(self, config=None):
        self.addParameter(
            QgsProcessingParameterEnum(
                self.INPUT_TYPE,
                self.tr('Select input type'),
                options=['OSMnx Place(s)', 'QGIS Layer'],
                defaultValue=0
            )
        )
        self.addParameter(
            QgsProcessingParameterString(
                self.PLACE_NAME,
                self.tr('Place Name(s) (semicolon-separated)'),
                defaultValue='',
                optional=True
            )
        )
        self.addParameter(
            QgsProcessingParameterFeatureSource(
                self.LAYER_INPUT,
                self.tr('Road Network Layer'),
                [QgsProcessing.TypeVectorLine, QgsProcessing.TypeVectorPolygon],
                optional=True
            )
        )
        self.addParameter(
            QgsProcessingParameterFileDestination(
                self.OUTPUT_NETWORK,
                self.tr('Network Visualization (PNG)'),
                fileFilter='PNG Files (*.png)'
            )
        )
        self.addParameter(
            QgsProcessingParameterFileDestination(
                self.OUTPUT_ROSE,
                self.tr('Orientation Rose Diagram (PNG)'),
                fileFilter='PNG Files (*.png)'
            )
        )

    def processAlgorithm(self, parameters, context, feedback):
        mode = self.parameterAsEnum(parameters, self.INPUT_TYPE, context)
        output_net = self.parameterAsString(parameters, self.OUTPUT_NETWORK, context)
        output_rose = self.parameterAsString(parameters, self.OUTPUT_ROSE, context)

        # OSMnx mode
        if mode == 0:
            place_input = self.parameterAsString(parameters, self.PLACE_NAME, context)
            if not place_input:
                raise QgsProcessingException(self.tr('Place name(s) cannot be empty.'))
            places = [p.strip() for p in place_input.split(';') if p.strip()]
            n = len(places)
            feedback.pushInfo(self.tr(f'Processing {n} place(s) via OSMnx'))

            # Network subplots
            ncols = int(np.ceil(np.sqrt(n)))
            nrows = int(np.ceil(n / ncols))
            fig_net, axes_net = plt.subplots(nrows, ncols, figsize=(ncols*5, nrows*5))
            axes_net = np.atleast_2d(axes_net)
            for ax, place in zip(axes_net.flat, places):
                feedback.pushInfo(self.tr(f'Plotting network: {place}'))
                G = ox.graph_from_place(place, network_type='drive')
                Gu = ox.bearing.add_edge_bearings(ox.convert.to_undirected(G))
                edges = ox.graph_to_gdfs(Gu, nodes=False, edges=True)
                edges.plot(ax=ax, linewidth=0.8, color='#333333')
                ax.set_title(place, fontsize=14, fontweight='bold')
                ax.set_aspect('equal', adjustable='box')
                ax.axis('off')
            for ax in axes_net.flat[n:]:
                ax.set_visible(False)
            fig_net.tight_layout()
            fig_net.savefig(output_net, dpi=300, bbox_inches='tight')
            plt.close(fig_net)
            feedback.pushInfo(self.tr(f'Network plot saved to {output_net}'))

            # Orientation rose
            fig_rose, axes_rose = plt.subplots(
                nrows, ncols,
                figsize=(ncols*5, nrows*5),
                subplot_kw={'projection': 'polar'}
            )
            axes_rose = np.atleast_2d(axes_rose)
            for ax, place in zip(axes_rose.flat, places):
                feedback.pushInfo(self.tr(f'Plotting orientation rose: {place}'))
                G = ox.graph_from_place(place, network_type='drive')
                Gu = ox.bearing.add_edge_bearings(ox.convert.to_undirected(G))
                entropy = ox.bearing.orientation_entropy(Gu)
                ox.plot.plot_orientation(Gu, ax=ax, title='', area=True)
                ax.set_title(place, fontsize=16, fontweight='bold', pad=30)
                ax.text(0.5, 1.06, f'Entropy = {entropy:.2f}', transform=ax.transAxes,
                        ha='center', va='bottom', fontsize=10)
            for ax in axes_rose.flat[n:]:
                ax.set_visible(False)
            fig_rose.suptitle('Street Network Orientation', fontsize=14, y=1.02)
            fig_rose.tight_layout()
            fig_rose.subplots_adjust(hspace=0.35)
            fig_rose.savefig(output_rose, dpi=300, bbox_inches='tight')
            plt.close(fig_rose)
            feedback.pushInfo(self.tr(f'Orientation rose saved to {output_rose}'))

        # QGIS Layer mode
        else:
            layer = self.parameterAsVectorLayer(parameters, self.LAYER_INPUT, context)
            if layer is None:
                raise QgsProcessingException(self.tr('Road network layer is required.'))
            if layer.featureCount() == 0:
                raise QgsProcessingException(self.tr('Input layer contains no features.'))
            feedback.pushInfo(self.tr('Reading features from QGIS layer'))

            path = layer.source()
            edges = gpd.read_file(path)

            # Plot network
            fig_net, ax_net = plt.subplots(figsize=(10, 10))
            for geom in edges.geometry:
                if not geom.geom_type.endswith('LineString'):
                    boundary = geom.boundary
                    if isinstance(boundary, MultiLineString):
                        segments = list(boundary.geoms)
                    else:
                        segments = [boundary]
                elif isinstance(geom, MultiLineString):
                    segments = list(geom.geoms)
                else:
                    segments = [geom]
                for line in segments:
                    x, y = line.xy
                    ax_net.plot(x, y, linewidth=0.8, color='#333333')
            ax_net.set_aspect('equal', adjustable='box')
            xmin, xmax = ax_net.get_xlim()
            ymin, ymax = ax_net.get_ylim()
            dx = (xmax - xmin) * 0.02
            dy = (ymax - ymin) * 0.02
            ax_net.set_xlim(xmin - dx, xmax + dx)
            ax_net.set_ylim(ymin - dy, ymax + dy)
            ax_net.set_title('Road Network: QGIS Layer')
            ax_net.axis('off')
            fig_net.tight_layout()
            fig_net.savefig(output_net, dpi=300, bbox_inches='tight')
            plt.close(fig_net)
            feedback.pushInfo(self.tr(f'Network visualization saved to {output_net}'))

            # Compute bearings
            def _bearing(p1, p2):
                dx, dy = p2[0] - p1[0], p2[1] - p1[1]
                ang = math.degrees(math.atan2(dx, dy))
                return (ang + 360) % 360

            bearings = []
            for geom in edges.geometry:
                if not geom.geom_type.endswith('LineString'):
                    boundary = geom.boundary
                    if isinstance(boundary, MultiLineString):
                        segments = list(boundary.geoms)
                    else:
                        segments = [boundary]
                elif isinstance(geom, MultiLineString):
                    segments = list(geom.geoms)
                else:
                    segments = [geom]
                for line in segments:
                    coords = list(line.coords)
                    for i in range(len(coords) - 1):
                        bearings.append(_bearing(coords[i], coords[i+1]))

            # ————— Modifikasi untuk symmetric rose —————
            thetas = np.deg2rad(bearings)
            # ubah jadi [0, π)
            orientations = np.mod(thetas, np.pi)
            n_bins = 36
            bins = np.linspace(0, np.pi, n_bins//2 + 1)
            counts, _ = np.histogram(orientations, bins=bins)
            angles = bins[:-1] + np.diff(bins)/2
            width = np.diff(bins)[0]

            fig_rose, axr = plt.subplots(subplot_kw={'projection': 'polar'})
            for angle, count in zip(angles, counts):
                axr.bar(angle, count, width=width, align='center', edgecolor='k', facecolor='C0')
                axr.bar(angle + np.pi, count, width=width, align='center', edgecolor='k', facecolor='C0')
            axr.set_theta_zero_location('N')
            axr.set_theta_direction(-1)
            probs = counts / counts.sum()
            entropy = -(probs * np.log2(probs + 1e-9)).sum()
            axr.text(0.5, 1.05, f'Entropy = {entropy:.2f}',
                     transform=axr.transAxes, ha='center', va='bottom', fontsize=10)
            fig_rose.set_size_inches(8, 8)
            fig_rose.tight_layout()
            fig_rose.savefig(output_rose, dpi=300, bbox_inches='tight')
            plt.close(fig_rose)
            feedback.pushInfo(self.tr(f'Orientation rose saved to {output_rose}'))

        return {self.OUTPUT_NETWORK: output_net, self.OUTPUT_ROSE: output_rose}
