# -*- coding: utf-8 -*-

"""
/***************************************************************************
 RoadSlopeCalculator
                                 A QGIS plugin
 This algorithm is used to calculate the longitudinal slope of forest paths and roads, based on a 2D line vector layer and a DEM (Digital Elevation Model). 
It should not be used on national road systems, such as highways, motorways, national and regional roads, which have engineering works designed to smooth the slope (bridges, excavations, landfills, etc.), unless the user has access to a high precision Digital Surface Model (DSM).
 Generated by Plugin Builder: http://g-sherman.github.io/Qgis-Plugin-Builder/
                              -------------------
        begin                : 2021-03-09
        copyright            : (C) 2021 by Antonio Sobral Almeida
        email                : 66124.almeida@gmail.com
 ***************************************************************************/

/***************************************************************************
 *                                                                         *
 *   This program is free software; you can redistribute it and/or modify  *
 *   it under the terms of the GNU General Public License as published by  *
 *   the Free Software Foundation; either version 2 of the License, or     *
 *   (at your option) any later version.                                   *
 *                                                                         *
 ***************************************************************************/
"""

from qgis.core import QgsProcessing
from qgis.core import QgsProcessingAlgorithm
from qgis.core import QgsProcessingMultiStepFeedback
from qgis.core import QgsProcessingParameterRasterLayer
from qgis.core import QgsProcessingParameterFeatureSink
from qgis.core import QgsProcessingParameterBoolean
from qgis.core import QgsProcessingParameterVectorLayer
from qgis.core import QgsProcessingParameterNumber
import processing


class RoadSlopeCalculatorAlgorithm(QgsProcessingAlgorithm):

    def initAlgorithm(self, config=None):
        self.addParameter(QgsProcessingParameterRasterLayer('MDT', 'Digital Elevation Model (DEM)', defaultValue=None))
        self.addParameter(QgsProcessingParameterFeatureSink('CalculatedRoadSlopes', 'Calculated road slopes', type=QgsProcessing.TypeVectorAnyGeometry, createByDefault=True, supportsAppend=True, defaultValue=None))
        self.addParameter(QgsProcessingParameterBoolean('VERBOSE_LOG', 'Verbose logging', optional=True, defaultValue=False))
        self.addParameter(QgsProcessingParameterVectorLayer('Shapedecaminhos', 'Road network vector layer', types=[QgsProcessing.TypeVectorLine], defaultValue=None))
        self.addParameter(QgsProcessingParameterNumber('Comprimentodossegmentos', 'Segment length (=> 5 times pixel size)', type=QgsProcessingParameterNumber.Double, maxValue=1.79769e+308, defaultValue=250))

    def processAlgorithm(self, parameters, context, model_feedback):
        # Use a multi-step feedback, so that individual child algorithm progress reports are adjusted for the
        # overall progress through the model
        feedback = QgsProcessingMultiStepFeedback(5, model_feedback)
        results = {}
        outputs = {}

        # Split lines by maximum length
        alg_params = {
            'INPUT': parameters['Shapedecaminhos'],
            'LENGTH': parameters['Comprimentodossegmentos'],
            'OUTPUT': QgsProcessing.TEMPORARY_OUTPUT
        }
        outputs['SplitLinesByMaximumLength'] = processing.run('native:splitlinesbylength', alg_params, context=context, feedback=feedback, is_child_algorithm=True)

        feedback.setCurrentStep(1)
        if feedback.isCanceled():
            return {}

        # Drape (set Z value from raster)
        alg_params = {
            'BAND': 1,
            'INPUT': outputs['SplitLinesByMaximumLength']['OUTPUT'],
            'NODATA': -9999,
            'RASTER': parameters['MDT'],
            'SCALE': 1,
            'OUTPUT': QgsProcessing.TEMPORARY_OUTPUT
        }
        outputs['DrapeSetZValueFromRaster'] = processing.run('native:setzfromraster', alg_params, context=context, feedback=feedback, is_child_algorithm=True)

        feedback.setCurrentStep(2)
        if feedback.isCanceled():
            return {}

        # Extract Z values
        alg_params = {
            'COLUMN_PREFIX': 'z_',
            'INPUT': outputs['DrapeSetZValueFromRaster']['OUTPUT'],
            'SUMMARIES': [0,1],
            'OUTPUT': QgsProcessing.TEMPORARY_OUTPUT
        }
        outputs['ExtractZValues'] = processing.run('native:extractzvalues', alg_params, context=context, feedback=feedback, is_child_algorithm=True)

        feedback.setCurrentStep(3)
        if feedback.isCanceled():
            return {}

        # Field calculator1
        alg_params = {
            'FIELD_LENGTH': 10,
            'FIELD_NAME': 'Length',
            'FIELD_PRECISION': 2,
            'FIELD_TYPE': 0,
            'FORMULA': ' $length ',
            'INPUT': outputs['ExtractZValues']['OUTPUT'],
            'OUTPUT': QgsProcessing.TEMPORARY_OUTPUT
        }
        outputs['FieldCalculator1'] = processing.run('qgis:fieldcalculator', alg_params, context=context, feedback=feedback, is_child_algorithm=True)

        feedback.setCurrentStep(4)
        if feedback.isCanceled():
            return {}

        # Field calculator2
        alg_params = {
            'FIELD_LENGTH': 10,
            'FIELD_NAME': 'Slope_%',
            'FIELD_PRECISION': 3,
            'FIELD_TYPE': 0,
            'FORMULA': '(abs(\"z_first\" - \"z_last\") / \"length\" ) *100',
            'INPUT': outputs['FieldCalculator1']['OUTPUT'],
            'OUTPUT': parameters['CalculatedRoadSlopes']
        }
        outputs['FieldCalculator2'] = processing.run('qgis:fieldcalculator', alg_params, context=context, feedback=feedback, is_child_algorithm=True)
        results['CalculatedRoadSlopes'] = outputs['FieldCalculator2']['OUTPUT']
        return results

    def name(self):
        return 'Road slope calculator'

    def displayName(self):
        return 'Road slope calculator'

    def shortHelpString(self):
        return """<html><body><h2>Algorithm description</h2>
<p>This algorithm is used to calculate the longitudinal slope of forest paths and roads, based on a 2D line vector layer and a DEM (Digital Elevation Model). 
It should not be used on national road systems, such as highways, motorways, national and regional roads, which have engineering works designed to smooth the slope (bridges, excavations, landfills, etc.), unless the user has access to a high precision Digital Surface Model (DSM).
 
Input parameters: 
1 - Segment length: to calculate the slope, the initial vector layer will be segmented, and the value entered here determines the length of these segments. It is advisable that the length of the segments is equal or greater than 5 times the pixel size of the DEM. Slopes of segments whose lengths are less than 3 times the pixel size, should be considered with caution, or even discarded. Default value is 250 map units, but can accept any other. 
2 - Digital Elevation Model (DEM): any elevation raster, with elevation values in same units as road network vector layer lengths. 
3 - Road network vector layer: any 2D line layer; please note that this layer will be segmented, so it is recommended that the length of the features in this layer be much longer than the segment length. Make sure this vector layer is completely within the DEM data area; otherwise, slope values of segments outside the DEM data area will be meaningless.
 
Outputs: a new line vector layer, segmented, with two new fields: 
"Length", with the current length of the segmentation (note that there will always be residual segments, of less than the chosen length); 
"Slope_%", with the longitudinal slope of each segment, in percentage. </p>
<h2>Input parameters</h2>
<h3>Digital Elevation Model (DEM)</h3>
<p></p>
<h3>Calculated road slopes</h3>
<p></p>
<h3>Verbose logging</h3>
<p></p>
<h3>Road network vector layer</h3>
<p></p>
<h3>Segment length (=> 5 times pixel size)</h3>
<p></p>
<h2>Outputs</h2>
<h3>Calculated road slopes</h3>
<p></p>
<br><p align="right">Algorithm author: Sobral Almeida</p><p align="right">Help author: Sobral Almeida</p><p align="right">Algorithm version: 1.0</p></body></html>"""

    def createInstance(self):
        return RoadSlopeCalculatorAlgorithm()
