"""
/***************************************************************************
 RMCGeo
                                 A QGIS plugin
 Conjunto de ferramentas para simplificar tarefas geoespaciais.
                             -------------------
        begin                : 2025-01-10
        copyright            : (C) 2025 by Rodolfo Martins de Carvalho
        email                : rodolfomartins09@gmail.com
        git sha              : $Format:%H$
 ***************************************************************************/

/***************************************************************************
 *                                                                         *
 *   This program is free software; you can redistribute it and/or modify  *
 *   it under the terms of the GNU General Public License as published by  *
 *   the Free Software Foundation; either version 2 of the License, or     *
 *   (at your option) any later version.                                   *
 *                                                                         *
 ***************************************************************************/
"""

from qgis.core import QgsPointXY, QgsGeometry, QgsFeature, QgsVectorLayer, QgsWkbTypes
from qgis.gui import QgsMapTool
from PyQt5.QtWidgets import QDialog, QMessageBox
from qgis.PyQt import uic
from qgis.utils import iface
from PyQt5.QtGui import QPixmap
from PyQt5.QtCore import Qt
import os

# Carrega o arquivo .ui
FORM_CLASS, _ = uic.loadUiType(os.path.join(
    os.path.dirname(__file__), '..', 'ui', 'point_insert.ui'))

class PointInsert(QgsMapTool):
    def __init__(self, canvas, dialog):
        super().__init__(canvas)
        self.canvas = canvas
        self.dialog = dialog
        self.last_map_point = None

    def canvasPressEvent(self, event):
        # Verifica se foi clique com botão direito
        if event.button() == 2:  # 2 é o botão direito do mouse
            self.canvas.unsetMapTool(self)
            return
            
        pos = event.pos()
        self.last_map_point = self.toMapCoordinates(pos)        
        # Verifica se há uma camada ativa e se está em modo de edição
        layer = iface.activeLayer()
        if not layer:
            QMessageBox.warning(self.dialog, "Aviso", "Nenhuma camada selecionada!")
            return
            
        if not layer.isEditable():
            QMessageBox.warning(self.dialog, "Aviso", "A camada precisa estar em modo de edição!")
            return
            
        # Verifica se a camada é do tipo ponto ou multiponto
        geometry_type = layer.geometryType()
        if geometry_type not in [QgsWkbTypes.PointGeometry, QgsWkbTypes.MultiPoint]:
            QMessageBox.warning(self.dialog, "Aviso", "A camada deve ser do tipo ponto ou multiponto!")
            return
            
        self.show_coordinate_input(pos, self.last_map_point)

    def show_coordinate_input(self, pos, map_point):
        if self.dialog.isVisible():
            self.dialog.close()

        self.dialog.eastInput.setPlaceholderText(f"Este (X): {map_point.x():.2f}")
        self.dialog.northInput.setPlaceholderText(f"Norte (Y): {map_point.y():.2f}")
        screen_pos = self.canvas.mapToGlobal(pos)
        self.dialog.move(screen_pos)
        self.dialog.show()
        self.dialog.eastInput.setFocus()

    def add_point(self, east, north):
        print("DEBUG: add_point chamado")
        print(f"DEBUG: Coordenadas recebidas - Este: {east}, Norte: {north}")
        
        try:
            x = float(east) if east else self.last_map_point.x()
            y = float(north) if north else self.last_map_point.y()
        except ValueError as e:
            iface.messageBar().pushMessage("Erro", "Coordenadas inválidas!", level=1)
            return

        layer = iface.activeLayer()
        if not layer:
            iface.messageBar().pushMessage("Erro", "Nenhuma camada selecionada!", level=1)
            return
            
        if not isinstance(layer, QgsVectorLayer):
            iface.messageBar().pushMessage("Erro", "A camada selecionada não é uma camada vetorial!", level=1)
            return
            
        if not layer.isEditable():
            iface.messageBar().pushMessage("Erro", "A camada precisa estar em modo de edição!", level=1)
            return

        # Verifica se a camada é do tipo ponto ou multiponto
        geometry_type = layer.geometryType()
        if geometry_type not in [QgsWkbTypes.PointGeometry, QgsWkbTypes.MultiPoint]:
            iface.messageBar().pushMessage("Erro", "A camada deve ser do tipo ponto ou multiponto!", level=1)
            return        
        try:
            # Cria o ponto
            point = QgsPointXY(x, y)
            
            # Cria a geometria
            geom = QgsGeometry.fromPointXY(point)
            
            # Cria a feature
            feature = QgsFeature(layer.fields())
            feature.setGeometry(geom)

            # Adiciona o ponto à camada
            success = layer.addFeature(feature)
            
            if success:
                # Força a atualização do estilo
                layer.triggerRepaint()
                # Atualiza o canvas
                self.canvas.refresh()
            else:
                iface.messageBar().pushMessage("Erro", "Falha ao adicionar o ponto à camada!", level=1)
                
        except Exception as e:
            iface.messageBar().pushMessage("Erro", f"Erro ao criar ponto: {str(e)}", level=1)

    def deactivate(self):
        if self.dialog.isVisible():
            self.dialog.close()
        super().deactivate()

class PointInsertDialog(QDialog, FORM_CLASS):
    def __init__(self, iface):
        super().__init__()
        self.setupUi(self)
        self.iface = iface
        self.canvas = iface.mapCanvas()
        self.tool = None

        # Carrega o ícone SVG
        icon_path = os.path.join(os.path.dirname(__file__), '..', 'icon.svg')
        if os.path.exists(icon_path):
            pixmap = QPixmap(icon_path)
            self.label_3.setPixmap(pixmap.scaled(24, 24, Qt.KeepAspectRatio, Qt.SmoothTransformation))

        # Conecta os botões
        self.confirmButton.clicked.connect(self.on_confirm)
        self.closeButton.clicked.connect(self.close)

    def run(self):
        # Verifica se há uma camada ativa e se é do tipo ponto ou multiponto
        layer = self.iface.activeLayer()
        if not layer:
            QMessageBox.warning(self, "Aviso", "Selecione uma camada de pontos ou multipontos!")
            return
            
        geometry_type = layer.geometryType()
        if geometry_type not in [QgsWkbTypes.PointGeometry, QgsWkbTypes.MultiPoint]:
            QMessageBox.warning(self, "Aviso", "Selecione uma camada de pontos ou multipontos!")
            return

        # Se a ferramenta já existe, apenas reativa
        if self.tool:
            self.canvas.setMapTool(self.tool)
        else:
            self.activate_tool()

    def activate_tool(self):
        if not self.tool:
            self.tool = PointInsert(self.canvas, self)
        self.canvas.setMapTool(self.tool)

    def on_confirm(self):
        if self.tool:
            east = self.eastInput.text()
            north = self.northInput.text()
            self.tool.add_point(east, north)
            self.close()

    def unload(self):
        if self.tool:
            self.canvas.unsetMapTool(self.tool)
            self.tool = None

def run(iface):
    tool = PointInsertDialog(iface=iface)
    tool.run()

def unload():
    pass