"""
/***************************************************************************
 RMCGeo
                                 A QGIS plugin
 Conjunto de ferramentas para simplificar tarefas geoespaciais.
                             -------------------
        begin                : 2025-01-10
        copyright            : (C) 2025 by Rodolfo Martins de Carvalho
        email                : rodolfomartins09@gmail.com
        git sha              : $Format:%H$
 ***************************************************************************/

/***************************************************************************
 *                                                                         *
 *   This program is free software; you can redistribute it and/or modify  *
 *   it under the terms of the GNU General Public License as published by  *
 *   the Free Software Foundation; either version 2 of the License, or     *
 *   (at your option) any later version.                                   *
 *                                                                         *
 ***************************************************************************/
"""

import os
import math
from qgis.PyQt import uic
from qgis.PyQt import QtWidgets
from qgis.PyQt.QtWidgets import QApplication
from qgis.PyQt.QtGui import QIcon
from qgis.PyQt.QtCore import QSize

FORM_CLASS, _ = uic.loadUiType(os.path.join(
    os.path.dirname(__file__), '..', 'ui', 'gms_to_decimal.ui'))


class GmsToDecimal(QtWidgets.QDialog, FORM_CLASS):
    def __init__(self, parent=None, iface=None):
        """Constructor."""
        super(GmsToDecimal, self).__init__(parent)
        self.iface = iface
        self.setupUi(self)

        self.clipboard = QApplication.clipboard()

        # Configura o campo de saída decimal como somente leitura
        self.saida_decimal.setReadOnly(True)
        
        # Carrega o ícone SVG no canto inferior esquerdo
        self.carregar_logo()

        # Conexões dos sinais
        self.buttonconvertg.clicked.connect(self.convertparadecimal)  # Botão para DMS -> Decimal
        self.buttonconvertd.clicked.connect(self.convertparagraus)  # Botão para Decimal -> DMS
        self.copyqline.clicked.connect(self.copy)  # Copiar Grau Decimal
        self.copyqline2.clicked.connect(self.copy_dms)  # Copiar DMS
        
        # Ativa validação em tempo real
        self.add_grau.textEdited.connect(self.digitagraus)  # Validação em tempo real para graus
        self.add_minuto.textEdited.connect(self.digitagraus)  # Validação em tempo real para minutos
        self.add_segundo.textEdited.connect(self.digitagraus)  # Validação em tempo real para segundos
        self.entrada_decimal.textEdited.connect(self.validar_decimal)  # Validação para grau decimal
        
        # Configuração dos ícones
        self.copyqline.setIcon(QIcon(':/images/themes/default/mActionEditCopy.svg'))
        self.copyqline2.setIcon(QIcon(':/images/themes/default/mActionEditCopy.svg'))

    def carregar_logo(self):
        """Carrega o ícone SVG do plugin no canto inferior esquerdo."""
        # Caminho para o ícone SVG
        svg_path = os.path.join(os.path.dirname(os.path.dirname(__file__)), 'icon.svg')
            
        # Verifica se o arquivo existe
        if os.path.exists(svg_path):
            # Cria um QIcon a partir do arquivo SVG
            icon = QIcon(svg_path)
            # Converte o QIcon para QPixmap com o tamanho desejado
            pixmap = icon.pixmap(QSize(32, 32))
            # Define o pixmap no QLabel
            self.logoLabel.setPixmap(pixmap)

    def validate_input(self, sender, text, min_val, max_val, allow_float=False):
        """Validação de entrada com suporte para números inteiros e float."""
        if not text:  # Permite campo vazio
            return True
            
        try:
            if allow_float:
                valor = float(text)
            else:
                valor = int(text)
                
            return min_val <= valor <= max_val
        except ValueError:
            return False

    def digitagraus(self, text):
        """Validação em tempo real para entradas de graus, minutos e segundos."""
        try:
            sender = self.sender()
            valid = True
            
            if sender == self.add_grau:
                valid = self.validate_input(sender, text, -180, 180)
            elif sender == self.add_minuto:
                valid = self.validate_input(sender, text, 0, 59)
            elif sender == self.add_segundo:
                valid = self.validate_input(sender, text, 0, 59, allow_float=True)
                
            # Aplica estilo visual para indicar erro
            if not valid and text:
                sender.setStyleSheet("background-color: #ffcccc;")
            else:
                sender.setStyleSheet("")
                
        except Exception as e:
            print(f"Erro em digitagraus: {e}")

    def validar_decimal(self, text):
        """Validação em tempo real para entrada de grau decimal."""
        try:
            sender = self.sender()
            valid = True
            
            if text:
                try:
                    valor = float(text)
                    valid = -180 <= valor <= 180
                except ValueError:
                    valid = False
            
            # Aplica estilo visual para indicar erro
            if not valid and text:
                sender.setStyleSheet("background-color: #ffcccc;")
                # Mostra mensagem de ajuda no status bar
                if self.iface and float(text) < -180 or float(text) > 180:
                    self.iface.statusBarIface().showMessage("Valor deve estar entre -180 e 180 graus", 3000)
            else:
                sender.setStyleSheet("")
                
        except Exception as e:
            print(f"Erro na validação decimal: {e}")

    def convertparadecimal(self):
        """Converte Graus, Minutos e Segundos para Graus Decimais."""
        try:
            # Obtém os valores dos campos
            grau_text = self.add_grau.text().strip()
            minuto_text = self.add_minuto.text().strip()
            segundo_text = self.add_segundo.text().strip()
            
            # Verifica se pelo menos o campo de graus foi preenchido
            if not grau_text:
                self.saida_decimal.setText("Informe ao menos o valor em graus")
                return
                
            # Conversão dos valores para números
            graus = int(grau_text)
            minutos = int(minuto_text) if minuto_text else 0
            segundos = float(segundo_text) if segundo_text else 0.0

            # Validação de intervalos
            if graus < -180 or graus > 180:
                self.saida_decimal.setText("Graus devem estar entre -180 e 180")
                return
            if minutos < 0 or minutos > 59:
                self.saida_decimal.setText("Minutos devem estar entre 0 e 59")
                return
            if segundos < 0 or segundos > 59.999:
                self.saida_decimal.setText("Segundos devem estar entre 0 e 59.999")
                return

            # Determina o sinal para cálculo correto
            sinal = -1 if graus < 0 else 1
            graus_abs = abs(graus)
            
            # Conversão para graus decimais mantendo o sinal
            decimal = sinal * (graus_abs + (minutos / 60) + (segundos / 3600))

            # Exibe o resultado com 6 casas decimais
            self.saida_decimal.setText(f"{decimal:.6f}")

        except ValueError as e:
            self.saida_decimal.setText("Erro de formato numérico")
            print(f"Erro na conversão para decimal: {e}")
        except Exception as e:
            self.saida_decimal.setText("Erro na conversão")
            print(f"Erro inesperado: {e}")

    def convertparagraus(self):
        """Converte Graus Decimais para Graus, Minutos e Segundos."""
        try:
            decimal_text = self.entrada_decimal.text().strip()
            
            if not decimal_text:
                self.saida_grau.setText("")
                self.saida_minuto.setText("")
                self.saida_segundo.setText("")
                return
                
            decimal = float(decimal_text)
            
            # Verifica se o valor está dentro do intervalo válido
            if decimal < -180 or decimal > 180:
                self.saida_grau.setText("Valor inválido")
                self.saida_minuto.setText("")
                self.saida_segundo.setText("")
                if self.iface:
                    self.iface.statusBarIface().showMessage("Valor deve estar entre -180 e 180 graus", 3000)
                return
            
            # Tratamento do sinal
            sinal = -1 if decimal < 0 else 1
            decimal_abs = abs(decimal)
            
            # Cálculos para conversão
            graus_abs = math.floor(decimal_abs)
            minutos_float = (decimal_abs - graus_abs) * 60
            minutos = math.floor(minutos_float)
            segundos = (minutos_float - minutos) * 60
            
            # Correção para arredondamentos
            if round(segundos, 2) >= 60:
                segundos = 0
                minutos += 1
                if minutos >= 60:
                    minutos = 0
                    graus_abs += 1
            
            # Aplica o sinal apenas aos graus
            graus = sinal * graus_abs
            
            # Exibe os resultados
            self.saida_grau.setText(str(graus))
            self.saida_minuto.setText(str(minutos))
            self.saida_segundo.setText(f"{segundos:.2f}")
            
        except ValueError:
            self.saida_grau.setText("Erro")
            self.saida_minuto.setText("Erro")
            self.saida_segundo.setText("Erro")
        except Exception as e:
            print(f"Erro inesperado na conversão para GMS: {e}")

    def copy(self):
        """Copia o valor de graus decimais para a área de transferência."""
        try:
            s = self.saida_decimal.text().strip()
            if not s:
                if self.iface:
                    self.iface.statusBarIface().showMessage("Não há valor para copiar", 3000)
                return
                
            self.clipboard.setText(s)
            if self.iface:
                self.iface.statusBarIface().showMessage(f"'{s}' Copiado para a Área de Transferência", 3000)
            
        except Exception as e:
            if self.iface:
                self.iface.statusBarIface().showMessage(f"Erro ao copiar: {str(e)}", 5000)
            print(f"Erro ao copiar: {e}")

    def copy_dms(self):
        """Copia o valor em graus, minutos e segundos para a área de transferência."""
        try:
            grau = self.saida_grau.text().strip()
            minuto = self.saida_minuto.text().strip()
            segundo = self.saida_segundo.text().strip()
            
            if not grau and not minuto and not segundo:
                if self.iface:
                    self.iface.statusBarIface().showMessage("Não há valor para copiar", 3000)
                return
            
            s = f"{grau}° {minuto}' {segundo}\""
            self.clipboard.setText(s)
            
            if self.iface:
                self.iface.statusBarIface().showMessage(f"'{s}' Copiado para a Área de Transferência", 3000)
                
        except Exception as e:
            if self.iface:
                self.iface.statusBarIface().showMessage(f"Erro ao copiar: {str(e)}", 5000)
            print(f"Erro ao copiar DMS: {e}")

    def reset_fields(self):
        """Redefine todos os campos de entrada e saída para valores padrão."""
        self.add_grau.setText("")
        self.add_minuto.setText("")
        self.add_segundo.setText("")
        self.entrada_decimal.setText("")
        self.saida_decimal.setText("")
        self.saida_grau.setText("")
        self.saida_minuto.setText("")
        self.saida_segundo.setText("")
        
        # Limpa qualquer estilo aplicado
        self.add_grau.setStyleSheet("")
        self.add_minuto.setStyleSheet("")
        self.add_segundo.setStyleSheet("")
        self.entrada_decimal.setStyleSheet("")

def run(iface):
    """Função para executar o diálogo de conversão GMS para decimal."""
    dlg = GmsToDecimal(iface=iface)
    dlg.reset_fields()
    dlg.show()
    dlg.exec_()

def unload():
    """Função para limpar recursos quando o plugin for descarregado."""
    pass