# -*- coding: utf-8 -*-
"""
/***************************************************************************
 RiverBankErosionAccretion_V2
                                 A QGIS plugin
 RiverBank Erosion and Accretion Analyzer v2 is an advanced QGIS plugin designed to quantify riverbank dynamics by analyzing spatial data over multiple time periods. It provides automated workflows for calculating, visualizing, and reporting erosion and accretion zones along a river’s course.
 Generated by Plugin Builder: http://g-sherman.github.io/Qgis-Plugin-Builder/
                              -------------------
        begin                : 2025-10-24
        git sha              : $Format:%H$
        copyright            : (C) 2025 by Srija Roy
        email                : roysrija7@gmail.com
 ***************************************************************************/

/***************************************************************************
 *                                                                         *
 *   This program is free software; you can redistribute it and/or modify  *
 *   it under the terms of the GNU General Public License as published by  *
 *   the Free Software Foundation; either version 2 of the License, or     *
 *   (at your option) any later version.                                   *
 *                                                                         *
 ***************************************************************************/
"""
# -*- coding: utf-8 -*-
# -*- coding: utf-8 -*-
"""
RiverBankErosionAccretion_V2
Advanced QGIS plugin to quantify riverbank erosion and accretion
by analyzing Year-1 and Year-2 waterbody polygons.
"""

import os
import numpy as np
import geopandas as gpd
from shapely.geometry import box, Polygon, MultiPolygon, GeometryCollection

from qgis.PyQt.QtWidgets import (
    QAction, QDialog, QLabel, QLineEdit, QPushButton,
    QGridLayout, QFileDialog, QMessageBox, QComboBox
)
from qgis.PyQt.QtGui import QIcon
from qgis.PyQt.QtCore import Qt

from qgis.core import (
    QgsCoordinateReferenceSystem,
    QgsCoordinateTransform,
    QgsCoordinateTransformContext,
)
from qgis.gui import QgsMapLayerComboBox, QgsProjectionSelectionWidget


PLUGIN_MENU = "&RiverBank Erosion and Accretion Analyzer (v2)"
CELL_SIZE_M = 30.0
TOP_N = 15


class RiverBankErosionAccretion_V2:
    """
    Core logic:
      1) Generate per-year fishnets for Year-1 and Year-2 polygons.
      2) Build a common analysis grid for both years.
      3) Calculate unchanged, erosion, and accretion areas per grid.
      4) Aggregate to user-selected stretch lengths.
      5) Export grid-based, stretch-based, and Top-15 outputs (Shapefile + CSV).
    """

    def __init__(self, iface):
        self.iface = iface
        self.plugin_dir = os.path.dirname(__file__)
        self.action = None
        self.dlg = None
        self.stretch_m = 3000
        self.cell_size_m = CELL_SIZE_M
        self.top_n = TOP_N

    # ---------------- GUI setup ----------------
    def initGui(self):
        icon_path = os.path.join(self.plugin_dir, "icon.png")
        icon = QIcon(icon_path) if os.path.exists(icon_path) else QIcon()
        self.action = QAction(icon, "RiverBank Erosion and Accretion Analyzer (v2)", self.iface.mainWindow())
        self.action.triggered.connect(self.run)
        self.iface.addToolBarIcon(self.action)
        self.iface.addPluginToMenu(PLUGIN_MENU, self.action)

    def unload(self):
        if self.action:
            self.iface.removeToolBarIcon(self.action)
            self.iface.removePluginMenu(PLUGIN_MENU, self.action)

    # ---------------- Dialog builder ----------------
    def build_ui(self):
        self.dlg = QDialog(self.iface.mainWindow())
        self.dlg.setWindowTitle("RiverBank Erosion & Accretion Analyzer (v2)")

        header = QLabel(
            "<b>RiverBank Erosion & Accretion Analyzer v2</b><br>"
            "<small>Select Year-1 and Year-2 polygon layers, choose a projected CRS, "
            "stretch distance, and output folder.</small>"
        )
        header.setAlignment(Qt.AlignCenter)

        lbl_y1 = QLabel("Year 1 Waterbody (Polygon)")
        lbl_y2 = QLabel("Year 2 Waterbody (Polygon)")
        lbl_crs = QLabel("Projected Coordinate System")
        lbl_str = QLabel("Stretch distance")
        lbl_out = QLabel("Output folder")

        self.comboYear1 = QgsMapLayerComboBox()
        self.comboYear2 = QgsMapLayerComboBox()
        for box in (self.comboYear1, self.comboYear2):
            try:
                box.setFilters(box.GeometryType.PolygonLayer)
            except Exception:
                pass

        # CRS selection
        self.projTarget = QgsProjectionSelectionWidget()
        self.projTarget.setToolTip("Select a projected CRS (equal-area preferred).")

        # Stretch options
        self.comboStretch = QComboBox()
        self.comboStretch.addItems(["90 m", "3 km", "6 km", "9 km"])
        self.comboStretch.setToolTip("Stretch aggregation length.")

        self.editOutDir = QLineEdit()
        self.editOutDir.setReadOnly(True)
        self.btnBrowseOut = QPushButton("Browse…")
        self.btnStart = QPushButton("Start Analysis")

        # Layout
        grid = QGridLayout(self.dlg)
        r = 0
        grid.addWidget(header, r, 0, 1, 3); r += 1
        grid.addWidget(lbl_y1, r, 0); grid.addWidget(self.comboYear1, r, 1, 1, 2); r += 1
        grid.addWidget(lbl_y2, r, 0); grid.addWidget(self.comboYear2, r, 1, 1, 2); r += 1
        grid.addWidget(lbl_crs, r, 0); grid.addWidget(self.projTarget, r, 1, 1, 2); r += 1
        grid.addWidget(lbl_str, r, 0); grid.addWidget(self.comboStretch, r, 1); r += 1
        grid.addWidget(lbl_out, r, 0); grid.addWidget(self.editOutDir, r, 1); grid.addWidget(self.btnBrowseOut, r, 2); r += 1
        grid.addWidget(self.btnStart, r, 1)

        self.btnBrowseOut.clicked.connect(self._pick_outdir)
        self.btnStart.clicked.connect(self._start)

    def run(self):
        if self.dlg is None:
            self.build_ui()
        self.dlg.show()

    # ---------------- UI Actions ----------------
    def _pick_outdir(self):
        folder = QFileDialog.getExistingDirectory(self.iface.mainWindow(), "Select Output Folder")
        if folder:
            self.editOutDir.setText(folder)

    def _start(self):
        lyr1 = self.comboYear1.currentLayer()
        lyr2 = self.comboYear2.currentLayer()
        outdir = self.editOutDir.text().strip()
        crs = self.projTarget.crs() if self._is_valid_crs_widget() else QgsCoordinateReferenceSystem()

        if not lyr1 or not lyr2:
            QMessageBox.warning(self.dlg, "Missing Inputs", "Please select both Year-1 and Year-2 polygon layers.")
            return
        if not outdir:
            QMessageBox.warning(self.dlg, "Missing Output Folder", "Please choose an output folder.")
            return

        # Stretch distance
        label = self.comboStretch.currentText().strip().lower()
        self.stretch_m = {"90 m": 90, "3 km": 3000, "6 km": 6000, "9 km": 9000}.get(label, 3000)

        if not crs.isValid():
            crs = self._suggest_equal_area_crs(lyr1, lyr2)

        try:
            self._process_core(lyr1, lyr2, crs.authid(), outdir)
            QMessageBox.information(self.dlg, "Finished", f"Processing complete.\nResults saved to:\n{outdir}")
        except Exception as e:
            QMessageBox.critical(self.dlg, "Error", f"Processing failed:\n{e}")

    # ---------------- Core Processing ----------------
    def _process_core(self, y1_layer, y2_layer, target_crs_authid, out_dir):
        y1 = gpd.read_file(y1_layer.source())
        y2 = gpd.read_file(y2_layer.source())

        if target_crs_authid:
            y1 = y1.to_crs(target_crs_authid)
            y2 = y2.to_crs(target_crs_authid)

        # Create per-year fishnets
        fishnet_y1 = self._create_fishnet(y1.total_bounds, self.cell_size_m, target_crs_authid)
        fishnet_y2 = self._create_fishnet(y2.total_bounds, self.cell_size_m, target_crs_authid)
        fishnet_y1.to_file(os.path.join(out_dir, "Fishnet_Y1.shp"))
        fishnet_y2.to_file(os.path.join(out_dir, "Fishnet_Y2.shp"))

        # Union extent for common analysis fishnet
        union_bounds = gpd.GeoSeries([y1.unary_union.union(y2.unary_union)], crs=y1.crs).total_bounds
        fishnet = self._create_fishnet(union_bounds, self.cell_size_m, target_crs_authid)
        fishnet['grid_id'] = np.arange(1, len(fishnet) + 1)
        fishnet.to_file(os.path.join(out_dir, "Fishnet_Analysis.shp"))

        # Compute erosion, accretion, unchanged
        per_grid = self._per_grid_change(fishnet, y1, y2, out_dir, target_crs_authid)
        self._aggregate_by_stretch(per_grid, self.stretch_m, self.cell_size_m, out_dir, top_n=self.top_n)

    # ---------------- Geometry helpers ----------------
    def _create_fishnet(self, bounds, cell_m, crs_authid):
        minx, miny, maxx, maxy = [float(v) for v in bounds]
        xs = np.arange(minx, maxx, cell_m)
        ys = np.arange(miny, maxy, cell_m)
        polys = [box(x, y, x + cell_m, y + cell_m) for x in xs for y in ys]
        return gpd.GeoDataFrame({'geometry': polys}, crs=crs_authid)

    def _extract_polygons(self, geom):
        """Extract only polygon/multipolygon parts from a geometry."""
        if geom is None or geom.is_empty:
            return None
        if isinstance(geom, (Polygon, MultiPolygon)):
            return geom
        if isinstance(geom, GeometryCollection):
            polys = [g for g in geom.geoms if isinstance(g, (Polygon, MultiPolygon))]
            if not polys:
                return None
            if len(polys) == 1:
                return polys[0]
            return MultiPolygon([
                p for g in polys
                for p in (g.geoms if isinstance(g, MultiPolygon) else [g])
            ])
        return None

    def _per_grid_change(self, grid_gdf, y1_gdf, y2_gdf, out_dir, crs_authid):
        if crs_authid:
            grid_gdf = grid_gdf.to_crs(crs_authid)
            y1_gdf = y1_gdf.to_crs(crs_authid)
            y2_gdf = y2_gdf.to_crs(crs_authid)

        results, erosion_rows, accretion_rows, unchanged_rows = [], [], [], []

        for grid_id, g in grid_gdf[['grid_id', 'geometry']].itertuples(index=False):
            y1_clip = y1_gdf[y1_gdf.intersects(g)]
            y2_clip = y2_gdf[y2_gdf.intersects(g)]

            if y1_clip.empty and y2_clip.empty:
                results.append((grid_id, 0.0, 0.0, 0.0))
                continue

            W1 = y1_clip.intersection(g).unary_union if not y1_clip.empty else None
            W2 = y2_clip.intersection(g).unary_union if not y2_clip.empty else None
            W1 = self._extract_polygons(W1)
            W2 = self._extract_polygons(W2)

            if (W1 is None or W1.is_empty) and (W2 is None or W2.is_empty):
                results.append((grid_id, 0.0, 0.0, 0.0))
                continue

            unchanged_area = erosion_area = accretion_area = 0.0

            # Unchanged = W1 ∩ W2
            if W1 and W2:
                inter = self._extract_polygons(W1.intersection(W2))
                if inter:
                    unchanged_area = inter.area
                    unchanged_rows.append((grid_id, inter))

            # Erosion = W2 \ W1 (new water)
            if W2:
                er = self._extract_polygons(W2.difference(W1) if W1 else W2)
                if er:
                    erosion_area = er.area
                    erosion_rows.append((grid_id, er))

            # Accretion = W1 \ W2 (land gained)
            if W1:
                ac = self._extract_polygons(W1.difference(W2) if W2 else W1)
                if ac:
                    accretion_area = ac.area
                    accretion_rows.append((grid_id, ac))

            results.append((grid_id, erosion_area, accretion_area, unchanged_area))

        per = gpd.GeoDataFrame(results, columns=['grid_id', 'erosion_m2', 'accretion_m2', 'unchanged_m2'])
        out = grid_gdf.merge(per, on='grid_id', how='left').fillna(0.0)
        out['erosion_km2'] = out['erosion_m2'] / 1e6
        out['accretion_km2'] = out['accretion_m2'] / 1e6
        out.to_file(os.path.join(out_dir, 'PerGrid_ErosionAccretion.shp'))

        def save_if_any(rows, fname, field, crs):
            if not rows: return
            gdf = gpd.GeoDataFrame(rows, columns=['grid_id', 'geometry'], crs=crs)
            gdf[field] = gdf.geometry.area
            gdf.to_file(os.path.join(out_dir, fname))

        save_if_any(erosion_rows, 'Erosion_Grids.shp', 'erosion_m2', out.crs)
        save_if_any(accretion_rows, 'Accretion_Grids.shp', 'accretion_m2', out.crs)
        save_if_any(unchanged_rows, 'Unchanged_Grids.shp', 'unchanged_m2', out.crs)

        return out

    def _aggregate_by_stretch(self, per_grid_gdf, stretch_m, cell_m, out_dir, top_n=15):
        grids_per_stretch = max(1, int(round(stretch_m / cell_m)))
        df = per_grid_gdf.copy()
        df['stretch_id'] = ((df['grid_id'] - 1) // grids_per_stretch).astype(int)
        stretch = df.dissolve(by='stretch_id', aggfunc={'erosion_km2': 'sum', 'accretion_km2': 'sum'})
        stretch.to_file(os.path.join(out_dir, 'Stretch_Summary.shp'))

        top_e = stretch.nlargest(top_n, 'erosion_km2').reset_index()
        top_a = stretch.nlargest(top_n, 'accretion_km2').reset_index()
        top_e.to_file(os.path.join(out_dir, f'Top{top_n}_Erosion_Stretches.shp'))
        top_a.to_file(os.path.join(out_dir, f'Top{top_n}_Accretion_Stretches.shp'))

        top_e[['stretch_id', 'erosion_km2']].to_csv(os.path.join(out_dir, f'Top{top_n}_Erosion_Stretches.csv'), index=False)
        top_a[['stretch_id', 'accretion_km2']].to_csv(os.path.join(out_dir, f'Top{top_n}_Accretion_Stretches.csv'), index=False)

    # ---------------- CRS helpers ----------------
    def _is_valid_crs_widget(self):
        try:
            return self.projTarget.crs().isValid()
        except Exception:
            return False

    def _suggest_equal_area_crs(self, lyr1, lyr2):
        """Suggest equal-area CRS based on centroid latitude."""
        try:
            ext = lyr1.extent().united(lyr2.extent())
            wgs84 = QgsCoordinateReferenceSystem("EPSG:4326")
            ct = QgsCoordinateTransform(lyr1.crs(), wgs84, QgsCoordinateTransformContext())
            center = ct.transform(ext.center())
            lat = center.y()
            if -43 <= lat <= 43:
                return QgsCoordinateReferenceSystem("EPSG:6933")  # World Cylindrical Equal Area
            elif lat > 43:
                return QgsCoordinateReferenceSystem("EPSG:3577")  # Albers Northern Hemisphere
            else:
                return QgsCoordinateReferenceSystem("EPSG:3035")  # LAEA Europe
        except Exception:
            return QgsCoordinateReferenceSystem("EPSG:6933")
