# -*- coding: utf-8 -*-

"""
/***************************************************************************
 SegmentationBoxes
                                 A QGIS plugin
 Segment a river polygon into boxes with given length and width
 Generated by Plugin Builder: http://g-sherman.github.io/Qgis-Plugin-Builder/
                              -------------------
        begin                : 2021-03-30
        copyright            : (C) 2021 by J. Pierson, UMR 6554 LETG, CNRS
        email                : julie.pierson@univ-brest.fr
 ***************************************************************************/

/***************************************************************************
 *                                                                         *
 *   This program is free software; you can redistribute it and/or modify  *
 *   it under the terms of the GNU General Public License as published by  *
 *   the Free Software Foundation; either version 2 of the License, or     *
 *   (at your option) any later version.                                   *
 *                                                                         *
 ***************************************************************************/
"""

__author__ = 'J. Pierson, UMR 6554 LETG, CNRS'
__date__ = '2021-03-30'
__copyright__ = '(C) 2021 by J. Pierson, UMR 6554 LETG, CNRS'

# This will get replaced with a git SHA1 when you do a git archive

__revision__ = '$Format:%H$'

from qgis.PyQt.QtCore import QCoreApplication
from qgis.core import (QgsProcessing,
                       QgsMessageLog,
                       QgsProcessingAlgorithm,
                       QgsProcessingParameterDistance,
                       QgsProcessingParameterVectorDestination,
                       QgsWkbTypes,
                       QgsProcessingParameterVectorLayer,
                       QgsVectorLayer)
import processing


class SegmentationBoxesAlgorithm(QgsProcessingAlgorithm):
    """
    This is an example algorithm that takes a vector layer and
    creates a new identical one.

    It is meant to be used as an example of how to create your own
    algorithms and explain methods and variables used to do it. An
    algorithm like this will be available in all elements, and there
    is not need for additional work.

    All Processing algorithms should extend the QgsProcessingAlgorithm
    class.
    """

    # Constants used to refer to parameters and outputs. They will be
    # used when calling the algorithm from another algorithm, or when
    # calling from the QGIS console.

    INPUT = 'INPUT'
    CENTERLINE = 'CENTERLINE'
    LENGTH = 'LENGTH'
    WIDTH = 'WIDTH'
    OUTPUT = 'OUTPUT'
    CENTERLINE_OUTPUT = 'CENTERLINE_OUTPUT'

    def initAlgorithm(self, config):
        """
        Here we define the inputs and output of the algorithm, along
        with some other properties.
        """

        # input layer (line or polygon)
        self.addParameter(
            QgsProcessingParameterVectorLayer(
                self.INPUT,
                self.tr('Input line or polygon layer'),
                [QgsProcessing.TypeVectorAnyGeometry ]
            )
        )
            
        # input centerline layer (optional)
        self.addParameter(
            QgsProcessingParameterVectorLayer(
                self.CENTERLINE,
                self.tr('Input centerline layer, can be used to speed up calculations if input layer is polygon'),
                [QgsProcessing.TypeVectorLine],
                optional=True
            )
        )
            
        # length of boxes
        self.addParameter(
            QgsProcessingParameterDistance(
                self.LENGTH,
                self.tr('Length of boxes'),
                '',
                self.INPUT
            )
        )
        
        # width of boxes (will NOT be used if input layer is polygon)
        self.addParameter(
            QgsProcessingParameterDistance(
                self.WIDTH,
                self.tr('Width of boxes, will NOT be used if input layer is polygon'),
                '',
                self.INPUT,
                optional=True
            )
        )

        # ouput polygon layer
        self.addParameter(
            QgsProcessingParameterVectorDestination(
                self.OUTPUT,
                self.tr('Output polygon layer')
                #QgsProcessing.TypeVectorPolygon
            )
        )
            
        # ouput centerline layer, created if input is polygon
        self.addParameter(
            QgsProcessingParameterVectorDestination(
                self.CENTERLINE_OUTPUT,
                self.tr('Output centerline layer, can be created if input layer is polygon'),
                optional=True
                #QgsProcessing.TypeVectorPolygon
            )
        )

    def processAlgorithm(self, parameters, context, feedback):
        """
        Here is where the processing itself takes place.
        """
        
        # Retrieve inputs
        river = self.parameterAsVectorLayer(parameters, self.INPUT, context)
        centerline = self.parameterAsVectorLayer(parameters, self.CENTERLINE, context)
        length = self.parameterAsInt(parameters, self.LENGTH, context)
        width = self.parameterAsInt(parameters, self.WIDTH, context)
       
        # if input layer is polygon and no centerline provided
        if river.geometryType() == QgsWkbTypes.PolygonGeometry and not centerline:
            # check topology
            river = self.checkTopology(river, context, feedback)
            # calculate the centerline for the input layer       
            centerline_output = self.createCenterline(river, parameters, context, feedback)
            centerline = centerline_output['output']
        # if input layer is line, it is considered as centerline
        if river.geometryType() == QgsWkbTypes.LineGeometry:
            centerline = river
            
        # if no polygon layer provided
        if river.geometryType() == QgsWkbTypes.LineGeometry:
            # if centerline is composed of multiple lines, merge them
            features = centerline.getFeatures()
            nb_features = sum(1 for _ in features)
            if nb_features > 1:
                centerline = self.mergeLines(centerline, context, feedback)
            # create buffer layer around centerline
            buffer_layer = self.createBuffer(centerline, width, context, feedback)
              
        # create points along centerline at a given interval
        points_layer = self.createPoints(centerline, length, context, feedback)
        
        # create Thiessen polygons
        thiessen_layer = self.createThiessen(points_layer, context, feedback)
        
        # if polygon layer is provided, clip thiessen by polygon
        if river.geometryType() == QgsWkbTypes.PolygonGeometry:
            boxes = self.clip(thiessen_layer, river, parameters, context, feedback)
        # else, clip by buffer layer
        else:
            boxes = self.clip(thiessen_layer, buffer_layer, parameters, context, feedback)


        # Return the results of the algorithm : segmentation boxes layer, centerline layer if wanted
        try:
            return {self.OUTPUT:boxes['OUTPUT'], self.CENTERLINE_OUTPUT:centerline_output['output']}
        except NameError:
            return {self.OUTPUT:boxes['OUTPUT']}
        
    def checkTopology(self, river, context, feedback):
        message = 'Checking topology for river layer...'
        feedback.pushInfo(QCoreApplication.translate('Segmentation Boxes', message))
        invalid = False
        # check validity for each feature in layer
        for feature in river.getFeatures():
            geom = feature.geometry()
            # if an invalid geometry is encountered
            if not geom.isGeosValid():
                invalid = True
                break
            
        # if layer has invalid geometries
        if invalid == True:
            message = 'Invalid geometry found, trying buffering by zero...'
            feedback.pushInfo(QCoreApplication.translate('Segmentation Boxes', message))
            # try buffering layer by 0
            buffer_param = {'INPUT' : river,
                      'DISTANCE' : 0,
                      'SEGMENTS' : 5,
                      'END_CAP_STYLE' : 0, # 0 : round, 1 : flat, 2 : square
                      'JOIN_STYLE' : 0, #  0 : round, 1 : square angle, 2 : oblique
                      'MITER_LIMIT' : 2,
                      'DISSOLVE' : False,
                      'OUTPUT' : 'buffer'}
            buffer_result = processing.run("native:buffer", buffer_param, is_child_algorithm=True, context=context, feedback=feedback)
            river = buffer_result['OUTPUT']
            if feedback.isCanceled():
                return {}
            
            # check validity again
            river = QgsVectorLayer(river, "river", "ogr")
            for feature in river.getFeatures():
                geom = feature.geometry()
                # if an invalid geometry is encountered again, exit with error message
                if not geom.isGeosValid():
                    message = 'Didn''t work ! Invalid geometry in river layer, please check topology'
                    feedback.reportError(QCoreApplication.translate('Distance along river', message))
                    return {}
                
        # if script wasn't exited :
        message = 'geometry is valid !'
        feedback.pushInfo(QCoreApplication.translate('Segmentation Boxes', message))
        return river
    
    def createCenterline(self, polygon, parameters, context, feedback):
        message = 'Creating  centerline with grass voronoi.skeleton algorithm...'
        feedback.pushInfo(QCoreApplication.translate('Segmentation Boxes', message))
        # voronoi.skeleton parameters
        skeleton_param = {'input' : polygon,
                  'smoothness' : 0.1,
                  'thin' : -1,
                  'output' : parameters[self.CENTERLINE_OUTPUT]} # to output this layer in QGIS as well
        # run voronoi.skeleton
        skeleton_result = processing.run("grass7:v.voronoi.skeleton", skeleton_param, is_child_algorithm=True, context=context, feedback=feedback)
        #centerline = skeleton_result['output']
        # Check for cancelation
        if feedback.isCanceled():
            return {}
        # return centerline
        return skeleton_result
    
    def mergeLines(self, line, context, feedback):
        message = 'Grouping lines with dissolve algorithm...'
        feedback.pushInfo(QCoreApplication.translate('Segmentation Boxes', message))
        # dissolve parameters
        dissolve_param = {'INPUT' : line,
                  'FIELD' : None,
                  #'OUTPUT' : parameters[self.OUTPUT]} # to output this layer in QGIS as well
                  'OUTPUT' : 'dissolve'}
        # run dissolve
        dissolve_result = processing.run("native:dissolve", dissolve_param, is_child_algorithm=True, context=context, feedback=feedback)
        dissolve_layer = dissolve_result['OUTPUT']
        # Check for cancelation
        if feedback.isCanceled():
            return {}
        # return dissolved layer
        return dissolve_layer
    
    def createBuffer(self, line, width, context, feedback):
        message = 'Creating buffer with buffer algorithm...'
        feedback.pushInfo(QCoreApplication.translate('Segmentation Boxes', message))
        # buffer parameters
        buffer_param = {'INPUT' : line,
                  'DISTANCE' : width,
                  'SEGMENTS' : 5,
                  'END_CAP_STYLE' : 0, # 0 : round, 1 : flat, 2 : square
                  'JOIN_STYLE' : 0, #  0 : round, 1 : square angle, 2 : oblique
                  'MITER_LIMIT' : 2,
                  'DISSOLVE' : True,
                  #'OUTPUT' : parameters[self.OUTPUT]} # to output this layer in QGIS as well
                  'OUTPUT' : 'buffer'}
        # run buffer
        buffer_result = processing.run("native:buffer", buffer_param, is_child_algorithm=True, context=context, feedback=feedback)
        buffer_layer = buffer_result['OUTPUT']
        # Check for cancelation
        if feedback.isCanceled():
            return {}
        # return buffered layer
        return buffer_layer
    
    def createPoints(self, line, length, context, feedback):
        message = 'Creating points along lines with pointsalonglines algorithm...'
        feedback.pushInfo(QCoreApplication.translate('Segmentation Boxes', message))
        # pointsalonglines parameters
        pointsalonglines_param = {'INPUT' : line,
                  'DISTANCE' : length,
                  'START_OFFSET' : length/2,
                  'END_OFFSET' : 0,
                  #'OUTPUT' : parameters[self.OUTPUT]} # to output this layer in QGIS as well
                  'OUTPUT' : 'points along lines'}
        # run pointsalonglines
        points_result = processing.run("native:pointsalonglines", pointsalonglines_param, is_child_algorithm=True, context=context, feedback=feedback)
        points_layer = points_result['OUTPUT']
        # Check for cancelation
        if feedback.isCanceled():
            return {}
        # return point layer
        return points_layer
    
    def createThiessen(self, points, context, feedback):
        message = 'Creating Thiessen polygons with QGIS voronoipolygons algorithm...'
        feedback.pushInfo(QCoreApplication.translate('Segmentation Boxes', message))
        # thiessen parameters
        thiessen_param = {'INPUT' : points,
                  'BUFFER' : 10,
                  #'OUTPUT' : parameters[self.OUTPUT]} # to output this layer in QGIS as well
                  'OUTPUT' : 'thiessen'}
        # run voronoipolygons algorithm
        thiessen_results = processing.run("qgis:voronoipolygons", thiessen_param, is_child_algorithm=True, context=context, feedback=feedback)
        thiessen_layer = thiessen_results['OUTPUT']
        # Check for cancelation
        if feedback.isCanceled():
            return {}
        # return thiessen polygons layer
        return thiessen_layer
    
    def clip(self, toclip, mask, parameters, context, feedback):
        message = 'Clipping Thiessen Polygons with clip algorithm...'
        feedback.pushInfo(QCoreApplication.translate('Segmentation Boxes', message))
        # thiessen parameters
        clip_param = {'INPUT' : toclip,
                  'OVERLAY' : mask,
                  'OUTPUT' : parameters[self.OUTPUT]}
        # run clip algorithm
        clip_results = processing.run("native:clip", clip_param, is_child_algorithm=True, context=context, feedback=feedback)
        # Check for cancelation
        if feedback.isCanceled():
            return {}
        # return segmentation boxes layer
        return clip_results

    def name(self):
        """
        Returns the algorithm name, used for identifying the algorithm. This
        string should be fixed for the algorithm, and must not be localised.
        The name should be unique within each provider. Names should contain
        lowercase alphanumeric characters only and no spaces or other
        formatting characters.
        """
        return 'Segmentation Boxes'

    def displayName(self):
        """
        Returns the translated algorithm name, which should be used for any
        user-visible display of the algorithm name.
        """
        return self.tr(self.name())

    def group(self):
        """
        Returns the name of the group this algorithm belongs to. This string
        should be localised.
        """
        return self.tr(self.groupId())

    def groupId(self):
        """
        Returns the unique ID of the group this algorithm belongs to. This
        string should be fixed for the algorithm, and must not be localised.
        The group id should be unique within each provider. Group id should
        contain lowercase alphanumeric characters only and no spaces or other
        formatting characters.
        """
        return ''

    def tr(self, string):
        return QCoreApplication.translate('Processing', string)

    def createInstance(self):
        return SegmentationBoxesAlgorithm()
