import csv
import os

from qgis.core import QgsProject, QgsVectorLayer
from qgis.PyQt import uic
from qgis.PyQt.QtWidgets import (
    QComboBox,
    QDialog,
    QDialogButtonBox,
    QFileDialog,
    QTableWidgetItem,
)


class WeightsGenerator(QDialog):
    """
    Main widget.
    """

    def __init__(self, project, iface) -> None:
        super(QDialog, self).__init__()
        self.iface = iface
        self.project = project

        ui_path = os.path.join(
            os.path.dirname(__file__), "ui", "weights_generator_dialog.ui"
        )
        uic.loadUi(ui_path, self)

        self.global_buttons.accepted.connect(self.main)
        self.global_buttons.rejected.connect(self.reject)
        self.textBrowser_direction.setOpenExternalLinks(True)
        self.textBrowser_weight.setOpenExternalLinks(True)

        self.pushButton_exposure.clicked.connect(self.load_csv_to_table)
        self.pushButton_vulnerability.clicked.connect(self.load_csv_to_table)
        self.pushButton_coping_capacity.clicked.connect(self.load_csv_to_table)
        self.pushButton_default.clicked.connect(self.set_default_weights)
        self.pushButton_clear.clicked.connect(self.clear_window)
        self.global_buttons.button(
            QDialogButtonBox.StandardButton.Cancel
        ).clicked.connect(self.cancel_action)
        self.pushButton_loadweightsfile.clicked.connect(self.load_weight_to_table)

    def load_csv_to_table(self):
        # Determine the category and corresponding QLineEdit based on the button clicked
        if self.sender() == self.pushButton_exposure:
            category_value = "exp"
            line_edit = self.lineEdit_exposure
        elif self.sender() == self.pushButton_vulnerability:
            category_value = "vul"
            line_edit = self.lineEdit_vulnerability
        elif self.sender() == self.pushButton_coping_capacity:
            category_value = "cop"
            line_edit = self.lineEdit_coping_capacity
        else:
            category_value = ""
            line_edit = None  # No QLineEdit associated

        # Open file dialog to select CSV file
        file_path, _ = QFileDialog.getOpenFileName(
            None, "Open CSV File", "", "CSV Files (*.csv);;All Files (*)"
        )

        if not file_path:
            return  # User canceled the selection

        # Set the file path in the corresponding QLineEdit
        if line_edit:
            line_edit.setText(file_path)

        # Read the CSV file and extract column names
        with open(file_path, newline="", encoding="utf-8") as csvfile:
            sample = csvfile.read(1024)
            csvfile.seek(0)
            try:
                dialect = csv.Sniffer().sniff(sample, delimiters=[",", ";"])
            except csv.Error:
                # Default to comma if sniffing fails
                dialect = csv.get_dialect("excel")

            reader = csv.reader(csvfile, dialect)
            headers = next(reader)  # Get the first row (column names)

            # Filter column names that don't start with "ADM"
            filtered_columns = [col for col in headers if not col.startswith("ADM")]

            # Remove existing rows with the same category
            rows_to_delete = []
            for row in range(self.tableWidget.rowCount()):
                if (
                    self.tableWidget.item(row, 1)
                    and self.tableWidget.item(row, 1).text() == category_value
                ):
                    rows_to_delete.append(row)

            # Delete from bottom to top to avoid row index shifting
            for row in reversed(rows_to_delete):
                self.tableWidget.removeRow(row)

            # Add new rows for the filtered columns
            for col_name in filtered_columns:
                row_position = self.tableWidget.rowCount()
                self.tableWidget.insertRow(row_position)
                self.tableWidget.setItem(
                    row_position, 0, QTableWidgetItem(col_name)
                )  # variable_name
                self.tableWidget.setItem(
                    row_position, 1, QTableWidgetItem(category_value)
                )  # category

            # Add QComboBox to column 3 for all rows
            options = ["1", "-1"]
            column_index = 3
            for row in range(self.tableWidget.rowCount()):
                combo = QComboBox()
                combo.addItems(options)
                self.tableWidget.setCellWidget(row, column_index, combo)

    def set_default_weights(self):
        for row in range(self.tableWidget.rowCount()):
            self.tableWidget.setItem(row, 2, QTableWidgetItem("1"))  # Set weight to 1

    def main(self) -> None:
        """Executed when OK button is clicked."""
        # Open a file dialog to select the output CSV file location
        file_path, _ = QFileDialog.getSaveFileName(
            self, "Save CSV File", "", "CSV Files (*.csv);;All Files (*)"
        )

        if not file_path:
            return  # User canceled the file dialog

        # Ensure the file has a .csv extension
        if not file_path.endswith(".csv"):
            file_path += ".csv"

        # Open the file in write mode
        with open(file_path, mode="w", newline="", encoding="utf-8") as csvfile:
            writer = csv.writer(csvfile)

            # Write the header row
            header = [
                self.tableWidget.horizontalHeaderItem(col).text()
                for col in range(self.tableWidget.columnCount())
            ]
            writer.writerow(header)

            # Write the data rows
            for row in range(self.tableWidget.rowCount()):
                row_data = []
                for col in range(self.tableWidget.columnCount()):
                    cell_widget = self.tableWidget.cellWidget(row, col)
                    if isinstance(cell_widget, QComboBox):
                        row_data.append(cell_widget.currentText())
                    else:
                        item = self.tableWidget.item(row, col)
                        row_data.append(item.text() if item else "")
                writer.writerow(row_data)

        # Now, load the CSV file as a table in QGIS (non-spatial)
        layer = QgsVectorLayer(
            f"file:///{file_path}?type=csv&delimiter=,&detectTypes=no",
            "Weights",
            "delimitedtext",
        )

        if not layer.isValid():
            print("Failed to load the CSV layer!")
            return

        # Add the CSV table to the QGIS project
        QgsProject.instance().addMapLayer(layer)

        # Close the dialog (indicating the process is complete)
        self.accept()

    def reset_plugin(self):
        """Resets all UI elements to their default state."""
        self.tableWidget.setRowCount(0)  # Clear all rows in tableWidget
        self.lineEdit_exposure.clear()  # Clear the exposure line edit
        self.lineEdit_vulnerability.clear()  # Clear the vulnerability line edit
        self.lineEdit_coping_capacity.clear()  # Clear the coping capacity line edit

    def closeEvent(self, event):
        """Reset the plugin when the window is closed."""
        self.reset_plugin()
        event.accept()  # Accept the close event

    def cancel_action(self):
        """Resets the plugin when Cancel is clicked and closes the dialog."""
        self.reset_plugin()  # Reset all fields
        self.close()  # Close the dialog

    def clear_window(self):
        """Clears all fields but keeps the window open."""
        self.reset_plugin()  # Resets table and text fields

    def load_weight_to_table(self):
        # Open file dialog to select CSV file
        file_path, _ = QFileDialog.getOpenFileName(
            None, "Load Weight CSV", "", "CSV Files (*.csv);;All Files (*)"
        )

        if not file_path:
            return  # User canceled

        # Clear existing table rows
        self.tableWidget.setRowCount(0)

        # Read the CSV and add each row to the table
        with open(file_path, newline="", encoding="utf-8") as csvfile:
            reader = csv.DictReader(csvfile)
            for row_data in reader:
                variable_name = row_data.get("variable_name", "").strip()
                category = row_data.get("category", "").strip()
                weight = row_data.get("weight", "").strip()
                direction = row_data.get("direction", "").strip()

                if variable_name:
                    row_position = self.tableWidget.rowCount()
                    self.tableWidget.insertRow(row_position)

                    # Column 0: variable_name
                    self.tableWidget.setItem(
                        row_position, 0, QTableWidgetItem(variable_name)
                    )

                    # Column 1: category
                    self.tableWidget.setItem(
                        row_position, 1, QTableWidgetItem(category)
                    )

                    # Column 2: weight
                    self.tableWidget.setItem(row_position, 2, QTableWidgetItem(weight))

                    # Column 3: direction (as QComboBox)
                    combo = QComboBox()
                    combo.addItems(["1", "-1"])
                    if direction in ["1", "-1"]:
                        combo.setCurrentText(direction)
                    self.tableWidget.setCellWidget(row_position, 3, combo)
