# -*- coding: utf-8 -*-

"""
/***************************************************************************
 RiskAssessment
                                 A QGIS plugin
 Risk assessment calculation for forecast based financing
 Generated by Plugin Builder: http://g-sherman.github.io/Qgis-Plugin-Builder/
                              -------------------
        begin                : 2023-10-11
        copyright            : (C) 2023 by HeiGIT gGmbH
        email                : info@heigit.org
 ***************************************************************************/

/***************************************************************************
 *                                                                         *
 *   This program is free software; you can redistribute it and/or modify  *
 *   it under the terms of the GNU General Public License as published by  *
 *   the Free Software Foundation; either version 2 of the License, or     *
 *   (at your option) any later version.                                   *
 *                                                                         *
 ***************************************************************************/
"""

__author__ = 'HeiGIT gGmbH'
__date__ = '2023-10-11'
__copyright__ = '(C) 2023 by HeiGIT gGmbH'

# This will get replaced with a git SHA1 when you do a git archive

__revision__ = '$Format:%H$'


from PyQt5.QtCore import QVariant
from qgis.PyQt.QtCore import QCoreApplication
from qgis._core import QgsField, QgsFields, Qgis
from qgis.core import (QgsProcessing,
                       QgsFeatureSink,
                       QgsProcessingAlgorithm,
                       QgsProcessingParameterFeatureSource,
                       QgsProcessingParameterFeatureSink,
                       QgsProcessingParameterFile,
                       QgsMessageLog)

import pandas as pd
from numpy import isnan

from .utils import *



class RiskAssessmentAlgorithm(QgsProcessingAlgorithm):
    """
    This is an example algorithm that takes a vector layer and
    creates a new identical one.

    It is meant to be used as an example of how to create your own
    algorithms and explain methods and variables used to do it. An
    algorithm like this will be available in all elements, and there
    is not need for additional work.

    All Processing algorithms should extend the QgsProcessingAlgorithm
    class.
    """

    # Constants used to refer to parameters and outputs. They will be
    # used when calling the algorithm from another algorithm, or when
    # calling from the QGIS console.

    OUTPUT = 'OUTPUT'
    DISTRICTS = 'DISTRICTS'
    EXPOSURE = 'EXPOSURE'
    COPING = 'COPING'
    VUL = 'VULNERABILITY'
    WEIGHTS = 'WEIGHTS'
    TRANSFORM = 'TRANSFORM'
    POP = 'POPULATION'
    INPUT = 'INPUT'

    def initAlgorithm(self, config):
        """
        Here we define the inputs and output of the algorithm, along
        with some other properties.
        """

        # We add the input vector features source. It can have any kind of
        # geometry.
        self.addParameter(QgsProcessingParameterFeatureSource(
                name=self.DISTRICTS,
                description=self.tr('<b>Inputs</b><br>Administrative Boundaries'),
                types=[QgsProcessing.TypeVectorPolygon]
        ))

        self.addParameter(QgsProcessingParameterFile(
            name=self.EXPOSURE,
            description=self.tr('Exposure Indicators (.csv)'),
            extension='csv'
        ))

        self.addParameter(QgsProcessingParameterFile(
            name=self.COPING,
            description=self.tr('Coping Capacity Indicators (.csv)'),
            extension='csv'
        ))

        self.addParameter(QgsProcessingParameterFile(
            name=self.VUL,
            description=self.tr('Vulnerability Indicators (.csv)'),
            extension='csv'
        ))

        self.addParameter(QgsProcessingParameterFile(
            name=self.WEIGHTS,
            description=self.tr('Weights (.csv)'),
            extension='csv'
        ))

        self.addParameter(QgsProcessingParameterFile(
            name=self.TRANSFORM,
            description=self.tr('Transformation (.csv)'),
            extension='csv'
        ))

        self.addParameter(QgsProcessingParameterFile(
            name=self.POP,
            description=self.tr('Population Data (.csv)'),
            extension='csv'
        ))

        self.addParameter(QgsProcessingParameterFeatureSink(
            name=self.OUTPUT,
            description=self.tr('<b>Outputs</b><br>Risk Assessment Vector Layer'),
        ))


    def processAlgorithm(self, parameters, context, feedback):
        districts = self.parameterAsSource(parameters, self.DISTRICTS, context)

        #############
        # Read Data #
        #############
        exposure_file = self.parameterAsFile(parameters, self.EXPOSURE, context)
        coping_file = self.parameterAsFile(parameters, self.COPING, context)
        vul_file = self.parameterAsFile(parameters, self.VUL, context)
        weights_file = self.parameterAsFile(parameters, self.WEIGHTS, context)
        transform_file = self.parameterAsFile(parameters, self.TRANSFORM, context)
        pop_file = self.parameterAsFile(parameters, self.POP, context)

        coping = pd.read_csv(coping_file, index_col="ADM2_PCODE")
        exposure = pd.read_csv(exposure_file, index_col="ADM2_PCODE")
        vulnerability = pd.read_csv(vul_file, index_col="ADM2_PCODE")
        transform = pd.read_csv(transform_file)
        weights = pd.read_csv(weights_file)
        pop = pd.read_csv(pop_file, index_col="ADM2_PCODE")

        id_col = "ADM2_PCODE"

        # Check if data loading was successful
        if not (len(districts) == len(coping) == len(exposure) == len(vulnerability)):
            QgsMessageLog.logMessage("Error: Datasets do not have the same size.", level=Qgis.Critical)
            return

        ####################
        # Pre-Process Data #
        ####################

        # Add prefix to variable names
        # The resulting prefix length of 4 chars is significant later
        pre_coping = "cop"
        pre_exposure = "exp"
        pre_vulnerability = "vul"

        coping = coping.rename(
            columns={col: f"{pre_coping}_{col}" for col in coping.columns if not col.startswith("ADM")})
        exposure = exposure.rename(
            columns={col: f"{pre_exposure}_{col}" for col in exposure.columns if not col.startswith("ADM")})
        vulnerability = vulnerability.rename(
            columns={col: f"{pre_vulnerability}_{col}" for col in vulnerability.columns if not col.startswith("ADM")})

        # Drop not needed columns, so they don't interfere in joining.
        coping = coping.drop([col for col in coping.columns if col.startswith("ADM") and col != id_col], axis=1)
        exposure = exposure.drop([col for col in exposure.columns if col.startswith("ADM") and col != id_col], axis=1)
        vulnerability = vulnerability.drop([col for col in vulnerability.columns if col.startswith("ADM") and col != id_col], axis=1)

        indicators = coping.join(exposure).join(vulnerability)

        # Check if all dataframes were merged successfully
        if not (len(districts) == len(indicators)):
            QgsMessageLog.logMessage("Datasets do not have the same size after merging.", level=Qgis.Critical)
            return

        indicators_with_pop = indicators.join(pop)

        ####################
        # Risk Calculation #
        ####################

        # Divide variables defined in transform by population
        indicators_by_population = transform_by_pop(indicators_with_pop, transform)

        # worldpop is not needed anymore, so it can be dropped
        indicators_by_population = indicators_by_population.drop([col for col in indicators_by_population.columns if col.startswith("wpop")], axis=1)

        normalized_indicators = normalize_indicators(indicators_by_population)

        full_indicators = guess_missing_indicators(normalized_indicators)

        scores = calculate_scores(full_indicators, weights)
        #scores = calculate_scores(normalized_indicators, weights)

        number_of_indicators = {
            "cop" : sum([1 for col in scores.columns if col.startswith("cop")]),
            "vul": sum([1 for col in scores.columns if col.startswith("vul")]),
            "exp": sum([1 for col in scores.columns if col.startswith("exp")])
        }

        normal_scores = normalize_scores(scores, number_of_indicators)

        # Calculate susceptibility and risk scores
        normal_scores["sus"] = calculate_geometric_mean(normal_scores["vul"], normal_scores["cop"])
        normal_scores["risk"] = calculate_geometric_mean(normal_scores["exp"], normal_scores["sus"])

        # Rank the risk scores
        normal_scores["ranking"] = normal_scores["risk"].rank(ascending=False)

        # define fields in output
        district_fields = districts.fields()
        name = district_fields.field(district_fields.lookupField("ADM2_EN"))
        pcode = district_fields.field(district_fields.lookupField("ADM2_PCODE"))

        output_fields = QgsFields()
        output_fields.append(name)
        output_fields.append(pcode)

        for field_name in scores.columns:
            # TODO: get appropriate data types
            output_fields.append(QgsField(field_name, QVariant.Double))

        (output, dest_id) = self.parameterAsSink(parameters, self.OUTPUT, context, output_fields,
                                                 districts.wkbType(), districts.sourceCrs())

        # Compute the number of steps to display within the progress bar and
        # get features from source
        total = 100.0 / districts.featureCount() if districts.featureCount() else 0
        features = districts.getFeatures()

        for current, feature in enumerate(features):
            # Stop the algorithm if cancel button has been clicked
            if feedback.isCanceled():
                break

            feature_name = feature["ADM2_EN"]
            feature_pcode= feature["ADM2_PCODE"]

            # setting fields will reset all attributes.
            feature.setFields(output_fields)

            feature['ADM2_EN'] = feature_name
            feature['ADM2_PCODE'] = feature_pcode

            data = normal_scores.loc[feature_pcode]

            for label in data.index:
                if isnan(data[label]):
                    feature[label] = -999
                    continue
                feature[label] = float(data[label])

            # Add a feature in the sink
            output.addFeature(feature, QgsFeatureSink.FastInsert)

            # Update the progress bar
            feedback.setProgress(int(current * total))

        # Rename output layer
        if context.willLoadLayerOnCompletion(dest_id):
            QgsMessageLog.logMessage("Renaming.")
            layer_details = context.layerToLoadOnCompletionDetails(dest_id)
            layer_details.name = "Risk Assessment Output"
        else:
            QgsMessageLog.logMessage("Cannt rename - Layer not loaded?")


        # Return the results of the algorithm. In this case our only result is
        # the feature sink which contains the processed features, but some
        # algorithms may return multiple feature sinks, calculated numeric
        # statistics, etc. These should all be included in the returned
        # dictionary, with keys matching the feature corresponding parameter
        # or output names.
        return {self.OUTPUT: dest_id}

    def shortHelpString(self):
        return """
        This algorithm calculates a risk score based on the three indicators exposure, susceptibility and coping capacity provided by the user and is applicable across different countries and disaster contexts.
        The user can assign different weights and directions to the indicators via the weights inputfile and receive different population measures based on (worldpop) population data and offset them against chosen indicators via the population and transform inputfile.
        By processing the inserted information, it will become clear which areas are predicted to be most severely impacted.

        The required input files must follow a given structure which can be found <a href=https://giscience.github.io/gis-training-resource-center/content/GIS_AA/en_qgis_risk_assessment_plugin.html">in the documentation.</a>
        The required "csv"-file containing population information based on Worldpop Sex & Age 2020 will be provided globally on admin level 2 with this plugin in the near future.
        To use the experimental plugin, please contact HeiGIT so that HeiGIT can provide the population-file for your country.
        """

    def helpUrl(self):
        return "https://giscience.github.io/gis-training-resource-center/content/GIS_AA/en_qgis_risk_assessment_plugin.html"


    def name(self):
        """
        Returns the algorithm name, used for identifying the algorithm. This
        string should be fixed for the algorithm, and must not be localised.
        The name should be unique within each provider. Names should contain
        lowercase alphanumeric characters only and no spaces or other
        formatting characters.
        """
        return 'Calculate Risk Assessment'

    def displayName(self):
        """
        Returns the translated algorithm name, which should be used for any
        user-visible display of the algorithm name.
        """
        return self.tr(self.name())

    def group(self):
        """
        Returns the name of the group this algorithm belongs to. This string
        should be localised.
        """
        return self.tr(self.groupId())

    def groupId(self):
        """
        Returns the unique ID of the group this algorithm belongs to. This
        string should be fixed for the algorithm, and must not be localised.
        The group id should be unique within each provider. Group id should
        contain lowercase alphanumeric characters only and no spaces or other
        formatting characters.
        """
        return 'Risk Assessment'

    def tr(self, string):
        return QCoreApplication.translate('Processing', string)

    def createInstance(self):
        return RiskAssessmentAlgorithm()
