import math

from qgis._core import QgsMessageLog

LAYER_PREFIXES = ["cop", "exp", "vul"]

# Function to divide selected variables by population
def transform_by_pop(indicators_with_pop_df, transform_df):
    column_names = indicators_with_pop_df.columns
    for _, row in transform_df.iterrows():
        column_to_transform = row['variable_name']
        new_name = row['new_name']
        # account for internally prefixed columns
        if column_to_transform not in column_names:
            match = [x for x in column_names if x[4:].startswith(column_to_transform)]
            QgsMessageLog.logMessage(str(column_names))
            if not match:
                raise ValueError(f"Invalid variable_name for ${column_to_transform}")
            column_to_transform = match[0]
        indicators_with_pop_df[new_name] = indicators_with_pop_df[column_to_transform] / indicators_with_pop_df["wpop_total"] * row["factor"]

        # the old column will interfere with score calculation later
        indicators_with_pop_df = indicators_with_pop_df.drop(columns=column_to_transform)

    return indicators_with_pop_df

# Function to normalize indicators
def normalize_indicators(indicators_df):
    def normalize(x):
        range_min = x.min()
        range_max = x.max()
        if math.isnan(range_min) or math.isnan(range_max):
            return x
        return (x - range_min) / (range_max - range_min)

    for prefix in LAYER_PREFIXES:
        columns_to_normalize = [col for col in indicators_df.columns if col.startswith(prefix + "_")]
        indicators_df[columns_to_normalize] = indicators_df[columns_to_normalize].apply(normalize, axis=0)

    return indicators_df


# Missing indicators will lead to problems in score calculation and are thus guessed based on a heuristic.
# Current heuristic: use the worst value possible
def guess_missing_indicators(indicators_df):
    # coping: Low values are bad
    coping_columns = [col for col in indicators_df.columns if col.startswith("cop" + "_")]
    indicators_df[coping_columns] = indicators_df[coping_columns].fillna(0)

    # vuln: High values are bad
    vulnerability_columns = [col for col in indicators_df.columns if col.startswith("vul" + "_")]
    indicators_df[vulnerability_columns] = indicators_df[vulnerability_columns].fillna(1)

    # exposure: High values are bad
    exposure_columns = [col for col in indicators_df.columns if col.startswith("exp" + "_")]
    indicators_df[exposure_columns] = indicators_df[exposure_columns].fillna(1)

    return indicators_df

# Function to calculate scores
def calculate_scores(values_df, weights_df):
    def calculate_score(row, prefix):
        weighted_sum = 0
        for _, weight_row in weights_df.iterrows():
            column_name = f"{weight_row['variable_name']}"
            if column_name.startswith(prefix):
                if column_name not in row.axes[0].values:
                    print(column_name + "weight not applicable")
                    continue
                if weight_row['direction'] == -1:
                    row[column_name] = 1 - row[column_name]
                weighted_sum += row[column_name] * weight_row['weight']
            else:
                continue
        return weighted_sum

    scores = values_df.copy()
    for prefix in LAYER_PREFIXES:
        columns_to_score = [col for col in values_df.columns if col.startswith(prefix + "_")]
        scores[prefix] = values_df[columns_to_score].apply(calculate_score, args=[prefix], axis=1)

    return scores


# Normalize scores depending on numbers of indicators.
# For coping, calculate a "lack of coping"-value afterwards.
def normalize_scores(scores, number_of_indicators):
    for prefix in LAYER_PREFIXES:
        scores[prefix] = scores[prefix].transform(lambda x: x / number_of_indicators[prefix])

    scores["cop"] = scores["cop"].transform(lambda x: 1 - x)
    return scores


# Function to calculate geometric mean of two columns
def calculate_geometric_mean(col1, col2):
    n = len(col1)
    if n == 0:
        raise ValueError("The input must not be empty.")

    if len(col2) != n:
        raise ValueError("The inputs must have the same length.")

    gm_col = [math.exp((math.log(col1[i]) + math.log(col2[i])) / 2) for i in range(n)]
    return gm_col

