# -*- coding: utf-8 -*-
"""
/***************************************************************************
 ReNCATPreprocessorDialog
                                 A QGIS plugin
 Determine plausible locations for a specific type of facility based on user-selected layers.
 Generated by Plugin Builder: http://g-sherman.github.io/Qgis-Plugin-Builder/
                             -------------------
        begin                : 2024-10-22
        git sha              : $Format:%H$
        copyright            : (C) 2024 by E Moog
        email                : ermoog@sandia.gov
 ***************************************************************************/

/***************************************************************************
 *                                                                         *
 *   This program is free software; you can redistribute it and/or modify  *
 *   it under the terms of the GNU General Public License as published by  *
 *   the Free Software Foundation; either version 2 of the License, or     *
 *   (at your option) any later version.                                   *
 *                                                                         *
 ***************************************************************************/
"""

import os
from pathlib import Path

from qgis.PyQt import QtWidgets, QtCore, QtGui
from qgis.core import QgsProject, QgsRasterLayer, QgsVectorLayer, QgsFieldProxyModel
from qgis.gui import QgsFieldComboBox, QgsMapLayerComboBox

from .dataBridge import dataBridge



class LabelHeader(QtWidgets.QLabel):
    def __init__(self, text: str, parent=None):
        QtWidgets.QLabel.__init__(self, parent)
        self.setText(text)
        ftheader = QtGui.QFont()
        ftheader.setPointSize(20)
        self.setFont(ftheader)

class ReNCATPreprocessorDialog(QtWidgets.QDialog):
    def __init__(self, preprocessor, parent=None):
        super(ReNCATPreprocessorDialog, self).__init__(parent)
        self.setWindowTitle("ReNCAT Preprocessor Menu")


        # Create a tab layout for the dialog
        tablayout = QtWidgets.QTabWidget(self)

            
        self._layout = tablayout

        ftreg = QtGui.QFont()
        ftreg.setPointSize(12)
        self._layout.setFont(ftreg)



        self._preprocessor = preprocessor
        self._dataBridge = dataBridge()



        self._initMenuMakeIntro(self._layout)
        self._initMenuMakeRaster(self._layout)
        self._initMenuMakePoints(self._layout)
        self._initMenuReNCATOutput(self._layout)

    # ---------MAKE MENU TABS ------------

    def _initMenuMakeIntro(self, layout):
        """
        Make intro panel for the plugin. Returns nothing.
        """

        tabWidget = QtWidgets.QWidget()

        # ------------ header details--------------

        # create the header layout item
        headerLayout = QtWidgets.QVBoxLayout()
  
        # Add a header layout
        header_label = QtWidgets.QLabel("ReNCAT Preprocessor")
        header_label.setAlignment(QtCore.Qt.AlignCenter)
        ft = QtGui.QFont()
        ft.setPointSize(30)

        header_label.setFont(ft)
        headerLayout.addWidget(header_label)

        # Add a descriptive paragraph at the beginning
        description_label = QtWidgets.QLabel("\n".join([
            "This plugin provides potential locations for facilities based on the weighting of the layers that you select.",
            "To make a raster layer based on selected layers, go to the 'Make Raster' tab.",
            "If there already exists a raster layer and points are to be extracted, go to the 'Make Points from Raster' tab.",
            "To export a points layer formatted as input for ReNCAT, go to the 'Make ReNCAT Input from Points' tab.",
            "\n\n"
        ]))
        description_label.setAlignment(QtCore.Qt.AlignCenter)
        description_label.setWordWrap(True)
        headerLayout.addWidget(description_label)

        credits_label = QtWidgets.QLabel("\n".join([
            "This plugin was developed by E. Moog and R. Garrett, Sandia National Laboratories, with assistance and advice from:",
            "Amanda Wachtel",
            "Darryl Melander",
            "Jimmy Quiroz",
            "Olga Hart",
            "Will Vining",
            "\n\n",
            "Please address questions or issues to our Github page (link TK) or to the corresponding developer, ermoog@sandia.gov"
        ]))
        credits_label.setWordWrap(True)
        credits_label.setAlignment(QtCore.Qt.AlignCenter)
        headerLayout.addWidget(credits_label)

        button_box = QtWidgets.QDialogButtonBox()


        button_box.addButton(QtWidgets.QDialogButtonBox.Cancel)
        button_box.rejected.connect(self.reject)

        headerLayout.addWidget(button_box)
        headerLayout.addStretch()
        tabWidget.setLayout(headerLayout)

        tabScrollArea = QtWidgets.QScrollArea()
        tabScrollArea.setWidget(tabWidget)
        tabScrollArea.setVerticalScrollBarPolicy(QtCore.Qt.ScrollBarAlwaysOn)
        tabScrollArea.setWidgetResizable(True)

        layout.addTab(tabScrollArea, "About")

        self.setMinimumSize(self.size().expandedTo(tabWidget.size().__mul__(3)))

    def _initMenuMakeRaster(self, mainlayout):
        """
        Handle the tab for creating the raster. Returns nothing.
        """
        rasterTabWidget = QtWidgets.QWidget()

        rasterTabLayout = QtWidgets.QVBoxLayout()

        # ------------ header details--------------

        # create the header layout item
        headerLayout = QtWidgets.QVBoxLayout()

        # Add a header layout
        header_label = LabelHeader("Make Raster")
        header_label.setAlignment(QtCore.Qt.AlignCenter)

        headerLayout.addWidget(header_label)

        # Add a descriptive paragraph at the beginning
        description_label = QtWidgets.QLabel("\n".join([
            "All input layers must be in the same projected CRS, either in meters or feet.",
            "The output layer will be in the same CRS as the input layer.\n",
            "If there already exists a raster layer and points are to be extracted, go to the 'Make Points from Raster' tab."
        ]))
        description_label.setWordWrap(True)
        description_label.setAlignment(QtCore.Qt.AlignCenter)
        headerLayout.addWidget(description_label)

        rasterTabLayout.addLayout(headerLayout)

        # ---------- grid resolution info ------

        # create layout item for the grid resolution stuff
        gridLayout = QtWidgets.QVBoxLayout()

        # Add a header
        grid_header_label = LabelHeader("Select grid size")
        grid_header_label.setAlignment(QtCore.Qt.AlignLeft)

        gridLayout.addWidget(grid_header_label)

        # Create a horizontal layout for the spatial resolution
        spatial_layout = QtWidgets.QHBoxLayout()

        # Create label, and input field for spatial resolution
        spatial_label = QtWidgets.QLabel(
            "Desired size of grid in the units of the layers, either meters or feet (min 1, maximum 100,000)"
        )
        spatial_input_field = QtWidgets.QSpinBox()
        spatial_input_field.setRange(1, int(1e5))  # Set range
        spatial_input_field.setValue(100)  # Default value
        self.spatial_resolution = spatial_input_field

        # Add the label, and input field to the horizontal layout
        spatial_layout.addWidget(spatial_label)
        spatial_layout.addWidget(spatial_input_field)

        # Add the horizontal layout
        gridLayout.addLayout(spatial_layout)

        rasterTabLayout.addLayout(gridLayout)

        # --------layer extent selector -------

        # Create a horizontal layout for the extent info
        extent_layout = QtWidgets.QHBoxLayout()

        # Create label, and input field for extent info
        extent_label = QtWidgets.QLabel(
            "Use the intersection of layers' extents? (Default: union)"
        )
        extent_label.setToolTip(
            "Union means that the extent of all the selected layers will be incorporated, whereas "
            "intersection will only include the area where all layers overlap. So if you have a layer that "
            "only partially overlaps with the others, and you only want points in that layer, use the "
            "intersection option. If you want to consider points outside that layer, use the default, which is union."
        )

        checkbox_useIntersection = QtWidgets.QCheckBox()
        self.useIntersection = checkbox_useIntersection

        # Add the label, and input field to the horizontal layout
        extent_layout.addWidget(extent_label)
        extent_layout.addWidget(checkbox_useIntersection)

        # Add the horizontal layout
        rasterTabLayout.addLayout(extent_layout) 

        # ------------- layer-related info ----------

        # Add a header
        layer_header_label = LabelHeader("Select layers and weights")
        layer_header_label.setAlignment(QtCore.Qt.AlignLeft)

        rasterTabLayout.addWidget(layer_header_label)  # item 5

        # Provide instructions for the layers - may need multiple of these because they keep getting cut off, I think
        layer_description_label1 = QtWidgets.QLabel("\n".join([
            "To include a layer, check the 'include' box to the left of that layer.\n",
            "If there is a column (field) with data that should be used, instead of simply using the "
            "locations of the items in a vector layer, check "
            "the 'Use field' checkbox to the left of the field dropdown. If using a field, the field must be "
            "numeric and nonzero. ", 
            "Raster layers will have their first channel used, and no others.\n", 
            "Weights are 1 (not important) to 10 (extremely important).\n",
            "'Radius' describes how far away a point or line affects its surroundings, "
            "in the units of the layer (meters or feet). Radius is not used for polygon layers.",
            "Note that if a radius is so large that the entire area of interest is covered with a single value, "
            "this creates problems with the overall evaluation. "
            "In such cases the offending layer is ignored - check the 'warnings' panel for "
            "notifications of this. One hint that this has happened is that the results layer "
            "may have an error that says it is unavailable.\n"      
        ]))

        layer_description_label1.setWordWrap(True)
        rasterTabLayout.addWidget(layer_description_label1)
        

        # Create a column header layout for the checkboxes
        header_layout = QtWidgets.QHBoxLayout()
        include_label = QtWidgets.QLabel("Include?")
        layername_label = QtWidgets.QLabel("Layer Name")
        fieldname_label = QtWidgets.QLabel("Field to use")
        includefield_label = QtWidgets.QLabel("Use field?")
        weight_label = QtWidgets.QLabel("Weight")
        radius_label = QtWidgets.QLabel("Radius")

        header_layout.addWidget(include_label)
        header_layout.addWidget(layername_label)
        header_layout.addWidget(includefield_label)
        header_layout.addWidget(fieldname_label)
        header_layout.addWidget(weight_label)
        header_layout.addWidget(radius_label)
        
        # magic numbers so the text fields don't get cut off and don't get too large
        include_label.setFixedWidth(100)
        includefield_label.setFixedWidth(100)
        fieldname_label.setFixedWidth(200)
        weight_label.setFixedWidth(50)
        radius_label.setFixedWidth(75)

        # Add the header layout to the main layout
        rasterTabLayout.addLayout(header_layout)  

        # Get all layers in the current QGIS project
        layers = QgsProject.instance().mapLayers().values()

        # Create a dictionary to hold layer names, checkboxes, and their corresponding input fields
        self.layer_inputs = {}

        # Create input fields for each layer
        for layer in layers:
            layer_name = layer.name()

            # Create a horizontal layout for each layer
            layer_layout = QtWidgets.QHBoxLayout()

            # Create items describing how to handle the layer
            checkbox_uselayer = QtWidgets.QCheckBox()
            label = QtWidgets.QLabel(layer_name)
            checkbox_usefield = (
                QtWidgets.QCheckBox()
            )  # add a checkbox if a field should be used, or the locations
            fielddropdown = QgsFieldComboBox()
            fielddropdown.setLayer(layer)
            fielddropdown.setFilters(QgsFieldProxyModel.Numeric)

            # this is the weight of the layer with respect to other layers
            weight_field = QtWidgets.QSpinBox()
            weight_field.setRange(1, 10)  # Set range from 1 to 10
            weight_field.setValue(5)  # Default value

            radius_field = QtWidgets.QSpinBox()
            radius_field.setRange(0, int(1e6))
            radius_field.setValue(int(1e4))
            # enable spinbox if is a line or points layer
            radius_field.setDisabled(True)
            if isinstance(layer, QgsVectorLayer):
                if layer.geometryType().name in ["Point", "Line"]: 
                    radius_field.setDisabled(False)
            
            

            checkbox_uselayer.setFixedWidth(50)
            checkbox_usefield.setFixedWidth(50)
            fielddropdown.setFixedWidth(200)
            weight_field.setFixedWidth(50)
            radius_field.setFixedWidth(75)

            # Add the checkbox, label, and input field to the horizontal layout
            layer_layout.addWidget(checkbox_uselayer)

            layer_layout.addWidget(label)
            layer_layout.addWidget(checkbox_usefield)
            layer_layout.addWidget(fielddropdown)

            layer_layout.addWidget(weight_field)
            layer_layout.addWidget(radius_field)

            # Add the horizontal layout to the main vertical layout
            rasterTabLayout.addLayout(layer_layout)


            # Store the checkbox and input field in the dictionary
            self.layer_inputs[layer_name] = (
                checkbox_uselayer,
                checkbox_usefield,
                fielddropdown,
                weight_field,
                radius_field,
            )



        # --------- Add OK and Cancel buttons---------------

        makeRasterButton = QtWidgets.QPushButton(self.tr("Generate Raster"))

        button_box = QtWidgets.QDialogButtonBox()
        button_box.addButton(makeRasterButton, QtWidgets.QDialogButtonBox.ActionRole)

        button_box.addButton(QtWidgets.QDialogButtonBox.Cancel)
        makeRasterButton.clicked.connect(self.callPreprocessorMakeRasters)


        button_box.rejected.connect(self.reject)
        rasterTabLayout.addWidget(button_box)

        rasterTabLayout.addStretch()
        rasterTabWidget.setLayout(rasterTabLayout)
        rasterTabWidget.setMinimumSize(
            rasterTabWidget.size().expandedTo(rasterTabWidget.size().__mul__(1.5)) # magic to add space because otherwise things get cramped.
        )
        rasterTabWidget.setMinimumSize(
            rasterTabWidget.size().expandedTo(
                rasterTabWidget.size() + QtCore.QSize(0, 125) # magic to expand the height slightly, otherwise text gets cut off.
            )
        )

        tabScrollArea = QtWidgets.QScrollArea()
        tabScrollArea.setWidget(rasterTabWidget)
        tabScrollArea.setVerticalScrollBarPolicy(QtCore.Qt.ScrollBarAlwaysOn)
        tabScrollArea.setHorizontalScrollBarPolicy(QtCore.Qt.ScrollBarAlwaysOn)
        tabScrollArea.setWidgetResizable(True)
        

        mainlayout.addTab(tabScrollArea, "Make Raster")

    def _initMenuMakePoints(self, mainlayout):
        """
        Create tab for making points from an existing raster. Returns nothing.
        """

        pointsTabLayout = QtWidgets.QVBoxLayout()

        # ---------- select raster layer on which to operate --------
        selectRaster_header = LabelHeader("Select Raster Layer For Generating Points")


        pointsTabLayout.addWidget(selectRaster_header)

        selectRaster_layout = QtWidgets.QHBoxLayout()

        selectRaster_label = QtWidgets.QLabel(
            "Select which raster layer to use to make points: "
        )

        layers = QgsProject.instance().mapLayers().values()
        rasters = {i.name(): i for i in layers if type(i) != QgsRasterLayer}
        selectRaster_comboBox = QgsMapLayerComboBox()
        selectRaster_comboBox.setExceptedLayerList(rasters.values())

        self._selectRaster_comboBox = selectRaster_comboBox

        selectRaster_layout.addWidget(selectRaster_label)
        selectRaster_layout.addWidget(selectRaster_comboBox)

        pointsTabLayout.addLayout(selectRaster_layout)

        # ------ return points filter options  -----

        # create horizontal layout
        hlayout_filter = QtWidgets.QHBoxLayout()

        # Create label, boxes, and input field
        filter_header = LabelHeader("Filter Results")
        filter_label = QtWidgets.QLabel("\n".join([
            "Select which goodness values to return.",
            "If 'Return all' is selected, then all points will be returned regardless of goodness.",
            "If 'Return if at least specific value' is selected, a specific value should "
            "be put in the value box, by which to filter.",
            "If no option is selected, only points associated with the maximum goodness value will be returned."
        ]))
        filter_label.setWordWrap(True)

        # add checkboxes
        filter_all_label = QtWidgets.QLabel("Return all")
        filter_all_label.setFixedWidth(100)
        filter_all_checkbox = QtWidgets.QCheckBox()
        filter_all_checkbox.setFixedWidth(20)
        filter_minimum_label = QtWidgets.QLabel("Return if at least specific value: ")
        filter_minimum_label.setFixedWidth(220)
        filter_minimum_checkbox = QtWidgets.QCheckBox()
        filter_minimum_checkbox.setFixedWidth(20)
        self.filter_all_checkbox = filter_all_checkbox
        self.filter_minimum_checkbox = filter_minimum_checkbox

        # # add spinbox
        filter_spinbox = QtWidgets.QDoubleSpinBox()
        filter_spinbox.setRange(int(-1e6), int(1e6))
        filter_spinbox.setValue(1)  # Default value
        self.filter_spinbox = filter_spinbox

        # Add the label, and input field to the horizontal layout
        hlayout_filter.addWidget(filter_all_checkbox)
        hlayout_filter.addWidget(filter_all_label)
        hlayout_filter.addWidget(filter_minimum_checkbox)
        hlayout_filter.addWidget(filter_minimum_label)
        hlayout_filter.addWidget(filter_spinbox)
        hlayout_filter.addStretch()

        # Add the horizontal layout
        pointsTabLayout.addWidget(filter_header)
        pointsTabLayout.addWidget(filter_label)
        pointsTabLayout.addLayout(hlayout_filter)

        # --------- Add OK and Cancel buttons---------------

        makePointsButton = QtWidgets.QPushButton(self.tr("Generate Points"))
        makePointsButton.clicked.connect(self.callPreprocessorMakePoints)

        button_box = QtWidgets.QDialogButtonBox()
        button_box.addButton(makePointsButton, QtWidgets.QDialogButtonBox.ActionRole)
        button_box.addButton(QtWidgets.QDialogButtonBox.Cancel)
        button_box.rejected.connect(self.reject)

        pointsTabLayout.addWidget(button_box)

        pointsTabLayout.addStretch()
        pointsTabWidget = QtWidgets.QWidget()
        pointsTabWidget.setLayout(pointsTabLayout)

        mainlayout.addTab(pointsTabWidget, "Make Points from Raster")

    def _initMenuReNCATOutput(self, layout):
        """
        Creates the tab that includes ReNCAT output options.
        """

        rencatTabLayout = QtWidgets.QVBoxLayout()

        # ------------ add header and descriptors ------------

        header = LabelHeader("Export for ReNCAT Use")

        description = QtWidgets.QLabel("\n".join([
            "Export the points layer as a JSON file for use in ReNCAT.",
            "Points' locations will be in lat/long, WGS84 (EPSG: 4326)."
        ]))
        description.setWordWrap(True)

        rencatTabLayout.addWidget(header)
        rencatTabLayout.addWidget(description)

        # -------- add layer selector for the points layer to export ----------
        layerDescription = QtWidgets.QLabel("Points layer to export:")

        layers = QgsProject.instance().mapLayers().values()
        vectors = {
            i.name(): i
            for i in layers
            if (
                (type(i) != QgsVectorLayer)
                or (type(i) == QgsVectorLayer and i.geometryType().name != "Point")
            )
        }
        selectExportLayer_comboBox = QgsMapLayerComboBox()
        selectExportLayer_comboBox.setExceptedLayerList(vectors.values())
        self._exportLayer = selectExportLayer_comboBox

        layerLayout = QtWidgets.QHBoxLayout()
        layerLayout.addWidget(layerDescription)
        layerLayout.addWidget(selectExportLayer_comboBox)

        rencatTabLayout.addLayout(layerLayout)
        
        
        # -------- add input box for the sector --------------
        sectorDescription = QtWidgets.QLabel("Type of facility that is located at these points (e.g., 'EV charger' or 'grocery store'):")
        sectorText = QtWidgets.QLineEdit()
        self._sectorText = sectorText
        
        sectorLayout = QtWidgets.QHBoxLayout()
        sectorLayout.addWidget(sectorDescription)
        sectorLayout.addWidget(sectorText)
        rencatTabLayout.addLayout(sectorLayout)

        # ----------add input boxes for the json file and descriptors ----------

        boxDescription = QtWidgets.QLabel("Save location for ReNCAT-formatted file: ")
        boxText = QtWidgets.QLineEdit()
        self._exportBoxText = boxText
        boxFileSelect = QtWidgets.QPushButton("...")

        boxFileSelect.clicked.connect(self.saveFileDialog)

        inputLayout = QtWidgets.QHBoxLayout()
        inputLayout.addWidget(boxDescription)
        inputLayout.addWidget(boxText)
        inputLayout.addWidget(boxFileSelect)

        rencatTabLayout.addLayout(inputLayout)

        # --------- Add OK and Cancel buttons---------------

        exportButton = QtWidgets.QPushButton(self.tr("Export Points"))
        exportButton.clicked.connect(self.callPreprocessorReNCATExport)

        button_box = QtWidgets.QDialogButtonBox()
        button_box.addButton(exportButton, QtWidgets.QDialogButtonBox.ActionRole)
        button_box.addButton(QtWidgets.QDialogButtonBox.Cancel)
        button_box.rejected.connect(self.reject)

        rencatTabLayout.addWidget(button_box)

        rencatTabLayout.addStretch()

        rencatTabWidget = QtWidgets.QWidget()
        rencatTabWidget.setLayout(rencatTabLayout)

        layout.addTab(rencatTabWidget, "Export Points for ReNCAT")
        
        
        
        
    # # --------- CUSTOM ACTION FOR RESIZING --------------
    def resizeEvent(self, event): 
       self._layout.setMinimumSize(event.size())
       self._layout.setMaximumSize(event.size())
    

    

    # ---------------CUSTOM ACTIONS FOR PROCESSING -------------
    def callPreprocessorMakeRasters(self):
        """
        Calls a method of the preprocessor to generate rasters based on user input data
        """
        self._preprocessor.createRasterStep(self)

    def callPreprocessorMakePoints(self):
        """
        Calls a method of the preprocessor to generate points based on the selected raster and values.

        """
        self._preprocessor.createPointsStep(self)

    def saveFileDialog(self):
        """
        Handles the dialog window for the save file in the ReNCAT processing. Adapted from
        https://www.tutorialspoint.com/pyqt/pyqt_qfiledialog_widget.htm
        and
        https://www.pythontutorial.net/pyqt/pyqt-qfiledialog/
        """

        filename, ok = QtWidgets.QFileDialog.getSaveFileName(
            self,
            "Selected Save File",
            str(Path.home()),
            self.tr("JSON files (*.json)"),
        )
        if filename:
            path = Path(filename)
            self._exportBoxText.setText(str(path))

    def callPreprocessorReNCATExport(self):
        """
        Calls a method of the preprocessor to export the selected points layer in a ReNCAT-friendly format.
        """
        self._preprocessor.exportReNCATStep(self)

    # -------------- GETTERS --------------------

    def get_layer_input_values(self):
        """Return a dictionary of layer names and their corresponding input values."""
        ret = {}
        for layer_name, (
            checkbox_uselayer,
            checkbox_usefield,
            intralayer_weight_fieldlabel,
            interlayer_weight_field,
            radius_field,
        ) in self.layer_inputs.items():
            if checkbox_uselayer.isChecked():
                ret[layer_name] = {
                    "field": intralayer_weight_fieldlabel.currentField(),  # this is going to be a string
                    "usefield": checkbox_usefield.isChecked(),
                    "weight": interlayer_weight_field.value(),
                    "radius": radius_field.value(),
                }
            else:
                ret[layer_name] = None

        return ret

    def get_grid_resolution(self):
        """Return the desired input layer resolution in meters"""
        return self.spatial_resolution.value()

    def get_minimum_return_value_info(self):
        """
        Return info about the minimum desirable return value.

        Possible return values:
            "ALL": return everything regardless of goodness
            "MAX": return only those points with the maximum value
            <some float value>: return points with this value or above.

            If no box is checked, returns MAX.
        """
        if (
            self.filter_all_checkbox.isChecked()
            and self.filter_minimum_checkbox.isChecked()
        ):  # if you've checked both, just use max.
            return "MAX"
        elif self.filter_all_checkbox.isChecked():
            return "ALL"
        elif self.filter_minimum_checkbox.isChecked():
            return self.filter_spinbox.value()
        else:
            return "MAX"

    def get_extent_type(self):
        """
        Return whether to use the union, or the intersection, of the extents, based on the checkbox the user selected.
        """

        if self.useIntersection.isChecked():
            return "intersection"
        else:
            return "union"

    def get_rasterLayerToProcess(self):
        """
        Return the QgsRasterLayer to turn into points, or None if no such field selected.
        """
        return self._selectRaster_comboBox.currentLayer()

    def get_dataBridge(self):
        return self._dataBridge

    def get_exportLayer(self):
        return self._exportLayer.currentLayer()

    def get_exportPath(self):
        """
        The path to which to export the ReNCAT file.
        """
        return self._exportBoxText.text()
        
    def get_sectorText(self): 
        """
        What to label the facilities' sector as when exporting for ReNCAT.
        """
        if self._sectorText.text() == "": 
            return "facility"
        else: 
            return self._sectorText.text()

    # # ---------------SETTERS ---------------


