# -*- coding: utf-8 -*-
"""
/***************************************************************************
 ReNCATPreprocessor
                                 A QGIS plugin
 Determine plausible locations for a specific type of facility based on user-selected layers.
 Generated by Plugin Builder: http://g-sherman.github.io/Qgis-Plugin-Builder/
                              -------------------
        begin                : 2024-10-22
        git sha              : $Format:%H$
        email                : ermoog@sandia.gov
 ***************************************************************************/

/***************************************************************************
 *                                                                         *
 *   This program is free software; you can redistribute it and/or modify  *
 *   it under the terms of the GNU General Public License as published by  *
 *   the Free Software Foundation; either version 2 of the License, or     *
 *   (at your option) any later version.                                   *
 *                                                                         *
 ***************************************************************************/
"""
from qgis.PyQt.QtCore import QSettings, QTranslator, QCoreApplication
from qgis.PyQt.QtGui import QIcon, QResizeEvent
from qgis.PyQt.QtWidgets import QAction
from qgis.core import QgsProject

# Initialize Qt resources from file resources.py
from .resources import *

# Import the code for the dialog
from .rencat_preprocessor_dialog import ReNCATPreprocessorDialog
from .dataBridge import dataBridge
from .preprocessorCalculator import preprocessorCalculator
from .rencatIO import rencatInputWriter

import os.path


class ReNCATPreprocessor:
    """QGIS Plugin Implementation."""

    def __init__(self, iface):
        """Constructor.

        :param iface: An interface instance that will be passed to this class
            which provides the hook by which you can manipulate the QGIS
            application at run time.
        :type iface: QgsInterface
        """
        # Save reference to the QGIS interface
        self.iface = iface
        # initialize plugin directory
        self.plugin_dir = os.path.dirname(__file__)
        # initialize locale
        locale = QSettings().value("locale/userLocale")[0:2]
        locale_path = os.path.join(
            self.plugin_dir, "i18n", "ReNCATPreprocessor_{}.qm".format(locale)
        )

        if os.path.exists(locale_path):
            self.translator = QTranslator()
            self.translator.load(locale_path)
            QCoreApplication.installTranslator(self.translator)

        # Declare instance attributes
        self.actions = []
        self.menu = self.tr("&ReNCAT Preprocessor")

        # Check if plugin was started the first time in current QGIS session
        # Must be set in initGui() to survive plugin reloads
        self.first_start = None

    # noinspection PyMethodMayBeStatic
    def tr(self, message):
        """Get the translation for a string using Qt translation API.

        We implement this ourselves since we do not inherit QObject.

        :param message: String for translation.
        :type message: str, QString

        :returns: Translated version of message.
        :rtype: QString
        """
        # noinspection PyTypeChecker,PyArgumentList,PyCallByClass
        return QCoreApplication.translate("ReNCATPreprocessor", message)

    def add_action(
        self,
        icon_path,
        text,
        callback,
        enabled_flag=True,
        add_to_menu=True,
        add_to_toolbar=True,
        status_tip=None,
        whats_this=None,
        parent=None,
    ):
        """Add a toolbar icon to the toolbar.

        :param icon_path: Path to the icon for this action. Can be a resource
            path (e.g. ':/plugins/foo/bar.png') or a normal file system path.
        :type icon_path: str

        :param text: Text that should be shown in menu items for this action.
        :type text: str

        :param callback: Function to be called when the action is triggered.
        :type callback: function

        :param enabled_flag: A flag indicating if the action should be enabled
            by default. Defaults to True.
        :type enabled_flag: bool

        :param add_to_menu: Flag indicating whether the action should also
            be added to the menu. Defaults to True.
        :type add_to_menu: bool

        :param add_to_toolbar: Flag indicating whether the action should also
            be added to the toolbar. Defaults to True.
        :type add_to_toolbar: bool

        :param status_tip: Optional text to show in a popup when mouse pointer
            hovers over the action.
        :type status_tip: str

        :param parent: Parent widget for the new action. Defaults None.
        :type parent: QWidget

        :param whats_this: Optional text to show in the status bar when the
            mouse pointer hovers over the action.

        :returns: The action that was created. Note that the action is also
            added to self.actions list.
        :rtype: QAction
        """

        icon = QIcon(icon_path)
        action = QAction(icon, text, parent)
        action.triggered.connect(callback)
        action.setEnabled(enabled_flag)

        if status_tip is not None:
            action.setStatusTip(status_tip)

        if whats_this is not None:
            action.setWhatsThis(whats_this)

        if add_to_toolbar:
            # Adds plugin icon to Plugins toolbar
            self.iface.addToolBarIcon(action)

        if add_to_menu:
            self.iface.addPluginToMenu(self.menu, action)

        self.actions.append(action)

        return action

    def initGui(self):
        """Create the menu entries and toolbar icons inside the QGIS GUI."""

        icon_path = ":/plugins/rencat_preprocessor/icon.png"
        self.add_action(
            icon_path,
            text=self.tr("Preprocess layers for ReNCAT"),
            callback=self.run,
            parent=self.iface.mainWindow(),
        )

        # will be set False in run()
        self.first_start = True

    def unload(self):
        """Removes the plugin menu item and icon from QGIS GUI."""
        for action in self.actions:
            self.iface.removePluginMenu(self.tr("&ReNCAT Preprocessor"), action)
            self.iface.removeToolBarIcon(action)

    def run(self):
        """Run method that performs all the real work"""

        # Create the dialog with elements (after translation) and keep reference
        # Only create GUI ONCE in callback, so that it will only load when the plugin is started
        if self.first_start:
            self.first_start = False
            self.dlg = ReNCATPreprocessorDialog(self)
        else:
            self.dlg = ReNCATPreprocessorDialog(self)

        
        
        # need this otherwise if you try to resize, it minimizes the entire panel
        self.dlg.setMinimumSize(QtCore.QSize(700,700)) 
        
        # need this because otherwise upon opening it sets the dimensions to the 
        # entire screen. Why? no idea.    
        self.dlg.setMaximumSize(QtCore.QSize(2500,5000)) 
    
        # show the dialog
        self.dlg.show()
        # Run the dialog event loop
        result = self.dlg.exec_()
        



    def createRasterStep(self, dlg):
        """
        Helper function called by the preprocessor dialog for managing creation of the raster.
        """
        dBridge = dlg.get_dataBridge()

        dBridge.importDataFromDialog(dlg)

        # check that there's at least one layer selected
        dBridge.checkAtLeastOneLayer()
        
        # validate
        dBridge.checkLayersCRSandUnits()
        dBridge.checkLayerTypes()

        # create the calculator object
        pC = preprocessorCalculator()
        pC.importFromDataBridge(dBridge)

        # ensure that all layers are either already rasters OR creates raster images of them
        pC.makeRasters()
        # these two lines are useful for debugging - they add every raster 
        # layer to the main project.
        # for layer in pC.getRasterLayers(): 
            # QgsProject.instance().addMapLayer(layer)

        # create raster layer based on the values
        rast = pC.calculateCombinedRaster()
        QgsProject.instance().addMapLayer(rast)

        

    def createPointsStep(self, dlg):
        """
        Helper function called by the preprocessor for making the points for a pre-existing raster.
        """

        dBridge = dlg.get_dataBridge()
        
        dBridge.importDataFromDialog(
            dlg
        )  # this needs to happen anyway due to additional info from the new panel

        pC = preprocessorCalculator()
        pC.importFromDataBridge(dBridge)

        rast = dBridge.getRasterLayerToProcess()

        # get the centroids of the points with the desired value
        desired_value = pC.findMinimumDesirableRasterValue(
            rast, dBridge
        )  # get the value to filter on

        # filter on that value while making the centroids of the raster pixels
        points_layer = pC.extractPointsFromRasterResults(rast, desired_value)

        QgsProject.instance().addMapLayer(points_layer)

    def exportReNCATStep(self, dlg):
        """
        Helper function called by the dialog for exporting the points in a
        user-selected layer in a ReNCAT-style format, to a user-selected
        path.
        """
        dBridge = dlg.get_dataBridge()

        dBridge.importDataFromDialog(
            dlg
        )  # this needs to happen anyway due to additional info from the new panel

        pC = preprocessorCalculator()
        pC.importFromDataBridge(dBridge)

        ids, xvals, yvals = pC.preparePointsLayerForRencatExport(
            dBridge.getExportLayer()
        )
        
        sectors = [dBridge.getExportSector()]*len(ids)
        RIW = rencatInputWriter(dBridge)
        RIW.createRencatInputFile(ids, yvals, xvals, sectors)
