import json
import pandas as pd
import numpy as np

from . import dataBridge


class rencatFacility:
    """
    Helper class for the rencatInput
    """

    def __init__(
        self,
        objectid,
        latitude,
        longitude,
        sector
    ):
        self._id = (objectid,)
        self._latitude = (latitude,)
        self._longitude = (longitude,)
        self._sector = (sector,)

    def asDict(self):
        dct = {
            "id": str(self._id[0]),
            "latitude": self._latitude[0],
            "longitude": self._longitude[0],
            "category": self._sector[0],
        }

        return dct

    def id(self):
        return self._id


class rencatInput:
    """
    Helper class for the rencatInputWriter
    """

    def __init__(self):
        self._facilities = {}
        self._model = {}

    def asDict(self):
        self._model = {
            "facilities": [self._facilities[fac].asDict() for fac in self._facilities],
            "serviceWeights": {},
        }

        return {"model": self._model}

    def addFacility(self, fac: rencatFacility):
        if fac.id() in self._facilities:
            raise ValueError(
                "Found repeated facility index when writing out for ReNCAT."
            )
        else:
            self._facilities[fac.id()] = fac

    def numFacilities(self):
        return len(self._facilities)


class rencatInputWriter:
    def __init__(self, databridge: dataBridge.dataBridge):
        self._dataBridge = databridge

    def _createRencatInputFile(
        self,
        outputPath,
        facilityIds,
        facilityLats,
        facilityLongs,
        facilitySectors,
    ):
        """
        Creates the rencat input file that is an optional output of this
        plugin.

        inputs:
            outputPath: str
                the path to which to write the output file.


            facilityIds: list or other 1-d iterable.
                Indexes of the different facilities.

            facilityLats: list or other 1-d iterable.
                Latitudes of the facilities.

            facilityLongs: list or other 1-d iterable.
                Longitudes of the facilities.
            
            facilitySectors: list or other 1-d iterable.
                Sector of each facility.

        output: none

        side effects: writes a json string to the provided output path,
            containing information that would be used in the command line
            standalone burden calculator part of rencat, in the format
            rencat likes.
        """

        # create the rencat input class object
        r_I = rencatInput()

        # add a bunch of facilities to it
        for idx, val in enumerate(facilityIds):
            r_I.addFacility(
                rencatFacility(
                    val,
                    facilityLats[idx],
                    facilityLongs[idx],
                    facilitySectors[idx],
                )
            )

        # get the thing as a dict.
        rDict = r_I.asDict()

        # write to file
        with open(outputPath, "w") as f:
            json.dump(rDict, f, indent=4, ensure_ascii=False)

    def createRencatInputFile(self, ids, lats, lons, sectors):
        self._createRencatInputFile(
            self._dataBridge.getExportPath(),
            ids,
            lats,
            lons,
            sectors
        )
